// Copyright (C) 2009 Martin Sandve Alnes
//
// This file is part of SyFi.
//
// SyFi is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// SyFi is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with SyFi. If not, see <http://www.gnu.org/licenses/>.
//
// First added:  2009-01-01
// Last changed: 2009-04-01
//
// This demo program solves a hyperelasticity problem
// in 3D using the Saint-Vernant-Kirchoff model.

#include <iostream>
#include <vector>
#include <dolfin.h>
#include "generated_code/HyperElasticitySVK.h"

using namespace dolfin; 
using namespace HyperElasticitySVK;


// ---------------------------------------- Functions

class BodyForce: public Expression 
{
public:
  BodyForce() : Expression(3) {}

  void eval(Array<double>& values, const Array<double>& x) const
  {
    const double dx = x[0] - 0.5;
    const double dy = x[1] - 0.5;
    const double dz = x[2] - 0.5;

    values[0] = 0.0;
    values[1] = 0.0;
    values[2] = 0.0;
  }
};

class BoundaryPressure: public Expression 
{
public:
  void eval(Array<double>& values, const Array<double>& x) const
  {
    values[0] = 0.0;
  }
};

// Dirichlet boundary condition for rotation at right end
class Rotation : public Expression 
{
  public: 
  Rotation() : Expression(3) {}
  void eval(Array<double>& values, const Array<double>& x) const
  {
    // Center of rotation
    double y0 = 0.5;
    double z0 = 0.5;
    
    // Angle of rotation (60 degrees)
    double theta = 0.5236*2;
    
    // New coordinates
    double y = y0 + (x[1] - y0)*cos(theta) - (x[2] - z0)*sin(theta);
    double z = z0 + (x[1] - y0)*sin(theta) + (x[2] - z0)*cos(theta);
    
    // Clamp at right end
    values[0] = 0.0;
    values[1] = y - x[1];
    values[2] = z - x[2];
  }
};

// ---------------------------------------- Subdomains

// Sub domain for clamp at left end
class Left : public SubDomain
{
  bool inside(const Array<double>& x, bool on_boundary) const
  {
    return on_boundary && x[0] < 0.0 + DOLFIN_EPS;
  }
};

// Sub domain for rotation at right end
class Right : public SubDomain
{
  bool inside(const Array<double>& x, bool on_boundary) const
  {
    return on_boundary && x[0] > 1.0 - DOLFIN_EPS;
  }
};

class EntireBoundary: public SubDomain
{
  bool inside(const Array<double>& x, bool on_boundary) const
  {
    return on_boundary;
  }
};

// ---------------------------------------- Main program

int main(int argc, char**argv)
{
    // Geometry
    info("Mesh");
    unsigned n = 12;
    UnitCube mesh(n, n, n);

    // Function spaces
    info("Function spaces");
    Form_a_F::TestSpace V(mesh);
    CoefficientSpace_sigma_0 P(mesh);
    CoefficientSpace_mu C(mesh);
    
    // Coefficient functions
    info("Functions");
    
    // Displacement at current and two previous timesteps
    Function u(V);
    u.vector().zero();
    
    // External forces
    BodyForce g;
    BoundaryPressure sigma_0;

    // BCs
    Constant uleft(3, 0.0, 0.0);
    Rotation uright;

    // Material parameters
    Constant mu(4.0);
    Constant lamda(6.0);

    // Forms
    info("Forms");
    Form_a_f f(mesh);
    f.u = u; f.mu = mu; f.lamda = lamda;  
    Form_a_F L(V);
    L.u = u; L.g = g; L.sigma_0 = sigma_0; L.mu = mu; L.lamda = lamda;  
    Form_a_J a(V, V);
    a.u = u; a.sigma_0 = sigma_0; a.mu = mu; a.lamda = lamda;  

    // Setup boundary conditions
    info("Boundary conditions");

    // Dirichlet boundary conditions
    Left left_boundary;
    DirichletBC leftbc(V, uleft, left_boundary);
    Right right_boundary;
    DirichletBC rightbc(V, uright, right_boundary);
    
    std::vector<const BoundaryCondition*> bcs;
    bcs.push_back(&leftbc);
    bcs.push_back(&rightbc);

    // Create variational problem (a, L, bcs, cell_domains, exterior_facet_domains, interior_facet_domains, nonlinear)
    
    // Solve!
    info("Solving");
    solve(a==L, u, bcs); 

    // Postprocessing
    info("Postprocessing");
    
    // Compute internal energy for u and exact solution
    double f_value = assemble(f);

    // Compute F at u for verification
    Vector F;
    assemble(F, L);

    // Print limits of u
    cout << endl;
    cout << "==========================" << endl;
    cout << "Norm of u_h = " << u.vector().norm("l2") << endl;
    cout << "Min  of u_h = " << u.vector().min() << endl;
    cout << "Max  of u_h = " << u.vector().max() << endl;
    cout << "==========================" << endl;

    cout << endl;
    cout << "==========================" << endl;
    cout << "f(u) = " << f_value << endl;
    cout << "Norm of F = " << F.norm("l2") << endl;
    cout << "Min  of F = " << F.min() << endl;
    cout << "Max  of F = " << F.max() << endl;
    cout << "==========================" << endl;
    
    // Write displacement to file
    info("Writing to file");
    
    File ufile("u.pvd");
    ufile << u;

    //plot(w);
    info("Done!");
    
}

