///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// geo adaptation
//
// author:
//      Pierre.Saramito@imag.fr
//
// date: 2 may 2001
//
// ============================================================================
//  includes
// ============================================================================

#include "rheolef/geo.h"
#include "rheolef/iorheo.h"
#include "rheolef/rheostream.h" // i/o utility
#include "rheolef/field.h"
#include "rheolef/form.h"
#include "rheolef/tiny_element.h"
using namespace std;
namespace rheolef { 


// =======================================================================
//! calculate a metric adapted to the criterion field u0
// =======================================================================
void
georep_calculate_metric_bamg (const field& u0, field& metric, const adapt_option_type& opt)
{
    // bool clean = false; // files may be used at next iteration !
    bool verbose = iorheo::getverbose(clog);
    typedef georep::size_type size_type;

    const geo& g = u0.get_geo();
    string coord = g.coordinate_system();
    if (g.dimension() != 2) {
      error_macro ("adapt: " << g.dimension() << " not yet supported");
    }
    if (u0.n_component() != 1) {
      error_macro ("adapt: multi-component key not yet supported");
    }
    //
    // project key-field u in P1, if different approx
    //
    field u;
    if (u0.get_approx() != "P1") {
        const space& U = u0.get_space();
        space V (g, "P1");
        form p (U, V, "mass");
        form m (V, V, "mass");
        ssk<Float> fact_m = ldlt(m.uu);
        u = field(V);
        u.u = fact_m.solve(p.uu*u0.u + p.ub*u0.b);
    } else {
        u = u0;
    }
    //
    // write the key field in .bb format
    //
    string name_i = g.name();
    string key_name = name_i + ".bb";
    ofstream key (key_name.c_str());
    if (verbose) cerr << "! file \"" << key_name << "\" created." << endl;
    if (opt.double_precision) key << setprecision(numeric_limits<Float>::digits10) ;
    key << "2 1 " << u.size() << " 2" << endl;
    for (size_type dof = 0; dof < u.size(); dof++) {
        key << u.at(dof) << endl;
    }
    key << endl;
    key.close();
    string    basename  = g.basename();
    size_type i         = g.serial_number();
    string    bamg_name;
    if (i == numeric_limits<size_type>::max()) {
        //
        // write the mesh field in bamg format
        //
        i = 0;
        bamg_name = basename + ".bamg";
        ofstream bamg_stream (bamg_name.c_str());
        if (verbose) cerr << "! file \"" << bamg_name << "\" created." << endl;
	bamg_stream << setanglecorner(opt.anglecorner);
        if (opt.double_precision) bamg_stream << setprecision(numeric_limits<Float>::digits10) ;
        bamg_stream << bamg << g;
        // create also a cad file : basename + ".cad"
        bamg_stream.close();
    } else {
        bamg_name   = name_i   + ".bamg";
        i = g.serial_number();
        if (!file_exists (bamg_name)) {
            error_macro ("adapt: iteration "<<i+1<<": file `"<<bamg_name<<"' not found.");
        }
        string cad_name    = basename + ".cad";
        if (!file_exists (cad_name)) {
          error_macro ("adapt: iteration "<<i+1<<": file `"<<cad_name<<"' not found.");
        }
    }
    //
    // run bamg
    //
    string adapt_name = basename + "-" + itos(i+1);
    string adapt_bamg_name = adapt_name + ".bamg";
    string metric_name = adapt_name + ".metric";
    string adapt_key_name  = adapt_name + ".bb";
    int status = 0;
    string command = opt.generator;
    command += " -coef "     + ftos(opt.hcoef);
    command += " -errg "     + ftos(opt.err);
    command += " -hmin "     + ftos(opt.hmin);
    command += " -hmax "     + ftos(opt.hmax);
    command += " -ratio "    + ftos(opt.ratio);
    command += " -iso ";    // isotropic metric wanted
    command += " " + opt.additional + " ";
    command += " -nbv "      + itos(opt.n_vertices_max);
    command += " -NbJacobi " + itos(opt.n_smooth_metric);
    command += " -CutOff "   + ftos(opt.cutoff);
    if (opt.splitpbedge) {
        command += " -splitpbedge ";
    }
    if (opt.thetaquad != numeric_limits<Float>::max()) {
        command += " -thetaquad " + ftos(opt.thetaquad);
    }
    command += " -b "  + bamg_name 
            + " -Mbb " + key_name 
            + " -o "   + adapt_bamg_name
            + " -oM "  + metric_name;

    if (verbose) {
        command += " 1>&2";
        clog << "! " << command << endl << flush << flush;
    } else {
        command += " >/dev/null";
    }
    status = system (command.c_str());
    //
    // check status and files creation
    //
    if (status!=0) {
        // change the orientation in .cad
        string change_or="sed -e 's,^2 1 1 0$,2 1 -1 0,' < " + basename + ".cad > /tmp/cad; mv /tmp/cad " + basename + ".cad";
        status = system(change_or.c_str());
        status |= system (command.c_str());
    }
    check_macro (status == 0, "bamg exits with " << status << " status");
    check_macro (file_exists (adapt_bamg_name),
        "bamg failed to create `" << adapt_bamg_name << "' file");
    //
    // load bamg-generated metric
    //
    ifstream metric_file(metric_name.c_str());
    if (verbose) clog << "! load `" << metric_name << "'" << endl << flush;
    metric_file >> bamg >> metric;
    metric_file.close();
    //
    // clear data
    //
    if (opt.clean) {
      string command = "/bin/rm -f " + bamg_name + " " + key_name 
          + " " + adapt_bamg_name;
      string cad_name    = basename + ".cad";
      if (file_exists (cad_name)) command += " " + cad_name;
      if (verbose) clog << "! " << command << endl << flush;
      status = system (command.c_str());
    }
}

void
georep_calculate_metric_bamg (const Vector<field>& criteria, field& metric, const adapt_option_type& opt)
{
    //bool clean = false; // files may be used at next iteration !
    bool verbose = iorheo::getverbose(clog);
    typedef georep::size_type size_type;

    const geo& g = criteria.at(0).get_geo();
    string coord = g.coordinate_system();
    if (g.dimension() != 2) {
        error_macro ("adapt: " << g.dimension() << " not yet supported");
    }
    for (size_t i=1; i<criteria.size(); i++) {
        if (criteria.at(i).n_component() != 1) {
            error_macro ("adapt: multi-component key not yet supported");
        }
        if (criteria.at(i).get_approx() != "P1"
         || criteria.at(i).size() != criteria.at(0).size())  {
            error_macro ("adapt: P1 criteria should be provided");
        }
    }
    //
    // write the key field in .bb format
    //
    string name_i = g.name();
    string key_name = name_i + ".bb";
    ofstream key (key_name.c_str());
    if (verbose) cerr << "! file \"" << key_name << "\" created." << endl;
    if (opt.double_precision) key << setprecision(numeric_limits<Float>::digits10) ;
    key << "2 " << criteria.size() << " " <<  criteria.at(0).size() << " 2" << endl;
    for (size_type dof = 0; dof < criteria.at(0).size(); dof++) {
        for (size_type i =0; i<criteria.size(); i++) {
            key << criteria.at(i).at(dof) << endl;
        }
    }
    key << endl;
    key.close();
    string    basename  = g.basename();
    size_type i         = g.serial_number();
    string    bamg_name;
    if (i == numeric_limits<size_type>::max()) {
        //
        // write the mesh field in bamg format
        //
        i = 0;
        bamg_name = basename + ".bamg";
        ofstream bamg_stream (bamg_name.c_str());
        if (opt.double_precision) bamg_stream << setprecision(numeric_limits<Float>::digits10) ;
	bamg_stream << setanglecorner(opt.anglecorner);
        if (verbose) cerr << "! file \"" << bamg_name << "\" created." << endl;
        bamg_stream << bamg << g;
        // create also a cad file : basename + ".cad"
        bamg_stream.close();
    } else {
        bamg_name   = name_i   + ".bamg";
        i = g.serial_number();
        if (!file_exists (bamg_name)) {
           error_macro ("adapt: iteration "<<i+1<<": file `"<<bamg_name<<"' not found.");
        }
        string cad_name    = basename + ".cad";
        if (!file_exists (cad_name)) {
           error_macro ("adapt: iteration "<<i+1<<": file `"<<cad_name<<"' not found.");
        }
    }
    //
    // run bamg
    //
    string adapt_name = basename + "-" + itos(i+1);
    string adapt_bamg_name = adapt_name + ".bamg";
    string metric_name = adapt_name + ".metric";
    string adapt_key_name  = adapt_name + ".bb";
    int status = 0;
    string command = opt.generator;
    command += " -coef "     + ftos(opt.hcoef);
    command += " -errg "     + ftos(opt.err);
    command += " -hmin "     + ftos(opt.hmin);
    command += " -hmax "     + ftos(opt.hmax);
    command += " -ratio "    + ftos(opt.ratio);
    command += " -iso " ;        // isotropic metric wanted
    command += " " + opt.additional + " ";
    command += " -nbv "      + itos(opt.n_vertices_max);
    command += " -NbJacobi " + itos(opt.n_smooth_metric);
    command += " -CutOff "   + ftos(opt.cutoff);
    if (opt.splitpbedge) {
        command += " -splitpbedge ";
    }
    if (opt.thetaquad != numeric_limits<Float>::max()) {
        command += " -thetaquad " + ftos(opt.thetaquad);
    }
    command += " -b "  + bamg_name 
            + " -Mbb " + key_name 
            + " -o "   + adapt_bamg_name
            + " -oM "  + metric_name;

    if (verbose) {
        command += " 1>&2";
        clog << "! " << command << endl << flush;
    } else {
        command += " >/dev/null";
    }
    status = system (command.c_str());
    //
    // check status and files creation
    //
    check_macro (status == 0, "bamg exits with " << status << " status");
    check_macro (file_exists (adapt_bamg_name),
        "bamg failed to create `" << adapt_bamg_name << "' file");
    //
    // load bamg-generated metric
    //
    ifstream metric_file(metric_name.c_str());
    if (verbose) clog << "! load `" << metric_name << "'" << endl << flush;
        metric_file >> bamg >> metric;
    metric_file.close();
    //
    // clear data
    //
    if (opt.clean) {
      string command = "/bin/rm -f " + bamg_name + " " + key_name 
         + " " + adapt_bamg_name;
      string cad_name    = basename + ".cad";
      if (file_exists (cad_name)) command += " " + cad_name;
      if (verbose) clog << "! " << command << endl << flush;
      status = system (command.c_str());
    }
}

// =======================================================================
// adapt according to a metric
// =======================================================================
void
georep_metric_adapt_bamg (const field& mh0, georep& adapt_g, const adapt_option_type& opt)
{
    //   bool clean = false; // now false is simply a default in opt.clean
    bool verbose = iorheo::getverbose(clog);
    typedef georep::size_type size_type;

    const geo& g = mh0.get_geo();
    string coord = g.coordinate_system();
    if (g.dimension() != 2) {
        error_macro ("adapt: " << g.dimension() << " not yet supported");
    }
    field mh;
    if (mh0.n_component() != 1) {
        if (mh0.get_approx() != "P1") error_macro("Multicomponent metric should be P1");
          mh=mh0;
    } else if (mh0.get_approx() != "P1") {
        //
        // project metric-field u in P1, if different approx
        //
        const space& U = mh0.get_space();
        space V (g, "P1");
        form p (U, V, "mass");
        form m (V, V, "mass");
        ssk<Float> fact_m = ldlt(m.uu);
        mh = field(V);
        mh.u = fact_m.solve(p.uu*mh0.u + p.ub*mh0.b);
    } else {
        mh = mh0;
    }
    //
    // write the metric field in .metric format
    //
    string name_i = g.name();
    string key_name = name_i + ".bamgmtr";
    ofstream key (key_name.c_str());
    if (verbose) cerr << "! file \"" << key_name << "\" created." << endl;
    if (opt.double_precision) key << setprecision(numeric_limits<Float>::digits10) ;
    key << bamg << mh;
    key.close();
    string    basename  = g.basename();
    size_type i         = g.serial_number();
    string    bamg_name;
    if (i == numeric_limits<size_type>::max()) {
        //
        // write the mesh field in bamg format
        //
        i = 0;
        bamg_name = basename + ".bamg";
        ofstream bamg_stream (bamg_name.c_str());
        if (opt.double_precision) bamg_stream << setprecision(numeric_limits<Float>::digits10) ;
	bamg_stream << setanglecorner(opt.anglecorner);
        if (verbose) cerr << "! file \"" << bamg_name << "\" created." << endl;
        bamg_stream << bamg << g;
        // create also a cad file : basename + ".cad"
        bamg_stream.close();
    } else {
        bamg_name   = name_i   + ".bamg";
        i = g.serial_number();
        if (!file_exists (bamg_name)) {
            error_macro ("adapt: iteration "<<i+1<<": file `"<<bamg_name<<"' not found.");
        }
        string cad_name    = basename + ".cad";
        if (!file_exists (cad_name)) {
           error_macro ("adapt: iteration "<<i+1<<": file `"<<cad_name<<"' not found.");
        }
    }
    //
    // run bamg
    //
    string adapt_name = basename + "-" + itos(i+1);
    string adapt_bamg_name = adapt_name + ".bamg";
    int status = 0;
    string command =  opt.generator;
    command += " -coef "     + ftos(opt.hcoef);
    command += " -errg "     + ftos(opt.err);
    command += " -hmin "     + ftos(opt.hmin);
    command += " -hmax "     + ftos(opt.hmax);
    command += " -ratio "    + ftos(opt.ratio);
    command += " -anisomax " + ftos(opt.anisomax);
    command += " " + opt.additional + " ";
    command += " -nbv "      + itos(opt.n_vertices_max);
    command += " -nbs "      + itos(opt.n_vertices_max);
    command += " -NbJacobi " + itos(opt.n_smooth_metric);
    command += " -CutOff "   + ftos(opt.cutoff);
    if (opt.splitpbedge) {
        command += " -splitpbedge ";
    }
    if (opt.thetaquad != numeric_limits<Float>::max()) {
        command += " -thetaquad " + ftos(opt.thetaquad);
    }
    command += " -b "  + bamg_name 
            + " -M " + key_name 
            + " -o "   + adapt_bamg_name;
    if (verbose) {
        command += " 1>&2";
        clog << "! " << command << endl << flush;
    } else {
        command += " >/dev/null";
    }
    status = system (command.c_str());
    //
    // check status and files creation
    //
    if (status!=0) {
        // change the orientation in .cad
        string change_or="sed -e 's,^2 1 1 0$,2 1 -1 0,' < " + basename + ".cad > /tmp/cad; mv /tmp/cad " + basename + ".cad";
        status = system(change_or.c_str());
        status |= system (command.c_str());
    }
    check_macro (status == 0, "bamg exits with " << status << " status");
    check_macro (file_exists (adapt_bamg_name),
        "bamg failed to create `" << adapt_bamg_name << "' file");
    //
    // append domain names
    //
    ofstream dmn (adapt_bamg_name.c_str(), std::ios::app);
    // count the 1D domains:
    size_type ndom_edge = 0;
    for (size_type k = 0; k < g.n_domain(); k++) {
        if (g.get_domain(k).dimension() == 1) ndom_edge++;
    }
    dmn << "EdgeDomainNames" << endl
        << ndom_edge << endl;
    for (size_type k = 0; k < g.n_domain(); k++) {
        if (g.get_domain(k).dimension() != 1) continue;
        dmn << g.get_domain(k).name() << endl;
    }
    // count the 0D domains:
    size_type ndom_vertice = 0;
    for (size_type k = 0; k < g.n_domain(); k++) {
        if (g.get_domain(k).dimension() == 0) ndom_vertice++;
    }
    dmn << "VerticeDomainNames" << endl
        << ndom_vertice << endl;
    for (size_type k = 0; k < g.n_domain(); k++) {
        if (g.get_domain(k).dimension() != 0) continue;
        dmn << g.get_domain(k).name() << endl;
    }
    dmn.close();
    //
    // load bamg adapted mesh
    //
    ifstream adapt(adapt_bamg_name.c_str());
    if (verbose) clog << "! load `" << adapt_bamg_name << "'" << endl << flush;
    adapt >> setbasename(basename) >> bamg >> adapt_g; 
    adapt.close();
    adapt_g.set_name(g.familyname());
    adapt_g.set_number(g.number());
    adapt_g.set_serial_number(i+1);
    adapt_g.set_coordinate_system(coord);
    adapt_g.upgrade();
    //
    // clear data
    //
    if (opt.clean) {
      string command = "/bin/rm -f " + bamg_name + " " + key_name 
          + " " + adapt_bamg_name;
      string cad_name    = basename + ".cad";
      if (file_exists (cad_name)) command += " " + cad_name;
      if (verbose) clog << "! " << command << endl << flush;
      status = system (command.c_str());
    }
}
// =======================================================================
// adapt
// =======================================================================
void
georep_adapt_bamg (const field& u0, georep& adapt_g, const adapt_option_type& opt, bool do_interpolate)
{
    // bool clean = false; // files may be used at next iteration !
    bool verbose = iorheo::getverbose(clog);
    typedef georep::size_type size_type;

    const geo& g = u0.get_geo();
    string coord = g.coordinate_system();
    if (g.dimension() != 2) {
        error_macro ("adapt: " << g.dimension() << " not yet supported");
    }
    if (u0.n_component() != 1) {
        error_macro ("adapt: multi-component key not yet supported");
    }
    //
    // project key-field u in P1, if different approx
    //
    field u;
    if (u0.get_approx() != "P1") {
        const space& U = u0.get_space();
        space V (g, "P1");
        form p (U, V, "mass");
        form m (V, V, "mass");
        ssk<Float> fact_m = ldlt(m.uu);
        u = field(V);
        u.u = fact_m.solve(p.uu*u0.u + p.ub*u0.b);
    } else {
        u = u0;
    }
    //
    // write the key field in .bb format
    //
    string name_i = g.name();
    string key_name = name_i + ".bb";
    ofstream key (key_name.c_str());
    if (verbose) cerr << "! file \"" << key_name << "\" created." << endl;
    if (opt.double_precision) key << setprecision(numeric_limits<Float>::digits10) ;
    key << "2 1 " << u.size() << " 2" << endl;
    for (size_type dof = 0; dof < u.size(); dof++) {
        key << u.at(dof) << endl;
    }
    key << endl;
    key.close();
    string    basename  = g.basename();
    size_type i         = g.serial_number();
    string    bamg_name;
    if (i == numeric_limits<size_type>::max()) {
        //
        // write the mesh field in bamg format
        //
        i = 0;
        bamg_name = basename + ".bamg";
        string cad_name    = basename + ".cad";
        if (file_exists (bamg_name) && file_exists(cad_name)) {
           if (verbose) cerr << "! both files \"" << bamg_name << "\" and \"" << cad_name
		            << "\" exists: not recreated." << endl;
        } else {
           if (file_exists (bamg_name)) {
             if (verbose) cerr << "! file \"" << bamg_name << "\" is missing: both .bamg and .cad recreated." << endl;
	   }
	   if (file_exists (cad_name)) {
             if (verbose) cerr << "! file \"" << cad_name << "\" is missing: both .bamg and .cad recreated." << endl;
	   }
           ofstream bamg_stream (bamg_name.c_str());
           if (opt.double_precision) bamg_stream << setprecision(numeric_limits<Float>::digits10) ;
	   bamg_stream << setanglecorner(opt.anglecorner);
           if (verbose) cerr << "! file \"" << bamg_name << "\" created." << endl;
           bamg_stream << bamg << g;
           // create also a cad file : basename + ".cad"
           bamg_stream.close();
        }
    } else {
        bamg_name   = name_i   + ".bamg";
        i = g.serial_number();
        if (!file_exists (bamg_name)) {
            error_macro ("adapt: iteration "<<i+1<<": file `"<<bamg_name<<"' not found.");
        }
        string cad_name    = basename + ".cad";
        if (!file_exists (cad_name)) {
          error_macro ("adapt: iteration "<<i+1<<": file `"<<cad_name<<"' not found.");
        }
    }
    //
    // run bamg
    //
    string adapt_name = basename + "-" + itos(i+1);
    string adapt_bamg_name = adapt_name + ".bamg";
    string adapt_key_name  = adapt_name + ".bb";
    int status = 0;
    string command = opt.generator;
    command += " -coef "     + ftos(opt.hcoef);
    command += " -errg "     + ftos(opt.err);
    command += " -hmin "     + ftos(opt.hmin);
    command += " -hmax "     + ftos(opt.hmax);
    command += " -ratio "    + ftos(opt.ratio);
    command += " -anisomax " + ftos(opt.anisomax);
    command += " " + opt.additional + " ";
    command += " -nbv "      + itos(opt.n_vertices_max);
    command += " -NbJacobi " + itos(opt.n_smooth_metric);
    command += " -CutOff "   + ftos(opt.cutoff);
    if (opt.splitpbedge) {
        command += " -splitpbedge ";
    }
    if (opt.thetaquad != numeric_limits<Float>::max()) {
        command += " -thetaquad " + ftos(opt.thetaquad);
    }
    command += " -b "  + bamg_name 
            + " -Mbb " + key_name 
            + " -o "   + adapt_bamg_name;

    if (do_interpolate) {
        command += " -wbb " + adapt_key_name;
    }
    if (verbose) {
        command += " 1>&2";
        clog << "! " << command << endl << flush;
    } else {
        command += " >/dev/null";
    }
    status = system (command.c_str());
    //
    // check status and files creation
    //
    check_macro (status == 0, "bamg exits with " << status << " status");
    check_macro (file_exists (adapt_bamg_name),
        "bamg failed to create `" << adapt_bamg_name << "' file");
    check_macro (!do_interpolate || file_exists (adapt_key_name),
        "bamg failed to create `" << adapt_key_name << "' file");
    //
    // append domain names
    //
    ofstream dmn (adapt_bamg_name.c_str(), std::ios::app);
    // count the 1D domains:
    size_type ndom_edge = 0;
    for (size_type k = 0; k < g.n_domain(); k++) {
        if (g.get_domain(k).dimension() == 1) ndom_edge++;
    }
    dmn << "EdgeDomainNames" << endl
        << ndom_edge << endl;
    for (size_type k = 0; k < g.n_domain(); k++) {
        if (g.get_domain(k).dimension() != 1) continue;
        dmn << g.get_domain(k).name() << endl;
    }
    // count the 0D domains:
    size_type ndom_vertice = 0;
    for (size_type k = 0; k < g.n_domain(); k++) {
        if (g.get_domain(k).dimension() == 0) ndom_vertice++;
    }
    dmn << "VerticeDomainNames" << endl
        << ndom_vertice << endl;
    for (size_type k = 0; k < g.n_domain(); k++) {
        if (g.get_domain(k).dimension() != 0) continue;
        dmn << g.get_domain(k).name() << endl;
    }
    dmn.close();
    //
    // load bamg adapted mesh
    //
    ifstream adapt(adapt_bamg_name.c_str());
    if (verbose) clog << "! load `" << adapt_bamg_name << "'" << endl << flush;
    adapt >> setbasename(basename) >> bamg >> adapt_g; 
    adapt.close();
    adapt_g.set_name(g.familyname());
    adapt_g.set_number(g.number());
    adapt_g.set_serial_number(i+1);
    adapt_g.set_coordinate_system(coord);
    adapt_g.upgrade();
    //
    // clear data
    //
    if (opt.clean) {
      string command = "/bin/rm -f " + bamg_name + " " + key_name 
         + " " + adapt_bamg_name;
      string cad_name    = basename + ".cad";
      if (file_exists (cad_name)) command += " " + cad_name;
      if (verbose) clog << "! " << command << endl << flush;
      status = system (command.c_str());
    }
}
// obsolete:
void
georep_adapt_bamg (const Vector<field>& criteria, georep& adapt_g, const adapt_option_type& opt, bool do_interpolate)
{
    //bool clean = false; // files may be used at next iteration !
    bool verbose = iorheo::getverbose(clog);
    typedef georep::size_type size_type;

    const geo& g = criteria.at(0).get_geo();
    string coord = g.coordinate_system();
    if (g.dimension() != 2) {
        error_macro ("adapt: " << g.dimension() << " not yet supported");
    }
    for (size_t i=1; i<criteria.size(); i++) {
        if (criteria.at(i).n_component() != 1) {
            error_macro ("adapt: multi-component key not yet supported");
        }
        if (criteria.at(i).get_approx() != "P1"
         || criteria.at(i).size() != criteria.at(0).size()) {
            error_macro ("adapt: P1 criteria should be provided");
        }
    }
    //
    // write the key field in .bb format
    //
    string name_i = g.name();
    string key_name = name_i + ".bb";
    ofstream key (key_name.c_str());
    if (verbose) cerr << "! file \"" << key_name << "\" created." << endl;
    if (opt.double_precision) key << setprecision(numeric_limits<Float>::digits10) ;
    key << "2 " << criteria.size() << " " <<  criteria.at(0).size() << " 2" << endl;
    for (size_type dof = 0; dof < criteria.at(0).size(); dof++) {
        for (size_type i =0; i<criteria.size(); i++) {
            key << criteria.at(i).at(dof) << endl;
        }
    }
    key << endl;
    key.close();
    string    basename  = g.basename();
    size_type i         = g.serial_number();
    string    bamg_name;
    if (i == numeric_limits<size_type>::max()) {
        //
        // write the mesh field in bamg format
        //
        i = 0;
        bamg_name = basename + ".bamg";
        ofstream bamg_stream (bamg_name.c_str());
        if (opt.double_precision) bamg_stream << setprecision(numeric_limits<Float>::digits10) ;
	bamg_stream << setanglecorner(opt.anglecorner);
        if (verbose) cerr << "! file \"" << bamg_name << "\" created." << endl;
        bamg_stream << bamg << g;
        // create also a cad file : basename + ".cad"
        bamg_stream.close();
    } else {
        bamg_name   = name_i   + ".bamg";
        i = g.serial_number();
        if (!file_exists (bamg_name)) {
           error_macro ("adapt: iteration "<<i+1<<": file `"<<bamg_name<<"' not found.");
        }
        string cad_name    = basename + ".cad";
        if (!file_exists (cad_name)) {
           error_macro ("adapt: iteration "<<i+1<<": file `"<<cad_name<<"' not found.");
        }
    }
    //
    // run bamg
    //
    string adapt_name = basename + "-" + itos(i+1);
    string adapt_bamg_name = adapt_name + ".bamg";
    string adapt_key_name  = adapt_name + ".bb";
    int status = 0;
    string command = opt.generator;
    command += " -errg "     + ftos(opt.err);
    command += " -coef "     + ftos(opt.hcoef);
    command += " -hmin "     + ftos(opt.hmin);
    command += " -hmax "     + ftos(opt.hmax);
    command += " -ratio "    + ftos(opt.ratio);
    command += " -anisomax " + ftos(opt.anisomax);
    command += " " + opt.additional + " ";
    command += " -nbv "      + itos(opt.n_vertices_max);
    command += " -NbJacobi " + itos(opt.n_smooth_metric);
    command += " -CutOff "   + ftos(opt.cutoff);
    if (opt.splitpbedge) {
        command += " -splitpbedge ";
    }
    if (opt.thetaquad != numeric_limits<Float>::max()) {
        command += " -thetaquad " + ftos(opt.thetaquad);
    }
    command += " -b "  + bamg_name 
            + " -Mbb " + key_name 
            + " -o "   + adapt_bamg_name;

    if (do_interpolate) {
        command += " -wbb " + adapt_key_name;
    }
    if (verbose) {
        command += " 1>&2";
        clog << "! " << command << endl << flush;
    } else {
        command += " >/dev/null";
    }
    status = system (command.c_str());
    //
    // check status and files creation
    //
    if (status!=0) {
        // change the orientation in .cad
        string change_or="sed -e 's,^2 1 1 0$,2 1 -1 0,' < " + basename + ".cad > /tmp/cad; mv /tmp/cad " + basename + ".cad";
        status = system(change_or.c_str());
        status |= system (command.c_str());
    }
    check_macro (status == 0, "bamg exits with " << status << " status");
    check_macro (file_exists (adapt_bamg_name),
        "bamg failed to create `" << adapt_bamg_name << "' file");
    check_macro (!do_interpolate || file_exists (adapt_key_name),
        "bamg failed to create `" << adapt_key_name << "' file");
    //
    // append domain names
    //
    ofstream dmn (adapt_bamg_name.c_str(), std::ios::app);
    size_type ndom=0;
    for (size_type k = 0; k < g.n_domain(); k++) 
        if (g.get_domain(k).dimension() == 1) ndom++;
    dmn << "EdgeDomainNames" << endl
        << ndom << endl;
    for (size_type k = 0; k < g.n_domain(); k++) {
        if (g.get_domain(k).dimension() != 1) continue;
        dmn << g.get_domain(k).name() << endl;
    }
    dmn.close();
    //
    // load bamg adapted mesh
    //
    ifstream adapt(adapt_bamg_name.c_str());
    if (verbose) clog << "! load `" << adapt_bamg_name << "'" << endl << flush;
    adapt >> setbasename(basename) >> bamg >> adapt_g; 
    adapt.close();
    adapt_g.set_name(g.familyname());
    adapt_g.set_number(g.number());
    adapt_g.set_serial_number(i+1);
    adapt_g.set_coordinate_system(coord);
    adapt_g.upgrade();
    //
    // clear data
    //
    if (opt.clean) {
      string command = "/bin/rm -f " + bamg_name + " " + key_name 
            + " " + adapt_bamg_name;
      string cad_name    = basename + ".cad";
      if (file_exists (cad_name)) command += " " + cad_name;
      if (verbose) clog << "! " << command << endl << flush;
      status = system (command.c_str());
    }
}
}// namespace rheolef
