//
// SessionImpl.cpp
//
// $Id: //poco/1.3/Data/SQLite/src/SessionImpl.cpp#9 $
//
// Library: Data/SQLite
// Package: SQLite
// Module:  SessionImpl
//
// Copyright (c) 2006, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#include "Poco/Data/SQLite/SessionImpl.h"
#include "Poco/Data/SQLite/Utility.h"
#include "Poco/Data/SQLite/SQLiteStatementImpl.h"
#if defined(POCO_UNBUNDLED)
#include <sqlite3.h>
#else
#include "sqlite3.h"
#endif
#include <cstdlib>


namespace Poco {
namespace Data {
namespace SQLite {


const std::string SessionImpl::BEGIN_TRANSACTION("BEGIN ");
const std::string SessionImpl::COMMIT_TRANSACTION("COMMIT");
const std::string SessionImpl::ABORT_TRANSACTION("ROLLBACK");


SessionImpl::SessionImpl(const std::string& fileName):
	_dbFileName(fileName),
	_pDB(0),
	_transactionMode("DEFERRED"),
	_maxRetryAttempts(DEFAULT_MAX_RETRY_ATTEMPTS),
	_minRetrySleep(DEFAULT_MIN_RETRY_SLEEP),
	_maxRetrySleep(DEFAULT_MAX_RETRY_SLEEP),
	_connected(false)
{
	addProperty("transactionMode", &SessionImpl::setTransactionMode, &SessionImpl::getTransactionMode);
	addProperty("maxRetryAttempts", &SessionImpl::setMaxRetryAttempts, &SessionImpl::getMaxRetryAttempts);
	addProperty("minRetrySleep", &SessionImpl::setMinRetrySleep, &SessionImpl::getMinRetrySleep);
	addProperty("maxRetrySleep", &SessionImpl::setMaxRetrySleep, &SessionImpl::getMaxRetrySleep);
	open();
}


SessionImpl::~SessionImpl()
{
	close();
}


Poco::Data::StatementImpl* SessionImpl::createStatementImpl()
{
	poco_check_ptr (_pDB);
	return new SQLiteStatementImpl(_pDB, _maxRetryAttempts, _minRetrySleep, _maxRetrySleep);
}


void SessionImpl::begin()
{
	SQLiteStatementImpl tmp(_pDB, _maxRetryAttempts, _minRetrySleep, _maxRetrySleep);
	tmp.add(BEGIN_TRANSACTION + _transactionMode);
	tmp.execute();
}


void SessionImpl::commit()
{
	SQLiteStatementImpl tmp(_pDB, _maxRetryAttempts, _minRetrySleep, _maxRetrySleep);
	tmp.add(COMMIT_TRANSACTION);
	tmp.execute();
}


void SessionImpl::rollback()
{
	SQLiteStatementImpl tmp(_pDB, _maxRetryAttempts, _minRetrySleep, _maxRetrySleep);
	tmp.add(ABORT_TRANSACTION);
	tmp.execute();
}


void SessionImpl::open()
{
	int rc = sqlite3_open(_dbFileName.c_str(), &_pDB);

	if (rc != 0)
	{
		close();
		Utility::throwException(rc);
	}

	sqlite3_extended_result_codes(_pDB, 1);
	_connected = true;
}


void SessionImpl::close()
{
	if (_pDB)
	{
		sqlite3_close(_pDB);
		_pDB = 0;
	}

	_connected = false;
}


bool SessionImpl::isConnected()
{
	return _connected;
}


bool SessionImpl::isTransaction()
{
	return sqlite3_get_autocommit(_pDB) == 0;
}


void SessionImpl::setTransactionMode(const std::string& prop, const Poco::Any& value)
{
	_transactionMode = Poco::RefAnyCast<std::string>(value);
}


Poco::Any SessionImpl::getTransactionMode(const std::string& prop)
{
	return Poco::Any(_transactionMode);
}


void SessionImpl::setMaxRetryAttempts(const std::string& prop, const Poco::Any& value)
{
	int maxRetryAttempts = Poco::RefAnyCast<int>(value);
	if (maxRetryAttempts < 0) throw Poco::InvalidArgumentException("maxRetryAttempts must be >= 0");
	
	_maxRetryAttempts = maxRetryAttempts;
}


Poco::Any SessionImpl::getMaxRetryAttempts(const std::string& prop)
{
	return Poco::Any(_maxRetryAttempts);
}


void SessionImpl::setMinRetrySleep(const std::string& prop, const Poco::Any& value)
{
	int minRetrySleep = Poco::RefAnyCast<int>(value);
	if (minRetrySleep < 0 || minRetrySleep > _maxRetrySleep) throw Poco::InvalidArgumentException("minRetrySleep must be >= 0 and <= maxRetrySleep");

	_minRetrySleep = minRetrySleep;
}


Poco::Any SessionImpl::getMinRetrySleep(const std::string& prop)
{
	return Poco::Any(_minRetrySleep);
}


void SessionImpl::setMaxRetrySleep(const std::string& prop, const Poco::Any& value)
{
	int maxRetrySleep = Poco::RefAnyCast<int>(value);
	if (maxRetrySleep < _minRetrySleep) throw Poco::InvalidArgumentException("maxRetrySleep must be >= minRetrySleep");

	_maxRetrySleep = maxRetrySleep;
}


Poco::Any SessionImpl::getMaxRetrySleep(const std::string& prop)
{
	return Poco::Any(_maxRetrySleep);
}


} } } // namespace Poco::Data::SQLite
