/* eart2000.cpp: functions for computing Earth coordinates

Copyright (C) 2010, Project Pluto

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
02110-1301, USA.    */

#include <math.h>

/* EART2000.CPP

   This computes a heliocentric J2000 position of the Earth,  using
the VSOP theory.  I don't use it very often,  since the VSOP theory
for all the planets is given in the file VSOP.BIN on the Guide CD-ROM.
(This is the truncated VSOP used in Meeus' _Astronomical Algorithms_.
The full VSOP is in BIG_VSOP.BIN.)

   The input is a time in millennia from J2000 = JD 2451545.  The
output fills an array of six doubles with the X, Y, Z,  ecliptic
longitude,  ecliptic latitude,  and distance,  all in J2000 coords. */

int get_earth_loc( const double t_millennia, double *results);

static const int count[18] = {64, 34, 20, 7, 2, 0,       /* 127 */
                         5,  7,  4, 3, 2, 0,       /* +21 = 148 */
                        40, 10,  6, 2, 1, 0};      /* +59 = 207 */


static const double coeffs[207 * 3] = {
      175347046.,      0.000000000,        0.000000000,
        3341656.,      4.669256804,     6283.075849991,
          34894.,      4.626102422,    12566.151699983,
           3497.,      2.744117834,     5753.384884897,
           3418.,      2.828865798,        3.523118349,
           3136.,      3.627670418,    77713.771468121,
           2676.,      4.418083454,     7860.419392439,
           2343.,      6.135162144,     3930.209696220,
           1324.,      0.742463417,    11506.769769794,
           1273.,      2.037096579,      529.690965095,
           1199.,      1.109629462,     1577.343542448,
            990.,      5.232680721,     5884.926846583,
            902.,      2.045054465,       26.298319800,
            857.,      3.508491523,      398.149003408,
            780.,      1.178826820,     5223.693919802,
            753.,      2.533390528,     5507.553238667,
            505.,      4.582926000,    18849.227549974,
            492.,      4.205057118,      775.522611324,
            357.,      2.919541145,        0.067310303,
            317.,      5.849019485,    11790.629088659,
            284.,      1.898692409,      796.298006816,
            271.,      0.314862554,    10977.078804699,
            243.,      0.344814459,     5486.777843175,
            206.,      4.806466315,     2544.314419883,
            205.,      1.869537703,     5573.142801433,
            202.,      2.457677902,     6069.776754553,
            156.,      0.833060846,      213.299095438,
            132.,      3.411182927,     2942.463423292,
            126.,      1.082954595,       20.775395492,
            115.,      0.645449117,        0.980321068,
            103.,      0.635998456,     4694.002954708,
            102.,      0.975692803,    15720.838784878,
            102.,      4.266798020,        7.113547001,
             99.,      6.209929269,     2146.165416475,
             98.,      0.681013424,      155.420399434,
             86.,      5.983226313,   161000.685737674,
             85.,      1.298707648,     6275.962302991,
             85.,      3.670800930,    71430.695618129,
             80.,      1.807912871,    17260.154654690,
             79.,      3.036974587,    12036.460734888,
             75.,      1.755089133,     5088.628839767,
             74.,      3.503194150,     3154.687084896,
             74.,      4.679266339,      801.820931124,
             70.,      0.832976214,     9437.762934887,
             62.,      3.977639128,     8827.390269875,
             61.,      1.818398930,     7084.896781115,
             57.,      2.784304586,     6286.598968340,
             56.,      4.386948654,    14143.495242431,
             56.,      3.470060599,     6279.552731642,
             52.,      0.189149472,    12139.553509107,
             52.,      1.332827399,     1748.016413067,
             51.,      0.283068329,     5856.477659115,
             49.,      0.487350142,     1194.447010225,
             41.,      5.368175929,     8429.241266467,
             41.,      2.398509387,    19651.048481098,
             39.,      6.168330210,    10447.387839604,
             37.,      6.041338632,    10213.285546211,
             37.,      2.569574818,     1059.381930189,
             36.,      1.708758088,     2352.866153772,
             36.,      1.775968892,     6812.766815086,
             33.,      0.593102786,    17789.845619785,
             30.,      0.442944642,    83996.847318112,
             30.,      2.739751241,     1349.867409659,
             25.,      3.164708917,     4690.479836359,
   628307584999.,      0.000000000,        0.000000000,
         206059.,      2.678234558,     6283.075849991,
           4303.,      2.635122335,    12566.151699983,
            425.,      1.590469820,        3.523118349,
            119.,      5.795557656,       26.298319800,
            109.,      2.966310107,     1577.343542448,
             93.,      2.592111095,    18849.227549974,
             72.,      1.138405812,      529.690965095,
             68.,      1.874533003,      398.149003408,
             67.,      4.409328320,     5507.553238667,
             59.,      2.888157906,     5223.693919802,
             56.,      2.174717400,      155.420399434,
             45.,      0.397995029,      796.298006816,
             36.,      0.468754372,      775.522611324,
             29.,      2.647322546,        7.113547001,
             21.,      5.341382751,        0.980321068,
             19.,      1.846283760,     5486.777843175,
             19.,      4.968551795,      213.299095438,
             17.,      2.991167606,     6275.962302991,
             16.,      0.032165873,     2544.314419883,
             16.,      1.430493013,     2146.165416475,
             15.,      1.204697937,    10977.078804699,
             12.,      2.834322821,     1748.016413067,
             12.,      3.258050820,     5088.628839767,
             12.,      5.273797604,     1194.447010225,
             12.,      2.075020801,     4694.002954708,
             11.,      0.766147230,      553.569402842,
             10.,      1.302634234,     6286.598968340,
             10.,      4.239258653,     1349.867409659,
              9.,      2.699568270,      242.728603974,
              9.,      5.644760860,      951.718406251,
              8.,      5.300561729,     2352.866153772,
              6.,      2.650345140,     9437.762934887,
              6.,      4.666337263,     4690.479836359,
           8722.,      1.072536356,     6283.075849991,
            991.,      3.141592654,        0.000000000,
            295.,      0.437173503,    12566.151699983,
             27.,      0.052956361,        3.523118349,
             16.,      5.188202157,       26.298319800,
             16.,      3.685047122,      155.420399434,
              9.,      0.296671147,    18849.227549974,
              9.,      2.057063196,    77713.771468121,
              7.,      0.826915410,      775.522611324,
              5.,      4.662432317,     1577.343542448,
              4.,      1.030670323,        7.113547001,
              4.,      3.440433695,     5573.142801433,
              3.,      5.140212246,      796.298006816,
              3.,      6.054793185,     5507.553238667,
              3.,      1.192400085,      242.728603974,
              3.,      6.117058654,      529.690965095,
              3.,      0.303632482,      398.149003408,
              3.,      2.279664343,      553.569402842,
              2.,      4.376661180,     5223.693919802,
              2.,      3.754350955,        0.980321068,
            289.,      5.841731497,     6283.075849991,
             21.,      6.049839390,    12566.151699983,
              3.,      5.195605796,      155.420399434,
              3.,      3.141592654,        0.000000000,
              1.,      4.721976120,        3.523118349,
              1.,      5.969048992,      242.728603974,
              1.,      5.541829032,    18849.227549974,
              8.,      4.141173214,     6283.075849991,
              1.,      3.275736442,    12566.151699983,
            280.,      3.198701560,    84334.661581308,
            102.,      5.422486193,     5507.553238667,
             80.,      3.880132045,     5223.693919802,
             44.,      3.704446898,     2352.866153772,
             32.,      4.000263698,     1577.343542448,
         227778.,      3.413766205,     6283.075849991,
           3806.,      3.370634238,    12566.151699983,
           3620.,      0.000000000,        0.000000000,
             72.,      3.327775497,    18849.227549974,
              8.,      3.891904036,     5507.553238667,
              8.,      1.794896072,     5223.693919802,
              6.,      5.197894248,     2352.866153772,
           9721.,      5.151928099,     6283.075849991,
            233.,      3.141592654,        0.000000000,
            134.,      0.644062130,    12566.151699983,
              7.,      1.073333978,    18849.227549974,
            276.,      0.594800971,     6283.075849991,
             17.,      3.141592654,        0.000000000,
              4.,      0.117505753,    12566.151699983,
              6.,      2.267340298,     6283.075849991,
              1.,      0.000000000,        0.000000000,
      100013989.,      0.000000000,        0.000000000,
        1670700.,      3.098463503,     6283.075849991,
          13956.,      3.055246095,    12566.151699983,
           3084.,      5.198466744,    77713.771468121,
           1628.,      1.173875581,     5753.384884897,
           1576.,      2.846852149,     7860.419392439,
            925.,      5.452922367,    11506.769769794,
            542.,      4.564091515,     3930.209696220,
            472.,      3.661000221,     5884.926846583,
            346.,      0.963686273,     5507.553238667,
            329.,      5.899836861,     5223.693919802,
            307.,      0.298671395,     5573.142801433,
            243.,      4.273495308,    11790.629088659,
            212.,      5.847144613,     1577.343542448,
            186.,      5.021997107,    10977.078804699,
            175.,      3.011936367,    18849.227549974,
            110.,      5.055106359,     5486.777843175,
             98.,      0.886813113,     6069.776754553,
             87.,      5.689564189,    15720.838784878,
             86.,      1.270791253,   161000.685737674,
             65.,      0.272513414,    17260.154654690,
             63.,      0.921770540,      529.690965095,
             57.,      2.013742922,    83996.847318112,
             56.,      5.241597992,    71430.695618129,
             49.,      3.245012404,     2544.314419883,
             47.,      2.577998532,      775.522611324,
             45.,      5.537156638,     9437.762934887,
             43.,      6.011102580,     6275.962302991,
             39.,      5.360638329,     4694.002954708,
             38.,      2.392553440,     8827.390269875,
             37.,      0.829612818,    19651.048481098,
             37.,      4.901075873,    12139.553509107,
             36.,      1.674471358,    12036.460734888,
             35.,      1.842706933,     2942.463423292,
             33.,      0.243702217,     7084.896781115,
             32.,      0.183682999,     5088.628839767,
             32.,      1.777756421,      398.149003408,
             28.,      1.213448875,     6286.598968340,
             28.,      1.899344278,     6279.552731642,
             26.,      4.588968631,    10447.387839604,
         103019.,      1.107489682,     6283.075849991,
           1721.,      1.064423004,    12566.151699983,
            702.,      3.141592654,        0.000000000,
             32.,      1.021685833,    18849.227549974,
             31.,      2.843584440,     5507.553238667,
             25.,      1.319065703,     5223.693919802,
             18.,      1.424287091,     1577.343542448,
             10.,      5.913852484,    10977.078804699,
              9.,      1.420468544,     6275.962302991,
              9.,      0.271581929,     5486.777843175,
           4359.,      5.784551338,     6283.075849991,
            124.,      5.579354280,    12566.151699983,
             12.,      3.141592654,        0.000000000,
              9.,      3.627778931,    77713.771468121,
              6.,      1.869589051,     5573.142801433,
              3.,      5.470348797,    18849.227549974,
            145.,      4.273194339,     6283.075849991,
              7.,      3.917062617,    12566.151699983,
              4.,      2.563890163,     6283.075849991 };

int get_earth_loc( const double t_millennia, double *results)
{
   int i, j, k;
   double const *tptr = coeffs;
   double power, sum;
   double val[3];

   for( i = 0; i < 3; i++)
      {
      val[i] = 0;
      power = 1.;
      for( j = 0; j < 6; j++)
         {
         sum = 0.;
         for( k = 0; k < count[j + i * 6]; k++, tptr += 3)
            sum += tptr[0] * cos( tptr[1] + tptr[2] * t_millennia);
         val[i] += sum * power;
         power *= t_millennia;
         }
      val[i] *= 1.e-8;
      }
   *results++ = val[2] * cos( val[0]) * cos( val[1]);
   *results++ = val[2] * sin( val[0]) * cos( val[1]);
   *results++ = val[2] *                sin( val[1]);
   *results++ = val[0];
   *results++ = val[1];
   *results++ = val[2];
   return( 0);
}
