/*	test_Configuration

PIRL CVS ID: test_Configuration.java,v 1.9 2012/04/16 06:05:20 castalia Exp

Copyright (C) 2003-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

import java.net.URL;
import java.util.*;
import java.io.*;
import PIRL.PVL.*;
import PIRL.Configuration.Configuration;
import PIRL.Configuration.Configuration_Exception;

import	PIRL.Utilities.Checker;


public class test_Configuration
{

/**	The name of the test configuration file we'll be using.

	This file will be created during the testing, in the directory from
	which the testing is conducted.
*/
private static final String
	CONFIGURATION_FILENAME = "test_Configuration.conf";


public static void main
	(
	String[] arguments
	)
{
System.out.println
	("*** test_Configuration: " + Configuration.ID);

Configuration
	config_A = null,
	config_B = null;
String
	expected,
	path,
	value;
boolean
	condition;
int
	status = 0;
Checker
	checker = new Checker ();

if (arguments.length > 0 &&
	arguments[0].startsWith ("-v"))
	checker.Verbose = true;

try
	{
	//	Constructors:
	System.out.println
		("=== Constructors:");

	//	Construct from a Parameter.
	String
		config_A_list =
			"P0 = V0\n" +
			"absolute_link = /G0/G1\n" +
			"relative_link = G1\n" +
			"GROUP = G0\n" +
			"	P1a = V1a\n" +
			"	P1b = V1b\n" +
			"	P1c = V1c\n" +
			"	GROUP = G1\n" +
			"		P2 = V2\n" +
			"	END_GROUP\n" +
			"END_GROUP\n" +
			"END\n";
	Parameter
		parameter = new Parameter
			(
			"Test_Configuration",
			(Parser)(new Parser (config_A_list)
				.Strict (false)
				.Crosshatch_Comments (true)
				.Filter_Input (false))
			);
	config_A = new Configuration (parameter);
	checker.Check
		("new Configuration (Parameter)",
		config_A_list, Trimmed_Description (config_A));

	//	Copy the Configuration; change the case sensitivity.
	condition = ! config_A.Case_Sensitive ();
	config_A.Case_Sensitive (condition);
	config_B = new Configuration (config_A);
	checker.Check
		("new Configuration (Configuration)",
		config_A_list, Trimmed_Description (config_B));
	checker.Check
		("Case sensitivity copied",
		condition, config_B.Case_Sensitive ());

	//	Construct from a file.
	File
		file = new File (CONFIGURATION_FILENAME);
	FileWriter
		filewriter = new FileWriter (file);
	filewriter.write (config_A_list);
	filewriter.close ();
	config_A = new Configuration (CONFIGURATION_FILENAME);
	checker.Check
		("new Configuration (\"" + CONFIGURATION_FILENAME + "\")",
		config_A_list, Trimmed_Description (config_A));

	//	Construct from a class relative system resource.
	File
		moved_file =
			new File (".." + File.separator + CONFIGURATION_FILENAME);
	if (file.renameTo (moved_file))
		{
		Configuration.Relative_to_Class (Configuration.class);
		config_A = new Configuration (CONFIGURATION_FILENAME);
		checker.Check
			("new Configuration (\"" + CONFIGURATION_FILENAME
				+ "\"), Relative_to_Class " + Configuration.class,
			config_A_list, Trimmed_Description (config_A));

		Configuration.Relative_to_Class (null);
		moved_file.delete ();
		filewriter = new FileWriter (file);
		filewriter.write (config_A_list);
		filewriter.close ();
		}

	//	Construct from a URL.
	URL
		url = new URL ("file:" + CONFIGURATION_FILENAME);
	config_A = new Configuration (url);
	checker.Check
		("new Configuration (URL), URL: " + url,
		config_A_list, Trimmed_Description (config_A));

	//	Set default filename and construct from it.
	Configuration.Default_Filename (CONFIGURATION_FILENAME);
	config_A = new Configuration ((String)null);
	checker.Check
		("new Configuration (null), Default_Filename: " + Configuration.Default_Filename (),
		config_A_list, Trimmed_Description (config_A));

	//	Construct from an InputStream.
	String
		config_B_list =
			"Type = MySQL\n" +
			"Database = test\n" +
			"Group = MySQL\n" +
			"	Database = MOC\n" +
			"	Host = PIRL.LPL.Arizona.edu\n" +
			"	Group = MOC\n" +
			"		password = MOC_password\n" +
			"		Group = MGSC\n" +
			"			key = third\n" +
			"		End_Group\n" +
			"	End_Group\n" +
			"	/* Alias (link) name for the MOC group */\n" +
			"	Mars = MOC\n" +
			"End_Group\n" +
			"\n" +
			"Group = Text\n" +
			"	Database = MGSC\n" +
			"end_group\n" +
			"group = mysql\n" +
			"	password = foobar\n" +
			"	group = moc\n" +
			"		group = MGSC\n" +
			"			fields = (first, second, third, fourth)\n" +
			"			ID = product_ID\n" +
			"			Primary_Selection = fields\n" +
			"		end_group\n" +
			"		table = MGSC\n" +
			"	end_group\n" +
			"end_group\n" +
			"password = barfoo\n" +
			"Name = MySQL/MOC/MGSC/ID\n" +
			"END\n";
	if (checker.Verbose)
		System.out.print
			("=== InputStream content:\n"
			+ config_B_list);
	StringBufferInputStream
		input_stream = new StringBufferInputStream (config_B_list);
	config_B = new Configuration (input_stream);
	//	After Coalesce:
	config_B_list =
		"Type = MySQL\n" +
		"Database = test\n" +
		"password = barfoo\n" +
		"Name = MySQL/MOC/MGSC/ID\n" +
		"GROUP = MySQL\n" +
		"	Database = MOC\n" +
		"	Mars = MOC\n" +
		"	password = foobar\n" +
		"	GROUP = MOC\n" +
		"		password = MOC_password\n" +
		"		table = MGSC\n" +
		"		GROUP = MGSC\n" +
		"			key = third\n" +
		"			fields = \n" +
		"				(first, second, third, fourth)\n" +
		"			ID = product_ID\n" +
		"			Primary_Selection = fields\n" +
		"		END_GROUP\n" +
		"	END_GROUP\n" +
		"END_GROUP\n" +
		"GROUP = Text\n" +
		"	Database = MGSC\n" +
		"END_GROUP\n" +
		"END\n";
	checker.Check
		("new Configuration (InputStream)",
		config_B_list, Trimmed_Description (config_B));


	//	Get:
	System.out.println
		("=== Get:");

	path = "/mysql/moc/mgsc/key";
	config_B.Case_Sensitive (true);
	expected = null;
	checker.Check
		("Case_Sensitive (true); Get_One (\"" + path + "\")",
		expected, config_B.Get_One (path));

	config_B.Case_Sensitive (false);
	expected = "third";
	checker.Check
		("Case_Sensitive (false); Get_One (\"" + path + "\")",
		expected, config_B.Get_One (path));

	path = "/mysql/moc/mgsc/Password";
	expected = "MOC_password";
	checker.Check
		("Get_One (\"" + path + "\")",
		expected, config_B.Get_One (path));

	path = "/mysql/moc/mgsc/type";
	expected = "MySQL";
	checker.Check
		("Get_One (\"" + path + "\")",
		expected, config_B.Get_One (path));

	path = "/mysql/moc/mgsc/NO_SUCH_PARAMETER";
	expected = null;
	checker.Check
		("Get_One (\"" + path + "\")",
		expected, config_B.Get_One (path));


	//	Set:
	System.out.println
		("=== Set:");

	path = "/mysql/moc/mgsc/key";
	value = "SECOND";
	checker.Check
		("Set (\"" + path + "\", \"" + value + "\") replaced existing value",
		true, config_B.Set (path, value));
	checker.Check
		("Set (\"" + path + "\", \"" + value + "\") value confirmation",
		value, config_B.Get_One (path));

	path = "/mysql/moc/mgsc/INDEX";
	value = "Index";
	checker.Check
		("Set (\"" + path + "\", \" " + value + "\") replaced existing value",
		false, config_B.Set (path, value));
	checker.Check
		("Set (\"" + path + "\", \"" + value + "\") value confirmation",
		value, config_B.Get_One (path));

	path = "/mysql/moc/NEW/KEY";
	value = "Key";
	checker.Check
		("Set_Conditionally (\"" + path + "\", \" " + value + "\") new parameter",
		false, config_B.Set_Conditionally (path, value));
	checker.Check
		("Set_Conditionally (\"" + path + "\", \"" + value + "\") value confirmation",
		value, config_B.Get_One (path));
	expected = value;
	value = "KEY_REPLACED";
	checker.Check
		("Set_Conditionally (\"" + path + "\", \" " + value + "\") existing parameter",
		true, config_B.Set_Conditionally (path, value));
	checker.Check
		("Set_Conditionally (\"" + path + "\", \"" + value + "\") value confirmation",
		expected, config_B.Get_One (path));


	//	Links:
	System.out.println
		("=== Links:");

	expected = "product_ID";
	checker.Check
		("Get_Linked_One (\"ID\")",
		expected, config_B.Get_Linked_One ("ID"));

	checker.Check
		("Get_Linked_One (\"Name\")",
		expected, config_B.Get_Linked_One ("Name"));

	expected = "[first, second, third, fourth]";
	checker.Check
		("Get_Linked (\"Primary_Selection\")",
		expected, config_B.Get_Linked ("Primary_Selection").toString ());


	//	Group:
	System.out.println
		("=== Groups:");

	config_B.Add_Environment ();
	checker.Check
		("Environment parameters group",
		Configuration.ENVIRONMENT,
		config_B.Group_Pathname (Configuration.ENVIRONMENT));
	if (checker.Verbose)
		{
		Configuration
			environment = config_B.Group (Configuration.ENVIRONMENT);
		if (environment == null)
			System.out.println
				("No " + Configuration.ENVIRONMENT + " found.");
		else
			System.out.println
				(environment.Description ());
		}


	config_B_list =
		"password = MOC_password\n" +
		"table = MGSC\n" +
		"Alias = MOC\n" +
		"Database = MOC\n" +
		"Mars = MOC\n" +
		"Type = MySQL\n" +
		"Name = MySQL/MOC/MGSC/ID\n" +
		"GROUP = MGSC\n" +
		"	key = SECOND\n" +
		"	fields = \n" +
		"		(first, second, third, fourth)\n" +
		"	ID = product_ID\n" +
		"	Primary_Selection = fields\n" +
		"	INDEX = Index\n" +
		"END_GROUP\n" +
		"GROUP = NEW\n" +
		"	KEY = Key\n" +
		"END_GROUP\n" +
		"END\n";
	config_B = config_B.Group ("Mars");
	checker.Check
		("Group (\"Mars\")",
		config_B_list, Trimmed_Description (config_B));


	//	Groups & Links:
	System.out.println
		("=== Groups w/Links:");

	//	Group_Pathname.
	expected = "/G0/G1";
	checker.Check
		("Group_Pathname (\"G1\")",
		expected, config_A.Group_Pathname ("G1"));
	checker.Check
		("Group_Pathname (\"absolute_link\")",
		expected, config_A.Group_Pathname ("absolute_link"));
	checker.Check
		("Group_Pathname (\"relative_link\")",
		expected, config_A.Group_Pathname ("relative_link"));


	//	Include:
	System.out.println
		("=== Include:");

	Configuration.Include_Default = false;
	path = "@INCLUDE";
	value = CONFIGURATION_FILENAME;
	checker.Check
		("Set (\"" + path + "\", \"" + value + "\")",
		false, config_B.Set (path, value));
	checker.Check
		("Set (\"" + path + "\", \"" + value + "\") value confirmation",
		value, config_B.Get_One (path));
	//	Place the include at the end of the assignment list.
	config_B.Coalesce (); 
	//	Copy the template.
	config_A = new Configuration (config_B);
	config_B.Include ();
	config_B_list = config_B_list.replaceAll ("/ID\\n", 
		"/ID\n"+ config_A_list.replaceAll ("END\\n", ""));
	checker.Check
		("Include",
		config_B_list, Trimmed_Description (config_B));

	//	Resolved reference
	config_B = new Configuration (config_A);
	path = "@INCLUDE";
	value = "${INDEX}";
	checker.Check
		("Set (\"" + path + "\", \"" + value + "\")",
		true, config_B.Set (path, value));
	path = "INDEX";
	value = CONFIGURATION_FILENAME;
	checker.Check
		("Set (\"" + path + "\", \"" + value + "\")",
		true, config_B.Set (path, value));
	config_B.Include ();
	config_B_list = config_B_list.replaceAll ("Index", CONFIGURATION_FILENAME);
	checker.Check
		("Include with resolved reference",
		config_B_list, Trimmed_Description (config_B));

	//	Recursive include.
	config_A_list = "@INCLUDE = " + CONFIGURATION_FILENAME + '\n'
		+ config_A_list;
	filewriter = new FileWriter (new File (CONFIGURATION_FILENAME));
	filewriter.write (config_A_list);
	filewriter.close ();
	try
		{
		config_A.Include ();
		expected = "Configuration_Exception - recursive include";
		value = Trimmed_Description (config_A);
		}
	catch (Configuration_Exception exception)
		{
		expected = exception.getMessage ();
		value = expected;
		}
	checker.Check
		("Recursive include exception",
		expected, value);
	}

catch (Exception exception)
	{
	status = 1;
	System.out.println
		("\n!!! Unexpected exception:\n"
		+ exception + '\n'
		+ exception.getMessage ());
	}

if (status == 0 &&
	checker.Checks_Total == checker.Checks_Passed)
	new File (CONFIGURATION_FILENAME).delete ();

System.out.println ("\n" +
	"Checks: " + checker.Checks_Total + '\n' +
	"Passed: " + checker.Checks_Passed);

System.exit (status);
}

/**	Trims a Configuration listing of boilerplate.
<p>
	The Configuration has its default "User", "Host", and "Classpath"
	parameters removed and its Name is changed to the
	Parser.CONTAINER_NAME so the enclosing Group will not be included
	in the listing. A Listing of the Configuration is written to a
	String. Then the original Name is restored.
<p>
	@param	configuration	The Configuration to be listed.
	@return	The String containing the listing. If an Exception occurs
		while generating the listing, the exception message will be
		returned. If the lister encounters a Warning condition, the
		warning message will be appended to the listing. If the configuration
		is null, null will be returned.
*/
private static String Trimmed_Description
	(
	Configuration configuration
	)
{
if (configuration == null)
	return null;

//	Remove DEFAULTS.
configuration.Remove ("User");
configuration.Remove ("Host");
configuration.Remove ("Classpath");
//	Remove Environment variables.
configuration.Remove_Group (Configuration.ENVIRONMENT);

StringWriter
	description = new StringWriter ();
Lister
	lister = new Lister (description).Strict (false);
//	Prevent the top level from being written.
String
	name = configuration.Name ();
configuration.Name (Parser.CONTAINER_NAME);
try {lister.Write (configuration);}
catch (Exception exception)
	{
	//	Restore the original name.
	configuration.Name (name);
	return exception.getMessage ();
	}
//	Restore the original name.
configuration.Name (name);

if (lister.Warning () != null)
	description.getBuffer ()
		.append ("\nWarning:\n")
		.append (lister.Warning ().getMessage ());

return description.toString ();
}

}
