/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment unit test for pgmimage.c
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


#include <tests/check/common/pgmcheck.h>
#include <pgm/pgm.h>
#include <string.h>
#include <stdlib.h>
#include <gdk-pixbuf/gdk-pixbuf.h>

/* Fixture stuff */

#define TEST_IMAGE_FILE "./test-image.png"
#define TEST_IMAGE_SMALL_FILE "./test-small-image.png"
#define UNEXISTING_FILE "/blah/foo/bar/notexistingfilethatdoesnotexist.png"


static void
write_image (const gchar *file_name, guint side_size)
{
  GdkPixbuf *pixbuf = NULL;
  guchar *data = malloc (side_size * side_size * 3);
  guint i = 0, rowstride = side_size*3;

  for (i=0; i < side_size * side_size * 3; i+=3)
    {
      data[i] = (i / 256) % 0xff;
      data[i+1] = i % 0xff;
      data[i+2] = (3 * i + 23) % 0xff;
    }

  pixbuf = gdk_pixbuf_new_from_data (data, GDK_COLORSPACE_RGB, FALSE,
                                     8, side_size, side_size, rowstride, NULL, NULL);
  gdk_pixbuf_save (pixbuf, file_name, "png", NULL, NULL);

  g_object_unref (pixbuf);
}

static void
test_image_unchecked_setup (void)
{
  write_image (TEST_IMAGE_SMALL_FILE, 16);
  write_image (TEST_IMAGE_FILE, 1024);
}

static void
test_image_unchecked_teardown (void)
{
  g_unlink (TEST_IMAGE_FILE);
  g_unlink (TEST_IMAGE_SMALL_FILE);
}

/* test image master/slave delete */
PGM_START_TEST (test_image_master_slave_del)
{
  PgmError ret;
  PgmImage *img1, *img2;

  /* create 2 floating images with refcount 1 */
  img1 = PGM_IMAGE (pgm_image_new ());
  ASSERT_OBJECT_REFCOUNT (img1, "img1", 1);
  img2 = PGM_IMAGE (pgm_image_new ());
  ASSERT_OBJECT_REFCOUNT (img2, "img2", 1);

  /* make img1 a slave of img2, refcounts should not change */
  ret = pgm_image_set_from_image (img1, img2);
  fail_if (ret != PGM_ERROR_OK, "image_set_from_image error");
  ASSERT_OBJECT_REFCOUNT (img1, "img1", 1);
  ASSERT_OBJECT_REFCOUNT (img2, "img2", 1);

  /* del master image, the refcount of the two images are still the same */
  ret = pgm_image_clear (img2);
  fail_if (ret != PGM_ERROR_OK, "image_clear error");
  ASSERT_OBJECT_REFCOUNT (img1, "img1", 1);
  ASSERT_OBJECT_REFCOUNT (img2, "img2", 1);

  /* stop the slavery */
  ret = pgm_image_clear (img1);
  fail_if (ret != PGM_ERROR_OK, "image_clear error");
  ASSERT_OBJECT_REFCOUNT (img1, "img1", 1);
  ASSERT_OBJECT_REFCOUNT (img2, "img2", 1);

  /* clean up our references */
  gst_object_unref (GST_OBJECT_CAST (img1));
  gst_object_unref (GST_OBJECT_CAST (img2));
}
PGM_END_TEST;


/* test default values */
PGM_START_TEST (test_image_default_values)
{
  PgmError ret;
  PgmImage *image;
  PgmImageAlignment alignment;
  PgmImageLayoutType layout;
  PgmImageInterpType interp;
  PgmImageStorageType storage;
  guint numerator, denominator;

  image = PGM_IMAGE (pgm_image_new ());

  /* alignment */
  ret = pgm_image_get_alignment (image, &alignment);
  fail_if (ret != PGM_ERROR_OK, "image_get_alignment error");
  fail_if (alignment != PGM_IMAGE_CENTER, "bad default alignment");

  /* aspect style */
  ret = pgm_image_get_layout (image, &layout);
  fail_if (ret != PGM_ERROR_OK, "image_get_layout error");
  fail_if (layout != PGM_IMAGE_SCALED, "bad default drawable style");

  /* interp */
  ret = pgm_image_get_interp (image, &interp);
  fail_if (ret != PGM_ERROR_OK, "image_get_interp error");
  fail_if (interp != PGM_IMAGE_BILINEAR, "bad default interp");

  /* aspect-ratio */
  ret = pgm_image_get_aspect_ratio (image, &numerator, &denominator);
  fail_if (ret != PGM_ERROR_OK, "image_get_aspect_ratio error");
  fail_if (numerator != 0 || denominator != 1, "bad default aspect-ratio");

  /* storage type */
  ret = pgm_image_get_storage_type (image, &storage);
  fail_if (ret != PGM_ERROR_OK, "image_get_storage_type error");
  fail_if (storage != PGM_IMAGE_EMPTY, "bad default storage type");

  gst_object_unref (GST_OBJECT_CAST (image));
}
PGM_END_TEST;


/* test set/get correctness */
PGM_START_TEST (test_image_set_get_correctness)
{
  PgmError ret;
  PgmImage *image;
  PgmImageAlignment alignment;
  PgmImageLayoutType layout;
  PgmImageInterpType interp;
  guint numerator, denominator;

  image = PGM_IMAGE (pgm_image_new ());

  /* alignment */
  ret = pgm_image_set_alignment (image, PGM_IMAGE_BOTTOM);
  fail_if (ret != PGM_ERROR_OK, "image_set_alignment error");
  ret = pgm_image_get_alignment (image, &alignment);
  fail_if (ret != PGM_ERROR_OK, "image_get_alignment error");
  fail_if (alignment != PGM_IMAGE_BOTTOM, "alignment not set");

  /* layout type */
  ret = pgm_image_set_layout (image, PGM_IMAGE_TILED);
  fail_if (ret != PGM_ERROR_OK, "image_set_layout error");
  ret = pgm_image_get_layout (image, &layout);
  fail_if (ret != PGM_ERROR_OK, "image_get_layout error");
  fail_if (layout != PGM_IMAGE_TILED, "laytout not set");

  /* interp */
  ret = pgm_image_set_interp (image, PGM_IMAGE_NEAREST);
  fail_if (ret != PGM_ERROR_OK, "image_set_interp error");
  ret = pgm_image_get_interp (image, &interp);
  fail_if (ret != PGM_ERROR_OK, "image_get_interp error");
  fail_if (interp != PGM_IMAGE_NEAREST, "interp not set");

  /* aspect-ratio */
  ret = pgm_image_set_aspect_ratio (image, 16, 9);
  fail_if (ret != PGM_ERROR_OK, "image_set_aspect_ratio error");
  ret = pgm_image_get_aspect_ratio (image, &numerator, &denominator);
  fail_if (ret != PGM_ERROR_OK, "image_get_aspect_ratio error");
  fail_if (numerator != 16 || denominator != 9, "aspect-ratio not set");

  gst_object_unref (GST_OBJECT_CAST (image));
}
PGM_END_TEST;

PGM_START_TEST (test_image_buffer)
{
  PgmImage *image = NULL;
  PgmImageStorageType storage = PGM_IMAGE_EMPTY;
  guchar image_data[192] = {
    0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc,
    0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc,
    0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc,
    0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc,
    0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc,
    0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc,
    0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc,
    0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc,
    0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc,
    0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc,
    0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc,
    0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc,
    0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc,
    0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc,
    0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc,
    0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc
  };

  image = PGM_IMAGE (pgm_image_new_from_buffer (PGM_IMAGE_RGB,
                                                8, 8, 24, 192,
                                                image_data));
  fail_if (NULL == image, "pgm_image_new_from_buffer failed");

  ASSERT_PGM_SUCCESS (pgm_image_get_storage_type (image, &storage));
  fail_unless (PGM_IMAGE_BUFFER == storage,
               "storage type should be PGM_IMAGE_BUFFER");

  ASSERT_PGM_SUCCESS (pgm_image_clear (image));

  ASSERT_PGM_SUCCESS (pgm_image_get_storage_type (image, &storage));
  fail_unless (PGM_IMAGE_EMPTY == storage,
               "storage type should be PGM_IMAGE_EMPTY");

  ASSERT_PGM_ERROR_WITH_CRITICAL
    (pgm_image_set_from_buffer (NULL, PGM_IMAGE_RGB, 8, 8, 24, 192,
                                image_data));
  ASSERT_PGM_SUCCESS (pgm_image_get_storage_type (image, &storage));
  fail_unless (PGM_IMAGE_EMPTY == storage,
               "storage type should be PGM_IMAGE_EMPTY");
  ASSERT_PGM_ERROR_WITH_CRITICAL
    (pgm_image_set_from_buffer (image, PGM_IMAGE_RGB, 8, 8, 24, 192, NULL));
  ASSERT_PGM_SUCCESS (pgm_image_get_storage_type (image, &storage));
  fail_unless (PGM_IMAGE_EMPTY == storage,
               "storage type should be PGM_IMAGE_EMPTY");

  ASSERT_PGM_SUCCESS
    (pgm_image_set_from_buffer (image, PGM_IMAGE_RGB, 8, 8, 24, 192,
                                image_data));
  ASSERT_PGM_SUCCESS (pgm_image_get_storage_type (image, &storage));
  fail_unless (PGM_IMAGE_BUFFER == storage,
               "storage type should be PGM_IMAGE_BUFFER");

  gst_object_unref (image);
}
PGM_END_TEST

PGM_START_TEST (test_image_from_file)
{
  PgmImage *image = NULL;
  PgmImageStorageType storage = PGM_IMAGE_EMPTY;

  image = PGM_IMAGE (pgm_image_new_from_file (NULL, 0));
  fail_if (NULL != image, "created an image for NULL filename");

  image = PGM_IMAGE (pgm_image_new_from_file (TEST_IMAGE_FILE, 0));
  fail_if (NULL == image, "image creation failed with right params");
  ASSERT_PGM_SUCCESS (pgm_image_get_storage_type (image, &storage));

  ASSERT_PGM_SUCCESS (pgm_image_clear (image));
  ASSERT_PGM_SUCCESS (pgm_image_get_storage_type (image, &storage));
  fail_unless (PGM_IMAGE_EMPTY == storage,
               "storage type should be PGM_IMAGE_EMPTY");

  ASSERT_PGM_ERROR_WITH_CRITICAL
      (pgm_image_set_from_file (NULL, TEST_IMAGE_FILE, 0));
  ASSERT_PGM_SUCCESS (pgm_image_get_storage_type (image, &storage));
  fail_unless (PGM_IMAGE_EMPTY == storage,
               "storage type should be PGM_IMAGE_EMPTY");

  ASSERT_PGM_ERROR (pgm_image_set_from_file (image, NULL, 0));
  ASSERT_PGM_SUCCESS (pgm_image_get_storage_type (image, &storage));
  fail_unless (PGM_IMAGE_EMPTY == storage,
               "storage type should be PGM_IMAGE_EMPTY");

  ASSERT_PGM_SUCCESS (pgm_image_set_from_file (image, TEST_IMAGE_FILE, 0));

  gst_object_unref (image);
}
PGM_END_TEST

static gpointer
thread_init (gpointer data)
{
  pgm_main ();
}

PGM_START_TEST (test_image_dispose_while_loading)
{
  PgmDrawable *image = NULL;
  GThread *pgm_thread  = NULL;

  pgm_init (NULL, NULL);

  pgm_thread = g_thread_create ( thread_init, NULL, TRUE, NULL);
  fail_unless (NULL != pgm_thread, "thread creation");

  g_usleep (500000);

  image = pgm_image_new ();
  fail_unless (NULL != image, "image creation");

  ASSERT_PGM_SUCCESS (pgm_image_set_from_file ( PGM_IMAGE (image),
                                                TEST_IMAGE_FILE, 0));
  gst_object_unref (image);

  pgm_main_quit ();
  g_thread_join (pgm_thread);
  pgm_deinit ();
}
PGM_END_TEST

static void
image_loaded_cb (PgmImage *image, gpointer  user_data)
{
  gboolean *image_loaded = user_data;

  *image_loaded = TRUE;

  pgm_main_quit ();
}

PGM_START_TEST (test_image_small_image)
{
  PgmDrawable *image = NULL;
  GThread *pgm_thread  = NULL;
  gboolean image_loaded = FALSE;

  pgm_init (NULL, NULL);

  image = pgm_image_new ();
  fail_unless (NULL != image, "image creation");

  g_signal_connect (image, "file-loaded", G_CALLBACK (image_loaded_cb),
                    &image_loaded);

  ASSERT_PGM_SUCCESS (pgm_image_set_from_file (PGM_IMAGE (image),
                                               TEST_IMAGE_SMALL_FILE, 0));
  pgm_main ();

  fail_unless (image_loaded, "small image loading");

  gst_object_unref (image);
  pgm_deinit ();
}
PGM_END_TEST

PGM_START_TEST (test_image_gst_buffer)
{
  /* FIXME: write gst_buffer tests */
}
PGM_END_TEST

/* Fight against rounding issues using a threshold comparison of 1 */
#define IMAGE_FROM_DRAWABLE_TEST(x_drb,y_drb,x_img,y_img)               \
  G_STMT_START {                                                        \
    ASSERT_PGM_SUCCESS (pgm_image_from_drawable (image, &x, &y,         \
                                                 x_drb, y_drb));        \
    fail_unless (abs (x-x_img) <= 1);                                   \
    fail_unless (abs (y-y_img) <= 1);                                   \
  } G_STMT_END

#define IMAGE_FROM_DRAWABLE_3_TESTS(x1,y1,x2,y2,x3,y3) \
  G_STMT_START {                                       \
    IMAGE_FROM_DRAWABLE_TEST (0.0f, 0.0f, x1, y1);     \
    IMAGE_FROM_DRAWABLE_TEST (2.0f, 1.0f, x2, y2);     \
    IMAGE_FROM_DRAWABLE_TEST (-1.0f, -1.0f, x3, y3);   \
  } G_STMT_END

PGM_START_TEST (test_image_from_drawable)
{
  PgmImage *image;
  GThread *pgm_thread;
  gint x, y;
  guint old_n, old_d;

  pgm_init (NULL, NULL);

  pgm_thread = g_thread_create (thread_init, NULL, TRUE, NULL);
  fail_unless (NULL != pgm_thread, "thread creation");

  image = PGM_IMAGE (pgm_image_new_from_file (TEST_IMAGE_FILE, 0));
  fail_if (NULL == image, "failed creating image");

  g_usleep (500000); /* we hope the image will get loaded in that time */

  ASSERT_PGM_SUCCESS (pgm_drawable_set_size (PGM_DRAWABLE (image), 2.0f, 1.0f));

  /* filled */
  ASSERT_PGM_SUCCESS (pgm_image_set_layout (image, PGM_IMAGE_FILLED));

  IMAGE_FROM_DRAWABLE_TEST (0.0f, 0.0f, 0, 0);
  IMAGE_FROM_DRAWABLE_TEST (2.0f, 1.0f, 1024, 1024);
  IMAGE_FROM_DRAWABLE_TEST (1.9999f, 0.9999f, 1023, 1023);
  IMAGE_FROM_DRAWABLE_TEST (4.0f, 2.0f, 2048, 2048);
  IMAGE_FROM_DRAWABLE_TEST (1.0f, 0.75f, 512, 768);
  IMAGE_FROM_DRAWABLE_TEST (-1.0f, -1.0f, -512, -1024);

  /* we trust the stuff to be linear and from now on consider it enough to test
   * only for (0,0), (2,1) and (-1,-1) with IMAGE_FROM_DRAWABLE_3_TESTS() */

  /* scaled */
  ASSERT_PGM_SUCCESS (pgm_image_set_layout (image, PGM_IMAGE_SCALED));

  ASSERT_PGM_SUCCESS (pgm_image_set_alignment (image, PGM_IMAGE_LEFT));
  IMAGE_FROM_DRAWABLE_3_TESTS (0, 0, 2048, 1024, -1024, -1024);

  ASSERT_PGM_SUCCESS (pgm_image_set_alignment (image, PGM_IMAGE_CENTER));
  IMAGE_FROM_DRAWABLE_3_TESTS (-512, 0, 1536, 1024, -1536, -1024);

  ASSERT_PGM_SUCCESS (pgm_image_set_alignment (image, PGM_IMAGE_RIGHT));
  IMAGE_FROM_DRAWABLE_3_TESTS (-1024, 0, 1024, 1024, -2048, -1024);

  ASSERT_PGM_SUCCESS (pgm_drawable_set_size (PGM_DRAWABLE (image), 1.0f, 2.0f));

  ASSERT_PGM_SUCCESS (pgm_image_set_alignment (image, PGM_IMAGE_TOP));
  IMAGE_FROM_DRAWABLE_3_TESTS (0, 0, 2048, 1024, -1024, -1024);

  ASSERT_PGM_SUCCESS (pgm_image_set_alignment (image, PGM_IMAGE_CENTER));
  IMAGE_FROM_DRAWABLE_3_TESTS (0, -512, 2048, 512, -1024, -1536);

  ASSERT_PGM_SUCCESS (pgm_image_set_alignment (image, PGM_IMAGE_BOTTOM));
  IMAGE_FROM_DRAWABLE_3_TESTS (0, -1024, 2048, 0, -1024, -2048);

  /* zoomed */
  ASSERT_PGM_SUCCESS (pgm_image_set_layout (image, PGM_IMAGE_ZOOMED));
  ASSERT_PGM_SUCCESS (pgm_drawable_set_size (PGM_DRAWABLE (image), 2.0f, 1.0f));

  ASSERT_PGM_SUCCESS (pgm_image_set_alignment (image, PGM_IMAGE_TOP));
  IMAGE_FROM_DRAWABLE_3_TESTS (0, 0, 1024, 512, -512, -512);

  ASSERT_PGM_SUCCESS (pgm_image_set_alignment (image, PGM_IMAGE_CENTER));
  IMAGE_FROM_DRAWABLE_3_TESTS (0, 256, 1024, 768, -512, -256);

  ASSERT_PGM_SUCCESS (pgm_image_set_alignment (image, PGM_IMAGE_BOTTOM));
  IMAGE_FROM_DRAWABLE_3_TESTS (0, 512, 1024, 1024, -512, 0);

  ASSERT_PGM_SUCCESS (pgm_drawable_set_size (PGM_DRAWABLE (image), 1.0f, 2.0f));

  ASSERT_PGM_SUCCESS (pgm_image_set_alignment (image, PGM_IMAGE_LEFT));
  IMAGE_FROM_DRAWABLE_3_TESTS (0, 0, 1024, 512, -512, -512);

  ASSERT_PGM_SUCCESS (pgm_image_set_alignment (image, PGM_IMAGE_CENTER));
  IMAGE_FROM_DRAWABLE_3_TESTS (256, 0, 1280, 512, -256, -512);

  ASSERT_PGM_SUCCESS (pgm_image_set_alignment (image, PGM_IMAGE_RIGHT));
  IMAGE_FROM_DRAWABLE_3_TESTS (512, 0, 1536, 512, 0, -512);

  /* and now a few tests with changed aspect ratio */
  ASSERT_PGM_SUCCESS (pgm_image_get_aspect_ratio (image, &old_n, &old_d));
  ASSERT_PGM_SUCCESS (pgm_image_set_aspect_ratio (image, 3, 2));
  ASSERT_PGM_SUCCESS (pgm_drawable_set_size (PGM_DRAWABLE (image), 2.0f, 1.0f));
  ASSERT_PGM_SUCCESS (pgm_image_set_layout (image, PGM_IMAGE_SCALED));
  ASSERT_PGM_SUCCESS (pgm_image_set_alignment (image, PGM_IMAGE_RIGHT));

  IMAGE_FROM_DRAWABLE_3_TESTS (-341, 0, 1024, 1024, -1024, -1024);

  ASSERT_PGM_SUCCESS (pgm_image_set_aspect_ratio (image, old_n, old_d));

  gst_object_unref (image);

  pgm_main_quit ();
  g_thread_join (pgm_thread);
  pgm_deinit ();
}
PGM_END_TEST

PGM_START_TEST (test_image_from_drawable_fail)
{
  PgmImage *image;
  gint x, y;

  image = PGM_IMAGE (pgm_image_new_from_file (TEST_IMAGE_FILE, 0));
  fail_if (NULL == image, "image creation failed");

  ASSERT_PGM_ERROR_WITH_CRITICAL (
    pgm_image_from_drawable (NULL, &x, &y, 0.0f, 0.0f));

  ASSERT_PGM_ERROR_WITH_CRITICAL (
    pgm_image_from_drawable (image, NULL, &y, 0.0f, 0.0f));

  ASSERT_PGM_ERROR_WITH_CRITICAL (
    pgm_image_from_drawable (image, &x, NULL, 0.0f, 0.0f));

  gst_object_unref (image);
}
PGM_END_TEST

#define IMAGE_TO_DRAWABLE_TEST(x_image, y_image, x_drawable, y_drawable) \
  do { \
    ASSERT_PGM_SUCCESS (pgm_image_to_drawable (image, &x, &y, \
                                                 x_image, y_image)); \
    fail_unless_similar_float (x, x_drawable); \
    fail_unless_similar_float (y, y_drawable); \
  }while(0)

#define IMAGE_TO_DRAWABLE_3_TESTS(x1,y1,x2,y2,x3,y3) \
    do { \
      IMAGE_TO_DRAWABLE_TEST (0, 0, x1, y1); \
      IMAGE_TO_DRAWABLE_TEST (1024, 1024, x2, y2); \
      IMAGE_TO_DRAWABLE_TEST (-1024, -512, x3, y3); \
    }while(0)


PGM_START_TEST (test_image_to_drawable)
{
  PgmImage *image;
  GThread *pgm_thread;
  gfloat x, y;
  guint old_n, old_d;

  pgm_init (NULL, NULL);

  pgm_thread = g_thread_create (thread_init, NULL, TRUE, NULL);
  fail_unless (NULL != pgm_thread, "thread creation");

  image = PGM_IMAGE (pgm_image_new_from_file (TEST_IMAGE_FILE, 0));
  fail_if (NULL == image, "failed creating image");

  g_usleep (500000); /* we hope the image will get loaded in that time */

  ASSERT_PGM_SUCCESS (pgm_drawable_set_size (PGM_DRAWABLE (image), 2.0f, 1.0f));

  /* filled */
  ASSERT_PGM_SUCCESS (pgm_image_set_layout (image, PGM_IMAGE_FILLED));
  IMAGE_TO_DRAWABLE_3_TESTS (0.0f, 0.0f, 2.0f, 1.0f, -2.0f, -0.5f);

  /* scaled */
  ASSERT_PGM_SUCCESS (pgm_image_set_layout (image, PGM_IMAGE_SCALED));

  ASSERT_PGM_SUCCESS (pgm_image_set_alignment (image, PGM_IMAGE_LEFT));
  IMAGE_TO_DRAWABLE_3_TESTS (0.0f, 0.0f, 1.0f, 1.0f, -1.0f, -0.5f);

  ASSERT_PGM_SUCCESS (pgm_image_set_alignment (image, PGM_IMAGE_CENTER));
  IMAGE_TO_DRAWABLE_3_TESTS (0.5f, 0.0f, 1.5f, 1.0f, -0.5f, -0.5f);

  ASSERT_PGM_SUCCESS (pgm_image_set_alignment (image, PGM_IMAGE_RIGHT));
  IMAGE_TO_DRAWABLE_3_TESTS (1.0f, 0.0f, 2.0f, 1.0f, 0.0f, -0.5f);

  ASSERT_PGM_SUCCESS (pgm_drawable_set_size (PGM_DRAWABLE (image), 1.0f, 2.0f));

  ASSERT_PGM_SUCCESS (pgm_image_set_alignment (image, PGM_IMAGE_TOP));
  IMAGE_TO_DRAWABLE_3_TESTS (0.0f, 0.0f, 1.0f, 1.0f, -1.0f, -0.5f);

  ASSERT_PGM_SUCCESS (pgm_image_set_alignment (image, PGM_IMAGE_CENTER));
  IMAGE_TO_DRAWABLE_3_TESTS (0.0f, 0.5f, 1.0f, 1.5f, -1.0f, 0.0f);

  ASSERT_PGM_SUCCESS (pgm_image_set_alignment (image, PGM_IMAGE_BOTTOM));
  IMAGE_TO_DRAWABLE_3_TESTS (0.0f, 1.0f, 1.0f, 2.0f, -1.0f, 0.5f);

  /* zoomed */
  ASSERT_PGM_SUCCESS (pgm_image_set_layout (image, PGM_IMAGE_ZOOMED));

  ASSERT_PGM_SUCCESS (pgm_image_set_alignment (image, PGM_IMAGE_LEFT));
  IMAGE_TO_DRAWABLE_3_TESTS (0.0f, 0.0f, 2.0f, 2.0f, -2.0f, -1.0f);

  ASSERT_PGM_SUCCESS (pgm_image_set_alignment (image, PGM_IMAGE_CENTER));
  IMAGE_TO_DRAWABLE_3_TESTS (-0.5f, 0.0f, 1.5f, 2.0f, -2.5f, -1.0f);

  ASSERT_PGM_SUCCESS (pgm_image_set_alignment (image, PGM_IMAGE_RIGHT));
  IMAGE_TO_DRAWABLE_3_TESTS (-1.0f, 0.0f, 1.0f, 2.0f, -3.0f, -1.0f);

  ASSERT_PGM_SUCCESS (pgm_drawable_set_size (PGM_DRAWABLE (image), 2.0f, 1.0f));

  ASSERT_PGM_SUCCESS (pgm_image_set_alignment (image, PGM_IMAGE_TOP));
  IMAGE_TO_DRAWABLE_3_TESTS (0.0f, 0.0f, 2.0f, 2.0f, -2.0f, -1.0f);

  ASSERT_PGM_SUCCESS (pgm_image_set_alignment (image, PGM_IMAGE_CENTER));
  IMAGE_TO_DRAWABLE_3_TESTS (0.0f, -0.5f, 2.0f, 1.5f, -2.0f, -1.5f);

  ASSERT_PGM_SUCCESS (pgm_image_set_alignment (image, PGM_IMAGE_BOTTOM));
  IMAGE_TO_DRAWABLE_3_TESTS (0.0f, -1.0f, 2.0f, 1.0f, -2.0f, -2.0f);

  /* and now a few tests with changed aspect ratio */
  ASSERT_PGM_SUCCESS (pgm_image_get_aspect_ratio (image, &old_n, &old_d));
  ASSERT_PGM_SUCCESS (pgm_image_set_aspect_ratio (image, 3, 2));
  ASSERT_PGM_SUCCESS (pgm_drawable_set_size (PGM_DRAWABLE (image), 2.0f, 1.0f));
  ASSERT_PGM_SUCCESS (pgm_image_set_layout (image, PGM_IMAGE_SCALED));
  ASSERT_PGM_SUCCESS (pgm_image_set_alignment (image, PGM_IMAGE_RIGHT));

  IMAGE_TO_DRAWABLE_3_TESTS (0.5f, 0.0f, 2.0f, 1.0f, -1.0f, -0.5f);

  ASSERT_PGM_SUCCESS (pgm_image_set_aspect_ratio (image, old_n, old_d));

  gst_object_unref (image);

  pgm_main_quit ();
  g_thread_join (pgm_thread);
  pgm_deinit ();
}
PGM_END_TEST

PGM_START_TEST (test_image_to_drawable_fail)
{
  PgmImage *image;
  gfloat x, y;

  image = PGM_IMAGE (pgm_image_new_from_file (TEST_IMAGE_FILE, 0));
  fail_if (NULL == image, "image creation failed");

  ASSERT_PGM_ERROR_WITH_CRITICAL (pgm_image_to_drawable (NULL, &x, &y, 0, 0));
  ASSERT_PGM_ERROR_WITH_CRITICAL (pgm_image_to_drawable (image,
                                                         NULL, &y,0, 0));
  ASSERT_PGM_ERROR_WITH_CRITICAL (pgm_image_to_drawable (image,
                                                         &x, NULL, 0, 0));

  gst_object_unref (image);
}
PGM_END_TEST

Suite *
pgm_image_suite (void)
{
  Suite *s = suite_create ("PgmImage");
  TCase *tc_chain = tcase_create ("pgmimage tests");

  PGM_USE_STANDARD_FIXTURES (tc_chain);
  tcase_add_unchecked_fixture (tc_chain, test_image_unchecked_setup,
                               test_image_unchecked_teardown);

  suite_add_tcase (s, tc_chain);
  tcase_add_test (tc_chain, test_image_master_slave_del);
  tcase_add_test (tc_chain, test_image_default_values);
  tcase_add_test (tc_chain, test_image_set_get_correctness);
  tcase_add_test (tc_chain, test_image_buffer);
  tcase_add_test (tc_chain, test_image_from_file);
  tcase_add_test (tc_chain, test_image_dispose_while_loading);
  tcase_add_test (tc_chain, test_image_small_image);
  tcase_add_test (tc_chain, test_image_from_drawable);
  tcase_add_test (tc_chain, test_image_from_drawable_fail);
  tcase_add_test (tc_chain, test_image_to_drawable);
  tcase_add_test (tc_chain, test_image_to_drawable_fail);

  return s;
}

GST_CHECK_MAIN (pgm_image);
