(random_displacements)=

# Random displacements

Phonopy supports generating two types of random displacements.

- Random directions with constant displacement distance
- Random sampling of harmonic oscillator probability densities of phonon modes

In both cases, phonopy generates dataset of displacements of supercells in
`phonopy_disp.yaml`, supercells with displacements, and a perfect supercell. The
number of the supercells with displacements are specified by
{ref}`random_displacements_tag` tag (or `--rd` option) in either case.

Forces on atoms in supercells with displacements are calculated by a force
calculator and force constants are computed by a force constants calculator. As
the force constants calculator, currently only {ref}`fc_calculator_alm_tag` is
supported.

## Related setting tags

- {ref}`random_displacements_tag` (`--rd`)
- {ref}`random_displacement_temperature_tag` (`--rd-temperature`)
- {ref}`random_seed_tag` (`--random-seed`)
- {ref}`fc_calculator_tag` (`--fc-calc`)
- {ref}`fc_calculator_options_tag` (`--fc-calc-opt`)

## Random directions with constant displacement distance

See also {ref}`random_displacements_tag`.

(random_displacements_random_direction_example)=
### Example

An example to generate the random direction diplacements with distance of 0.03
Angstrom in `example/NaCl-rd` directory is as shown below,

```
 % phonopy --rd 10 --dim 2 2 2 --pa auto --amplitude 0.03 -c POSCAR-unitcell
         _
  _ __ | |__   ___  _ __   ___   _ __  _   _
 | '_ \| '_ \ / _ \| '_ \ / _ \ | '_ \| | | |
 | |_) | | | | (_) | | | | (_) || |_) | |_| |
 | .__/|_| |_|\___/|_| |_|\___(_) .__/ \__, |
 |_|                            |_|    |___/
                                      2.20.0

Compiled with OpenMP support (max 10 threads).
Python version 3.11.4
Spglib version 2.1.0

Crystal structure was read from "POSCAR-unitcell".
Unit of length: angstrom
Displacements creation mode
  Number of supercells with random displacements: 10
  Displacement distance: 0.03
Settings:
  Supercell: [2 2 2]
Spacegroup: Fm-3m (225)
Use -v option to watch primitive cell, unit cell, and supercell structures.

"phonopy_disp.yaml" and supercells have been created.

Summary of calculation was written in "phonopy_disp.yaml".
                 _
   ___ _ __   __| |
  / _ \ '_ \ / _` |
 |  __/ | | | (_| |
  \___|_| |_|\__,_|
```

In the generated `phonopy_disp.yaml`, dataset of displacements is stored as follows:

```
% cat phonopy_disp.yaml
...
displacements:
- #    1
  - displacement: # 1
      [  -0.0201336051051884,  0.0137526506253330,  0.0174786311319239 ]
  - displacement: # 2
      [  -0.0176810186962932,  0.0037245664471804, -0.0239480517504424 ]
  - displacement: # 3
      [  -0.0028626950563326, -0.0032145292137935,  0.0296895904139501 ]
  - displacement: # 4
      [   0.0251664434224757,  0.0149681266628547,  0.0065272742908559 ]
  - displacement: # 5
      [  -0.0136393731379345, -0.0226372187886819, -0.0141959087739228 ]
  - displacement: # 6
      [   0.0299122348895043, -0.0003839832933863, -0.0022606991718315 ]
  - displacement: # 7
      [  -0.0265153637733336, -0.0030921513486749, -0.0136884653633881 ]
  - displacement: # 8
      [   0.0284712562630672, -0.0049105574746481, -0.0080792321473599 ]
...
```

where the displacements are given in the Cartesian coordinates. The supercell
with displacements in the VASP interface is found as

```
% cat POSCAR-001
generated by phonopy
1.0
11.3806029523513423 0.0000000000000000 0.0000000000000000
0.0000000000000000 11.3806029523513423 0.0000000000000000
0.0000000000000000 0.0000000000000000 11.3806029523513423
Na Cl
32 32
Direct
0.9982308841465181 0.0012084289982625 0.0015358264588532
0.4984463899873917 0.0003272732088778 0.9978957132718971
0.9997484584016929 0.4997175431541499 0.0026087888786082
0.5022113453503161 0.5013152314271505 0.0005735438023965
0.9988015245593718 0.9980108945999205 0.4987526224371979
0.5026283523829750 0.9999662598462494 0.4998013550616521
0.9976701266282333 0.4997282963511142 0.4987972108841070
0.5025017353107100 0.4995685151748806 0.4992900875128334
...
```

It is easy to verify consistency between them, e.g.,

```python
np.dot(ph.displacements[0], np.linalg.inv(ph.supercell.cell)) + ph.supercell.scaled_positions
```

Once forces on atoms in these supercells with displacements, `FORCE_SETS` can be generated by the usual manner. For the `NaCl-rd` example,

```
% phonopy -f force-calcs/disp-{001..010}/vasprun.xml
```

where `phonopy_disp.yaml` has to be located in the current directory where the
above command is executed. The `FORCE_SETS` file is written in the
{ref}`file_forces_type_2` format. To compute force constants for the type-2
displacements-forces dataset, an external force constants calculator is
necessary, e.g., ALM (https://github.com/ttadano/ALM). With installed ALM, force
constants can be calculated by

```
% phonopy --alm --writefc
```
With this command, the force constants are written in `FORCE_CONSTANTS`.

(random_displacements_at_temperatures)=

## Random sampling of harmonic oscillator probability densities of phonon modes

Random displacements corresponding to a specific temperature are generated with
the combination of the {ref}`random_displacements_tag` (`--rd`) and
{ref}`random_displacement_temperature_tag` tags. This is based on the harmonic
phonon approximation. Since each phonon follows the probability density function
of the harmonic oscillator, the random displacements are generated by randomly
sampling the probability density functions of phonons at the commensurate
points. More details are found in the paper “Implementation
strategies in phonopy and phono3py” (see {ref}`citation_of_phonopy`).

### Example

To generate the random displacements in this approach, phonon information is
required. At the first step, phonons at the commensurate points are calculated.
Then, the random displacements are generated in the next step. If
`FORCE_CONSTANTS` file exists in the current directory, as an example, this is
performed by

```
% phonopy --readfc --rd 1000 --temperature 300
```

where 1000 supercells with displacements at 300K are generated.

### Visualization of displacement distance distribution

Distribution of displacement distances may be visualized and checked by a python
script like below.

```python
import matplotlib.pyplot as plt
import numpy as np
from phonopy.interface.phonopy_yaml import PhonopyYaml
phyml = PhonopyYaml()
phyml.read("phonopy_disp.yaml")
plt.hist(np.linalg.norm(phyml.dataset['displacements'].reshape(-1, 3), axis=1), 100)
plt.show()
```

```{image} NaCl-disp-dist.png
:scale: 50
```

### Another example

In the below, a step-by-step example is presented.

1. Compute supercell force sets in the usual manner with small displacements.
   Generate `FORCE_SETS` in the current directory.
2. Create new directory and copy `FORCE_SETS` and input crystal structure, or
   `phonopy_disp.yaml` to the new directory, then change to the new directory.
3. Using `FORCE_SETS`, generate supercells with random displacements at finite
   temperature using this tag. `phonopy_disp.yaml` is generated at this step.
   Therefore, if `phonopy_disp.yaml` already exists in this directory, it is
   overwritten. The required number of supercells depends on your system and
   also the purpose. It can be 10, or can be 1000.
4. Using those supercells with random displacements, calculate supercell forces
   with some force calculator with VASP, QE, etc.
5. Generate `FORCE_SETS` in this directory, which overwrites previous
   `FORCE_SETS`.
6. Run the force constants calculation, e.g., with ALM. ALM does the fitting
   force constants using the dataset of displacements and forces stored in
   `FORCE_SETS`. Run also phonon calculation. This is done for example,

   ```
   phonopy -v --alm --mesh 10 10 10 -p
   ```

   or

   ```bash
   phonopy -v --alm --writefc
   phonopy --readfc --mesh 10 10 10 -p
   ```

   Force constants calculated as above contain a part of temperature effect from
   atomic displacements. To even better include the temperature effect, this
   calculation has to be done self-consistently, i.e., it is required to repeat
   the loop over the steps 2-5 until the phonon structure is converged. The
   solution corresponds to a stochastic self-consistent harmonic approximation
   ({ref}`reference_sscha`) approach.

It is also possible to perform the same calculation as above using the phonopy API.
