#include <gtk/gtk.h>
#include <jansson.h>
#include <gcrypt.h>
#include "password-cb.h"
#include "message-dialogs.h"
#include "common/exports.h"


static void show_ret_msg_dialog (GtkWidget   *mainwin,
                                 const gchar *fpath,
                                 const gchar *ret_msg);


void
export_data_cb (GSimpleAction *simple,
                GVariant      *parameter __attribute__((unused)),
                gpointer       user_data)
{
    const gchar *action_name = g_action_get_name (G_ACTION(simple));
    AppData *app_data = (AppData *)user_data;

    const gchar *base_dir = NULL;
#ifndef USE_FLATPAK_APP_FOLDER
    base_dir = g_get_home_dir ();
#else
    base_dir = g_get_user_data_dir ();
#endif

    gboolean encrypted;
    if ((g_strcmp0 (action_name, "export_andotp") == 0) || (g_strcmp0 (action_name, "export_aegis") == 0)) {
        encrypted = TRUE;
    } else {
        encrypted = FALSE;
    }

    GtkFileChooserNative *fl_diag = gtk_file_chooser_native_new ("Export file",
                                                                 GTK_WINDOW(app_data->main_window),
                                                                 GTK_FILE_CHOOSER_ACTION_SAVE,
                                                                 "OK",
                                                                 "Cancel");
    gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER(fl_diag), base_dir);
    gtk_file_chooser_set_do_overwrite_confirmation (GTK_FILE_CHOOSER(fl_diag), TRUE);
    gtk_file_chooser_set_select_multiple (GTK_FILE_CHOOSER(fl_diag), FALSE);

    const gchar *filename = NULL;
    if (g_strcmp0 (action_name, ANDOTP_EXPORT_ACTION_NAME) == 0 || g_strcmp0 (action_name, ANDOTP_EXPORT_PLAIN_ACTION_NAME) == 0) {
        filename = (encrypted == TRUE) ? "andotp_exports.json.aes" : "andotp_exports.json";
    } else if (g_strcmp0 (action_name, FREEOTPPLUS_EXPORT_ACTION_NAME) == 0) {
        filename = "freeotpplus-exports.txt";
    } else if (g_strcmp0 (action_name, AEGIS_EXPORT_ACTION_NAME) == 0 || g_strcmp0 (action_name, AEGIS_EXPORT_PLAIN_ACTION_NAME) == 0) {
        filename = (encrypted == TRUE) ? "aegis_encrypted.json" : "aegis_export_plain.json";
    } else {
        show_message_dialog (app_data->main_window, "Invalid export action.", GTK_MESSAGE_ERROR);
        return;
    }

    gtk_file_chooser_set_current_name (GTK_FILE_CHOOSER(fl_diag), filename);

    gchar *export_file_abs_path = NULL;
    gint native_diag_res = gtk_native_dialog_run (GTK_NATIVE_DIALOG(fl_diag));
    if (native_diag_res == GTK_RESPONSE_ACCEPT) {
        export_file_abs_path = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER(fl_diag));
    }
    g_object_unref (fl_diag);

    if (export_file_abs_path == NULL) {
        show_message_dialog (app_data->main_window, "Invalid export file name/path.", GTK_MESSAGE_ERROR);
        return;
    }

    gchar *password = NULL, *ret_msg = NULL;
    if (g_strcmp0 (action_name, ANDOTP_EXPORT_ACTION_NAME) == 0 || g_strcmp0 (action_name, ANDOTP_EXPORT_PLAIN_ACTION_NAME) == 0) {
        if (encrypted == TRUE) {
            password = prompt_for_password (app_data, NULL, NULL, TRUE);
            if (password == NULL) {
                return;
            }
        }
        ret_msg = export_andotp (export_file_abs_path, password, app_data->db_data->json_data);
        show_ret_msg_dialog (app_data->main_window, export_file_abs_path, ret_msg);
    } else if (g_strcmp0 (action_name, FREEOTPPLUS_EXPORT_ACTION_NAME) == 0) {
        ret_msg = export_freeotpplus (export_file_abs_path, app_data->db_data->json_data);
        show_ret_msg_dialog (app_data->main_window, export_file_abs_path, ret_msg);
    } else if (g_strcmp0 (action_name, AEGIS_EXPORT_ACTION_NAME) == 0 || g_strcmp0 (action_name, AEGIS_EXPORT_PLAIN_ACTION_NAME) == 0) {
        if (encrypted == TRUE) {
            password = prompt_for_password (app_data, NULL, NULL, TRUE);
            if (password == NULL) {
                return;
            }
        }
        ret_msg = export_aegis (export_file_abs_path, app_data->db_data->json_data, password);
        show_ret_msg_dialog (app_data->main_window, export_file_abs_path, ret_msg);
    } else {
        show_message_dialog (app_data->main_window, "Invalid export action.", GTK_MESSAGE_ERROR);
        return;
    }
    g_free (ret_msg);
    g_free (export_file_abs_path);
    if (encrypted == TRUE) {
        gcry_free (password);
    }
}


static void
show_ret_msg_dialog (GtkWidget   *mainwin,
                     const gchar *fpath,
                     const gchar *ret_msg)
{
    GtkMessageType msg_type;
    gchar *message = NULL;

    if (ret_msg != NULL) {
        message = g_strconcat ("Error while exporting data: ", ret_msg, NULL);
        msg_type = GTK_MESSAGE_ERROR;
    } else {
        message = g_strconcat ("Data successfully exported to ", fpath, NULL);
        msg_type = GTK_MESSAGE_INFO;
    }
    show_message_dialog (mainwin, message, msg_type);
    g_free (message);
}
