//*******************************************************************
// Copyright (C) 2000 ImageLinks Inc.
//
// License:  LGPL
//
// See LICENSE.txt file in the top level directory for more details.
// 
// Author:  David Burken
//
// Description:
//
// Contains application definition "create_cg" app.
//
// It takes an image handler as input and creates a course grid model.
//
//*******************************************************************
//  $Id: create_cg.cpp 9111 2006-06-14 12:34:08Z gpotts $


#include <ossim/ossimConfig.h>
#include <ossim/base/ossimFilename.h>
#include <ossim/base/ossimDpt.h>
#include <ossim/base/ossimKeywordlist.h>
#include <ossim/init/ossimInit.h>
#include <ossim/imaging/ossimImageHandler.h>
#include <ossim/imaging/ossimImageHandlerRegistry.h>
#include <ossim/projection/ossimProjectionFactoryRegistry.h>
#include <ossim/projection/ossimProjection.h>
#include <ossim/projection/ossimCoarseGridModel.h>
#include <ossim/base/ossimArgumentParser.h>
#include <ossim/base/ossimApplicationUsage.h>

int main(int argc, char *argv[])
{
   ossimInit::instance()->initialize(argc, argv);

   double error = .1;
   ossim_int32 minSpacing = 100;
   ossimFilename outputBaseName;
   
   ossimFilename file = argv[argc-1];
   ossimRefPtr<ossimImageHandler> ih = ossimImageHandlerRegistry::instance()->open(file);
   
   std::string	tempString;
   ossimArgumentParser::ossimParameter	stringParam(tempString);
   ossimArgumentParser argumentParser(&argc, argv);
   ossimInit::instance()->addOptions(argumentParser);
   ossimInit::instance()->initialize(argumentParser);
   argumentParser.getApplicationUsage()->setApplicationName(argumentParser.getApplicationName());
   
   argumentParser.getApplicationUsage()->setDescription(argumentParser.getApplicationName()+" creates OSSIM coarse grid geom from image.");
   
   argumentParser.getApplicationUsage()->setCommandLineUsage(argumentParser.getApplicationName()+" [options] <input	file>");
   
   argumentParser.getApplicationUsage()->addCommandLineOption("-e", "<error>  default is .1.  This is basically the pixel accuracy of the grid");
   
   argumentParser.getApplicationUsage()->addCommandLineOption("-s","<min_line_sample_spacing> give the min line sample spacing for the coarse grid.  Helps in converging. Basically will stop splitting if number of pixels per grid cell gets\nbelow this number.");
   
   argumentParser.getApplicationUsage()->addCommandLineOption("-o", "<base output name> Specify a new base output name default it's base name is <image file>.dat and <image file>.geom.\nIf specified then the output is <base output name>.dat and <base output name>.geom where the .dat is the coarse grid and the .geom is the model");
   
   argumentParser.getApplicationUsage()->addCommandLineOption("-h or --help", "Shows help");
   
   if(argc < 2)
   {
      argumentParser.getApplicationUsage()->write(std::cout);
      exit(0);
   }
   
   std::vector<ossim_uint32> entryList;
   ossimString filename;
   if(argumentParser.read("-h") || argumentParser.read("--help"))
   {
      argumentParser.getApplicationUsage()->write(std::cout);
      exit(0);
   }
   
   if(argumentParser.read("-e", error))
   {
      if(error < 0.0)
      {
         error = .1;
      }
   }
   if(argumentParser.read("-s", minSpacing))
   {
      if(minSpacing < 1)
      {
         minSpacing = 100;
      }
   }
   
   argumentParser.read("-o", outputBaseName);
   ossimKeywordlist kwl;
   if(ih.valid())
   {
      if(outputBaseName == "")
      {
         outputBaseName = file;
      }

      ossimRefPtr<ossimProjection> proj;
      ih->getImageGeometry(kwl);

      proj = ossimProjectionFactoryRegistry::instance()->createProjection(kwl);
      if(!proj.valid())
      {
         proj = ossimProjectionFactoryRegistry::instance()->createProjection(file);
      }
      
      ossimCoarseGridModel::setInterpolationError(error);
      ossimCoarseGridModel::setMinGridSpacing(minSpacing);
      if(proj.valid())
      {
         kwl.clear();
         ossimCoarseGridModel cg;
         cg.buildGrid(ih->getBoundingRect(), proj.get());

         cg.saveState(kwl);
         
         kwl.write(outputBaseName.setExtension("geom"));
         cg.saveCoarseGrid(outputBaseName.setExtension("dat"));
      }
   }
   return 0;
}
