/* OpenVAS Manager
 * $Id$
 * Description: Manager Manage library: SQL backend.
 *
 * Authors:
 * Matthew Mundell <matthew.mundell@greenbone.net>
 * Timo Pollmeier <timo.pollmeier@greenbone.net>
 *
 * Copyright:
 * Copyright (C) 2009-2013 Greenbone Networks GmbH
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 */

/**
 * @file  manage_sql.c
 * @brief The OpenVAS Manager management library (SQLite implementation).
 *
 * This file defines the SQLite specific portions of the OpenVAS manager
 * management library.
 */

#define _GNU_SOURCE

#include "manage_sql.h"
#include "manage_acl.h"
#include "lsc_user.h"
#include "sql.h"
#include "scanner.h"
#include "utils.h"

#include <arpa/inet.h>
#include <assert.h>
#include <ctype.h>
#include <dirent.h>
#include <errno.h>
#include <gcrypt.h>
#include <glib/gstdio.h>
#include <locale.h>
#include <pwd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/un.h>
#include <sys/wait.h>
#include <unistd.h>
#include <sys/time.h>
#include <grp.h>

#include <openvas/base/openvas_string.h>
#include <openvas/base/openvas_file.h>
#include <openvas/base/openvas_hosts.h>
#include <openvas/misc/openvas_auth.h>
#include <openvas/misc/ldap_connect_auth.h>
#include <openvas/misc/radius.h>
#include <openvas/misc/openvas_logging.h>
#include <openvas/misc/openvas_uuid.h>
#include <openvas/misc/openvas_server.h>
#include <openvas/misc/openvas_ssh.h>
#include <openvas/base/pwpolicy.h>
#include <openvas/omp/omp.h>

#undef G_LOG_DOMAIN
/**
 * @brief GLib log domain.
 */
#define G_LOG_DOMAIN "md manage"

/**
 * @brief Scanner (openvassd) address.
 */
#define OPENVASSD_ADDRESS OPENVAS_RUN_DIR "/openvassd.sock"


/* Headers from backend specific manage_xxx.c file. */

int
manage_create_sql_functions ();

void
create_tables ();

void
check_db_sequences ();

static int
check_db_encryption_key ();

void
manage_attach_databases ();

int
manage_cert_loaded ();

int
manage_scap_loaded ();


/* Headers for symbols defined in manage.c which are private to libmanage. */

/**
 * @brief Flag to force authentication to succeed.
 *
 * 1 if set via scheduler, 2 if set via event, else 0.
 */
int authenticate_allow_all;

const char *threat_message_type (const char *);

const char *message_type_threat (const char *);

int delete_reports (task_t);

int delete_slave_task (const char *, int, const char *, const char *,
                       const char *);

int
set_certs (const char *, const char *, const char *);

gchar *
predefined_report_format_dir (const gchar *);

int
stop_task_internal (task_t);

int
validate_username (const gchar *);


/* Port range headers. */

void
make_port_ranges_iana_tcp_2012 (port_list_t);

void
make_port_ranges_iana_tcp_udp_2012 (port_list_t);

void
make_port_ranges_all_tcp_nmap_5_51_top_100 (port_list_t);

void
make_port_ranges_all_tcp_nmap_5_51_top_1000 (port_list_t);

void
make_port_ranges_nmap_5_51_top_2000_top_100 (port_list_t);

void
make_config_discovery (char *const, const char * const);

void
make_config_discovery_service_detection (char * const);

void
make_config_host_discovery (char *const, const char * const);

void
make_config_system_discovery (char *const, const char * const);

int
check_config_discovery (const char *);

void
check_config_host_discovery (const char *);

int
check_config_system_discovery (const char *);


/* Static headers. */

static void
nvt_selector_add (const char*, const char*, const char*, int);

static int
nvt_selector_families_growing (const char*);

static int
nvt_selector_nvts_growing_2 (const char*, int);

static void
nvt_selector_remove_selector (const char*, const char*, int);

static void
update_config_caches (config_t);

static void
update_all_config_caches ();

static gchar*
select_config_nvts (config_t, const char*, int, const char*);

int
family_count ();

static int
report_counts_cache_exists (report_t, int, int);

static void report_severity_data (report_t, const char *, const get_data_t *,
                                  severity_data_t*, severity_data_t*);

static gchar* apply_report_format (gchar *, gchar *, gchar *, gchar *,
                                   GList **);

static int cache_report_counts (report_t, int, int, severity_data_t*, int);

static char*
task_owner_uuid (task_t);

static int
insert_nvt_selectors (const char *, const array_t*);

static int
validate_param_value (report_format_t, report_format_param_t param, const char *,
                      const char *);

int
delete_task_lock (task_t, int);

gchar*
clean_hosts (const char *, int*);

int
create_port_list_unique (const char *, const char *, const char *,
                         port_list_t *);

int
valid_type (const char*);

static gboolean
find_user_by_name (const char *, user_t *user);

gboolean
find_group (const char*, group_t*);

gboolean
find_role (const char *, group_t *);

gboolean
find_role_with_permission (const char *, role_t *, const char *);

static gboolean
find_user_by_name (const char *, user_t *);

int
user_ensure_in_db (const gchar *, const gchar *);

static int
verify_report_format_internal (report_format_t);

void
cleanup_prognosis_iterator ();

int
set_password (const gchar *, const gchar *, const gchar *, gchar **);

void
permissions_set_locations (const char *, resource_t, resource_t, int);

void
permissions_set_orphans (const char *, resource_t, int);

void
permissions_set_subjects (const char *, resource_t, resource_t, int);

static resource_t
permission_resource (permission_t);

resource_t
permission_subject (permission_t);

char *
permission_subject_type (permission_t);

void
tags_set_locations (const char *, resource_t, resource_t, int);

void
tags_set_orphans (const char *, resource_t, int);

int
role_is_predefined (role_t);

int
role_is_predefined_id (const char *);

static int
create_permission_internal (const char *, const char *, const char *,
                            const char *, const char *, const char *,
                            permission_t *);

static int
check_config_families ();

static int
task_second_last_report (task_t, report_t *);

static gchar *
new_secinfo_message (event_t, const void*, alert_t);

static gchar *
new_secinfo_list (event_t, const void*, alert_t, int*);

static void
check_for_new_scap ();

static void
check_for_new_cert ();

static void
check_for_updated_scap ();

static void
check_for_updated_cert ();

static gchar*
results_extra_where (const get_data_t*, report_t, const gchar*,
                     int, int, int, const gchar*);

static int
report_counts_id_full (report_t, int *, int *, int *, int *, int *, int *,
                       double *, const get_data_t*, const char* ,
                       int *, int *, int *, int *, int *, int *, double *);

static int
check_report_format (const gchar *);


/* Variables. */

/**
 * @brief Function to fork a connection that will accept OMP requests.
 */
int (*manage_fork_connection) (openvas_connection_t *, gchar*) = NULL;

/**
 * @brief Function to mark progress.
 */
void (*progress) () = NULL;

/**
 * @brief Max number of hosts per target.
 */
static int max_hosts = MANAGE_MAX_HOSTS;

/**
 * @brief Default max number of bytes of reports included in email alerts.
 */
#define MAX_CONTENT_LENGTH 20000

/**
 * @brief Maximum number of bytes of reports included in email alerts.
 *
 * A value less or equal to 0 allows any size.
 */
static int max_content_length = MAX_CONTENT_LENGTH;

/**
 * @brief Default max number of bytes of reports attached to email alerts.
 */
#define MAX_ATTACH_LENGTH 1048576

/**
 * @brief Maximum number of bytes of reports attached to email alerts.
 *
 * A value less or equal to 0 allows any size.
 */
static int max_attach_length = MAX_ATTACH_LENGTH;

/**
 * @brief Default max number of bytes of user-defined message in email alerts.
 */
#define MAX_EMAIL_MESSAGE_LENGTH 2000

/**
 * @brief Maximum number of bytes of user-defined message text in email alerts.
 *
 * A value less or equal to 0 allows any size.
 */
static int max_email_message_length = MAX_EMAIL_MESSAGE_LENGTH;

/**
 * @brief Memory cache of NVT information from the database.
 */
nvtis_t* nvti_cache = NULL;

/**
 * @brief Name of the database file.
 */
gchar* task_db_name = NULL;

/**
 * @brief Whether a transaction has been opened and not committed yet.
 */
static gboolean in_transaction;

/**
 * @brief Time of reception of the currently processed message.
 */
static struct timeval last_msg;


/* OMP commands. */

/**
 * @brief The OMP command list.
 */
command_t omp_commands[]
 = {{"AUTHENTICATE", "Authenticate with the manager." },
    {"COMMANDS",     "Run a list of commands."},
    {"CREATE_AGENT", "Create an agent."},
    {"CREATE_ALERT", "Create an alert."},
    {"CREATE_ASSET", "Create an asset."},
    {"CREATE_CONFIG", "Create a config."},
    {"CREATE_CREDENTIAL", "Create a credential."},
    {"CREATE_FILTER", "Create a filter."},
    {"CREATE_GROUP", "Create a group."},
    {"CREATE_NOTE", "Create a note."},
    {"CREATE_OVERRIDE", "Create an override."},
    {"CREATE_PERMISSION", "Create a permission."},
    {"CREATE_PORT_LIST", "Create a port list."},
    {"CREATE_PORT_RANGE", "Create a port range in a port list."},
    {"CREATE_REPORT", "Create a report."},
    {"CREATE_REPORT_FORMAT", "Create a report format."},
    {"CREATE_ROLE", "Create a role."},
    {"CREATE_SCANNER", "Create a scanner."},
    {"CREATE_SCHEDULE", "Create a schedule."},
    {"CREATE_TAG", "Create a tag."},
    {"CREATE_TARGET", "Create a target."},
    {"CREATE_TASK", "Create a task."},
    {"CREATE_USER", "Create a new user."},
    {"DELETE_AGENT", "Delete an agent."},
    {"DELETE_ALERT", "Delete an alert."},
    {"DELETE_ASSET", "Delete an asset."},
    {"DELETE_CONFIG", "Delete a config."},
    {"DELETE_CREDENTIAL", "Delete a credential."},
    {"DELETE_FILTER", "Delete a filter."},
    {"DELETE_GROUP", "Delete a group."},
    {"DELETE_NOTE", "Delete a note."},
    {"DELETE_OVERRIDE", "Delete an override."},
    {"DELETE_PERMISSION", "Delete a permission."},
    {"DELETE_PORT_LIST", "Delete a port list."},
    {"DELETE_PORT_RANGE", "Delete a port range."},
    {"DELETE_REPORT", "Delete a report."},
    {"DELETE_REPORT_FORMAT", "Delete a report format."},
    {"DELETE_ROLE", "Delete a role."},
    {"DELETE_SCANNER", "Delete a scanner."},
    {"DELETE_SCHEDULE", "Delete a schedule."},
    {"DELETE_TAG", "Delete a tag."},
    {"DELETE_TARGET", "Delete a target."},
    {"DELETE_TASK", "Delete a task."},
    {"DELETE_USER", "Delete an existing user."},
    {"DESCRIBE_AUTH", "Get details about the used authentication methods."},
    {"EMPTY_TRASHCAN", "Empty the trashcan."},
    {"GET_AGENTS", "Get all agents."},
    {"GET_AGGREGATES", "Get aggregates of resources."},
    {"GET_ALERTS", "Get all alerts."},
    {"GET_ASSETS", "Get all assets."},
    {"GET_CONFIGS", "Get all configs."},
    {"GET_CREDENTIALS", "Get all credentials."},
    {"GET_FEEDS", "Get details of one or all feeds this Manager uses."},
    {"GET_FILTERS", "Get all filters."},
    {"GET_GROUPS", "Get all groups."},
    {"GET_INFO", "Get raw information for a given item."},
    {"GET_NOTES", "Get all notes."},
    {"GET_NVTS", "Get one or all available NVTs."},
    {"GET_NVT_FAMILIES", "Get a list of all NVT families."},
    {"GET_OVERRIDES", "Get all overrides."},
    {"GET_PERMISSIONS", "Get all permissions."},
    {"GET_PORT_LISTS", "Get all port lists."},
    {"GET_PREFERENCES", "Get preferences for all available NVTs."},
    {"GET_REPORTS", "Get all reports."},
    {"GET_REPORT_FORMATS", "Get all report formats."},
    {"GET_RESULTS", "Get results."},
    {"GET_ROLES", "Get all roles."},
    {"GET_SCANNERS", "Get all scanners."},
    {"GET_SCHEDULES", "Get all schedules."},
    {"GET_SETTINGS", "Get all settings."},
    {"GET_SYSTEM_REPORTS", "Get all system reports."},
    {"GET_TAGS", "Get all tags."},
    {"GET_TARGETS", "Get all targets."},
    {"GET_TASKS", "Get all tasks."},
    {"GET_USERS", "Get all users."},
    {"GET_VERSION", "Get the OpenVAS Manager Protocol version."},
    {"HELP", "Get this help text."},
    {"MODIFY_AGENT", "Modify an existing agent."},
    {"MODIFY_ALERT", "Modify an existing alert."},
    {"MODIFY_ASSET", "Modify an existing asset."},
    {"MODIFY_AUTH", "Modify the authentication methods."},
    {"MODIFY_CONFIG", "Update an existing config."},
    {"MODIFY_CREDENTIAL", "Modify an existing credential."},
    {"MODIFY_FILTER", "Modify an existing filter."},
    {"MODIFY_GROUP", "Modify an existing group."},
    {"MODIFY_NOTE", "Modify an existing note."},
    {"MODIFY_OVERRIDE", "Modify an existing override."},
    {"MODIFY_PERMISSION", "Modify an existing permission."},
    {"MODIFY_PORT_LIST", "Modify an existing port list."},
    {"MODIFY_REPORT", "Modify an existing report."},
    {"MODIFY_REPORT_FORMAT", "Modify an existing report format."},
    {"MODIFY_ROLE", "Modify an existing role."},
    {"MODIFY_SCANNER", "Modify an existing scanner."},
    {"MODIFY_SCHEDULE", "Modify an existing schedule."},
    {"MODIFY_SETTING", "Modify an existing setting."},
    {"MODIFY_TAG", "Modify an existing tag."},
    {"MODIFY_TARGET", "Modify an existing target."},
    {"MODIFY_TASK", "Update an existing task."},
    {"MODIFY_USER", "Modify a user."},
    {"MOVE_TASK", "Assign task to another slave scanner, even while running."},
    {"RESTORE", "Restore a resource."},
    {"RESUME_TASK", "Resume a stopped task."},
    {"RUN_WIZARD", "Run a wizard."},
    {"START_TASK", "Manually start an existing task."},
    {"STOP_TASK", "Stop a running task."},
    {"SYNC_CERT", "Synchronize with a CERT feed."},
    {"SYNC_CONFIG", "Synchronize a config with a scanner."},
    {"SYNC_FEED", "Synchronize with an NVT feed."},
    {"SYNC_SCAP", "Synchronize with a SCAP feed."},
    {"TEST_ALERT", "Run an alert."},
    {"VERIFY_AGENT", "Verify an agent."},
    {"VERIFY_REPORT_FORMAT", "Verify a report format."},
    {"VERIFY_SCANNER", "Verify a scanner."},
    {NULL, NULL}};

/**
 * @brief Check whether a command name is valid.
 *
 * @param[in]  name  Command name.
 *
 * @return 1 yes, 0 no.
 */
int
valid_omp_command (const char* name)
{
  command_t *command;
  command = omp_commands;
  while (command[0].name)
    if (strcasecmp (command[0].name, name) == 0)
      return 1;
    else
      command++;
  return 0;
}

/**
 * @brief Get the type associated with an OMP command.
 *
 * @param[in]  name  Command name.
 *
 * @return Freshly allocated type name if any, else NULL.
 */
gchar *
omp_command_type (const char* name)
{
  const char *under;
  under = strchr (name, '_');
  if (under && (strlen (under) > 1))
    {
      gchar *command;
      under++;
      command = g_strdup (under);
      if (command[strlen (command) - 1] == 's')
        command[strlen (command) - 1] = '\0';
      if (valid_type (command))
        return command;
      g_free (command);
    }
  return NULL;
}

/**
 * @brief Check whether an OMP command takes a resource.
 *
 * MODIFY_TARGET, for example, takes a target.
 *
 * @param[in]  name  Command name.
 *
 * @return 1 if takes resource, else 0.
 */
static int
omp_command_takes_resource (const char* name)
{
  assert (name);
  return strcasecmp (name, "AUTHENTICATE")
         && strcasecmp (name, "COMMANDS")
         && strcasestr (name, "CREATE_") != name
         && strcasestr (name, "DESCRIBE_") != name
         && strcasecmp (name, "EMPTY_TRASHCAN")
         && strcasecmp (name, "GET_VERSION")
         && strcasecmp (name, "HELP")
         && strcasecmp (name, "RUN_WIZARD")
         && strcasestr (name, "SYNC_") != name;
}


/* General helpers. */

/**
 * @brief Check if a resource with a certain name exists already.
 *
 * Conflicting resource can be global or owned by the current user.
 *
 * @param[in]   name      Name of resource to check for.
 * @param[in]   type      Type of resource.
 * @param[in]   resource  Resource to ignore, 0 otherwise.
 */
static gboolean
resource_with_name_exists (const char *name, const char *type,
                           resource_t resource)
{
  int ret;
  char *quoted_name, *quoted_type;

  assert (type);
  if (!name)
    return 0;
  quoted_name = sql_quote (name);
  quoted_type = sql_quote (type);
  if (resource)
    ret = sql_int ("SELECT COUNT(*) FROM %ss"
                   " WHERE name = '%s'"
                   " AND id != %llu"
                   " AND " ACL_USER_OWNS () ";",
                   quoted_type, quoted_name, resource,
                   current_credentials.uuid);
  else
    ret = sql_int ("SELECT COUNT(*) FROM %ss"
                   " WHERE name = '%s'"
                   " AND " ACL_USER_OWNS () ";",
                   quoted_type, quoted_name, current_credentials.uuid);

  g_free (quoted_name);
  g_free (quoted_type);
  return !!ret;
}

/**
 * @brief Check if a resource with a certain name exists already.
 *
 * Conflicting resource can be owned by anybody.
 *
 * @param[in]   name      Name of resource to check for.
 * @param[in]   type      Type of resource.
 * @param[in]   resource  Resource to ignore, 0 otherwise.
 */
static gboolean
resource_with_name_exists_global (const char *name, const char *type,
                                  resource_t resource)
{
  int ret;
  char *quoted_name, *quoted_type;

  assert (type);
  if (!name)
    return 0;
  quoted_name = sql_quote (name);
  quoted_type = sql_quote (type);
  if (resource)
    ret = sql_int ("SELECT COUNT(*) FROM %ss"
                   " WHERE name = '%s'"
                   " AND id != %llu;",
                   quoted_type, quoted_name, resource);
  else
    ret = sql_int ("SELECT COUNT(*) FROM %ss"
                   " WHERE name = '%s';",
                   quoted_type, quoted_name);

  g_free (quoted_name);
  g_free (quoted_type);
  return !!ret;
}

/**
 * @brief Test whether a string equal to a given string exists in an array.
 *
 * @param[in]  array   Array of gchar* pointers.
 * @param[in]  string  String.
 *
 * @return 1 if a string equal to \arg string exists in \arg array, else 0.
 */
static int
member (GPtrArray *array, const char *string)
{
  const gchar *item;
  int index = 0;
  while ((item = (gchar*) g_ptr_array_index (array, index++)))
    if (strcmp (item, string) == 0) return 1;
  return 0;
}

/**
 * @brief Ensure a string is in an array.
 *
 * @param[in]  array   Array.
 * @param[in]  string  String.  Copied into array.
 */
void
array_add_new_string (array_t *array, const gchar *string)
{
  guint index;
  for (index = 0; index < array->len; index++)
    if (strcmp (g_ptr_array_index (array, index), string) == 0)
      return;
  array_add (array, g_strdup (string));
}

/**
 * @brief Find a resource in the trashcan given a UUID.
 *
 * @param[in]   type      Type of resource.
 * @param[in]   uuid      UUID of resource.
 * @param[out]  resource  Resource return, 0 if succesfully failed to find
 *                        resource.
 *
 * @return FALSE on success (including if failed to find resource), TRUE on
 *         error.
 */
static gboolean
find_trash (const char *type, const char *uuid, resource_t *resource)
{
  gchar *quoted_uuid;

  if (!uuid)
    return FALSE;
  quoted_uuid = sql_quote (uuid);
  if (acl_user_owns_trash_uuid (type, quoted_uuid) == 0)
    {
      g_free (quoted_uuid);
      *resource = 0;
      return FALSE;
    }
  switch (sql_int64 (resource,
                     "SELECT id FROM %ss_trash WHERE uuid = '%s';",
                     type,
                     quoted_uuid))
    {
      case 0:
        break;
      case 1:        /* Too few rows in result of query. */
        *resource = 0;
        break;
      default:       /* Programming error. */
        assert (0);
      case -1:
        g_free (quoted_uuid);
        return TRUE;
        break;
    }

  g_free (quoted_uuid);
  return FALSE;
}

/**
 * @brief Convert an OTP time into seconds since epoch.
 *
 * Use UTC as timezone.
 *
 * @param[in]  text_time  Time as text in ctime format.
 *
 * @return Time since epoch.
 */
static int
parse_otp_time (const char *text_time)
{
  int epoch_time;
  struct tm tm;
  gchar *tz;

  /* Scanner sends UTC in ctime format: "Wed Jun 30 21:49:08 1993". */

  /* Store current TZ. */
  tz = getenv ("TZ") ? g_strdup (getenv ("TZ")) : NULL;

  if (setenv ("TZ", "UTC", 1) == -1)
    {
      g_warning ("%s: Failed to switch to UTC", __FUNCTION__);
      if (tz != NULL)
        setenv ("TZ", tz, 1);
      g_free (tz);
      return 0;
    }

  if (strptime ((char*) text_time, "%a %b %d %H:%M:%S %Y", &tm) == NULL)
    {
      g_warning ("%s: Failed to parse time", __FUNCTION__);
      if (tz != NULL)
        setenv ("TZ", tz, 1);
      g_free (tz);
      return 0;
    }
  epoch_time = mktime (&tm);
  if (epoch_time == -1)
    {
      g_warning ("%s: Failed to make time", __FUNCTION__);
      if (tz != NULL)
        setenv ("TZ", tz, 1);
      g_free (tz);
      return 0;
    }

  /* Revert to stored TZ. */
  if (tz)
    {
      if (setenv ("TZ", tz, 1) == -1)
        {
          g_warning ("%s: Failed to switch to original TZ", __FUNCTION__);
          g_free (tz);
          return 0;
        }
    }
  else
    unsetenv ("TZ");

  g_free (tz);
  return epoch_time;
}

/**
 * @brief Convert a ctime into seconds since epoch.
 *
 * Use the current timezone.
 *
 * @param[in]  text_time  Time as text in ctime format.
 *
 * @return Time since epoch.
 */
static int
parse_ctime (const char *text_time)
{
  int epoch_time;
  struct tm tm;

  /* ctime format: "Wed Jun 30 21:49:08 1993". */

  memset (&tm, 0, sizeof (struct tm));
  if (strptime ((char*) text_time, "%a %b %d %H:%M:%S %Y", &tm) == NULL)
    {
      g_warning ("%s: Failed to parse time", __FUNCTION__);
      return 0;
    }
  epoch_time = mktime (&tm);
  if (epoch_time == -1)
    {
      g_warning ("%s: Failed to make time", __FUNCTION__);
      return 0;
    }

  return epoch_time;
}

/**
 * @brief Convert an ISO time into seconds since epoch.
 *
 * For backward compatibility, if the conversion fails try parse in ctime
 * format.
 *
 * @param[in]  text_time  Time as text in ISO format: 2011-11-03T09:23:28+02:00.
 *
 * @return Time since epoch.  0 on error.
 */
int
parse_iso_time (const char *text_time)
{
  int epoch_time;
  struct tm tm;

  memset (&tm, 0, sizeof (struct tm));
  tm.tm_isdst = -1;
  if (strptime ((char*) text_time, "%FT%T%z", &tm) == NULL)
    {
      gchar *tz;

      memset (&tm, 0, sizeof (struct tm));
      tm.tm_isdst = -1;
      if (strptime ((char*) text_time, "%FT%TZ", &tm) == NULL)
        {
          /* Try time without timezone suffix, applying timezone of user */

          memset (&tm, 0, sizeof (struct tm));
          tm.tm_isdst = -1;
          if (strptime ((char*) text_time, "%FT%T", &tm) == NULL)
            return parse_ctime (text_time);

          /* Store current TZ. */
          tz = getenv ("TZ") ? g_strdup (getenv ("TZ")) : NULL;

          if (setenv ("TZ", current_credentials.timezone, 1) == -1)
            {
              g_warning ("%s: Failed to switch to timezone %s",
                         __FUNCTION__, current_credentials.timezone);
              if (tz != NULL)
                setenv ("TZ", tz, 1);
              g_free (tz);
              return 0;
            }

          memset (&tm, 0, sizeof (struct tm));
          tm.tm_isdst = -1;
          if (strptime ((char*) text_time, "%FT%T", &tm) == NULL)
            {
              assert (0);
              g_warning ("%s: Failed to parse time", __FUNCTION__);
              if (tz != NULL)
                setenv ("TZ", tz, 1);
              g_free (tz);
              return 0;
            }
        }
      else
        {
          /* Time has "Z" suffix for UTC */

          /* Store current TZ. */
          tz = getenv ("TZ") ? g_strdup (getenv ("TZ")) : NULL;

          if (setenv ("TZ", "UTC", 1) == -1)
            {
              g_warning ("%s: Failed to switch to UTC", __FUNCTION__);
              if (tz != NULL)
                setenv ("TZ", tz, 1);
              g_free (tz);
              return 0;
            }

          memset (&tm, 0, sizeof (struct tm));
          tm.tm_isdst = -1;
          if (strptime ((char*) text_time, "%FT%TZ", &tm) == NULL)
            {
              assert (0);
              g_warning ("%s: Failed to parse time", __FUNCTION__);
              if (tz != NULL)
                setenv ("TZ", tz, 1);
              g_free (tz);
              return 0;
            }
        }

      epoch_time = mktime (&tm);
      if (epoch_time == -1)
        {
          g_warning ("%s: Failed to make time", __FUNCTION__);
          if (tz != NULL)
            setenv ("TZ", tz, 1);
          g_free (tz);
          return 0;
        }

      /* Revert to stored TZ. */
      if (tz)
        {
          if (setenv ("TZ", tz, 1) == -1)
            {
              g_warning ("%s: Failed to switch to original TZ", __FUNCTION__);
              g_free (tz);
              return 0;
            }
        }
      else
        unsetenv ("TZ");

      g_free (tz);
    }
  else
    {
      gchar *tz, *new_tz;
      int offset_hour, offset_minute;
      char sign;

      /* Get the timezone offset from the string. */

      if (sscanf ((char*) text_time,
                  "%*u-%*u-%*uT%*u:%*u:%*u%[-+]%d:%d",
                  &sign, &offset_hour, &offset_minute)
          != 3)
        {
          /* Perhaps %z is an acronym like "CEST".  Assume it's local time. */
          epoch_time = mktime (&tm);
          if (epoch_time == -1)
            {
              g_warning ("%s: Failed to make time", __FUNCTION__);
              return 0;
            }
          return epoch_time;
        }

      /* Store current TZ. */

      tz = getenv ("TZ") ? g_strdup (getenv ("TZ")) : NULL;

      /* Switch to the timezone in the time string. */

      new_tz = g_strdup_printf ("UTC%c%d:%d",
                                sign == '-' ? '+' : '-',
                                offset_hour,
                                offset_minute);
      if (setenv ("TZ", new_tz, 1) == -1)
        {
          g_warning ("%s: Failed to switch to %s", __FUNCTION__, new_tz);
          g_free (new_tz);
          if (tz != NULL)
            setenv ("TZ", tz, 1);
          g_free (tz);
          return 0;
        }
      g_free (new_tz);

      /* Parse time again under the new timezone. */

      memset (&tm, 0, sizeof (struct tm));
      tm.tm_isdst = -1;
      if (strptime ((char*) text_time, "%FT%T%z", &tm) == NULL)
        {
          assert (0);
          g_warning ("%s: Failed to parse time", __FUNCTION__);
          if (tz != NULL)
            setenv ("TZ", tz, 1);
          g_free (tz);
          return 0;
        }

      epoch_time = mktime (&tm);
      if (epoch_time == -1)
        {
          g_warning ("%s: Failed to make time", __FUNCTION__);
          if (tz != NULL)
            setenv ("TZ", tz, 1);
          g_free (tz);
          return 0;
        }

      /* Revert to stored TZ. */
      if (tz)
        {
          if (setenv ("TZ", tz, 1) == -1)
            {
              g_warning ("%s: Failed to switch to original TZ", __FUNCTION__);
              g_free (tz);
              return 0;
            }
        }
      else
        unsetenv ("TZ");

      g_free (tz);
    }

  return epoch_time;
}

/**
 * @brief Calculate difference between now and epoch_time in days
 *
 * @param[in]  epoch_time  Time in seconds from epoch.
 *
 * @return Int days bettween now and epoch_time or -1 if epoch_time is in the
 * past
 */
int
days_from_now (time_t *epoch_time)
{
    time_t now = time (NULL);
    int diff = *epoch_time - now;

    if (diff < 0) return -1;
    return diff / 86400; /* 60 sec * 60 min * 24 h */
}

/**
 * @brief Create an ISO time from seconds since epoch.
 *
 * @param[in]  epoch_time  Time in seconds from epoch.
 * @param[out] abbrev      Abbreviation for current timezone.
 *
 * @return Pointer to ISO time in static memory, or NULL on error.
 */
static char *
iso_time_internal (time_t *epoch_time, const char **abbrev)
{
  struct tm *tm;
  static char time_string[100];

  tm = localtime (epoch_time);
  if (timezone == 0)
    {
      if (strftime (time_string, 98, "%FT%TZ", tm) == 0)
        return NULL;

      if (abbrev)
        *abbrev = "UTC";
    }
  else
    {
      int len;

      if (strftime (time_string, 98, "%FT%T%z", tm) == 0)
        return NULL;

      /* Insert the ISO 8601 colon by hand. */
      len = strlen (time_string);
      time_string[len + 1] = '\0';
      time_string[len] = time_string[len - 1];
      time_string[len - 1] = time_string[len - 2];
      time_string[len - 2] = ':';

      if (abbrev)
        {
          static char abbrev_string[100];
          if (strftime (abbrev_string, 98, "%Z", tm) == 0)
            return NULL;
          *abbrev = abbrev_string;
        }
    }

  return time_string;
}

/**
 * @brief Create an ISO time from seconds since epoch.
 *
 * @param[in]  epoch_time  Time in seconds from epoch.
 *
 * @return Pointer to ISO time in static memory, or NULL on error.
 */
char *
iso_time (time_t *epoch_time)
{
  return iso_time_internal (epoch_time, NULL);
}

/**
 * @brief Create an ISO time from seconds since epoch, given a timezone.
 *
 * @param[in]  epoch_time  Time in seconds from epoch.
 * @param[in]  timezone    Timezone.
 * @param[out] abbrev      Timezone abbreviation.
 *
 * @return Pointer to ISO time in static memory, or NULL on error.
 */
char *
iso_time_tz (time_t *epoch_time, const char *timezone, const char **abbrev)
{
  gchar *tz;
  char *ret;

  if (timezone == NULL)
    return iso_time (epoch_time);

  /* Store current TZ. */
  tz = getenv ("TZ") ? g_strdup (getenv ("TZ")) : NULL;

  if (setenv ("TZ", timezone, 1) == -1)
    {
      g_warning ("%s: Failed to switch to timezone", __FUNCTION__);
      if (tz != NULL)
        setenv ("TZ", tz, 1);
      g_free (tz);
      return iso_time (epoch_time);
    }

  tzset ();
  ret = iso_time_internal (epoch_time, abbrev);

  /* Revert to stored TZ. */
  if (tz)
    {
      if (setenv ("TZ", tz, 1) == -1)
        {
          g_warning ("%s: Failed to switch to original TZ", __FUNCTION__);
          g_free (tz);
          return ret;
        }
    }
  else
    unsetenv ("TZ");

  g_free (tz);
  return ret;
}

/**
 * @brief Find a string in an array.
 *
 * @param[in]  array   Array.
 * @param[in]  string  String.
 *
 * @return The string from the array if found, else NULL.
 */
static gchar*
array_find_string (array_t *array, const gchar *string)
{
  guint index;
  for (index = 0; index < array->len; index++)
    {
      gchar *ele;
      ele = (gchar*) g_ptr_array_index (array, index);
      if (ele && (strcmp (ele, string) == 0))
        return ele;
    }
  return NULL;
}

/**
 * @brief Find a string in a glib style string vector.
 *
 * @param[in]  vector  Vector.
 * @param[in]  string  String.
 *
 * @return The string from the vector if found, else NULL.
 */
static const gchar*
vector_find_string (const gchar **vector, const gchar *string)
{
  if (vector == NULL)
    return NULL;
  while (*vector)
    if (strcmp (*vector, string) == 0)
      return *vector;
    else
      vector++;
  return NULL;
}

/**
 * @brief Find a filter string in a glib style string vector.
 *
 * @param[in]  vector  Vector.
 * @param[in]  string  String.
 *
 * @return 1 if found, 2 if found with underscore prefix, else NULL.
 */
static int
vector_find_filter (const gchar **vector, const gchar *string)
{
  gchar *underscore;
  if (vector_find_string (vector, string))
    return 1;
  underscore = g_strdup_printf ("_%s", string);
  if (vector_find_string (vector, underscore))
    {
      g_free (underscore);
      return 2;
    }
  g_free (underscore);
  return 0;
}

/**
 * @brief Extract a tag from an OTP tag list.
 *
 * @param[in]   tags  Tag list.
 * @param[out]  tag   Tag name.
 *
 * @return Newly allocated tag value.
 */
gchar *
tag_value (const gchar *tags, const gchar *tag)
{
  gchar **split, **point;

  /* creation_date=2009-04-09 14:18:58 +0200 (Thu, 09 Apr 2009)|... */

  if (tags == NULL)
    return g_strdup ("");

  split = g_strsplit (tags, "|", 0);
  point = split;

  while (*point)
    {
      if ((strlen (*point) > strlen (tag))
          && (strncmp (*point, tag, strlen (tag)) == 0)
          && ((*point)[strlen (tag)] == '='))
        {
          gchar *ret;
          ret = g_strdup (*point + strlen (tag) + 1);
          g_strfreev (split);
          return ret;
        }
      point++;
    }
  g_strfreev (split);
  return g_strdup ("");
}

/**
 * @brief Try convert an OTP NVT tag time string into epoch time.
 *
 * @param[in]   string   String.
 * @param[out]  seconds  Time as seconds since the epoch.
 *
 * @return -1 failed to parse time, -2 failed to make time, -3 failed to parse
 *         timezone offset, 0 success.
 */
int
parse_time (const gchar *string, int *seconds)
{
  int epoch_time, offset;
  struct tm tm;

  if ((strcmp ((char*) string, "") == 0)
      || (strcmp ((char*) string, "$Date: $") == 0)
      || (strcmp ((char*) string, "$Date$") == 0)
      || (strcmp ((char*) string, "$Date:$") == 0)
      || (strcmp ((char*) string, "$Date") == 0)
      || (strcmp ((char*) string, "$$") == 0))
    {
      if (seconds)
        *seconds = 0;
      return 0;
    }

  /* Parse the time. */

  /* 2011-08-09 08:20:34 +0200 (Tue, 09 Aug 2011) */
  /* $Date: 2012-02-17 16:05:26 +0100 (Fr, 17. Feb 2012) $ */
  /* $Date: Fri, 11 Nov 2011 14:42:28 +0100 $ */
  memset (&tm, 0, sizeof (struct tm));
  if (strptime ((char*) string, "%F %T %z", &tm) == NULL)
    {
      memset (&tm, 0, sizeof (struct tm));
      if (strptime ((char*) string, "$Date: %F %T %z", &tm) == NULL)
        {
          memset (&tm, 0, sizeof (struct tm));
          if (strptime ((char*) string, "%a %b %d %T %Y %z", &tm) == NULL)
            {
              memset (&tm, 0, sizeof (struct tm));
              if (strptime ((char*) string, "$Date: %a, %d %b %Y %T %z", &tm)
                  == NULL)
                {
                  memset (&tm, 0, sizeof (struct tm));
                  if (strptime ((char*) string, "$Date: %a %b %d %T %Y %z", &tm)
                      == NULL)
                    {
                      g_warning ("%s: Failed to parse time: %s",
                                 __FUNCTION__, string);
                      return -1;
                    }
                }
            }
        }
    }
  epoch_time = mktime (&tm);
  if (epoch_time == -1)
    {
      g_warning ("%s: Failed to make time: %s", __FUNCTION__, string);
      return -2;
    }

  /* Get the timezone offset from the string. */

  if ((sscanf ((char*) string, "%*u-%*u-%*u %*u:%*u:%*u %d%*[^]]", &offset)
               != 1)
      && (sscanf ((char*) string, "$Date: %*u-%*u-%*u %*u:%*u:%*u %d%*[^]]",
                  &offset)
          != 1)
      && (sscanf ((char*) string, "%*s %*s %*s %*u:%*u:%*u %*u %d%*[^]]",
                  &offset)
          != 1)
      && (sscanf ((char*) string,
                  "$Date: %*s %*s %*s %*u %*u:%*u:%*u %d%*[^]]",
                  &offset)
          != 1)
      && (sscanf ((char*) string, "$Date: %*s %*s %*s %*u:%*u:%*u %*u %d%*[^]]",
                  &offset)
          != 1))
    {
      g_warning ("%s: Failed to parse timezone offset: %s", __FUNCTION__,
                 string);
      return -3;
    }

  /* Use the offset to convert to UTC. */

  if (offset < 0)
    {
      epoch_time += ((-offset) / 100) * 60 * 60;
      epoch_time += ((-offset) % 100) * 60;
    }
  else if (offset > 0)
    {
      epoch_time -= (offset / 100) * 60 * 60;
      epoch_time -= (offset % 100) * 60;
    }

  if (seconds)
    *seconds = epoch_time;
  return 0;
}

/**
 * @brief Get last time NVT alerts were checked.
 *
 * @return Last check time.
 */
static int
nvts_check_time ()
{
  return sql_int ("SELECT"
                  " CASE WHEN EXISTS (SELECT * FROM meta"
                  "                   WHERE name = 'nvts_check_time')"
                  "      THEN CAST ((SELECT value FROM meta"
                  "                  WHERE name = 'nvts_check_time')"
                  "                 AS INTEGER)"
                  "      ELSE 0"
                  "      END;");
}

/**
 * @brief Get last time SCAP SecInfo alerts were checked.
 */
static int
scap_check_time ()
{
  return sql_int ("SELECT"
                  " CASE WHEN EXISTS (SELECT * FROM meta"
                  "                   WHERE name = 'scap_check_time')"
                  "      THEN CAST ((SELECT value FROM meta"
                  "                  WHERE name = 'scap_check_time')"
                  "                 AS INTEGER)"
                  "      ELSE 0"
                  "      END;");
}

/**
 * @brief Get last time CERT SecInfo alerts were checked.
 */
static int
cert_check_time ()
{
  return sql_int ("SELECT"
                  " CASE WHEN EXISTS (SELECT * FROM meta"
                  "                   WHERE name = 'cert_check_time')"
                  "      THEN CAST ((SELECT value FROM meta"
                  "                  WHERE name = 'cert_check_time')"
                  "                 AS INTEGER)"
                  "      ELSE 0"
                  "      END;");
}

/**
 * @brief Setup for an option process.
 *
 * @param[in]  log_config  Log configuration.
 * @param[in]  database    Database.
 *
 * @return 0 success, -1 error, -2 database is wrong version,
 *         -3 database needs to be initialised from server.
 */
int
manage_option_setup (GSList *log_config, const gchar *database)
{
  const gchar *db;
  int ret;

  if (openvas_auth_init ())
    {
      fprintf (stderr, "Authentication init failed\n");
      return -1;
    }

  db = database ? database : sql_default_database ();

  ret = init_manage_helper (log_config, db, MANAGE_ABSOLUTE_MAX_IPS_PER_TARGET, NULL);
  assert (ret != -4);
  switch (ret)
    {
      case 0:
        break;
      case -2:
        fprintf (stderr, "Database is wrong version.\n");
        return ret;
      case -3:
        fprintf (stderr,
                 "Database must be initialised"
                 " (with --update or --rebuild).\n");
        return ret;
      default:
        fprintf (stderr, "Internal error.\n");
        return ret;
    }

  init_manage_process (0, db);

  return 0;
}

/**
 * @brief Cleanup for an option process.
 */
void
manage_option_cleanup ()
{
  cleanup_manage_process (TRUE);
}

/**
 * @brief Iterator column.
 */
typedef struct
{
  gchar *select;       ///< Column for SELECT.
  gchar *filter;       ///< Filter column name.  NULL to use select_column.
  keyword_type_t type; ///< Type of column.
} column_t;

/**
 * @brief Print an array of columns.
 *
 * @param[in]  columns  Columns.
 */
void
column_array_print (column_t *columns)
{
  g_debug ("%s: %p", __FUNCTION__, columns);
  g_debug ("%s: {", __FUNCTION__);
  while (columns->select)
    {
      g_debug ("%s:   { \"%s\", \"%s\", %i }",
               __FUNCTION__,
               columns->select,
               columns->filter ? columns->filter : "NULL",
               columns->type);
      columns++;
    }
  g_debug ("%s: }", __FUNCTION__);
}

/**
 * @brief Copy an array of columns.
 *
 * @param[in]  columns  Columns.
 */
column_t *
column_array_copy (column_t *columns)
{
  column_t *point, *start;
  int count;

  count = 1;
  point = columns;
  while (point->select)
    {
      point++;
      count++;
    }

  g_debug ("%s: %i", __FUNCTION__, count);
  start = g_malloc0 (sizeof (column_t) * count);
  point = start;

  while (columns->select)
    {
      point->select = g_strdup (columns->select);
      point->filter = columns->filter ? g_strdup (columns->filter) : NULL;
      point->type = columns->type;
      point++;
      columns++;
    }
  return start;
}

/**
 * @brief Free an array of columns.
 *
 * @param[in]  columns  Columns.
 */
void
column_array_free (column_t *columns)
{
  while (columns->filter)
    {
      g_free (columns->select);
      g_free (columns->filter);
      columns++;
    }
  g_free (columns);
}

/**
 * @brief Set the select clause of a column in an array of columns.
 *
 * Frees the existing select clause.
 *
 * @param[in]  columns  Columns.
 * @param[in]  filter   Filter term name.
 * @param[in]  select   Select clause.
 */
void
column_array_set (column_t *columns, const gchar *filter, gchar *select)
{
  while (columns->select)
    {
      if (columns->filter && (strcmp (columns->filter, filter) == 0))
        {
          g_free (columns->select);
          columns->select = select;
          break;
        }
      columns++;
    }
}


/* Filter utilities. */

/**
 * @brief Get the symbol of a keyword relation.
 *
 * @param[in]  relation  Relation.
 *
 * @return Relation symbol.
 */
const char *
keyword_relation_symbol (keyword_relation_t relation)
{
  switch (relation)
    {
      case KEYWORD_RELATION_APPROX:        return "~";
      case KEYWORD_RELATION_COLUMN_ABOVE:  return ">";
      case KEYWORD_RELATION_COLUMN_APPROX: return "~";
      case KEYWORD_RELATION_COLUMN_EQUAL:  return "=";
      case KEYWORD_RELATION_COLUMN_BELOW:  return "<";
      case KEYWORD_RELATION_COLUMN_REGEXP: return ":";
      default:                             return "";
    }
}

/**
 * @brief Free a keyword.
 *
 * @param[in]  keyword  Filter keyword.
 */
static void
keyword_free (keyword_t* keyword)
{
  g_free (keyword->string);
  g_free (keyword->column);
}

/**
 * @brief Parse a filter keyword.
 *
 * @param[in]  keyword  Filter keyword.
 */
static void
parse_keyword (keyword_t* keyword)
{
  gchar *string;
  int digits;

  if (keyword->column == NULL && keyword->equal == 0)
    {
      keyword->relation = KEYWORD_RELATION_APPROX;
      keyword->type = KEYWORD_TYPE_STRING;
      return;
    }

  /* Special values to substitute */

  if (keyword->column
      && (strcasecmp (keyword->column, "severity") == 0
          || strcasecmp (keyword->column, "new_severity") == 0))
    {
      if (strcasecmp (keyword->string, "Log") == 0)
        {
          keyword->double_value = SEVERITY_LOG;
          keyword->type = KEYWORD_TYPE_DOUBLE;
          return;
        }
      if (strcasecmp (keyword->string, "False Positive") == 0)
        {
          keyword->double_value = SEVERITY_FP;
          keyword->type = KEYWORD_TYPE_DOUBLE;
          return;
        }
      else if (strcasecmp (keyword->string, "Debug") == 0)
        {
          keyword->double_value = SEVERITY_DEBUG;
          keyword->type = KEYWORD_TYPE_DOUBLE;
          return;
        }
      else if (strcasecmp (keyword->string, "Error") == 0)
        {
          keyword->double_value = SEVERITY_ERROR;
          keyword->type = KEYWORD_TYPE_DOUBLE;
          return;
        }
    }

  /* The type. */

  string = keyword->string;
  if (*string == '\0')
    {
      keyword->type = KEYWORD_TYPE_STRING;
      return;
    }
  if (*string && *string == '-' && strlen (string) > 1) string++;
  digits = 0;
  while (*string && isdigit (*string))
    {
      digits = 1;
      string++;
    }
  if (digits == 0)
    keyword->type = KEYWORD_TYPE_STRING;
  else if (*string)
    {
      struct tm date;
      gchar next;
      int parsed_integer;
      double parsed_double;
      char dummy[2];
      memset (&date, 0, sizeof (date));
      next = *(string + 1);
      if (next == '\0' && *string == 's')
        {
          time_t now;
          now = time (NULL);
          keyword->integer_value = now + atoi (keyword->string);
          keyword->type = KEYWORD_TYPE_INTEGER;
        }
      else if (next == '\0' && *string == 'm')
        {
          time_t now;
          now = time (NULL);
          keyword->integer_value = now + (atoi (keyword->string) * 60);
          keyword->type = KEYWORD_TYPE_INTEGER;
        }
      else if (next == '\0' && *string == 'h')
        {
          time_t now;
          now = time (NULL);
          keyword->integer_value = now + (atoi (keyword->string) * 3600);
          keyword->type = KEYWORD_TYPE_INTEGER;
        }
      else if (next == '\0' && *string == 'd')
        {
          time_t now;
          now = time (NULL);
          keyword->integer_value = now + (atoi (keyword->string) * 86400);
          keyword->type = KEYWORD_TYPE_INTEGER;
        }
      else if (next == '\0' && *string == 'w')
        {
          time_t now;
          now = time (NULL);
          keyword->integer_value = now + atoi (keyword->string) * 604800;
          keyword->type = KEYWORD_TYPE_INTEGER;
        }
      else if (next == '\0' && *string == 'M')
        {
          time_t now;
          now = time (NULL);
          keyword->integer_value = add_months (now, atoi (keyword->string));
          keyword->type = KEYWORD_TYPE_INTEGER;
        }
      else if (next == '\0' && *string == 'y')
        {
          time_t now;
          now = time (NULL);
          keyword->integer_value = add_months (now,
                                               atoi (keyword->string) * 12);
          keyword->type = KEYWORD_TYPE_INTEGER;
        }
      else if (strptime (keyword->string, "%Y-%m-%dT%H:%M", &date))
        {
          keyword->integer_value = mktime (&date);
          keyword->type = KEYWORD_TYPE_INTEGER;
        }
      else if (memset (&date, 0, sizeof (date)),
               strptime (keyword->string, "%Y-%m-%d", &date))
        {
          keyword->integer_value = mktime (&date);
          keyword->type = KEYWORD_TYPE_INTEGER;
        }
      else if (sscanf (keyword->string, "%d%1s", &parsed_integer, dummy) == 1)
        {
          keyword->integer_value = parsed_integer;
          keyword->type = KEYWORD_TYPE_INTEGER;
        }
      else if (sscanf (keyword->string, "%lf%1s", &parsed_double, dummy) == 1
               && parsed_double <= DBL_MAX)
        {
          keyword->double_value = parsed_double;
          keyword->type = KEYWORD_TYPE_DOUBLE;
        }
      else
        keyword->type = KEYWORD_TYPE_STRING;
    }
  else
    {
      keyword->integer_value = atoi (keyword->string);
      keyword->type = KEYWORD_TYPE_INTEGER;
    }
}

/**
 * @brief Check whether a keyword has any effect in the filter.
 *
 * Some keywords are redundant, like a second sort= keyword.
 *
 * @param[in]  array    Array of existing keywords.
 * @param[in]  keyword  Keyword under consideration.
 *
 * @return 0 no, 1 yes.
 */
static int
keyword_applies (array_t *array, const keyword_t *keyword)
{
  if (keyword->column
      && ((strcmp (keyword->column, "sort") == 0)
          || (strcmp (keyword->column, "sort-reverse") == 0))
      && (keyword->relation == KEYWORD_RELATION_COLUMN_EQUAL))
    {
      int index;

      index = array->len;
      while (index--)
        {
          keyword_t *item;
          item = (keyword_t*) g_ptr_array_index (array, index);
          if (item->column
              && ((strcmp (item->column, "sort") == 0)
                  || (strcmp (item->column, "sort-reverse") == 0)))
            return 0;
        }
      return 1;
    }

  if (keyword->column
      && (strcmp (keyword->column, "first") == 0))
    {
      int index;

      index = array->len;
      while (index--)
        {
          keyword_t *item;
          item = (keyword_t*) g_ptr_array_index (array, index);
          if (item->column && (strcmp (item->column, "first") == 0))
            return 0;
        }
    }

  if (keyword->column
      && (strcmp (keyword->column, "rows") == 0))
    {
      int index;

      index = array->len;
      while (index--)
        {
          keyword_t *item;
          item = (keyword_t*) g_ptr_array_index (array, index);
          if (item->column && (strcmp (item->column, "rows") == 0))
            return 0;
        }
    }

  if (keyword->column
      && (strcmp (keyword->column, "apply_overrides") == 0))
    {
      int index;

      index = array->len;
      while (index--)
        {
          keyword_t *item;
          item = (keyword_t*) g_ptr_array_index (array, index);
          if (item->column && (strcmp (item->column, "apply_overrides") == 0))
            return 0;
        }
    }

  if (keyword->column
      && (strcmp (keyword->column, "autofp") == 0))
    {
      int index;

      index = array->len;
      while (index--)
        {
          keyword_t *item;
          item = (keyword_t*) g_ptr_array_index (array, index);
          if (item->column && (strcmp (item->column, "autofp") == 0))
            return 0;
        }
    }

  if (keyword->column
      && (strcmp (keyword->column, "delta_states") == 0))
    {
      int index;

      index = array->len;
      while (index--)
        {
          keyword_t *item;
          item = (keyword_t*) g_ptr_array_index (array, index);
          if (item->column && (strcmp (item->column, "delta_states") == 0))
            return 0;
        }
    }

  if (keyword->column
      && (strcmp (keyword->column, "levels") == 0))
    {
      int index;

      index = array->len;
      while (index--)
        {
          keyword_t *item;
          item = (keyword_t*) g_ptr_array_index (array, index);
          if (item->column && (strcmp (item->column, "levels") == 0))
            return 0;
        }
    }

  if (keyword->column
      && (strcmp (keyword->column, "min_qod") == 0))
    {
      int index;

      index = array->len;
      while (index--)
        {
          keyword_t *item;
          item = (keyword_t*) g_ptr_array_index (array, index);
          if (item->column && (strcmp (item->column, "min_qod") == 0))
            return 0;
        }
    }

  if (keyword->column
      && (strcmp (keyword->column, "notes") == 0))
    {
      int index;

      index = array->len;
      while (index--)
        {
          keyword_t *item;
          item = (keyword_t*) g_ptr_array_index (array, index);
          if (item->column && (strcmp (item->column, "notes") == 0))
            return 0;
        }
    }

  if (keyword->column
      && (strcmp (keyword->column, "overrides") == 0))
    {
      int index;

      index = array->len;
      while (index--)
        {
          keyword_t *item;
          item = (keyword_t*) g_ptr_array_index (array, index);
          if (item->column && (strcmp (item->column, "overrides") == 0))
            return 0;
        }
    }

  if (keyword->column
      && (strcmp (keyword->column, "result_hosts_only") == 0))
    {
      int index;

      index = array->len;
      while (index--)
        {
          keyword_t *item;
          item = (keyword_t*) g_ptr_array_index (array, index);
          if (item->column && (strcmp (item->column, "result_hosts_only") == 0))
            return 0;
        }
    }

  if (keyword->column
      && (strcmp (keyword->column, "timezone") == 0))
    {
      int index;

      index = array->len;
      while (index--)
        {
          keyword_t *item;
          item = (keyword_t*) g_ptr_array_index (array, index);
          if (item->column && (strcmp (item->column, "timezone") == 0))
            return 0;
        }
    }

  return 1;
}

/**
 * @brief Free a split filter.
 *
 * @param[in]  split  Split filter.
 */
void
filter_free (array_t *split)
{
  keyword_t **point;
  for (point = (keyword_t**) split->pdata; *point; point++)
    keyword_free (*point);
  array_free (split);
}

/**
 * @brief Flag to control the default sorting produced by split filter.
 *
 * If this is true, and the filter does not specify a sort field, then
 * split_filter will not insert a default sort term, so that the random
 * (and fast) table order in the database will be used.
 */
int table_order_if_sort_not_specified = 0;

/**
 * @brief Split the filter term into parts.
 *
 * @param[in]  given_filter  Filter term.
 *
 * @return Array of strings, the parts.
 */
array_t *
split_filter (const gchar* given_filter)
{
  int in_quote, between;
  array_t *parts;
  const gchar *current_part, *filter;
  keyword_t *keyword;

  assert (given_filter);

  filter = given_filter;
  parts = make_array ();
  in_quote = 0;
  between = 1;
  keyword = NULL;
  current_part = filter;  /* To silence compiler warning. */
  while (*filter)
    {
      switch (*filter)
        {
          case '=':
            if (between)
              {
                /* Empty index.  Start a part. */
                keyword = g_malloc0 (sizeof (keyword_t));
                keyword->equal = 1;
                current_part = filter + 1;
                between = 0;
                break;
              }
          case ':':
          case '~':
          case '>':
          case '<':
            if (between)
              {
                /* Empty index.  Just start a part for now. */
                keyword = g_malloc0 (sizeof (keyword_t));
                current_part = filter;
                between = 0;
                break;
              }
            if (in_quote)
              break;
            /* End of an index. */
            if (keyword == NULL)
              {
                assert (0);
                break;
              }
            if (keyword->column)
              /* Already had an index char. */
              break;
            if (filter <= (current_part - 1))
              {
                assert (0);
                break;
              }
            keyword->column = g_strndup (current_part,
                                         filter - current_part);
            current_part = filter + 1;
            switch (*filter)
              {
                case '=':
                  keyword->relation = KEYWORD_RELATION_COLUMN_EQUAL;
                  break;
                case '~':
                  keyword->relation = KEYWORD_RELATION_COLUMN_APPROX;
                  break;
                case '>':
                  keyword->relation = KEYWORD_RELATION_COLUMN_ABOVE;
                  break;
                case '<':
                  keyword->relation = KEYWORD_RELATION_COLUMN_BELOW;
                  break;
                case ':':
                  keyword->relation = KEYWORD_RELATION_COLUMN_REGEXP;
                  break;
              }
            break;

          case ' ':
          case '\t':
          case '\n':
          case '\r':
            if (in_quote || between)
              break;
            /* End of a part. */
            if (keyword == NULL)
              {
                assert (0);
                break;
              }
            keyword->string = g_strndup (current_part, filter - current_part);
            parse_keyword (keyword);
            if (keyword_applies (parts, keyword))
              array_add (parts, keyword);
            keyword = NULL;
            between = 1;
            break;

          case '"':
            if (in_quote)
              {
                /* End of a quoted part. */
                if (keyword == NULL)
                  {
                    assert (0);
                    break;
                  }
                keyword->quoted = 1;
                keyword->string = g_strndup (current_part,
                                             filter - current_part);
                parse_keyword (keyword);
                if (keyword_applies (parts, keyword))
                  array_add (parts, keyword);
                keyword = NULL;
                in_quote = 0;
                between = 1;
              }
            else if (between)
              {
                /* Start of a quoted part. */
                keyword = g_malloc0 (sizeof (keyword_t));
                in_quote = 1;
                current_part = filter + 1;
                between = 0;
              }
            else if (keyword->column && filter == current_part)
              {
                /* A quoted index. */
                in_quote = 1;
                current_part++;
              }
            else if (keyword->equal && filter == current_part)
              {
                /* A quoted exact term, like ="abc". */
                in_quote = 1;
                current_part++;
              }
            /* Else just a quote in a keyword, like ab"cd. */
            break;

          default:
            if (between)
              {
                /* Start of a part. */
                keyword = g_malloc0 (sizeof (keyword_t));
                current_part = filter;
                between = 0;
              }
            break;
        }
      filter++;
    }
  if (between == 0)
    {
      if (keyword == NULL)
        assert (0);
      else
        {
          keyword->quoted = in_quote;
          keyword->string = g_strdup (current_part);
          parse_keyword (keyword);
          if (keyword_applies (parts, keyword))
            array_add (parts, keyword);
          keyword = NULL;
        }
    }
  assert (keyword == NULL);

  {
    int index, first, max, sort;
    keyword_t *keyword;

    index = parts->len;
    first = max = sort = 0;
    while (index--)
      {
        keyword_t *item;
        item = (keyword_t*) g_ptr_array_index (parts, index);
        if (item->column && (strcmp (item->column, "first") == 0))
          first = 1;
        else if (item->column && (strcmp (item->column, "rows") == 0))
          max = 1;
        else if (item->column
                 && ((strcmp (item->column, "sort") == 0)
                     || (strcmp (item->column, "sort-reverse") == 0)))
          sort = 1;
      }

    if (first == 0)
      {
        keyword = g_malloc0 (sizeof (keyword_t));
        keyword->column = g_strdup ("first");
        keyword->string = g_strdup ("1");
        keyword->type = KEYWORD_TYPE_STRING;
        keyword->relation = KEYWORD_RELATION_COLUMN_EQUAL;
        array_add (parts, keyword);
      }

    if (max == 0)
      {
        keyword = g_malloc0 (sizeof (keyword_t));
        keyword->column = g_strdup ("rows");
        /* If there was a filter, make max_return default to Rows Per
         * Page.  This keeps the pre-filters OMP behaviour when the filter
         * is empty, but is more convenenient for clients that set the
         * filter. */
        if (strlen (given_filter))
          keyword->string = g_strdup ("-2");
        else
          keyword->string = g_strdup ("-1");
        keyword->type = KEYWORD_TYPE_STRING;
        keyword->relation = KEYWORD_RELATION_COLUMN_EQUAL;
        array_add (parts, keyword);
      }

    if (table_order_if_sort_not_specified == 0 && sort == 0)
      {
        keyword = g_malloc0 (sizeof (keyword_t));
        keyword->column = g_strdup ("sort");
        keyword->string = g_strdup ("name");
        keyword->type = KEYWORD_TYPE_STRING;
        keyword->relation = KEYWORD_RELATION_COLUMN_EQUAL;
        array_add (parts, keyword);
      }
  }

  array_add (parts, NULL);

  return parts;
}

/**
 * @brief Get info from a filter.
 *
 * It's up to the caller to ensure that max is adjusted for Max Rows Per Page
 * (by calling manage_max_rows).
 *
 * @param[in]   filter      Filter.
 * @param[out]  first       Number of first item.
 * @param[out]  max         Max number of rows.
 * @param[out]  sort_field  Sort field.
 * @param[out]  sort_order  Sort order.
 */
void
manage_filter_controls (const gchar *filter, int *first, int *max,
                        gchar **sort_field, int *sort_order)
{
  keyword_t **point;
  array_t *split;

  if (filter == NULL)
    {
      if (first)
        *first = 1;
      if (max)
        *max = -1;
      if (sort_field)
        *sort_field = g_strdup ("name");
      if (sort_order)
        *sort_order = 1;
      return;
    }

  split = split_filter (filter);
  point = (keyword_t**) split->pdata;
  if (first)
    {
      *first = 1;
      while (*point)
        {
          keyword_t *keyword;

          keyword = *point;
          if (keyword->column && (strcmp (keyword->column, "first") == 0))
            {
              *first = atoi (keyword->string);
              if (*first < 0)
                *first = 0;
              break;
            }
          point++;
        }
    }

  point = (keyword_t**) split->pdata;
  if (max)
    {
      *max = -1;
      while (*point)
        {
          keyword_t *keyword;

          keyword = *point;
          if (keyword->column && (strcmp (keyword->column, "rows") == 0))
            {
              *max = atoi (keyword->string);
              if (*max == -2)
                setting_value_int (SETTING_UUID_ROWS_PER_PAGE, max);
              else if (*max < 1)
                *max = -1;
              break;
            }
          point++;
        }
    }

  point = (keyword_t**) split->pdata;
  if (sort_field || sort_order)
    {
      if (sort_field) *sort_field = NULL;
      if (sort_order) *sort_order = 1;
      while (*point)
        {
          keyword_t *keyword;

          keyword = *point;
          if (keyword->column
              && (strcmp (keyword->column, "sort") == 0))
            {
              if (sort_field) *sort_field = g_strdup (keyword->string);
              if (sort_order) *sort_order = 1;
              break;
            }
          if (keyword->column
              && (strcmp (keyword->column, "sort-reverse") == 0))
            {
              if (sort_field) *sort_field = g_strdup (keyword->string);
              if (sort_order) *sort_order = 0;
              break;
            }
          point++;
        }
      if (sort_field && (*sort_field == NULL))
        *sort_field = g_strdup ("name");
    }

  filter_free (split);
  return;
}

/**
 * @brief Get an int column from a filter split.
 *
 * @param[in]  point   Filter split.
 * @param[in]  column  Name of column.
 * @param[out] val     Value of column.
 *
 * @return 0 success, 1 fail.
 */
int
filter_control_int (keyword_t **point, const char *column, int *val)
{
  if (val)
    while (*point)
      {
        keyword_t *keyword;

        keyword = *point;
        if (keyword->column
            && (strcmp (keyword->column, column) == 0))
          {
            *val = atoi (keyword->string);
            return 0;
          }
        point++;
      }
  return 1;
}

/**
 * @brief Get a string column from a filter split.
 *
 * @param[in]  point   Filter split.
 * @param[in]  column  Name of column.
 * @param[out] string  Value of column, freshly allocated.
 *
 * @return 0 success, 1 fail.
 */
int
filter_control_str (keyword_t **point, const char *column, gchar **string)
{
  if (string)
    while (*point)
      {
        keyword_t *keyword;

        keyword = *point;
        if (keyword->column
            && (strcmp (keyword->column, column) == 0))
          {
            *string = g_strdup (keyword->string);
            return 0;
          }
        point++;
      }
  return 1;
}

/**
 * @brief Get info from a filter for report.
 *
 * It's up to the caller to ensure that max is adjusted for Max Rows Per Page
 * (by calling manage_max_rows).
 *
 * @param[in]   filter      Filter.
 * @param[out]  first       Number of first item.
 * @param[out]  max         Max number of rows.
 * @param[out]  sort_field  Sort field.
 * @param[out]  sort_order  Sort order.
 * @param[out]  result_hosts_only  Whether to show only hosts with results.
 * @param[out]  min_qod        Minimum QoD base of included results.  All
 *                              results if NULL.
 * @param[out]  levels         String describing threat levels (message types)
 *                             to include in count (for example, "hmlgd" for
 *                             High, Medium, Low, loG and Debug).  All levels if
 *                             NULL.
 * @param[out]  delta_states   String describing delta states to include in count
 *                             (for example, "sngc" Same, New, Gone and Changed).
 *                             All levels if NULL.
 * @param[out]  search_phrase      Phrase that results must include.  All results
 *                                 if NULL or "".
 * @param[out]  search_phrase_exact  Whether search phrase is exact.
 * @param[out]  autofp             Whether to apply auto FP filter.
 * @param[out]  notes              Whether to include notes.
 * @param[out]  overrides          Whether to include overrides.
 * @param[out]  apply_overrides    Whether to apply overrides.
 * @param[out]  zone               Timezone.
 */
void
manage_report_filter_controls (const gchar *filter, int *first, int *max,
                               gchar **sort_field, int *sort_order,
                               int *result_hosts_only, gchar **min_qod,
                               gchar **levels, gchar **delta_states,
                               gchar **search_phrase, int *search_phrase_exact,
                               int *autofp, int *notes, int *overrides,
                               int *apply_overrides, gchar **zone)
{
  keyword_t **point;
  array_t *split;
  int val;
  gchar *string;

  if (filter == NULL)
    return;

  split = split_filter (filter);
  point = (keyword_t**) split->pdata;
  if (first)
    {
      *first = 1;
      while (*point)
        {
          keyword_t *keyword;

          keyword = *point;
          if (keyword->column && (strcmp (keyword->column, "first") == 0))
            {
              *first = atoi (keyword->string);
              if (*first < 0)
                *first = 0;
              break;
            }
          point++;
        }
      /* Switch from 1 to 0 indexing. */

      (*first)--;
    }

  point = (keyword_t**) split->pdata;
  if (max)
    {
      *max = 100;
      while (*point)
        {
          keyword_t *keyword;

          keyword = *point;
          if (keyword->column && (strcmp (keyword->column, "rows") == 0))
            {
              *max = atoi (keyword->string);
              if (*max == -2)
                setting_value_int (SETTING_UUID_ROWS_PER_PAGE, max);
              else if (*max < 1)
                *max = -1;
              break;
            }
          point++;
        }
    }

  point = (keyword_t**) split->pdata;
  if (sort_field || sort_order)
    {
      if (sort_field) *sort_field = NULL;
      if (sort_order) *sort_order = 1;
      while (*point)
        {
          keyword_t *keyword;

          keyword = *point;
          if (keyword->column
              && (strcmp (keyword->column, "sort") == 0))
            {
              if (sort_field) *sort_field = g_strdup (keyword->string);
              if (sort_order) *sort_order = 1;
              break;
            }
          if (keyword->column
              && (strcmp (keyword->column, "sort-reverse") == 0))
            {
              if (sort_field) *sort_field = g_strdup (keyword->string);
              if (sort_order) *sort_order = 0;
              break;
            }
          point++;
        }
      if (sort_field && (*sort_field == NULL))
        // FIX name??
        *sort_field = g_strdup ("name");
    }

  if (search_phrase)
    {
      GString *phrase;
      phrase = g_string_new ("");
      point = (keyword_t**) split->pdata;
      if (search_phrase_exact)
        *search_phrase_exact = 0;
      while (*point)
        {
          keyword_t *keyword;

          keyword = *point;
          if (keyword->column == NULL)
            {
              if (search_phrase_exact && keyword->equal)
                /* If one term is "exact" then the search is "exact", because
                 * for reports the filter terms are combined into a single
                 * search term. */
                *search_phrase_exact = 1;
              g_string_append_printf (phrase, "%s ", keyword->string);
            }
          point++;
        }
      *search_phrase = g_strchomp (phrase->str);
      g_string_free (phrase, FALSE);
    }

  if (result_hosts_only)
    {
      if (filter_control_int ((keyword_t **) split->pdata,
                              "result_hosts_only",
                              &val))
        *result_hosts_only = 1;
      else
        *result_hosts_only = val;
    }

  if (autofp)
    {
      if (filter_control_int ((keyword_t **) split->pdata,
                              "autofp",
                              &val))
        *autofp = 0;
      else
        *autofp = val;
    }

  if (notes)
    {
      if (filter_control_int ((keyword_t **) split->pdata,
                              "notes",
                              &val))
        *notes = 1;
      else
        *notes = val;
    }

  if (overrides)
    {
      if (filter_control_int ((keyword_t **) split->pdata,
                              "overrides",
                              &val))
        *overrides = 1;
      else
        *overrides = val;
    }

  if (apply_overrides)
    {
      if (filter_control_int ((keyword_t **) split->pdata,
                              "apply_overrides",
                              &val))
        {
          if (filter_control_int ((keyword_t **) split->pdata,
                                  "overrides",
                                  &val))
            *apply_overrides = 1;
          else
            *apply_overrides = val;
        }
      else
        *apply_overrides = val;
    }

  if (delta_states)
    {
      if (filter_control_str ((keyword_t **) split->pdata,
                              "delta_states",
                              &string))
        *delta_states = NULL;
      else
        *delta_states = string;
    }

  if (levels)
    {
      if (filter_control_str ((keyword_t **) split->pdata,
                              "levels",
                              &string))
        *levels = NULL;
      else
        *levels = string;
    }

  if (min_qod)
    {
      if (filter_control_str ((keyword_t **) split->pdata,
                              "min_qod",
                              &string))
        *min_qod = NULL;
      else
        *min_qod = string;
    }

  if (zone)
    {
      if (filter_control_str ((keyword_t **) split->pdata,
                              "timezone",
                              &string))
        *zone = NULL;
      else
        *zone = string;
    }

  filter_free (split);
  return;
}

/**
 * @brief Append relation to filter.
 *
 * @param[in]  clean     Filter.
 * @param[in]  keyword   Keyword
 * @param[in]  relation  Relation char.
 */
static void
append_relation (GString *clean, keyword_t *keyword, const char relation)
{
  if (strcmp (keyword->column, "rows") == 0)
    {
      int max;

      if (strcmp (keyword->string, "-2") == 0)
        setting_value_int (SETTING_UUID_ROWS_PER_PAGE, &max);
      else
        max = atoi (keyword->string);

      g_string_append_printf (clean,
                              " %s%c%i",
                              keyword->column,
                              relation,
                              manage_max_rows (max));
    }
  else if (keyword->quoted)
    g_string_append_printf (clean,
                            " %s%c\"%s\"",
                            keyword->column,
                            relation,
                            keyword->string);
  else
    g_string_append_printf (clean,
                            " %s%c%s",
                            keyword->column,
                            relation,
                            keyword->string);
}

/**
 * @brief Clean a filter, removing a keyword in the process.
 *
 * @param[in]  filter  Filter.
 * @param[in]  column  Keyword to remove, or NULL.
 *
 * @return Cleaned filter.
 */
gchar *
manage_clean_filter_remove (const gchar *filter, const gchar *column)
{
  GString *clean;
  keyword_t **point;
  array_t *split;

  if (filter == NULL)
    return g_strdup ("");

  clean = g_string_new ("");
  split = split_filter (filter);
  point = (keyword_t**) split->pdata;
  while (*point)
    {
      keyword_t *keyword;

      keyword = *point;
      if (keyword->column
          && column
          && strlen (column)
          && ((strcasecmp (keyword->column, column) == 0)
              || (keyword->column[0] == '_'
                  && strcasecmp (keyword->column + 1, column) == 0)))
        {
          /* Remove this keyword. */;
        }
      else if (keyword->column)
        switch (keyword->relation)
          {
            case KEYWORD_RELATION_COLUMN_EQUAL:
              append_relation (clean, keyword, '=');
              break;
            case KEYWORD_RELATION_COLUMN_APPROX:
              append_relation (clean, keyword, '~');
              break;
            case KEYWORD_RELATION_COLUMN_ABOVE:
              append_relation (clean, keyword, '>');
              break;
            case KEYWORD_RELATION_COLUMN_BELOW:
              append_relation (clean, keyword, '<');
              break;
            case KEYWORD_RELATION_COLUMN_REGEXP:
              append_relation (clean, keyword, ':');
              break;

            case KEYWORD_RELATION_APPROX:
              if (keyword->quoted)
                g_string_append_printf (clean, " \"%s\"", keyword->string);
              else
                g_string_append_printf (clean, " %s", keyword->string);
              break;
          }
      else
        if (keyword->quoted)
          g_string_append_printf (clean, " %s\"%s\"",
                                  keyword->equal ? "=" : "",
                                  keyword->string);
        else
          g_string_append_printf (clean, " %s%s",
                                  keyword->equal ? "=" : "",
                                  keyword->string);
      point++;
    }
  filter_free (split);
  return g_strstrip (g_string_free (clean, FALSE));
}

/**
 * @brief Clean a filter.
 *
 * @param[in]  filter  Filter.
 *
 * @return Cleaned filter.
 */
gchar *
manage_clean_filter (const gchar *filter)
{
  return manage_clean_filter_remove (filter, NULL);
}

/**
 * @brief Return SQL join words for filter_clause.
 *
 * @param[in]  first         Whether keyword is first.
 * @param[in]  last_was_and  Whether last keyword was "and".
 * @param[in]  last_was_not  Whether last keyword was "not".
 *
 * @return SQL join words.
 */
static const char *
get_join (int first, int last_was_and, int last_was_not)
{
  const char *pre;
  if (first)
    {
      if (last_was_not)
        pre = "NOT ";
      else
        pre = "";
    }
  else
    {
      if (last_was_and)
        {
          if (last_was_not)
            pre = " AND NOT ";
          else
            pre = " AND ";
        }
      else
        {
          if (last_was_not)
            pre = " OR NOT ";
          else
            pre = " OR ";
        }
    }
  return pre;
}

/**
 * @brief Get the column expression for a filter column.
 *
 * @param[in]  select_columns  SELECT columns.
 * @param[in]  filter_column   Filter column.
 * @param[out] type            Type of returned column.
 *
 * @return Column for the SELECT statement.
 */
static gchar *
columns_select_column_single (column_t *select_columns,
                              const char *filter_column,
                              keyword_type_t* type)
{
  column_t *columns;
  if (type)
    *type = KEYWORD_TYPE_UNKNOWN;
  if (select_columns == NULL)
    return NULL;
  columns = select_columns;
  while ((*columns).select)
    {
      if ((*columns).filter
          && strcmp ((*columns).filter, filter_column) == 0)
        {
          if (type)
            *type = (*columns).type;
          return (*columns).select;
        }
      if ((*columns).filter
          && *((*columns).filter)
          && *((*columns).filter) == '_'
          && strcmp (((*columns).filter) + 1, filter_column) == 0)
        {
          if (type)
            *type = (*columns).type;
          return (*columns).select;
        }
      columns++;
    }
  columns = select_columns;
  while ((*columns).select)
    {
      if (strcmp ((*columns).select, filter_column) == 0)
        {
          if (type)
            *type = (*columns).type;
          return (*columns).select;
        }
      columns++;
    }
  return NULL;
}

/**
 * @brief Get the selection term for a filter column.
 *
 * @param[in]  select_columns  SELECT columns.
 * @param[in]  where_columns   WHERE "columns".
 * @param[in]  filter_column   Filter column.
 *
 * @return Column for the SELECT statement.
 */
static gchar *
columns_select_column (column_t *select_columns,
                       column_t *where_columns,
                       const char *filter_column)
{
  gchar *column;
  column = columns_select_column_single (select_columns, filter_column, NULL);
  if (column)
    return column;
  return columns_select_column_single (where_columns, filter_column, NULL);
}

/**
 * @brief Get the selection term for a filter column.
 *
 * @param[in]  select_columns  SELECT columns.
 * @param[in]  where_columns   WHERE "columns".
 * @param[in]  filter_column   Filter column.
 * @param[out] type            Type of the returned column.
 *
 * @return Column for the SELECT statement.
 */
static gchar *
columns_select_column_with_type (column_t *select_columns,
                                 column_t *where_columns,
                                 const char *filter_column,
                                 keyword_type_t* type)
{
  gchar *column;
  column = columns_select_column_single (select_columns, filter_column, type);
  if (column)
    return column;
  return columns_select_column_single (where_columns, filter_column, type);
}

/**
 * @brief Return column list for SELECT statement.
 *
 * @param[in]  select_columns  SELECT columns.
 *
 * @return Column list for the SELECT statement.
 */
static gchar *
columns_build_select (column_t *select_columns)
{
  if (select_columns == NULL)
    return g_strdup ("''");

  if ((*select_columns).select)
    {
      column_t *columns;
      GString *select;

      columns = select_columns;
      select = g_string_new ("");
      g_string_append (select, (*columns).select);
      if ((*columns).filter)
        g_string_append_printf (select, " AS %s", (*columns).filter);
      columns++;
      while ((*columns).select)
       {
         g_string_append_printf (select, ", %s", (*columns).select);
         if ((*columns).filter)
           g_string_append_printf (select, " AS %s", (*columns).filter);
         columns++;
       }
      return g_string_free (select, FALSE);
    }
  return g_strdup ("''");
}

/**
 * @brief Check whether a keyword applies to a column.
 *
 * @param[in]  keyword  Keyword.
 * @param[in]  column   Column.
 *
 * @return 1 if applies, else 0.
 */
static int
keyword_applies_to_column (keyword_t *keyword, const char* column)
{
  if ((strcmp (column, "threat") == 0)
      && (strstr ("None", keyword->string) == NULL)
      && (strstr ("False Positive", keyword->string) == NULL)
      && (strstr ("Debug", keyword->string) == NULL)
      && (strstr ("Error", keyword->string) == NULL)
      && (strstr ("Alarm", keyword->string) == NULL)
      && (strstr ("High", keyword->string) == NULL)
      && (strstr ("Medium", keyword->string) == NULL)
      && (strstr ("Low", keyword->string) == NULL)
      && (strstr ("Log", keyword->string) == NULL))
    return 0;
  if ((strcmp (column, "trend") == 0)
      && (strstr ("more", keyword->string) == NULL)
      && (strstr ("less", keyword->string) == NULL)
      && (strstr ("up", keyword->string) == NULL)
      && (strstr ("down", keyword->string) == NULL)
      && (strstr ("same", keyword->string) == NULL))
    return 0;
  if ((strcmp (column, "status") == 0)
      && (strstr ("Delete Requested", keyword->string) == NULL)
      && (strstr ("Ultimate Delete Requested", keyword->string) == NULL)
      && (strstr ("Done", keyword->string) == NULL)
      && (strstr ("New", keyword->string) == NULL)
      && (strstr ("Running", keyword->string) == NULL)
      && (strstr ("Stop Requested", keyword->string) == NULL)
      && (strstr ("Stopped", keyword->string) == NULL)
      && (strstr ("Internal Error", keyword->string) == NULL))
    return 0;
  return 1;
}

/**
 * @brief Return SQL WHERE clause for restricting a SELECT to a filter term.
 *
 * @param[in]  type     Resource type.
 * @param[in]  filter   Filter term.
 * @param[in]  filter_columns  Filter columns.
 * @param[in]  select_columns  SELECT columns.
 * @param[in]  where_columns   Columns in SQL that only appear in WHERE clause.
 * @param[out] trash           Whether the trash table is being queried.
 * @param[out] order_return  If given then order clause.
 * @param[out] first_return  If given then first row.
 * @param[out] max_return    If given then max rows.
 * @param[out] permissions   When given then permissions string vector.
 * @param[out] owner_filter  When given then value of owner keyword.
 *
 * @return WHERE clause for filter if one is required, else NULL.
 */
static gchar *
filter_clause (const char* type, const char* filter,
               const char **filter_columns, column_t *select_columns,
               column_t *where_columns, int trash, gchar **order_return,
               int *first_return, int *max_return, array_t **permissions,
               gchar **owner_filter)
{
  GString *clause, *order;
  keyword_t **point;
  int first_keyword, first_order, last_was_and, last_was_not, last_was_re, skip;
  array_t *split;

  if (filter == NULL)
    filter = "";

  while (*filter && isspace (*filter)) filter++;

  if (permissions)
    *permissions = make_array ();

  if (owner_filter)
    *owner_filter = NULL;

  /* Add SQL to the clause for each keyword or phrase. */

  if (max_return)
    *max_return = -1;

  clause = g_string_new ("");
  order = g_string_new ("");
  /* NB This may add terms that are missing, like "sort". */
  split = split_filter (filter);
  point = (keyword_t**) split->pdata;
  first_keyword = 1;
  last_was_and = 0;
  last_was_not = 0;
  last_was_re = 0;
  first_order = 1;
  while (*point)
    {
      gchar *quoted_keyword;
      int index;
      keyword_t *keyword;

      skip = 0;

      keyword = *point;

      if ((keyword->column == NULL)
          && (strlen (keyword->string) == 0))
        {
          point++;
          continue;
        }

      if ((keyword->column == NULL)
          && (strcasecmp (keyword->string, "or") == 0))
        {
          point++;
          continue;
        }

      if ((keyword->column == NULL)
          && (strcasecmp (keyword->string, "and") == 0))
        {
          last_was_and = 1;
          point++;
          continue;
        }

      if ((keyword->column == NULL)
          && (strcasecmp (keyword->string, "not") == 0))
        {
          last_was_not = 1;
          point++;
          continue;
        }

      if ((keyword->column == NULL)
          && (strcasecmp (keyword->string, "re") == 0))
        {
          last_was_re = 1;
          point++;
          continue;
        }

      if ((keyword->column == NULL)
          && (strcasecmp (keyword->string, "regexp") == 0))
        {
          last_was_re = 1;
          point++;
          continue;
        }

      /* Check for ordering parts, like sort=name or sort-reverse=string. */

      if (keyword->column && (strcasecmp (keyword->column, "sort") == 0))
        {
          if (vector_find_filter (filter_columns, keyword->string) == 0)
            {
              point++;
              continue;
            }

          if (first_order)
            {
              if ((strcmp (type, "task") == 0)
                  && (strcmp (keyword->string, "threat") == 0))
                {
                  gchar *column;
                  column = columns_select_column (select_columns,
                                                  where_columns,
                                                  keyword->string);
                  assert (column);
                  g_string_append_printf (order,
                                          " ORDER BY order_threat (%s) ASC",
                                          column);
                }
              else if (strcmp (keyword->string, "severity") == 0
                       || strcmp (keyword->string, "original_severity") == 0
                       || strcmp (keyword->string, "cvss") == 0
                       || strcmp (keyword->string, "cvss_base") == 0
                       || strcmp (keyword->string, "max_cvss") == 0
                       || strcmp (keyword->string, "fp_per_host") == 0
                       || strcmp (keyword->string, "log_per_host") == 0
                       || strcmp (keyword->string, "low_per_host") == 0
                       || strcmp (keyword->string, "medium_per_host") == 0
                       || strcmp (keyword->string, "high_per_host") == 0)
                {
                  gchar *column;
                  column = columns_select_column (select_columns,
                                                  where_columns,
                                                  keyword->string);
                  g_string_append_printf (order,
                                          " ORDER BY CASE CAST (%s AS text)"
                                          " WHEN '' THEN NULL"
                                          " ELSE CAST (%s AS REAL) END ASC",
                                          column,
                                          column);
                }
              else if (strcmp (keyword->string, "roles") == 0)
                {
                  gchar *column;
                  column = columns_select_column (select_columns,
                                                  where_columns,
                                                  keyword->string);
                  assert (column);
                  g_string_append_printf (order,
                                          " ORDER BY"
                                          " CASE WHEN %s %s 'Admin.*'"
                                          " THEN '0' || %s"
                                          " ELSE '1' || %s END ASC",
                                          column,
                                          sql_regexp_op (),
                                          column,
                                          column);
                }
              else if ((strcmp (keyword->string, "created") == 0)
                       || (strcmp (keyword->string, "modified") == 0)
                       || (strcmp (keyword->string, "published") == 0)
                       || (strcmp (keyword->string, "qod") == 0)
                       || (strcmp (keyword->string, "cves") == 0)
                       || (strcmp (keyword->string, "high") == 0)
                       || (strcmp (keyword->string, "medium") == 0)
                       || (strcmp (keyword->string, "low") == 0)
                       || (strcmp (keyword->string, "log") == 0)
                       || (strcmp (keyword->string, "false_positive") == 0)
                       || (strcmp (keyword->string, "hosts") == 0)
                       || (strcmp (keyword->string, "result_hosts") == 0)
                       || (strcmp (keyword->string, "latest_severity") == 0)
                       || (strcmp (keyword->string, "highest_severity") == 0)
                       || (strcmp (keyword->string, "average_severity") == 0))
                {
                  gchar *column;
                  column = columns_select_column (select_columns,
                                                  where_columns,
                                                  keyword->string);
                  assert (column);
                  g_string_append_printf (order,
                                          " ORDER BY %s ASC",
                                          column);
                }
              else if ((strcmp (keyword->string, "ips") == 0)
                       || (strcmp (keyword->string, "total") == 0)
                       || (strcmp (keyword->string, "tcp") == 0)
                       || (strcmp (keyword->string, "udp") == 0))
                {
                  gchar *column;
                  column = columns_select_column (select_columns,
                                                  where_columns,
                                                  keyword->string);
                  assert (column);
                  g_string_append_printf (order,
                                          " ORDER BY CAST (%s AS INTEGER) ASC",
                                          column);
                }
              else if (strcmp (keyword->string, "ip") == 0)
                {
                  gchar *column;
                  column = columns_select_column (select_columns,
                                                  where_columns,
                                                  keyword->string);
                  assert (column);
                  g_string_append_printf (order,
                                          " ORDER BY order_inet (%s) ASC",
                                          column);
                }
              else if ((strcmp (type, "note")
                        && strcmp (type, "override"))
                       || (strcmp (keyword->string, "nvt")
                           && strcmp (keyword->string, "name")))
                {
                  gchar *column;
                  column = columns_select_column (select_columns,
                                                  where_columns,
                                                  keyword->string);
                  assert (column);
                  g_string_append_printf (order, " ORDER BY lower (%s) ASC",
                                          column);
                }
              else
                /* Special case for notes text sorting. */
                g_string_append_printf (order,
                                        " ORDER BY nvt ASC,"
                                        "          lower (%ss%s.text) ASC",
                                        type,
                                        trash ? "_trash" : "");
              first_order = 0;
            }
          else
            /* To help the client split_filter restricts the filter to one
             * sorting term, preventing this from happening. */
            g_string_append_printf (order, ", %s ASC",
                                    keyword->string);
          point++;
          continue;
        }
      else if (keyword->column
               && (strcasecmp (keyword->column, "sort-reverse") == 0))
        {
          if (vector_find_filter (filter_columns, keyword->string) == 0)
            {
              point++;
              continue;
            }

          if (first_order)
            {
              if ((strcmp (type, "task") == 0)
                  && (strcmp (keyword->string, "threat") == 0))
                {
                  gchar *column;
                  column = columns_select_column (select_columns,
                                                  where_columns,
                                                  keyword->string);
                  assert (column);
                  g_string_append_printf (order,
                                          " ORDER BY order_threat (%s) DESC",
                                          column);
                }
              else if (strcmp (keyword->string, "severity") == 0
                       || strcmp (keyword->string, "original_severity") == 0
                       || strcmp (keyword->string, "cvss") == 0
                       || strcmp (keyword->string, "cvss_base") == 0
                       || strcmp (keyword->string, "max_cvss") == 0
                       || strcmp (keyword->string, "fp_per_host") == 0
                       || strcmp (keyword->string, "log_per_host") == 0
                       || strcmp (keyword->string, "low_per_host") == 0
                       || strcmp (keyword->string, "medium_per_host") == 0
                       || strcmp (keyword->string, "high_per_host") == 0)
                {
                  gchar *column;
                  column = columns_select_column (select_columns,
                                                  where_columns,
                                                  keyword->string);
                  g_string_append_printf (order,
                                          " ORDER BY CASE CAST (%s AS text)"
                                          " WHEN '' THEN NULL"
                                          " ELSE CAST (%s AS REAL) END DESC",
                                          column,
                                          column);
                }
              else if (strcmp (keyword->string, "roles") == 0)
                {
                  gchar *column;
                  column = columns_select_column (select_columns,
                                                  where_columns,
                                                  keyword->string);
                  assert (column);
                  g_string_append_printf (order,
                                          " ORDER BY"
                                          " CASE WHEN %s %s 'Admin.*'"
                                          " THEN '0' || %s"
                                          " ELSE '1' || %s END DESC",
                                          column,
                                          sql_regexp_op (),
                                          column,
                                          column);
                }
              else if ((strcmp (keyword->string, "created") == 0)
                       || (strcmp (keyword->string, "modified") == 0)
                       || (strcmp (keyword->string, "published") == 0)
                       || (strcmp (keyword->string, "qod") == 0)
                       || (strcmp (keyword->string, "cves") == 0)
                       || (strcmp (keyword->string, "high") == 0)
                       || (strcmp (keyword->string, "medium") == 0)
                       || (strcmp (keyword->string, "low") == 0)
                       || (strcmp (keyword->string, "log") == 0)
                       || (strcmp (keyword->string, "false_positive") == 0)
                       || (strcmp (keyword->string, "hosts") == 0)
                       || (strcmp (keyword->string, "result_hosts") == 0)
                       || (strcmp (keyword->string, "latest_severity") == 0)
                       || (strcmp (keyword->string, "highest_severity") == 0)
                       || (strcmp (keyword->string, "average_severity") == 0))
                {
                  gchar *column;
                  column = columns_select_column (select_columns,
                                                  where_columns,
                                                  keyword->string);
                  assert (column);
                  g_string_append_printf (order,
                                          " ORDER BY %s DESC",
                                          column);
                }
              else if ((strcmp (keyword->string, "ips") == 0)
                       || (strcmp (keyword->string, "total") == 0)
                       || (strcmp (keyword->string, "tcp") == 0)
                       || (strcmp (keyword->string, "udp") == 0))
                {
                  gchar *column;
                  column = columns_select_column (select_columns,
                                                  where_columns,
                                                  keyword->string);
                  assert (column);
                  g_string_append_printf (order,
                                          " ORDER BY CAST (%s AS INTEGER) DESC",
                                          column);
                }
              else if (strcmp (keyword->string, "ip") == 0)
                {
                  gchar *column;
                  column = columns_select_column (select_columns,
                                                  where_columns,
                                                  keyword->string);
                  assert (column);
                  g_string_append_printf (order,
                                          " ORDER BY order_inet (%s) DESC",
                                          column);
                }
              else if ((strcmp (type, "note")
                        && strcmp (type, "override"))
                       || (strcmp (keyword->string, "nvt")
                           && strcmp (keyword->string, "name")))
                {
                  gchar *column;
                  g_debug ("   %s: select_columns: %p", __FUNCTION__, select_columns);
                  g_debug ("   %s: where_columns: %p", __FUNCTION__, where_columns);
                  g_debug ("   %s: keyword->string: %p", __FUNCTION__, keyword->string);
                  column = columns_select_column (select_columns,
                                                  where_columns,
                                                  keyword->string);
                  assert (column);
                  g_string_append_printf (order, " ORDER BY lower (%s) DESC",
                                          column);
                }
              else
                /* Special case for notes text sorting. */
                g_string_append_printf (order,
                                        " ORDER BY nvt DESC,"
                                        "          lower (%ss%s.text) DESC",
                                        type,
                                        trash ? "_trash" : "");
              first_order = 0;
            }
          else
            /* To help the client split_filter restricts the filter to one
             * sorting term, preventing this from happening. */
            g_string_append_printf (order, ", %s DESC",
                                    keyword->string);
          point++;
          continue;
        }
      else if (keyword->column
               && (strcasecmp (keyword->column, "first") == 0))
        {
          if (first_return)
            {
              /* Subtract 1 to switch from 1 to 0 indexing. */
              *first_return = atoi (keyword->string) - 1;
              if (*first_return < 0)
                *first_return = 0;
            }

          point++;
          continue;
        }
      else if (keyword->column
               && (strcasecmp (keyword->column, "rows") == 0))
        {
          if (max_return)
            *max_return = atoi (keyword->string);

          point++;
          continue;
        }
      else if (keyword->column
               && (strcasecmp (keyword->column, "permission") == 0))
        {
          if (permissions)
            array_add (*permissions, g_strdup (keyword->string));

          point++;
          continue;
        }
      /* Add tag criteria to clause: tag name with optional value */
      else if (keyword->column
               && (strcasecmp (keyword->column, "tag") == 0))
        {
          gchar **tag_split, *tag_name, *tag_value;
          int value_given, value_numeric;

          quoted_keyword = sql_quote (keyword->string);
          tag_split = g_strsplit (quoted_keyword, "=", 2);
          tag_name = g_strdup (tag_split[0] ? tag_split[0] : "");

          if (tag_split[0] && tag_split[1])
            {
              tag_value = g_strdup (tag_split[1]);
              value_given = 1;
            }
          else
            {
              tag_value = g_strdup ("");
              value_given = 0;
            }

          value_numeric = 0;
          if (value_given)
            {
              double test_d;
              value_numeric = (sscanf (tag_value, "%lf", &test_d) > 0);
            }

          if (keyword->relation == KEYWORD_RELATION_COLUMN_EQUAL
              || keyword->relation == KEYWORD_RELATION_COLUMN_ABOVE
              || keyword->relation == KEYWORD_RELATION_COLUMN_BELOW)
            {
              if (strcasecmp (type, "allinfo") == 0)
                g_string_append_printf (clause,
                                        "%s"
                                        "(EXISTS"
                                        "  (SELECT * FROM tags"
                                        "   WHERE tags.name = '%s'"
                                        "   AND tags.active != 0"
                                        "   AND tags.resource_uuid"
                                        "         = allinfo.uuid"
                                        "   AND tags.resource_type"
                                        "         = allinfo.type"
                                        "   %s%s%s))",
                                        get_join (first_keyword, last_was_and,
                                                  last_was_not),
                                        tag_name,
                                        (value_given
                                          ? (value_numeric
                                              ? "AND CAST"
                                                " (tags.value AS REAL) = "
                                              : "AND CAST"
                                                " (tags.value AS TEXT) = '")
                                          : ""),
                                        value_given ? tag_value : "",
                                        (value_given && value_numeric == 0
                                          ? "'"
                                          : ""));
              else
                g_string_append_printf (clause,
                                        "%s"
                                        "(EXISTS"
                                        "  (SELECT * FROM tags"
                                        "   WHERE tags.name = '%s'"
                                        "   AND tags.active != 0"
                                        "   AND tags.resource_uuid = %ss.uuid"
                                        "   AND tags.resource_type = '%s'"
                                        "   %s%s%s))",
                                        get_join (first_keyword, last_was_and,
                                                  last_was_not),
                                        tag_name,
                                        type,
                                        type,
                                        (value_given
                                          ? (value_numeric
                                              ? "AND CAST"
                                                " (tags.value AS REAL) = "
                                              : "AND CAST"
                                                " (tags.value AS TEXT) = '")
                                          : ""),
                                        value_given ? tag_value : "",
                                        (value_given && value_numeric == 0
                                          ? "'"
                                          : ""));
            }
          else if (keyword->relation == KEYWORD_RELATION_COLUMN_APPROX)
            {
              if (strcasecmp (type, "allinfo") == 0)
                g_string_append_printf (clause,
                                        "%s"
                                        "(EXISTS"
                                        "  (SELECT * FROM tags"
                                        "   WHERE tags.name %s '%%%%%s%%%%'"
                                        "   AND tags.active != 0"
                                        "   AND tags.resource_uuid"
                                        "         = allinfo.uuid"
                                        "   AND tags.resource_type"
                                        "         = allinfo.type"
                                        "   AND tags.value %s '%%%%%s%%%%'))",
                                        get_join (first_keyword, last_was_and,
                                                  last_was_not),
                                        sql_ilike_op (),
                                        tag_name,
                                        sql_ilike_op (),
                                        tag_value);
              else
                g_string_append_printf (clause,
                                        "%s"
                                        "(EXISTS"
                                        "  (SELECT * FROM tags"
                                        "   WHERE tags.name %s '%%%%%s%%%%'"
                                        "   AND tags.active != 0"
                                        "   AND tags.resource_uuid = %ss.uuid"
                                        "   AND tags.resource_type = '%s'"
                                        "   AND tags.value %s '%%%%%s%%%%'))",
                                        get_join (first_keyword, last_was_and,
                                                  last_was_not),
                                        sql_ilike_op (),
                                        tag_name,
                                        type,
                                        type,
                                        sql_ilike_op (),
                                        tag_value);
            }
          else if (keyword->relation == KEYWORD_RELATION_COLUMN_REGEXP)
            {
              if (strcasecmp (type, "allinfo") == 0)
                g_string_append_printf (clause,
                                        "%s"
                                        "(EXISTS"
                                        "  (SELECT * FROM tags"
                                        "   WHERE tags.name %s '%s'"
                                        "   AND tags.active != 0"
                                        "   AND tags.resource_uuid"
                                        "         = allinfo.uuid"
                                        "   AND tags.resource_type"
                                        "         = allinfo.type"
                                        "   AND tags.value"
                                        "       %s '%s'))",
                                        get_join (first_keyword, last_was_and,
                                                  last_was_not),
                                        sql_regexp_op (),
                                        tag_name,
                                        sql_regexp_op (),
                                        tag_value);
              else
                g_string_append_printf (clause,
                                        "%s"
                                        "(EXISTS"
                                        "  (SELECT * FROM tags"
                                        "   WHERE tags.name %s '%s'"
                                        "   AND tags.active != 0"
                                        "   AND tags.resource_uuid = %ss.uuid"
                                        "   AND tags.resource_type = '%s'"
                                        "   AND tags.value"
                                        "       %s '%s'))",
                                        get_join (first_keyword, last_was_and,
                                                  last_was_not),
                                        sql_regexp_op (),
                                        tag_name,
                                        type,
                                        type,
                                        sql_regexp_op (),
                                        tag_value);
            }

          g_strfreev(tag_split);
          g_free(tag_name);
          g_free(tag_value);
          first_keyword = 0;
          last_was_and = 0;
          last_was_not = 0;
        }

      /* Add SQL to the clause for each column name. */

      quoted_keyword = NULL;

      if (keyword->relation == KEYWORD_RELATION_COLUMN_EQUAL)
        {
          if (vector_find_filter (filter_columns, keyword->column) == 0)
            {
              last_was_and = 0;
              last_was_not = 0;
              point++;
              continue;
            }

          if (keyword->column
              && (strlen (keyword->column) > 3)
              && (strcmp (keyword->column + strlen (keyword->column) - 3, "_id")
                  == 0)
              && strcasecmp (keyword->column, "nvt_id"))
            {
              gchar *type_term;

              type_term = g_strndup (keyword->column,
                                     strlen (keyword->column) - 3);
              if (valid_type (type_term) == 0)
                {
                  g_free (type_term);
                  last_was_and = 0;
                  last_was_not = 0;
                  point++;
                  continue;
                }

              quoted_keyword = sql_quote (keyword->string);
              if (strcmp (quoted_keyword, ""))
                g_string_append_printf (clause,
                                        "%s(((SELECT id FROM %ss"
                                        "     WHERE %ss.uuid = '%s')"
                                        "     = %ss.%s"
                                        "     OR %ss.%s IS NULL"
                                        "     OR %ss.%s = 0)",
                                        get_join (first_keyword,
                                                  last_was_and,
                                                  last_was_not),
                                        type_term,
                                        type_term,
                                        quoted_keyword,
                                        type,
                                        type_term,
                                        type,
                                        type_term,
                                        type,
                                        type_term);
              else
                g_string_append_printf (clause,
                                        "%s((%ss.%s IS NULL"
                                        "   OR %ss.%s = 0)",
                                        get_join (first_keyword,
                                                  last_was_and,
                                                  last_was_not),
                                        type,
                                        type_term,
                                        type,
                                        type_term);

              g_free (type_term);
            }
          else if (keyword->column && strcmp (keyword->column, "owner"))
            {
              gchar *column;
              keyword_type_t column_type;
              quoted_keyword = sql_quote (keyword->string);
              column = columns_select_column_with_type (select_columns,
                                                        where_columns,
                                                        keyword->column,
                                                        &column_type);
              assert (column);
              if (keyword->type == KEYWORD_TYPE_INTEGER
                  && (column_type == KEYWORD_TYPE_INTEGER
                      || column_type == KEYWORD_TYPE_DOUBLE))
                g_string_append_printf (clause,
                                        "%s(CAST (%s AS NUMERIC) = %i",
                                        get_join (first_keyword, last_was_and,
                                                  last_was_not),
                                        column,
                                        keyword->integer_value);
          else if (keyword->type == KEYWORD_TYPE_DOUBLE
                   && (column_type == KEYWORD_TYPE_DOUBLE
                       || column_type == KEYWORD_TYPE_INTEGER))
                g_string_append_printf (clause,
                                        "%s(CAST (%s AS REAL)"
                                        " = CAST (%f AS REAL)",
                                        get_join (first_keyword, last_was_and,
                                                  last_was_not),
                                        column,
                                        keyword->double_value);
              else if (strcmp (quoted_keyword, ""))
                g_string_append_printf (clause,
                                        "%s(CAST (%s AS TEXT) = '%s'",
                                        get_join (first_keyword, last_was_and,
                                                  last_was_not),
                                        column,
                                        quoted_keyword);
              else
                g_string_append_printf (clause,
                                        "%s((%s IS NULL OR CAST (%s AS TEXT) = '%s')",
                                        get_join (first_keyword, last_was_and,
                                                  last_was_not),
                                        column,
                                        column,
                                        quoted_keyword);
            }
          else
            {
              /* Skip term.  Owner filtering is done via where_owned. */
              skip = 1;
              if (owner_filter && (*owner_filter == NULL))
                *owner_filter = g_strdup (keyword->string);
            }
        }
      else if (keyword->relation == KEYWORD_RELATION_COLUMN_APPROX)
        {
          gchar *column;

          if (vector_find_filter (filter_columns, keyword->column) == 0)
            {
              last_was_and = 0;
              last_was_not = 0;
              point++;
              continue;
            }

          quoted_keyword = sql_quote (keyword->string);
          column = columns_select_column (select_columns,
                                          where_columns,
                                          keyword->column);
          assert (column);
          g_string_append_printf (clause,
                                  "%s(CAST (%s AS TEXT) %s '%%%%%s%%%%'",
                                  get_join (first_keyword, last_was_and,
                                            last_was_not),
                                  column,
                                  sql_ilike_op (),
                                  quoted_keyword);
        }
      else if (keyword->relation == KEYWORD_RELATION_COLUMN_ABOVE)
        {
          gchar *column;
          keyword_type_t column_type;

          if (vector_find_filter (filter_columns, keyword->column) == 0)
            {
              last_was_and = 0;
              last_was_not = 0;
              point++;
              continue;
            }

          quoted_keyword = sql_quote (keyword->string);
          column = columns_select_column_with_type (select_columns,
                                                    where_columns,
                                                    keyword->column,
                                                    &column_type);
          assert (column);
          if (keyword->type == KEYWORD_TYPE_INTEGER
              && (column_type == KEYWORD_TYPE_INTEGER
                  || column_type == KEYWORD_TYPE_DOUBLE))
            g_string_append_printf (clause,
                                    "%s(CAST (%s AS NUMERIC) > %i",
                                    get_join (first_keyword, last_was_and,
                                              last_was_not),
                                    column,
                                    keyword->integer_value);
          else if (keyword->type == KEYWORD_TYPE_DOUBLE
                   && (column_type == KEYWORD_TYPE_DOUBLE
                       || column_type == KEYWORD_TYPE_INTEGER))
            g_string_append_printf (clause,
                                    "%s(CAST (%s AS REAL)"
                                    " > CAST (%f AS REAL)",
                                    get_join (first_keyword, last_was_and,
                                              last_was_not),
                                    column,
                                    keyword->double_value);
          else
            g_string_append_printf (clause,
                                    "%s(CAST (%s AS TEXT) > '%s'",
                                    get_join (first_keyword, last_was_and,
                                              last_was_not),
                                    column,
                                    quoted_keyword);
        }
      else if (keyword->relation == KEYWORD_RELATION_COLUMN_BELOW)
        {
          gchar *column;
          keyword_type_t column_type;

          if (vector_find_filter (filter_columns, keyword->column) == 0)
            {
              last_was_and = 0;
              last_was_not = 0;
              point++;
              continue;
            }

          quoted_keyword = sql_quote (keyword->string);
          column = columns_select_column_with_type (select_columns,
                                                    where_columns,
                                                    keyword->column,
                                                    &column_type);
          assert (column);
          if (keyword->type == KEYWORD_TYPE_INTEGER
              && (column_type == KEYWORD_TYPE_INTEGER
                  || column_type == KEYWORD_TYPE_DOUBLE))
            g_string_append_printf (clause,
                                    "%s(CAST (%s AS NUMERIC) < %i",
                                    get_join (first_keyword, last_was_and,
                                              last_was_not),
                                    column,
                                    keyword->integer_value);
          else if (keyword->type == KEYWORD_TYPE_DOUBLE
                   && (column_type == KEYWORD_TYPE_DOUBLE
                       || column_type == KEYWORD_TYPE_INTEGER))
            g_string_append_printf (clause,
                                    "%s(CAST (%s AS REAL)"
                                    " < CAST (%f AS REAL)",
                                    get_join (first_keyword, last_was_and,
                                              last_was_not),
                                    column,
                                    keyword->double_value);
          else
            g_string_append_printf (clause,
                                    "%s(CAST (%s AS TEXT) < '%s'",
                                    get_join (first_keyword, last_was_and,
                                              last_was_not),
                                    column,
                                    quoted_keyword);
        }
      else if (keyword->relation == KEYWORD_RELATION_COLUMN_REGEXP)
        {
          gchar *column;

          if (vector_find_filter (filter_columns, keyword->column) == 0)
            {
              last_was_and = 0;
              last_was_not = 0;
              point++;
              continue;
            }

          quoted_keyword = sql_quote (keyword->string);
          column = columns_select_column (select_columns,
                                          where_columns,
                                          keyword->column);
          assert (column);
          g_string_append_printf (clause,
                                  "%s(CAST (%s AS TEXT) %s '%s'",
                                  get_join (first_keyword, last_was_and,
                                            last_was_not),
                                  column,
                                  sql_regexp_op (),
                                  quoted_keyword);
        }
      else if (keyword->equal)
        {
          const char *filter_column;

          /* Keyword like "=example". */

          g_string_append_printf (clause,
                                  "%s(",
                                  (first_keyword
                                    ? ""
                                    : (last_was_and ? " AND " : " OR ")));

          quoted_keyword = sql_quote (keyword->string);
          if (last_was_not)
            for (index = 0;
                 (filter_column = filter_columns[index]) != NULL;
                 index++)
              {
                gchar *select_column;
                keyword_type_t column_type;

                select_column = columns_select_column_with_type (select_columns,
                                                                 where_columns,
                                                                 filter_column,
                                                                 &column_type);
                assert (select_column);

                if (keyword->type == KEYWORD_TYPE_INTEGER
                    && (column_type == KEYWORD_TYPE_INTEGER
                        || column_type == KEYWORD_TYPE_DOUBLE))
                  g_string_append_printf (clause,
                                          "%s"
                                          "(%s IS NULL"
                                          " OR CAST (%s AS NUMERIC)"
                                          "    != %i)",
                                          (index ? " AND " : ""),
                                          select_column,
                                          select_column,
                                          keyword->integer_value);
                else if (keyword->type == KEYWORD_TYPE_DOUBLE
                         && (column_type == KEYWORD_TYPE_DOUBLE
                             || column_type == KEYWORD_TYPE_INTEGER))
                  g_string_append_printf (clause,
                                          "%s"
                                          "(%s IS NULL"
                                          " OR CAST (%s AS REAL)"
                                          "    != CAST (%f AS REAL))",
                                          (index ? " AND " : ""),
                                          select_column,
                                          select_column,
                                          keyword->double_value);
                else
                  g_string_append_printf (clause,
                                          "%s"
                                          "(%s IS NULL"
                                          " OR CAST (%s AS TEXT)"
                                          "    != '%s')",
                                          (index ? " AND " : ""),
                                          select_column,
                                          select_column,
                                          quoted_keyword);
              }
          else
            for (index = 0;
                 (filter_column = filter_columns[index]) != NULL;
                 index++)
              {
                gchar *select_column;
                keyword_type_t column_type;

                select_column = columns_select_column_with_type (select_columns,
                                                                 where_columns,
                                                                 filter_column,
                                                                 &column_type);
                assert (select_column);

                if (keyword->type == KEYWORD_TYPE_INTEGER
                    && (column_type == KEYWORD_TYPE_INTEGER
                        || column_type == KEYWORD_TYPE_DOUBLE))
                  g_string_append_printf (clause,
                                          "%sCAST (%s AS NUMERIC)"
                                          " = %i",
                                          (index ? " OR " : ""),
                                          select_column,
                                          keyword->integer_value);
                else if (keyword->type == KEYWORD_TYPE_DOUBLE
                         && (column_type == KEYWORD_TYPE_DOUBLE
                             || column_type == KEYWORD_TYPE_INTEGER))
                  g_string_append_printf (clause,
                                          "%sCAST (%s AS REAL)"
                                          " = CAST (%f AS REAL)",
                                          (index ? " OR " : ""),
                                          select_column,
                                          keyword->double_value);
                else
                  g_string_append_printf (clause,
                                          "%sCAST (%s AS TEXT)"
                                          " = '%s'",
                                          (index ? " OR " : ""),
                                          select_column,
                                          quoted_keyword);
              }
        }
      else
        {
          const char *filter_column;

          g_string_append_printf (clause,
                                  "%s(",
                                  (first_keyword
                                    ? ""
                                    : (last_was_and ? " AND " : " OR ")));

          quoted_keyword = sql_quote (keyword->string);
          if (last_was_not)
            for (index = 0;
                 (filter_column = filter_columns[index]) != NULL;
                 index++)
              {
                gchar *select_column;
                keyword_type_t column_type;
                int column_type_matches = 0;

                select_column = columns_select_column_with_type (select_columns,
                                                                 where_columns,
                                                                 filter_column,
                                                                 &column_type);

                if (column_type != KEYWORD_TYPE_INTEGER
                    && column_type != KEYWORD_TYPE_DOUBLE)
                  column_type_matches = 1;

                if (keyword_applies_to_column (keyword, filter_column)
                    && select_column && column_type_matches)
                  g_string_append_printf (clause,
                                          "%s"
                                          "(%s IS NULL"
                                          " OR CAST (%s AS TEXT)"
                                          "    NOT %s '%s%s%s')",
                                          (index ? " AND " : ""),
                                          select_column,
                                          select_column,
                                          last_was_re
                                           ? sql_regexp_op ()
                                           : sql_ilike_op (),
                                          last_was_re ? "" : "%%",
                                          quoted_keyword,
                                          last_was_re ? "" : "%%");
                else
                  g_string_append_printf (clause,
                                          "%s t ()",
                                          (index ? " AND " : ""));
              }
          else
            for (index = 0;
                 (filter_column = filter_columns[index]) != NULL;
                 index++)
              {
                gchar *select_column;
                keyword_type_t column_type;
                int column_type_matches = 0;

                select_column = columns_select_column_with_type (select_columns,
                                                                 where_columns,
                                                                 filter_column,
                                                                 &column_type);
                if (column_type != KEYWORD_TYPE_INTEGER
                    && column_type != KEYWORD_TYPE_DOUBLE)
                  column_type_matches = 1;

                if (keyword_applies_to_column (keyword, filter_column)
                    && select_column && column_type_matches)
                  g_string_append_printf (clause,
                                          "%sCAST (%s AS TEXT)"
                                          " %s '%s%s%s'",
                                          (index ? " OR " : ""),
                                          select_column,
                                          last_was_re
                                           ? sql_regexp_op ()
                                           : sql_ilike_op (),
                                          last_was_re ? "" : "%%",
                                          quoted_keyword,
                                          last_was_re ? "" : "%%");
                else
                  g_string_append_printf (clause,
                                          "%snot t ()",
                                          (index ? " OR " : ""));
              }
        }

      if (skip == 0)
        {
          g_string_append (clause, ")");
          first_keyword = 0;
          last_was_and = 0;
          last_was_not = 0;
          last_was_re = 0;
        }
      g_free (quoted_keyword);

      point++;
    }
  filter_free (split);

  if (order_return)
    *order_return = g_string_free (order, FALSE);
  else
    g_string_free (order, TRUE);

  if (max_return)
    {
      if (*max_return == -2)
        setting_value_int (SETTING_UUID_ROWS_PER_PAGE, max_return);
      else if (*max_return < 1)
        *max_return = -1;

      *max_return = manage_max_rows (*max_return);
    }

  if (strlen (clause->str))
    return g_string_free (clause, FALSE);

  g_string_free (clause, TRUE);
  return NULL;
}


/* Resources. */

/**
 * @brief Filter columns for GET iterator.
 */
#define ANON_GET_ITERATOR_FILTER_COLUMNS "uuid", \
 "created", "modified", "_owner"

/**
 * @brief Filter columns for GET iterator.
 */
#define GET_ITERATOR_FILTER_COLUMNS "uuid", "name", "comment", \
 "created", "modified", "_owner"

/**
 * @brief Columns for GET iterator, as a single string.
 *
 * @param[in]  prefix  Column prefix.
 */
#define GET_ITERATOR_COLUMNS_STRING                                \
  "id, uuid, name, comment, iso_time (creation_time),"             \
  " iso_time (modification_time), creation_time AS created,"       \
  " modification_time AS modified"

/**
 * @brief Columns for GET iterator.
 *
 * @param[in]  prefix  Column prefix.
 */
#define GET_ITERATOR_COLUMNS_PREFIX(prefix)                                 \
  { prefix "id", NULL, KEYWORD_TYPE_INTEGER },                              \
  { prefix "uuid", NULL, KEYWORD_TYPE_STRING },                             \
  { prefix "name", NULL, KEYWORD_TYPE_STRING },                             \
  { prefix "comment", NULL, KEYWORD_TYPE_STRING },                          \
  { " iso_time (" prefix "creation_time)", NULL, KEYWORD_TYPE_STRING },     \
  { " iso_time (" prefix "modification_time)", NULL, KEYWORD_TYPE_STRING }, \
  { prefix "creation_time", "created", KEYWORD_TYPE_INTEGER },              \
  { prefix "modification_time", "modified", KEYWORD_TYPE_INTEGER }

/**
 * @brief Columns for GET iterator.
 *
 * @param[in]  table  Table.
 */
#define GET_ITERATOR_COLUMNS(table)                                             \
  GET_ITERATOR_COLUMNS_PREFIX(""),                                              \
  {                                                                             \
    "(SELECT name FROM users AS inner_users"                                    \
    " WHERE inner_users.id = " G_STRINGIFY (table) ".owner)",                   \
    "_owner",                                                                   \
    KEYWORD_TYPE_STRING                                                         \
  },                                                                            \
  { "owner", NULL, KEYWORD_TYPE_INTEGER }

/**
 * @brief Number of columns for GET iterator.
 */
#define GET_ITERATOR_COLUMN_COUNT 10

/**
 * @brief Check whether a resource type name is valid.
 *
 * @param[in]  type  Type of resource.
 *
 * @return 1 yes, 0 no.
 */
int
valid_type (const char* type)
{
  return (strcasecmp (type, "agent") == 0)
         || (strcasecmp (type, "alert") == 0)
         || (strcasecmp (type, "asset") == 0)
         || (strcasecmp (type, "config") == 0)
         || (strcasecmp (type, "credential") == 0)
         || (strcasecmp (type, "filter") == 0)
         || (strcasecmp (type, "group") == 0)
         || (strcasecmp (type, "host") == 0)
         || (strcasecmp (type, "info") == 0)
         || (strcasecmp (type, "note") == 0)
         || (strcasecmp (type, "os") == 0)
         || (strcasecmp (type, "override") == 0)
         || (strcasecmp (type, "permission") == 0)
         || (strcasecmp (type, "port_list") == 0)
         || (strcasecmp (type, "report") == 0)
         || (strcasecmp (type, "report_format") == 0)
         || (strcasecmp (type, "result") == 0)
         || (strcasecmp (type, "role") == 0)
         || (strcasecmp (type, "scanner") == 0)
         || (strcasecmp (type, "schedule") == 0)
         || (strcasecmp (type, "tag") == 0)
         || (strcasecmp (type, "target") == 0)
         || (strcasecmp (type, "task") == 0)
         || (strcasecmp (type, "user") == 0);
}

/**
 * @brief Return pretty name of type.
 *
 * @param[in]  type  Database name.
 *
 * @return 1 yes, 0 no.
 */
const char *
type_pretty_name (const char* type)
{
  if (strcasecmp (type, "agent") == 0)
    return "Agent";
  if (strcasecmp (type, "alert") == 0)
    return "Alert";
  if (strcasecmp (type, "asset") == 0)
    return "Asset";
  if (strcasecmp (type, "config") == 0)
    return "Config";
  if (strcasecmp (type, "credential") == 0)
    return "Credential";
  if (strcasecmp (type, "filter") == 0)
    return "Filter";
  if (strcasecmp (type, "note") == 0)
    return "Note";
  if (strcasecmp (type, "override") == 0)
    return "Override";
  if (strcasecmp (type, "permission") == 0)
    return "Permission";
  if (strcasecmp (type, "port_list") == 0)
    return "Port List";
  if (strcasecmp (type, "report") == 0)
    return "Report";
  if (strcasecmp (type, "report_format") == 0)
    return "Report Format";
  if (strcasecmp (type, "result") == 0)
    return "Result";
  if (strcasecmp (type, "role") == 0)
    return "Role";
  if (strcasecmp (type, "scanner") == 0)
    return "Scanner";
  if (strcasecmp (type, "schedule") == 0)
    return "Schedule";
  if (strcasecmp (type, "tag") == 0)
    return "Tag";
  if (strcasecmp (type, "target") == 0)
    return "Target";
  if (strcasecmp (type, "task") == 0)
    return "Task";
  if (strcasecmp (type, "info") == 0)
    return "SecInfo";
  return "";
}

/**
 * @brief Return DB name of type.
 *
 * @param[in]  type  Database or pretty name.
 *
 * @return Database name of type if possible, else NULL.
 */
const char *
type_db_name (const char* type)
{
  if (type == NULL)
    return NULL;

  if (valid_type (type))
    return type;

  if (strcasecmp (type, "Agent") == 0)
    return "agent";
  if (strcasecmp (type, "Alert") == 0)
    return "alert";
  if (strcasecmp (type, "Asset") == 0)
    return "asset";
  if (strcasecmp (type, "Config") == 0)
    return "config";
  if (strcasecmp (type, "Credential") == 0)
    return "credential";
  if (strcasecmp (type, "Filter") == 0)
    return "filter";
  if (strcasecmp (type, "Note") == 0)
    return "note";
  if (strcasecmp (type, "Override") == 0)
    return "override";
  if (strcasecmp (type, "Permission") == 0)
    return "permission";
  if (strcasecmp (type, "Port List") == 0)
    return "port_list";
  if (strcasecmp (type, "Report") == 0)
    return "report";
  if (strcasecmp (type, "Report Format") == 0)
    return "report_format";
  if (strcasecmp (type, "Result") == 0)
    return "result";
  if (strcasecmp (type, "Role") == 0)
    return "role";
  if (strcasecmp (type, "Scanner") == 0)
    return "scanner";
  if (strcasecmp (type, "Schedule") == 0)
    return "schedule";
  if (strcasecmp (type, "Tag") == 0)
    return "tag";
  if (strcasecmp (type, "Target") == 0)
    return "target";
  if (strcasecmp (type, "Task") == 0)
    return "task";
  if (strcasecmp (type, "SecInfo") == 0)
    return "info";
  return NULL;
}

/**
 * @brief Check whether a type has a name and comment.
 *
 * @param[in]  type          Type of resource.
 *
 * @return 1 yes, 0 no.
 */
int
type_named (const char *type)
{
  return strcasecmp (type, "note")
         && strcasecmp (type, "override");
}

/**
 * @brief Check whether a type has a comment.
 *
 * @param[in]  type  Type of resource.
 *
 * @return 1 yes, 0 no.
 */
int
type_has_comment (const char *type)
{
  return strcasecmp (type, "report_format");
}

/**
 * @brief Check whether a resource type has an owner.
 *
 * @param[in]  type  Type of resource.
 *
 * @return 1 yes, 0 no.
 */
int
type_owned (const char* type)
{
  return (strcasecmp (type, "info")
          && strcasecmp (type, "nvt")
          && strcasecmp (type, "cve")
          && strcasecmp (type, "cpe")
          && strcasecmp (type, "ovaldef")
          && strcasecmp (type, "cert_bund_adv")
          && strcasecmp (type, "dfn_cert_adv")
          && strcasecmp (type, "allinfo"));
}

/**
 * @brief Check whether the trash is in the real table.
 *
 * @param[in]  type  Type of resource.
 *
 * @return 1 yes, 0 no.
 */
static int
type_trash_in_table (const char *type)
{
  return strcasecmp (type, "task") == 0;
}

/**
 * @brief Find a resource given a UUID.
 *
 * @param[in]   type       Type of resource.
 * @param[in]   uuid       UUID of resource.
 * @param[out]  resource   Resource return, 0 if succesfully failed to find resource.
 *
 * @return FALSE on success (including if failed to find resource), TRUE on error.
 */
gboolean
find_resource (const char* type, const char* uuid, resource_t* resource)
{
  gchar *quoted_uuid;
  quoted_uuid = sql_quote (uuid);
  if (acl_user_owns_uuid (type, quoted_uuid, 0) == 0)
    {
      g_free (quoted_uuid);
      *resource = 0;
      return FALSE;
    }
  // TODO should really check type
  switch (sql_int64 (resource,
                     "SELECT id FROM %ss WHERE uuid = '%s'%s;",
                     type,
                     quoted_uuid,
                     strcmp (type, "task") ? "" : " AND hidden < 2"))
    {
      case 0:
        break;
      case 1:        /* Too few rows in result of query. */
        *resource = 0;
        break;
      default:       /* Programming error. */
        assert (0);
      case -1:
        g_free (quoted_uuid);
        return TRUE;
        break;
    }

  g_free (quoted_uuid);
  return FALSE;
}

/**
 * @brief Find a resource given a UUID and a permission.
 *
 * @param[in]   type        Type of resource.
 * @param[in]   uuid        UUID of resource.
 * @param[out]  resource    Resource return, 0 if succesfully failed to find
 *                          resource.
 * @param[in]   permission  Permission.
 * @param[in]   trash       Whether resource is in trashcan.
 *
 * @return FALSE on success (including if failed to find resource), TRUE on
 *         error.
 */
gboolean
find_resource_with_permission (const char* type, const char* uuid,
                               resource_t* resource, const char *permission,
                               int trash)
{
  gchar *quoted_uuid;
  if (uuid == NULL)
    return TRUE;
  if ((type == NULL) || (valid_type (type) == 0))
    return TRUE;
  quoted_uuid = sql_quote (uuid);
  if (acl_user_has_access_uuid (type, quoted_uuid, permission, trash) == 0)
    {
      g_free (quoted_uuid);
      *resource = 0;
      return FALSE;
    }
  switch (sql_int64 (resource,
                     "SELECT id FROM %ss%s WHERE uuid = '%s'%s;",
                     type,
                     (strcmp (type, "task") && trash) ? "_trash" : "",
                     quoted_uuid,
                     strcmp (type, "task")
                      ? ""
                      : (trash ? " AND hidden = 2" : " AND hidden < 2")))
    {
      case 0:
        break;
      case 1:        /* Too few rows in result of query. */
        *resource = 0;
        break;
      default:       /* Programming error. */
        assert (0);
      case -1:
        g_free (quoted_uuid);
        return TRUE;
        break;
    }

  g_free (quoted_uuid);
  return FALSE;
}

/**
 * @brief Find a resource given a name.
 *
 * @param[in]   type      Type of resource.
 * @param[in]   name      A resource name.
 * @param[out]  resource  Resource return, 0 if succesfully failed to find
 *                        resource.
 *
 * @return FALSE on success (including if failed to find resource), TRUE on
 *         error.
 */
static gboolean
find_resource_by_name (const char* type, const char* name, resource_t *resource)
{
  gchar *quoted_name;
  quoted_name = sql_quote (name);
  // TODO should really check type
  switch (sql_int64 (resource,
                     "SELECT id FROM %ss WHERE name = '%s'"
                     " ORDER BY id DESC;",
                     type,
                     quoted_name))
    {
      case 0:
        break;
      case 1:        /* Too few rows in result of query. */
        *resource = 0;
        break;
      default:       /* Programming error. */
        assert (0);
      case -1:
        g_free (quoted_name);
        return TRUE;
        break;
    }

  g_free (quoted_name);
  return FALSE;
}

/**
 * @brief Find a resource given a UUID and a permission.
 *
 * @param[in]   type        Type of resource.
 * @param[in]   name        Name of resource.
 * @param[out]  resource    Resource return, 0 if succesfully failed to find
 *                          resource.
 * @param[in]   permission  Permission.
 *
 * @return FALSE on success (including if failed to find resource), TRUE on
 *         error.
 */
gboolean
find_resource_by_name_with_permission (const char *type, const char *name,
                                       resource_t *resource,
                                       const char *permission)
{
  gchar *quoted_name;
  assert (strcmp (type, "task"));
  if (name == NULL)
    return TRUE;
  quoted_name = sql_quote (name);
  // TODO should really check type
  switch (sql_int64 (resource,
                     "SELECT id FROM %ss WHERE name = '%s'"
                     " ORDER BY id DESC;",
                     type,
                     quoted_name))
    {
      case 0:
        {
          gchar *uuid;

          uuid = sql_string ("SELECT uuid FROM %ss WHERE id = %llu;",
                             type, *resource);
          if (acl_user_has_access_uuid (type, uuid, permission, 0) == 0)
            {
              g_free (uuid);
              g_free (quoted_name);
              *resource = 0;
              return FALSE;
            }
          g_free (uuid);
        }
        break;
      case 1:        /* Too few rows in result of query. */
        *resource = 0;
        break;
      default:       /* Programming error. */
        assert (0);
      case -1:
        g_free (quoted_name);
        return TRUE;
        break;
    }

  g_free (quoted_name);
  return FALSE;
}

/**
 * @brief Create a resource from an existing resource.
 *
 * @param[in]  type          Type of resource.
 * @param[in]  name          Name of new resource.  NULL to copy from existing.
 * @param[in]  comment       Comment on new resource.  NULL to copy from existing.
 * @param[in]  resource_id   UUID of existing resource.
 * @param[in]  columns       Extra columns in resource.
 * @param[in]  make_name_unique  When name NULL, whether to make existing name
 *                               unique.
 * @param[out] new_resource  Address for new resource, or NULL.
 * @param[out] old_resource  Address for existing resource, or NULL.
 *
 * @return 0 success, 1 resource exists already, 2 failed to find existing
 *         resource, 99 permission denied, -1 error.
 */
static int
copy_resource_lock (const char *type, const char *name, const char *comment,
                    const char *resource_id, const char *columns,
                    int make_name_unique, resource_t* new_resource,
                    resource_t *old_resource)
{
  gchar *quoted_name, *quoted_uuid, *uniquify, *command;
  int named;
  user_t owner;
  resource_t resource;
  resource_t new;

  if (resource_id == NULL)
    return -1;

  command = g_strdup_printf ("create_%s", type);
  if (acl_user_may (command) == 0)
    {
      g_free (command);
      return 99;
    }
  g_free (command);

  command = g_strdup_printf ("get_%ss", type);
  if (find_resource_with_permission (type, resource_id, &resource, command, 0))
    {
      g_free (command);
      return -1;
    }
  g_free (command);

  if (resource == 0)
    return 2;

  if (find_user_by_name (current_credentials.username, &owner)
      || owner == 0)
    {
      return -1;
    }

  if (strcmp (type, "permission") == 0)
    {
      resource_t perm_resource;
      perm_resource = permission_resource (resource);
      if ((perm_resource == 0)
          && (acl_user_can_everything (current_credentials.uuid) == 0))
        /* Only admins can copy permissions that apply to whole commands. */
        return 99;
    }

  named = type_named (type);

  if (named && name && *name && resource_with_name_exists (name, type, 0))
    return 1;
  if (name && *name)
    quoted_name = sql_quote (name);
  else
    quoted_name = NULL;
  quoted_uuid = sql_quote (resource_id);

  /* Copy the existing resource. */

  if (make_name_unique)
    uniquify = g_strdup_printf ("uniquify ('%s', name, %llu, '%cClone')",
                                type,
                                owner,
                                strcmp (type, "user") ? ' ' : '_');
  else
    uniquify = g_strdup ("name");
  if (named && comment && strlen (comment))
    {
      gchar *quoted_comment;
      quoted_comment = sql_nquote (comment, strlen (comment));
      sql ("INSERT INTO %ss"
           " (uuid, owner, name, comment, creation_time, modification_time%s%s)"
           " SELECT make_uuid (),"
           "        (SELECT id FROM users where users.uuid = '%s'),"
           "        %s%s%s, '%s', m_now (), m_now ()%s%s"
           " FROM %ss WHERE uuid = '%s';",
           type,
           columns ? ", " : "",
           columns ? columns : "",
           current_credentials.uuid,
           quoted_name ? "'" : "",
           quoted_name ? quoted_name : uniquify,
           quoted_name ? "'" : "",
           quoted_comment,
           columns ? ", " : "",
           columns ? columns : "",
           type,
           quoted_uuid);
      g_free (quoted_comment);
    }
  else if (named)
    sql ("INSERT INTO %ss"
         " (uuid, owner, name%s, creation_time, modification_time%s%s)"
         " SELECT make_uuid (),"
         "        (SELECT id FROM users where users.uuid = '%s'),"
         "        %s%s%s%s, m_now (), m_now ()%s%s"
         " FROM %ss WHERE uuid = '%s';",
         type,
         type_has_comment (type) ? ", comment" : "",
         columns ? ", " : "",
         columns ? columns : "",
         current_credentials.uuid,
         quoted_name ? "'" : "",
         quoted_name ? quoted_name : uniquify,
         quoted_name ? "'" : "",
         type_has_comment (type) ? ", comment" : "",
         columns ? ", " : "",
         columns ? columns : "",
         type,
         quoted_uuid);
  else
    sql ("INSERT INTO %ss"
         " (uuid, owner, creation_time, modification_time%s%s)"
         " SELECT make_uuid (), (SELECT id FROM users where users.uuid = '%s'),"
         "        m_now (), m_now ()%s%s"
         " FROM %ss WHERE uuid = '%s';",
         type,
         columns ? ", " : "",
         columns ? columns : "",
         current_credentials.uuid,
         columns ? ", " : "",
         columns ? columns : "",
         type,
         quoted_uuid);

  new = sql_last_insert_id ();

  /* Copy attached tags */
  sql ("INSERT INTO tags"
       " (uuid, owner, name, comment, creation_time, modification_time,"
       "  resource_type, resource, resource_uuid, resource_location,"
       "  active, value)"
       " SELECT make_uuid (), (SELECT id FROM users where users.uuid = '%s'),"
       "        name, comment, m_now (), m_now (),"
       "        resource_type, %llu, (SELECT uuid FROM %ss WHERE id = %llu),"
       "        resource_location, active, value"
       " FROM tags WHERE resource_type = '%s' AND resource = %llu"
       "           AND resource_location = " G_STRINGIFY (LOCATION_TABLE) ";",
       current_credentials.uuid,
       new,
       type,
       new,
       type,
       resource);

  if (new_resource)
    *new_resource = new;

  if (old_resource)
    *old_resource = resource;

  g_free (quoted_uuid);
  g_free (quoted_name);
  g_free (uniquify);
  if (sql_last_insert_id () == 0)
    return -1;
  return 0;
}

/**
 * @brief Create a resource from an existing resource.
 *
 * @param[in]  type          Type of resource.
 * @param[in]  name          Name of new resource.  NULL to copy from existing.
 * @param[in]  comment       Comment on new resource.  NULL to copy from existing.
 * @param[in]  resource_id   UUID of existing resource.
 * @param[in]  columns       Extra columns in resource.
 * @param[in]  make_name_unique  When name NULL, whether to make existing name
 *                               unique.
 * @param[out] new_resource  New resource.
 *
 * @return 0 success, 1 resource exists already, 2 failed to find existing
 *         resource, 99 permission denied, -1 error.
 */
int
copy_resource (const char *type, const char *name, const char *comment,
               const char *resource_id, const char *columns,
               int make_name_unique, resource_t* new_resource)
{
  int ret;

  assert (current_credentials.uuid);

  sql_begin_immediate ();

  ret = copy_resource_lock (type, name, comment, resource_id, columns,
                            make_name_unique, new_resource, NULL);

  if (ret)
    sql_rollback ();
  else
    sql_commit ();

  return ret;
}

/**
 * @brief Get whether a resource exists.
 *
 * @param[in]  type      Type.
 * @param[in]  resource  Resource.
 * @param[in]  location  Location.
 *
 * @return 1 yes, 0 no, -1 error in type.
 */
int
resource_exists (const char *type, resource_t resource, int location)
{
  if (valid_db_resource_type (type) == 0)
    return -1;

  if (location == LOCATION_TABLE)
    return sql_int ("SELECT EXISTS (SELECT id FROM %ss WHERE id = %llu);",
                    type,
                    resource);
  return sql_int ("SELECT EXISTS (SELECT id FROM %ss%s WHERE id = %llu);",
                  type,
                  strcmp (type, "task") ? "_trash" : "",
                  resource);
}

/**
 * @brief Get the name of a resource.
 *
 * @param[in]  type      Type.
 * @param[in]  uuid      UUID.
 * @param[in]  location  Location.
 * @param[out] name      Return for freshly allocated name.
 *
 * @return 0 success, 1 error in type.
 */
int
resource_name (const char *type, const char *uuid, int location, char **name)
{
  if (valid_db_resource_type (type) == 0)
    return 1;

  if (strcasecmp (type, "note") == 0)
    *name = sql_string ("SELECT 'Note for: '"
                        " || (SELECT name"
                        "     FROM nvts"
                        "     WHERE nvts.uuid = notes%s.nvt)"
                        " FROM notes%s"
                        " WHERE uuid = '%s';",
                        location == LOCATION_TABLE ? "" : "_trash",
                        location == LOCATION_TABLE ? "" : "_trash",
                        uuid);
  else if (strcasecmp (type, "override") == 0)
    *name = sql_string ("SELECT 'Override for: '"
                        " || (SELECT name"
                        "     FROM nvts"
                        "     WHERE nvts.uuid = overrides%s.nvt)"
                        " FROM overrides%s"
                        " WHERE uuid = '%s';",
                        location == LOCATION_TABLE ? "" : "_trash",
                        location == LOCATION_TABLE ? "" : "_trash",
                        uuid);
  else if (strcasecmp (type, "report") == 0)
    *name = sql_string ("SELECT (SELECT name FROM tasks WHERE id = task)"
                        " || ' - '"
                        " || (SELECT"
                        "       CASE (SELECT end_time FROM tasks"
                        "             WHERE id = task)"
                        "       WHEN 0 THEN 'N/A'"
                        "       ELSE (SELECT iso_time (end_time)"
                        "             FROM tasks WHERE id = task)"
                        "    END)"
                        " FROM reports"
                        " WHERE uuid = '%s';",
                        uuid);
  else if (strcasecmp (type, "result") == 0)
    *name = sql_string ("SELECT (SELECT name FROM tasks WHERE id = task)"
                        " || ' - '"
                        " || (SELECT name FROM nvts WHERE oid = nvt)"
                        " || ' - '"
                        " || (SELECT"
                        "       CASE (SELECT end_time FROM tasks"
                        "             WHERE id = task)"
                        "       WHEN 0 THEN 'N/A'"
                        "       ELSE (SELECT iso_time (end_time)"
                        "             FROM tasks WHERE id = task)"
                        "    END)"
                        " FROM results"
                        " WHERE uuid = '%s';",
                        uuid);
  else if (location == LOCATION_TABLE)
    *name = sql_string ("SELECT name"
                        " FROM %ss"
                        " WHERE uuid = '%s';",
                        type,
                        uuid);
  else if ((strcmp (type, "nvt"))
           && (strcmp (type, "cpe"))
           && (strcmp (type, "cve"))
           && (strcmp (type, "ovaldef"))
           && (strcmp (type, "cert_bund_adv"))
           && (strcmp (type, "dfn_cert_adv"))
           && (strcmp (type, "report"))
           && (strcmp (type, "result"))
           && (strcmp (type, "user")))
    *name = sql_string ("SELECT name"
                        " FROM %ss%s"
                        " WHERE uuid = '%s';",
                        type,
                        strcmp (type, "task") ? "_trash" : "",
                        uuid);
  else
    *name = NULL;

  return 0;
}

/**
 * @brief Get the name of a resource.
 *
 * @param[in]  type      Type.
 * @param[in]  uuid      UUID.
 * @param[out] name      Return for freshly allocated name.
 *
 * @return 0 success, 1 error in type.
 */
int
manage_resource_name (const char *type, const char *uuid, char **name)
{
  return resource_name (type, uuid, LOCATION_TABLE, name);
}

/**
 * @brief Get the name of a trashcan resource.
 *
 * @param[in]  type      Type.
 * @param[in]  uuid      UUID.
 * @param[out] name      Return for freshly allocated name.
 *
 * @return 0 success, 1 error in type.
 */
int
manage_trash_resource_name (const char *type, const char *uuid, char **name)
{
  return resource_name (type, uuid, LOCATION_TRASH, name);
}

/**
 * @brief Get the UUID of a resource.
 *
 * @param[in]  type      Type.
 * @param[in]  resource  Resource.
 *
 * @return Freshly allocated UUID on success, else NULL.
 */
gchar *
resource_uuid (const gchar *type, resource_t resource)
{
  assert (valid_db_resource_type (type));

  return sql_string ("SELECT uuid FROM %ss WHERE id = %llu;",
                     type,
                     resource);
}

/**
 * @brief Initialise a GET iterator, including observed resources.
 *
 * This version includes the pre_sql arg.
 *
 * @param[in]  iterator        Iterator.
 * @param[in]  type            Type of resource.
 * @param[in]  get             GET data.
 * @param[in]  select_columns         Columns for SQL.
 * @param[in]  trash_select_columns   Columns for SQL trash case.
 * @param[in]  where_columns          WHERE columns.  These are columns that
 *                                    can be used for filtering and searching,
 *                                    but are not accessed (so column has no
 *                                    iterator access function).
 * @param[in]  trash_where_columns    WHERE columns for trashcan.
 * @param[in]  filter_columns  Columns for filter.
 * @param[in]  distinct        Whether the query should be distinct.  Skipped
 *                             for trash and single resource.
 * @param[in]  extra_tables    Extra tables to join in FROM clause.
 * @param[in]  extra_where     Extra WHERE clauses.  Skipped for single
 *                             resource.
 * @param[in]  owned           Only get items owned by the current user.
 * @param[in]  ignore_id       Whether to ignore id (e.g. for report results).
 * @param[in]  extra_order     Extra ORDER clauses.
 * @param[in]  pre_sql         SQL to add before the SELECT.  Useful for WITH.
 *
 * @return 0 success, 1 failed to find resource, 2 failed to find filter, -1
 *         error.
 */
static int
init_get_iterator2_pre (iterator_t* iterator, const char *type,
                        const get_data_t *get, column_t *select_columns,
                        column_t *trash_select_columns,
                        column_t *where_columns,
                        column_t *trash_where_columns,
                        const char **filter_columns, int distinct,
                        const char *extra_tables,
                        const char *extra_where, int owned,
                        int ignore_id,
                        const char *extra_order,
                        const char *pre_sql,
                        int assume_permitted)
{
  int first, max;
  gchar *clause, *order, *filter, *owned_clause;
  array_t *permissions;
  resource_t resource = 0;
  gchar *owner_filter;
  gchar *columns;

  assert (get);

  if (select_columns == NULL)
    {
      assert (0);
      return -1;
    }

  if (ignore_id)
    {
      resource = 0;
    }
  else if (get->id && owned && (current_credentials.uuid == NULL))
    {
      gchar *quoted_uuid = sql_quote (get->id);
      switch (sql_int64 (&resource,
                         "SELECT id FROM %ss WHERE uuid = '%s';",
                         type, quoted_uuid))
        {
          case 0:
            break;
          case 1:        /* Too few rows in result of query. */
            g_free (quoted_uuid);
            return 1;
            break;
          default:       /* Programming error. */
            assert (0);
          case -1:
            g_free (quoted_uuid);
            return -1;
            break;
        }
      g_free (quoted_uuid);
    }
  else if (get->id && owned)
    {
      /* For now assume that the permission is "get_<type>".  Callers wishing
       * to iterate over a single resource with other permissions can use
       * uuid= in the filter (instead of passing get->id). */
      const char* permission;
      /* Special case: "get_assets" subtypes */
      if (strcasecmp (type, "host") == 0
          || strcasecmp (type, "os") == 0)
        permission = "get_assets";
      else
        permission = NULL;

      if (find_resource_with_permission (type, get->id, &resource, permission,
                                         get->trash))
        return -1;
      if (resource == 0)
        return 1;
    }

  if (get->filt_id && strcmp (get->filt_id, "0"))
    {
      if (get->filter_replacement)
        /* Replace the filter term with one given by the caller.  This is
         * used by GET_REPORTS to use the default filter with any task (when
         * given the special value of -3 in filt_id). */
        filter = g_strdup (get->filter_replacement);
      else
        filter = filter_term (get->filt_id);
      if (filter == NULL)
        return 2;
    }
  else
    filter = NULL;

  clause = filter_clause (type, filter ? filter : get->filter, filter_columns,
                          (get->trash && trash_select_columns)
                           ? trash_select_columns
                           : select_columns,
                          (get->trash && trash_where_columns)
                           ? trash_where_columns
                           : where_columns,
                          get->trash, &order, &first, &max, &permissions,
                          &owner_filter);

  g_free (filter);

  if (resource)
    /* Ownership test is done above by find function. */
    owned_clause = g_strdup (" t ()");
  else if (assume_permitted)
    owned_clause = g_strdup (" t ()");
  else
    owned_clause = acl_where_owned (type, get, owned, owner_filter, resource,
                                    permissions);

  g_free (owner_filter);
  array_free (permissions);

  if (get->trash && trash_select_columns)
    columns = columns_build_select (trash_select_columns);
  else
    columns = columns_build_select (select_columns);

  if (get->ignore_pagination
      && ((strcmp (type, "host") == 0)
          || (strcmp (type, "os") == 0)
          || (strcmp (type, "task") == 0)
          || (strcmp (type, "report") == 0)
          || (strcmp (type, "result") == 0)))
    {
      first = 0;
      max = -1;
    }

  if (strlen (order) == 0)
    {
      g_free (order);
      order = NULL;
    }

  if (resource && get->trash)
    init_iterator (iterator,
                   "%sSELECT %s"
                   " FROM %ss%s %s"
                   " WHERE id = %llu"
                   " AND %s"
                   "%s%s;",
                   pre_sql ? pre_sql : "",
                   columns,
                   type,
                   type_trash_in_table (type) ? "" : "_trash",
                   extra_tables ? extra_tables : "",
                   resource,
                   owned_clause,
                   order ? order : "",
                   order ? (extra_order ? extra_order : "") : "");
  else if (get->trash)
    init_iterator (iterator,
                   "%sSELECT %s"
                   " FROM %ss%s %s"
                   " WHERE"
                   "%s"
                   "%s"
                   "%s%s;",
                   pre_sql ? pre_sql : "",
                   columns,
                   type,
                   type_trash_in_table (type) ? "" : "_trash",
                   extra_tables ? extra_tables : "",
                   owned_clause,
                   extra_where ? extra_where : "",
                   order ? order : "",
                   order ? (extra_order ? extra_order : "") : "");
  else if (resource)
    init_iterator (iterator,
                   "%sSELECT %s"
                   " FROM %ss %s"
                   " WHERE id = %llu"
                   " AND %s"
                   "%s%s;",
                   pre_sql ? pre_sql : "",
                   columns,
                   type,
                   extra_tables ? extra_tables : "",
                   resource,
                   owned_clause,
                   order ? order : "",
                   order ? (extra_order ? extra_order : "") : "");
  else if (distinct == 0)
    init_iterator (iterator,
                   "%sSELECT %s"
                   " FROM %ss %s"
                   " WHERE %s"
                   " %s%s%s%s%s%s"
                   " LIMIT %s OFFSET %i;",
                   pre_sql ? pre_sql : "",
                   columns,
                   type,
                   extra_tables ? extra_tables : "",
                   owned_clause,
                   clause ? " AND (" : "",
                   clause ? clause : "",
                   clause ? ")" : "",
                   extra_where ? extra_where : "",
                   order ? order : "",
                   order ? (extra_order ? extra_order : "") : "",
                   sql_select_limit (max),
                   first);
  else
    {
      init_iterator (iterator,
                   "%s%sSELECT %s"
                   " FROM %ss %s"
                   " WHERE"
                   " %s"
                   "%s%s%s%s%s%s"
                   " LIMIT %s OFFSET %i%s;",
                   pre_sql ? pre_sql : "",
                   distinct ? "SELECT DISTINCT * FROM (" : "",
                   columns,
                   type,
                   extra_tables ? extra_tables : "",
                   owned_clause,
                   clause ? " AND (" : "",
                   clause ? clause : "",
                   clause ? ")" : "",
                   extra_where ? extra_where : "",
                   order ? order : "",
                   order ? (extra_order ? extra_order : "") : "",
                   sql_select_limit (max),
                   first,
                   distinct ? ") AS subquery_for_distinct" : "");
    }

  g_free (columns);
  g_free (owned_clause);
  g_free (order);
  g_free (clause);
  return 0;
}

/**
 * @brief Initialise a GET iterator, including observed resources.
 *
 * @param[in]  iterator        Iterator.
 * @param[in]  type            Type of resource.
 * @param[in]  get             GET data.
 * @param[in]  select_columns         Columns for SQL.
 * @param[in]  trash_select_columns   Columns for SQL trash case.
 * @param[in]  where_columns          WHERE columns.  These are columns that
 *                                    can be used for filtering and searching,
 *                                    but are not accessed (so column has no
 *                                    iterator access function).
 * @param[in]  trash_where_columns    WHERE columns for trashcan.
 * @param[in]  filter_columns  Columns for filter.
 * @param[in]  distinct        Whether the query should be distinct.  Skipped
 *                             for trash and single resource.
 * @param[in]  extra_tables    Extra tables to join in FROM clause.
 * @param[in]  extra_where     Extra WHERE clauses.  Skipped for single
 *                             resource.
 * @param[in]  owned           Only get items owned by the current user.
 * @param[in]  ignore_id       Whether to ignore id (e.g. for report results).
 * @param[in]  extra_order     Extra ORDER clauses.
 *
 * @return 0 success, 1 failed to find resource, 2 failed to find filter, -1
 *         error.
 */
static int
init_get_iterator2 (iterator_t* iterator, const char *type,
                    const get_data_t *get, column_t *select_columns,
                    column_t *trash_select_columns,
                    column_t *where_columns,
                    column_t *trash_where_columns,
                    const char **filter_columns, int distinct,
                    const char *extra_tables,
                    const char *extra_where, int owned,
                    int ignore_id,
                    const char *extra_order)
{
  return init_get_iterator2_pre (iterator, type, get, select_columns,
                                 trash_select_columns, where_columns,
                                 trash_where_columns, filter_columns, distinct,
                                 extra_tables, extra_where, owned, ignore_id,
                                 extra_order, NULL, 0);
}

/**
 * @brief Initialise a GET iterator, including observed resources.
 *
 * @param[in]  iterator        Iterator.
 * @param[in]  type            Type of resource.
 * @param[in]  get             GET data.
 * @param[in]  select_columns         Columns for SQL.
 * @param[in]  trash_select_columns   Columns for SQL trash case.
 * @param[in]  filter_columns  Columns for filter.
 * @param[in]  distinct        Whether the query should be distinct.  Skipped
 *                             for trash and single resource.
 * @param[in]  extra_tables    Extra tables to join in FROM clause.
 * @param[in]  extra_where     Extra WHERE clauses.  Skipped for single
 *                             resource.
 * @param[in]  owned           Only get items owned by the current user.
 *
 * @return 0 success, 1 failed to find resource, 2 failed to find filter, -1
 *         error.
 */
static int
init_get_iterator (iterator_t* iterator, const char *type,
                   const get_data_t *get, column_t *select_columns,
                   column_t *trash_select_columns,
                   const char **filter_columns, int distinct,
                   const char *extra_tables,
                   const char *extra_where, int owned)
{
  return init_get_iterator2 (iterator, type, get, select_columns,
                             trash_select_columns, NULL, NULL, filter_columns,
                             distinct, extra_tables, extra_where, owned, FALSE,
                             NULL);
}

// FIX
column_t *
type_select_columns (const char *type, int);

column_t *
type_where_columns (const char *type, int);

/**
 * @brief Append expression for a column to an array.
 *
 * @param[in]  columns         Array.
 * @param[in]  column_name     Name of column.
 * @param[in]  select_columns  Definition of "SELECT" columns.
 * @param[in]  where_columns   Definition of "WHERE" columns.
 */
static void
append_column (GArray *columns, const gchar *column_name,
               column_t *select_columns, column_t *where_columns)
{
  int i = 0;
  while (select_columns[i].select != NULL)
    {
      gchar *select = NULL;
      if (strcmp (select_columns[i].select, column_name) == 0
          || (select_columns[i].filter
              && strcmp (select_columns[i].filter, column_name) == 0))
        {
          select = g_strdup (select_columns[i].select);
          g_array_append_val (columns, select);
          break;
        }
      i++;
    }
  if (select_columns[i].select == NULL && where_columns)
    {
      i = 0;
      while (where_columns[i].select != NULL)
        {
          gchar *select = NULL;
          if (strcmp (where_columns[i].select, column_name) == 0
              || (where_columns[i].filter
                  && strcmp (where_columns[i].filter, column_name) == 0))
            {
              select = g_strdup (where_columns[i].select);
              g_array_append_val (columns, select);
              break;
            }
          i++;
        }
    }
}

/**
 * @brief Initialise a GET_AGGREGATES iterator, including observed resources.
 *
 * @param[in]  iterator        Iterator.
 * @param[in]  type            Type of resource.
 * @param[in]  get             GET data.
 * @param[in]  distinct        Whether the query should be distinct.  Skipped
 *                             for trash and single resource.
 * @param[in]  data_columns    Columns to calculate statistics for.
 * @param[in]  group_column    Column to group data by.
 * @param[in]  subgroup_column Column to further group data by.
 * @param[in]  text_columns    Columns to get text from.
 * @param[in]  sort_data       GArray of sorting data.
 * @param[in]  first_group     Row number to start iterating from.
 * @param[in]  max_groups      Maximum number of rows.
 * @param[in]  extra_tables    Join tables.  Skipped for trash and single
 *                             resource.
 * @param[in]  extra_where     Extra WHERE clauses.  Skipped for single
 *                             resource.
 *
 * @return 0 success, 1 failed to find resource, 2 failed to find filter,
 *         3 invalid stat_column, 4 invalid group_column, 5 invalid type,
 *         6 trashcan not used by type, 7 invalid text column, 8 invalid
 *         subgroup_column, -1 error.
 */
int
init_aggregate_iterator (iterator_t* iterator, const char *type,
                         const get_data_t *get, int distinct,
                         GArray *data_columns,
                         const char *group_column, const char *subgroup_column,
                         GArray *text_columns, GArray *sort_data,
                         int first_group, int max_groups,
                         const char *extra_tables, const char *extra_where)
{
  int owned;
  gchar *autofp_str, *apply_overrides_str, *min_qod_str;
  int autofp, apply_overrides, min_qod;
  column_t *select_columns, *where_columns;
  gchar *columns;
  gchar *trash_columns;
  gchar *opts_table;
  const char **filter_columns;
  int first, max;
  gchar *from_table, *trash_extra, *clause, *filter_order, *filter;
  gchar *owned_clause;
  array_t *permissions;
  resource_t resource = 0;
  gchar *owner_filter;
  GString *aggregate_select, *outer_col_select;
  gchar *aggregate_group_by;
  gchar *outer_group_by_column, *outer_subgroup_column;
  gchar *select_group_column, *select_subgroup_column;
  GArray *select_data_columns, *select_text_columns;
  GString *order_clause;

  assert (get);

  if (get->id)
    g_warning ("%s: Called with an id parameter", __FUNCTION__);

  if ((manage_scap_loaded () == FALSE
       && (strcmp (type, "cve") == 0
           || strcmp (type, "cpe") == 0
           || strcmp (type, "ovaldef") == 0
           || strcmp (type, "allinfo") == 0))
      || (manage_cert_loaded () == FALSE
          && (strcmp (type, "cert_bund_adv") == 0
              || strcmp (type, "dfn_cert_adv") == 0
              || strcmp (type, "allinfo") == 0)))
    {
      // Init a dummy iterator if SCAP or CERT DB is required but unavailable.
      init_iterator (iterator,
                     "SELECT NULL LIMIT %s",
                     sql_select_limit (0));
      return 0;
    }

  if (get->filt_id && strcmp (get->filt_id, "0"))
    {
      if (get->filter_replacement)
        /* Replace the filter term with one given by the caller.  This is
         * used by GET_REPORTS to use the default filter with any task (when
         * given the special value of -3 in filt_id). */
        filter = g_strdup (get->filter_replacement);
      else
        filter = filter_term (get->filt_id);
      if (filter == NULL)
        return 2;
    }
  else
    filter = NULL;

  autofp_str = filter_term_value (filter ? filter : get->filter,
                                  "autofp");
  autofp = autofp_str ? atoi (autofp_str) : 0;
  apply_overrides_str = filter_term_value (filter ? filter : get->filter,
                                           "apply_overrides");
  apply_overrides = apply_overrides_str ? atoi (apply_overrides_str) : 1;
  min_qod_str = filter_term_value (filter ? filter : get->filter, "min_qod");
  min_qod = (min_qod_str && strcmp (min_qod_str, ""))
              ? atoi (min_qod_str) : MIN_QOD_DEFAULT;
  g_free (apply_overrides_str);
  g_free (min_qod_str);

  select_columns = type_select_columns (type, apply_overrides);
  columns = type_columns (type, apply_overrides);
  trash_columns = type_trash_columns (type, apply_overrides);
  where_columns = type_where_columns (type, apply_overrides);
  filter_columns = type_filter_columns (type, apply_overrides);
  opts_table = type_opts_table (type, autofp, apply_overrides, min_qod);

  if (columns == NULL || filter_columns == NULL)
    {
      g_free (columns);
      g_free (trash_columns);
      g_free (opts_table);
      return 5;
    }
  if (get->trash && trash_columns == NULL)
    {
      g_free (columns);
      g_free (trash_columns);
      g_free (opts_table);
      return 6;
    }

  owned = type_owned (type);

  if (data_columns && data_columns->len > 0)
    {
      int i;
      for (i = 0; i < data_columns->len; i++)
        {
          if (vector_find_filter (filter_columns,
                                  g_array_index (data_columns, gchar*, i))
              == 0)
            {
              g_free (columns);
              g_free (trash_columns);
              g_free (opts_table);
              return 3;
            }
        }
    }
  if (text_columns && text_columns->len > 0)
    {
      int i;
      for (i = 0; i < text_columns->len; i++)
        {
          if (vector_find_filter (filter_columns,
                                  g_array_index (text_columns, gchar*, i))
              == 0)
            {
              g_free (columns);
              g_free (trash_columns);
              g_free (opts_table);
              return 7;
            }
        }
    }
  if (group_column && vector_find_filter (filter_columns, group_column) == 0)
    {
      g_free (columns);
      g_free (trash_columns);
      g_free (opts_table);
      return 4;
    }
  if (subgroup_column
      && vector_find_filter (filter_columns, subgroup_column) == 0)
    {
      g_free (columns);
      g_free (trash_columns);
      g_free (opts_table);
      return 8;
    }
  select_data_columns = g_array_new (TRUE, TRUE, sizeof (gchar*));
  select_text_columns = g_array_new (TRUE, TRUE, sizeof (gchar*));

  clause = filter_clause (type, filter ? filter : get->filter, filter_columns,
                          select_columns, where_columns, get->trash,
                          &filter_order, &first, &max, &permissions,
                          &owner_filter);

  g_free (filter);

  if (resource)
    /* Ownership test is done above by find function. */
    owned_clause = g_strdup (" 1");
  else
    owned_clause = acl_where_owned (type, get, owned, owner_filter, resource,
                                permissions);

  if (strcasecmp (type, "TASK") == 0)
    {
      if (get->trash)
        trash_extra = g_strdup (" AND hidden = 2");
      else
        trash_extra = g_strdup (" AND hidden = 0");
    }
  else if (strcasecmp (type, "REPORT") == 0)
    {
      if (get->trash)
        trash_extra = g_strdup (" AND (SELECT hidden FROM tasks"
                                "      WHERE tasks.id = task)"
                                "     = 2");
      else
        trash_extra = g_strdup (" AND (SELECT hidden FROM tasks"
                                "      WHERE tasks.id = task)"
                                "     = 0");
    }
  else if (strcasecmp (type, "RESULT") == 0)
    {
      trash_extra = results_extra_where (get, 0, NULL, autofp, apply_overrides,
                                         setting_dynamic_severity_int (),
                                         filter ? filter : get->filter);
    }
  else
    trash_extra = g_strdup ("");

  g_free (owner_filter);
  array_free (permissions);

  select_group_column = NULL;
  select_subgroup_column = NULL;

  if (group_column)
    {
      select_group_column
        = g_strdup (columns_select_column (select_columns,
                                           where_columns,
                                           group_column));
      if (subgroup_column)
        {
          select_subgroup_column
            = g_strdup (columns_select_column (select_columns,
                                               where_columns,
                                               subgroup_column));
        }
    }

  if (data_columns && data_columns->len > 0)
    {
      int column_index;
      for (column_index = 0; column_index < data_columns->len; column_index++)
        append_column (select_data_columns,
                       g_array_index (data_columns, gchar*, column_index),
                       select_columns,
                       where_columns);
    }

  if (text_columns && text_columns->len > 0)
    {
      int column_index;
      for (column_index = 0; column_index < text_columns->len; column_index++)
        append_column (select_text_columns,
                       g_array_index (text_columns, gchar*, column_index),
                       select_columns,
                       where_columns);
    }

  /* Round time fields to the next day to reduce amount of rows returned
   * This returns "pseudo-UTC" dates which are used by the GSA charts because
   *  the JavaScript Date objects do not support setting the timezone.
   */
  if (column_is_timestamp (group_column))
    if (sql_is_sqlite3 ())
      outer_group_by_column
        = g_strdup_printf ("CAST (strftime ('%%s',"
                           "                date(%s, 'unixepoch',"
                           "                     'localtime'),"
                           "                'utc')"
                           "      AS INTEGER)",
                           "aggregate_group_value");
    else
      outer_group_by_column
        = g_strdup_printf ("EXTRACT (EPOCH FROM"
                           "           date_trunc ('day',"
                           "           TIMESTAMP WITH TIME ZONE 'epoch'"
                           "           + (%s) * INTERVAL '1 second'))"
                           "  :: integer",
                           "aggregate_group_value");
  else
    outer_group_by_column = g_strdup ("aggregate_group_value");

  if (column_is_timestamp (subgroup_column))
    if (sql_is_sqlite3 ())
      outer_subgroup_column
        = g_strdup_printf ("CAST (strftime ('%%s',"
                           "                date(%s, 'unixepoch',"
                           "                     'localtime'),"
                           "                'utc')"
                           "      AS INTEGER)",
                           "aggregate_subgroup_value");
    else
      outer_subgroup_column
        = g_strdup_printf ("EXTRACT (EPOCH FROM"
                           "           date_trunc ('day',"
                           "           TIMESTAMP WITH TIME ZONE 'epoch'"
                           "           + (%s) * INTERVAL '1 second'))"
                           "  :: integer",
                           "aggregate_subgroup_value");
  else
    outer_subgroup_column = g_strdup ("aggregate_subgroup_value");

  if (sort_data)
    {
      order_clause = g_string_new ("ORDER BY");

      int sort_index;
      for (sort_index = 0; sort_index < sort_data->len; sort_index++) {
        sort_data_t *sort_data_item;
        const gchar *sort_field, *sort_stat;
        int sort_order;
        gchar *order_column;

        sort_data_item = g_array_index (sort_data, sort_data_t*, sort_index);
        sort_field = sort_data_item->field;
        sort_stat = sort_data_item->stat;
        sort_order = sort_data_item->order;

        if (sort_stat && strcmp (sort_stat, "count") == 0)
          order_column = g_strdup ("outer_count");
        else if (sort_stat && strcmp (sort_stat, "value") == 0)
          order_column = g_strdup ("outer_group_column");
        else if (sort_field
                 && group_column
                 && strcmp (sort_field, "")
                 && strcmp (sort_field, group_column)
                 && (subgroup_column == NULL
                     || strcmp (sort_field, subgroup_column)))
          {
            int index;
            order_column = NULL;
            for (index = 0;
                 index < data_columns->len && order_column == NULL;
                 index++)
              {
                gchar *column = g_array_index (data_columns, gchar*, index);
                if (strcmp (column, sort_field) == 0)
                  {
                    if (sort_stat == NULL || strcmp (sort_stat, "") == 0
                        || (   strcmp (sort_stat, "min")
                            && strcmp (sort_stat, "max")
                            && strcmp (sort_stat, "mean")
                            && strcmp (sort_stat, "sum")))
                      order_column = g_strdup_printf ("max (aggregate_max_%d)",
                                                      index);
                    else if (strcmp (sort_stat, "mean") == 0)
                      order_column = g_strdup_printf ("sum (aggregate_avg_%d)",
                                                      index);
                    else
                      order_column = g_strdup_printf ("%s (aggregate_%s_%d)",
                                                      sort_stat, sort_stat,
                                                      index);
                  }
              }

            for (index = 0;
                index < text_columns->len && order_column == NULL;
                index++)
              {
                gchar *column = g_array_index (text_columns, gchar*, index);
                if (strcmp (column, sort_field) == 0)
                  {
                    order_column = g_strdup_printf ("max (text_column_%d)",
                                                    index);
                  }
              }
          }
        else if (sort_field && subgroup_column
                && strcmp (sort_field, subgroup_column) == 0)
          order_column = g_strdup ("outer_subgroup_column");
        else
          order_column = g_strdup ("outer_group_column");

        if (subgroup_column && sort_index == 0)
          {
            xml_string_append (order_clause,
                               " outer_group_column %s, %s %s",
                               sort_order ? "ASC" : "DESC",
                               order_column,
                               sort_order ? "ASC" : "DESC");
          }
        else
          {
            xml_string_append (order_clause,
                               "%s %s %s",
                               sort_index > 0 ? "," : "",
                               order_column,
                               sort_order ? "ASC" : "DESC");
          }
        g_free (order_column);
      }

      if (sort_data->len == 0)
        g_string_append (order_clause, " outer_group_column ASC");
    }
  else
    order_clause = g_string_new ("");

  aggregate_select = g_string_new ("");
  outer_col_select = g_string_new ("");
  if (group_column && strcmp (group_column, ""))
    {
      if (subgroup_column && strcmp (subgroup_column, ""))
        {
          g_string_append_printf (aggregate_select,
                             " count(*) AS aggregate_count,"
                             " %s AS aggregate_group_value,"
                             " %s AS aggregate_subgroup_value",
                             select_group_column,
                             select_subgroup_column);

          aggregate_group_by = g_strdup_printf (" GROUP BY %s, %s",
                                                select_group_column,
                                                select_subgroup_column);
        }
      else
        {
          g_string_append_printf (aggregate_select,
                             " count(*) AS aggregate_count,"
                             " %s AS aggregate_group_value,"
                             " CAST (NULL AS TEXT) AS aggregate_subgroup_value",
                             select_group_column);

          aggregate_group_by = g_strdup_printf (" GROUP BY %s",
                                                select_group_column);
        }


    }
  else
    {
      g_string_append_printf (aggregate_select,
                         " count(*) AS aggregate_count,"
                         " CAST (NULL AS TEXT) AS aggregate_group_value,"
                         " CAST (NULL AS TEXT) AS aggregate_subgroup_value");

      aggregate_group_by = g_strdup ("");
    }

  int col_index;
  for (col_index = 0; col_index < select_data_columns->len; col_index ++)
    {
      gchar *select_data_column = g_array_index (select_data_columns, gchar*,
                                                 col_index);
      // TODO: Test type of column (string, number, timestamp)
      g_string_append_printf (aggregate_select,
                              ","
                              " min(CAST (%s AS real)) AS aggregate_min_%d,"
                              " max(CAST (%s AS real)) AS aggregate_max_%d,"
                              " avg(CAST (%s AS real)) * count(*)"
                              "   AS aggregate_avg_%d,"
                              " sum(CAST (%s AS real))"
                              "   AS aggregate_sum_%d",
                              select_data_column,
                              col_index,
                              select_data_column,
                              col_index,
                              select_data_column,
                              col_index,
                              select_data_column,
                              col_index);
      g_string_append_printf (outer_col_select,
                              ", min(aggregate_min_%d),"
                              " max (aggregate_max_%d),"
                              " sum (aggregate_avg_%d) / sum(aggregate_count),"
                              " sum (aggregate_sum_%d)",
                              col_index, col_index, col_index, col_index);
    }
  for (col_index = 0; col_index < select_text_columns->len; col_index ++)
    {
      gchar *select_text_column = g_array_index (select_text_columns, gchar*,
                                                 col_index);
      g_string_append_printf (aggregate_select,
                              ", max (%s) as text_column_%d",
                              select_text_column,
                              col_index);
      g_string_append_printf (outer_col_select,
                              ", max (text_column_%d)",
                              col_index);
    }

  from_table = type_table (type, get->trash);

  init_iterator (iterator,
                 "SELECT sum(aggregate_count) AS outer_count,"
                 " %s AS outer_group_column,"
                 " %s AS outer_subgroup_column"
                 " %s"
                 " FROM (SELECT%s %s"
                 "       FROM %s%s%s"
                 "       WHERE"
                 "       %s%s"
                 "       %s%s%s%s"
                 "       %s)"
                 "      AS agg_sub"
                 " GROUP BY outer_group_column, outer_subgroup_column %s"
                 " LIMIT %s OFFSET %d;",
                 outer_group_by_column,
                 outer_subgroup_column,
                 outer_col_select->str,
                 distinct ? " DISTINCT" : "",
                 aggregate_select->str,
                 from_table,
                 opts_table ? opts_table : "",
                 extra_tables ? extra_tables : "",
                 owned_clause,
                 trash_extra,
                 clause ? " AND (" : "",
                 clause ? clause : "",
                 clause ? ")" : "",
                 extra_where ? extra_where : "",
                 aggregate_group_by,
                 order_clause->str,
                 sql_select_limit (max_groups),
                 first_group);

  g_free (columns);
  g_free (trash_columns);
  g_free (opts_table);
  g_free (owned_clause);
  g_free (trash_extra);
  g_free (filter_order);
  g_string_free (order_clause, TRUE);
  g_free (clause);
  g_free (aggregate_group_by);
  g_string_free (aggregate_select, TRUE);
  g_string_free (outer_col_select, TRUE);
  g_free (outer_group_by_column);
  g_free (outer_subgroup_column);
  g_free (select_group_column);
  g_free (select_subgroup_column);
  return 0;
}

#define AGGREGATE_ITERATOR_OFFSET 3
#define AGGREGATE_ITERATOR_N_STATS 4

/**
 * @brief Get the count from an aggregate iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The count of resources in the current group.
 */
int
aggregate_iterator_count (iterator_t* iterator)
{
  return sql_column_int (iterator->stmt, 0);
}

/**
 * @brief Get the minimum from an aggregate iterator.
 *
 * @param[in]  iterator           Iterator.
 * @param[in]  data_column_index  Index of the data column to get min of.
 *
 * @return The minimum value in the current group.
 */
double
aggregate_iterator_min (iterator_t* iterator, int data_column_index)
{
  return sql_column_double (iterator->stmt,
                            AGGREGATE_ITERATOR_OFFSET
                            + data_column_index * AGGREGATE_ITERATOR_N_STATS);
}

/**
 * @brief Get the maximum from an aggregate iterator.
 *
 * @param[in]  iterator           Iterator.
 * @param[in]  data_column_index  Index of the data column to get max of.
 *
 * @return The maximum value in the current group.
 */
double
aggregate_iterator_max (iterator_t* iterator, int data_column_index)
{
  return sql_column_double (iterator->stmt,
                            AGGREGATE_ITERATOR_OFFSET + 1
                            + data_column_index * AGGREGATE_ITERATOR_N_STATS);
}

/**
 * @brief Get the mean from an aggregate iterator.
 *
 * @param[in]  iterator           Iterator.
 * @param[in]  data_column_index  Index of the data column to get mean of.
 *
 * @return The mean value in the current group.
 */
double
aggregate_iterator_mean (iterator_t* iterator, int data_column_index)
{
  return sql_column_double (iterator->stmt,
                            AGGREGATE_ITERATOR_OFFSET + 2
                            + data_column_index * AGGREGATE_ITERATOR_N_STATS);
}

/**
 * @brief Get the sum from a statistics iterator.
 *
 * @param[in]  iterator           Iterator.
 * @param[in]  data_column_index  Index of the data column to get sum of.
 *
 * @return The sum of values in the current group.
 */
double
aggregate_iterator_sum (iterator_t* iterator, int data_column_index)
{
  return sql_column_double (iterator->stmt,
                            AGGREGATE_ITERATOR_OFFSET + 3
                            + data_column_index * AGGREGATE_ITERATOR_N_STATS);
}

/**
 * @brief Get the value of a text column from an aggregate iterator.
 *
 * @param[in]  iterator           Iterator.
 * @param[in]  text_column_index  Index of the text column to get.
 * @param[in]  data_columns       Number of data columns.
 *
 * @return The value, or NULL if iteration is complete.  Freed by
 *         cleanup_iterator.
 */
const char*
aggregate_iterator_text (iterator_t* iterator, int text_column_index,
                         int data_columns)
{
  const char *ret;
  if (iterator->done) return NULL;
  ret = (const char*) sql_column_text (iterator->stmt,
                                       AGGREGATE_ITERATOR_OFFSET
                                        + (data_columns
                                           * AGGREGATE_ITERATOR_N_STATS)
                                        + text_column_index);
  return ret;
}

/**
 * @brief Get the value of the group column from a statistics iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The value, or NULL if iteration is complete.  Freed by
 *         cleanup_iterator.
 */
const char*
aggregate_iterator_value (iterator_t* iterator)
{
  const char *ret;
  if (iterator->done) return NULL;
  ret = (const char*) sql_column_text (iterator->stmt, 1);
  return ret;
}

/**
 * @brief Get the value of the subgroup column from an aggregate iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The value, or NULL if iteration is complete.  Freed by
 *         cleanup_iterator.
 */
const char*
aggregate_iterator_subgroup_value (iterator_t* iterator)
{
  const char *ret;
  if (iterator->done) return NULL;
  ret = (const char*) sql_column_text (iterator->stmt, 2);
  return ret;
}

/**
 * @brief Count number of a particular resource.
 *
 * @param[in]  type              Type of resource.
 * @param[in]  get               GET params.
 * @param[in]  select_columns    SELECT columns.
 * @param[in]  trash_select_columns  SELECT columns for trashcan.
 * @param[in]  where_columns     WHERE columns.
 * @param[in]  trash_where_columns   WHERE columns for trashcan.
 * @param[in]  filter_columns        Extra columns.
 * @param[in]  distinct          Whether the query should be distinct.  Skipped
 *                               for trash and single resource.
 * @param[in]  extra_tables      Join tables.  Skipped for trash and single
 *                               resource.
 * @param[in]  extra_where       Extra WHERE clauses.  Skipped for trash and
 *                               single resource.
 * @param[in]  owned             Only count items owned by current user.
 *
 * @return Total number of resources in filtered set.
 */
static int
count2 (const char *type, const get_data_t *get, column_t *select_columns,
        column_t *trash_select_columns, column_t *where_columns,
        column_t *trash_where_columns, const char **filter_columns, int distinct,
        const char *extra_tables, const char *extra_where, int owned)
{
  int ret;
  gchar *clause, *owned_clause, *owner_filter, *columns, *filter;
  array_t *permissions;

  assert (get);

  if (get->filt_id && strcmp (get->filt_id, "0"))
    {
      filter = filter_term (get->filt_id);
      if (filter == NULL)
        return -1;
    }
  else
    filter = NULL;

  clause = filter_clause (type, filter ? filter : get->filter, filter_columns,
                          get->trash && trash_select_columns
                           ? trash_select_columns
                           : select_columns,
                          get->trash && trash_where_columns
                           ? trash_where_columns
                           : where_columns,
                          get->trash, NULL, NULL, NULL, &permissions,
                          &owner_filter);

  g_free (filter);

  owned_clause = acl_where_owned (type, get, owned, owner_filter, 0, permissions);

  g_free (owner_filter);
  array_free (permissions);

  if (get->trash && trash_select_columns)
    columns = columns_build_select (trash_select_columns);
  else
    columns = columns_build_select (select_columns);

  if ((distinct == 0)
      && (extra_tables == NULL)
      && (clause == NULL)
      && (extra_where == NULL)
      && (strcmp (owned_clause, " t ()") == 0))
    ret = sql_int ("SELECT count (*) FROM %ss%s;",
                   type,
                   get->trash && strcmp (type, "task") ? "_trash" : "");
  else
    ret = sql_int ("SELECT count (%scount_id)"
                   " FROM (SELECT %ss%s.id AS count_id"
                   "       FROM %ss%s%s"
                   "       WHERE %s"
                   "       %s%s%s%s) AS subquery;",
                   distinct ? "DISTINCT " : "",
                   type,
                   get->trash && strcmp (type, "task") ? "_trash" : "",
                   type,
                   get->trash && strcmp (type, "task") ? "_trash" : "",
                   extra_tables ? extra_tables : "",
                   owned_clause,
                   clause ? " AND (" : "",
                   clause ? clause : "",
                   clause ? ") " : "",
                   extra_where ? extra_where : "");

  g_free (columns);
  g_free (owned_clause);
  g_free (clause);
  return ret;
}

/**
 * @brief Count number of a particular resource.
 *
 * @param[in]  type              Type of resource.
 * @param[in]  get               GET params.
 * @param[in]  select_columns    SELECT columns.
 * @param[in]  trash_select_columns  SELECT columns for trashcan.
 * @param[in]  filter_columns        Extra columns.
 * @param[in]  distinct          Whether the query should be distinct.  Skipped
 *                               for trash and single resource.
 * @param[in]  extra_tables      Join tables.  Skipped for trash and single
 *                               resource.
 * @param[in]  extra_where       Extra WHERE clauses.  Skipped for trash and
 *                               single resource.
 * @param[in]  owned             Only count items owned by current user.
 *
 * @return Total number of resources in filtered set.
 */
static int
count (const char *type, const get_data_t *get, column_t *select_columns,
       column_t *trash_select_columns, const char **filter_columns,
       int distinct, const char *extra_tables, const char *extra_where,
       int owned)
{
  return count2 (type, get, select_columns, trash_select_columns, NULL, NULL,
                 filter_columns, distinct, extra_tables, extra_where, owned);
}

/**
 * @brief Count number of info of a given subtype with a given name.
 *
 * @param[in]  type  GET_INFO subtype.
 * @param[out] name  Name of the info item.
 *
 * @return Total number of get_info items of given type, -1 on error.
 */
int
info_name_count (const char *type, const char *name)
{
  gchar *quoted_name;
  int count;
  assert(type);
  assert(name);

  quoted_name = sql_quote (name);
  count =  sql_int ("SELECT COUNT(id)"
                    " FROM %ss"
                    " WHERE name = '%s';",
                    type,
                    quoted_name);
  g_free (quoted_name);

  return count;
}

/**
 * @brief Return whether a resource is predefined.
 *
 * @param[in]  type      Type of resource.
 * @param[in]  resource  Resource.
 *
 * @return 1 if predefined, else 0.
 */
static int
resource_predefined (const gchar *type, resource_t resource)
{
  assert (valid_type (type));
  return sql_int ("SELECT EXISTS (SELECT * FROM resources_predefined"
                  "               WHERE resource_type = '%s'"
                  "               AND resource = %llu);",
                  type,
                  resource);
}

/**
 * @brief Mark a resource as predefined.
 *
 * Currently only report formats use this.
 */
static void
resource_set_predefined (const gchar *type, resource_t resource, int enable)
{
  assert (valid_type (type));

  sql ("DELETE FROM resources_predefined"
       " WHERE resource_type = '%s'"
       " AND resource = %llu;",
       type,
       resource);

  if (enable)
    sql ("INSERT into resources_predefined (resource_type, resource)"
         " VALUES ('%s', %llu);",
         type,
         resource);
}



/**
 * @brief Return the database version supported by this manager.
 *
 * @return Database version supported by this manager.
 */
int
manage_db_supported_version ()
{
  return OPENVASMD_DATABASE_VERSION;
}

/**
 * @brief Return the database version of the actual database.
 *
 * @return Database version read from database, -2 if database is empty,
 *         -1 on error.
 */
int
manage_db_version ()
{
  int number;
  char *version;

  if (manage_db_empty ())
    return -2;

  version = sql_string ("SELECT value FROM %s.meta"
                        " WHERE name = 'database_version' LIMIT 1;",
                        sql_schema ());
  if (version)
    {
      number = atoi (version);
      free (version);
      return number;
    }
  return -1;
}

/**
 * @brief Return the database version supported by this manager.
 *
 * @return Database version supported by this manager.
 */
int
manage_scap_db_supported_version ()
{
  return OPENVASMD_SCAP_DATABASE_VERSION;
}

/**
 * @brief Return the database version of the actual database.
 *
 * @return Database version read from database if possible, else -1.
 */
int
manage_scap_db_version ()
{
  if (manage_scap_loaded () == 0)
    return -1;

  int number;
  char *version = sql_string ("SELECT value FROM scap.meta"
                              " WHERE name = 'database_version' LIMIT 1;");
  if (version)
    {
      number = atoi (version);
      free (version);
      return number;
    }
  return -1;
}

/**
 * @brief Return the database version supported by this manager.
 *
 * @return Database version supported by this manager.
 */
int
manage_cert_db_supported_version ()
{
  return OPENVASMD_CERT_DATABASE_VERSION;
}

/**
 * @brief Return the database version of the actual database.
 *
 * @return Database version read from database if possible, else -1.
 */
int
manage_cert_db_version ()
{
  if (manage_cert_loaded () == 0)
    return -1;

  int number;
  char *version = sql_string ("SELECT value FROM cert.meta"
                              " WHERE name = 'database_version' LIMIT 1;");
  if (version)
    {
      number = atoi (version);
      free (version);
      return number;
    }
  return -1;
}

/**
 * @brief Returns associated name for a tcp/ip port.
 *
 * @param   number      Port number to get name for.
 * @param   protocol    Protocol type of port.
 *
 * @return  associated name for port if found, NULL otherwise.
 */
char *
manage_port_name (int number, const char *protocol)
{
  if (protocol == NULL || number <= 0 || number > 65535)
    return NULL;

  return sql_string ("SELECT name FROM port_names"
                     " WHERE number = %i AND protocol = '%s' LIMIT 1;",
                     number, protocol);
}

/**
 * @brief Returns formatted port number, protocol and iana name from
 * @brief field in "number/proto" form.
 *
 * @param   field       Number/Protocol string.
 *
 * @return  Formatted port name string, NULL if error.
 */
gchar *
port_name_formatted (const char *field)
{
  int number;
  char *protocol, *port_name;

  if (field == NULL)
    return NULL;

  protocol = g_newa (char, strlen (field));

  if (sscanf (field, "%i/%s",
              &number, protocol)
      != 2)
    return g_strdup (field);

  port_name = manage_port_name (number, protocol);
  if (port_name)
    {
      char *formatted = g_strdup_printf
                         ("%i/%s (IANA: %s)",
                          number,
                          protocol,
                          port_name);
      free (port_name);
      return formatted;
    }
  else
    return g_strdup (field);
}

/**
 * @brief Set the database version of the actual database.
 *
 * Caller must organise transaction.
 *
 * @param  version  New version number.
 */
void
set_db_version (int version)
{
  sql ("DELETE FROM %s.meta WHERE name = 'database_version';",
       sql_schema ());
  sql ("INSERT INTO %s.meta (name, value)"
       " VALUES ('database_version', '%i');",
       sql_schema (),
       version);
}


/**
 * @brief Encrypt, re-encrypt or decrypt all credentials
 *
 * All plaintext credentials in the credentials table are
 * encrypted, all already encrypted credentials are encrypted again
 * using the latest key.
 *
 * @param[in] decrypt_flag  If true decrypt all credentials.
 *
 * @return 0 success, -1 error.
 */
static int
encrypt_all_credentials (gboolean decrypt_flag)
{
  iterator_t iterator;
  unsigned long ntotal, nencrypted, nreencrypted, ndecrypted;

  init_iterator (&iterator,
                 "SELECT id,"
                 " (SELECT value FROM credentials_data"
                 "  WHERE credential = credentials.id"
                 "  AND type = 'secret'),"
                 " (SELECT value FROM credentials_data"
                 "  WHERE credential = credentials.id"
                 "  AND type = 'password'),"
                 " (SELECT value FROM credentials_data"
                 "  WHERE credential = credentials.id"
                 "  AND type = 'private_key')"
                 " FROM credentials");
  iterator.crypt_ctx = lsc_crypt_new ();

  sql_begin_immediate ();

  ntotal = nencrypted = nreencrypted = ndecrypted = 0;
  while (next (&iterator))
    {
      long long int rowid;
      const char *secret, *password, *privkey;

      ntotal++;
      if (!(ntotal % 10))
        g_message ("  %lu credentials so far processed", ntotal);

      rowid    = iterator_int64 (&iterator, 0);
      secret   = iterator_string (&iterator, 1);
      password = iterator_string (&iterator, 2);
      privkey  = iterator_string (&iterator, 3);

      /* If there is no secret, password or private key, skip the row.  */
      if (!secret && !password && !privkey)
        continue;

      if (secret)
        {
          lsc_crypt_flush (iterator.crypt_ctx);
          password = lsc_crypt_get_password (iterator.crypt_ctx, secret);
          privkey  = lsc_crypt_get_private_key (iterator.crypt_ctx, secret);

          /* If there is no password or private key, skip the row.  */
          if (!password && !privkey)
            continue;

          nreencrypted++;
        }
      else
        {
          if (decrypt_flag)
            continue; /* Skip non-encrypted rows.  */

          nencrypted++;
        }

      if (decrypt_flag)
        {
          set_credential_data (rowid, "password", password);
          set_credential_data (rowid, "private_key", privkey);
          set_credential_data (rowid, "secret", NULL);
          sql ("UPDATE credentials SET"
               " modification_time = m_now ()"
               " WHERE id = %llu;", rowid);
          ndecrypted++;
        }
      else
        {
          char *encblob;

          if (password && privkey)
            encblob = lsc_crypt_encrypt (iterator.crypt_ctx,
                                         "password", password,
                                         "private_key", privkey, NULL);
          else if (password)
            encblob = lsc_crypt_encrypt (iterator.crypt_ctx,
                                         "password", password, NULL);
          else
            encblob = lsc_crypt_encrypt (iterator.crypt_ctx,
                                         "private_key", privkey, NULL);

          if (!encblob)
            {
              sql_rollback ();
              cleanup_iterator (&iterator);
              return -1;
            }
          set_credential_data (rowid, "password", NULL);
          set_credential_data (rowid, "private_key", NULL);
          set_credential_data (rowid, "secret", encblob);
          sql ("UPDATE credentials SET"
               " modification_time = m_now ()"
               " WHERE id = %llu;", rowid);
          g_free (encblob);
        }
    }

  sql_commit ();

  if (decrypt_flag)
    g_message ("%lu out of %lu credentials decrypted",
               ndecrypted, ntotal);
  else
    g_message ("%lu out of %lu credentials encrypted and %lu re-encrypted",
               nencrypted, ntotal, nreencrypted);
  cleanup_iterator (&iterator);
  return 0;
}

/**
 * @brief Encrypt or re-encrypt all credentials
 *
 * All plaintext credentials in the credentials table are
 * encrypted, all already encrypted credentials are encrypted again
 * using the latest key.
 *
 * @param[in] log_config    Log configuration.
 * @param[in] database      Location of manage database.
 *
 * @return 0 success, -1 error,
 *         -2 database is wrong version, -3 database needs to be initialised
 *         from server.
 */
int
manage_encrypt_all_credentials (GSList *log_config, const gchar *database)
{
  int ret;

  g_info ("   (Re-)encrypting all credentials.\n");

  ret = manage_option_setup (log_config, database);
  if (ret)
    return ret;

  ret = encrypt_all_credentials (FALSE);
  if (ret)
    printf ("Encryption failed.\n");
  else
    printf ("Encryption succeeded.\n");

  manage_option_cleanup ();

  return ret;
}

/**
 * @brief Decrypt all credentials
 *
 * @param[in] log_config    Log configuration.
 * @param[in] database      Location of manage database.
 *
 * @return 0 success, -1 error,
 *         -2 database is wrong version, -3 database needs to be initialised
 *         from server.
 */
int
manage_decrypt_all_credentials (GSList *log_config, const gchar *database)
{
  int ret;

  g_info ("   Decrypting all credentials.\n");

  ret = manage_option_setup (log_config, database);
  if (ret)
    return ret;

  ret = encrypt_all_credentials (TRUE);
  if (ret)
    printf ("Decryption failed.\n");
  else
    printf ("Decryption succeeded.\n");

  manage_option_cleanup ();

  return ret;
}


/* Collation. */

/**
 * @brief Collate two message type strings.
 *
 * A lower threat is considered less than a higher threat, so Medium is
 * less than High.
 *
 * @param[in]  data     Dummy for callback.
 * @param[in]  one_len  Length of first string.
 * @param[in]  arg_one  First string.
 * @param[in]  two_len  Length of second string.
 * @param[in]  arg_two  Second string.
 *
 * @return -1, 0 or 1 if first is less than, equal to or greater than second.
 */
int
collate_message_type (void* data,
                      int one_len, const void* arg_one,
                      int two_len, const void* arg_two)
{
  const char* one = (const char*) arg_one;
  const char* two = (const char*) arg_two;

  if (strncmp (one, "Security Hole", one_len) == 0)
    {
      if (strncmp (two, "Security Hole", two_len) == 0)
        return 0;
      return 1;
    }
  if (strncmp (two, "Security Hole", two_len) == 0) return -1;

  if (strncmp (one, "Security Warning", one_len) == 0)
    {
      if (strncmp (two, "Security Warning", two_len) == 0)
        return 0;
      return 1;
    }
  if (strncmp (two, "Security Warning", two_len) == 0) return -1;

  if (strncmp (one, "Security Note", one_len) == 0)
    {
      if (strncmp (two, "Security Note", two_len) == 0)
        return 0;
      return 1;
    }
  if (strncmp (two, "Security Note", two_len) == 0) return -1;

  if (strncmp (one, "Log Message", one_len) == 0)
    {
      if (strncmp (two, "Log Message", two_len) == 0)
        return 0;
      return 1;
    }
  if (strncmp (two, "Log Message", two_len) == 0) return -1;

  if (strncmp (one, "Debug Message", one_len) == 0)
    {
      if (strncmp (two, "Debug Message", two_len) == 0)
        return 0;
      return 1;
    }
  if (strncmp (two, "Debug Message", two_len) == 0) return -1;

  if (strncmp (one, "Error Message", one_len) == 0)
    {
      if (strncmp (two, "Error Message", two_len) == 0)
        return 0;
      return 1;
    }
  if (strncmp (two, "Error Message", two_len) == 0) return -1;

  return strncmp (one, two, MIN (one_len, two_len));
}

/**
 * @brief Compare two number strings for collate_ip.
 *
 * @param[in]  one_arg  First string.
 * @param[in]  two_arg  Second string.
 *
 * @return -1, 0 or 1 if first is less than, equal to or greater than second.
 */
static int
collate_ip_compare (const char *one_arg, const char *two_arg)
{
  int one = atoi (one_arg);
  int two = atoi (two_arg);
  return one == two ? 0 : (one < two ? -1 : 1);
}

/**
 * @brief Collate two IP addresses.
 *
 * For example, 127.0.0.2 is less than 127.0.0.3 and 127.0.0.10.
 *
 * Only works correctly for IPv4 addresses.
 *
 * @param[in]  data     Dummy for callback.
 * @param[in]  one_len  Length of first IP (a string).
 * @param[in]  arg_one  First string.
 * @param[in]  two_len  Length of second IP (a string).
 * @param[in]  arg_two  Second string.
 *
 * @return -1, 0 or 1 if first is less than, equal to or greater than second.
 */
int
collate_ip (void* data,
            int one_len, const void* arg_one,
            int two_len, const void* arg_two)
{
  int ret, one_dot, two_dot;
  char one_a[4], one_b[4], one_c[4], one_d[4];
  char two_a[4], two_b[4], two_c[4], two_d[4];
  const char* one = (const char*) arg_one;
  const char* two = (const char*) arg_two;

  if ((sscanf (one, "%3[0-9].%3[0-9].%3[0-9].%n%3[0-9]",
               one_a, one_b, one_c, &one_dot, one_d)
       == 4)
      && (sscanf (two, "%3[0-9].%3[0-9].%3[0-9].%n%3[0-9]",
                  two_a, two_b, two_c, &two_dot, two_d)
          == 4))
    {
      int ret = collate_ip_compare (one_a, two_a);
      if (ret) return ret < 0 ? -1 : 1;

      ret = collate_ip_compare (one_b, two_b);
      if (ret) return ret < 0 ? -1 : 1;

      ret = collate_ip_compare (one_c, two_c);
      if (ret) return ret < 0 ? -1 : 1;

      /* Ensure that the last number is limited to digits in the arg. */
      one_d[one_len - one_dot] = '\0';
      two_d[two_len - two_dot] = '\0';

      ret = collate_ip_compare (one_d, two_d);
      if (ret) return ret < 0 ? -1 : 1;

      return 0;
    }

  ret = strncmp (one, two, MIN (one_len, two_len));
  return ret == 0 ? 0 : (ret < 0 ? -1 : 1);
}


/* Access control. */

/**
 * @brief Generate accessor for an SQL iterator.
 *
 * This convenience macro is used to generate an accessor returning a
 * const string pointer.
 *
 * @param[in]  name  Name of accessor.
 * @param[in]  col   Column number to access.
 */
#define DEF_ACCESS(name, col)                                     \
const char*                                                       \
name (iterator_t* iterator)                                       \
{                                                                 \
  const char *ret;                                                \
  if (iterator->done) return NULL;                                \
  ret = iterator_string (iterator, col);                          \
  return ret;                                                     \
}

/**
 * @brief Initialise a task user iterator.
 *
 * @param[in]  iterator  Iterator.
 * @param[in]  task      Task.
 */
void
init_task_user_iterator (iterator_t *iterator, task_t task)
{
  init_iterator (iterator,
                 "SELECT DISTINCT 1, resource, subject,"
                 " (SELECT name FROM users"
                 "  WHERE users.id = permissions.subject)"
                 " FROM permissions"
                 /* Any permission implies 'get_tasks'. */
                 " WHERE resource_type = 'task'"
                 " AND resource = %llu"
                 " AND resource_location = " G_STRINGIFY (LOCATION_TABLE)
                 " AND subject_type = 'user'"
                 " AND subject_location = " G_STRINGIFY (LOCATION_TABLE) ";",
                 task);
}

DEF_ACCESS (task_user_iterator_name, 3);

/**
 * @brief Initialise a task group iterator.
 *
 * @param[in]  iterator  Iterator.
 * @param[in]  task      Task.
 */
void
init_task_group_iterator (iterator_t *iterator, task_t task)
{
  init_iterator (iterator,
                 "SELECT DISTINCT 1, resource, subject,"
                 " (SELECT name FROM groups"
                 "  WHERE groups.id = permissions.subject),"
                 " (SELECT uuid FROM groups"
                 "  WHERE groups.id = permissions.subject)"
                 " FROM permissions"
                 /* Any permission implies 'get_tasks'. */
                 " WHERE resource_type = 'task'"
                 " AND resource = %llu"
                 " AND resource_location = " G_STRINGIFY (LOCATION_TABLE)
                 " AND subject_type = 'group'"
                 " AND subject_location = " G_STRINGIFY (LOCATION_TABLE) ";",
                 task);
}

DEF_ACCESS (task_group_iterator_name, 3);

DEF_ACCESS (task_group_iterator_uuid, 4);

/**
 * @brief Initialise a task role iterator.
 *
 * @param[in]  iterator  Iterator.
 * @param[in]  task      Task.
 */
void
init_task_role_iterator (iterator_t *iterator, task_t task)
{
  init_iterator (iterator,
                 "SELECT DISTINCT 1, resource, subject,"
                 " (SELECT name FROM roles"
                 "  WHERE roles.id = permissions.subject),"
                 " (SELECT uuid FROM roles"
                 "  WHERE roles.id = permissions.subject)"
                 " FROM permissions"
                 /* Any permission implies 'get'. */
                 " WHERE resource_type = 'task'"
                 " AND resource = %llu"
                 " AND resource_location = " G_STRINGIFY (LOCATION_TABLE)
                 " AND subject_type = 'role'",
                 task);
}

DEF_ACCESS (task_role_iterator_name, 3);

DEF_ACCESS (task_role_iterator_uuid, 4);


/* Events and Alerts. */

/**
 * @brief Check if any SecInfo alerts are due.
 *
 * @param[in]  log_config  Log configuration.
 * @param[in]  database    Location of manage database.
 *
 * @return 0 success, -1 error,
 *         -2 database is wrong version, -3 database needs to be initialised
 *         from server.
 */
int
manage_check_alerts (GSList *log_config, const gchar *database)
{
  int ret;

  g_info ("   Checking alerts.\n");

  ret = manage_option_setup (log_config, database);
  if (ret)
    return ret;

  /* Setup a dummy user, so that create_user will work. */
  current_credentials.uuid = "";

  if (manage_scap_loaded ())
    {
      int max_time;

      max_time
       = sql_int ("SELECT %s"
                  "        ((SELECT max (modification_time) FROM scap.cves),"
                  "         (SELECT max (modification_time) FROM scap.cpes),"
                  "         (SELECT max (modification_time) FROM scap.ovaldefs),"
                  "         (SELECT max (creation_time) FROM scap.cves),"
                  "         (SELECT max (creation_time) FROM scap.cpes),"
                  "         (SELECT max (creation_time) FROM scap.ovaldefs));",
                  sql_greatest ());

      if (sql_int ("SELECT NOT EXISTS (SELECT * FROM meta"
                   "                   WHERE name = 'scap_check_time')"))
        sql ("INSERT INTO meta (name, value)"
             " VALUES ('scap_check_time', %i);",
             max_time);
      else if (sql_int ("SELECT value = '0' FROM meta"
                        " WHERE name = 'scap_check_time';"))
        sql ("UPDATE meta SET value = %i"
             " WHERE name = 'scap_check_time';",
             max_time);
      else
        {
          check_for_new_scap ();
          check_for_updated_scap ();
          sql ("UPDATE meta SET value = %i"
               " WHERE name = 'scap_check_time';",
               max_time);
        }
    }

  if (manage_cert_loaded ())
    {
      int max_time;

      max_time
       = sql_int ("SELECT"
                  " %s"
                  "  ((SELECT max (modification_time) FROM cert.cert_bund_advs),"
                  "   (SELECT max (modification_time) FROM cert.dfn_cert_advs),"
                  "   (SELECT max (creation_time) FROM cert.cert_bund_advs),"
                  "   (SELECT max (creation_time) FROM cert.dfn_cert_advs));",
                  sql_greatest ());

      if (sql_int ("SELECT NOT EXISTS (SELECT * FROM meta"
                   "                   WHERE name = 'cert_check_time')"))
        sql ("INSERT INTO meta (name, value)"
             " VALUES ('cert_check_time', %i);",
             max_time);
      else if (sql_int ("SELECT value = '0' FROM meta"
                        " WHERE name = 'cert_check_time';"))
        sql ("UPDATE meta SET value = %i"
             " WHERE name = 'cert_check_time';",
             max_time);
      else
        {
          check_for_new_cert ();
          check_for_updated_cert ();
          sql ("UPDATE meta SET value = %i"
               " WHERE name = 'cert_check_time';",
               max_time);
        }
    }

  current_credentials.uuid = NULL;

  manage_option_cleanup ();

  return ret;
}

/**
 * @brief Find a alert for a specific permission, given a UUID.
 *
 * @param[in]   uuid        UUID of alert.
 * @param[out]  alert       Alert return, 0 if succesfully failed to find alert.
 * @param[in]   permission  Permission.
 *
 * @return FALSE on success (including if failed to find alert), TRUE on error.
 */
gboolean
find_alert_with_permission (const char* uuid, alert_t* alert,
                            const char *permission)
{
  return find_resource_with_permission ("alert", uuid, alert, permission, 0);
}

/**
 * @brief Validate an email address.
 *
 * @param[in]  address  Email address.
 *
 * @return 0 success, 1 failure.
 */
static int
validate_email (const char* address)
{
  gchar **split, *point;

  assert (address);

  split = g_strsplit (address, "@", 0);

  if (split[0] == NULL || split[1] == NULL || split[2])
    {
      g_strfreev (split);
      return 1;
    }

  /* Local part. */
  point = split[0];
  while (*point)
    if (isalnum (*point)
        || strchr ("!#$%&'*+-/=?^_`{|}~", *point)
        || ((*point == '.')
            && (point > split[0])
            && point[1]
            && (point[1] != '.')
            && (point[-1] != '.')))
      point++;
    else
      {
        g_strfreev (split);
        return 1;
      }

  /* Domain. */
  point = split[1];
  while (*point)
    if (isalnum (*point)
        || strchr ("-_", *point)  /* RFC actually forbids _. */
        || ((*point == '.')
            && (point > split[1])
            && point[1]
            && (point[1] != '.')
            && (point[-1] != '.')))
      point++;
    else
      {
        g_strfreev (split);
        return 1;
      }

  g_strfreev (split);
  return 0;
}

/**
 * @brief Validate an email address list.
 *
 * @param[in]  list  Comma separated list of email addresses.
 *
 * @return 0 success, 1 failure.
 */
static int
validate_email_list (const char *list)
{
  gchar **split, **point;

  assert (list);

  split = g_strsplit (list, ",", 0);

  if (split[0] == NULL)
    {
      g_strfreev (split);
      return 1;
    }

  point = split;
  while (*point)
    {
      const char *address;
      address = *point;
      while (*address && (*address == ' ')) address++;
      if (validate_email (address))
        {
          g_strfreev (split);
          return 1;
        }
      point++;
    }

  g_strfreev (split);
  return 0;
}

/**
 * @brief Validate condition data for an alert.
 *
 * @param[in]  name      Name.
 * @param[in]  data      Data to validate.
 * @param[in]  condition The condition.
 *
 * @return 0 on success, 1 unexpected data name, 2 syntax error in data,
 *         3 failed to find filter for condition, -1 internal error.
 */
static int
validate_alert_condition_data (gchar *name, gchar* data,
                               alert_condition_t condition)
{
  if (condition == ALERT_CONDITION_ALWAYS)
    return 1;
  if (condition == ALERT_CONDITION_SEVERITY_AT_LEAST)
    {
      if (strcmp (name, "severity"))
        return 1;

      if (g_regex_match_simple ("^(-1(\\.0)?|[0-9](\\.[0-9])?|10(\\.0))$",
                                data ? data : "",
                                0,
                                0)
          == 0)
        return 2;
    }
  else if (condition == ALERT_CONDITION_SEVERITY_CHANGED)
    {
      if (strcmp (name, "direction"))
        return 1;

      if (g_regex_match_simple ("^(increased|decreased|changed)$",
                                data ? data : "",
                                0,
                                0)
          == 0)
        return 2;
    }
  else if (condition == ALERT_CONDITION_FILTER_COUNT_AT_LEAST)
    {
      if (strcmp (name, "filter_id") == 0)
        {
          filter_t filter;
          if (data == NULL)
            return 3;
          filter = 0;
          if (find_filter_with_permission (data, &filter, "get_filters"))
            return -1;
          if (filter == 0)
            return 3;
          return 0;
        }

      if (strcmp (name, "count"))
        return 1;
    }
  else if (condition == ALERT_CONDITION_FILTER_COUNT_CHANGED)
    {
      if (strcmp (name, "filter_id") == 0)
        {
          filter_t filter;
          if (data == NULL)
            return 3;
          filter = 0;
          if (find_filter_with_permission (data, &filter, "get_filters"))
            return -1;
          if (filter == 0)
            return 3;
          return 0;
        }

      if (strcmp (name, "direction")
          && strcmp (name, "count"))
        return 1;

      if (strcmp (name, "direction") == 0
          && g_regex_match_simple ("^(increased|decreased|changed)$",
                                   data ? data : "",
                                   0,
                                   0)
             == 0)
        return 2;
    }


  return 0;
}

/**
 * @brief Validate event data for an alert.
 *
 * @param[in]  name   Name.
 * @param[in]  data   Data to validate.
 * @param[in]  event  The event.
 *
 * @return 0 on success, 1 unexpected data name, 2 syntax error in data.
 */
static int
validate_alert_event_data (gchar *name, gchar* data, event_t event)
{
  if (event == EVENT_NEW_SECINFO || event == EVENT_UPDATED_SECINFO)
    {
      if (strcmp (name, "secinfo_type"))
        return 1;

      if (data == NULL)
        return 2;

      if (strcasecmp (data, "nvt")
          && strcasecmp (data, "cve")
          && strcasecmp (data, "cpe")
          && strcasecmp (data, "cert_bund_adv")
          && strcasecmp (data, "dfn_cert_adv")
          && strcasecmp (data, "ovaldef"))
        return 2;
    }
  return 0;
}

/**
 * @brief Validate method data for the SCP method.
 *
 * @param[in]  method          Method that data corresponds to.
 * @param[in]  name            Name of data.
 * @param[in]  data            The data.
 *
 * @return 0 valid, 15 error in SCP host, 17 failed to find report format for
 *         SCP method, 18 error in SCP credential, 19 error in SCP path,
 *         -1 error.
 */
int
validate_scp_data (alert_method_t method, const gchar *name, gchar **data)
{
  if (method == ALERT_METHOD_SCP
      && strcmp (name, "scp_credential") == 0)
    {
      credential_t credential;
      if (find_credential_with_permission (*data, &credential,
                                           "get_credentials"))
        return -1;
      else if (credential == 0)
        return 18;
      else
        {
          gchar *username;
          username = credential_value (credential, "username");

          if (username == NULL || strlen (username) == 0)
            {
              g_free (username);
              return 18;
            }

          if (strchr (username, '@') || strchr (username, ':'))
            {
              g_free (username);
              return 18;
            }

          g_free (username);
        }
    }

  if (method == ALERT_METHOD_SCP
      && strcmp (name, "scp_path") == 0)
    {
      if (strlen (*data) == 0)
        return 19;
    }

  if (method == ALERT_METHOD_SCP
      && strcmp (name, "scp_host") == 0)
    {
      int type;
      gchar *stripped;

      stripped = g_strstrip (g_strdup (*data));
      type = openvas_get_host_type (stripped);
      g_free (stripped);
      if ((type != HOST_TYPE_IPV4)
          && (type != HOST_TYPE_IPV6)
          && (type != HOST_TYPE_NAME))
        return 15;
    }

  if (method == ALERT_METHOD_SCP
      && strcmp (name, "scp_report_format") == 0)
    {
      report_format_t report_format;

      report_format = 0;
      if (find_report_format_with_permission (*data,
                                              &report_format,
                                              "get_report_formats"))
        return -1;
      if (report_format == 0)
        return 17;
    }

  return 0;
}

/**
 * @brief Validate method data for the Send method.
 *
 * @param[in]  method          Method that data corresponds to.
 * @param[in]  name            Name of data.
 * @param[in]  data            The data.
 *
 * @return 0 valid, 12 error in Send host, 13 error in Send port, 14 failed
 *         to find report format for Send method, -1 error.
 */
int
validate_send_data (alert_method_t method, const gchar *name, gchar **data)
{
  if (method == ALERT_METHOD_SEND
      && strcmp (name, "send_host") == 0)
    {
      int type;
      gchar *stripped;

      stripped = g_strstrip (g_strdup (*data));
      type = openvas_get_host_type (stripped);
      g_free (stripped);
      if ((type != HOST_TYPE_IPV4)
          && (type != HOST_TYPE_IPV6)
          && (type != HOST_TYPE_NAME))
        return 12;
    }

  if (method == ALERT_METHOD_SEND
      && strcmp (name, "send_port") == 0)
    {
      int port;
      gchar *stripped, *end;

      stripped = g_strstrip (g_strdup (*data));
      port = strtol (stripped, &end, 10);
      if (*end != '\0')
        {
          g_free (stripped);
          return 13;
        }

      g_free (stripped);
      g_free (*data);
      *data = g_strdup_printf ("%i", port);
    }

  if (method == ALERT_METHOD_SEND
      && strcmp (name, "send_report_format") == 0)
    {
      report_format_t report_format;

      report_format = 0;
      if (find_report_format_with_permission (*data,
                                              &report_format,
                                              "get_report_formats"))
        return -1;
      if (report_format == 0)
        return 14;
    }

  return 0;
}

/**
 * @brief Validate method data for the Send method.
 *
 * @param[in]  method          Method that data corresponds to.
 * @param[in]  name            Name of data.
 * @param[in]  data            The data.
 *
 * @return 0 valid, 40 invalid credential, 41 invalid SMB share path,
 *         42 invalid SMB file path, -1 error.
 */
int
validate_smb_data (alert_method_t method, const gchar *name, gchar **data)
{
  if (method == ALERT_METHOD_SMB)
    {
      if (strcmp (name, "smb_credential") == 0)
        {
          credential_t credential;
          if (find_credential_with_permission (*data, &credential,
                                              "get_credentials"))
            return -1;
          else if (credential == 0)
            return 40;
          else
            {
              gchar *username;
              username = credential_value (credential, "username");

              if (username == NULL || strlen (username) == 0)
                {
                  g_free (username);
                  return 40;
                }

              if (strchr (username, '@') || strchr (username, ':'))
                {
                  g_free (username);
                  return 40;
                }

              g_free (username);
            }
        }

      if (strcmp (name, "smb_share_path") == 0)
        {
          if (g_regex_match_simple ("^(?>\\\\\\\\|\\/\\/)[^:?<>|]+"
                                    "(?>\\\\|\\/)[^:?<>|]+$", *data, 0, 0)
              == FALSE)
            {
              return 41;
            }
        }

      if (strcmp (name, "smb_file_path") == 0)
        {
          if (g_regex_match_simple ("^[^:?<>|]+$", *data, 0, 0)
              == FALSE)
            {
              return 42;
            }
        }
    }

  return 0;
}

/**
 * @brief Validate method data for the TippingPoint method.
 *
 * @param[in]  method          Method that data corresponds to.
 * @param[in]  name            Name of data.
 * @param[in]  data            The data.
 *
 * @return 0 valid, 50 invalid credential, 51 invalid hostname,
 *  52 invalid certificate, 53 invalid TLS workaround setting.
 */
int
validate_tippingpoint_data (alert_method_t method, const gchar *name,
                             gchar **data)
{
  if (method == ALERT_METHOD_TIPPINGPOINT)
    {
      if (strcmp (name, "tp_sms_credential") == 0)
        {
          credential_t credential;
          if (find_credential_with_permission (*data, &credential,
                                               "get_credentials"))
            return -1;
          else if (credential == 0)
            return 50;
          else
            {
              if (strcmp (credential_type (credential), "up"))
                return 50;

            }
        }

      if (strcmp (name, "tp_sms_hostname") == 0)
        {
          if (g_regex_match_simple ("^[0-9A-Za-z][0-9A-Za-z.-]*$",
                                    *data, 0, 0)
              == FALSE)
            {
              return 51;
            }
        }

      if (strcmp (name, "tp_sms_tls_certificate") == 0)
        {
          // TODO: Check certificate, return 52 on failure
        }

      if (strcmp (name, "tp_sms_tls_workaround") == 0)
        {
          if (g_regex_match_simple ("^0|1$", *data, 0, 0)
              == FALSE)
            {
              return 53;
            }
        }
    }

  return 0;
}

/**
 * @brief Check alert params.
 *
 * @param[in]  event           Type of event.
 * @param[in]  condition       Event condition.
 * @param[in]  method          Escalation method.
 *
 * @return 0 success, 20 method does not match event, 21 condition does not
 *         match event.
 */
static int
check_alert_params (event_t event, alert_condition_t condition,
                    alert_method_t method)
{
  if (event == EVENT_NEW_SECINFO || event == EVENT_UPDATED_SECINFO)
    {
      if (method == ALERT_METHOD_HTTP_GET
          || method == ALERT_METHOD_SOURCEFIRE
          || method == ALERT_METHOD_START_TASK
          || method == ALERT_METHOD_VERINICE)
        return 20;

      if (condition == ALERT_CONDITION_SEVERITY_AT_LEAST
          || condition == ALERT_CONDITION_SEVERITY_CHANGED
          || condition == ALERT_CONDITION_FILTER_COUNT_CHANGED)
        return 21;
    }
  return 0;
}

/**
 * @brief Create an alert.
 *
 * @param[in]  name            Name of alert.
 * @param[in]  comment         Comment on alert.
 * @param[in]  filter_id       Filter.
 * @param[in]  event           Type of event.
 * @param[in]  event_data      Type-specific event data.
 * @param[in]  condition       Event condition.
 * @param[in]  condition_data  Condition-specific data.
 * @param[in]  method          Escalation method.
 * @param[in]  method_data     Data for escalation method.
 * @param[out] alert       Created alert on success.
 *
 * @return 0 success, 1 escalation exists already, 2 validation of email failed,
 *         3 failed to find filter, 4 type must be "result" if specified,
 *         5 unexpected condition data name, 6 syntax error in condition data,
 *         7 email subject too long, 8 email message too long, 9 failed to find
 *         filter for condition, 12 error in Send host, 13 error in Send port,
 *         14 failed to find report format for Send method, 15 error in
 *         SCP host, 17 failed to find report format for SCP method, 18 error
 *         in SCP credential, 19 error in SCP path, 20 method does not match
 *         event, 21 condition does not match event, 31 unexpected event data
 *         name, 32 syntax error in event data, 40 invalid SMB credential
 *       , 41 invalid SMB share path, 42 invalid SMB file path,
 *         50 invalid TippingPoint credential, 51 invalid TippingPoint hostname,
 *         52 invalid TippingPoint certificate, 53 invalid TippingPoint TLS
 *         workaround setting.
 *         99 permission denied, -1 error.
 */
int
create_alert (const char* name, const char* comment, const char* filter_id,
              event_t event, GPtrArray* event_data,
              alert_condition_t condition, GPtrArray* condition_data,
              alert_method_t method, GPtrArray* method_data,
              alert_t *alert)
{
  int index, ret;
  gchar *item, *quoted_comment;
  gchar *quoted_name;
  filter_t filter;

  assert (current_credentials.uuid);

  sql_begin_immediate ();

  if (acl_user_may ("create_alert") == 0)
    {
      sql_rollback ();
      return 99;
    }

  ret = check_alert_params (event, condition, method);
  if (ret)
    {
      sql_rollback ();
      return ret;
    }

  filter = 0;
  if (event != EVENT_NEW_SECINFO && event != EVENT_UPDATED_SECINFO && filter_id
      && strcmp (filter_id, "0"))
    {
      char *type;

      if (find_filter_with_permission (filter_id, &filter, "get_filters"))
        {
          sql_rollback ();
          return -1;
        }

      if (filter == 0)
        {
          sql_rollback ();
          return 3;
        }

      /* Filter type must be result if specified. */

      type = sql_string ("SELECT type FROM filters WHERE id = %llu;",
                         filter);
      if (type && strcasecmp (type, "result"))
        {
          free (type);
          sql_rollback ();
          return 4;
        }
      free (type);
    }

  if (resource_with_name_exists (name, "alert", 0))
    {
      sql_rollback ();
      return 1;
    }
  quoted_name = sql_quote (name);
  quoted_comment = sql_quote (comment ?: "");

  sql ("INSERT INTO alerts (uuid, owner, name, comment, event, condition,"
       " method, filter, creation_time, modification_time)"
       " VALUES (make_uuid (),"
       " (SELECT id FROM users WHERE users.uuid = '%s'),"
       " '%s', '%s', %i, %i, %i, %llu, m_now (), m_now ());",
       current_credentials.uuid,
       quoted_name,
       quoted_comment,
       event,
       condition,
       method,
       filter);

  g_free (quoted_comment);
  g_free (quoted_name);

  *alert = sql_last_insert_id ();

  index = 0;
  while ((item = (gchar*) g_ptr_array_index (condition_data, index++)))
    {
      int validation_result;
      gchar *name = sql_quote (item);
      gchar *data = sql_quote (item + strlen (item) + 1);

      validation_result = validate_alert_condition_data (name, data, condition);

      if (validation_result)
        {
          g_free (name);
          g_free (data);
          sql_rollback ();

          switch (validation_result)
            {
              case 1:
                return 5;
              case 2:
                return 6;
              case 3:
                return 9;
              default:
                return -1;
            }
        }

      sql ("INSERT INTO alert_condition_data (alert, name, data)"
           " VALUES (%llu, '%s', '%s');",
           *alert,
           name,
           data);
      g_free (name);
      g_free (data);
    }

  index = 0;
  while ((item = (gchar*) g_ptr_array_index (event_data, index++)))
    {
      int validation_result;
      gchar *name = sql_quote (item);
      gchar *data = sql_quote (item + strlen (item) + 1);

      validation_result = validate_alert_event_data (name, data, event);

      if (validation_result)
        {
          g_free (name);
          g_free (data);
          sql_rollback ();

          switch (validation_result)
            {
              case 1:
                return 31;
              case 2:
                return 32;
              default:
                return -1;
            }
        }

      sql ("INSERT INTO alert_event_data (alert, name, data)"
           " VALUES (%llu, '%s', '%s');",
           *alert,
           name,
           data);
      g_free (name);
      g_free (data);
    }

  index = 0;
  while ((item = (gchar*) g_ptr_array_index (method_data, index++)))
    {
      int ret;
      gchar *name, *data;

      name = sql_quote (item);
      data = sql_quote (item + strlen (item) + 1);

      if (method == ALERT_METHOD_EMAIL
          && strcmp (name, "to_address") == 0
          && validate_email_list (data))
        {
          g_free (name);
          g_free (data);
          sql_rollback ();
          return 2;
        }

      if (method == ALERT_METHOD_EMAIL
          && strcmp (name, "from_address") == 0
          && validate_email (data))
        {
          g_free (name);
          g_free (data);
          sql_rollback ();
          return 2;
        }

      if (method == ALERT_METHOD_EMAIL
          && strcmp (name, "subject") == 0
          && strlen (data) > 80)
        {
          g_free (name);
          g_free (data);
          sql_rollback ();
          return 7;
        }

      if (method == ALERT_METHOD_EMAIL
          && strcmp (name, "message") == 0
          && strlen (data) > max_email_message_length)
        {
          g_free (name);
          g_free (data);
          sql_rollback ();
          return 8;
        }

      ret = validate_scp_data (method, name, &data);
      if (ret)
        {
          g_free (name);
          g_free (data);
          sql_rollback ();
          return ret;
        }

      ret = validate_send_data (method, name, &data);
      if (ret)
        {
          g_free (name);
          g_free (data);
          sql_rollback ();
          return ret;
        }

      ret = validate_smb_data (method, name, &data);
      if (ret)
        {
          g_free (name);
          g_free (data);
          sql_rollback ();
          return ret;
        }

      ret = validate_tippingpoint_data (method, name, &data);
      if (ret)
        {
          g_free (name);
          g_free (data);
          sql_rollback ();
          return ret;
        }

      sql ("INSERT INTO alert_method_data (alert, name, data)"
           " VALUES (%llu, '%s', '%s');",
           *alert,
           name,
           data);
      g_free (name);
      g_free (data);
    }

  sql_commit ();

  return 0;
}

/**
 * @brief Create an alert from an existing alert.
 *
 * @param[in]  name          Name of new alert. NULL to copy from existing.
 * @param[in]  comment       Comment on new alert. NULL to copy from
 *                           existing.
 * @param[in]  alert_id      UUID of existing alert.
 * @param[out] new_alert     New alert.
 *
 * @return 0 success, 1 alert exists already, 2 failed to find existing
 *         alert, 99 permission denied, -1 error.
 */
int
copy_alert (const char* name, const char* comment, const char* alert_id,
            alert_t* new_alert)
{
  int ret;
  alert_t new, old;

  assert (current_credentials.uuid);

  if (alert_id == NULL)
    return -1;

  sql_begin_immediate ();

  ret = copy_resource_lock ("alert", name, comment, alert_id,
                            "event, condition, method, filter",
                            1, &new, &old);
  if (ret)
    {
      sql_rollback ();
      return ret;
    }

  /* Copy the alert condition data */
  sql ("INSERT INTO alert_condition_data (alert, name, data)"
       " SELECT %llu, name, data FROM alert_condition_data"
       "  WHERE alert = %llu;",
       new,
       old);

  /* Copy the alert event data */
  sql ("INSERT INTO alert_event_data (alert, name, data)"
       " SELECT %llu, name, data FROM alert_event_data"
       "  WHERE alert = %llu;",
       new,
       old);

  /* Copy the alert method data */
  sql ("INSERT INTO alert_method_data (alert, name, data)"
       " SELECT %llu, name, data FROM alert_method_data"
       "  WHERE alert = %llu;",
       new,
       old);

  sql_commit ();
  if (new_alert) *new_alert = new;
  return 0;
}

/**
 * @brief Modify an alert.
 *
 * @param[in]   alert_id        UUID of alert.
 * @param[in]   name            Name of alert.
 * @param[in]   comment         Comment on alert.
 * @param[in]   filter_id       Filter.
 * @param[in]   event           Type of event.
 * @param[in]   event_data      Type-specific event data.
 * @param[in]   condition       Event condition.
 * @param[in]   condition_data  Condition-specific data.
 * @param[in]   method          Escalation method.
 * @param[in]   method_data     Data for escalation method.
 *
 * @return 0 success, 1 failed to find alert, 2 alert with new name exists,
 *         3 alert_id required, 4 failed to find filter, 5 filter type must be
 *         result if specified, 6 Provided email address not valid,
 *         7 unexpected condition data name, 8 syntax error in condition data,
 *         9 email subject too long, 10 email message too long, 11 failed to
 *         find filter for condition, 12 error in Send host, 13 error in Send
 *         port, 14 failed to find report format for Send method, 15 error in
 *         SCP host, 17 failed to find report format for SCP method, 18 error
 *         in SCP credential, 19 error in SCP path, 20 method does not match
 *         event, 21 condition does not match event, 31 unexpected event data
 *         name, 32 syntax error in event data, 40 invalid SMB credential
 *       , 41 invalid SMB share path, 42 invalid SMB file path,
 *         50 invalid TippingPoint credential, 51 invalid TippingPoint hostname,
 *         52 invalid TippingPoint certificate, 53 invalid TippingPoint TLS
 *         workaround setting.
 *         99 permission denied, -1 internal error.
 */
int
modify_alert (const char *alert_id, const char *name, const char *comment,
              const char *filter_id, event_t event, GPtrArray *event_data,
              alert_condition_t condition, GPtrArray *condition_data,
              alert_method_t method, GPtrArray *method_data)
{
  int index, ret;
  gchar *quoted_name, *quoted_comment, *item;
  alert_t alert;
  filter_t filter;

  if (alert_id == NULL)
    return 3;

  sql_begin_immediate ();

  assert (current_credentials.uuid);

  if (acl_user_may ("modify_alert") == 0)
    {
      sql_rollback ();
      return 99;
    }

  ret = check_alert_params (event, condition, method);
  if (ret)
    {
      sql_rollback ();
      return ret;
    }

  alert = 0;
  if (find_alert_with_permission (alert_id, &alert, "modify_alert"))
    {
      sql_rollback ();
      return -1;
    }

  if (alert == 0)
    {
      sql_rollback ();
      return 1;
    }

  /* Check whether an alert with the same name exists already. */
  if (resource_with_name_exists (name, "alert", alert))
    {
      sql_rollback ();
      return 2;
    }

  /* Check filter. */
  filter = 0;
  if (event != EVENT_NEW_SECINFO && event != EVENT_UPDATED_SECINFO && filter_id
      && strcmp (filter_id, "0"))
    {
      char *type;

      if (find_filter_with_permission (filter_id, &filter, "get_filters"))
        {
          sql_rollback ();
          return -1;
        }

      if (filter == 0)
        {
          sql_rollback ();
          return 4;
        }

      /* Filter type must be report if specified. */

      type = sql_string ("SELECT type FROM filters WHERE id = %llu;",
                         filter);
      if (type && strcasecmp (type, "result"))
        {
          free (type);
          sql_rollback ();
          return 5;
        }
      free (type);
    }

  quoted_name = sql_quote (name ?: "");
  quoted_comment = sql_quote (comment ? comment : "");

  sql ("UPDATE alerts SET"
       " name = '%s',"
       " comment = '%s',"
       " filter = %llu,"
       " modification_time = m_now ()"
       " WHERE id = %llu;",
       quoted_name,
       quoted_comment,
       filter,
       alert);

  g_free (quoted_comment);
  g_free (quoted_name);

  /* Modify alert event */
  if (event != EVENT_ERROR)
    {
      sql ("UPDATE alerts set event = %i WHERE id = %llu", event, alert);
      sql ("DELETE FROM alert_event_data WHERE alert = %llu", alert);
      index = 0;
      while ((item = (gchar*) g_ptr_array_index (event_data, index++)))
        {
          int validation_result;
          gchar *name = sql_quote (item);
          gchar *data = sql_quote (item + strlen (item) + 1);

          validation_result = validate_alert_event_data (name, data, event);

          if (validation_result)
            {
              g_free (name);
              g_free (data);
              sql_rollback ();

              switch (validation_result)
                {
                  case 1:
                    return 31;
                  case 2:
                    return 32;
                  default:
                    return -1;
                }
            }

          sql ("INSERT INTO alert_event_data (alert, name, data)"
               " VALUES (%llu, '%s', '%s');",
               alert,
               name,
               data);
          g_free (name);
          g_free (data);
        }
    }

  /* Modify alert condition */
  if (condition != ALERT_CONDITION_ERROR)
    {
      sql ("UPDATE alerts set condition = %i WHERE id = %llu",
           condition,
           alert);
      sql ("DELETE FROM alert_condition_data WHERE alert = %llu", alert);
      index = 0;
      while ((item = (gchar*) g_ptr_array_index (condition_data, index++)))
        {
          int validation_result;
          gchar *name = sql_quote (item);
          gchar *data = sql_quote (item + strlen (item) + 1);

          validation_result = validate_alert_condition_data (name, data,
                                                             condition);

          if (validation_result)
            {
              g_free (name);
              g_free (data);
              sql_rollback ();

              switch (validation_result)
                {
                  case 1:
                    return 7;
                  case 2:
                    return 8;
                  case 3:
                    return 11;
                  default:
                    return -1;
                }
            }

          sql ("INSERT INTO alert_condition_data (alert, name, data)"
               " VALUES (%llu, '%s', '%s');",
               alert,
               name,
               data);
          g_free (name);
          g_free (data);
        }
    }

  /* Modify alert method */
  if (method != ALERT_METHOD_ERROR)
    {
      sql ("UPDATE alerts set method = %i WHERE id = %llu", method, alert);
      sql ("DELETE FROM alert_method_data WHERE alert = %llu", alert);
      index = 0;
      while ((item = (gchar*) g_ptr_array_index (method_data, index++)))
        {
          int ret;
          gchar *name, *data;

          name = sql_quote (item);
          data = sql_quote (item + strlen (item) + 1);

          if (method == ALERT_METHOD_EMAIL
              && strcmp (name, "to_address") == 0
              && validate_email_list (data))
            {
              g_free (name);
              g_free (data);
              sql_rollback ();
              return 6;
            }

          if (method == ALERT_METHOD_EMAIL
              && strcmp (name, "from_address") == 0
              && validate_email (data))
            {
              g_free (name);
              g_free (data);
              sql_rollback ();
              return 6;
            }

          if (method == ALERT_METHOD_EMAIL
              && strcmp (name, "subject") == 0
              && strlen (data) > 80)
            {
              g_free (name);
              g_free (data);
              sql_rollback ();
              return 9;
            }

          if (method == ALERT_METHOD_EMAIL
              && strcmp (name, "message") == 0
              && strlen (data) > max_email_message_length)
            {
              g_free (name);
              g_free (data);
              sql_rollback ();
              return 10;
            }

          ret = validate_scp_data (method, name, &data);
          if (ret)
            {
              g_free (name);
              g_free (data);
              sql_rollback ();
              return ret;
            }

          ret = validate_send_data (method, name, &data);
          if (ret)
            {
              g_free (name);
              g_free (data);
              sql_rollback ();
              return ret;
            }

          ret = validate_smb_data (method, name, &data);
          if (ret)
            {
              g_free (name);
              g_free (data);
              sql_rollback ();
              return ret;
            }

          ret = validate_tippingpoint_data (method, name, &data);
          if (ret)
            {
              g_free (name);
              g_free (data);
              sql_rollback ();
              return ret;
            }

          sql ("INSERT INTO alert_method_data (alert, name, data)"
               " VALUES (%llu, '%s', '%s');",
               alert,
               name,
               data);
          g_free (name);
          g_free (data);
        }
    }

  sql_commit ();

  return 0;
}

/**
 * @brief Delete an alert.
 *
 * @param[in]  alert_id  UUID of alert.
 * @param[in]  ultimate      Whether to remove entirely, or to trashcan.
 *
 * @return 0 success, 1 fail because a task refers to the alert, 2 failed
 *         to find target, 99 permission denied, -1 error.
 */
int
delete_alert (const char *alert_id, int ultimate)
{
  alert_t alert = 0;

  sql_begin_immediate ();

  if (acl_user_may ("delete_alert") == 0)
    {
      sql_rollback ();
      return 99;
    }

  if (find_alert_with_permission (alert_id, &alert, "delete_alert"))
    {
      sql_rollback ();
      return -1;
    }

  if (alert == 0)
    {
      if (find_trash ("alert", alert_id, &alert))
        {
          sql_rollback ();
          return -1;
        }
      if (alert == 0)
        {
          sql_rollback ();
          return 2;
        }
      if (ultimate == 0)
        {
          /* It's already in the trashcan. */
          sql_commit ();
          return 0;
        }

      /* Check if it's in use by a task in the trashcan. */
      if (sql_int ("SELECT count(*) FROM task_alerts"
                   " WHERE alert = %llu"
                   " AND alert_location = " G_STRINGIFY (LOCATION_TRASH) ";",
                   alert))
        {
          sql_rollback ();
          return 1;
        }

      permissions_set_orphans ("alert", alert, LOCATION_TRASH);
      tags_set_orphans ("alert", alert, LOCATION_TRASH);

      sql ("DELETE FROM alert_condition_data_trash WHERE alert = %llu;",
           alert);
      sql ("DELETE FROM alert_event_data_trash WHERE alert = %llu;",
           alert);
      sql ("DELETE FROM alert_method_data_trash WHERE alert = %llu;",
           alert);
      sql ("DELETE FROM alerts_trash WHERE id = %llu;", alert);
      sql_commit ();
      return 0;
    }

  if (ultimate == 0)
    {
      alert_t trash_alert;

      if (sql_int ("SELECT count(*) FROM task_alerts"
                   " WHERE alert = %llu"
                   " AND alert_location = " G_STRINGIFY (LOCATION_TABLE)
                   " AND (SELECT hidden < 2 FROM tasks"
                   "      WHERE id = task_alerts.task);",
                   alert))
        {
          sql_rollback ();
          return 1;
        }

      sql ("INSERT INTO alerts_trash"
           " (uuid, owner, name, comment, event, condition, method, filter,"
           "  filter_location, creation_time, modification_time)"
           " SELECT uuid, owner, name, comment, event, condition, method,"
           "        filter, " G_STRINGIFY (LOCATION_TABLE) ", creation_time,"
           "        m_now ()"
           " FROM alerts WHERE id = %llu;",
           alert);

      trash_alert = sql_last_insert_id ();

      sql ("INSERT INTO alert_condition_data_trash"
           " (alert, name, data)"
           " SELECT %llu, name, data"
           " FROM alert_condition_data WHERE alert = %llu;",
           trash_alert,
           alert);

      sql ("INSERT INTO alert_event_data_trash"
           " (alert, name, data)"
           " SELECT %llu, name, data"
           " FROM alert_event_data WHERE alert = %llu;",
           trash_alert,
           alert);

      sql ("INSERT INTO alert_method_data_trash"
           " (alert, name, data)"
           " SELECT %llu, name, data"
           " FROM alert_method_data WHERE alert = %llu;",
           trash_alert,
           alert);

      /* Update the location of the alert in any trashcan tasks. */
      sql ("UPDATE task_alerts"
           " SET alert = %llu,"
           "     alert_location = " G_STRINGIFY (LOCATION_TRASH)
           " WHERE alert = %llu"
           " AND alert_location = " G_STRINGIFY (LOCATION_TABLE) ";",
           trash_alert,
           alert);

      permissions_set_locations ("alert", alert, trash_alert,
                                 LOCATION_TRASH);
      tags_set_locations ("alert", alert, trash_alert,
                          LOCATION_TRASH);
    }
  else if (sql_int ("SELECT count(*) FROM task_alerts"
                    " WHERE alert = %llu"
                    " AND alert_location = " G_STRINGIFY (LOCATION_TABLE) ";",
                    alert))
    {
      sql_rollback ();
      return 1;
    }
  else
    {
      permissions_set_orphans ("alert", alert, LOCATION_TABLE);
      tags_set_orphans ("alert", alert, LOCATION_TABLE);
    }

  sql ("DELETE FROM alert_condition_data WHERE alert = %llu;",
       alert);
  sql ("DELETE FROM alert_event_data WHERE alert = %llu;", alert);
  sql ("DELETE FROM alert_method_data WHERE alert = %llu;", alert);
  sql ("DELETE FROM alerts WHERE id = %llu;", alert);
  sql_commit ();
  return 0;
}

/**
 * @brief Return the UUID of an alert.
 *
 * @param[in]  alert  Alert.
 *
 * @return UUID of alert.
 */
char *
alert_uuid (alert_t alert)
{
  return sql_string ("SELECT uuid FROM alerts WHERE id = %llu;",
                     alert);
}

/**
 * @brief Return the name of an alert.
 *
 * @param[in]  alert  Alert.
 *
 * @return Name of alert.
 */
char *
alert_name (alert_t alert)
{
  return sql_string ("SELECT name FROM alerts WHERE id = %llu;", alert);
}

/**
 * @brief Return the UUID of the owner of an alert.
 *
 * @param[in]  alert  Alert.
 *
 * @return UUID of owner.
 */
char *
alert_owner_uuid (alert_t alert)
{
  return sql_string ("SELECT uuid FROM users"
                     " WHERE id = (SELECT owner FROM alerts WHERE id = %llu);",
                     alert);
}

/**
 * @brief Return the UUID of the filter of an alert.
 *
 * @param[in]  alert  Alert.
 *
 * @return UUID if there's a filter, else NULL.
 */
char *
alert_filter_id (alert_t alert)
{
  return sql_string ("SELECT"
                     " (CASE WHEN (SELECT filter IS NULL OR filter = 0"
                     "             FROM alerts WHERE id = %llu)"
                     "  THEN NULL"
                     "  ELSE (SELECT uuid FROM filters"
                     "        WHERE id = (SELECT filter FROM alerts"
                     "                    WHERE id = %llu))"
                     "  END);",
                     alert,
                     alert);
}

/**
 * @brief Return the condition associated with an alert.
 *
 * @param[in]  alert  Alert.
 *
 * @return Condition.
 */
static alert_condition_t
alert_condition (alert_t alert)
{
  return sql_int ("SELECT condition FROM alerts WHERE id = %llu;",
                  alert);
}

/**
 * @brief Return the method associated with an alert.
 *
 * @param[in]  alert  Alert.
 *
 * @return Method.
 */
static alert_method_t
alert_method (alert_t alert)
{
  return sql_int ("SELECT method FROM alerts WHERE id = %llu;",
                  alert);
}

/**
 * @brief Return the event associated with an alert.
 *
 * @param[in]  alert  Alert.
 *
 * @return Event.
 */
static event_t
alert_event (alert_t alert)
{
  return sql_int ("SELECT event FROM alerts WHERE id = %llu;",
                  alert);
}

/**
 * @brief Filter columns for alert iterator.
 */
#define ALERT_ITERATOR_FILTER_COLUMNS                                         \
 { GET_ITERATOR_FILTER_COLUMNS, "event", "condition", "method",               \
   "filter",  NULL }

/**
 * @brief Alert iterator columns.
 */
#define ALERT_ITERATOR_COLUMNS                                                \
 {                                                                            \
   GET_ITERATOR_COLUMNS (alerts),                                             \
   { "event", NULL, KEYWORD_TYPE_INTEGER },                                   \
   { "condition", NULL, KEYWORD_TYPE_INTEGER },                               \
   { "method", NULL, KEYWORD_TYPE_INTEGER },                                  \
   { "filter", NULL, KEYWORD_TYPE_INTEGER },                                  \
   { G_STRINGIFY (LOCATION_TABLE), NULL, KEYWORD_TYPE_INTEGER },              \
   { NULL, NULL, KEYWORD_TYPE_UNKNOWN }                                       \
 }

/**
 * @brief Alert iterator columns for trash case.
 */
#define ALERT_ITERATOR_TRASH_COLUMNS                                          \
 {                                                                            \
   GET_ITERATOR_COLUMNS (alerts_trash),                                       \
   { "event", NULL, KEYWORD_TYPE_INTEGER },                                   \
   { "condition", NULL, KEYWORD_TYPE_INTEGER },                               \
   { "method", NULL, KEYWORD_TYPE_INTEGER },                                  \
   { "filter", NULL, KEYWORD_TYPE_STRING },                                   \
   { "filter_location", NULL, KEYWORD_TYPE_INTEGER},                          \
   { NULL, NULL, KEYWORD_TYPE_UNKNOWN }                                       \
 }

/**
 * @brief Count the number of alerts.
 *
 * @param[in]  get  GET params.
 *
 * @return Total number of alerts filtered set.
 */
int
alert_count (const get_data_t *get)
{
  static const char *filter_columns[] = ALERT_ITERATOR_FILTER_COLUMNS;
  static column_t columns[] = ALERT_ITERATOR_COLUMNS;
  static column_t trash_columns[] = ALERT_ITERATOR_TRASH_COLUMNS;
  return count ("alert", get, columns, trash_columns, filter_columns, 0, 0, 0,
                  TRUE);
}

/**
 * @brief Return whether a alert is in use by a task.
 *
 * @param[in]  alert  Alert.
 *
 * @return 1 if in use, else 0.
 */
int
alert_in_use (alert_t alert)
{
  return !!sql_int ("SELECT count (*) FROM task_alerts WHERE alert = %llu;",
                    alert);
}

/**
 * @brief Return whether a trashcan alert is in use by a task.
 *
 * @param[in]  alert  Alert.
 *
 * @return 1 if in use, else 0.
 */
int
trash_alert_in_use (alert_t alert)
{
  return !!sql_int ("SELECT count(*) FROM task_alerts"
                    " WHERE alert = %llu"
                    " AND alert_location = " G_STRINGIFY (LOCATION_TRASH),
                    alert);
}

/**
 * @brief Return whether a alert is writable.
 *
 * @param[in]  alert  Alert.
 *
 * @return 1 if writable, else 0.
 */
int
alert_writable (alert_t alert)
{
    return 1;
}

/**
 * @brief Return whether a trashcan alert is writable.
 *
 * @param[in]  alert  Alert.
 *
 * @return 1 if writable, else 0.
 */
int
trash_alert_writable (alert_t alert)
{
    return 1;
}

/**
 * @brief Initialise an alert iterator, including observed alerts.
 *
 * @param[in]  iterator    Iterator.
 * @param[in]  get         GET data.
 *
 * @return 0 success, 1 failed to find alert, failed to find filter (filt_id),
 *         -1 error.
 */
int
init_alert_iterator (iterator_t* iterator, const get_data_t *get)
{
  static const char *filter_columns[] = ALERT_ITERATOR_FILTER_COLUMNS;
  static column_t columns[] = ALERT_ITERATOR_COLUMNS;
  static column_t trash_columns[] = ALERT_ITERATOR_TRASH_COLUMNS;

  return init_get_iterator (iterator,
                            "alert",
                            get,
                            columns,
                            trash_columns,
                            filter_columns,
                            0,
                            NULL,
                            NULL,
                            TRUE);
}

/**
 * @brief Return the event from an alert iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return Event of the alert or NULL if iteration is complete.
 */
int
alert_iterator_event (iterator_t* iterator)
{
  int ret;
  if (iterator->done) return -1;
  ret = iterator_int (iterator, GET_ITERATOR_COLUMN_COUNT);
  return ret;
}

/**
 * @brief Return the condition from an alert iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return Condition of the alert or NULL if iteration is complete.
 */
int
alert_iterator_condition (iterator_t* iterator)
{
  int ret;
  if (iterator->done) return -1;
  ret = iterator_int (iterator, GET_ITERATOR_COLUMN_COUNT + 1);
  return ret;
}

/**
 * @brief Return the method from an alert iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return Method of the alert or NULL if iteration is complete.
 */
int
alert_iterator_method (iterator_t* iterator)
{
  int ret;
  if (iterator->done) return -1;
  ret = iterator_int (iterator, GET_ITERATOR_COLUMN_COUNT + 2);
  return ret;
}

/**
 * @brief Return the filter from an alert iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return Filter of the alert or NULL if iteration is complete.
 */
filter_t
alert_iterator_filter (iterator_t* iterator)
{
  if (iterator->done) return -1;
  return (filter_t) iterator_int64 (iterator, GET_ITERATOR_COLUMN_COUNT + 3);
}

/**
 * @brief Return the filter UUID from an alert iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return UUID of filter of the alert or NULL if iteration is complete.
 */
char *
alert_iterator_filter_uuid (iterator_t* iterator)
{
  filter_t filter;

  if (iterator->done) return NULL;

  filter = alert_iterator_filter (iterator);
  if (filter)
    {
      if (iterator_int (iterator, GET_ITERATOR_COLUMN_COUNT + 4)
          == LOCATION_TABLE)
        return filter_uuid (filter);
      return trash_filter_uuid (filter);
    }
  return NULL;
}

/**
 * @brief Return the filter name from an alert iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return Name of filter of the alert or NULL if iteration is complete.
 */
char *
alert_iterator_filter_name (iterator_t* iterator)
{
  filter_t filter;

  if (iterator->done) return NULL;

  filter = alert_iterator_filter (iterator);
  if (filter)
    {
      if (iterator_int (iterator, GET_ITERATOR_COLUMN_COUNT + 4)
          == LOCATION_TABLE)
        return filter_name (filter);
      return trash_filter_name (filter);
    }
  return NULL;
}

/**
 * @brief Return the location of an alert iterator filter.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return 0 in table, 1 in trash.
 */
int
alert_iterator_filter_trash (iterator_t* iterator)
{
  if (iterator->done) return 0;
  if (alert_iterator_filter (iterator)
      && (iterator_int (iterator, GET_ITERATOR_COLUMN_COUNT + 4)
          == LOCATION_TRASH))
    return 1;
  return 0;
}

/**
 * @brief Return the filter readable state from an alert iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return Whether filter is readable.
 */
int
alert_iterator_filter_readable (iterator_t* iterator)
{
  filter_t filter;

  if (iterator->done) return 0;

  filter = alert_iterator_filter (iterator);
  if (filter)
    {
      char *uuid;
      uuid = alert_iterator_filter_uuid (iterator);
      if (uuid)
        {
          int readable;
          readable = acl_user_has_access_uuid
                      ("filter", uuid, "get_filters",
                       iterator_int (iterator, GET_ITERATOR_COLUMN_COUNT + 4)
                       == LOCATION_TRASH);
          free (uuid);
          return readable;
        }
    }
  return 0;
}

/**
 * @brief Initialise an alert data iterator.
 *
 * @param[in]  iterator   Iterator.
 * @param[in]  alert  Alert.
 * @param[in]  trash      Whether to iterate over trashcan alert data.
 * @param[in]  table      Type of data: "condition", "event" or "method",
 *                        corresponds to substring of the table to select
 *                        from.
 */
void
init_alert_data_iterator (iterator_t *iterator, alert_t alert,
                          int trash, const char *table)
{
  init_iterator (iterator,
                 "SELECT name, data FROM alert_%s_data%s"
                 " WHERE alert = %llu;",
                 table,
                 trash ? "_trash" : "",
                 alert);
}

/**
 * @brief Return the name from an alert data iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return Name of the alert data or NULL if iteration is complete.
 */
const char*
alert_data_iterator_name (iterator_t* iterator)
{
  const char *ret;
  if (iterator->done) return NULL;
  ret = iterator_string (iterator, 0);
  return ret;
}

/**
 * @brief Return the data from an alert data iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 *
 * @return Data of the alert data or NULL if iteration is complete.
 */
const char*
alert_data_iterator_data (iterator_t* iterator)
{
  const char *ret;
  if (iterator->done) return NULL;
  ret = iterator_string (iterator, 1);
  return ret;
}

/**
 * @brief Return data associated with an alert.
 *
 * @param[in]  alert  Alert.
 * @param[in]  type       Type of data: "condition", "event" or "method".
 * @param[in]  name       Name of the data.
 *
 * @return Freshly allocated data if it exists, else NULL.
 */
char *
alert_data (alert_t alert, const char *type, const char *name)
{
  gchar *quoted_name;
  char *data;

  assert (strcmp (type, "condition") == 0
          || strcmp (type, "event") == 0
          || strcmp (type, "method") == 0);

  quoted_name = sql_quote (name);
  data = sql_string ("SELECT data FROM alert_%s_data"
                     " WHERE alert = %llu AND name = '%s';",
                     type,
                     alert,
                     quoted_name);
  g_free (quoted_name);
  return data;
}

/**
 * @brief Initialise a task alert iterator.
 *
 * @param[in]  iterator  Iterator.
 * @param[in]  task      Task.
 * @param[in]  event     Event.
 */
void
init_task_alert_iterator (iterator_t* iterator, task_t task, event_t event)
{
  gchar *owned_clause;
  get_data_t get;
  array_t *permissions;

  get.trash = 0;
  permissions = make_array ();
  array_add (permissions, g_strdup ("get_alerts"));
  owned_clause = acl_where_owned ("alert", &get, 0, "any", 0, permissions);
  array_free (permissions);

  if (task && event)
    init_iterator (iterator,
                   "SELECT alerts.id, alerts.uuid, alerts.name"
                   " FROM alerts, task_alerts"
                   " WHERE task_alerts.task = %llu AND event = %i"
                   " AND task_alerts.alert = alerts.id"
                   " AND %s;",
                   task,
                   event,
                   owned_clause);
  else if (task)
    init_iterator (iterator,
                   "SELECT alerts.id, alerts.uuid, alerts.name"
                   " FROM alerts, task_alerts"
                   " WHERE task_alerts.task = %llu"
                   " AND task_alerts.alert = alerts.id"
                   " AND %s;",
                   task,
                   owned_clause);
  else if (event)
    init_iterator (iterator,
                   "SELECT alerts.id, alerts.uuid, alerts.name"
                   " FROM alerts"
                   " WHERE event = %i"
                   " AND %s;",
                   event,
                   owned_clause);
  else
    init_iterator (iterator,
                   "SELECT alerts.id, alerts.uuid, alerts.name"
                   " FROM alerts"
                   " WHERE %s;",
                   owned_clause);

  g_free (owned_clause);
}

/**
 * @brief Get the alert from a task alert iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return alert.
 */
alert_t
task_alert_iterator_alert (iterator_t* iterator)
{
  if (iterator->done) return 0;
  return (task_t) iterator_int64 (iterator, 0);
}

/**
 * @brief Get the UUID from a task alert iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return UUID, or NULL if iteration is complete.  Freed by
 *         cleanup_iterator.
 */
DEF_ACCESS (task_alert_iterator_uuid, 1);

/**
 * @brief Get the name from a task alert iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return Name, or NULL if iteration is complete.  Freed by
 *         cleanup_iterator.
 */
DEF_ACCESS (task_alert_iterator_name, 2);

/**
 * @brief Send an email.
 *
 * @param[in]  to_address    Address to send to.
 * @param[in]  from_address  Address to send to.
 * @param[in]  subject       Subject of email.
 * @param[in]  body          Body of email.
 * @param[in]  attachment    Attachment in line broken base64, or NULL.
 * @param[in]  attachment_type  Attachment MIME type, or NULL.
 * @param[in]  attachment_name  Base file name of the attachment, or NULL.
 * @param[in]  attachment_extension  Attachment file extension, or NULL.
 *
 * @return 0 success, -1 error.
 */
static int
email (const char *to_address, const char *from_address, const char *subject,
       const char *body, const gchar *attachment, const char *attachment_type,
       const char *attachment_name, const char *attachment_extension)
{
  int ret, content_fd, to_fd;
  gchar *command;
  GError *error = NULL;
  char content_file_name[] = "/tmp/openvasmd-content-XXXXXX";
  char to_file_name[] = "/tmp/openvasmd-to-XXXXXX";
  FILE *content_file;

  content_fd = mkstemp (content_file_name);
  if (content_fd == -1)
    {
      g_warning ("%s: mkstemp: %s\n", __FUNCTION__, strerror (errno));
      return -1;
    }

  g_debug ("   EMAIL to %s from %s subject: %s, body: %s",
          to_address, from_address, subject, body);

  content_file = fdopen (content_fd, "w");
  if (content_file == NULL)
    {
      g_warning ("%s: %s", __FUNCTION__, strerror (errno));
      close (content_fd);
      return -1;
    }

  if (fprintf (content_file,
               "To: %s\n"
               "From: %s\n"
               "Subject: %s\n"
               "%s%s%s"
               "\n"
               "%s"
               "%s\n",
               to_address,
               from_address ? from_address
                            : "automated@openvas.org",
               subject,
               (attachment
                 ? "MIME-Version: 1.0\n"
                   "Content-Type: multipart/mixed;"
                   " boundary=\""
                 : "Content-Type: text/plain; charset=utf-8\n"
                   "Content-Transfer-Encoding: 8bit"),
               /* @todo Future callers may give email containing this string. */
               (attachment ? "=-=-=-=-=" : ""),
               (attachment ? "\"\n" : ""),
               (attachment ? "--=-=-=-=-=\n"
                             "Content-Type: text/plain; charset=utf-8\n"
                             "Content-Transfer-Encoding: 8bit\n"
                             "Content-Disposition: inline\n"
                             "\n"
                           : ""),
               body)
      < 0)
    {
      g_warning ("%s: output error", __FUNCTION__);
      fclose (content_file);
      return -1;
    }

  if (attachment)
    {
      int len;

      if (fprintf (content_file,
                   "--=-=-=-=-=\n"
                   "Content-Type: %s\n"
                   "Content-Disposition: attachment;"
                   " filename=\"%s.%s\"\n"
                   "Content-Transfer-Encoding: base64\n"
                   "Content-Description: Report\n\n",
                   attachment_type,
                   attachment_name,
                   attachment_extension)
          < 0)
        {
          g_warning ("%s: output error", __FUNCTION__);
          fclose (content_file);
          return -1;
        }

      len = strlen (attachment);
      while (len)
        if (len > 72)
          {
            if (fprintf (content_file,
                         "%.*s\n",
                         72,
                         attachment)
                < 0)
              {
                g_warning ("%s: output error", __FUNCTION__);
                fclose (content_file);
                return -1;
              }
            attachment += 72;
            len -= 72;
          }
        else
          {
            if (fprintf (content_file,
                         "%s\n",
                         attachment)
                < 0)
              {
                g_warning ("%s: output error", __FUNCTION__);
                fclose (content_file);
                return -1;
              }
            break;
          }

      if (fprintf (content_file,
                   "--=-=-=-=-=--\n")
          < 0)
        {
          g_warning ("%s: output error", __FUNCTION__);
          fclose (content_file);
          return -1;
        }
    }

  while (fflush (content_file))
    if (errno == EINTR)
      continue;
    else
      {
        g_warning ("%s", strerror (errno));
        fclose (content_file);
        return -1;
      }

  to_fd = mkstemp (to_file_name);
  if (to_fd == -1)
    {
      g_warning ("%s: mkstemp: %s\n", __FUNCTION__, strerror (errno));
      fclose (content_file);
      return -1;
    }

  g_file_set_contents (to_file_name, to_address, strlen (to_address), &error);
  if (error)
    {
      g_warning ("%s", error->message);
      g_error_free (error);
      fclose (content_file);
      close (to_fd);
      return -1;
    }

  command = g_strdup_printf ("xargs -a %s -I XXX"
                             " /usr/sbin/sendmail XXX < %s"
                             " > /dev/null 2>&1",
                             to_file_name,
                             content_file_name);

  g_debug ("   command: %s\n", command);

  ret = system (command);
  if ((ret == -1) || WEXITSTATUS (ret))
    {
      g_warning ("%s: system failed with ret %i, %i, %s\n",
                 __FUNCTION__,
                 ret,
                 WEXITSTATUS (ret),
                 command);
      g_free (command);
      fclose (content_file);
      close (to_fd);
      unlink (content_file_name);
      unlink (to_file_name);
      return -1;
    }
  g_free (command);
  fclose (content_file);
  close (to_fd);
  unlink (content_file_name);
  unlink (to_file_name);
  return 0;
}

/**
 * @brief GET an HTTP resource.
 *
 * @param[in]  url  URL.
 *
 * @return 0 success, -1 error.
 */
static int
http_get (const char *url)
{
  int ret;
  gchar *standard_out = NULL;
  gchar *standard_err = NULL;
  gint exit_status;
  gchar **cmd;

  g_debug ("   HTTP_GET %s", url);

  cmd = (gchar **) g_malloc (5 * sizeof (gchar *));
  cmd[0] = g_strdup ("/usr/bin/wget");
  cmd[1] = g_strdup ("-O");
  cmd[2] = g_strdup ("-");
  cmd[3] = g_strdup (url);
  cmd[4] = NULL;
  g_debug ("%s: Spawning in /tmp/: %s %s %s %s\n",
           __FUNCTION__, cmd[0], cmd[1], cmd[2], cmd[3]);
  if ((g_spawn_sync ("/tmp/",
                     cmd,
                     NULL,                  /* Environment. */
                     G_SPAWN_SEARCH_PATH,
                     NULL,                  /* Setup function. */
                     NULL,
                     &standard_out,
                     &standard_err,
                     &exit_status,
                     NULL)
       == FALSE)
      || (WIFEXITED (exit_status) == 0)
      || WEXITSTATUS (exit_status))
    {
      g_debug ("%s: wget failed: %d (WIF %i, WEX %i)",
               __FUNCTION__,
               exit_status,
               WIFEXITED (exit_status),
               WEXITSTATUS (exit_status));
      g_debug ("%s: stdout: %s\n", __FUNCTION__, standard_out);
      g_debug ("%s: stderr: %s\n", __FUNCTION__, standard_err);
      ret = -1;
    }
  else
    {
      if (strlen (standard_out) > 80)
        standard_out[80] = '\0';
      g_debug ("   HTTP_GET %s: %s", url, standard_out);
      ret = 0;
    }

  g_free (cmd[0]);
  g_free (cmd[1]);
  g_free (cmd[2]);
  g_free (cmd[3]);
  g_free (cmd[4]);
  g_free (cmd);
  g_free (standard_out);
  g_free (standard_err);
  return ret;
}

/**
 * @brief Initialize common files and variables for an alert script.
 *
 * The temporary file / dir parameters will be modified by mkdtemp / mkstemp
 *  to contain the actual path.
 * The extra data is meant for data that should not be logged like passwords.
 *
 * @param[in]     report_filename Filename for the report or NULL for default.
 * @param[in]     report          Report that should be sent.
 * @param[in]     report_size     Size of the report.
 * @param[in]     extra_content   Optional extra data, e.g. credentials
 * @param[in]     extra_size      Optional extra data length
 * @param[in,out] report_dir      Template for temporary report directory
 * @param[out]    report_path     Pointer to store path to report file at
 * @param[out]    error_file      Temporary file for error messages
 * @param[out]    extra_file      Temporary extra data file
 *
 * @return 0 success, -1 error.
 */
static int
alert_script_init (const char *report_filename, const char* report,
                   size_t report_size,
                   const char *extra_content, size_t extra_size,
                   char *report_dir,
                   gchar **report_path, gchar **error_path, gchar **extra_path)
{
  GError *error;

  /* Create temp directory */

  if (mkdtemp (report_dir) == NULL)
    {
      g_warning ("%s: mkdtemp failed\n", __FUNCTION__);
      return -1;
    }

  /* Create report file */

  *report_path = g_strdup_printf ("%s/%s",
                                  report_dir,
                                  report_filename ? report_filename
                                                  : "report");

  error = NULL;
  g_file_set_contents (*report_path, report, report_size, &error);
  if (error)
    {
      g_warning ("%s: could not write report: %s",
                 __FUNCTION__, error->message);
      g_error_free (error);
      g_free (*report_path);
      openvas_file_remove_recurse (report_dir);
      return -1;
    }

  /* Create error file */

  *error_path = g_strdup_printf ("%s/error_XXXXXX", report_dir);

  if (mkstemp (*error_path) == -1)
    {
      g_warning ("%s: mkstemp for error output failed\n", __FUNCTION__);
      openvas_file_remove_recurse (report_dir);
      g_free (*report_path);
      g_free (*error_path);
      return -1;
    }

  /* Create extra data file */

  if (extra_content)
    {
      *extra_path = g_strdup_printf ("%s/extra_XXXXXX", report_dir);
      if (mkstemp (*extra_path) == -1)
        {
          g_warning ("%s: mkstemp for extra data failed\n", __FUNCTION__);
          openvas_file_remove_recurse (report_dir);
          g_free (*report_path);
          g_free (*error_path);
          g_free (*extra_path);
          return -1;
        }

      error = NULL;
      g_file_set_contents (*extra_path, extra_content, extra_size, &error);
      if (error)
        {
          g_warning ("%s: could not write extra data: %s",
                    __FUNCTION__, error->message);
          g_error_free (error);
          openvas_file_remove_recurse (report_dir);
          g_free (*report_path);
          g_free (*error_path);
          g_free (*extra_path);
          return -1;
        }
    }
  else
    *extra_path = NULL;

  return 0;
}

/**
 * @brief Execute the alert script.
 *
 * @param[in]  alert_id      UUID of the alert.
 * @param[in]  command_args  Args for the "alert" script.
 * @param[in]  report_path   Path to temporary file containing the report
 * @param[in]  report_dir    Temporary directory for the report
 * @param[in]  error_path    Path to the script error message file
 * @param[in]  extra_path    Path to the extra data file
 * @param[out] message       Custom error message generated by the script
 *
 * @return 0 success, -1 error, -5 alert script failed.
 */
static int
alert_script_exec (const char *alert_id, const char *command_args,
                   const char *report_path, const char *report_dir,
                   const char *error_path, const char *extra_path,
                   gchar **message)
{
  gchar *script, *script_dir;
  GError *error;

  /* Setup script file name. */
  script_dir = g_build_filename (OPENVAS_DATA_DIR,
                                 "openvasmd",
                                 "global_alert_methods",
                                 alert_id,
                                 NULL);

  script = g_build_filename (script_dir, "alert", NULL);

  if (!g_file_test (script, G_FILE_TEST_EXISTS))
    {
      g_warning ("%s: Failed to find alert script: %s\n",
           __FUNCTION__,
           script);
      g_free (script);
      g_free (script_dir);
      return -1;
    }

  /* Run the script */
  {
    gchar *command;
    char *previous_dir;
    int ret;

    /* Change into the script directory. */

    /** @todo NULL arg is glibc extension. */
    previous_dir = getcwd (NULL, 0);
    if (previous_dir == NULL)
      {
        g_warning ("%s: Failed to getcwd: %s\n",
                   __FUNCTION__,
                   strerror (errno));
        g_free (previous_dir);
        g_free (script);
        g_free (script_dir);
        return -1;
      }

    if (chdir (script_dir))
      {
        g_warning ("%s: Failed to chdir: %s\n",
                   __FUNCTION__,
                   strerror (errno));
        g_free (previous_dir);
        g_free (script);
        g_free (script_dir);
        return -1;
      }
    g_free (script_dir);

    /* Call the script. */

    if (extra_path)
      command = g_strdup_printf ("%s %s %s %s"
                                 " > /dev/null 2> %s",
                                 script,
                                 command_args,
                                 extra_path,
                                 report_path,
                                 error_path);
    else
      command = g_strdup_printf ("%s %s %s"
                                 " > /dev/null 2> %s",
                                 script,
                                 command_args,
                                 report_path,
                                 error_path);
    g_free (script);

    g_debug ("   command: %s\n", command);

    if (geteuid () == 0)
      {
        pid_t pid;
        struct passwd *nobody;

        /* Run the command with lower privileges in a fork. */

        nobody = getpwnam ("nobody");
        if ((nobody == NULL)
            || chown (report_dir, nobody->pw_uid, nobody->pw_gid)
            || chown (report_path, nobody->pw_uid, nobody->pw_gid)
            || chown (error_path, nobody->pw_uid, nobody->pw_gid)
            || (extra_path && chown (extra_path, nobody->pw_uid,
                                     nobody->pw_gid)))
          {
            g_warning ("%s: Failed to set permissions for user nobody: %s\n",
                       __FUNCTION__,
                       strerror (errno));
            g_free (previous_dir);
            g_free (command);
            return -1;
          }

        pid = fork ();
        switch (pid)
          {
            case 0:
              {
                /* Child.  Drop privileges, run command, exit. */

                cleanup_manage_process (FALSE);

                if (setgroups (0,NULL))
                  {
                    g_warning ("%s (child): setgroups: %s\n",
                               __FUNCTION__, strerror (errno));
                    exit (EXIT_FAILURE);
                  }
                if (setgid (nobody->pw_gid))
                  {
                    g_warning ("%s (child): setgid: %s\n",
                               __FUNCTION__,
                               strerror (errno));
                    exit (EXIT_FAILURE);
                  }
                if (setuid (nobody->pw_uid))
                  {
                    g_warning ("%s (child): setuid: %s\n",
                               __FUNCTION__,
                               strerror (errno));
                    exit (EXIT_FAILURE);
                  }

                ret = system (command);
                /*
                 * Check shell command exit status, assuming 0 means success.
                 */
                if (ret == -1)
                  {
                    g_warning ("%s (child):"
                               " system failed with ret %i, %i, %s\n",
                               __FUNCTION__,
                               ret,
                               WEXITSTATUS (ret),
                               command);
                    exit (EXIT_FAILURE);
                  }
                else if (ret != 0)
                  {
                    if (g_file_get_contents (error_path, message,
                                             NULL, &error) == FALSE)
                      {
                        g_warning ("%s: failed to test error message: %s",
                                    __FUNCTION__, error->message);
                        g_error_free (error);
                        if (message)
                          g_free (*message);
                        exit (EXIT_FAILURE);
                      }

                    if (message == NULL)
                      exit (EXIT_FAILURE);
                    else if (*message == NULL || strcmp (*message, "") == 0)
                      {
                        g_free (*message);
                        *message
                          = g_strdup_printf ("Exited with code %d.",
                                              WEXITSTATUS (ret));

                        if (g_file_set_contents (error_path, *message,
                                                 strlen (*message),
                                                 &error) == FALSE)
                          {
                            g_warning ("%s: failed to write error message:"
                                        " %s",
                                        __FUNCTION__, error->message);
                            g_error_free (error);
                            g_free (*message);
                            exit (EXIT_FAILURE);
                          }
                      }

                    g_free (*message);
                    exit (2);
                  }

                exit (EXIT_SUCCESS);
              }

            case -1:
              /* Parent when error. */

              g_warning ("%s: Failed to fork: %s\n",
                         __FUNCTION__,
                         strerror (errno));
              if (chdir (previous_dir))
                g_warning ("%s: and chdir failed\n",
                           __FUNCTION__);
              g_free (previous_dir);
              g_free (command);
              return -1;
              break;

            default:
              {
                int status;

                /* Parent on success.  Wait for child, and check result. */

                while (waitpid (pid, &status, 0) < 0)
                  {
                    if (errno == ECHILD)
                      {
                        g_warning ("%s: Failed to get child exit status",
                                   __FUNCTION__);
                        if (chdir (previous_dir))
                          g_warning ("%s: and chdir failed\n",
                                     __FUNCTION__);
                        g_free (previous_dir);
                        return -1;
                      }
                    if (errno == EINTR)
                      continue;
                    g_warning ("%s: wait: %s",
                               __FUNCTION__,
                               strerror (errno));
                    if (chdir (previous_dir))
                      g_warning ("%s: and chdir failed\n",
                                 __FUNCTION__);
                    g_free (previous_dir);
                    return -1;
                  }
                if (WIFEXITED (status))
                  switch (WEXITSTATUS (status))
                    {
                    case EXIT_SUCCESS:
                      break;
                    case 2: // script failed
                      if (message)
                        {
                          GError *error = NULL;
                          if (g_file_get_contents (error_path, message,
                                                   NULL, &error) == FALSE)
                            {
                              g_warning ("%s: failed to get error message: %s",
                                         __FUNCTION__, error->message);
                              g_error_free (error);
                            }

                          if (strcmp (*message, "") == 0)
                            {
                              g_free (*message);
                              *message = NULL;
                            }
                        }
                      if (chdir (previous_dir))
                        g_warning ("%s: chdir failed\n",
                                   __FUNCTION__);
                      return -5;
                    case EXIT_FAILURE:
                    default:
                      g_warning ("%s: child failed, %s\n",
                                 __FUNCTION__,
                                 command);
                      if (chdir (previous_dir))
                        g_warning ("%s: and chdir failed\n",
                                   __FUNCTION__);
                      g_free (previous_dir);
                      return -1;
                    }
                else
                  {
                    g_warning ("%s: child failed, %s\n",
                               __FUNCTION__,
                               command);
                    if (chdir (previous_dir))
                      g_warning ("%s: and chdir failed\n",
                                 __FUNCTION__);
                    g_free (previous_dir);
                    return -1;
                  }

                /* Child succeeded, continue to process result. */

                break;
              }
          }
      }
    else
      {
        /* Just run the command as the current user. */

        ret = system (command);
        /* Ignore the shell command exit status, because we've not
         * specified what it must be in the past. */
        if (ret == -1)
          {
            g_warning ("%s: system failed with ret %i, %i, %s\n",
                       __FUNCTION__,
                       ret,
                       WEXITSTATUS (ret),
                       command);
            if (chdir (previous_dir))
              g_warning ("%s: and chdir failed\n",
                         __FUNCTION__);
            g_free (previous_dir);
            g_free (command);
            return -1;
          }
        else if (ret)
          {
            if (message)
              {
                GError *error = NULL;
                if (g_file_get_contents (error_path, message, NULL, &error)
                      == FALSE)
                  {
                    g_warning ("%s: failed to get error message: %s",
                               __FUNCTION__, error->message);
                    g_error_free (error);
                  }

                if (strcmp (*message, "") == 0)
                  {
                    g_free (*message);
                    *message = NULL;
                  }

                if (*message == NULL)
                  {
                    *message
                      = g_strdup_printf ("Exited with code %d.",
                                         WEXITSTATUS (ret));
                  }
              }
            g_free (previous_dir);
            g_free (command);
            return -5;
          }
      }

    g_free (command);

    /* Change back to the previous directory. */

    if (chdir (previous_dir))
      {
        g_warning ("%s: Failed to chdir back: %s\n",
                   __FUNCTION__,
                   strerror (errno));
        g_free (previous_dir);
        return -1;
      }
    g_free (previous_dir);
  }
  return 0;
}

/**
 * @brief Clean up common files and variables for running alert script.
 *
 * @param[in]  report_dir   The temporary directory.
 * @param[in]  report_path  The temporary report file path to free.
 * @param[in]  error_path   The temporary error file path to free.
 * @param[in]  extra_path   The temporary extra data file path to free.
 *
 * @return 0 success, -1 error.
 */
static int
alert_script_cleanup (const char *report_dir,
                      gchar *report_path, gchar *error_path, gchar *extra_path)
{
  openvas_file_remove_recurse (report_dir);
  g_free (report_path);
  g_free (error_path);
  g_free (extra_path);
  return 0;
}

/**
 * @brief Run an alert's "alert" script with one file of extra data.
 *
 * @param[in]  alert_id         ID of alert.
 * @param[in]  command_args     Args for the "alert" script.
 * @param[in]  report_filename  Optional report file name, default: "report"
 * @param[in]  report           Report that should be sent.
 * @param[in]  report_size      Size of the report.
 * @param[in]  extra_content    Optional extra data like passwords
 * @param[in]  extra_size       Size of the report.
 * @param[out] script_message   Custom error message of the script.
 *
 * @return 0 success, -1 error, -5 alert script failed.
 */
static int
run_alert_script (const char *alert_id, const char *command_args,
                  const char *report_filename, const char *report,
                  size_t report_size,
                  const char *extra_content, size_t extra_size,
                  gchar **message)
{
  char report_dir[] = "/tmp/openvasmd_alert_XXXXXX";
  gchar *report_path, *error_path, *extra_path;
  int ret;

  if (message)
    *message = NULL;

  if (report == NULL)
    return -1;

  g_debug ("report: %s", report);

  /* Setup files. */
  ret = alert_script_init (report_filename, report, report_size,
                           extra_content, extra_size,
                           report_dir,
                           &report_path, &error_path, &extra_path);
  if (ret)
    return ret;

  /* Run the script */
  ret = alert_script_exec (alert_id, command_args, report_path, report_dir,
                           error_path, extra_path, message);
  if (ret)
    {
      alert_script_cleanup (report_dir, report_path, error_path, extra_path);
      return ret;
    }

  /* Remove the directory. */
  ret = alert_script_cleanup (report_dir, report_path, error_path, extra_path);

  return ret;
}

/**
 * @brief Send an SNMP TRAP to a host.
 *
 * @param[in]  community  Community.
 * @param[in]  agent      Agent.
 * @param[in]  message    Message.
 * @param[out] script_message  Custom error message of the script.
 *
 * @return 0 success, -1 error, -5 alert script failed.
 */
static int
snmp_to_host (const char *community, const char *agent, const char *message,
              gchar **script_message)
{
  gchar *clean_community, *clean_agent, *clean_message, *command_args;
  int ret;

  g_debug ("SNMP to host: %s", agent);

  if (community == NULL || agent == NULL || message == NULL)
    {
      g_warning ("%s: parameter was NULL", __FUNCTION__);
      return -1;
    }

  clean_community = g_shell_quote (community);
  clean_agent = g_shell_quote (agent);
  clean_message = g_shell_quote (message);
  command_args = g_strdup_printf ("%s %s %s", clean_community, clean_agent,
                                  clean_message);
  g_free (clean_community);
  g_free (clean_agent);
  g_free (clean_message);

  ret = run_alert_script ("9d435134-15d3-11e6-bf5c-28d24461215b", command_args,
                          "report", "", 0, NULL, 0, script_message);

  g_free (command_args);
  return ret;
}

/**
 * @brief Send a report to a host via TCP.
 *
 * @param[in]  host         Address of host.
 * @param[in]  port         Port of host.
 * @param[in]  report      Report that should be sent.
 * @param[in]  report_size Size of the report.
 * @param[out] script_message  Custom error message of the script.
 *
 * @return 0 success, -1 error, -5 alert script failed.
 */
static int
send_to_host (const char *host, const char *port,
              const char *report, int report_size,
              gchar **script_message)
{
  gchar *clean_host, *clean_port, *command_args;
  int ret;

  g_debug ("send to host: %s:%s", host, port);

  if (host == NULL)
    return -1;

  clean_host = g_shell_quote (host);
  clean_port = g_shell_quote (port);
  command_args = g_strdup_printf ("%s %s", clean_host, clean_port);
  g_free (clean_host);
  g_free (clean_port);

  ret = run_alert_script ("4a398d42-87c0-11e5-a1c0-28d24461215b", command_args,
                          "report", report, report_size, NULL, 0,
                          script_message);

  g_free (command_args);
  return ret;
}

/**
 * @brief Send a report to a host via TCP.
 *
 * @param[in]  password     Password.
 * @param[in]  username     Username.
 * @param[in]  host         Address of host.
 * @param[in]  path         Destination filename with path.
 * @param[in]  known_hosts  Content for known_hosts file.
 * @param[in]  report       Report that should be sent.
 * @param[in]  report_size  Size of the report.
 * @param[out] script_message  Custom error message of the alert script.
 *
 * @return 0 success, -1 error, -5 alert script failed.
 */
static int
scp_to_host (const char *password, const char *username, const char *host,
             const char *path, const char *known_hosts, const char *report,
             int report_size, gchar **script_message)
{
  gchar *clean_username, *clean_host, *clean_path;
  gchar *clean_known_hosts, *command_args;
  int ret;

  g_debug ("scp to host: %s@%s:%s", username, host, path);

  if (password == NULL || username == NULL || host == NULL || path == NULL)
    return -1;

  if (known_hosts == NULL)
    known_hosts = "";

  clean_username = g_shell_quote (username);
  clean_host = g_shell_quote (host);
  clean_path = g_shell_quote (path);
  clean_known_hosts = g_shell_quote (known_hosts);
  command_args = g_strdup_printf ("%s %s %s %s", clean_username,
                                  clean_host, clean_path, clean_known_hosts);
  g_free (clean_username);
  g_free (clean_host);
  g_free (clean_path);
  g_free (clean_known_hosts);

  ret = run_alert_script ("2db07698-ec49-11e5-bcff-28d24461215b",
                          command_args, "report", report, report_size,
                          password, strlen (password), script_message);

  g_free (command_args);
  return ret;
}

/**
 * @brief Send a report to a host via SMB.
 *
 * @param[in]  password       Password.
 * @param[in]  username       Username.
 * @param[in]  share_path     Name/address of host and name of the share.
 * @param[in]  file_path      Destination filename with path inside the share.
 * @param[in]  report         Report that should be sent.
 * @param[in]  report_size    Size of the report.
 * @param[out] script_message Custom error message of the alert script.
 *
 * @return 0 success, -1 error, -5 alert script failed.
 */
static int
smb_send_to_host (const char *password, const char *username,
                  const char *share_path, const char *file_path,
                  const char *report, gsize report_size,
                  gchar **script_message)
{
  gchar *clean_share_path, *clean_file_path;
  gchar *authfile_content;
  gchar *command_args;
  int ret;

  g_debug ("smb as %s to share: %s, path: %s", username, share_path, file_path);

  if (password == NULL || username == NULL 
      || share_path == NULL || file_path == NULL)
    return -1;

  clean_share_path = g_shell_quote (share_path);
  clean_file_path = g_shell_quote (file_path);
  authfile_content = g_strdup_printf ("username = %s\n"
                                      "password = %s\n",
                                      username, password);
  command_args = g_strdup_printf ("%s %s",
                                  clean_share_path, clean_file_path);
  g_free (clean_share_path);
  g_free (clean_file_path);

  ret = run_alert_script ("c427a688-b653-40ab-a9d0-d6ba842a9d63", command_args,
                          "report", report, report_size,
                          authfile_content, strlen (authfile_content),
                          script_message);

  g_free (authfile_content);
  g_free (command_args);
  return ret;
}

/**
 * @brief Send a report to a Sourcefire Defense Center.
 *
 * @param[in]  ip         IP of center.
 * @param[in]  port       Port of center.
 * @param[in]  pkcs12_64  PKCS12 content in base64.
 * @param[in]  report     Report in "Sourcefire" format.
 *
 * @return 0 success, -1 error.
 */
static int
send_to_sourcefire (const char *ip, const char *port, const char *pkcs12_64,
                    const char *report)
{
  gchar *script, *script_dir;
  gchar *report_file, *pkcs12_file, *pkcs12;
  gchar *clean_ip, *clean_port;
  char report_dir[] = "/tmp/openvasmd_escalate_XXXXXX";
  GError *error;
  gsize pkcs12_len;

  if ((report == NULL) || (ip == NULL) || (port == NULL))
    return -1;

  g_debug ("send to sourcefire: %s:%s", ip, port);
  g_debug ("report: %s", report);

  /* Setup files. */

  if (mkdtemp (report_dir) == NULL)
    {
      g_warning ("%s: mkdtemp failed\n", __FUNCTION__);
      return -1;
    }

  report_file = g_strdup_printf ("%s/report.csv", report_dir);

  error = NULL;
  g_file_set_contents (report_file, report, strlen (report), &error);
  if (error)
    {
      g_warning ("%s", error->message);
      g_error_free (error);
      g_free (report_file);
      return -1;
    }

  pkcs12_file = g_strdup_printf ("%s/pkcs12", report_dir);

  if (strlen (pkcs12_64))
    pkcs12 = (gchar*) g_base64_decode (pkcs12_64, &pkcs12_len);
  else
    {
      pkcs12 = g_strdup ("");
      pkcs12_len = 0;
    }

  error = NULL;
  g_file_set_contents (pkcs12_file, pkcs12, pkcs12_len, &error);
  if (error)
    {
      g_warning ("%s", error->message);
      g_error_free (error);
      g_free (report_file);
      g_free (pkcs12_file);
      return -1;
    }

  /* Setup file names. */

  script_dir = g_build_filename (OPENVAS_DATA_DIR,
                                 "openvasmd",
                                 "global_alert_methods",
                                 "cd1f5a34-6bdc-11e0-9827-002264764cea",
                                 NULL);

  script = g_build_filename (script_dir, "alert", NULL);

  if (!g_file_test (script, G_FILE_TEST_EXISTS))
    {
      g_free (report_file);
      g_free (pkcs12_file);
      g_free (script);
      g_free (script_dir);
      return -1;
    }

  {
    gchar *command;
    char *previous_dir;
    int ret;

    /* Change into the script directory. */

    /** @todo NULL arg is glibc extension. */
    previous_dir = getcwd (NULL, 0);
    if (previous_dir == NULL)
      {
        g_warning ("%s: Failed to getcwd: %s\n",
                   __FUNCTION__,
                   strerror (errno));
        g_free (report_file);
        g_free (pkcs12_file);
        g_free (previous_dir);
        g_free (script);
        g_free (script_dir);
        return -1;
      }

    if (chdir (script_dir))
      {
        g_warning ("%s: Failed to chdir: %s\n",
                   __FUNCTION__,
                   strerror (errno));
        g_free (report_file);
        g_free (pkcs12_file);
        g_free (previous_dir);
        g_free (script);
        g_free (script_dir);
        return -1;
      }
    g_free (script_dir);

    /* Call the script. */

    clean_ip = g_shell_quote (ip);
    clean_port = g_shell_quote (port);

    command = g_strdup_printf ("%s %s %s %s %s > /dev/null"
                               " 2> /dev/null",
                               script,
                               clean_ip,
                               clean_port,
                               pkcs12_file,
                               report_file);
    g_free (script);
    g_free (clean_ip);
    g_free (clean_port);

    g_debug ("   command: %s\n", command);

    if (geteuid () == 0)
      {
        pid_t pid;
        struct passwd *nobody;

        /* Run the command with lower privileges in a fork. */

        nobody = getpwnam ("nobody");
        if ((nobody == NULL)
            || chown (report_dir, nobody->pw_uid, nobody->pw_gid)
            || chown (report_file, nobody->pw_uid, nobody->pw_gid)
            || chown (pkcs12_file, nobody->pw_uid, nobody->pw_gid))
          {
            g_warning ("%s: Failed to set permissions for user nobody: %s\n",
                       __FUNCTION__,
                       strerror (errno));
            g_free (report_file);
            g_free (pkcs12_file);
            g_free (previous_dir);
            return -1;
          }
        g_free (report_file);
        g_free (pkcs12_file);

        pid = fork ();
        switch (pid)
          {
          case 0:
              {
                /* Child.  Drop privileges, run command, exit. */
                cleanup_manage_process (FALSE);

                if (setgroups (0,NULL))
                  {
                    g_warning ("%s (child): setgroups: %s\n",
                               __FUNCTION__, strerror (errno));
                    exit (EXIT_FAILURE);
                  }
                if (setgid (nobody->pw_gid))
                  {
                    g_warning ("%s (child): setgid: %s\n",
                               __FUNCTION__,
                               strerror (errno));
                    exit (EXIT_FAILURE);
                  }
                if (setuid (nobody->pw_uid))
                  {
                    g_warning ("%s (child): setuid: %s\n",
                               __FUNCTION__,
                               strerror (errno));
                    exit (EXIT_FAILURE);
                  }

                ret = system (command);
                /* Ignore the shell command exit status, because we've not
                 * specified what it must be in the past. */
                if (ret == -1)
                  {
                    g_warning ("%s (child):"
                               " system failed with ret %i, %i, %s\n",
                               __FUNCTION__,
                               ret,
                               WEXITSTATUS (ret),
                               command);
                    exit (EXIT_FAILURE);
                  }

                exit (EXIT_SUCCESS);
              }

          case -1:
            /* Parent when error. */

            g_warning ("%s: Failed to fork: %s\n",
                       __FUNCTION__,
                       strerror (errno));
            if (chdir (previous_dir))
              g_warning ("%s: and chdir failed\n",
                         __FUNCTION__);
            g_free (previous_dir);
            g_free (command);
            return -1;
            break;

          default:
              {
                int status;

                /* Parent on success.  Wait for child, and check result. */

                g_free (command);

                while (waitpid (pid, &status, 0) < 0)
                  {
                    if (errno == ECHILD)
                      {
                        g_warning ("%s: Failed to get child exit status",
                                   __FUNCTION__);
                        if (chdir (previous_dir))
                          g_warning ("%s: and chdir failed\n",
                                     __FUNCTION__);
                        g_free (previous_dir);
                        return -1;
                      }
                    if (errno == EINTR)
                      continue;
                    g_warning ("%s: wait: %s",
                               __FUNCTION__,
                               strerror (errno));
                    if (chdir (previous_dir))
                      g_warning ("%s: and chdir failed\n",
                                 __FUNCTION__);
                    g_free (previous_dir);
                    return -1;
                  }
                if (WIFEXITED (status))
                  switch (WEXITSTATUS (status))
                    {
                    case EXIT_SUCCESS:
                      break;
                    case EXIT_FAILURE:
                    default:
                      g_warning ("%s: child failed, %s\n",
                                 __FUNCTION__,
                                 command);
                      if (chdir (previous_dir))
                        g_warning ("%s: and chdir failed\n",
                                   __FUNCTION__);
                      g_free (previous_dir);
                      return -1;
                    }
                else
                  {
                    g_warning ("%s: child failed, %s\n",
                               __FUNCTION__,
                               command);
                    if (chdir (previous_dir))
                      g_warning ("%s: and chdir failed\n",
                                 __FUNCTION__);
                    g_free (previous_dir);
                    return -1;
                  }

                /* Child succeeded, continue to process result. */

                break;
              }
          }
      }
    else
      {
        /* Just run the command as the current user. */
        g_free (report_file);
        g_free (pkcs12_file);

        ret = system (command);
        /* Ignore the shell command exit status, because we've not
         * specified what it must be in the past. */
        if (ret == -1)
          {
            g_warning ("%s: system failed with ret %i, %i, %s\n",
                       __FUNCTION__,
                       ret,
                       WEXITSTATUS (ret),
                       command);
            if (chdir (previous_dir))
              g_warning ("%s: and chdir failed\n",
                         __FUNCTION__);
            g_free (previous_dir);
            g_free (command);
            return -1;
          }

        g_free (command);
      }

    /* Change back to the previous directory. */

    if (chdir (previous_dir))
      {
        g_warning ("%s: Failed to chdir back: %s\n",
                   __FUNCTION__,
                   strerror (errno));
        g_free (previous_dir);
        return -1;
      }
    g_free (previous_dir);

    /* Remove the directory. */

    openvas_file_remove_recurse (report_dir);

    return 0;
  }
}

/**
 * @brief Send a report to a verinice.PRO server.
 *
 * @param[in]  url          URL of the server.
 * @param[in]  username     Username for server access.
 * @param[in]  password     Password for server access.
 * @param[in]  archive      Verinice archive that should be sent.
 * @param[in]  archive_size Size of the verinice archive
 *
 * @return 0 success, -1 error.
 */
static int
send_to_verinice (const char *url, const char *username, const char *password,
                  const char *archive, int archive_size)
{
  gchar *script, *script_dir;
  gchar *archive_file;
  gchar *clean_url, *clean_username, *clean_password;
  char archive_dir[] = "/tmp/openvasmd_alert_XXXXXX";
  GError *error;

  if ((archive == NULL) || (url == NULL))
    return -1;

  g_debug ("send to verinice: %s", url);
  g_debug ("archive: %s", archive);

  /* Setup files. */

  if (mkdtemp (archive_dir) == NULL)
    {
      g_warning ("%s: mkdtemp failed\n", __FUNCTION__);
      return -1;
    }

  archive_file = g_strdup_printf ("%s/archive.vna", archive_dir);

  error = NULL;
  g_file_set_contents (archive_file, archive, archive_size, &error);
  if (error)
    {
      g_warning ("%s", error->message);
      g_error_free (error);
      g_free (archive_file);
      return -1;
    }

  /* Setup file names. */
  script_dir = g_build_filename (OPENVAS_DATA_DIR,
                                 "openvasmd",
                                 "global_alert_methods",
                                 "f9d97653-f89b-41af-9ba1-0f6ee00e9c1a",
                                 NULL);

  script = g_build_filename (script_dir, "alert", NULL);

  if (!g_file_test (script, G_FILE_TEST_EXISTS))
    {
      g_warning ("%s: Failed to find alert script: %s\n",
           __FUNCTION__,
           script);
      g_free (archive_file);
      g_free (script);
      g_free (script_dir);
      return -1;
    }

  {
    gchar *command;
    gchar *log_command; /* Command with password removed. */
    char *previous_dir;
    int ret;

    /* Change into the script directory. */

    /** @todo NULL arg is glibc extension. */
    previous_dir = getcwd (NULL, 0);
    if (previous_dir == NULL)
      {
        g_warning ("%s: Failed to getcwd: %s\n",
                   __FUNCTION__,
                   strerror (errno));
        g_free (archive_file);
        g_free (previous_dir);
        g_free (script);
        g_free (script_dir);
        return -1;
      }

    if (chdir (script_dir))
      {
        g_warning ("%s: Failed to chdir: %s\n",
                   __FUNCTION__,
                   strerror (errno));
        g_free (archive_file);
        g_free (previous_dir);
        g_free (script);
        g_free (script_dir);
        return -1;
      }
    g_free (script_dir);

    /* Call the script. */

    clean_url = g_shell_quote (url);
    clean_username = g_shell_quote (username);
    clean_password = g_shell_quote (password);

    command = g_strdup_printf ("%s %s %s %s %s > /dev/null"
                               " 2> /dev/null",
                               script,
                               clean_url,
                               clean_username,
                               clean_password,
                               archive_file);
    log_command = g_strdup_printf ("%s %s %s ****** %s > /dev/null"
                                   " 2> /dev/null",
                                   script,
                                   clean_url,
                                   clean_username,
                                   archive_file);
    g_free (script);
    g_free (clean_url);
    g_free (clean_username);
    g_free (clean_password);

    g_debug ("   command: %s\n", log_command);

    if (geteuid () == 0)
      {
        pid_t pid;
        struct passwd *nobody;

        /* Run the command with lower privileges in a fork. */

        nobody = getpwnam ("nobody");
        if ((nobody == NULL)
            || chown (archive_dir, nobody->pw_uid, nobody->pw_gid)
            || chown (archive_file, nobody->pw_uid, nobody->pw_gid))
          {
            g_warning ("%s: Failed to set permissions for user nobody: %s\n",
                       __FUNCTION__,
                       strerror (errno));
            g_free (previous_dir);
            g_free (archive_file);
            g_free (command);
            g_free (log_command);
            return -1;
          }
        g_free (archive_file);

        pid = fork ();
        switch (pid)
          {
          case 0:
              {
                /* Child.  Drop privileges, run command, exit. */

                cleanup_manage_process (FALSE);

                if (setgroups (0,NULL))
                  {
                    g_warning ("%s (child): setgroups: %s\n",
                               __FUNCTION__, strerror (errno));
                    exit (EXIT_FAILURE);
                  }
                if (setgid (nobody->pw_gid))
                  {
                    g_warning ("%s (child): setgid: %s\n",
                               __FUNCTION__,
                               strerror (errno));
                    exit (EXIT_FAILURE);
                  }
                if (setuid (nobody->pw_uid))
                  {
                    g_warning ("%s (child): setuid: %s\n",
                               __FUNCTION__,
                               strerror (errno));
                    exit (EXIT_FAILURE);
                  }

                ret = system (command);
                /* Ignore the shell command exit status, because we've not
                 * specified what it must be in the past. */
                if (ret == -1)
                  {
                    g_warning ("%s (child):"
                               " system failed with ret %i, %i, %s\n",
                               __FUNCTION__,
                               ret,
                               WEXITSTATUS (ret),
                               log_command);
                    exit (EXIT_FAILURE);
                  }

                exit (EXIT_SUCCESS);
              }

          case -1:
            /* Parent when error. */

            g_warning ("%s: Failed to fork: %s\n",
                       __FUNCTION__,
                       strerror (errno));
            if (chdir (previous_dir))
              g_warning ("%s: and chdir failed\n",
                         __FUNCTION__);
            g_free (previous_dir);
            g_free (command);
            g_free (log_command);
            return -1;
            break;

          default:
              {
                int status;

                /* Parent on success.  Wait for child, and check result. */

                while (waitpid (pid, &status, 0) < 0)
                  {
                    if (errno == ECHILD)
                      {
                        g_warning ("%s: Failed to get child exit status",
                                   __FUNCTION__);
                        if (chdir (previous_dir))
                          g_warning ("%s: and chdir failed\n",
                                     __FUNCTION__);
                        g_free (previous_dir);
                        return -1;
                      }
                    if (errno == EINTR)
                      continue;
                    g_warning ("%s: wait: %s",
                               __FUNCTION__,
                               strerror (errno));
                    if (chdir (previous_dir))
                      g_warning ("%s: and chdir failed\n",
                                 __FUNCTION__);
                    g_free (previous_dir);
                    return -1;
                  }
                if (WIFEXITED (status))
                  switch (WEXITSTATUS (status))
                    {
                    case EXIT_SUCCESS:
                      break;
                    case EXIT_FAILURE:
                    default:
                      g_warning ("%s: child failed, %s\n",
                                 __FUNCTION__,
                                 log_command);
                      if (chdir (previous_dir))
                        g_warning ("%s: and chdir failed\n",
                                   __FUNCTION__);
                      g_free (previous_dir);
                      return -1;
                    }
                else
                  {
                    g_warning ("%s: child failed, %s\n",
                               __FUNCTION__,
                               log_command);
                    if (chdir (previous_dir))
                      g_warning ("%s: and chdir failed\n",
                                 __FUNCTION__);
                    g_free (previous_dir);
                    return -1;
                  }

                /* Child succeeded, continue to process result. */

                break;
              }
          }
      }
    else
      {
        /* Just run the command as the current user. */
        g_free (archive_file);

        ret = system (command);
        /* Ignore the shell command exit status, because we've not
         * specified what it must be in the past. */
        if (ret == -1)
          {
            g_warning ("%s: system failed with ret %i, %i, %s\n",
                       __FUNCTION__,
                       ret,
                       WEXITSTATUS (ret),
                       log_command);
            if (chdir (previous_dir))
              g_warning ("%s: and chdir failed\n",
                         __FUNCTION__);
            g_free (previous_dir);
            g_free (command);
            return -1;
          }

      }

    g_free (command);
    g_free (log_command);

    /* Change back to the previous directory. */

    if (chdir (previous_dir))
      {
        g_warning ("%s: Failed to chdir back: %s\n",
                   __FUNCTION__,
                   strerror (errno));
        g_free (previous_dir);
        return -1;
      }
    g_free (previous_dir);

    /* Remove the directory. */

    openvas_file_remove_recurse (archive_dir);

    return 0;
  }
}

/**
 * @brief Convert an XML report and send it to a TippingPoint SMS.
 *
 *
 * @return 0 success, -1 error.
 */
static int
send_to_tippingpoint (const char *report, size_t report_size,
                      const char *username, const char *password,
                      const char *hostname, const char *certificate,
                      int cert_workaround, gchar **message)
{
  const char *alert_id = "5b39c481-9137-4876-b734-263849dd96ce";
  char report_dir[] = "/tmp/openvasmd_alert_XXXXXX";
  gchar *auth_config, *report_path, *error_path, *extra_path, *cert_path;
  gchar *command_args, *hostname_clean, *convert_script;
  GError *error = NULL;
  int ret;

  /* Setup auth file contents */
  auth_config = g_strdup_printf ("machine %s\n"
                                 "login %s\n"
                                 "password %s\n",
                                 cert_workaround ? "Tippingpoint" : hostname,
                                 username, password);

  /* Setup common files. */
  ret = alert_script_init ("report", report, report_size,
                           auth_config, strlen (auth_config),
                           report_dir,
                           &report_path, &error_path, &extra_path);
  g_free (auth_config);

  if (ret)
    return ret;

  /* Setup certificate file */
  cert_path = g_build_filename (report_dir, "cacert.pem", NULL);

  if (g_file_set_contents (cert_path,
                           certificate, strlen (certificate),
                           &error) == FALSE)
    {
      g_warning ("%s: Failed to write TLS certificate to file: %s",
                 __FUNCTION__, error->message);
      alert_script_cleanup (report_dir, report_path, error_path, extra_path);
      g_free (cert_path);
      return -1;
    }

  if (geteuid () == 0)
    {
      struct passwd *nobody;

      /* Run the command with lower privileges in a fork. */

      nobody = getpwnam ("nobody");
      if ((nobody == NULL)
          || chown (cert_path, nobody->pw_uid, nobody->pw_gid))
        {
          g_warning ("%s: Failed to set permissions for user nobody: %s\n",
                      __FUNCTION__,
                      strerror (errno));
          g_free (cert_path);
          alert_script_cleanup (report_dir, report_path, error_path,
                                extra_path);
          return -1;
        }
    }

  /* Build args and run the script */
  hostname_clean = g_shell_quote (hostname);
  convert_script = g_build_filename (OPENVAS_DATA_DIR,
                                     "openvasmd",
                                     "global_alert_methods",
                                     alert_id,
                                     "report-convert.py",
                                     NULL);

  command_args = g_strdup_printf ("%s %s %d %s",
                                  hostname_clean, cert_path, cert_workaround,
                                  convert_script);

  g_free (hostname_clean);
  g_free (cert_path);
  g_free (convert_script);

  ret = alert_script_exec (alert_id, command_args, report_path, report_dir,
                           error_path, extra_path, message);
  if (ret)
    {
      alert_script_cleanup (report_dir, report_path, error_path, extra_path);
      return ret;
    }

  /* Remove the directory. */
  ret = alert_script_cleanup (report_dir, report_path, error_path, extra_path);
  return ret;
}

/**
 * @brief Format string for simple notice alert email.
 */
#define SIMPLE_NOTICE_FORMAT                                                  \
 "%s.\n"                                                                      \
 "\n"                                                                         \
 "After the event %s,\n"                                                      \
 "the following condition was met: %s\n"                                      \
 "\n"                                                                         \
 "This email escalation is not configured to provide more details.\n"         \
 "Full details are stored on the scan engine.\n"                              \
 "\n"                                                                         \
 "\n"                                                                         \
 "Note:\n"                                                                    \
 "This email was sent to you as a configured security scan escalation.\n"     \
 "Please contact your local system administrator if you think you\n"          \
 "should not have received it.\n"

/**
 * @brief Format string for simple notice alert email.
 */
#define SECINFO_SIMPLE_NOTICE_FORMAT                                          \
 "%s.\n"                                                                      \
 "\n"                                                                         \
 "After the event %s,\n"                                                      \
 "the following condition was met: %s\n"                                      \
 "\n"                                                                         \
 "This email escalation is not configured to provide more details.\n"         \
 "Full details are stored on the scan engine.\n"                              \
 "\n"                                                                         \
 "\n"                                                                         \
 "Note:\n"                                                                    \
 "This email was sent to you as a configured security scan escalation.\n"     \
 "Please contact your local system administrator if you think you\n"          \
 "should not have received it.\n"

/**
 * @brief Print an alert subject.
 *
 * @param[in]  subject     Format string for subject.
 * @param[in]  event       Event.
 * @param[in]  event_data  Event data.
 * @param[in]  alert       Alert.
 * @param[in]  task        Task.
 * @param[in]  total       Total number of resources (for SecInfo alerts).
 *
 * @return Freshly allocated subject.
 */
static gchar *
alert_subject_print (const gchar *subject, event_t event,
                     const void *event_data,
                     alert_t alert, task_t task, int total)
{
  int formatting;
  const gchar *point, *end;
  GString *new_subject;

  assert (subject);

  new_subject = g_string_new ("");
  for (formatting = 0, point = subject, end = (subject + strlen (subject));
       point < end;
       point++)
    if (formatting)
      {
        switch (*point)
          {
            case '$':
              g_string_append_c (new_subject, '$');
              break;
            case 'd':
              /* Date that the check was last performed. */
              if (event == EVENT_NEW_SECINFO || event == EVENT_UPDATED_SECINFO)
                {
                  char time_string[100];
                  time_t date;
                  struct tm *tm;

                  if (event_data && (strcasecmp (event_data, "nvt") == 0))
                    date = nvts_check_time ();
                  else if (type_is_scap (event_data))
                    date = scap_check_time ();
                  else
                    date = cert_check_time ();
                  tm = localtime (&date);
                  if (strftime (time_string, 98, "%F", tm) == 0)
                    break;
                  g_string_append (new_subject, time_string);
                }
              break;
            case 'e':
              {
                gchar *event_desc;
                event_desc = event_description (event, event_data,
                                                NULL);
                g_string_append (new_subject, event_desc);
                g_free (event_desc);
                break;
              }
            case 'n':
              {
                if (task)
                  {
                    char *name = task_name (task);
                    g_string_append (new_subject, name);
                    free (name);
                  }
                break;
              }
            case 'N':
              {
                /* Alert Name */
                char *name = alert_name (alert);
                g_string_append (new_subject, name);
                free (name);
                break;
              }
            case 'q':
              if (event == EVENT_NEW_SECINFO)
                g_string_append (new_subject, "New");
              else if (event == EVENT_UPDATED_SECINFO)
                g_string_append (new_subject, "Updated");
              break;
            case 's':
              /* Type. */
              if (event == EVENT_NEW_SECINFO || event == EVENT_UPDATED_SECINFO)
                g_string_append (new_subject, type_name (event_data));
              break;
            case 'S':
              /* Type, plural. */
              if (event == EVENT_NEW_SECINFO || event == EVENT_UPDATED_SECINFO)
                g_string_append (new_subject, type_name_plural (event_data));
              break;
            case 'T':
              g_string_append_printf (new_subject, "%i", total);
              break;
            case 'u':
              {
                /* Current user or owner of the Alert */
                if (current_credentials.username
                    && strcmp (current_credentials.username, ""))
                  {
                    g_string_append (new_subject, current_credentials.username);
                  }
                else
                  {
                    char *owner = alert_owner_uuid (alert);
                    gchar *name = user_name (owner);
                    g_string_append (new_subject, name);
                    free (owner);
                    g_free (name);
                  }
                break;
              }
            case 'U':
              {
                /* Alert UUID */
                char *uuid = alert_uuid (task);
                g_string_append (new_subject, uuid);
                free (uuid);
                break;
              }
            default:
              g_string_append_c (new_subject, '$');
              g_string_append_c (new_subject, *point);
              break;
          }
        formatting = 0;
      }
    else if (*point == '$')
      formatting = 1;
    else
      g_string_append_c (new_subject, *point);

  return g_string_free (new_subject, FALSE);
}

/**
 * @brief Print an alert message.
 *
 * @param[in]  message      Format string for message.
 * @param[in]  event        Event.
 * @param[in]  event_data   Event data.
 * @param[in]  task         Task.
 * @param[in]  alert        Alert.
 * @param[in]  condition    Alert condition.
 * @param[in]  format_name  Report format name.
 * @param[in]  filter       Filter.
 * @param[in]  term         Filter term.
 * @param[in]  zone         Timezone.
 * @param[in]  host_summary    Host summary.
 * @param[in]  content         The report, for inlining.
 * @param[in]  content_length  Length of content.
 * @param[in]  truncated       Whether the report was truncated.
 * @param[in]  total        Total number of resources (for SecInfo alerts).
 * @param[in]  max_length   Max allowed length of content.
 *
 * @return Freshly allocated message.
 */
static gchar *
alert_message_print (const gchar *message, event_t event,
                     const void *event_data, task_t task,
                     alert_t alert, alert_condition_t condition,
                     gchar *format_name, filter_t filter,
                     const gchar *term, const gchar *zone,
                     const gchar *host_summary, const gchar *content,
                     gsize content_length, int truncated, int total,
                     int max_length)
{
  int formatting;
  const gchar *point, *end;
  GString *new_message;

  assert (message);

  new_message = g_string_new ("");
  for (formatting = 0, point = message, end = (message + strlen (message));
       point < end;
       point++)
    if (formatting)
      {
        switch (*point)
          {
            case '$':
              g_string_append_c (new_message, '$');
              break;
            case 'c':
              {
                gchar *condition_desc;
                condition_desc = alert_condition_description
                                  (condition, alert);
                g_string_append (new_message, condition_desc);
                g_free (condition_desc);
                break;
              }
            case 'd':
              /* Date that the check was last performed. */
              if (event == EVENT_NEW_SECINFO || event == EVENT_UPDATED_SECINFO)
                {
                  char time_string[100];
                  time_t date;
                  struct tm *tm;

                  if (event_data && (strcasecmp (event_data, "nvt") == 0))
                    date = nvts_check_time ();
                  else if (type_is_scap (event_data))
                    date = scap_check_time ();
                  else
                    date = cert_check_time ();
                  tm = localtime (&date);
                  if (strftime (time_string, 98, "%F", tm) == 0)
                    break;
                  g_string_append (new_message, time_string);
                }
              break;
            case 'e':
              {
                gchar *event_desc;
                event_desc = event_description (event, event_data,
                                                NULL);
                g_string_append (new_message, event_desc);
                g_free (event_desc);
                break;
              }
            case 'H':
              {
                /* Host summary. */

                g_string_append (new_message,
                                 host_summary ? host_summary : "N/A");
                break;
              }
            case 'i':
              {
                if (content)
                  {
                    g_string_append_printf (new_message,
                                            "%.*s",
                                            /* Cast for 64 bit. */
                                            (int) MIN (content_length,
                                                       max_content_length),
                                            content);
                    if (content_length > max_content_length)
                      g_string_append_printf (new_message,
                                              "\n... (report truncated after"
                                              " %i characters)\n",
                                              max_content_length);
                  }

                break;
              }
            case 'n':
              if (task)
                {
                  char *name = task_name (task);
                  g_string_append (new_message, name);
                  free (name);
                }
              break;
            case 'N':
              {
                /* Alert Name */
                char *name = alert_name (alert);
                g_string_append (new_message, name);
                free (name);
                break;
              }
            case 'r':
              {
                /* Report format name. */

                g_string_append (new_message,
                                 format_name ? format_name : "N/A");
                break;
              }
            case 'F':
              {
                /* Name of filter. */

                if (filter)
                  {
                    char *name = filter_name (filter);
                    g_string_append (new_message, name);
                    free (name);
                  }
                else
                  g_string_append (new_message, "N/A");
                break;
              }
            case 'f':
              {
                /* Filter term. */

                g_string_append (new_message, term ? term : "N/A");
                break;
              }
            case 'q':
              {
                if (event == EVENT_NEW_SECINFO)
                  g_string_append (new_message, "New");
                else if (event == EVENT_UPDATED_SECINFO)
                  g_string_append (new_message, "Updated");
                break;
              }
            case 's':
              /* Type. */
              if (event == EVENT_NEW_SECINFO || event == EVENT_UPDATED_SECINFO)
                g_string_append (new_message, type_name (event_data));
              break;
            case 'S':
              /* Type, plural. */
              if (event == EVENT_NEW_SECINFO || event == EVENT_UPDATED_SECINFO)
                g_string_append (new_message, type_name_plural (event_data));
              break;
            case 't':
              {
                if (truncated)
                  g_string_append_printf (new_message,
                                          "Note: This report exceeds the"
                                          " maximum length of %i characters"
                                          " and thus\n"
                                          "was truncated.\n",
                                          max_length);
                break;
              }
            case 'T':
              {
                g_string_append_printf (new_message, "%i", total);
                break;
              }
            case 'u':
              {
                /* Current user or owner of the Alert */
                if (current_credentials.username
                    && strcmp (current_credentials.username, ""))
                  {
                    g_string_append (new_message, current_credentials.username);
                  }
                else
                  {
                    char *owner = alert_owner_uuid (alert);
                    gchar *name = user_name (owner);
                    g_string_append (new_message, name);
                    free (owner);
                    g_free (name);
                  }
                break;
              }
            case 'U':
              {
                /* Alert UUID */
                char *uuid = alert_uuid (task);
                g_string_append (new_message, uuid);
                free (uuid);
                break;
              }
            case 'z':
              {
                /* Timezone. */

                g_string_append (new_message, zone ? zone : "N/A");
                break;
              }

            case 'R':
            default:
              g_string_append_c (new_message, '$');
              g_string_append_c (new_message, *point);
              break;
          }
        formatting = 0;
      }
    else if (*point == '$')
      formatting = 1;
    else
      g_string_append_c (new_message, *point);

  return g_string_free (new_message, FALSE);
}

/**
 * @brief Print an SCP alert file path.
 *
 * @param[in]  message      Format string for message.
 * @param[in]  task         Task.
 *
 * @return Freshly allocated message.
 */
static gchar *
scp_alert_path_print (const gchar *message, task_t task)
{
  int formatting;
  const gchar *point, *end;
  GString *new_message;

  assert (message);

  new_message = g_string_new ("");
  for (formatting = 0, point = message, end = (message + strlen (message));
       point < end;
       point++)
    if (formatting)
      {
        switch (*point)
          {
            case '$':
              g_string_append_c (new_message, '$');
              break;
            case 'n':
              if (task)
                {
                  char *name = task_name (task);
                  g_string_append (new_message, name);
                  free (name);
                }
              break;
          }
        formatting = 0;
      }
    else if (*point == '$')
      formatting = 1;
    else
      g_string_append_c (new_message, *point);

  return g_string_free (new_message, FALSE);
}

/**
 * @brief Build and send email for SecInfo alert.
 *
 * @param[in]  alert       Alert.
 * @param[in]  task        Task.
 * @param[in]  event       Event.
 * @param[in]  event_data  Event data.
 * @param[in]  method      Method from alert.
 * @param[in]  condition   Condition from alert, which was met by event.
 * @param[in]  to_address    To address.
 * @param[in]  from_address  From address.
 *
 * @return 0 success, -1 error, -2 failed to find report format, -3 failed to
 *         find filter.
 */
int
email_secinfo (alert_t alert, task_t task, event_t event,
               const void* event_data, alert_method_t method,
               alert_condition_t condition, const gchar *to_address,
               const gchar *from_address)
{
  gchar *alert_subject, *message, *subject, *example, *list, *type, *base64;
  gchar *body;
  char *notice;
  int ret, count;

  list = new_secinfo_list (event, event_data, alert, &count);

  type = g_strdup (event_data);
  if (type && (example = strstr (type, "_example")))
    example[0] = '\0';

  /* Setup subject. */

  subject = g_strdup_printf
             ("[OpenVAS-Manager] %s %s arrived",
              event == EVENT_NEW_SECINFO ? "New" : "Updated",
              type_name_plural (type ? type : "nvt"));
  alert_subject = alert_data (alert, "method", "subject");
  if (alert_subject && strlen (alert_subject))
    {
      g_free (subject);
      subject = alert_subject_print (alert_subject, event,
                                     type, alert, task, count);
    }
  g_free (alert_subject);

  /* Setup body. */

  notice = alert_data (alert, "method", "notice");

  message = alert_data (alert, "method", "message");
  if (message == NULL || strlen (message) == 0)
    {
      g_free (message);
      if (notice && strcmp (notice, "0") == 0)
        /* Message with inlined report. */
        message = g_strdup (SECINFO_ALERT_MESSAGE_INCLUDE);
      else if (notice && strcmp (notice, "2") == 0)
        /* Message with attached report. */
        message = g_strdup (SECINFO_ALERT_MESSAGE_ATTACH);
      else
        /* Simple notice message. */
        message = NULL;
    }

  base64 = NULL;
  if (list && notice && strcmp (notice, "2") == 0)
    {
      /* Add list as text attachment. */
      if (max_attach_length <= 0
          || strlen (list) <= max_attach_length)
        base64 = g_base64_encode ((guchar*) list,
                                  strlen (list));
    }

  if (message && strlen (message))
    body = alert_message_print (message, event, type,
                                task, alert, condition,
                                NULL, 0, NULL, NULL, NULL,
                                list,
                                list ? strlen (list) : 0,
                                0, count, 0);
  else
    {
      gchar *event_desc, *condition_desc;
      event_desc = event_description (event, event_data, NULL);
      condition_desc = alert_condition_description
                        (condition, alert);
      body = g_strdup_printf (SECINFO_SIMPLE_NOTICE_FORMAT,
                              event_desc,
                              event_desc,
                              condition_desc);
      free (event_desc);
      free (condition_desc);
    }

  g_free (message);
  g_free (list);

  /* Send email. */

  ret = email (to_address, from_address, subject,
               body, base64,
               base64 ? "text/plain" : NULL,
               base64 ? "secinfo-alert" : NULL,
               base64 ? "txt" : NULL);
  g_free (body);
  g_free (type);
  g_free (subject);
  return ret;
}

/**
 * @brief Generate report content for alert
 *
 * @param[in]  alert  The alert the report is generated for.
 * @param[in]  report Report or NULL to get last report of task.
 * @param[in]  task   Task the report belongs to.
 * @param[in]  get    GET data for the report.
 * @param[in]  report_format_data_name  Name of alert data with report format.
 * @param[in]  fallback_format_id       UUID of fallback report format.
 * @param[in]  notes_details     Whether to include details of notes in report.
 * @param[in]  overrides_details Whether to include override details in report.
 * @param[out] content              Report content location.
 * @param[out] content_length       Length of report content.
 * @param[out] extension            File extension of report format.
 * @param[out] content_type         Content type of report format.
 * @param[out] term                 Filter term.
 * @param[out] host_summary         Summary of results per host.
 * @param[out] used_report_format   Report format used.
 *
 * @return 0 success, -1 error, -2 failed to find report format, -3 failed to
 *         find filter.
 */
static int
report_content_for_alert (alert_t alert, report_t report, task_t task,
                          const get_data_t *get,
                          const char *report_format_data_name,
                          const char *fallback_format_id,
                          int notes_details, int overrides_details,
                          gchar **content, gsize *content_length,
                          gchar **extension, gchar **content_type,
                          gchar **term, gchar **report_zone,
                          gchar **host_summary,
                          report_format_t *used_report_format)
{
  report_format_t report_format;
  char *filt_id, *format_uuid;
  filter_t filter;
  get_data_t *alert_filter_get;
  gchar *report_content;

  assert (content);

  // Get filter
  filt_id = alert_filter_id (alert);
  filter = 0;
  if (filt_id)
    {
      if (find_filter_with_permission (filt_id, &filter,
                                       "get_filters"))
        return -1;
      if (filter == 0)
        return -3;
    }

  if (filter)
    {
      alert_filter_get = g_malloc0 (sizeof (get_data_t));
      alert_filter_get->details = get->details;
      alert_filter_get->ignore_pagination = get->ignore_pagination;
      alert_filter_get->ignore_max_rows_per_page 
        = get->ignore_max_rows_per_page;
      alert_filter_get->filt_id = g_strdup (filt_id);
      alert_filter_get->filter = filter_term (filt_id);
    }
  else
    alert_filter_get = NULL;

  // Get last report from task if no report is given
  if (report == 0)
    switch (sql_int64 (&report,
                        "SELECT max (id) FROM reports"
                        " WHERE task = %llu",
                        task))
      {
        case 0:
          if (report)
            break;
        case 1:        /* Too few rows in result of query. */
        case -1:
          return -1;
          break;
        default:       /* Programming error. */
          assert (0);
          return -1;
      }

  // Get report format or use fallback
  if (report_format_data_name)
    format_uuid = alert_data (alert,
                              "method",
                              report_format_data_name);
  else
    format_uuid = NULL;

  if (format_uuid && strlen (format_uuid))
    {
      if (find_report_format_with_permission (format_uuid,
                                              &report_format,
                                              "get_report_formats")
          || (report_format == 0))
        {
          g_warning ("%s: Could not find RFP '%s' for %s",
                     __FUNCTION__, format_uuid,
                     alert_method_name (alert_method (alert)));
          g_free (format_uuid);
          return -2;
        }
      g_free (format_uuid);
    }
  else
    {
      g_free (format_uuid);
      if (find_report_format_with_permission
            (fallback_format_id,
             &report_format,
             "get_report_formats")
          || (report_format == 0))
        {
          g_warning ("%s: Could not find fallback RFP '%s' for %s",
                      __FUNCTION__, fallback_format_id,
                     alert_method_name (alert_method (alert)));
          return -2;
        }
    }

  // Generate report content
  report_content = manage_report (report,
                                  alert_filter_get ? alert_filter_get : get,
                                  report_format,
                                  notes_details,
                                  overrides_details,
                                  NULL, /* Type. */
                                  content_length,
                                  extension,
                                  content_type,
                                  term,
                                  report_zone,
                                  host_summary);

  if (alert_filter_get)
    {
      get_data_reset (alert_filter_get);
      g_free (alert_filter_get);
    }

  if (report_content == NULL)
    return -1;

  *content = report_content;
  *used_report_format = report_format;

  return 0;
}

/**
 * @brief Escalate an event.
 *
 * @param[in]  alert   Alert.
 * @param[in]  task        Task.
 * @param[in]  report      Report.  0 for most recent report.
 * @param[in]  event       Event.
 * @param[in]  event_data  Event data.
 * @param[in]  method      Method from alert.
 * @param[in]  condition   Condition from alert, which was met by event.
 * @param[in]  get         GET data for report.
 * @param[in]  notes_details      If notes, Whether to include details.
 * @param[in]  overrides_details  If overrides, Whether to include details.
 * @param[out] script_message  Custom error message from the script.
 *
 * @return 0 success, -1 error, -2 failed to find report format, -3 failed to
 *         find filter, -4 failed to find credential, -5 alert script failed.
 */
static int
escalate_2 (alert_t alert, task_t task, report_t report, event_t event,
            const void* event_data, alert_method_t method,
            alert_condition_t condition,
            const get_data_t *get, int notes_details, int overrides_details,
            gchar **script_message)
{
  char *name_alert, *name_task;
  gchar *event_desc, *alert_desc;

  if (script_message)
    *script_message = NULL;

  name_alert = alert_name (alert);
  name_task = task_name (task);
  event_desc = event_description (event, event_data, NULL);
  alert_desc = alert_condition_description (condition, alert);
  g_log ("event alert", G_LOG_LEVEL_MESSAGE,
         "The alert %s%s%s was triggered "
         "(Event: %s, Condition: %s)",
         name_alert,
         name_task ? " for task " : "",
         name_task ? name_task : "",
         event_desc,
         alert_desc);
  free (name_task);
  free (name_alert);
  free (event_desc);
  free (alert_desc);

  switch (method)
    {
      case ALERT_METHOD_EMAIL:
        {
          char *to_address;
          char *format_name;
          format_name = NULL;

          to_address = alert_data (alert, "method", "to_address");

          if (to_address)
            {
              int ret;
              gchar *body, *subject;
              char *name, *notice, *from_address, *filt_id;
              gchar *base64, *type, *extension;
              filter_t filter;
              get_data_t *alert_filter_get;

              base64 = NULL;
              type = NULL;
              extension = NULL;

              from_address = alert_data (alert,
                                         "method",
                                         "from_address");

              if (event == EVENT_NEW_SECINFO || event == EVENT_UPDATED_SECINFO)
                {
                  ret = email_secinfo (alert, task, event, event_data, method,
                                       condition, to_address, from_address);
                  free (to_address);
                  free (from_address);
                  return ret;
                }

              notice = alert_data (alert, "method", "notice");
              filt_id = alert_filter_id (alert);
              filter = 0;
              if (filt_id)
                {
                  if (find_filter_with_permission (filt_id, &filter,
                                                   "get_filters"))
                    return -1;
                  if (filter == 0)
                    return -3;
                }

              if (filter)
                {
                  alert_filter_get = g_malloc0 (sizeof (get_data_t));
                  alert_filter_get->details = get->details;
                  alert_filter_get->ignore_pagination = get->ignore_pagination;
                  alert_filter_get->ignore_max_rows_per_page
                    = get->ignore_max_rows_per_page;
                  alert_filter_get->filt_id = g_strdup (filt_id);
                  alert_filter_get->filter = filter_term (filt_id);
                }
              else
                alert_filter_get = NULL;

              name = task_name (task);
              if (notice && strcmp (notice, "0") == 0)
                {
                  gchar *event_desc, *condition_desc, *report_content;
                  gchar *alert_subject, *message;
                  gchar *term, *report_zone, *host_summary;
                  char *format_uuid;
                  report_format_t report_format = 0;
                  gsize content_length;

                  /* Message with inlined report. */

                  if (report == 0)
                    switch (sql_int64 (&report,
                                       "SELECT max (id) FROM reports"
                                       " WHERE task = %llu",
                                       task))
                      {
                        case 0:
                          if (report)
                            break;
                        case 1:        /* Too few rows in result of query. */
                        case -1:
                          free (notice);
                          free (name);
                          free (to_address);
                          free (from_address);
                          if (alert_filter_get)
                            {
                              get_data_reset (alert_filter_get);
                              g_free (alert_filter_get);
                            }
                          return -1;
                          break;
                        default:       /* Programming error. */
                          assert (0);
                          return -1;
                      }

                  format_uuid = alert_data (alert,
                                            "method",
                                            "notice_report_format");
                  if (((format_uuid == NULL)
                       || find_report_format_with_permission
                           (format_uuid, &report_format, "get_report_formats")
                       || (report_format == 0))
                      /* Fallback to TXT. */
                      && (find_report_format_with_permission
                           ("a3810a62-1f62-11e1-9219-406186ea4fc5",
                            &report_format,
                            "get_report_formats")
                          || (report_format == 0)))
                    {
                      g_free (format_uuid);
                      free (filt_id);
                      free (notice);
                      free (name);
                      free (to_address);
                      free (from_address);
                      if (alert_filter_get)
                        {
                          get_data_reset (alert_filter_get);
                          g_free (alert_filter_get);
                        }
                      return -2;
                    }
                  g_free (format_uuid);

                  event_desc = event_description (event, event_data, NULL);
                  term = NULL;
                  report_zone = NULL;
                  host_summary = NULL;
                  report_content = manage_report (report,
                                                  alert_filter_get
                                                    ? alert_filter_get : get,
                                                  report_format,
                                                  notes_details,
                                                  overrides_details,
                                                  NULL, /* Type. */
                                                  &content_length,
                                                  NULL,    /* Extension. */
                                                  NULL,   /* Content type. */
                                                  &term,
                                                  &report_zone,
                                                  &host_summary);
                  if (alert_filter_get)
                    {
                      get_data_reset (alert_filter_get);
                      g_free (alert_filter_get);
                    }
                  if (report_content == NULL)
                    {
                      free (event_desc);
                      free (filt_id);
                      free (notice);
                      free (name);
                      free (to_address);
                      free (from_address);
                      g_free (term);
                      g_free (report_zone);
                      g_free (host_summary);
                      return -1;
                    }
                  format_name = report_format_name (report_format);
                  condition_desc = alert_condition_description (condition,
                                                                alert);
                  subject = g_strdup_printf ("[OpenVAS-Manager] Task '%s': %s",
                                             name ? name : "Internal Error",
                                             event_desc);
                  alert_subject = alert_data (alert, "method", "subject");
                  if (alert_subject && strlen (alert_subject))
                    {
                      g_free (subject);
                      subject = alert_subject_print (alert_subject, event,
                                                     event_data,
                                                     alert, task, 0);
                    }
                  g_free (alert_subject);

                  message = alert_data (alert, "method", "message");
                  if (message == NULL || strlen (message) == 0)
                    {
                      g_free (message);
                      message = g_strdup (ALERT_MESSAGE_INCLUDE);
                    }
                  body = alert_message_print (message, event, event_data,
                                              task, alert, condition,
                                              format_name, filter,
                                              term, report_zone,
                                              host_summary, report_content,
                                              content_length,
                                              content_length
                                              > max_content_length,
                                              0,
                                              max_content_length);
                  g_free (message);
                  g_free (report_content);
                  g_free (event_desc);
                  g_free (condition_desc);
                  g_free (term);
                  g_free (report_zone);
                  g_free (host_summary);
                }
              else if (notice && strcmp (notice, "2") == 0)
                {
                  gchar *event_desc, *condition_desc, *report_content;
                  char *format_uuid;
                  report_format_t report_format = 0;
                  gsize content_length;
                  gchar *alert_subject, *message;
                  gchar *term, *report_zone, *host_summary;

                  /* Message with attached report. */

                  if (report == 0)
                    switch (sql_int64 (&report,
                                       "SELECT max (id) FROM reports"
                                       " WHERE task = %llu",
                                       task))
                      {
                        case 0:
                          if (report)
                            break;
                        case 1:        /* Too few rows in result of query. */
                        case -1:
                          free (notice);
                          free (name);
                          free (to_address);
                          free (from_address);
                          if (alert_filter_get)
                            {
                              get_data_reset (alert_filter_get);
                              g_free (alert_filter_get);
                            }
                          return -1;
                          break;
                        default:       /* Programming error. */
                          assert (0);
                          return -1;
                      }

                  format_uuid = alert_data (alert,
                                            "method",
                                            "notice_attach_format");
                  if (((format_uuid == NULL)
                       || find_report_format_with_permission
                           (format_uuid, &report_format, "get_report_formats")
                       || (report_format == 0))
                      /* Fallback to TXT. */
                      && (find_report_format_with_permission
                           ("19f6f1b3-7128-4433-888c-ccc764fe6ed5",
                            &report_format,
                            "get_report_formats")
                          || (report_format == 0)))
                    {
                      g_free (format_uuid);
                      free (filt_id);
                      free (notice);
                      free (name);
                      free (to_address);
                      free (from_address);
                      if (alert_filter_get)
                        {
                          get_data_reset (alert_filter_get);
                          g_free (alert_filter_get);
                        }
                      return -2;
                    }
                  g_free (format_uuid);

                  event_desc = event_description (event, event_data, NULL);
                  term = NULL;
                  report_zone = NULL;
                  host_summary = NULL;
                  report_content = manage_report (report,
                                                  alert_filter_get
                                                    ? alert_filter_get
                                                    : get,
                                                  report_format,
                                                  notes_details,
                                                  overrides_details,
                                                  NULL, /* Type. */
                                                  &content_length,
                                                  &extension,
                                                  &type,
                                                  &term,
                                                  &report_zone,
                                                  &host_summary);
                  if (alert_filter_get)
                    {
                      get_data_reset (alert_filter_get);
                      g_free (alert_filter_get);
                    }
                  if (report_content == NULL)
                    {
                      g_free (event_desc);
                      free (filt_id);
                      free (notice);
                      free (name);
                      free (to_address);
                      free (from_address);
                      return -1;
                    }
                  format_name = report_format_name (report_format);
                  condition_desc = alert_condition_description (condition,
                                                                    alert);
                  subject = g_strdup_printf ("[OpenVAS-Manager] Task '%s': %s",
                                             name ? name : "Internal Error",
                                             event_desc);
                  alert_subject = alert_data (alert, "method", "subject");
                  if (alert_subject && strlen (alert_subject))
                    {
                      g_free (subject);
                      subject = alert_subject_print (alert_subject, event,
                                                     event_data,
                                                     alert, task, 0);
                    }
                  g_free (alert_subject);
                  if (max_attach_length <= 0
                      || content_length <= max_attach_length)
                    base64 = g_base64_encode ((guchar*) report_content,
                                              content_length);
                  g_free (report_content);
                  message = alert_data (alert, "method", "message");
                  if (message == NULL || strlen (message) == 0)
                    {
                      g_free (message);
                      message = g_strdup (ALERT_MESSAGE_ATTACH);
                    }
                  body = alert_message_print (message, event, event_data,
                                              task, alert, condition,
                                              format_name, filter,
                                              term, report_zone,
                                              host_summary, NULL, 0,
                                              base64 == NULL,
                                              0,
                                              max_attach_length);
                  g_free (message);
                  g_free (event_desc);
                  g_free (condition_desc);
                  g_free (term);
                  g_free (report_zone);
                  g_free (host_summary);
                }
              else
                {
                  gchar *event_desc, *generic_desc, *condition_desc;
                  gchar *alert_subject, *message;

                  /* Simple notice message. */
                  format_name = NULL;
                  event_desc = event_description (event, event_data, name);
                  generic_desc = event_description (event, event_data, NULL);
                  condition_desc = alert_condition_description (condition,
                                                                    alert);

                  subject = g_strdup_printf ("[OpenVAS-Manager] Task '%s':"
                                             " An event occurred",
                                             name);

                  alert_subject = alert_data (alert, "method", "subject");
                  if (alert_subject && strlen (alert_subject))
                    {
                      g_free (subject);
                      subject = alert_subject_print (alert_subject, event,
                                                     event_data,
                                                     alert, task, 0);
                    }
                  g_free (alert_subject);

                  message = alert_data (alert, "method", "message");
                  if (message && strlen (message))
                    body = alert_message_print (message, event, event_data,
                                                task, alert, condition,
                                                NULL, 0, NULL, NULL, NULL,
                                                NULL, 0, 0, 0, 0);
                  else
                    body = g_strdup_printf (SIMPLE_NOTICE_FORMAT,
                                            event_desc,
                                            generic_desc,
                                            condition_desc);
                  g_free (message);
                  g_free (event_desc);
                  g_free (generic_desc);
                  g_free (condition_desc);
                  if (alert_filter_get)
                    {
                      get_data_reset (alert_filter_get);
                      g_free (alert_filter_get);
                    }
                }
              free (filt_id);
              free (notice);

              gchar *fname_format, *file_name;
              gchar *report_id, *creation_time, *modification_time;

              fname_format
                = sql_string ("SELECT value FROM settings"
                              " WHERE name"
                              "       = 'Report Export File Name'"
                              " AND " ACL_USER_OWNS ()
                              " ORDER BY coalesce (owner, 0) DESC LIMIT 1;",
                              current_credentials.uuid);

              report_id = report_uuid (report);

              creation_time
                = sql_string ("SELECT iso_time (start_time)"
                              " FROM reports"
                              " WHERE id = %llu",
                              report);

              modification_time
                = sql_string ("SELECT iso_time (end_time)"
                              " FROM reports"
                              " WHERE id = %llu",
                              report);

              file_name
                = openvas_export_file_name (fname_format,
                                            current_credentials.username,
                                            "report", report_id,
                                            creation_time, modification_time,
                                            name, format_name);
              ret = email (to_address, from_address, subject, body, base64,
                           type, file_name ? file_name : "openvas-report",
                           extension);

              free (extension);
              free (type);
              free (name);
              free (format_name);
              g_free (base64);
              free (to_address);
              free (from_address);
              g_free (subject);
              g_free (body);
              g_free (fname_format);
              g_free (file_name);
              g_free (report_id);
              g_free (creation_time);
              g_free (modification_time);
              return ret;
            }
          return -1;
        }
      case ALERT_METHOD_HTTP_GET:
        {
          char *url;

          if (event == EVENT_NEW_SECINFO || event == EVENT_UPDATED_SECINFO)
            {
              g_warning ("%s: Event \"%s NVTs arrived\" with method"
                         " \"HTTP Get\" not support",
                         __FUNCTION__,
                         event == EVENT_NEW_SECINFO ? "New" : "Updated");
              return -1;
            }

          url = alert_data (alert, "method", "URL");

          if (url)
            {
              int ret, formatting;
              gchar *point, *end;
              GString *new_url;

              new_url = g_string_new ("");
              for (formatting = 0, point = url, end = (url + strlen (url));
                   point < end;
                   point++)
                if (formatting)
                  {
                    switch (*point)
                      {
                        case '$':
                          g_string_append_c (new_url, '$');
                          break;
                        case 'c':
                          {
                            gchar *condition_desc;
                            condition_desc = alert_condition_description
                                              (condition, alert);
                            g_string_append (new_url, condition_desc);
                            g_free (condition_desc);
                            break;
                          }
                        case 'e':
                          {
                            gchar *event_desc;
                            event_desc = event_description (event, event_data,
                                                            NULL);
                            g_string_append (new_url, event_desc);
                            g_free (event_desc);
                            break;
                          }
                        case 'n':
                          {
                            char *name = task_name (task);
                            g_string_append (new_url, name);
                            free (name);
                            break;
                          }
                        default:
                          g_string_append_c (new_url, '$');
                          g_string_append_c (new_url, *point);
                          break;
                      }
                    formatting = 0;
                  }
                else if (*point == '$')
                  formatting = 1;
                else
                  g_string_append_c (new_url, *point);

              ret = http_get (new_url->str);
              g_string_free (new_url, TRUE);
              g_free (url);
              return ret;
            }
          return -1;
        }
      case ALERT_METHOD_SCP:
        {
          credential_t credential;
          char *credential_id;
          char *password, *username, *host, *path, *known_hosts, *filt_id;
          gchar *report_content, *format_uuid, *alert_path;
          gsize content_length;
          report_format_t report_format;
          int ret;
          filter_t filter;
          get_data_t *alert_filter_get;

          if (event == EVENT_NEW_SECINFO || event == EVENT_UPDATED_SECINFO)
            {
              gchar *message;

              credential_id = alert_data (alert, "method", "scp_credential");
              if (find_credential_with_permission (credential_id,
                                                   &credential,
                                                   "get_credentials"))
                {
                  return -1;
                }
              else if (credential == 0)
                {
                  return -4;
                }
              else
                {
                  message = new_secinfo_message (event, event_data, alert);

                  password = credential_encrypted_value (credential,
                                                         "password");
                  username = credential_value (credential, "username");

                  host = alert_data (alert, "method", "scp_host");
                  path = alert_data (alert, "method", "scp_path");
                  known_hosts = alert_data (alert, "method", "scp_known_hosts");

                  alert_path = scp_alert_path_print (path, task);
                  free (path);

                  ret = scp_to_host (password, username, host, alert_path,
                                     known_hosts, message, strlen (message),
                                     script_message);

                  g_free (message);
                  free (password);
                  free (username);
                  free (host);
                  g_free (alert_path);
                  free (known_hosts);

                  return ret;
                }
            }

          format_uuid = alert_data (alert,
                                    "method",
                                    "scp_report_format");
          if (format_uuid && strlen (format_uuid))
            {
              if (find_report_format_with_permission (format_uuid,
                                                      &report_format,
                                                      "get_report_formats")
                  || (report_format == 0))
                {
                  g_warning ("%s: Could not find SCP RFP '%s'", __FUNCTION__,
                             format_uuid);
                  g_free (format_uuid);
                  return -2;
                }
              g_free (format_uuid);
            }
          else
            {
              g_free (format_uuid);
              if (find_report_format_with_permission
                   ("a994b278-1f62-11e1-96ac-406186ea4fc5",
                    &report_format,
                    "get_report_formats")
                  || (report_format == 0))
                {
                  g_warning ("%s: Could not find XML RFP for SCP",
                             __FUNCTION__);
                  return -2;
                }
            }

          if (report == 0)
            switch (sql_int64 (&report,
                               "SELECT max (id) FROM reports"
                               " WHERE task = %llu",
                               task))
              {
                case 0:
                  if (report)
                    break;
                case 1:        /* Too few rows in result of query. */
                case -1:
                  return -1;
                  break;
                default:       /* Programming error. */
                  assert (0);
                  return -1;
              }

          filt_id = alert_filter_id (alert);
          filter = 0;
          if (filt_id)
            {
              if (find_filter_with_permission (filt_id, &filter, "get_filters"))
                return -1;
              if (filter == 0)
                return -3;
            }

          if (filter)
            {
              alert_filter_get = g_malloc0 (sizeof (get_data_t));
              alert_filter_get->details = get->details;
              alert_filter_get->ignore_pagination = get->ignore_pagination;
              alert_filter_get->ignore_max_rows_per_page
                = get->ignore_max_rows_per_page;
              alert_filter_get->filt_id = g_strdup (filt_id);
              alert_filter_get->filter = filter_term (filt_id);
            }
          else
            alert_filter_get = NULL;

          report_content = manage_report (report,
                                          alert_filter_get ? alert_filter_get
                                                           : get,
                                          report_format,
                                          notes_details, overrides_details,
                                          NULL, /* Type. */
                                          &content_length,
                                          NULL,    /* Extension. */
                                          NULL,    /* Content type. */
                                          NULL,
                                          NULL,
                                          NULL);
          free (filt_id);
          if (alert_filter_get)
            {
              get_data_reset (alert_filter_get);
              g_free (alert_filter_get);
            }
          if (report_content == NULL)
            {
              g_warning ("%s: Empty Report", __FUNCTION__);
              return -1;
            }

          credential_id = alert_data (alert, "method", "scp_credential");
          if (find_credential_with_permission (credential_id, &credential,
                                               "get_credentials"))
            {
              g_free (report_content);
              return -1;
            }
          else if (credential == 0)
            {
              g_free (report_content);
              return -4;
            }
          else
            {
              password = credential_encrypted_value (credential, "password");
              username = credential_value (credential, "username");

              host = alert_data (alert, "method", "scp_host");
              path = alert_data (alert, "method", "scp_path");
              known_hosts = alert_data (alert, "method", "scp_known_hosts");

              alert_path = scp_alert_path_print (path, task);
              free (path);

              ret = scp_to_host (password, username, host, alert_path,
                                 known_hosts, report_content, content_length,
                                 script_message);

              free (password);
              free (username);
              free (host);
              g_free (alert_path);
              free (known_hosts);
            }
          g_free (report_content);

          return ret;
        }
      case ALERT_METHOD_SEND:
        {
          char *host, *port, *filt_id;
          gchar *report_content, *format_uuid;
          gsize content_length;
          report_format_t report_format;
          int ret;
          filter_t filter;
          get_data_t *alert_filter_get;

          if (event == EVENT_NEW_SECINFO || event == EVENT_UPDATED_SECINFO)
            {
              gchar *message;

              message = new_secinfo_message (event, event_data, alert);
              host = alert_data (alert, "method", "send_host");
              port = alert_data (alert, "method", "send_port");

              g_debug ("send host: %s", host);
              g_debug ("send port: %s", port);

              ret = send_to_host (host, port, message, strlen (message),
                                  script_message);

              g_free (message);
              free (host);
              free (port);

              return ret;
            }

          format_uuid = alert_data (alert,
                                    "method",
                                    "send_report_format");
          if (format_uuid && strlen (format_uuid))
            {
              if (find_report_format_with_permission (format_uuid,
                                                      &report_format,
                                                      "get_report_formats")
                  || (report_format == 0))
                {
                  g_warning ("%s: Could not find Send RFP '%s'", __FUNCTION__,
                             format_uuid);
                  g_free (format_uuid);
                  return -2;
                }
              g_free (format_uuid);
            }
          else
            {
              g_free (format_uuid);
              if (find_report_format_with_permission
                   ("a994b278-1f62-11e1-96ac-406186ea4fc5",
                    &report_format,
                    "get_report_formats")
                  || (report_format == 0))
                {
                  g_warning ("%s: Could not find XML RFP for Send",
                             __FUNCTION__);
                  return -2;
                }
            }

          if (report == 0)
            switch (sql_int64 (&report,
                               "SELECT max (id) FROM reports"
                               " WHERE task = %llu",
                               task))
              {
                case 0:
                  if (report)
                    break;
                case 1:        /* Too few rows in result of query. */
                case -1:
                  return -1;
                  break;
                default:       /* Programming error. */
                  assert (0);
                  return -1;
              }

          filt_id = alert_filter_id (alert);
          filter = 0;
          if (filt_id)
            {
              if (find_filter_with_permission (filt_id, &filter, "get_filters"))
                return -1;
              if (filter == 0)
                return -3;
            }

          if (filter)
            {
              alert_filter_get = g_malloc0 (sizeof (get_data_t));
              alert_filter_get->details = get->details;
              alert_filter_get->ignore_pagination = get->ignore_pagination;
              alert_filter_get->ignore_max_rows_per_page
                = get->ignore_max_rows_per_page;
              alert_filter_get->filt_id = g_strdup (filt_id);
              alert_filter_get->filter = filter_term (filt_id);
            }
          else
            alert_filter_get = NULL;

          report_content = manage_report (report,
                                          alert_filter_get ? alert_filter_get
                                                           : get,
                                          report_format,
                                          notes_details, overrides_details,
                                          NULL, /* Type. */
                                          &content_length,
                                          NULL,    /* Extension. */
                                          NULL,    /* Content type. */
                                          NULL,
                                          NULL,
                                          NULL);
          if (alert_filter_get)
            {
              get_data_reset (alert_filter_get);
              g_free (alert_filter_get);
            }
          free (filt_id);
          if (report_content == NULL)
            {
              g_warning ("%s: Empty Report", __FUNCTION__);
              return -1;
            }

          host = alert_data (alert, "method", "send_host");
          port = alert_data (alert, "method", "send_port");

          g_debug ("send host: %s", host);
          g_debug ("send port: %s", port);

          ret = send_to_host (host, port, report_content, content_length,
                              script_message);

          free (host);
          free (port);
          g_free (report_content);

          return ret;
        }
      case ALERT_METHOD_SMB:
        {
          char *credential_id, *username, *password;
          char *share_path, *file_path_format;
          report_format_t report_format;
          char *name, *report_id, *format_name;
          char *creation_time, *modification_time;
          gchar *file_path, *report_content, *extension;
          gsize content_length;
          credential_t credential;
          int ret;

          credential_id = alert_data (alert, "method", "smb_credential");
          share_path = alert_data (alert, "method", "smb_share_path");
          file_path_format = alert_data (alert, "method", "smb_file_path");

          report_content = NULL;
          extension = NULL;
          report_format = 0;

          g_debug ("smb_credential: %s", credential_id);
          g_debug ("smb_share_path: %s", share_path);
          g_debug ("smb_file_path: %s", file_path_format);

          ret = report_content_for_alert
                  (alert, report, task, get,
                   "smb_report_format",
                   "a994b278-1f62-11e1-96ac-406186ea4fc5", /* XML fallback */
                   notes_details, overrides_details,
                   &report_content, &content_length, &extension,
                   NULL, NULL, NULL, NULL, &report_format);
          if (ret || report_content == NULL)
            {
              free (credential_id);
              free (share_path);
              free (file_path_format);
              g_free (report_content);
              g_free (extension);
              return ret ? ret : -1;
            }

          name = task_name (task);
          report_id = report_uuid (report);

          format_name
            = sql_string ("SELECT name"
                          " FROM report_formats"
                          " WHERE id = %llu",
                          report_format);

          creation_time
            = sql_string ("SELECT iso_time (start_time)"
                          " FROM reports"
                          " WHERE id = %llu",
                          report);

          modification_time
            = sql_string ("SELECT iso_time (end_time)"
                          " FROM reports"
                          " WHERE id = %llu",
                          report);

          file_path = openvas_export_file_name (file_path_format,
                                                current_credentials.username,
                                                "report",
                                                report_id,
                                                creation_time,
                                                modification_time,
                                                name,
                                                format_name);

          free (name);
          free (report_id);
          free (format_name);
          free (creation_time);
          free (modification_time);

          credential = 0;
          ret = find_credential_with_permission (credential_id, &credential,
                                                 "get_credentials");
          if (ret || credential == 0)
            {
              if (ret == 0)
                {
                  g_warning ("%s: Could not find credential %s",
                             __FUNCTION__, credential_id);
                }
              free (credential_id);
              free (share_path);
              free (file_path);
              g_free (report_content);
              g_free (extension);
              return ret ? -1 : -4;
            }

          username = credential_value (credential, "username");
          password = credential_encrypted_value (credential, "password");

          ret = smb_send_to_host (password, username, share_path, file_path,
                                  report_content, content_length,
                                  script_message);

          g_free (username);
          g_free (password);
          free (credential_id);
          free (share_path);
          free (file_path);
          g_free (report_content);
          g_free (extension);
          return ret;
        }
      case ALERT_METHOD_SNMP:
        {
          char *community, *agent, *snmp_message;
          int ret;
          gchar *message;

          community = alert_data (alert, "method", "snmp_community");
          agent = alert_data (alert, "method", "snmp_agent");
          snmp_message = alert_data (alert, "method", "snmp_message");

          g_debug ("snmp_message: %s", snmp_message);
          g_debug ("snmp_community: %s", community);
          g_debug ("snmp_agent: %s", agent);

          if (snmp_message)
            {
              if (event == EVENT_NEW_SECINFO || event == EVENT_UPDATED_SECINFO)
                {
                  int count;
                  gchar *list, *example, *type;

                  type = g_strdup (event_data);

                  if (type && (example = strstr (type, "_example")))
                    example[0] = '\0';

                  list = new_secinfo_list (event, event_data, alert, &count);
                  g_free (list);

                  message = alert_subject_print (snmp_message, event, type,
                                                 alert, task, count);

                  g_free (type);
                }
              else
                message = alert_subject_print (snmp_message, event, event_data,
                                               alert, task, 0);
            }
          else
            {
              gchar *event_desc;
              event_desc = event_description (event, event_data, NULL);
              message = g_strdup_printf ("%s", event_desc);
              g_free (event_desc);
            }

          ret = snmp_to_host (community, agent, message, script_message);

          free (agent);
          free (community);
          free (snmp_message);
          g_free (message);
          return ret;
        }
      case ALERT_METHOD_SOURCEFIRE:
        {
          char *ip, *port, *pkcs12, *filt_id;
          gchar *report_content;
          gsize content_length;
          report_format_t report_format;
          int ret;
          filter_t filter;
          get_data_t *alert_filter_get;

          if (event == EVENT_NEW_SECINFO || event == EVENT_UPDATED_SECINFO)
            {
              g_warning ("%s: Event \"%s NVTs arrived\" with method"
                         " \"Sourcefire\" not support",
                         __FUNCTION__,
                         event == EVENT_NEW_SECINFO ? "New" : "Updated");
              return -1;
            }

          if (lookup_report_format ("Sourcefire", &report_format)
              || (report_format == 0))
            return -2;

          if (report == 0)
            switch (sql_int64 (&report,
                               "SELECT max (id) FROM reports"
                               " WHERE task = %llu",
                               task))
              {
                case 0:
                  if (report)
                    break;
                case 1:        /* Too few rows in result of query. */
                case -1:
                  return -1;
                  break;
                default:       /* Programming error. */
                  assert (0);
                  return -1;
              }

          filt_id = alert_filter_id (alert);
          filter = 0;
          if (filt_id)
            {
              if (find_filter_with_permission (filt_id, &filter, "get_filters"))
                return -1;
              if (filter == 0)
                return -3;
            }

          if (filter)
            {
              alert_filter_get = g_malloc0 (sizeof (get_data_t));
              alert_filter_get->details = get->details;
              alert_filter_get->ignore_pagination = get->ignore_pagination;
              alert_filter_get->ignore_max_rows_per_page
                = get->ignore_max_rows_per_page;
              alert_filter_get->filt_id = g_strdup (filt_id);
              alert_filter_get->filter = filter_term (filt_id);
            }
          else
            alert_filter_get = NULL;

          report_content = manage_report (report,
                                          alert_filter_get ? alert_filter_get
                                                           : get,
                                          report_format,
                                          notes_details, overrides_details,
                                          NULL, /* Type. */
                                          &content_length,
                                          NULL,    /* Extension. */
                                          NULL,    /* Content type. */
                                          NULL,
                                          NULL,
                                          NULL);
          if (alert_filter_get)
            {
              get_data_reset (alert_filter_get);
              g_free (alert_filter_get);
            }
          if (report_content == NULL)
            return -1;

          ip = alert_data (alert, "method", "defense_center_ip");
          port = alert_data (alert, "method", "defense_center_port");
          if (port == NULL)
            port = g_strdup ("8307");
          pkcs12 = alert_data (alert, "method", "pkcs12");

          g_debug ("  sourcefire   ip: %s", ip);
          g_debug ("  sourcefire port: %s", port);
          g_debug ("sourcefire pkcs12: %s", pkcs12);

          ret = send_to_sourcefire (ip, port, pkcs12, report_content);

          free (filt_id);
          free (ip);
          g_free (port);
          free (pkcs12);
          g_free (report_content);

          return ret;
        }
      case ALERT_METHOD_SYSLOG:
        {
          char *submethod;
          gchar *message, *event_desc, *level;

          event_desc = event_description (event, event_data, NULL);
          message = g_strdup_printf ("%s: %s", event_name (event), event_desc);
          g_free (event_desc);

          submethod = alert_data (alert, "method", "submethod");
          level = g_strdup_printf ("event %s", submethod);
          g_free (submethod);

          g_debug ("  syslog level: %s", level);
          g_debug ("syslog message: %s", message);

          g_log (level, G_LOG_LEVEL_MESSAGE, "%s", message);

          g_free (level);
          g_free (message);

          return 0;
        }
      case ALERT_METHOD_TIPPINGPOINT:
        {
          int ret;
          report_format_t report_format;
          gchar *report_content, *extension;
          size_t content_length;
          char *credential_id, *username, *password, *hostname, *certificate;
          credential_t credential;
          char *tls_cert_workaround_str;
          int tls_cert_workaround;

          /* TLS certificate subject workaround setting */
          tls_cert_workaround_str 
            = alert_data (alert, "method",
                          "tp_sms_tls_workaround");
          if (tls_cert_workaround_str)
            tls_cert_workaround = !!(atoi (tls_cert_workaround_str));
          else
            tls_cert_workaround = 0;
          g_free (tls_cert_workaround_str);

          /* SSL / TLS Certificate */
          certificate = alert_data (alert, "method",
                                    "tp_sms_tls_certificate");

          /* Hostname / IP address */
          hostname = alert_data (alert, "method",
                                 "tp_sms_hostname");

          /* Credential */
          credential_id = alert_data (alert, "method",
                                      "tp_sms_credential");
          if (find_credential_with_permission (credential_id, &credential,
                                               "get_credentials"))
            {
              g_free (certificate);
              g_free (hostname);
              g_free (credential_id);
              return -1;
            }
          else if (credential == 0)
            {
              g_free (certificate);
              g_free (hostname);
              g_free (credential_id);
              return -4;
            }
          else
            {
              g_free (credential_id);
              username = credential_value (credential, "username");
              password = credential_encrypted_value (credential, "password");
            }

          /* Report content */
          extension = NULL;
          ret = report_content_for_alert
                  (alert, report, task, get,
                   NULL, /* Report format not configurable */
                   "a994b278-1f62-11e1-96ac-406186ea4fc5", /* XML fallback */
                   notes_details, overrides_details,
                   &report_content, &content_length, &extension,
                   NULL, NULL, NULL, NULL, &report_format);
          g_free (extension);
          if (ret)
            {
              g_free (username);
              g_free (password);
              g_free (hostname);
              g_free (certificate);
              return ret;
            }

          /* Send report */
          ret = send_to_tippingpoint (report_content, content_length,
                                      username, password, hostname,
                                      certificate, tls_cert_workaround,
                                      script_message);

          g_free (username);
          g_free (password);
          g_free (hostname);
          g_free (certificate);
          g_free (report_content);
          return ret;
        }
      case ALERT_METHOD_VERINICE:
        {
          credential_t credential;
          char *url, *username, *password, *filt_id;
          gchar *credential_id, *report_content, *format_uuid;
          gsize content_length;
          report_format_t report_format;
          int ret;
          filter_t filter;
          get_data_t *alert_filter_get;

          if (event == EVENT_NEW_SECINFO || event == EVENT_UPDATED_SECINFO)
            {
              g_warning ("%s: Event \"%s NVTs arrived\" with method"
                         " \"Verinice\" not support",
                         __FUNCTION__,
                         event == EVENT_NEW_SECINFO ? "New" : "Updated");
              return -1;
            }

          format_uuid = alert_data (alert,
                                    "method",
                                    "verinice_server_report_format");
          if (format_uuid && strlen (format_uuid))
            {
              if (find_report_format_with_permission (format_uuid,
                                                      &report_format,
                                                      "get_report_formats")
                  || (report_format == 0))
                {
                  g_warning ("Could not find Verinice RFP '%s'", format_uuid);
                  g_free (format_uuid);
                  return -2;
                }
              g_free (format_uuid);
            }
          else if (lookup_report_format ("Verinice ISM", &report_format)
              || (report_format == 0))
            {
              g_warning ("Could not find default verinice RFP");
              return -2;
            }

          if (report == 0)
            switch (sql_int64 (&report,
                               "SELECT max (id) FROM reports"
                               " WHERE task = %llu",
                               task))
              {
                case 0:
                  if (report)
                    break;
                case 1:        /* Too few rows in result of query. */
                case -1:
                  return -1;
                  break;
                default:       /* Programming error. */
                  assert (0);
                  return -1;
              }

          filt_id = alert_filter_id (alert);
          filter = 0;
          if (filt_id)
            {
              if (find_filter_with_permission (filt_id, &filter, "get_filters"))
                return -1;
              if (filter == 0)
                return -3;
            }

          if (filter)
            {
              alert_filter_get = g_malloc0 (sizeof (get_data_t));
              alert_filter_get->details = get->details;
              alert_filter_get->ignore_pagination = get->ignore_pagination;
              alert_filter_get->ignore_max_rows_per_page
                = get->ignore_max_rows_per_page;
              alert_filter_get->filt_id = g_strdup (filt_id);
              alert_filter_get->filter = filter_term (filt_id);
            }
          else
            alert_filter_get = NULL;

          report_content = manage_report (report,
                                          alert_filter_get ? alert_filter_get
                                                           : get,
                                          report_format,
                                          notes_details, overrides_details,
                                          NULL, /* Type. */
                                          &content_length,
                                          NULL,    /* Extension. */
                                          NULL,    /* Content type. */
                                          NULL,
                                          NULL,
                                          NULL);
          if (alert_filter_get)
            {
              get_data_reset (alert_filter_get);
              g_free (alert_filter_get);
            }
          if (report_content == NULL)
            {
              g_warning ("Empty Report");
              return -1;
            }

          url = alert_data (alert, "method", "verinice_server_url");

          credential_id = alert_data (alert, "method",
                                      "verinice_server_credential");
          if (find_credential_with_permission (credential_id, &credential,
                                               "get_credentials"))
            {
              free (filt_id);
              free (url);
              g_free (report_content);
              return -1;
            }
          else if (credential == 0)
            {
              free (filt_id);
              free (url);
              g_free (report_content);
              return -4;
            }
          else
            {
              username = credential_value (credential, "username");
              password = credential_encrypted_value (credential, "password");

              g_debug ("    verinice  url: %s", url);
              g_debug ("verinice username: %s", username);

              ret = send_to_verinice (url, username, password, report_content,
                                      content_length);

              free (filt_id);
              free (url);
              g_free (username);
              g_free (password);
              g_free (report_content);

              return ret;
            }
        }
      case ALERT_METHOD_START_TASK:
        {
          openvas_connection_t connection;
          char *task_id, *report_id;
          omp_authenticate_info_opts_t auth_opts;

          if (event == EVENT_NEW_SECINFO || event == EVENT_UPDATED_SECINFO)
            {
              g_warning ("%s: Event \"%s NVTs arrived\" with method"
                         " \"Start Task\" not support",
                         __FUNCTION__,
                         event == EVENT_NEW_SECINFO ? "New" : "Updated");
              return -1;
            }

          /* Run the callback to fork a child connected to the Manager. */

          if (manage_fork_connection == NULL)
            {
              g_warning ("%s: no connection fork available\n", __FUNCTION__);
              return -1;
            }

          task_id = alert_data (alert, "method", "start_task_task");

          switch (manage_fork_connection (&connection, current_credentials.uuid))
            {
              case 0:
                /* Child.  Break, stop task, exit. */
                break;

              case -1:
                /* Parent on error. */
                g_free (task_id);
                g_warning ("%s: fork failed\n", __FUNCTION__);
                return -1;
                break;

              default:
                /* Parent.  Continue with whatever lead to this escalation. */
                g_free (task_id);
                return 0;
                break;
            }

          /* Start the task. */

          auth_opts = omp_authenticate_info_opts_defaults;
          auth_opts.username = current_credentials.username;
          if (omp_authenticate_info_ext_c (&connection, auth_opts))
            {
              openvas_connection_free (&connection);
              exit (EXIT_FAILURE);
            }

          if (omp_start_task_report_c (&connection, task_id, &report_id))
            {
              g_free (task_id);
              openvas_connection_free (&connection);
              exit (EXIT_FAILURE);
            }

          g_free (task_id);
          g_free (report_id);
          openvas_connection_free (&connection);
          exit (EXIT_SUCCESS);
        }
      case ALERT_METHOD_ERROR:
      default:
        break;
    }
  return -1;
}

/**
 * @brief Escalate an event with preset report filtering.
 *
 * @param[in]  alert       Alert.
 * @param[in]  task        Task.
 * @param[in]  report      Report.
 * @param[in]  event       Event.
 * @param[in]  event_data  Event data.
 * @param[in]  method      Method from alert.
 * @param[in]  condition   Condition from alert, which was met by event.
 * @param[out] script_message  Custom error message from alert script.
 *
 * @return 0 success, -1 error, -2 failed to find report format for alert,
 *         -3 failed to find filter for alert, -4 failed to find credential,
 *         -5 alert script failed.
 */
static int
escalate_1 (alert_t alert, task_t task, report_t report, event_t event,
            const void* event_data, alert_method_t method,
            alert_condition_t condition, gchar **script_message)
{
  int ret;
  get_data_t get;
  char *results_filter;

  memset (&get, 0, sizeof (get_data_t));
  get.details = 1;

  results_filter = setting_filter ("Results");
  if (results_filter && strlen (results_filter))
    {
      get.filt_id = results_filter;
      get.filter = filter_term (results_filter);
    }
  else
    {
      get.filt_id = g_strdup ("0");
      get.filter = g_strdup_printf ("notes=1 overrides=1 sort-reverse=severity"
                                    " rows=%d",
                                    method == ALERT_METHOD_EMAIL ? 1000 : -1);
    }

  ret = escalate_2 (alert, task, report, event, event_data, method, condition,
                    &get, 1, 1, script_message);
  free (results_filter);
  g_free (get.filter);
  return ret;
}

/**
 * @brief Escalate an alert with task and event data.
 *
 * @param[in]  alert_id    Alert UUID.
 * @param[in]  task_id     Task UUID.
 * @param[in]  event       Event.
 * @param[in]  event_data  Event data.
 * @param[out] script_message  Custom error message from alert script.
 *
 * @return 0 success, 1 failed to find alert, 2 failed to find task,
 *         99 permission denied, -1 error, -2 failed to find report format
 *         for alert, -3 failed to find filter for alert, -4 failed to find
 *         credential for alert, -5 alert script failed.
 */
int
manage_alert (const char *alert_id, const char *task_id, event_t event,
              const void* event_data, gchar **script_message)
{
  alert_t alert;
  task_t task;
  alert_condition_t condition;
  alert_method_t method;

  if (acl_user_may ("test_alert") == 0)
    return 99;

  if (find_alert_with_permission (alert_id, &alert, "test_alert"))
    return -1;
  if (alert == 0)
    return 1;

  if (task_id == NULL || strcmp (task_id, "0") == 0)
    task = 0;
  else
    {
      if (find_task_with_permission (task_id, &task, NULL))
        return -1;
      if (task == 0)
        return 2;
    }

  condition = alert_condition (alert);
  method = alert_method (alert);
  return escalate_1 (alert, task, 0, event, event_data, method, condition,
                     script_message);
}

/**
 * @brief Header for "New NVTs" alert message.
 */
#define NEW_NVTS_HEADER                                                       \
/* Open-Xchange (OX) AppSuite XHTML File HTML Injection Vuln...  NoneAvailable       0.0 100% */ \
  "Name                                                          Solution Type  Severity  QOD\n" \
  "------------------------------------------------------------------------------------------\n"

/**
 * @brief Header for "New NVTs" alert message, when there's an OID.
 */
#define NEW_NVTS_HEADER_OID                                                   \
/* Open-Xchange (OX) AppSuite XHTML File HTML Injection Vuln...  NoneAvailable       0.0 100%  1.3... */ \
  "Name                                                          Solution Type  Severity  QOD  OID\n" \
  "------------------------------------------------------------------------------------------------\n"

/**
 * @brief Header for "New CVEs" alert message.
 */
#define NEW_CVES_HEADER                                                         \
/* CVE-2014-100001       6.8  Cross-site request forgery (CSRF) vulnerability in... */ \
  "Name             Severity  Description\n"                                    \
  "--------------------------------------------------------------------------------\n"

/**
 * @brief Header for "New CPEs" alert message.
 */
#define NEW_CPES_HEADER                                                        \
/* cpe:/a:.joomclan:com_joomclip                                 1024cms... */ \
  "Name                                                          Title\n"      \
  "------------------------------------------------------------------------------------------\n"

/**
 * @brief Header for "New CERT-Bund Advisories" alert message.
 */
#define NEW_CERT_BUNDS_HEADER                                                       \
/* CB-K13/0849  Novell SUSE Linux Enterprise Server: Mehrere Schwachstellen... */   \
  "Name         Title\n"                                                            \
  "------------------------------------------------------------------------------------------\n"

/**
 * @brief Header for "New DFN-CERT Advisories" alert message.
 */
#define NEW_DFN_CERTS_HEADER                                                   \
/* DFN-CERT-2008-1100  Denial of Service Schwachstelle in der... */            \
  "Name                Title\n"                                                \
  "------------------------------------------------------------------------------------------\n"

/**
 * @brief Header for "New CERT-Bund Advisories" alert message.
 */
#define NEW_OVAL_DEFS_HEADER                                                   \
/* oval:org.mitre.oval:def:100116  libtiff Malloc Error Denial of Service */   \
  "OVAL ID                         Title\n"                                    \
  "------------------------------------------------------------------------------------------\n"

/**
 * @brief Test an alert.
 *
 * @param[in]  alert_id    Alert UUID.
 * @param[out] script_message  Custom message from the alert script.
 *
 * @return 0 success, 1 failed to find alert, 2 failed to find task,
 *         99 permission denied, -1 error, -2 failed to find report format
 *         for alert, -3 failed to find filter for alert, -4 failed to find
 *         credential for alert, -5 alert script failed.
 */
int
manage_test_alert (const char *alert_id, gchar **script_message)
{
  int ret;
  alert_t alert;
  task_t task;
  report_t report;
  result_t result;
  char *task_id, *report_id;
  time_t now;
  const char *now_string;
  gchar *clean;

  if (acl_user_may ("test_alert") == 0)
    return 99;

  if (find_alert_with_permission (alert_id, &alert, "test_alert"))
    return -1;
  if (alert == 0)
    return 1;

  if (alert_event (alert) == EVENT_NEW_SECINFO
      || alert_event (alert) == EVENT_UPDATED_SECINFO)
    {
      int ret;
      char *alert_event_data;
      gchar *type;

      alert_event_data = alert_data (alert, "event", "secinfo_type");
      type = g_strdup_printf ("%s_example", alert_event_data ?: "NVT");
      free (alert_event_data);

      if (alert_event (alert) == EVENT_NEW_SECINFO)
        ret = manage_alert (alert_id, "0", EVENT_NEW_SECINFO, (void*) type,
                            script_message);
      else
        ret = manage_alert (alert_id, "0", EVENT_UPDATED_SECINFO, (void*) type,
                            script_message);

      g_free (type);

      return ret;
    }

  task = make_task (g_strdup ("Temporary Task for Alert"),
                    g_strdup (""),
                    0,  /* Exclude from assets. */
                    0); /* Skip event and log. */

  report_id = openvas_uuid_make ();
  if (report_id == NULL)
    return -1;
  task_uuid (task, &task_id);
  report = make_report (task, report_id, TASK_STATUS_DONE);
  result = make_result (task, "127.0.0.1", "telnet (23/tcp)",
                        "1.3.6.1.4.1.25623.1.0.10330", "Alarm",
                        "A telnet server seems to be running on this port.");
  if (result == 0)
    {
      ret = -1;
      goto exit;
    }
  now = time (NULL);
  now_string = ctime (&now);
  if (strlen (now_string) == 0)
    {
      ret = -1;
      goto exit;
    }
  clean = g_strdup (now_string);
  if (clean[strlen (clean) - 1] == '\n')
    clean[strlen (clean) - 1] = '\0';
  set_task_start_time_otp (task, g_strdup (clean));
  set_scan_start_time_otp (report, g_strdup (clean));
  set_scan_host_start_time_otp (report, "127.0.0.1", clean);
  report_add_result (report, result);
  set_scan_host_end_time_otp (report, "127.0.0.1", clean);
  set_scan_end_time_otp (report, clean);
  g_free (clean);
  ret = manage_alert (alert_id,
                      task_id,
                      EVENT_TASK_RUN_STATUS_CHANGED,
                      (void*) TASK_STATUS_DONE,
                      script_message);
 exit:
  delete_task (task, 1);
  free (task_id);
  free (report_id);
  return ret;
}

/**
 * @brief Return whether an event applies to a task and an alert.
 *
 * @param[in]  event       Event.
 * @param[in]  event_data  Event data.
 * @param[in]  task        Task.
 * @param[in]  alert       Alert.
 *
 * @return 1 if event applies, else 0.
 */
static int
event_applies (event_t event, const void *event_data, task_t task,
               alert_t alert)
{
  switch (event)
    {
      case EVENT_TASK_RUN_STATUS_CHANGED:
        {
          int ret;
          char *alert_event_data;

          alert_event_data = alert_data (alert, "event", "status");
          if (alert_event_data == NULL)
            return 0;
          ret = (task_run_status (task) == (task_status_t) event_data)
                && (strcmp (alert_event_data,
                            run_status_name_internal ((task_status_t)
                                                      event_data))
                    == 0);
          free (alert_event_data);
          return ret;
        }
      case EVENT_NEW_SECINFO:
      case EVENT_UPDATED_SECINFO:
        {
          char *alert_event_data;

          alert_event_data = alert_data (alert, "event", "secinfo_type");
          if (alert_event_data == NULL)
            return 0;
          if (strcasecmp (alert_event_data, event_data) == 0)
            return 1;
          return 0;
        }
      default:
        return 0;
    }
}

/**
 * @brief Return whether the condition of an alert is met by a task.
 *
 * @param[in]  task       Task.
 * @param[in]  report     Report.
 * @param[in]  alert      Alert.
 * @param[in]  condition  Condition.
 *
 * @return 1 if met, else 0.
 */
static int
condition_met (task_t task, report_t report, alert_t alert,
               alert_condition_t condition)
{
  switch (condition)
    {
      case ALERT_CONDITION_ALWAYS:
        return 1;
        break;
      case ALERT_CONDITION_FILTER_COUNT_AT_LEAST:
        {
          char *filter_id, *count_string;
          report_t last_report;
          int debugs, holes, infos, logs, warnings, false_positives;
          int count;
          double severity;

          /* True if there are at least the given number of results matched by
           * the given filter in the last finished report. */

          filter_id = alert_data (alert, "condition", "filter_id");
          count_string = alert_data (alert, "condition", "count");
          if (count_string)
            {
              count = atoi (count_string);
              free (count_string);
            }
          else
            count = 0;

          if (task == 0)
            {
              get_data_t get;
              int db_count, uuid_was_null;

              /* NVT event. */

              if (current_credentials.uuid == NULL)
                {
                  current_credentials.uuid = alert_owner_uuid (alert);
                  uuid_was_null = 1;
                }
              else
                uuid_was_null = 0;

              memset (&get, '\0', sizeof (get));
              if (filter_id && strlen (filter_id) && strcmp (filter_id, "0"))
                get.filt_id = filter_id;
              db_count = nvt_info_count (&get);
              if (uuid_was_null)
                {
                  free (current_credentials.uuid);
                  current_credentials.uuid = NULL;
                }
              if (db_count >= count)
                return 1;
              break;
            }

          if (report)
            last_report = report;
          else
            {
              last_report = 0;
              if (task_last_report (task, &last_report))
                g_warning ("%s: failed to get last report\n", __FUNCTION__);
            }

          g_debug ("%s: last_report: %llu", __FUNCTION__, last_report);
          if (last_report)
            {
              int db_count;
              get_data_t get;
              memset (&get, 0, sizeof (get_data_t));
              get.type = "result";
              get.filt_id = filter_id;
              report_counts_id (last_report, &debugs, &holes, &infos, &logs,
                                &warnings, &false_positives, &severity,
                                &get, NULL);

              db_count = debugs + holes + infos + logs + warnings
                         + false_positives;

              g_debug ("%s: count: %i vs %i", __FUNCTION__, db_count, count);
              if (db_count >= count)
                {
                  g_free (filter_id);
                  return 1;
                }
            }
          g_free (filter_id);
          break;
        }
      case ALERT_CONDITION_FILTER_COUNT_CHANGED:
        {
          char *direction, *filter_id, *count_string;
          report_t last_report;
          int debugs, holes, infos, logs, warnings, false_positives;
          int count;
          double severity;

          /* True if the number of results matched by the given filter in the
           * last finished report changed in the given direction with respect
           * to the second last finished report. */

          direction = alert_data (alert, "condition", "direction");
          filter_id = alert_data (alert, "condition", "filter_id");
          count_string = alert_data (alert, "condition", "count");
          if (count_string)
            {
              count = atoi (count_string);
              free (count_string);
            }
          else
            count = 0;

          if (report)
            last_report = report;
          else
            {
              last_report = 0;
              if (task_last_report (task, &last_report))
                g_warning ("%s: failed to get last report\n", __FUNCTION__);
            }

          if (last_report)
            {
              report_t second_last_report;
              int last_count;
              get_data_t get;
              get.type = "result";
              get.filt_id = filter_id;

              report_counts_id (last_report, &debugs, &holes, &infos, &logs,
                                &warnings, &false_positives, &severity,
                                &get, NULL);
              last_count = debugs + holes + infos + logs + warnings
                            + false_positives;

              second_last_report = 0;
              if (task_second_last_report (task, &second_last_report))
                g_warning ("%s: failed to get second last report\n", __FUNCTION__);

              if (second_last_report)
                {
                  int cmp, second_last_count;

                  report_counts_id (second_last_report, &debugs, &holes, &infos,
                                    &logs, &warnings, &false_positives,
                                    &severity, &get, NULL);
                  second_last_count = debugs + holes + infos + logs + warnings
                                      + false_positives;

                  cmp = last_count - second_last_count;
                  g_debug ("cmp: %i (vs %i)\n", cmp, count);
                  g_debug ("direction: %s\n", direction);
                  g_debug ("last_count: %i\n", last_count);
                  g_debug ("second_last_count: %i\n", second_last_count);
                  if (count < 0)
                    {
                      count = -count;
                      if (direction == NULL
                          || strcasecmp (direction, "increased") == 0)
                        {
                          free (direction);
                          direction = g_strdup ("decreased");
                        }
                      else if (strcasecmp (direction, "decreased") == 0)
                        {
                          free (direction);
                          direction = g_strdup ("increased");
                        }
                    }
                  if (direction == NULL)
                    {
                      /* Same as "increased". */
                      if (cmp >= count)
                        return 1;
                    }
                  else if (((strcasecmp (direction, "changed") == 0)
                            && (abs (cmp) >= count))
                           || ((strcasecmp (direction, "increased") == 0)
                               && (cmp >= count))
                           || ((strcasecmp (direction, "decreased") == 0)
                               && (cmp <= count)))
                    {
                      free (direction);
                      free (filter_id);
                      return 1;
                    }
                }
              else
                {
                  g_debug ("direction: %s\n", direction);
                  g_debug ("last_count: %i\n", last_count);
                  g_debug ("second_last_count NULL\n");
                  if (((strcasecmp (direction, "changed") == 0)
                       || (strcasecmp (direction, "increased") == 0))
                      && (last_count > 0))
                    {
                      free (direction);
                      free (filter_id);
                      return 1;
                    }
                }
            }
          free (direction);
          free (filter_id);
          break;
        }
      case ALERT_CONDITION_SEVERITY_AT_LEAST:
        {
          char *condition_severity_str;

          /* True if the threat level of the last finished report is at
           * least the given level. */

          condition_severity_str = alert_data (alert, "condition", "severity");

          if (condition_severity_str)
            {
              double condition_severity_dbl, task_severity_dbl;

              condition_severity_dbl = g_ascii_strtod (condition_severity_str,
                                                       0);
              task_severity_dbl = task_severity_double (task, 1,
                                                        MIN_QOD_DEFAULT, 0);

              if (task_severity_dbl >= condition_severity_dbl)
                {
                  free (condition_severity_str);
                  return 1;
                }
            }
          free (condition_severity_str);
          break;
        }
      case ALERT_CONDITION_SEVERITY_CHANGED:
        {
          char *direction;
          double last_severity, second_last_severity;

          /* True if the threat level of the last finished report changed
           * in the given direction with respect to the second last finished
           * report. */

          direction = alert_data (alert, "condition", "direction");
          last_severity = task_severity_double (task, 1,
                                                MIN_QOD_DEFAULT, 0);
          second_last_severity = task_severity_double (task, 1,
                                                       MIN_QOD_DEFAULT, 1);
          if (direction
              && last_severity > SEVERITY_MISSING
              && second_last_severity > SEVERITY_MISSING)
            {
              double cmp = last_severity - second_last_severity;
              g_debug ("cmp: %f\n", cmp);
              g_debug ("direction: %s\n", direction);
              g_debug ("last_level: %1.1f\n", last_severity);
              g_debug ("second_last_level: %1.1f\n", second_last_severity);
              if (((strcasecmp (direction, "changed") == 0) && cmp)
                  || ((strcasecmp (direction, "increased") == 0) && (cmp > 0))
                  || ((strcasecmp (direction, "decreased") == 0) && (cmp < 0)))
                {
                  free (direction);
                  return 1;
                }
            }
          else if (direction
                   && last_severity > SEVERITY_MISSING)
            {
              g_debug ("direction: %s\n", direction);
              g_debug ("last_level: %1.1f\n", last_severity);
              g_debug ("second_last_level NULL\n");
              if ((strcasecmp (direction, "changed") == 0)
                  || (strcasecmp (direction, "increased") == 0))
                {
                  free (direction);
                  return 1;
                }
            }
          free (direction);
          break;
        }
      default:
        break;
    }
  return 0;
}

/**
 * @brief Produce an event.
 *
 * @param[in]  task        Task.
 * @param[in]  report      Report.
 * @param[in]  event       Event.
 * @param[in]  event_data  Event type specific details.
 */
static void
event (task_t task, report_t report, event_t event, void* event_data)
{
  iterator_t alerts;
  GArray *alerts_triggered;
  guint index;

  g_debug ("   EVENT %i on task %llu", event, task);

  alerts_triggered = g_array_new (TRUE, TRUE, sizeof (alert_t));

  init_task_alert_iterator (&alerts, task, event);
  while (next (&alerts))
    {
      alert_t alert = task_alert_iterator_alert (&alerts);
      if (event_applies (event, event_data, task, alert))
        {
          alert_condition_t condition;

          condition = alert_condition (alert);
          if (condition_met (task, report, alert, condition))
            g_array_append_val (alerts_triggered, alert);
        }
    }
  cleanup_iterator (&alerts);

  /* Run the alerts outside the iterator, because they may take some
   * time and the iterator would prevent update processes (OMP MODIFY_XXX,
   * CREATE_XXX, ...) from locking the database. */
  index = alerts_triggered->len;
  while (index--)
    {
      alert_t alert;
      alert_condition_t condition;

      alert = g_array_index (alerts_triggered, alert_t, index);
      condition = alert_condition (alert);
      escalate_1 (alert,
                  task,
                  report,
                  event,
                  event_data,
                  alert_method (alert),
                  condition,
                  NULL);
    }

  g_array_free (alerts_triggered, TRUE);
}

/**
 * @brief Initialise an alert task iterator.
 *
 * Iterate over all tasks that use the alert.
 *
 * @param[in]  iterator   Iterator.
 * @param[in]  alert  Alert.
 * @param[in]  ascending  Whether to sort ascending or descending.
 */
void
init_alert_task_iterator (iterator_t* iterator, alert_t alert,
                              int ascending)
{
  gchar *available;
  get_data_t get;
  array_t *permissions;

  assert (alert);

  get.trash = 0;
  permissions = make_array ();
  array_add (permissions, g_strdup ("get_tasks"));
  available = acl_where_owned ("task", &get, 1, "any", 0, permissions);
  array_free (permissions);

  init_iterator (iterator,
                 "SELECT tasks.name, tasks.uuid, %s FROM tasks, task_alerts"
                 " WHERE tasks.id = task_alerts.task"
                 " AND task_alerts.alert = %llu"
                 " AND hidden = 0"
                 " ORDER BY tasks.name %s;",
                 available,
                 alert,
                 ascending ? "ASC" : "DESC");

  g_free (available);
}

/**
 * @brief Return the name from an alert task iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return Name of the task or NULL if iteration is complete.
 */
const char*
alert_task_iterator_name (iterator_t* iterator)
{
  const char *ret;
  if (iterator->done) return NULL;
  ret = iterator_string (iterator, 0);
  return ret;
}

/**
 * @brief Return the uuid from an alert task iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return UUID of the task or NULL if iteration is complete.
 */
const char*
alert_task_iterator_uuid (iterator_t* iterator)
{
  const char *ret;
  if (iterator->done) return NULL;
  ret = iterator_string (iterator, 1);
  return ret;
}

/**
 * @brief Get the read permission status from a GET iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return 1 if may read, else 0.
 */
int
alert_task_iterator_readable (iterator_t* iterator)
{
  if (iterator->done) return 0;
  return iterator_int (iterator, 2);
}


/* Task functions. */

/**
 * @brief Append value to field of task.
 *
 * @param[in]  task   Task.
 * @param[in]  field  Field.
 * @param[in]  value  Value.
 */
static void
append_to_task_string (task_t task, const char* field, const char* value)
{
  char* current;
  gchar* quote;
  current = sql_string ("SELECT %s FROM tasks WHERE id = %llu;",
                        field,
                        task);
  if (current)
    {
      gchar* new = g_strconcat ((const gchar*) current, value, NULL);
      free (current);
      quote = sql_nquote (new, strlen (new));
      g_free (new);
    }
  else
    quote = sql_nquote (value, strlen (value));
  sql ("UPDATE tasks SET %s = '%s', modification_time = m_now ()"
       " WHERE id = %llu;",
       field,
       quote,
       task);
  g_free (quote);
}

/**
 * @brief Filter columns for task iterator.
 */
#define TASK_ITERATOR_FILTER_COLUMNS                                          \
 { GET_ITERATOR_FILTER_COLUMNS, "status", "total", "first_report",            \
   "last_report", "threat", "trend", "severity", "schedule", "next_due",      \
   "first", "last", "false_positive", "log", "low", "medium", "high",         \
   "hosts", "result_hosts", "fp_per_host", "log_per_host", "low_per_host",    \
   "medium_per_host", "high_per_host", "target", NULL }

/**
 * @brief Task iterator columns.
 */
#define TASK_ITERATOR_COLUMNS_INNER                                         \
   { "run_status", NULL, KEYWORD_TYPE_INTEGER },                            \
   {                                                                        \
     "(SELECT count(*) FROM reports"                                        \
     " WHERE task = tasks.id)",                                             \
     "total",                                                               \
     KEYWORD_TYPE_INTEGER                                                   \
   },                                                                       \
   {                                                                        \
     "(SELECT uuid FROM reports WHERE task = tasks.id"                      \
     /* TODO 1 == TASK_STATUS_DONE */                                       \
     " AND scan_run_status = 1"                                             \
     " ORDER BY date ASC LIMIT 1)",                                         \
     "first_report",                                                        \
     KEYWORD_TYPE_STRING                                                    \
   },                                                                       \
   { "run_status_name (run_status)", "status", KEYWORD_TYPE_STRING },       \
   {                                                                        \
     "(SELECT uuid FROM reports WHERE task = tasks.id"                      \
     /* TODO 1 == TASK_STATUS_DONE */                                       \
     " AND scan_run_status = 1"                                             \
     " ORDER BY date DESC LIMIT 1)",                                        \
     "last_report",                                                         \
     KEYWORD_TYPE_STRING                                                    \
   },                                                                       \
   {                                                                        \
     "(SELECT count(*) FROM reports"                                        \
     /* TODO 1 == TASK_STATUS_DONE */                                       \
     " WHERE task = tasks.id AND scan_run_status = 1)",                     \
     NULL,                                                                  \
     KEYWORD_TYPE_INTEGER                                                   \
   },                                                                       \
   { "hosts_ordering", NULL, KEYWORD_TYPE_STRING },                         \
   { "scanner", NULL, KEYWORD_TYPE_INTEGER }

/**
 * @brief Task iterator WHERE columns.
 */
#define TASK_ITERATOR_WHERE_COLUMNS_INNER                                    \
   {                                                                         \
     "task_threat_level (id, opts.override, opts.min_qod)",                  \
     "threat",                                                               \
     KEYWORD_TYPE_STRING                                                     \
   },                                                                        \
   {                                                                         \
     "task_trend (id, opts.override, opts.min_qod)",                         \
     "trend",                                                                \
     KEYWORD_TYPE_STRING                                                     \
   },                                                                        \
   {                                                                         \
     "task_severity (id, opts.override, opts.min_qod)",                      \
     "severity",                                                             \
     KEYWORD_TYPE_DOUBLE                                                     \
   },                                                                        \
   {                                                                         \
     "(SELECT schedules.name FROM schedules"                                 \
     " WHERE schedules.id = tasks.schedule)",                                \
     "schedule",                                                             \
     KEYWORD_TYPE_STRING                                                     \
   },                                                                        \
   {                                                                         \
     "(CASE WHEN schedule_next_time IS NULL"                                 \
     " THEN -1"                                                              \
     " WHEN schedule_next_time = 0 AND tasks.schedule > 0"                   \
     " THEN (SELECT first_time"                                              \
     "       FROM schedules"                                                 \
     "       WHERE schedules.id = tasks.schedule)"                           \
     " ELSE schedule_next_time"                                              \
     " END)",                                                                \
     "next_due",                                                             \
     KEYWORD_TYPE_INTEGER                                                    \
   },                                                                        \
   {                                                                         \
     "(SELECT date FROM reports WHERE task = tasks.id"                       \
     /* TODO 1 == TASK_STATUS_DONE */                                        \
     " AND scan_run_status = 1"                                              \
     " ORDER BY date ASC LIMIT 1)",                                          \
     "first",                                                                \
     KEYWORD_TYPE_INTEGER                                                    \
   },                                                                        \
   {                                                                         \
     "(SELECT date FROM reports WHERE task = tasks.id"                       \
     /* TODO 1 == TASK_STATUS_DONE */                                        \
     " AND scan_run_status = 1"                                              \
     " ORDER BY date DESC LIMIT 1)",                                         \
     "last",                                                                 \
     KEYWORD_TYPE_INTEGER                                                    \
   },                                                                        \
   {                                                                         \
     "CASE WHEN target IS null OR opts.ignore_severity != 0 THEN 0 ELSE"     \
     " report_severity_count (task_last_report (id),"                        \
     "                        opts.override, opts.min_qod,"                  \
     "                        'False Positive')"                             \
     " END",                                                                 \
     "false_positive",                                                       \
     KEYWORD_TYPE_INTEGER                                                    \
   },                                                                        \
   {                                                                         \
     "CASE WHEN target IS null OR opts.ignore_severity != 0 THEN 0 ELSE"     \
     " report_severity_count (task_last_report (id),"                        \
     "                        opts.override, opts.min_qod, 'Log')"           \
     " END",                                                                 \
     "log",                                                                  \
     KEYWORD_TYPE_INTEGER                                                    \
   },                                                                        \
   {                                                                         \
     "CASE WHEN target IS null OR opts.ignore_severity != 0 THEN 0 ELSE"     \
     " report_severity_count (task_last_report (id),"                        \
     "                        opts.override, opts.min_qod, 'Low')"           \
     " END",                                                                 \
     "low",                                                                  \
     KEYWORD_TYPE_INTEGER                                                    \
   },                                                                        \
   {                                                                         \
     "CASE WHEN target IS null OR opts.ignore_severity != 0 THEN 0 ELSE"     \
     " report_severity_count (task_last_report (id),"                        \
     "                        opts.override, opts.min_qod, 'Medium')"        \
     " END",                                                                 \
     "medium",                                                               \
     KEYWORD_TYPE_INTEGER                                                    \
   },                                                                        \
   {                                                                         \
     "CASE WHEN target IS null OR opts.ignore_severity != 0 THEN 0 ELSE"     \
     " report_severity_count (task_last_report (id),"                        \
     "                        opts.override, opts.min_qod, 'High')"          \
     " END",                                                                 \
     "high",                                                                 \
     KEYWORD_TYPE_INTEGER                                                    \
   },                                                                        \
   {                                                                         \
     "CASE WHEN target IS null OR opts.ignore_severity != 0 THEN 0 ELSE"     \
     " report_host_count (task_last_report (id))"                            \
     " END",                                                                 \
     "hosts",                                                                \
     KEYWORD_TYPE_INTEGER                                                    \
   },                                                                        \
   {                                                                         \
     "CASE WHEN target IS null OR opts.ignore_severity != 0 THEN 0 ELSE"     \
     " report_result_host_count (task_last_report (id), opts.min_qod)"       \
     " END",                                                                 \
     "result_hosts",                                                         \
     KEYWORD_TYPE_INTEGER                                                    \
   },                                                                        \
   {                                                                         \
     "CASE WHEN target IS null OR opts.ignore_severity != 0 THEN 0 ELSE"     \
     " coalesce (report_severity_count (task_last_report (id),"              \
     "                                 opts.override, opts.min_qod,"         \
     "                                 'False Positive') * 1.0"              \
     "            / nullif (report_result_host_count (task_last_report (id),"\
     "                                                opts.min_qod), 0),"    \
     "          0)"                                                          \
     " END",                                                                 \
     "fp_per_host",                                                          \
     KEYWORD_TYPE_INTEGER                                                    \
   },                                                                        \
   {                                                                         \
     "CASE WHEN target IS null OR opts.ignore_severity != 0 THEN 0 ELSE"     \
     " coalesce (report_severity_count (task_last_report (id),"              \
     "                                 opts.override, opts.min_qod,"         \
     "                                 'Log') * 1.0"                         \
     "            / nullif (report_result_host_count (task_last_report (id),"\
     "                                                opts.min_qod), 0),"    \
     "          0)"                                                          \
     " END",                                                                 \
     "log_per_host",                                                         \
     KEYWORD_TYPE_INTEGER                                                    \
   },                                                                        \
   {                                                                         \
     "CASE WHEN target IS null OR opts.ignore_severity != 0 THEN 0 ELSE"     \
     " coalesce (report_severity_count (task_last_report (id),"              \
     "                                 opts.override, opts.min_qod,"         \
     "                                 'Low') * 1.0"                         \
     "            / nullif (report_result_host_count (task_last_report (id),"\
     "                                                opts.min_qod), 0),"    \
     "          0)"                                                          \
     " END",                                                                 \
     "low_per_host",                                                         \
     KEYWORD_TYPE_INTEGER                                                    \
   },                                                                        \
   {                                                                         \
     "CASE WHEN target IS null OR opts.ignore_severity != 0 THEN 0 ELSE"     \
     " coalesce (report_severity_count (task_last_report (id),"              \
     "                                 opts.override, opts.min_qod,"         \
     "                                 'Medium') * 1.0"                      \
     "            / nullif (report_result_host_count (task_last_report (id),"\
     "                                                opts.min_qod), 0),"    \
     "          0)"                                                          \
     " END",                                                                 \
     "medium_per_host",                                                      \
     KEYWORD_TYPE_INTEGER                                                    \
   },                                                                        \
   {                                                                         \
     "CASE WHEN target IS null OR opts.ignore_severity != 0 THEN 0 ELSE"     \
     " coalesce (report_severity_count (task_last_report (id),"              \
     "                                 opts.override, opts.min_qod,"         \
     "                                 'High') * 1.0"                        \
     "            / nullif (report_result_host_count (task_last_report (id),"\
     "                                                opts.min_qod), 0),"    \
     "          0)"                                                          \
     " END",                                                                 \
     "high_per_host",                                                        \
     KEYWORD_TYPE_INTEGER                                                    \
   },                                                                        \
   {                                                                         \
     "(SELECT name FROM targets WHERE id = target)",                         \
     "target",                                                               \
     KEYWORD_TYPE_STRING                                                     \
   }

/**
 * @brief Task iterator WHERE columns.
 */
#define TASK_ITERATOR_WHERE_COLUMNS                                         \
 {                                                                          \
   TASK_ITERATOR_WHERE_COLUMNS_INNER,                                       \
   { NULL, NULL, KEYWORD_TYPE_UNKNOWN }                                     \
 }

/**
 * @brief Task iterator columns.
 */
#define TASK_ITERATOR_COLUMNS                                               \
 {                                                                          \
   GET_ITERATOR_COLUMNS (tasks),                                            \
   TASK_ITERATOR_COLUMNS_INNER,                                             \
   { NULL, NULL, KEYWORD_TYPE_UNKNOWN }                                     \
 }

/**
 * @brief Task iterator minimal columns.
 */
#define TASK_ITERATOR_COLUMNS_MIN                                           \
 {                                                                          \
   GET_ITERATOR_COLUMNS (tasks),                                            \
   { NULL, NULL, KEYWORD_TYPE_UNKNOWN }                                     \
 }

/**
 * @brief Task iterator minimal WHERE columns.
 */
#define TASK_ITERATOR_WHERE_COLUMNS_MIN                                     \
 {                                                                          \
   TASK_ITERATOR_COLUMNS_INNER,                                             \
   TASK_ITERATOR_WHERE_COLUMNS_INNER,                                       \
   { NULL, NULL, KEYWORD_TYPE_UNKNOWN }                                     \
 }

/**
 * @brief Generate the extra_tables string for a task iterator.
 *
 * @param[in]  override  Whether to apply overrides.
 * @param[in]  min_qod   Minimum QoD of results to count.
 * @param[in]  ignore_severity  Whether to ignore severity data.
 * @return Newly allocated string with the extra_tables clause.
 */
static gchar*
task_iterator_opts_table (int override, int min_qod, int ignore_severity)
{
  return g_strdup_printf (", (SELECT"
                          "   %d AS override,"
                          "   %d AS min_qod,"
                          "   %d AS ignore_severity)"
                          "  AS opts",
                          override,
                          min_qod,
                          ignore_severity);
}

/**
 * @brief Initialise a task iterator, limited to current user's tasks.
 *
 * @param[in]  iterator    Task iterator.
 * @param[in]  trash       Whether to iterate over trashcan tasks.
 * @param[in]  ignore_severity  Whether to ignore severity data.
 */
static void
init_user_task_iterator (iterator_t* iterator, int trash, int ignore_severity)
{
  static column_t select_columns[] = TASK_ITERATOR_COLUMNS;
  gchar *extra_tables;
  gchar *columns;

  extra_tables = task_iterator_opts_table (0, MIN_QOD_DEFAULT,
                                           ignore_severity);

  columns = columns_build_select (select_columns);

  init_iterator (iterator,
                 "SELECT %s"
                 " FROM tasks%s"
                 " WHERE " ACL_USER_OWNS ()
                 "%s;",
                 columns,
                 extra_tables,
                 current_credentials.uuid,
                 trash ? " AND hidden = 2" : " AND hidden < 2");

  g_free (extra_tables);
  g_free (columns);
}

/**
 * @brief Initialise a task iterator.
 *
 * @param[in]  iterator    Task iterator.
 * @param[in]  get         GET data.
 *
 * @return 0 success, 1 failed to find target, 2 failed to find filter,
 *         -1 error.
 */
int
init_task_iterator (iterator_t* iterator, const get_data_t *get)
{
  static const char *filter_columns[] = TASK_ITERATOR_FILTER_COLUMNS;
  static column_t columns[] = TASK_ITERATOR_COLUMNS;
  static column_t where_columns[] = TASK_ITERATOR_WHERE_COLUMNS;
  static column_t columns_min[] = TASK_ITERATOR_COLUMNS_MIN;
  static column_t where_columns_min[] = TASK_ITERATOR_WHERE_COLUMNS_MIN;
  char *filter;
  gchar *value;
  int overrides, min_qod;
  gchar *extra_tables;
  int ret;

  if (get->filt_id && strcmp (get->filt_id, "0"))
    {
      filter = filter_term (get->filt_id);
      if (filter == NULL)
        return 2;
    }
  else
    filter = NULL;
  value = filter_term_value (filter ? filter : get->filter, "apply_overrides");
  overrides = value && strcmp (value, "0");
  g_free (value);

  value = filter_term_value (filter ? filter : get->filter, "min_qod");
  if (value == NULL || sscanf (value, "%d", &min_qod) != 1)
    min_qod = MIN_QOD_DEFAULT;
  g_free (value);
  free (filter);

  extra_tables = task_iterator_opts_table (overrides, min_qod, 0);

  ret = init_get_iterator2 (iterator,
                            "task",
                            get,
                            /* SELECT columns. */
                            get->minimal ? columns_min : columns,
                            get->minimal ? columns_min : columns,
                            /* Filterable columns not in SELECT columns. */
                            get->minimal ? where_columns_min : where_columns,
                            get->minimal ? where_columns_min : where_columns,
                            filter_columns,
                            0,
                            extra_tables,
                            get->trash
                             ? " AND hidden = 2"
                             : " AND hidden = 0",
                            current_credentials.uuid ? TRUE : FALSE,
                            FALSE,
                            NULL);

  g_free (extra_tables);
  return ret;
}

/**
 * @brief Get the run status from a task iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return Task run status.
 */
task_status_t
task_iterator_run_status (iterator_t* iterator)
{
  task_status_t ret;
  if (iterator->done) return TASK_STATUS_INTERNAL_ERROR;
  ret = (unsigned int) iterator_int (iterator, GET_ITERATOR_COLUMN_COUNT);
  return ret;
}

/**
 * @brief Get the number of reports of a task iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return Count of all task reports.
 */
int
task_iterator_total_reports (iterator_t* iterator)
{
  if (iterator->done) return 0;
  return iterator_int (iterator, GET_ITERATOR_COLUMN_COUNT + 1);
}

/**
 * @brief Get the first report UUID from a task iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return First report UUID.
 */
const char *
task_iterator_first_report (iterator_t* iterator)
{
  if (iterator->done) return 0;
  return iterator_string (iterator, GET_ITERATOR_COLUMN_COUNT + 2);
}

/**
 * @brief Get the run status name from a task iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return Task run status name.
 */
const char *
task_iterator_run_status_name (iterator_t* iterator)
{
  if (iterator->done) return 0;
  return iterator_string (iterator, GET_ITERATOR_COLUMN_COUNT + 3);
}

/**
 * @brief Get the last report UUID from a task iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return Last report UUID.
 */
const char *
task_iterator_last_report (iterator_t* iterator)
{
  if (iterator->done) return 0;
  return iterator_string (iterator, GET_ITERATOR_COLUMN_COUNT + 4);
}

/**
 * @brief Get the number of reports of a task iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return Count of all task reports.
 */
int
task_iterator_finished_reports (iterator_t *iterator)
{
  if (iterator->done) return 0;
  return iterator_int (iterator, GET_ITERATOR_COLUMN_COUNT + 5);
}

/**
 * @brief Get the hosts ordering value from a task iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return Task hosts ordering.
 */
const char *
task_iterator_hosts_ordering (iterator_t* iterator)
{
  if (iterator->done) return 0;
  return iterator_string (iterator, GET_ITERATOR_COLUMN_COUNT + 6);
}

/**
 * @brief Get the UUID of task scanner from a task iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return Task scanner if found, NULL otherwise.
 */
scanner_t
task_iterator_scanner (iterator_t* iterator)
{
  if (iterator->done) return 0;
  return iterator_int64 (iterator, GET_ITERATOR_COLUMN_COUNT + 7);
}

/**
 * @brief Return whether a task is in use by a task.
 *
 * @param[in]  task  Task.
 *
 * @return 0.
 */
int
task_in_use (task_t task)
{
  task_status_t status;
  status = task_run_status (task);
  return status == TASK_STATUS_DELETE_REQUESTED
         || status == TASK_STATUS_DELETE_WAITING
         || status == TASK_STATUS_DELETE_ULTIMATE_REQUESTED
         || status == TASK_STATUS_DELETE_ULTIMATE_WAITING
         || status == TASK_STATUS_REQUESTED
         || status == TASK_STATUS_RUNNING
         || status == TASK_STATUS_STOP_REQUESTED_GIVEUP
         || status == TASK_STATUS_STOP_REQUESTED
         || status == TASK_STATUS_STOP_WAITING;
}

/**
 * @brief Return whether a trashcan task is referenced by a task.
 *
 * @param[in]  task  Task.
 *
 * @return 0.
 */
int
trash_task_in_use (task_t task)
{
  return task_in_use (task);
}

/**
 * @brief Return whether a task is an Alterable Task.
 *
 * @param[in]  task  Task.
 *
 * @return 1 if Alterable, else 0.
 */
int
task_alterable (task_t task)
{
  return sql_int ("SELECT alterable FROM tasks"
                  " WHERE id = %llu",
                  task);
}

/**
 * @brief Return whether a task is writable.
 *
 * @param[in]  task  Task.
 *
 * @return 1 if writable, else 0.
 */
int
task_writable (task_t task)
{
  return sql_int ("SELECT hidden = 0 FROM tasks"
                  " WHERE id = %llu",
                  task);
}

/**
 * @brief Return whether a trashcan task is writable.
 *
 * @param[in]  task  Task.
 *
 * @return 1 if writable, else 0.
 */
int
trash_task_writable (task_t task)
{
  return sql_int ("SELECT hidden = 2 FROM tasks"
                  " WHERE id = %llu",
                  task);
}

/**
 * @brief Get the average duration of all finished reports of a task.
 *
 * @param[in]  task  Task.
 *
 * @return Average scan duration in seconds.
 */
int
task_average_scan_duration (task_t task)
{
  return sql_int ("SELECT avg (end_time - start_time) FROM reports"
                  " WHERE task = %llu"
                  "   AND scan_run_status = %d"
                  "   AND coalesce (end_time, 0) != 0"
                  "   AND coalesce (start_time, 0) != 0;",
                  task, TASK_STATUS_DONE);
}

/**
 * @brief Initialize the manage library for a process.
 *
 * Open the SQL database, attach secondary databases, and define functions.
 *
 * @param[in]  update_nvt_cache  0 operate normally, -1 just update NVT cache,
 *                               -2 just rebuild NVT cache.
 * @param[in]  database          Location of manage database.
 */
void
init_manage_process (int update_nvt_cache, const gchar *database)
{
  if (sql_is_open ())
    return;

  /* Open the database. */
  if (sql_open (database))
    {
      g_warning ("%s: sql_open failed\n", __FUNCTION__);
      abort ();
    }

  /* Attach the SCAP and CERT databases. */
  manage_attach_databases ();

  if (update_nvt_cache)
    {
      sql ("CREATE TEMPORARY TABLE old_nvts"
           " (oid TEXT, modification_time INTEGER);");
      sql ("INSERT INTO old_nvts (oid, modification_time)"
           " SELECT oid, modification_time FROM nvts;");
    }

  /* Define functions for SQL. */

  /* Lock to avoid an error return from Postgres when multiple processes
   * create a function at the same time. */
  sql_begin_exclusive ();
  if (manage_create_sql_functions ())
    {
      sql_rollback ();
      g_warning ("%s: failed to create functions", __FUNCTION__);
      abort ();
    }
  sql_commit ();
}

/**
 * @brief Reinitialize the manage library for a process.
 *
 * This is mandatory after a fork, to not carry open databases around (refer
 * to database documentation).
 */
void
reinit_manage_process ()
{
  cleanup_manage_process (FALSE);
  init_manage_process (0, task_db_name);
}

/**
 * @brief Setup config preferences for a config.
 *
 * @param[in]  config         The config.
 * @param[in]  safe_checks    safe_checks option: 1 for "yes", 0 for "no".
 * @param[in]  optimize_test  optimize_test option: 1 for "yes", 0 for "no".
 * @param[in]  port_range     port_range option: 1 for "yes", 0 for "no".
 */
static void
setup_full_config_prefs (config_t config, int safe_checks,
                         int optimize_test, int port_range)
{
  sql ("INSERT into config_preferences (config, type, name, value)"
       " VALUES (%i, 'SERVER_PREFS', 'max_hosts', '20');",
       config);
  sql ("INSERT into config_preferences (config, type, name, value)"
       " VALUES (%i, 'SERVER_PREFS', 'max_checks', '4');",
       config);
  sql ("INSERT into config_preferences (config, type, name, value)"
       " VALUES (%i, 'SERVER_PREFS', 'cgi_path', '/cgi-bin:/scripts');",
       config);
  if (port_range)
    sql ("INSERT into config_preferences (config, type, name, value)"
         " VALUES (%i, 'SERVER_PREFS', 'port_range', '1-65535');",
         config);
  else
    sql ("INSERT into config_preferences (config, type, name, value)"
         " VALUES (%i, 'SERVER_PREFS', 'port_range', 'default');",
         config);
  sql ("INSERT into config_preferences (config, type, name, value)"
       " VALUES (%i, 'SERVER_PREFS', 'auto_enable_dependencies', 'yes');",
       config);
  if (optimize_test)
    sql ("INSERT into config_preferences (config, type, name, value)"
         " VALUES (%i, 'SERVER_PREFS', 'optimize_test', 'yes');",
         config);
  else
    sql ("INSERT into config_preferences (config, type, name, value)"
         " VALUES (%i, 'SERVER_PREFS', 'optimize_test', 'no');",
         config);
  if (safe_checks)
    sql ("INSERT into config_preferences (config, type, name, value)"
         " VALUES (%i, 'SERVER_PREFS', 'safe_checks', 'yes');",
         config);
  else
    sql ("INSERT into config_preferences (config, type, name, value)"
         " VALUES (%i, 'SERVER_PREFS', 'safe_checks', 'no');",
         config);
  sql ("INSERT into config_preferences (config, type, name, value)"
       " VALUES (%i, 'SERVER_PREFS', 'use_mac_addr', 'no');",
       config);
  sql ("INSERT into config_preferences (config, type, name, value)"
       " VALUES (%i, 'SERVER_PREFS', 'unscanned_closed', 'yes');",
       config);
  sql ("INSERT into config_preferences (config, type, name, value)"
       " VALUES (%i, 'SERVER_PREFS', 'unscanned_closed_udp', 'yes');",
       config);
  sql ("INSERT into config_preferences (config, type, name, value)"
       " VALUES (%i, 'SERVER_PREFS', 'checks_read_timeout', '5');",
       config);
  sql ("INSERT into config_preferences (config, type, name, value)"
       " VALUES (%i, 'SERVER_PREFS', 'network_scan', 'no');",
       config);
  sql ("INSERT into config_preferences (config, type, name, value)"
       " VALUES (%i, 'SERVER_PREFS', 'non_simult_ports', '139, 445');",
       config);
  sql ("INSERT into config_preferences (config, type, name, value)"
       " VALUES (%i, 'SERVER_PREFS', 'plugins_timeout', '320');",
       config);
  sql ("INSERT into config_preferences (config, type, name, value)"
       " VALUES (%i, 'SERVER_PREFS', 'nasl_no_signature_check', 'yes');",
       config);

  sql ("INSERT into config_preferences (config, type, name, value)"
       " VALUES (%i, 'PLUGINS_PREFS',"
       " 'Ping Host[checkbox]:Mark unrechable Hosts as dead (not scanning)',"
       " 'yes');",
       config);
  sql ("INSERT into config_preferences (config, type, name, value)"
       " VALUES (%i, 'PLUGINS_PREFS',"
       " 'Login configurations[checkbox]:NTLMSSP',"
       " 'yes');",
       config);
}

/**
 * @brief Update the memory cache of NVTs.
 */
static void
update_nvti_cache ()
{
  iterator_t nvts;

  nvtis_free (nvti_cache);

  nvti_cache = nvtis_new ();

  init_iterator (&nvts,
                 "SELECT oid, name, family, cvss_base, cve, bid, xref, tag"
                 " FROM nvts;");
  while (next (&nvts))
    {
      nvti_t *nvti = nvti_new ();
      nvti_set_oid (nvti, iterator_string (&nvts, 0));
      nvti_set_name (nvti, iterator_string (&nvts, 1));
      nvti_set_family (nvti, iterator_string (&nvts, 2));
      nvti_set_cvss_base (nvti, iterator_string (&nvts, 3));
      nvti_set_cve (nvti, iterator_string (&nvts, 4));
      nvti_set_bid (nvti, iterator_string (&nvts, 5));
      nvti_set_xref (nvti, iterator_string (&nvts, 6));
      nvti_set_tag (nvti, iterator_string (&nvts, 7));
      nvtis_add (nvti_cache, nvti);
    }
  cleanup_iterator (&nvts);
}

/**
 * @brief Update the memory cache of NVTs, if this has been requested.
 *
 * @return 0 success, 1 failed to get lock, -1 error.
 */
int
manage_update_nvti_cache ()
{
  int ret;

  ret = sql_begin_immediate_giveup ();
  if (ret)
    return ret;
  if (sql_int ("SELECT value FROM %s.meta"
               " WHERE name = 'update_nvti_cache';",
               sql_schema ()))
    {
      update_nvti_cache ();
      sql ("UPDATE %s.meta SET value = 0 WHERE name = 'update_nvti_cache';",
           sql_schema ());
    }
  sql_commit ();
  return 0;
}

/**
 * @brief Insert a port range.
 */
#define RANGE(type, start, end)                                      \
  sql ("INSERT INTO port_ranges"                                     \
       " (uuid, port_list, type, start, \"end\", comment, exclude)"  \
       " VALUES"                                                     \
       " (make_uuid (), %llu, %i,"                                   \
       "  '" G_STRINGIFY (start) "',"                                \
       "  '" G_STRINGIFY (end) "',"                                  \
       "  '', 0)",                                                   \
       list,                                                         \
       type)

/**
 * @brief Make port ranges.
 *
 * Caller must lock the db.
 *
 * @param[in]  list  Port list.
 */
static void
make_port_ranges_openvas_default (port_list_t list)
{
  RANGE (PORT_PROTOCOL_TCP, 1, 5);
  RANGE (PORT_PROTOCOL_TCP, 7, 7);
  RANGE (PORT_PROTOCOL_TCP, 9, 9);
  RANGE (PORT_PROTOCOL_TCP, 11, 11);
  RANGE (PORT_PROTOCOL_TCP, 13, 13);
  RANGE (PORT_PROTOCOL_TCP, 15, 15);
  RANGE (PORT_PROTOCOL_TCP, 17, 25);
  RANGE (PORT_PROTOCOL_TCP, 27, 27);
  RANGE (PORT_PROTOCOL_TCP, 29, 29);
  RANGE (PORT_PROTOCOL_TCP, 31, 31);
  RANGE (PORT_PROTOCOL_TCP, 33, 33);
  RANGE (PORT_PROTOCOL_TCP, 35, 35);
  RANGE (PORT_PROTOCOL_TCP, 37, 39);
  RANGE (PORT_PROTOCOL_TCP, 41, 59);
  RANGE (PORT_PROTOCOL_TCP, 61, 224);
  RANGE (PORT_PROTOCOL_TCP, 242, 248);
  RANGE (PORT_PROTOCOL_TCP, 256, 268);
  RANGE (PORT_PROTOCOL_TCP, 280, 287);
  RANGE (PORT_PROTOCOL_TCP, 308, 322);
  RANGE (PORT_PROTOCOL_TCP, 333, 333);
  RANGE (PORT_PROTOCOL_TCP, 344, 700);
  RANGE (PORT_PROTOCOL_TCP, 702, 702);
  RANGE (PORT_PROTOCOL_TCP, 704, 707);
  RANGE (PORT_PROTOCOL_TCP, 709, 711);
  RANGE (PORT_PROTOCOL_TCP, 721, 721);
  RANGE (PORT_PROTOCOL_TCP, 723, 723);
  RANGE (PORT_PROTOCOL_TCP, 729, 731);
  RANGE (PORT_PROTOCOL_TCP, 740, 742);
  RANGE (PORT_PROTOCOL_TCP, 744, 744);
  RANGE (PORT_PROTOCOL_TCP, 747, 754);
  RANGE (PORT_PROTOCOL_TCP, 758, 765);
  RANGE (PORT_PROTOCOL_TCP, 767, 767);
  RANGE (PORT_PROTOCOL_TCP, 769, 777);
  RANGE (PORT_PROTOCOL_TCP, 780, 783);
  RANGE (PORT_PROTOCOL_TCP, 786, 787);
  RANGE (PORT_PROTOCOL_TCP, 799, 801);
  RANGE (PORT_PROTOCOL_TCP, 808, 808);
  RANGE (PORT_PROTOCOL_TCP, 810, 810);
  RANGE (PORT_PROTOCOL_TCP, 828, 829);
  RANGE (PORT_PROTOCOL_TCP, 847, 848);
  RANGE (PORT_PROTOCOL_TCP, 860, 860);
  RANGE (PORT_PROTOCOL_TCP, 871, 871);
  RANGE (PORT_PROTOCOL_TCP, 873, 873);
  RANGE (PORT_PROTOCOL_TCP, 886, 888);
  RANGE (PORT_PROTOCOL_TCP, 898, 898);
  RANGE (PORT_PROTOCOL_TCP, 900, 904);
  RANGE (PORT_PROTOCOL_TCP, 911, 913);
  RANGE (PORT_PROTOCOL_TCP, 927, 927);
  RANGE (PORT_PROTOCOL_TCP, 950, 950);
  RANGE (PORT_PROTOCOL_TCP, 953, 953);
  RANGE (PORT_PROTOCOL_TCP, 975, 975);
  RANGE (PORT_PROTOCOL_TCP, 989, 1002);
  RANGE (PORT_PROTOCOL_TCP, 1005, 1005);
  RANGE (PORT_PROTOCOL_TCP, 1008, 1008);
  RANGE (PORT_PROTOCOL_TCP, 1010, 1010);
  RANGE (PORT_PROTOCOL_TCP, 1023, 1027);
  RANGE (PORT_PROTOCOL_TCP, 1029, 1036);
  RANGE (PORT_PROTOCOL_TCP, 1040, 1040);
  RANGE (PORT_PROTOCOL_TCP, 1042, 1042);
  RANGE (PORT_PROTOCOL_TCP, 1045, 1045);
  RANGE (PORT_PROTOCOL_TCP, 1047, 1112);
  RANGE (PORT_PROTOCOL_TCP, 1114, 1117);
  RANGE (PORT_PROTOCOL_TCP, 1119, 1120);
  RANGE (PORT_PROTOCOL_TCP, 1122, 1127);
  RANGE (PORT_PROTOCOL_TCP, 1139, 1139);
  RANGE (PORT_PROTOCOL_TCP, 1154, 1155);
  RANGE (PORT_PROTOCOL_TCP, 1161, 1162);
  RANGE (PORT_PROTOCOL_TCP, 1168, 1170);
  RANGE (PORT_PROTOCOL_TCP, 1178, 1178);
  RANGE (PORT_PROTOCOL_TCP, 1180, 1181);
  RANGE (PORT_PROTOCOL_TCP, 1183, 1188);
  RANGE (PORT_PROTOCOL_TCP, 1194, 1194);
  RANGE (PORT_PROTOCOL_TCP, 1199, 1231);
  RANGE (PORT_PROTOCOL_TCP, 1233, 1286);
  RANGE (PORT_PROTOCOL_TCP, 1288, 1774);
  RANGE (PORT_PROTOCOL_TCP, 1776, 2028);
  RANGE (PORT_PROTOCOL_TCP, 2030, 2030);
  RANGE (PORT_PROTOCOL_TCP, 2032, 2035);
  RANGE (PORT_PROTOCOL_TCP, 2037, 2038);
  RANGE (PORT_PROTOCOL_TCP, 2040, 2065);
  RANGE (PORT_PROTOCOL_TCP, 2067, 2083);
  RANGE (PORT_PROTOCOL_TCP, 2086, 2087);
  RANGE (PORT_PROTOCOL_TCP, 2089, 2152);
  RANGE (PORT_PROTOCOL_TCP, 2155, 2155);
  RANGE (PORT_PROTOCOL_TCP, 2159, 2167);
  RANGE (PORT_PROTOCOL_TCP, 2170, 2177);
  RANGE (PORT_PROTOCOL_TCP, 2180, 2181);
  RANGE (PORT_PROTOCOL_TCP, 2190, 2191);
  RANGE (PORT_PROTOCOL_TCP, 2199, 2202);
  RANGE (PORT_PROTOCOL_TCP, 2213, 2213);
  RANGE (PORT_PROTOCOL_TCP, 2220, 2223);
  RANGE (PORT_PROTOCOL_TCP, 2232, 2246);
  RANGE (PORT_PROTOCOL_TCP, 2248, 2255);
  RANGE (PORT_PROTOCOL_TCP, 2260, 2260);
  RANGE (PORT_PROTOCOL_TCP, 2273, 2273);
  RANGE (PORT_PROTOCOL_TCP, 2279, 2289);
  RANGE (PORT_PROTOCOL_TCP, 2294, 2311);
  RANGE (PORT_PROTOCOL_TCP, 2313, 2371);
  RANGE (PORT_PROTOCOL_TCP, 2381, 2425);
  RANGE (PORT_PROTOCOL_TCP, 2427, 2681);
  RANGE (PORT_PROTOCOL_TCP, 2683, 2824);
  RANGE (PORT_PROTOCOL_TCP, 2826, 2854);
  RANGE (PORT_PROTOCOL_TCP, 2856, 2924);
  RANGE (PORT_PROTOCOL_TCP, 2926, 3096);
  RANGE (PORT_PROTOCOL_TCP, 3098, 3299);
  RANGE (PORT_PROTOCOL_TCP, 3302, 3321);
  RANGE (PORT_PROTOCOL_TCP, 3326, 3366);
  RANGE (PORT_PROTOCOL_TCP, 3372, 3403);
  RANGE (PORT_PROTOCOL_TCP, 3405, 3545);
  RANGE (PORT_PROTOCOL_TCP, 3547, 3707);
  RANGE (PORT_PROTOCOL_TCP, 3709, 3765);
  RANGE (PORT_PROTOCOL_TCP, 3767, 3770);
  RANGE (PORT_PROTOCOL_TCP, 3772, 3800);
  RANGE (PORT_PROTOCOL_TCP, 3802, 3802);
  RANGE (PORT_PROTOCOL_TCP, 3845, 3871);
  RANGE (PORT_PROTOCOL_TCP, 3875, 3876);
  RANGE (PORT_PROTOCOL_TCP, 3885, 3885);
  RANGE (PORT_PROTOCOL_TCP, 3900, 3900);
  RANGE (PORT_PROTOCOL_TCP, 3928, 3929);
  RANGE (PORT_PROTOCOL_TCP, 3939, 3939);
  RANGE (PORT_PROTOCOL_TCP, 3959, 3959);
  RANGE (PORT_PROTOCOL_TCP, 3970, 3971);
  RANGE (PORT_PROTOCOL_TCP, 3984, 3987);
  RANGE (PORT_PROTOCOL_TCP, 3999, 4036);
  RANGE (PORT_PROTOCOL_TCP, 4040, 4042);
  RANGE (PORT_PROTOCOL_TCP, 4045, 4045);
  RANGE (PORT_PROTOCOL_TCP, 4080, 4080);
  RANGE (PORT_PROTOCOL_TCP, 4096, 4100);
  RANGE (PORT_PROTOCOL_TCP, 4111, 4111);
  RANGE (PORT_PROTOCOL_TCP, 4114, 4114);
  RANGE (PORT_PROTOCOL_TCP, 4132, 4134);
  RANGE (PORT_PROTOCOL_TCP, 4138, 4138);
  RANGE (PORT_PROTOCOL_TCP, 4141, 4145);
  RANGE (PORT_PROTOCOL_TCP, 4154, 4154);
  RANGE (PORT_PROTOCOL_TCP, 4160, 4160);
  RANGE (PORT_PROTOCOL_TCP, 4199, 4200);
  RANGE (PORT_PROTOCOL_TCP, 4242, 4242);
  RANGE (PORT_PROTOCOL_TCP, 4300, 4300);
  RANGE (PORT_PROTOCOL_TCP, 4321, 4321);
  RANGE (PORT_PROTOCOL_TCP, 4333, 4333);
  RANGE (PORT_PROTOCOL_TCP, 4343, 4351);
  RANGE (PORT_PROTOCOL_TCP, 4353, 4358);
  RANGE (PORT_PROTOCOL_TCP, 4369, 4369);
  RANGE (PORT_PROTOCOL_TCP, 4400, 4400);
  RANGE (PORT_PROTOCOL_TCP, 4442, 4457);
  RANGE (PORT_PROTOCOL_TCP, 4480, 4480);
  RANGE (PORT_PROTOCOL_TCP, 4500, 4500);
  RANGE (PORT_PROTOCOL_TCP, 4545, 4547);
  RANGE (PORT_PROTOCOL_TCP, 4555, 4555);
  RANGE (PORT_PROTOCOL_TCP, 4557, 4557);
  RANGE (PORT_PROTOCOL_TCP, 4559, 4559);
  RANGE (PORT_PROTOCOL_TCP, 4567, 4568);
  RANGE (PORT_PROTOCOL_TCP, 4600, 4601);
  RANGE (PORT_PROTOCOL_TCP, 4658, 4662);
  RANGE (PORT_PROTOCOL_TCP, 4672, 4672);
  RANGE (PORT_PROTOCOL_TCP, 4752, 4752);
  RANGE (PORT_PROTOCOL_TCP, 4800, 4802);
  RANGE (PORT_PROTOCOL_TCP, 4827, 4827);
  RANGE (PORT_PROTOCOL_TCP, 4837, 4839);
  RANGE (PORT_PROTOCOL_TCP, 4848, 4849);
  RANGE (PORT_PROTOCOL_TCP, 4868, 4869);
  RANGE (PORT_PROTOCOL_TCP, 4885, 4885);
  RANGE (PORT_PROTOCOL_TCP, 4894, 4894);
  RANGE (PORT_PROTOCOL_TCP, 4899, 4899);
  RANGE (PORT_PROTOCOL_TCP, 4950, 4950);
  RANGE (PORT_PROTOCOL_TCP, 4983, 4983);
  RANGE (PORT_PROTOCOL_TCP, 4987, 4989);
  RANGE (PORT_PROTOCOL_TCP, 4998, 4998);
  RANGE (PORT_PROTOCOL_TCP, 5000, 5011);
  RANGE (PORT_PROTOCOL_TCP, 5020, 5025);
  RANGE (PORT_PROTOCOL_TCP, 5031, 5031);
  RANGE (PORT_PROTOCOL_TCP, 5042, 5042);
  RANGE (PORT_PROTOCOL_TCP, 5050, 5057);
  RANGE (PORT_PROTOCOL_TCP, 5060, 5061);
  RANGE (PORT_PROTOCOL_TCP, 5064, 5066);
  RANGE (PORT_PROTOCOL_TCP, 5069, 5069);
  RANGE (PORT_PROTOCOL_TCP, 5071, 5071);
  RANGE (PORT_PROTOCOL_TCP, 5081, 5081);
  RANGE (PORT_PROTOCOL_TCP, 5093, 5093);
  RANGE (PORT_PROTOCOL_TCP, 5099, 5102);
  RANGE (PORT_PROTOCOL_TCP, 5137, 5137);
  RANGE (PORT_PROTOCOL_TCP, 5145, 5145);
  RANGE (PORT_PROTOCOL_TCP, 5150, 5152);
  RANGE (PORT_PROTOCOL_TCP, 5154, 5154);
  RANGE (PORT_PROTOCOL_TCP, 5165, 5165);
  RANGE (PORT_PROTOCOL_TCP, 5190, 5193);
  RANGE (PORT_PROTOCOL_TCP, 5200, 5203);
  RANGE (PORT_PROTOCOL_TCP, 5222, 5222);
  RANGE (PORT_PROTOCOL_TCP, 5225, 5226);
  RANGE (PORT_PROTOCOL_TCP, 5232, 5232);
  RANGE (PORT_PROTOCOL_TCP, 5236, 5236);
  RANGE (PORT_PROTOCOL_TCP, 5250, 5251);
  RANGE (PORT_PROTOCOL_TCP, 5264, 5265);
  RANGE (PORT_PROTOCOL_TCP, 5269, 5269);
  RANGE (PORT_PROTOCOL_TCP, 5272, 5272);
  RANGE (PORT_PROTOCOL_TCP, 5282, 5282);
  RANGE (PORT_PROTOCOL_TCP, 5300, 5311);
  RANGE (PORT_PROTOCOL_TCP, 5314, 5315);
  RANGE (PORT_PROTOCOL_TCP, 5351, 5355);
  RANGE (PORT_PROTOCOL_TCP, 5400, 5432);
  RANGE (PORT_PROTOCOL_TCP, 5435, 5435);
  RANGE (PORT_PROTOCOL_TCP, 5454, 5456);
  RANGE (PORT_PROTOCOL_TCP, 5461, 5463);
  RANGE (PORT_PROTOCOL_TCP, 5465, 5465);
  RANGE (PORT_PROTOCOL_TCP, 5500, 5504);
  RANGE (PORT_PROTOCOL_TCP, 5510, 5510);
  RANGE (PORT_PROTOCOL_TCP, 5520, 5521);
  RANGE (PORT_PROTOCOL_TCP, 5530, 5530);
  RANGE (PORT_PROTOCOL_TCP, 5540, 5540);
  RANGE (PORT_PROTOCOL_TCP, 5550, 5550);
  RANGE (PORT_PROTOCOL_TCP, 5553, 5556);
  RANGE (PORT_PROTOCOL_TCP, 5566, 5566);
  RANGE (PORT_PROTOCOL_TCP, 5569, 5569);
  RANGE (PORT_PROTOCOL_TCP, 5595, 5605);
  RANGE (PORT_PROTOCOL_TCP, 5631, 5632);
  RANGE (PORT_PROTOCOL_TCP, 5666, 5666);
  RANGE (PORT_PROTOCOL_TCP, 5673, 5680);
  RANGE (PORT_PROTOCOL_TCP, 5688, 5688);
  RANGE (PORT_PROTOCOL_TCP, 5690, 5690);
  RANGE (PORT_PROTOCOL_TCP, 5713, 5717);
  RANGE (PORT_PROTOCOL_TCP, 5720, 5720);
  RANGE (PORT_PROTOCOL_TCP, 5729, 5730);
  RANGE (PORT_PROTOCOL_TCP, 5741, 5742);
  RANGE (PORT_PROTOCOL_TCP, 5745, 5746);
  RANGE (PORT_PROTOCOL_TCP, 5755, 5755);
  RANGE (PORT_PROTOCOL_TCP, 5757, 5757);
  RANGE (PORT_PROTOCOL_TCP, 5766, 5768);
  RANGE (PORT_PROTOCOL_TCP, 5771, 5771);
  RANGE (PORT_PROTOCOL_TCP, 5800, 5803);
  RANGE (PORT_PROTOCOL_TCP, 5813, 5813);
  RANGE (PORT_PROTOCOL_TCP, 5858, 5859);
  RANGE (PORT_PROTOCOL_TCP, 5882, 5882);
  RANGE (PORT_PROTOCOL_TCP, 5888, 5889);
  RANGE (PORT_PROTOCOL_TCP, 5900, 5903);
  RANGE (PORT_PROTOCOL_TCP, 5968, 5969);
  RANGE (PORT_PROTOCOL_TCP, 5977, 5979);
  RANGE (PORT_PROTOCOL_TCP, 5987, 5991);
  RANGE (PORT_PROTOCOL_TCP, 5997, 6010);
  RANGE (PORT_PROTOCOL_TCP, 6050, 6051);
  RANGE (PORT_PROTOCOL_TCP, 6064, 6073);
  RANGE (PORT_PROTOCOL_TCP, 6085, 6085);
  RANGE (PORT_PROTOCOL_TCP, 6100, 6112);
  RANGE (PORT_PROTOCOL_TCP, 6123, 6123);
  RANGE (PORT_PROTOCOL_TCP, 6141, 6150);
  RANGE (PORT_PROTOCOL_TCP, 6175, 6177);
  RANGE (PORT_PROTOCOL_TCP, 6200, 6200);
  RANGE (PORT_PROTOCOL_TCP, 6253, 6253);
  RANGE (PORT_PROTOCOL_TCP, 6255, 6255);
  RANGE (PORT_PROTOCOL_TCP, 6270, 6270);
  RANGE (PORT_PROTOCOL_TCP, 6300, 6300);
  RANGE (PORT_PROTOCOL_TCP, 6321, 6322);
  RANGE (PORT_PROTOCOL_TCP, 6343, 6343);
  RANGE (PORT_PROTOCOL_TCP, 6346, 6347);
  RANGE (PORT_PROTOCOL_TCP, 6373, 6373);
  RANGE (PORT_PROTOCOL_TCP, 6382, 6382);
  RANGE (PORT_PROTOCOL_TCP, 6389, 6389);
  RANGE (PORT_PROTOCOL_TCP, 6400, 6400);
  RANGE (PORT_PROTOCOL_TCP, 6455, 6456);
  RANGE (PORT_PROTOCOL_TCP, 6471, 6471);
  RANGE (PORT_PROTOCOL_TCP, 6500, 6503);
  RANGE (PORT_PROTOCOL_TCP, 6505, 6510);
  RANGE (PORT_PROTOCOL_TCP, 6543, 6543);
  RANGE (PORT_PROTOCOL_TCP, 6547, 6550);
  RANGE (PORT_PROTOCOL_TCP, 6558, 6558);
  RANGE (PORT_PROTOCOL_TCP, 6566, 6566);
  RANGE (PORT_PROTOCOL_TCP, 6580, 6582);
  RANGE (PORT_PROTOCOL_TCP, 6588, 6588);
  RANGE (PORT_PROTOCOL_TCP, 6620, 6621);
  RANGE (PORT_PROTOCOL_TCP, 6623, 6623);
  RANGE (PORT_PROTOCOL_TCP, 6628, 6628);
  RANGE (PORT_PROTOCOL_TCP, 6631, 6631);
  RANGE (PORT_PROTOCOL_TCP, 6665, 6670);
  RANGE (PORT_PROTOCOL_TCP, 6672, 6673);
  RANGE (PORT_PROTOCOL_TCP, 6699, 6701);
  RANGE (PORT_PROTOCOL_TCP, 6714, 6714);
  RANGE (PORT_PROTOCOL_TCP, 6767, 6768);
  RANGE (PORT_PROTOCOL_TCP, 6776, 6776);
  RANGE (PORT_PROTOCOL_TCP, 6788, 6790);
  RANGE (PORT_PROTOCOL_TCP, 6831, 6831);
  RANGE (PORT_PROTOCOL_TCP, 6841, 6842);
  RANGE (PORT_PROTOCOL_TCP, 6850, 6850);
  RANGE (PORT_PROTOCOL_TCP, 6881, 6889);
  RANGE (PORT_PROTOCOL_TCP, 6891, 6891);
  RANGE (PORT_PROTOCOL_TCP, 6901, 6901);
  RANGE (PORT_PROTOCOL_TCP, 6939, 6939);
  RANGE (PORT_PROTOCOL_TCP, 6961, 6966);
  RANGE (PORT_PROTOCOL_TCP, 6969, 6970);
  RANGE (PORT_PROTOCOL_TCP, 6998, 7015);
  RANGE (PORT_PROTOCOL_TCP, 7020, 7021);
  RANGE (PORT_PROTOCOL_TCP, 7030, 7030);
  RANGE (PORT_PROTOCOL_TCP, 7070, 7070);
  RANGE (PORT_PROTOCOL_TCP, 7099, 7100);
  RANGE (PORT_PROTOCOL_TCP, 7121, 7121);
  RANGE (PORT_PROTOCOL_TCP, 7161, 7161);
  RANGE (PORT_PROTOCOL_TCP, 7170, 7170);
  RANGE (PORT_PROTOCOL_TCP, 7174, 7174);
  RANGE (PORT_PROTOCOL_TCP, 7200, 7201);
  RANGE (PORT_PROTOCOL_TCP, 7210, 7210);
  RANGE (PORT_PROTOCOL_TCP, 7269, 7269);
  RANGE (PORT_PROTOCOL_TCP, 7273, 7273);
  RANGE (PORT_PROTOCOL_TCP, 7280, 7281);
  RANGE (PORT_PROTOCOL_TCP, 7283, 7283);
  RANGE (PORT_PROTOCOL_TCP, 7300, 7300);
  RANGE (PORT_PROTOCOL_TCP, 7320, 7320);
  RANGE (PORT_PROTOCOL_TCP, 7326, 7326);
  RANGE (PORT_PROTOCOL_TCP, 7391, 7392);
  RANGE (PORT_PROTOCOL_TCP, 7395, 7395);
  RANGE (PORT_PROTOCOL_TCP, 7426, 7431);
  RANGE (PORT_PROTOCOL_TCP, 7437, 7437);
  RANGE (PORT_PROTOCOL_TCP, 7464, 7464);
  RANGE (PORT_PROTOCOL_TCP, 7491, 7491);
  RANGE (PORT_PROTOCOL_TCP, 7501, 7501);
  RANGE (PORT_PROTOCOL_TCP, 7510, 7511);
  RANGE (PORT_PROTOCOL_TCP, 7544, 7545);
  RANGE (PORT_PROTOCOL_TCP, 7560, 7560);
  RANGE (PORT_PROTOCOL_TCP, 7566, 7566);
  RANGE (PORT_PROTOCOL_TCP, 7570, 7570);
  RANGE (PORT_PROTOCOL_TCP, 7575, 7575);
  RANGE (PORT_PROTOCOL_TCP, 7588, 7588);
  RANGE (PORT_PROTOCOL_TCP, 7597, 7597);
  RANGE (PORT_PROTOCOL_TCP, 7624, 7624);
  RANGE (PORT_PROTOCOL_TCP, 7626, 7627);
  RANGE (PORT_PROTOCOL_TCP, 7633, 7634);
  RANGE (PORT_PROTOCOL_TCP, 7648, 7649);
  RANGE (PORT_PROTOCOL_TCP, 7666, 7666);
  RANGE (PORT_PROTOCOL_TCP, 7674, 7676);
  RANGE (PORT_PROTOCOL_TCP, 7743, 7743);
  RANGE (PORT_PROTOCOL_TCP, 7775, 7779);
  RANGE (PORT_PROTOCOL_TCP, 7781, 7781);
  RANGE (PORT_PROTOCOL_TCP, 7786, 7786);
  RANGE (PORT_PROTOCOL_TCP, 7797, 7798);
  RANGE (PORT_PROTOCOL_TCP, 7800, 7801);
  RANGE (PORT_PROTOCOL_TCP, 7845, 7846);
  RANGE (PORT_PROTOCOL_TCP, 7875, 7875);
  RANGE (PORT_PROTOCOL_TCP, 7902, 7902);
  RANGE (PORT_PROTOCOL_TCP, 7913, 7913);
  RANGE (PORT_PROTOCOL_TCP, 7932, 7933);
  RANGE (PORT_PROTOCOL_TCP, 7967, 7967);
  RANGE (PORT_PROTOCOL_TCP, 7979, 7980);
  RANGE (PORT_PROTOCOL_TCP, 7999, 8005);
  RANGE (PORT_PROTOCOL_TCP, 8007, 8010);
  RANGE (PORT_PROTOCOL_TCP, 8022, 8022);
  RANGE (PORT_PROTOCOL_TCP, 8032, 8033);
  RANGE (PORT_PROTOCOL_TCP, 8044, 8044);
  RANGE (PORT_PROTOCOL_TCP, 8074, 8074);
  RANGE (PORT_PROTOCOL_TCP, 8080, 8082);
  RANGE (PORT_PROTOCOL_TCP, 8088, 8089);
  RANGE (PORT_PROTOCOL_TCP, 8098, 8098);
  RANGE (PORT_PROTOCOL_TCP, 8100, 8100);
  RANGE (PORT_PROTOCOL_TCP, 8115, 8116);
  RANGE (PORT_PROTOCOL_TCP, 8118, 8118);
  RANGE (PORT_PROTOCOL_TCP, 8121, 8122);
  RANGE (PORT_PROTOCOL_TCP, 8130, 8132);
  RANGE (PORT_PROTOCOL_TCP, 8160, 8161);
  RANGE (PORT_PROTOCOL_TCP, 8181, 8194);
  RANGE (PORT_PROTOCOL_TCP, 8199, 8201);
  RANGE (PORT_PROTOCOL_TCP, 8204, 8208);
  RANGE (PORT_PROTOCOL_TCP, 8224, 8225);
  RANGE (PORT_PROTOCOL_TCP, 8245, 8245);
  RANGE (PORT_PROTOCOL_TCP, 8311, 8311);
  RANGE (PORT_PROTOCOL_TCP, 8351, 8351);
  RANGE (PORT_PROTOCOL_TCP, 8376, 8380);
  RANGE (PORT_PROTOCOL_TCP, 8400, 8403);
  RANGE (PORT_PROTOCOL_TCP, 8416, 8417);
  RANGE (PORT_PROTOCOL_TCP, 8431, 8431);
  RANGE (PORT_PROTOCOL_TCP, 8443, 8444);
  RANGE (PORT_PROTOCOL_TCP, 8450, 8450);
  RANGE (PORT_PROTOCOL_TCP, 8473, 8473);
  RANGE (PORT_PROTOCOL_TCP, 8554, 8555);
  RANGE (PORT_PROTOCOL_TCP, 8649, 8649);
  RANGE (PORT_PROTOCOL_TCP, 8733, 8733);
  RANGE (PORT_PROTOCOL_TCP, 8763, 8765);
  RANGE (PORT_PROTOCOL_TCP, 8786, 8787);
  RANGE (PORT_PROTOCOL_TCP, 8804, 8804);
  RANGE (PORT_PROTOCOL_TCP, 8863, 8864);
  RANGE (PORT_PROTOCOL_TCP, 8875, 8875);
  RANGE (PORT_PROTOCOL_TCP, 8880, 8880);
  RANGE (PORT_PROTOCOL_TCP, 8888, 8894);
  RANGE (PORT_PROTOCOL_TCP, 8900, 8901);
  RANGE (PORT_PROTOCOL_TCP, 8910, 8911);
  RANGE (PORT_PROTOCOL_TCP, 8954, 8954);
  RANGE (PORT_PROTOCOL_TCP, 8989, 8989);
  RANGE (PORT_PROTOCOL_TCP, 8999, 9002);
  RANGE (PORT_PROTOCOL_TCP, 9006, 9006);
  RANGE (PORT_PROTOCOL_TCP, 9009, 9009);
  RANGE (PORT_PROTOCOL_TCP, 9020, 9026);
  RANGE (PORT_PROTOCOL_TCP, 9080, 9080);
  RANGE (PORT_PROTOCOL_TCP, 9090, 9091);
  RANGE (PORT_PROTOCOL_TCP, 9100, 9103);
  RANGE (PORT_PROTOCOL_TCP, 9110, 9111);
  RANGE (PORT_PROTOCOL_TCP, 9131, 9131);
  RANGE (PORT_PROTOCOL_TCP, 9152, 9152);
  RANGE (PORT_PROTOCOL_TCP, 9160, 9164);
  RANGE (PORT_PROTOCOL_TCP, 9200, 9207);
  RANGE (PORT_PROTOCOL_TCP, 9210, 9211);
  RANGE (PORT_PROTOCOL_TCP, 9217, 9217);
  RANGE (PORT_PROTOCOL_TCP, 9281, 9285);
  RANGE (PORT_PROTOCOL_TCP, 9287, 9287);
  RANGE (PORT_PROTOCOL_TCP, 9292, 9292);
  RANGE (PORT_PROTOCOL_TCP, 9321, 9321);
  RANGE (PORT_PROTOCOL_TCP, 9343, 9344);
  RANGE (PORT_PROTOCOL_TCP, 9346, 9346);
  RANGE (PORT_PROTOCOL_TCP, 9374, 9374);
  RANGE (PORT_PROTOCOL_TCP, 9390, 9390);
  RANGE (PORT_PROTOCOL_TCP, 9396, 9397);
  RANGE (PORT_PROTOCOL_TCP, 9400, 9400);
  RANGE (PORT_PROTOCOL_TCP, 9418, 9418);
  RANGE (PORT_PROTOCOL_TCP, 9495, 9495);
  RANGE (PORT_PROTOCOL_TCP, 9500, 9500);
  RANGE (PORT_PROTOCOL_TCP, 9535, 9537);
  RANGE (PORT_PROTOCOL_TCP, 9593, 9595);
  RANGE (PORT_PROTOCOL_TCP, 9600, 9600);
  RANGE (PORT_PROTOCOL_TCP, 9612, 9612);
  RANGE (PORT_PROTOCOL_TCP, 9704, 9704);
  RANGE (PORT_PROTOCOL_TCP, 9747, 9747);
  RANGE (PORT_PROTOCOL_TCP, 9753, 9753);
  RANGE (PORT_PROTOCOL_TCP, 9797, 9797);
  RANGE (PORT_PROTOCOL_TCP, 9800, 9802);
  RANGE (PORT_PROTOCOL_TCP, 9872, 9872);
  RANGE (PORT_PROTOCOL_TCP, 9875, 9876);
  RANGE (PORT_PROTOCOL_TCP, 9888, 9889);
  RANGE (PORT_PROTOCOL_TCP, 9898, 9901);
  RANGE (PORT_PROTOCOL_TCP, 9909, 9909);
  RANGE (PORT_PROTOCOL_TCP, 9911, 9911);
  RANGE (PORT_PROTOCOL_TCP, 9950, 9952);
  RANGE (PORT_PROTOCOL_TCP, 9990, 10005);
  RANGE (PORT_PROTOCOL_TCP, 10007, 10008);
  RANGE (PORT_PROTOCOL_TCP, 10012, 10012);
  RANGE (PORT_PROTOCOL_TCP, 10080, 10083);
  RANGE (PORT_PROTOCOL_TCP, 10101, 10103);
  RANGE (PORT_PROTOCOL_TCP, 10113, 10116);
  RANGE (PORT_PROTOCOL_TCP, 10128, 10128);
  RANGE (PORT_PROTOCOL_TCP, 10252, 10252);
  RANGE (PORT_PROTOCOL_TCP, 10260, 10260);
  RANGE (PORT_PROTOCOL_TCP, 10288, 10288);
  RANGE (PORT_PROTOCOL_TCP, 10607, 10607);
  RANGE (PORT_PROTOCOL_TCP, 10666, 10666);
  RANGE (PORT_PROTOCOL_TCP, 10752, 10752);
  RANGE (PORT_PROTOCOL_TCP, 10990, 10990);
  RANGE (PORT_PROTOCOL_TCP, 11000, 11001);
  RANGE (PORT_PROTOCOL_TCP, 11111, 11111);
  RANGE (PORT_PROTOCOL_TCP, 11201, 11201);
  RANGE (PORT_PROTOCOL_TCP, 11223, 11223);
  RANGE (PORT_PROTOCOL_TCP, 11319, 11321);
  RANGE (PORT_PROTOCOL_TCP, 11367, 11367);
  RANGE (PORT_PROTOCOL_TCP, 11371, 11371);
  RANGE (PORT_PROTOCOL_TCP, 11600, 11600);
  RANGE (PORT_PROTOCOL_TCP, 11720, 11720);
  RANGE (PORT_PROTOCOL_TCP, 11751, 11751);
  RANGE (PORT_PROTOCOL_TCP, 11965, 11965);
  RANGE (PORT_PROTOCOL_TCP, 11967, 11967);
  RANGE (PORT_PROTOCOL_TCP, 11999, 12006);
  RANGE (PORT_PROTOCOL_TCP, 12076, 12076);
  RANGE (PORT_PROTOCOL_TCP, 12109, 12109);
  RANGE (PORT_PROTOCOL_TCP, 12168, 12168);
  RANGE (PORT_PROTOCOL_TCP, 12172, 12172);
  RANGE (PORT_PROTOCOL_TCP, 12223, 12223);
  RANGE (PORT_PROTOCOL_TCP, 12321, 12321);
  RANGE (PORT_PROTOCOL_TCP, 12345, 12346);
  RANGE (PORT_PROTOCOL_TCP, 12361, 12362);
  RANGE (PORT_PROTOCOL_TCP, 12468, 12468);
  RANGE (PORT_PROTOCOL_TCP, 12701, 12701);
  RANGE (PORT_PROTOCOL_TCP, 12753, 12753);
  RANGE (PORT_PROTOCOL_TCP, 13160, 13160);
  RANGE (PORT_PROTOCOL_TCP, 13223, 13224);
  RANGE (PORT_PROTOCOL_TCP, 13701, 13702);
  RANGE (PORT_PROTOCOL_TCP, 13705, 13706);
  RANGE (PORT_PROTOCOL_TCP, 13708, 13718);
  RANGE (PORT_PROTOCOL_TCP, 13720, 13722);
  RANGE (PORT_PROTOCOL_TCP, 13724, 13724);
  RANGE (PORT_PROTOCOL_TCP, 13782, 13783);
  RANGE (PORT_PROTOCOL_TCP, 13818, 13822);
  RANGE (PORT_PROTOCOL_TCP, 14001, 14001);
  RANGE (PORT_PROTOCOL_TCP, 14033, 14034);
  RANGE (PORT_PROTOCOL_TCP, 14141, 14141);
  RANGE (PORT_PROTOCOL_TCP, 14145, 14145);
  RANGE (PORT_PROTOCOL_TCP, 14149, 14149);
  RANGE (PORT_PROTOCOL_TCP, 14194, 14194);
  RANGE (PORT_PROTOCOL_TCP, 14237, 14237);
  RANGE (PORT_PROTOCOL_TCP, 14936, 14937);
  RANGE (PORT_PROTOCOL_TCP, 15000, 15000);
  RANGE (PORT_PROTOCOL_TCP, 15126, 15126);
  RANGE (PORT_PROTOCOL_TCP, 15345, 15345);
  RANGE (PORT_PROTOCOL_TCP, 15363, 15363);
  RANGE (PORT_PROTOCOL_TCP, 16360, 16361);
  RANGE (PORT_PROTOCOL_TCP, 16367, 16368);
  RANGE (PORT_PROTOCOL_TCP, 16384, 16384);
  RANGE (PORT_PROTOCOL_TCP, 16660, 16661);
  RANGE (PORT_PROTOCOL_TCP, 16959, 16959);
  RANGE (PORT_PROTOCOL_TCP, 16969, 16969);
  RANGE (PORT_PROTOCOL_TCP, 16991, 16991);
  RANGE (PORT_PROTOCOL_TCP, 17007, 17007);
  RANGE (PORT_PROTOCOL_TCP, 17185, 17185);
  RANGE (PORT_PROTOCOL_TCP, 17219, 17219);
  RANGE (PORT_PROTOCOL_TCP, 17300, 17300);
  RANGE (PORT_PROTOCOL_TCP, 17770, 17772);
  RANGE (PORT_PROTOCOL_TCP, 18000, 18000);
  RANGE (PORT_PROTOCOL_TCP, 18181, 18187);
  RANGE (PORT_PROTOCOL_TCP, 18190, 18190);
  RANGE (PORT_PROTOCOL_TCP, 18241, 18241);
  RANGE (PORT_PROTOCOL_TCP, 18463, 18463);
  RANGE (PORT_PROTOCOL_TCP, 18769, 18769);
  RANGE (PORT_PROTOCOL_TCP, 18888, 18888);
  RANGE (PORT_PROTOCOL_TCP, 19191, 19191);
  RANGE (PORT_PROTOCOL_TCP, 19194, 19194);
  RANGE (PORT_PROTOCOL_TCP, 19283, 19283);
  RANGE (PORT_PROTOCOL_TCP, 19315, 19315);
  RANGE (PORT_PROTOCOL_TCP, 19398, 19398);
  RANGE (PORT_PROTOCOL_TCP, 19410, 19412);
  RANGE (PORT_PROTOCOL_TCP, 19540, 19541);
  RANGE (PORT_PROTOCOL_TCP, 19638, 19638);
  RANGE (PORT_PROTOCOL_TCP, 19726, 19726);
  RANGE (PORT_PROTOCOL_TCP, 20000, 20001);
  RANGE (PORT_PROTOCOL_TCP, 20005, 20005);
  RANGE (PORT_PROTOCOL_TCP, 20011, 20012);
  RANGE (PORT_PROTOCOL_TCP, 20034, 20034);
  RANGE (PORT_PROTOCOL_TCP, 20200, 20200);
  RANGE (PORT_PROTOCOL_TCP, 20202, 20203);
  RANGE (PORT_PROTOCOL_TCP, 20222, 20222);
  RANGE (PORT_PROTOCOL_TCP, 20670, 20670);
  RANGE (PORT_PROTOCOL_TCP, 20999, 21000);
  RANGE (PORT_PROTOCOL_TCP, 21490, 21490);
  RANGE (PORT_PROTOCOL_TCP, 21544, 21544);
  RANGE (PORT_PROTOCOL_TCP, 21590, 21590);
  RANGE (PORT_PROTOCOL_TCP, 21800, 21800);
  RANGE (PORT_PROTOCOL_TCP, 21845, 21849);
  RANGE (PORT_PROTOCOL_TCP, 22000, 22001);
  RANGE (PORT_PROTOCOL_TCP, 22222, 22222);
  RANGE (PORT_PROTOCOL_TCP, 22273, 22273);
  RANGE (PORT_PROTOCOL_TCP, 22289, 22289);
  RANGE (PORT_PROTOCOL_TCP, 22305, 22305);
  RANGE (PORT_PROTOCOL_TCP, 22321, 22321);
  RANGE (PORT_PROTOCOL_TCP, 22370, 22370);
  RANGE (PORT_PROTOCOL_TCP, 22555, 22555);
  RANGE (PORT_PROTOCOL_TCP, 22800, 22800);
  RANGE (PORT_PROTOCOL_TCP, 22951, 22951);
  RANGE (PORT_PROTOCOL_TCP, 23456, 23456);
  RANGE (PORT_PROTOCOL_TCP, 24000, 24006);
  RANGE (PORT_PROTOCOL_TCP, 24242, 24242);
  RANGE (PORT_PROTOCOL_TCP, 24249, 24249);
  RANGE (PORT_PROTOCOL_TCP, 24345, 24347);
  RANGE (PORT_PROTOCOL_TCP, 24386, 24386);
  RANGE (PORT_PROTOCOL_TCP, 24554, 24554);
  RANGE (PORT_PROTOCOL_TCP, 24677, 24678);
  RANGE (PORT_PROTOCOL_TCP, 24922, 24922);
  RANGE (PORT_PROTOCOL_TCP, 25000, 25009);
  RANGE (PORT_PROTOCOL_TCP, 25378, 25378);
  RANGE (PORT_PROTOCOL_TCP, 25544, 25544);
  RANGE (PORT_PROTOCOL_TCP, 25793, 25793);
  RANGE (PORT_PROTOCOL_TCP, 25867, 25867);
  RANGE (PORT_PROTOCOL_TCP, 25901, 25901);
  RANGE (PORT_PROTOCOL_TCP, 25903, 25903);
  RANGE (PORT_PROTOCOL_TCP, 26000, 26000);
  RANGE (PORT_PROTOCOL_TCP, 26208, 26208);
  RANGE (PORT_PROTOCOL_TCP, 26260, 26264);
  RANGE (PORT_PROTOCOL_TCP, 27000, 27010);
  RANGE (PORT_PROTOCOL_TCP, 27345, 27345);
  RANGE (PORT_PROTOCOL_TCP, 27374, 27374);
  RANGE (PORT_PROTOCOL_TCP, 27504, 27504);
  RANGE (PORT_PROTOCOL_TCP, 27665, 27665);
  RANGE (PORT_PROTOCOL_TCP, 27999, 27999);
  RANGE (PORT_PROTOCOL_TCP, 28001, 28001);
  RANGE (PORT_PROTOCOL_TCP, 29559, 29559);
  RANGE (PORT_PROTOCOL_TCP, 29891, 29891);
  RANGE (PORT_PROTOCOL_TCP, 30001, 30002);
  RANGE (PORT_PROTOCOL_TCP, 30100, 30102);
  RANGE (PORT_PROTOCOL_TCP, 30303, 30303);
  RANGE (PORT_PROTOCOL_TCP, 30999, 30999);
  RANGE (PORT_PROTOCOL_TCP, 31337, 31337);
  RANGE (PORT_PROTOCOL_TCP, 31339, 31339);
  RANGE (PORT_PROTOCOL_TCP, 31416, 31416);
  RANGE (PORT_PROTOCOL_TCP, 31457, 31457);
  RANGE (PORT_PROTOCOL_TCP, 31554, 31554);
  RANGE (PORT_PROTOCOL_TCP, 31556, 31556);
  RANGE (PORT_PROTOCOL_TCP, 31620, 31620);
  RANGE (PORT_PROTOCOL_TCP, 31765, 31765);
  RANGE (PORT_PROTOCOL_TCP, 31785, 31787);
  RANGE (PORT_PROTOCOL_TCP, 32261, 32261);
  RANGE (PORT_PROTOCOL_TCP, 32666, 32666);
  RANGE (PORT_PROTOCOL_TCP, 32768, 32780);
  RANGE (PORT_PROTOCOL_TCP, 32786, 32787);
  RANGE (PORT_PROTOCOL_TCP, 32896, 32896);
  RANGE (PORT_PROTOCOL_TCP, 33270, 33270);
  RANGE (PORT_PROTOCOL_TCP, 33331, 33331);
  RANGE (PORT_PROTOCOL_TCP, 33434, 33434);
  RANGE (PORT_PROTOCOL_TCP, 33911, 33911);
  RANGE (PORT_PROTOCOL_TCP, 34249, 34249);
  RANGE (PORT_PROTOCOL_TCP, 34324, 34324);
  RANGE (PORT_PROTOCOL_TCP, 34952, 34952);
  RANGE (PORT_PROTOCOL_TCP, 36865, 36865);
  RANGE (PORT_PROTOCOL_TCP, 37475, 37475);
  RANGE (PORT_PROTOCOL_TCP, 37651, 37651);
  RANGE (PORT_PROTOCOL_TCP, 38037, 38037);
  RANGE (PORT_PROTOCOL_TCP, 38201, 38201);
  RANGE (PORT_PROTOCOL_TCP, 38292, 38293);
  RANGE (PORT_PROTOCOL_TCP, 39681, 39681);
  RANGE (PORT_PROTOCOL_TCP, 40412, 40412);
  RANGE (PORT_PROTOCOL_TCP, 40841, 40843);
  RANGE (PORT_PROTOCOL_TCP, 41111, 41111);
  RANGE (PORT_PROTOCOL_TCP, 41508, 41508);
  RANGE (PORT_PROTOCOL_TCP, 41794, 41795);
  RANGE (PORT_PROTOCOL_TCP, 42508, 42510);
  RANGE (PORT_PROTOCOL_TCP, 43118, 43118);
  RANGE (PORT_PROTOCOL_TCP, 43188, 43190);
  RANGE (PORT_PROTOCOL_TCP, 44321, 44322);
  RANGE (PORT_PROTOCOL_TCP, 44333, 44334);
  RANGE (PORT_PROTOCOL_TCP, 44442, 44443);
  RANGE (PORT_PROTOCOL_TCP, 44818, 44818);
  RANGE (PORT_PROTOCOL_TCP, 45000, 45000);
  RANGE (PORT_PROTOCOL_TCP, 45054, 45054);
  RANGE (PORT_PROTOCOL_TCP, 45678, 45678);
  RANGE (PORT_PROTOCOL_TCP, 45966, 45966);
  RANGE (PORT_PROTOCOL_TCP, 47000, 47000);
  RANGE (PORT_PROTOCOL_TCP, 47557, 47557);
  RANGE (PORT_PROTOCOL_TCP, 47624, 47624);
  RANGE (PORT_PROTOCOL_TCP, 47806, 47806);
  RANGE (PORT_PROTOCOL_TCP, 47808, 47808);
  RANGE (PORT_PROTOCOL_TCP, 47891, 47891);
  RANGE (PORT_PROTOCOL_TCP, 48000, 48003);
  RANGE (PORT_PROTOCOL_TCP, 48556, 48556);
  RANGE (PORT_PROTOCOL_TCP, 49400, 49400);
  RANGE (PORT_PROTOCOL_TCP, 50000, 50004);
  RANGE (PORT_PROTOCOL_TCP, 50505, 50505);
  RANGE (PORT_PROTOCOL_TCP, 50776, 50776);
  RANGE (PORT_PROTOCOL_TCP, 51210, 51210);
  RANGE (PORT_PROTOCOL_TCP, 53001, 53001);
  RANGE (PORT_PROTOCOL_TCP, 54320, 54321);
  RANGE (PORT_PROTOCOL_TCP, 57341, 57341);
  RANGE (PORT_PROTOCOL_TCP, 59595, 59595);
  RANGE (PORT_PROTOCOL_TCP, 60177, 60177);
  RANGE (PORT_PROTOCOL_TCP, 60179, 60179);
  RANGE (PORT_PROTOCOL_TCP, 61439, 61441);
  RANGE (PORT_PROTOCOL_TCP, 61446, 61446);
  RANGE (PORT_PROTOCOL_TCP, 65000, 65000);
  RANGE (PORT_PROTOCOL_TCP, 65301, 65301);
}

/**
 * @brief Ensure the predefined scanner exists.
 *
 * @return 0 if success, -1 if error.
 */
static int
check_db_scanners ()
{
  if (sql_int ("SELECT count(*) FROM scanners WHERE uuid = '%s';",
               SCANNER_UUID_DEFAULT) == 0)
    {
      sql ("INSERT INTO scanners"
           " (uuid, owner, name, host, port, type, ca_pub, credential,"
           "  creation_time, modification_time)"
           " VALUES ('" SCANNER_UUID_DEFAULT "', NULL, 'OpenVAS Default',"
           " '" OPENVASSD_ADDRESS "', 0, %d, NULL, NULL, m_now (), m_now ());",
           SCANNER_TYPE_OPENVAS);
    }

  if (sql_int ("SELECT count(*) FROM scanners WHERE uuid = '%s';",
               SCANNER_UUID_CVE) == 0)
    sql ("INSERT INTO scanners"
         " (uuid, owner, name, host, port, type, ca_pub, credential,"
         "  creation_time, modification_time)"
         " VALUES ('" SCANNER_UUID_CVE "', NULL, 'CVE',"
         " '', 0, %d, NULL, NULL, m_now (), m_now ());",
         SCANNER_TYPE_CVE);

  return 0;
}

/**
 * @brief Ensure that the predefined port lists exist.
 */
static void
check_db_port_lists ()
{
  if (sql_int ("SELECT count(*) FROM port_lists"
               " WHERE uuid = '" PORT_LIST_UUID_DEFAULT "';")
      == 0)
    {
      port_list_t list;
      sql ("INSERT INTO port_lists (uuid, owner, name, comment, creation_time,"
           "                        modification_time)"
           " VALUES ('" PORT_LIST_UUID_DEFAULT "', NULL, 'OpenVAS Default',"
           " '', m_now (), m_now ())");
      list = sql_last_insert_id ();
      make_port_ranges_openvas_default (list);
    }

  if (sql_int ("SELECT count(*) FROM port_lists"
               " WHERE uuid = '" PORT_LIST_UUID_ALL_TCP "';")
      == 0)
    {
      port_list_t list;
      sql ("INSERT INTO port_lists (uuid, owner, name, comment, creation_time,"
           "                        modification_time)"
           " VALUES ('" PORT_LIST_UUID_ALL_TCP "', NULL, 'All TCP',"
           " '', m_now (), m_now ())");
      list = sql_last_insert_id ();
      RANGE (PORT_PROTOCOL_TCP, 1, 65535);
    }

  if (sql_int ("SELECT count(*) FROM port_lists"
               " WHERE uuid = '" PORT_LIST_UUID_ALL_TCP_NMAP_5_51_TOP_100 "';")
      == 0)
    {
      port_list_t list;
      sql ("INSERT INTO port_lists (uuid, owner, name, comment, creation_time,"
           "                        modification_time)"
           " VALUES ('" PORT_LIST_UUID_ALL_TCP_NMAP_5_51_TOP_100 "', NULL,"
           " 'All TCP and Nmap 5.51 top 100 UDP', '', m_now (), m_now ())");
      list = sql_last_insert_id ();
      make_port_ranges_all_tcp_nmap_5_51_top_100 (list);
    }

  if (sql_int ("SELECT count(*) FROM port_lists"
               " WHERE uuid = '" PORT_LIST_UUID_ALL_TCP_NMAP_5_51_TOP_1000 "';")
      == 0)
    {
      port_list_t list;
      sql ("INSERT INTO port_lists (uuid, owner, name, comment, creation_time,"
           "                        modification_time)"
           " VALUES ('" PORT_LIST_UUID_ALL_TCP_NMAP_5_51_TOP_1000 "', NULL,"
           " 'All TCP and Nmap 5.51 top 1000 UDP', '', m_now (), m_now ())");
      list = sql_last_insert_id ();
      make_port_ranges_all_tcp_nmap_5_51_top_1000 (list);
    }

  if (sql_int ("SELECT count(*) FROM port_lists"
               " WHERE uuid = '" PORT_LIST_UUID_ALL_PRIV_TCP "';")
      == 0)
    {
      port_list_t list;
      sql ("INSERT INTO port_lists (uuid, owner, name, comment, creation_time,"
           "                        modification_time)"
           " VALUES ('" PORT_LIST_UUID_ALL_PRIV_TCP "', NULL,"
           " 'All privileged TCP', '', m_now (), m_now ())");
      list = sql_last_insert_id ();
      RANGE (PORT_PROTOCOL_TCP, 1, 1023);
    }

  if (sql_int ("SELECT count(*) FROM port_lists"
               " WHERE uuid = '" PORT_LIST_UUID_ALL_PRIV_TCP_UDP "';")
      == 0)
    {
      port_list_t list;
      sql ("INSERT INTO port_lists (uuid, owner, name, comment, creation_time,"
           "                        modification_time)"
           " VALUES ('" PORT_LIST_UUID_ALL_PRIV_TCP_UDP "', NULL,"
           " 'All privileged TCP and UDP', '', m_now (), m_now ())");
      list = sql_last_insert_id ();
      RANGE (PORT_PROTOCOL_TCP, 1, 1023);
      RANGE (PORT_PROTOCOL_UDP, 1, 1023);
    }

  if (sql_int ("SELECT count(*) FROM port_lists"
               " WHERE uuid = '" PORT_LIST_UUID_ALL_IANA_TCP_2012 "';")
      == 0)
    {
      port_list_t list;
      sql ("INSERT INTO port_lists (uuid, owner, name, comment, creation_time,"
           "                        modification_time)"
           " VALUES ('" PORT_LIST_UUID_ALL_IANA_TCP_2012 "', NULL,"
           " 'All IANA assigned TCP 2012-02-10', '', m_now (), m_now ())");
      list = sql_last_insert_id ();
      make_port_ranges_iana_tcp_2012 (list);
    }

  if (sql_int ("SELECT count(*) FROM port_lists"
               " WHERE uuid = '" PORT_LIST_UUID_ALL_IANA_TCP_UDP_2012 "';")
      == 0)
    {
      port_list_t list;
      sql ("INSERT INTO port_lists (uuid, owner, name, comment, creation_time,"
           "                        modification_time)"
           " VALUES ('" PORT_LIST_UUID_ALL_IANA_TCP_UDP_2012 "', NULL,"
           " 'All IANA assigned TCP and UDP 2012-02-10', '', m_now (), m_now ())");
      list = sql_last_insert_id ();
      make_port_ranges_iana_tcp_udp_2012 (list);
    }

  if (sql_int ("SELECT count(*) FROM port_lists"
               " WHERE uuid = '" PORT_LIST_UUID_NMAP_5_51_TOP_2000_TOP_100 "';")
      == 0)
    {
      port_list_t list;
      sql ("INSERT INTO port_lists (uuid, owner, name, comment, creation_time,"
           "                        modification_time)"
           " VALUES ('" PORT_LIST_UUID_NMAP_5_51_TOP_2000_TOP_100 "', NULL,"
           " 'Nmap 5.51 top 2000 TCP and top 100 UDP', '', m_now (), m_now ())");
      list = sql_last_insert_id ();
      make_port_ranges_nmap_5_51_top_2000_top_100 (list);
    }
  /*
   * Ensure that the highest number in a port range is 65535.  At some point
   * ranges were initialised to 65536.
   *
   * This should be a migrator, but this way is easier to backport.  */
  sql ("UPDATE port_ranges SET \"end\" = 65535 WHERE \"end\" = 65536;");
  sql ("UPDATE port_ranges SET start = 65535 WHERE start = 65536;");
}

/**
 * @brief Bring UUIDs for single predefined report format up to date.
 *
 * @param[in]  old  Old UUID.
 * @param[in]  new  New UUID.
 */
static void
update_report_format_uuid (const char *old, const char *new)
{
  gchar *dir;

  dir = predefined_report_format_dir (old);
  if (g_file_test (dir, G_FILE_TEST_EXISTS))
    openvas_file_remove_recurse (dir);
  g_free (dir);

  sql ("UPDATE report_formats"
       " SET uuid = '%s', modification_time = m_now ()"
       " WHERE uuid = '%s';",
       new,
       old);

  sql ("UPDATE alert_method_data"
       " SET data = '%s'"
       " WHERE data = '%s';",
       new,
       old);
}

/**
 * @brief Bring report format UUIDs in database up to date.
 */
static void
update_report_format_uuids ()
{
  /* Same as migrate_58_to_59, to enable backporting r13519 to OpenVAS-5
   * without backporting the 58 to 59 migrator.  In future these should be
   * done here instead of in a migrator. */

  update_report_format_uuid ("a0704abb-2120-489f-959f-251c9f4ffebd",
                             "5ceff8ba-1f62-11e1-ab9f-406186ea4fc5");

  update_report_format_uuid ("b993b6f5-f9fb-4e6e-9c94-dd46c00e058d",
                             "6c248850-1f62-11e1-b082-406186ea4fc5");

  update_report_format_uuid ("929884c6-c2c4-41e7-befb-2f6aa163b458",
                             "77bd6c4a-1f62-11e1-abf0-406186ea4fc5");

  update_report_format_uuid ("9f1ab17b-aaaa-411a-8c57-12df446f5588",
                             "7fcc3a1a-1f62-11e1-86bf-406186ea4fc5");

  update_report_format_uuid ("f5c2a364-47d2-4700-b21d-0a7693daddab",
                             "9ca6fe72-1f62-11e1-9e7c-406186ea4fc5");

  update_report_format_uuid ("1a60a67e-97d0-4cbf-bc77-f71b08e7043d",
                             "a0b5bfb2-1f62-11e1-85db-406186ea4fc5");

  update_report_format_uuid ("19f6f1b3-7128-4433-888c-ccc764fe6ed5",
                             "a3810a62-1f62-11e1-9219-406186ea4fc5");

  update_report_format_uuid ("d5da9f67-8551-4e51-807b-b6a873d70e34",
                             "a994b278-1f62-11e1-96ac-406186ea4fc5");

  /* New updates go here.  Oldest must come first, so add at the end. */

  update_report_format_uuid ("7fcc3a1a-1f62-11e1-86bf-406186ea4fc5",
                             "a684c02c-b531-11e1-bdc2-406186ea4fc5");

  update_report_format_uuid ("a0b5bfb2-1f62-11e1-85db-406186ea4fc5",
                             "c402cc3e-b531-11e1-9163-406186ea4fc5");
}

/**
 * @brief Initialize the default settings.
 *
 * Ensure all the default manager settings exist.
 */
static void
check_db_settings ()
{
  if (sql_int ("SELECT count(*) FROM settings"
               " WHERE uuid = '6765549a-934e-11e3-b358-406186ea4fc5'"
               " AND " ACL_IS_GLOBAL () ";")
      == 0)
    sql ("INSERT into settings (uuid, owner, name, comment, value)"
         " VALUES"
         " ('6765549a-934e-11e3-b358-406186ea4fc5', NULL,"
         "  'User Interface Language',"
         "  'Preferred language to be used in client user interfaces.',"
         "  'Browser Language');");

  if (sql_int ("SELECT count(*) FROM settings"
               " WHERE uuid = '" SETTING_UUID_ROWS_PER_PAGE "'"
               " AND " ACL_IS_GLOBAL () ";")
      == 0)
    sql ("INSERT into settings (uuid, owner, name, comment, value)"
         " VALUES"
         " ('" SETTING_UUID_ROWS_PER_PAGE "', NULL, 'Rows Per Page',"
         "  'The default number of rows displayed in any listing.',"
         "  10);");

  if (sql_int ("SELECT count(*) FROM settings"
               " WHERE uuid = '" SETTING_UUID_MAX_ROWS_PER_PAGE "'"
               " AND " ACL_IS_GLOBAL () ";")
      == 0)
    sql ("INSERT into settings (uuid, owner, name, comment, value)"
         " VALUES"
         " ('" SETTING_UUID_MAX_ROWS_PER_PAGE "', NULL, 'Max Rows Per Page',"
         "  'The default maximum number of rows displayed in any listing.',"
         "  1000);");

  if (sql_int ("SELECT count(*) FROM settings"
               " WHERE uuid = 'f16bb236-a32d-4cd5-a880-e0fcf2599f59'"
               " AND " ACL_IS_GLOBAL () ";")
      == 0)
    sql ("INSERT into settings (uuid, owner, name, comment, value)"
         " VALUES"
         " ('f16bb236-a32d-4cd5-a880-e0fcf2599f59', NULL, 'Severity Class',"
         "  'Severity class used for severity bars.',"
         "  'nist');");

  if (sql_int ("SELECT count(*) FROM settings"
               " WHERE uuid = '77ec2444-e7f2-4a80-a59b-f4237782d93f'"
               " AND " ACL_IS_GLOBAL () ";")
      == 0)
    sql ("INSERT into settings (uuid, owner, name, comment, value)"
         " VALUES"
         " ('77ec2444-e7f2-4a80-a59b-f4237782d93f', NULL, 'Dynamic Severity',"
         "  'Whether to use dynamic severity scores by default.',"
         "  '0');");

  if (sql_int ("SELECT count(*) FROM settings"
               " WHERE uuid = '578a1c14-e2dc-45ef-a591-89d31391d007'"
               " AND " ACL_IS_GLOBAL () ";")
      == 0)
    sql ("INSERT into settings (uuid, owner, name, comment, value)"
         " VALUES"
         " ('578a1c14-e2dc-45ef-a591-89d31391d007', NULL, 'Auto-Refresh',"
         "  'The delay between automatic page refreshs in seconds.',"
         "  '0');");

  if (sql_int ("SELECT count(*) FROM settings"
               " WHERE uuid = 'a6ac88c5-729c-41ba-ac0a-deea4a3441f2'"
               " AND " ACL_IS_GLOBAL () ";")
      == 0)
    sql ("INSERT into settings (uuid, owner, name, comment, value)"
         " VALUES"
         " ('a6ac88c5-729c-41ba-ac0a-deea4a3441f2', NULL,"
         "  'Details Export File Name',"
         "  'File name format string for the export of resource details.',"
         "  '%%T-%%U');");

  if (sql_int ("SELECT count(*) FROM settings"
               " WHERE uuid = '0872a6ed-4f85-48c5-ac3f-a5ef5e006745'"
               " AND " ACL_IS_GLOBAL () ";")
      == 0)
    sql ("INSERT into settings (uuid, owner, name, comment, value)"
         " VALUES"
         " ('0872a6ed-4f85-48c5-ac3f-a5ef5e006745', NULL,"
         "  'List Export File Name',"
         "  'File name format string for the export of resource lists.',"
         "  '%%T-%%D');");

  if (sql_int ("SELECT count(*) FROM settings"
               " WHERE uuid = 'e1a2ae0b-736e-4484-b029-330c9e15b900'"
               " AND " ACL_IS_GLOBAL () ";")
      == 0)
    sql ("INSERT into settings (uuid, owner, name, comment, value)"
         " VALUES"
         " ('e1a2ae0b-736e-4484-b029-330c9e15b900', NULL,"
         "  'Report Export File Name',"
         "  'File name format string for the export of reports.',"
         "  '%%T-%%U');");

  if (sql_int ("SELECT count(*) FROM settings"
               " WHERE uuid = '7eda49c5-096c-4bef-b1ab-d080d87300df'"
               " AND " ACL_IS_GLOBAL () ";")
      == 0)
    sql ("INSERT into settings (uuid, owner, name, comment, value)"
         " VALUES"
         " ('7eda49c5-096c-4bef-b1ab-d080d87300df', NULL,"
         "  'Default Severity',"
         "  'Severity to use if none is specified or available from SecInfo.',"
         "  '10.0');");

  if (sql_int ("SELECT count(*) FROM settings"
               " WHERE uuid = 'a09285b0-2d47-49b6-a4ef-946ee71f1d5c'"
               " AND " ACL_IS_GLOBAL () ";")
      == 0)
    sql ("INSERT into settings (uuid, owner, name, comment, value)"
         " VALUES"
         " ('a09285b0-2d47-49b6-a4ef-946ee71f1d5c', NULL,"
         "  'Auto Cache Rebuild',"
         "  'Whether to rebuild report caches on changes affecting severity.',"
         "  '1');");
}

/**
 * @brief Add permission to role.
 *
 * Caller must ensure args are SQL escaped.
 *
 * @param[in]  role        Role.
 * @param[in]  permission  Permission.
 */
void
add_role_permission (const gchar *role, const gchar *permission)
{
  sql ("INSERT INTO permissions"
       " (uuid, owner, name, comment, resource_type, resource, resource_uuid,"
       "  resource_location, subject_type, subject, subject_location,"
       "  creation_time, modification_time)"
       " VALUES"
       " (make_uuid (), NULL, lower ('%s'), '', '',"
       "  0, '', " G_STRINGIFY (LOCATION_TABLE) ", 'role',"
       "  (SELECT id FROM roles WHERE uuid = '%s'),"
       "  " G_STRINGIFY (LOCATION_TABLE) ", m_now (), m_now ());",
       permission,
       role);
}

/**
 * @brief Refresh nvt_cves table.
 *
 * Caller must organise transaction.
 */
void
refresh_nvt_cves ()
{
  iterator_t nvts;

  sql ("DELETE FROM nvt_cves;");

  init_iterator (&nvts, "SELECT id, oid, cve FROM nvts;");
  while (next (&nvts))
    {
      gchar **split, **point;

      split = g_strsplit_set (iterator_string (&nvts, 2), " ,", 0);

      point = split;
      while (*point)
        {
          g_strstrip (*point);
          if (strlen (*point))
            {
              gchar *quoted_cve, *quoted_oid;

              quoted_cve = sql_insert (*point);
              quoted_oid = sql_insert (iterator_string (&nvts, 1));
              sql ("INSERT INTO nvt_cves (nvt, oid, cve_name)"
                   " VALUES (%llu, %s, %s);",
                   iterator_int64 (&nvts, 0),
                   quoted_oid,
                   quoted_cve);
              g_free (quoted_cve);
              g_free (quoted_oid);
            }
          point++;
        }
      g_strfreev (split);
    }
  cleanup_iterator (&nvts);

  if (sql_is_sqlite3 ())
    sql ("REINDEX nvt_cves_by_oid;");
}

/**
 * @brief Ensure that the databases are the right versions.
 *
 * @param[in]  nvt_cache_mode  True when running in NVT caching mode.
 *
 * @return 0 success, -1 error, -2 database is wrong version, -3 database
 *         needs to be initialised from server.
 */
int
check_db_versions (int nvt_cache_mode)
{
  char *database_version;
  int scap_db_version, cert_db_version;

  database_version = sql_string ("SELECT value FROM %s.meta"
                                 " WHERE name = 'database_version';",
                                 sql_schema ());
  if (nvt_cache_mode)
    {
      if (database_version
          && strcmp (database_version,
                     G_STRINGIFY (OPENVASMD_DATABASE_VERSION)))
        {
          g_message ("%s: database version of database: %s\n",
                     __FUNCTION__,
                     database_version);
          g_message ("%s: database version supported by manager: %s\n",
                     __FUNCTION__,
                     G_STRINGIFY (OPENVASMD_DATABASE_VERSION));
          g_free (database_version);
          return -2;
        }
      g_free (database_version);

      /* If database_version was NULL then meta was missing, so assume
       * that the database is missing, which is OK. */
    }
  else
    {
      long long int count;

      if (database_version)
        {
          if (strcmp (database_version,
                      G_STRINGIFY (OPENVASMD_DATABASE_VERSION)))
            {
              g_message ("%s: database version of database: %s\n",
                         __FUNCTION__,
                         database_version);
              g_message ("%s: database version supported by manager: %s\n",
                         __FUNCTION__,
                         G_STRINGIFY (OPENVASMD_DATABASE_VERSION));
              g_free (database_version);
              return -2;
            }
          g_free (database_version);

          /* Check that the database was initialised from the scanner.
           *
           * This can also fail after a migration, for example if the database
           * was created before NVT preferences were cached in the database.
           */

          if (sql_int64 (&count,
                         "SELECT count(*) FROM %s.meta"
                         " WHERE name = 'nvts_feed_version'"
                         " OR name = 'nvt_preferences_enabled';",
                         sql_schema ())
              || count < 2)
            g_warning ("database must be initialised from scanner"
                       " (with --update or --rebuild)");
        }
      else
        /* Assume database is missing. */
        g_warning ("database must be initialised from scanner"
                   " (with --update or --rebuild)");
    }

  /* Check SCAP database version. */

  scap_db_version = manage_scap_db_version ();
  if (scap_db_version == -1)
    g_message ("No SCAP database found");
  else if (scap_db_version != manage_scap_db_supported_version ())
    {
      g_message ("%s: database version of SCAP database: %i\n",
                 __FUNCTION__,
                 scap_db_version);
      g_message ("%s: SCAP database version supported by manager: %s\n",
                 __FUNCTION__,
                 G_STRINGIFY (OPENVASMD_SCAP_DATABASE_VERSION));
      return -2;
    }

  /* Check CERT database version. */

  cert_db_version = manage_cert_db_version ();
  if (cert_db_version == -1)
    g_message ("No CERT database found");
  else if (cert_db_version != manage_cert_db_supported_version ())
    {
      g_message ("%s: database version of CERT database: %i\n",
                 __FUNCTION__,
                 cert_db_version);
      g_message ("%s: CERT database version supported by manager: %s\n",
                 __FUNCTION__,
                 G_STRINGIFY (OPENVASMD_CERT_DATABASE_VERSION));
      return -2;
    }
  return 0;
}

/**
 * @brief Ensures the sanity of nvts cache in DB.
 */
static void
check_db_nvts ()
{
  /* Ensure the nvti cache update flag exists and is clear. */
  if (sql_int ("SELECT count(*) FROM %s.meta"
               " WHERE name = 'update_nvti_cache';",
               sql_schema ()))
    sql ("UPDATE %s.meta SET value = 0 WHERE name = 'update_nvti_cache';",
         sql_schema ());
  else
    sql ("INSERT INTO %s.meta (name, value)"
         " VALUES ('update_nvti_cache', 0);",
         sql_schema ());

  /*
   * Ensure every part of the predefined selector exists.
   * This restores entries lost due to the error solved 2010-08-13 by r8805.
   */
  if (sql_int ("SELECT count(*) FROM nvt_selectors WHERE name ="
               " '" MANAGE_NVT_SELECTOR_UUID_ALL "'"
               " AND type = " G_STRINGIFY (NVT_SELECTOR_TYPE_ALL) ";")
      == 0)
    {
      sql ("INSERT into nvt_selectors (name, exclude, type, family_or_nvt)"
           " VALUES ('" MANAGE_NVT_SELECTOR_UUID_ALL "', 0, "
           G_STRINGIFY (NVT_SELECTOR_TYPE_ALL) ", NULL);");
    }

  if (sql_int ("SELECT count(*) FROM nvt_selectors WHERE name ="
               " '" MANAGE_NVT_SELECTOR_UUID_ALL "'"
               " AND type = " G_STRINGIFY (NVT_SELECTOR_TYPE_NVT)
               " AND family_or_nvt = '1.3.6.1.4.1.25623.1.0.810002';")
      == 0)
    {
      sql ("INSERT into nvt_selectors"
           " (name, exclude, type, family_or_nvt, family)"
           " VALUES ('" MANAGE_NVT_SELECTOR_UUID_ALL "', 1, "
           G_STRINGIFY (NVT_SELECTOR_TYPE_NVT) ","
           /* OID of the "CPE Inventory" NVT. */
           " '1.3.6.1.4.1.25623.1.0.810002', 'Service detection');");
    }

  if (sql_int ("SELECT count(*) FROM nvt_selectors WHERE name ="
               " '" MANAGE_NVT_SELECTOR_UUID_ALL "'"
               " AND type = " G_STRINGIFY (NVT_SELECTOR_TYPE_NVT)
               " AND family_or_nvt = '1.3.6.1.4.1.25623.1.0.810003';")
      == 0)
    {
      sql ("INSERT into nvt_selectors"
           " (name, exclude, type, family_or_nvt, family)"
           " VALUES ('" MANAGE_NVT_SELECTOR_UUID_ALL "', 1, "
           G_STRINGIFY (NVT_SELECTOR_TYPE_NVT) ","
           /* OID of the "Host Summary" NVT. */
           " '1.3.6.1.4.1.25623.1.0.810003', 'General');");
    }

  if (sql_int ("SELECT count(*) FROM nvt_selectors WHERE name ="
               " '" MANAGE_NVT_SELECTOR_UUID_ALL "'"
               " AND type = " G_STRINGIFY (NVT_SELECTOR_TYPE_FAMILY)
               " AND family_or_nvt = 'Port scanners';")
      == 0)
    {
      sql ("INSERT into nvt_selectors"
           " (name, exclude, type, family_or_nvt, family)"
           " VALUES ('" MANAGE_NVT_SELECTOR_UUID_ALL "', 1, "
           G_STRINGIFY (NVT_SELECTOR_TYPE_FAMILY) ","
           " 'Port scanners', 'Port scanners');");
    }

  if (sql_int ("SELECT count(*) FROM nvt_selectors WHERE name ="
               " '" MANAGE_NVT_SELECTOR_UUID_ALL "'"
               " AND type = " G_STRINGIFY (NVT_SELECTOR_TYPE_NVT)
               " AND family_or_nvt = '1.3.6.1.4.1.25623.1.0.14259';")
      == 0)
    {
      sql ("INSERT into nvt_selectors"
           " (name, exclude, type, family_or_nvt, family)"
           " VALUES ('" MANAGE_NVT_SELECTOR_UUID_ALL "', 0, "
           G_STRINGIFY (NVT_SELECTOR_TYPE_NVT) ","
           /* OID of the "Nmap (NASL wrapper)" NVT. */
           " '1.3.6.1.4.1.25623.1.0.14259', 'Port scanners');");
    }

  if (sql_int ("SELECT count(*) FROM nvt_selectors WHERE name ="
               " '" MANAGE_NVT_SELECTOR_UUID_ALL "'"
               " AND type = " G_STRINGIFY (NVT_SELECTOR_TYPE_NVT)
               " AND family_or_nvt = '1.3.6.1.4.1.25623.1.0.100315';")
      == 0)
    {
      sql ("INSERT into nvt_selectors"
           " (name, exclude, type, family_or_nvt, family)"
           " VALUES ('" MANAGE_NVT_SELECTOR_UUID_ALL "', 0, "
           G_STRINGIFY (NVT_SELECTOR_TYPE_NVT) ","
           /* OID of the "Ping Host" NVT. */
           " '1.3.6.1.4.1.25623.1.0.100315', 'Port scanners');");
    }

  if (sql_int ("SELECT count(*) FROM nvt_selectors WHERE name ="
               " '" MANAGE_NVT_SELECTOR_UUID_ALL "'"
               " AND type = " G_STRINGIFY (NVT_SELECTOR_TYPE_NVT)
               " AND family_or_nvt = '1.3.6.1.4.1.25623.1.0.80109';")
      == 0)
    {
      sql ("INSERT into nvt_selectors"
           " (name, exclude, type, family_or_nvt, family)"
           " VALUES ('" MANAGE_NVT_SELECTOR_UUID_ALL "', 1, "
           G_STRINGIFY (NVT_SELECTOR_TYPE_NVT) ","
           /* OID of the "w3af (NASL wrapper)" NVT. */
           " '1.3.6.1.4.1.25623.1.0.80109', 'Web application abuses');");
    }
  /* Ensure the NVT CVE table is filled. */
  if (sql_int ("SELECT count (*) FROM nvt_cves;") == 0)
    refresh_nvt_cves ();
}

/**
 * @brief Ensure the predefined configs exist.
 */
static void
check_db_configs ()
{
  if (sql_int ("SELECT count(*) FROM configs"
               " WHERE name = 'Full and fast';")
      == 0)
    {
      config_t config;

      sqli (&config,
            "INSERT into configs (uuid, owner, name, nvt_selector, comment,"
            " family_count, nvt_count, nvts_growing, families_growing,"
            " type, creation_time, modification_time)"
            " VALUES ('" CONFIG_UUID_FULL_AND_FAST "', NULL, 'Full and fast',"
            " '" MANAGE_NVT_SELECTOR_UUID_ALL "',"
            " 'Most NVT''s; optimized by using previously collected information.',"
            " %i, %i, 1, 1, 0, m_now (), m_now ())",
            family_nvt_count (NULL) - family_nvt_count ("Port scanners") + 1,
            family_count ());

      /* Setup preferences for the config. */
      setup_full_config_prefs (config, 1, 1, 0);
    }

  if (sql_int ("SELECT count(*) FROM configs"
               " WHERE name = 'Full and fast ultimate';")
      == 0)
    {
      config_t config;

      sqli (&config,
            "INSERT into configs (uuid, owner, name, nvt_selector, comment,"
            " family_count, nvt_count, nvts_growing, families_growing,"
            " type, creation_time, modification_time)"
            " VALUES ('" CONFIG_UUID_FULL_AND_FAST_ULTIMATE "', NULL,"
            " 'Full and fast ultimate', '" MANAGE_NVT_SELECTOR_UUID_ALL "',"
            " 'Most NVT''s including those that can stop services/hosts;"
            " optimized by using previously collected information.',"
            " %i, %i, 1, 1, 0, m_now (), m_now ())",
            family_nvt_count (NULL) - family_nvt_count ("Port scanners") + 1,
            family_count ());

      /* Setup preferences for the config. */
      setup_full_config_prefs (config, 0, 1, 0);
    }

  if (sql_int ("SELECT count(*) FROM configs"
               " WHERE name = 'Full and very deep';")
      == 0)
    {
      config_t config;

      sqli (&config,
            "INSERT into configs (uuid, owner, name, nvt_selector, comment,"
            " family_count, nvt_count, nvts_growing, families_growing,"
            " type, creation_time, modification_time)"
            " VALUES ('" CONFIG_UUID_FULL_AND_VERY_DEEP "', NULL,"
            " 'Full and very deep', '" MANAGE_NVT_SELECTOR_UUID_ALL "',"
            " 'Most NVT''s; don''t trust previously collected information; slow.',"
            " %i, %i, 1, 1, 0, m_now (), m_now ())",
            family_nvt_count (NULL) - family_nvt_count ("Port scanners") + 1,
            family_count ());

      /* Setup preferences for the config. */
      setup_full_config_prefs (config, 1, 0, 1);
    }

  if (sql_int ("SELECT count(*) FROM configs"
               " WHERE name = 'Full and very deep ultimate';")
      == 0)
    {
      config_t config;

      sqli (&config,
            "INSERT into configs (uuid, owner, name, nvt_selector, comment,"
            " family_count, nvt_count, nvts_growing, families_growing,"
            " type, creation_time, modification_time)"
            " VALUES ('" CONFIG_UUID_FULL_AND_VERY_DEEP_ULTIMATE "',"
            " NULL, 'Full and very deep ultimate',"
            " '" MANAGE_NVT_SELECTOR_UUID_ALL "',"
            " 'Most NVT''s including those that can stop services/hosts;"
            " don''t trust previously collected information; slow.',"
            " %i, %i, 1, 1, 0, m_now (), m_now ())",
            family_nvt_count (NULL) - family_nvt_count ("Port scanners") + 1,
            family_count ());

      /* Setup preferences for the config. */
      setup_full_config_prefs (config, 0, 0, 1);
    }

  if (sql_int ("SELECT count(*) FROM configs"
               " WHERE uuid = '" CONFIG_UUID_EMPTY "';")
      == 0)
    {
      config_t config;

      sqli (&config,
            "INSERT into configs (uuid, name, owner, nvt_selector, comment,"
            " family_count, nvt_count, nvts_growing, families_growing,"
            " type, creation_time, modification_time)"
            " VALUES ('" CONFIG_UUID_EMPTY "', 'empty', NULL, 'empty',"
            " 'Empty and static configuration template.',"
            " 0, 0, 0, 0, 0, m_now (), m_now ())");

      /* Setup preferences for the config. */
      setup_full_config_prefs (config, 1, 1, 0);
    }

  if (sql_int ("SELECT count(*) FROM configs"
               " WHERE uuid = '%s';",
               CONFIG_UUID_DISCOVERY)
      == 0)
    make_config_discovery (CONFIG_UUID_DISCOVERY,
                           MANAGE_NVT_SELECTOR_UUID_DISCOVERY);

  /* Service Detection NVTs were all being accidentally removed by the
   * checks below.  Recover them. */

  if (sql_int ("SELECT count (*) FROM nvt_selectors"
               " WHERE name = '" MANAGE_NVT_SELECTOR_UUID_DISCOVERY "'"
               " AND family = 'Service detection'")
      == 0)
    make_config_discovery_service_detection
     (MANAGE_NVT_SELECTOR_UUID_DISCOVERY);

  /* In the Service Detection family, NVTs sometimes move to Product
   * Detection, and once an NVT was removed.  So remove those NVTs
   * from Service Detection in the NVT selector. */

  sql ("DELETE FROM nvt_selectors"
       " WHERE name = '" MANAGE_NVT_SELECTOR_UUID_DISCOVERY "'"
       " AND family = 'Service detection'"
       " AND (SELECT family FROM nvts"
       "      WHERE oid = nvt_selectors.family_or_nvt)"
       "     = 'Product detection';");

  if (sql_int ("SELECT EXISTS (SELECT * FROM nvts);"))
    sql ("DELETE FROM nvt_selectors"
         " WHERE name = '" MANAGE_NVT_SELECTOR_UUID_DISCOVERY "'"
         " AND family = 'Service detection'"
         " AND NOT EXISTS (SELECT * FROM nvts"
         "                 WHERE oid = nvt_selectors.family_or_nvt);");

  check_config_discovery (CONFIG_UUID_DISCOVERY);

  if (sql_int ("SELECT count(*) FROM configs"
               " WHERE uuid = '%s';",
               CONFIG_UUID_HOST_DISCOVERY)
      == 0)
    make_config_host_discovery (CONFIG_UUID_HOST_DISCOVERY,
                                MANAGE_NVT_SELECTOR_UUID_HOST_DISCOVERY);

  check_config_host_discovery (CONFIG_UUID_HOST_DISCOVERY);

  if (sql_int ("SELECT count(*) FROM configs"
               " WHERE uuid = '%s';",
               CONFIG_UUID_SYSTEM_DISCOVERY)
      == 0)
    make_config_system_discovery (CONFIG_UUID_SYSTEM_DISCOVERY,
                                  MANAGE_NVT_SELECTOR_UUID_SYSTEM_DISCOVERY);

  check_config_system_discovery (CONFIG_UUID_SYSTEM_DISCOVERY);
}

/**
 * @brief Ensure every report format has a unique UUID.
 *
 * @return 0 success, -1 error.
 */
static int
make_report_format_uuids_unique ()
{
  iterator_t rows;

  sql ("CREATE TEMPORARY TABLE duplicates"
       " AS SELECT id, uuid, make_uuid () AS new_uuid, owner,"
       "           (SELECT uuid FROM users"
       "            WHERE users.id = outer_report_formats.owner)"
       "           AS owner_uuid,"
       "           (SELECT owner from report_formats"
       "                              WHERE uuid = outer_report_formats.uuid"
       "                              ORDER BY id ASC LIMIT 1)"
       "           AS original_owner,"
       "           (SELECT uuid FROM users"
       "            WHERE users.id = (SELECT owner from report_formats"
       "                              WHERE uuid = outer_report_formats.uuid"
       "                              ORDER BY id ASC LIMIT 1))"
       "           AS original_owner_uuid"
       "    FROM report_formats AS outer_report_formats"
       "    WHERE id > (SELECT id from report_formats"
       "                WHERE uuid = outer_report_formats.uuid"
       "                ORDER BY id ASC LIMIT 1);");

  sql ("UPDATE alert_method_data"
       " SET data = (SELECT new_uuid FROM duplicates"
       "             WHERE duplicates.id = alert_method_data.alert)"
       " WHERE alert IN (SELECT id FROM duplicates);");

  /* Update UUIDs on disk. */
  init_iterator (&rows,
                 "SELECT id, uuid, new_uuid, owner, owner_uuid, original_owner,"
                 "       original_owner_uuid"
                 " FROM duplicates;");
  while (next (&rows))
    {
      gchar *dir, *new_dir;
      const char *old_uuid, *new_uuid;
      int copy;

      old_uuid = iterator_string (&rows, 1);
      new_uuid = iterator_string (&rows, 2);

      if (iterator_int64 (&rows, 3) == 0)
        {
          /* Old-style "global" report format.  I don't think this is possible
           * with any released version, so ignore. */
          continue;
        }
      else if (iterator_int64 (&rows, 5) == 0)
        {
          const char *owner_uuid;
          /* Dedicated subdir in user dir, but must be renamed. */
          copy = 0;
          owner_uuid = iterator_string (&rows, 4);
          dir = g_build_filename (OPENVAS_STATE_DIR,
                                  "openvasmd",
                                  "report_formats",
                                  owner_uuid,
                                  old_uuid,
                                  NULL);
          new_dir = g_build_filename (OPENVAS_STATE_DIR,
                                      "openvasmd",
                                      "report_formats",
                                      owner_uuid,
                                      new_uuid,
                                      NULL);
        }
      else
        {
          const char *owner_uuid, *original_owner_uuid;

          /* Two user-owned report formats, may be the same user. */

          owner_uuid = iterator_string (&rows, 4);
          original_owner_uuid = iterator_string (&rows, 6);

          /* Copy the subdir if both report formats owned by one user. */
          copy = owner_uuid
                 && original_owner_uuid
                 && (strcmp (owner_uuid, original_owner_uuid) == 0);

          dir = g_build_filename (OPENVAS_STATE_DIR,
                                  "openvasmd",
                                  "report_formats",
                                  owner_uuid,
                                  old_uuid,
                                  NULL);
          new_dir = g_build_filename (OPENVAS_STATE_DIR,
                                      "openvasmd",
                                      "report_formats",
                                      owner_uuid,
                                      new_uuid,
                                      NULL);
        }

      if (copy)
        {
          gchar *command;
          int ret;

          command = g_strdup_printf ("cp -a %s %s > /dev/null 2>&1",
                                     dir,
                                     new_dir);
          g_debug ("   command: %s\n", command);
          ret = system (command);
          g_free (command);

          if (ret == -1 || WEXITSTATUS (ret))
            {
              /* Presume dir missing, just log a warning. */
              g_warning ("%s: cp %s to %s failed",
                         __FUNCTION__, dir, new_dir);
            }
          else
            g_debug ("%s: copied %s to %s", __FUNCTION__, dir, new_dir);
        }
      else
        {
          if (rename (dir, new_dir))
            {
              g_warning ("%s: rename %s to %s: %s",
                         __FUNCTION__, dir, new_dir, strerror (errno));
              if (errno != ENOENT)
                {
                  g_free (dir);
                  g_free (new_dir);
                  sql_rollback ();
                  return -1;
                }
            }
          else
            g_debug ("%s: moved %s to %s", __FUNCTION__, dir, new_dir);
        }
      g_free (dir);
      g_free (new_dir);
    }
  cleanup_iterator (&rows);

  sql ("UPDATE report_formats"
       " SET uuid = (SELECT new_uuid FROM duplicates"
       "             WHERE duplicates.id = report_formats.id)"
       " WHERE id IN (SELECT id FROM duplicates);");

  if (sql_changes () > 0)
    g_debug ("%s: gave %d report format(s) new UUID(s) to keep UUIDs unique.",
             __FUNCTION__, sql_changes ());

  sql ("DROP TABLE duplicates;");

  return 0;
}

/**
 * @brief Check that trash report formats are correct.
 *
 * @return 0 success, -1 error.
 */
static int
check_db_trash_report_formats ()
{
  gchar *dir;
  struct stat state;

  dir = g_build_filename (OPENVAS_STATE_DIR,
                          "openvasmd",
                          "report_formats_trash",
                          NULL);

  if (g_lstat (dir, &state))
    {
      iterator_t report_formats;
      int count;

      if (errno != ENOENT)
        {
          g_warning ("%s: g_lstat (%s) failed: %s\n",
                     __FUNCTION__, dir, g_strerror (errno));
          g_free (dir);
          return -1;
        }

      /* Remove all trash report formats. */

      count = 0;
      init_iterator (&report_formats, "SELECT id FROM report_formats_trash;");
      while (next (&report_formats))
        {
          report_format_t report_format;

          report_format = iterator_int64 (&report_formats, 0);

          sql ("DELETE FROM alert_method_data_trash"
               " WHERE data = (SELECT original_uuid"
               "               FROM report_formats_trash"
               "               WHERE id = %llu)"
               " AND (name = 'notice_attach_format'"
               "      OR name = 'notice_report_format');",
               report_format);

          permissions_set_orphans ("report_format", report_format,
                                   LOCATION_TRASH);
          tags_set_orphans ("report_format", report_format, LOCATION_TRASH);

          sql ("DELETE FROM report_format_param_options_trash"
               " WHERE report_format_param"
               " IN (SELECT id from report_format_params_trash"
               "     WHERE report_format = %llu);",
               report_format);
          sql ("DELETE FROM report_format_params_trash WHERE report_format = %llu;",
               report_format);
          sql ("DELETE FROM report_formats_trash WHERE id = %llu;",
               report_format);

          count++;
        }
      cleanup_iterator (&report_formats);

      if (count)
        g_message ("Trash report format directory was missing."
                   " Removed all %i trash report formats.",
                   count);
    }

  g_free (dir);
  return 0;
}

/**
 * @brief Ensure the predefined report formats exist.
 *
 * @return 0 success, -1 error.
 */
static int
check_db_report_formats ()
{
  GError *error;
  GDir *dir;
  gchar *path;
  const gchar *report_format_path;
  iterator_t report_formats;

  if (check_db_trash_report_formats ())
    return -1;

  /* Bring report format UUIDs in database up to date. */
  update_report_format_uuids ();
  if (make_report_format_uuids_unique ())
    return -1;

  /* Open the global report format dir. */

  path = predefined_report_format_dir (NULL);

  dir = g_dir_open (path, 0, &error);
  if (dir == NULL)
    {
      g_warning ("%s: Failed to open directory '%s': %s",
                 __FUNCTION__, path, error->message);
      g_error_free (error);
      g_free (path);
      return -1;
    }
  g_free (path);

  /* Remember existing global report formats. */

  sql ("CREATE TEMPORARY TABLE report_formats_check"
       " AS SELECT id, uuid, name, owner, summary, description, extension,"
       "           content_type, signature, trust, trust_time, flags,"
       "           creation_time, modification_time"
       "    FROM report_formats"
       "    WHERE owner IS NULL;");

  sql ("CREATE TEMPORARY TABLE report_format_params_check"
       " AS SELECT id, name, report_format, type, value, type_min, type_max,"
       "           type_regex, fallback"
       "    FROM report_format_params"
       "    WHERE report_format IN (SELECT id FROM report_formats"
       "                            WHERE owner IS NULL);");

  /* Create or update global report formats from disk. */

  while ((report_format_path = g_dir_read_name (dir)))
    check_report_format (report_format_path);

  /* Remove previous global report formats that were not defined. */

  init_iterator (&report_formats,
                 "SELECT id, uuid, name FROM report_formats"
                 " WHERE uuid IN (SELECT uuid FROM report_formats_check)"
                 " AND (EXISTS (SELECT * FROM alert_method_data_trash"
                 "              WHERE data = report_formats.uuid"
                 "              AND (name = 'notice_attach_format'"
                 "                   OR name = 'notice_report_format'))"
                 "      OR EXISTS (SELECT * FROM alert_method_data"
                 "                 WHERE data = report_formats.uuid"
                 "                 AND (name = 'notice_attach_format'"
                 "                      OR name = 'notice_report_format')));");
  while (next (&report_formats))
    g_warning
     ("Removing old report format %s (%s) which is in use by an alert.\n"
      "Alert will fallback to TXT report format (%s), if TXT exists.",
      iterator_string (&report_formats, 2),
      iterator_string (&report_formats, 1),
      "a3810a62-1f62-11e1-9219-406186ea4fc5");
  cleanup_iterator (&report_formats);

  sql ("DELETE FROM report_format_param_options"
       " WHERE report_format_param"
       "       IN (SELECT id FROM report_format_params"
       "           WHERE report_format"
       "                 IN (SELECT id FROM report_formats"
       "                     WHERE uuid IN (SELECT uuid"
       "                                    FROM report_formats_check)));");

  sql ("DELETE FROM report_format_params"
       " WHERE report_format IN (SELECT id FROM report_formats"
       "                         WHERE uuid IN (SELECT uuid"
       "                                        FROM report_formats_check));");

  sql ("DELETE FROM resources_predefined"
       " WHERE resource_type = 'report_format'"
       " AND resource IN (SELECT id FROM report_formats_check);");

  sql ("DELETE FROM report_formats"
       " WHERE uuid IN (SELECT uuid FROM report_formats_check);");

  /* Forget the old global report formats. */

  sql ("DROP TABLE report_format_params_check;");
  sql ("DROP TABLE report_formats_check;");

  return 0;
}

/**
 * @brief Get trash directory of a report format.
 *
 * @param[in]  report_format_id  UUID of report format.  NULL for the
 *             base dir that holds the report format trash.
 *
 * @return Freshly allocated trash dir.
 */
static gchar *
report_format_trash_dir (const gchar *report_format_id)
{
  if (report_format_id)
    return g_build_filename (OPENVAS_STATE_DIR,
                             "openvasmd",
                             "report_formats_trash",
                             report_format_id,
                             NULL);

  return g_build_filename (OPENVAS_STATE_DIR,
                           "openvasmd",
                           "report_formats_trash",
                           NULL);
}

/**
 * @brief Ensure that the report formats trash directory matches the database.
 *
 * @return -1 if error, 0 if success.
 */
static int
check_db_report_formats_trash ()
{
  gchar *dir;
  GError *error;
  GDir *directory;
  const gchar *entry;

  dir = report_format_trash_dir (NULL);
  error = NULL;
  directory = g_dir_open (dir, 0, &error);

  if (directory == NULL)
    {
      assert (error);
      if (!g_error_matches (error, G_FILE_ERROR, G_FILE_ERROR_NOENT))
        {
          g_warning ("g_dir_open (%s) failed - %s\n", dir, error->message);
          g_error_free (error);
          g_free (dir);
          return -1;
        }
    }
  else
    {
      entry = NULL;
      while ((entry = g_dir_read_name (directory)) != NULL)
        {
          gchar *end;
          if (strtol (entry, &end, 10) < 0)
            /* Only interested in positive numbers. */
            continue;
          if (*end != '\0')
            /* Only interested in numbers. */
            continue;

          /* Check whether the db has a report format with this ID. */
          if (sql_int ("SELECT count(*) FROM report_formats_trash"
                       " WHERE id = %s;",
                       entry)
              == 0)
            {
              int ret;
              gchar *entry_path;

              /* Remove the directory. */

              entry_path = g_build_filename (dir, entry, NULL);
              ret = openvas_file_remove_recurse (entry_path);
              g_free (entry_path);
              if (ret)
                {
                  g_warning ("%s: failed to remove %s from %s",
                             __FUNCTION__, entry, dir);
                  g_dir_close (directory);
                  g_free (dir);
                  return -1;
                }
            }
        }
      g_dir_close (directory);
    }
  g_free (dir);
  return 0;
}

/**
 * @brief Ensure the predefined permissions exists.
 */
static void
check_db_permissions ()
{
  if (sql_int ("SELECT count(*) FROM permissions"
               " WHERE uuid = '" PERMISSION_UUID_ADMIN_EVERYTHING "';")
      == 0)
    sql ("INSERT INTO permissions"
         " (uuid, owner, name, comment, resource_type, resource, resource_uuid,"
         "  resource_location, subject_type, subject, subject_location,"
         "  creation_time, modification_time)"
         " VALUES"
         " ('" PERMISSION_UUID_ADMIN_EVERYTHING "', NULL, 'Everything', '', '',"
         "  0, '', " G_STRINGIFY (LOCATION_TABLE) ", 'role',"
         "  (SELECT id FROM roles WHERE uuid = '" ROLE_UUID_ADMIN "'),"
         "  " G_STRINGIFY (LOCATION_TABLE) ", m_now (), m_now ());");

  if (sql_int ("SELECT count(*) FROM permissions"
               " WHERE uuid = '" PERMISSION_UUID_SUPER_ADMIN_EVERYTHING "';")
      == 0)
    {
      sql ("INSERT INTO permissions"
           " (uuid, owner, name, comment, resource_type, resource, resource_uuid,"
           "  resource_location, subject_type, subject, subject_location,"
           "  creation_time, modification_time)"
           " VALUES"
           " ('" PERMISSION_UUID_SUPER_ADMIN_EVERYTHING "', NULL, 'Everything',"
           "  '', '', 0, '', " G_STRINGIFY (LOCATION_TABLE) ", 'role',"
           "  (SELECT id FROM roles WHERE uuid = '" ROLE_UUID_SUPER_ADMIN "'),"
           "  " G_STRINGIFY (LOCATION_TABLE) ", m_now (), m_now ());");
      sql ("INSERT INTO permissions"
           " (uuid, owner, name, comment, resource_type, resource, resource_uuid,"
           "  resource_location, subject_type, subject, subject_location,"
           "  creation_time, modification_time)"
           " VALUES"
           " (make_uuid (), NULL, 'Super',"
           "  '', '', 0, '', " G_STRINGIFY (LOCATION_TABLE) ", 'role',"
           "  (SELECT id FROM roles WHERE uuid = '" ROLE_UUID_SUPER_ADMIN "'),"
           "  " G_STRINGIFY (LOCATION_TABLE) ", m_now (), m_now ());");
    }

  if (sql_int ("SELECT count(*) FROM permissions"
               " WHERE subject_type = 'role'"
               " AND subject = (SELECT id FROM roles"
               "                WHERE uuid = '" ROLE_UUID_GUEST "')"
               " AND resource = 0;")
      <= 1)
    {
      /* Clean-up any remaining permissions. */
      sql ("DELETE FROM permissions WHERE subject_type = 'role'"
           " AND subject = (SELECT id FROM roles"
           "                WHERE uuid = '" ROLE_UUID_GUEST "');");
      add_role_permission (ROLE_UUID_GUEST, "AUTHENTICATE");
      add_role_permission (ROLE_UUID_GUEST, "COMMANDS");
      add_role_permission (ROLE_UUID_GUEST, "HELP");
      add_role_permission (ROLE_UUID_GUEST, "GET_AGGREGATES");
      add_role_permission (ROLE_UUID_GUEST, "GET_FILTERS");
      add_role_permission (ROLE_UUID_GUEST, "GET_INFO");
      add_role_permission (ROLE_UUID_GUEST, "GET_NVTS");
      add_role_permission (ROLE_UUID_GUEST, "GET_SETTINGS");
    }

  if (sql_int ("SELECT count(*) FROM permissions"
               " WHERE subject_type = 'role'"
               " AND subject = (SELECT id FROM roles"
               "                WHERE uuid = '" ROLE_UUID_INFO "')"
               " AND resource = 0;")
      <= 1)
    {
      /* Clean-up any remaining permissions. */
      sql ("DELETE FROM permissions WHERE subject_type = 'role'"
           " AND subject = (SELECT id FROM roles"
           "                WHERE uuid = '" ROLE_UUID_INFO "');");
      add_role_permission (ROLE_UUID_INFO, "AUTHENTICATE");
      add_role_permission (ROLE_UUID_INFO, "COMMANDS");
      add_role_permission (ROLE_UUID_INFO, "HELP");
      add_role_permission (ROLE_UUID_INFO, "GET_AGGREGATES");
      add_role_permission (ROLE_UUID_INFO, "GET_INFO");
      add_role_permission (ROLE_UUID_INFO, "GET_NVTS");
      add_role_permission (ROLE_UUID_INFO, "GET_SETTINGS");
      add_role_permission (ROLE_UUID_INFO, "MODIFY_SETTING");
    }

  if (sql_int ("SELECT count(*) FROM permissions"
               " WHERE subject_type = 'role'"
               " AND subject = (SELECT id FROM roles"
               "                WHERE uuid = '" ROLE_UUID_MONITOR "')"
               " AND resource = 0;")
      <= 1)
    {
      /* Clean-up any remaining permissions. */
      sql ("DELETE FROM permissions WHERE subject_type = 'role'"
           " AND subject = (SELECT id FROM roles"
           "                WHERE uuid = '" ROLE_UUID_MONITOR "');");
      add_role_permission (ROLE_UUID_MONITOR, "AUTHENTICATE");
      add_role_permission (ROLE_UUID_MONITOR, "COMMANDS");
      add_role_permission (ROLE_UUID_MONITOR, "GET_SETTINGS");
      add_role_permission (ROLE_UUID_MONITOR, "GET_SYSTEM_REPORTS");
      add_role_permission (ROLE_UUID_MONITOR, "HELP");
    }

  if (sql_int ("SELECT count(*) FROM permissions"
               " WHERE subject_type = 'role'"
               " AND subject = (SELECT id FROM roles"
               "                WHERE uuid = '" ROLE_UUID_USER "')"
               " AND resource = 0;")
      <= 1)
    {
      command_t *command;
      command = omp_commands;

      /* Clean-up any remaining permissions. */
      sql ("DELETE FROM permissions WHERE subject_type = 'role'"
           " AND subject = (SELECT id FROM roles"
           "                WHERE uuid = '" ROLE_UUID_USER "');");
      while (command[0].name)
        {
          if (strstr (command[0].name, "DESCRIBE_AUTH") == NULL
              && strcmp (command[0].name, "GET_VERSION")
              && strstr (command[0].name, "GROUP") == NULL
              && strstr (command[0].name, "ROLE") == NULL
              && strstr (command[0].name, "SYNC") == NULL
              && strstr (command[0].name, "USER") == NULL)
            add_role_permission (ROLE_UUID_USER, command[0].name);
          command++;
        }
    }

  if (sql_int ("SELECT count(*) FROM permissions"
               " WHERE subject_type = 'role'"
               " AND subject = (SELECT id FROM roles"
               "                WHERE uuid = '" ROLE_UUID_OBSERVER "')"
               " AND resource = 0;")
      <= 1)
    {
      command_t *command;
      command = omp_commands;
      /* Clean-up any remaining permissions. */
      sql ("DELETE FROM permissions WHERE subject_type = 'role'"
           " AND subject = (SELECT id FROM roles"
           "                WHERE uuid = '" ROLE_UUID_OBSERVER "');");
      while (command[0].name)
        {
          if ((strstr (command[0].name, "GET") == command[0].name)
              && strcmp (command[0].name, "GET_GROUPS")
              && strcmp (command[0].name, "GET_ROLES")
              && strcmp (command[0].name, "GET_USERS")
              && strcmp (command[0].name, "GET_VERSION"))
            add_role_permission (ROLE_UUID_OBSERVER, command[0].name);
          command++;
        }
      add_role_permission (ROLE_UUID_OBSERVER, "AUTHENTICATE");
      add_role_permission (ROLE_UUID_OBSERVER, "HELP");
      add_role_permission (ROLE_UUID_OBSERVER, "MODIFY_SETTING");
    }

  /* Migration from name LSC_CREDENTIAL to CREDENTIAL missed a case.  In
   * trunk this is done in migrate_185_to_186. */

  sql ("UPDATE permissions SET resource_type = 'credential'"
       " WHERE resource_type = 'lsc_credential';");

  sql ("UPDATE permissions_trash SET resource_type = 'credential'"
       " WHERE resource_type = 'lsc_credential';");
}

/**
 * @brief Ensure the predefined roles exists.
 */
static void
check_db_roles ()
{
  if (sql_int ("SELECT count(*) FROM roles WHERE uuid = '" ROLE_UUID_ADMIN "';")
      == 0)
    sql ("INSERT INTO roles"
         " (uuid, owner, name, comment, creation_time, modification_time)"
         " VALUES"
         " ('" ROLE_UUID_ADMIN "', NULL, 'Admin',"
         "  'Administrator.  Full privileges.',"
         " m_now (), m_now ());");

  if (sql_int ("SELECT count(*) FROM roles WHERE uuid = '" ROLE_UUID_GUEST "';")
      == 0)
    sql ("INSERT INTO roles"
         " (uuid, owner, name, comment, creation_time, modification_time)"
         " VALUES"
         " ('" ROLE_UUID_GUEST "', NULL, 'Guest',"
         "  'Guest.',"
         " m_now (), m_now ());");

  if (sql_int ("SELECT count(*) FROM roles WHERE uuid = '" ROLE_UUID_INFO "';")
      == 0)
    sql ("INSERT INTO roles"
         " (uuid, owner, name, comment, creation_time, modification_time)"
         " VALUES"
         " ('" ROLE_UUID_INFO "', NULL, 'Info',"
         "  'Information browser.',"
         " m_now (), m_now ());");

  if (sql_int ("SELECT count(*) FROM roles WHERE uuid = '" ROLE_UUID_MONITOR "';")
      == 0)
    sql ("INSERT INTO roles"
         " (uuid, owner, name, comment, creation_time, modification_time)"
         " VALUES"
         " ('" ROLE_UUID_MONITOR "', NULL, 'Monitor',"
         "  'Performance monitor.',"
         " m_now (), m_now ());");

  if (sql_int ("SELECT count(*) FROM roles WHERE uuid = '" ROLE_UUID_USER "';")
      == 0)
    sql ("INSERT INTO roles"
         " (uuid, owner, name, comment, creation_time, modification_time)"
         " VALUES"
         " ('" ROLE_UUID_USER "', NULL, 'User',"
         "  'Standard user.',"
         " m_now (), m_now ());");

  if (sql_int ("SELECT count(*) FROM roles WHERE uuid = '" ROLE_UUID_SUPER_ADMIN "';")
      == 0)
    sql ("INSERT INTO roles"
         " (uuid, owner, name, comment, creation_time, modification_time)"
         " VALUES"
         " ('" ROLE_UUID_SUPER_ADMIN "', NULL, 'Super Admin',"
         "  'Super administrator.  Full privileges with access to all users.',"
         " m_now (), m_now ());");

  if (sql_int ("SELECT count(*) FROM roles"
               " WHERE uuid = '" ROLE_UUID_OBSERVER "';")
      == 0)
    sql ("INSERT INTO roles"
         " (uuid, owner, name, comment, creation_time, modification_time)"
         " VALUES"
         " ('" ROLE_UUID_OBSERVER "', NULL, 'Observer',"
         "  'Observer.',"
         " m_now (), m_now ());");
}

/**
 * @brief Cleanup the auth_cache table.
 */
static void
clean_auth_cache ()
{
  sql ("DELETE FROM auth_cache;");
}

/**
 * @brief Ensure that the database is in order.
 *
 * @param[in]  check_encryption_key  Whether to check encryption key.
 *
 * @return 0 success, -1 error.
 */
static int
check_db (int check_encryption_key)
{
  sql_begin_exclusive ();
  create_tables ();
  check_db_sequences ();
  if (progress)
    progress ();
  set_db_version (OPENVASMD_DATABASE_VERSION);
  check_db_nvts ();
  check_db_configs ();
  check_db_port_lists ();
  clean_auth_cache ();
  if (check_db_scanners ())
    goto fail;
  if (check_db_report_formats ())
    goto fail;
  if (check_db_report_formats_trash ())
    goto fail;
  check_db_roles ();
  check_db_permissions ();
  check_db_settings ();
  cleanup_schedule_times ();
  if (check_encryption_key && check_db_encryption_key ())
    goto fail;
  if (progress)
    progress ();

  sql_commit ();
  return 0;

 fail:
  sql_rollback ();
  return -1;
}

/**
 * @brief Ensure the generate scripts are all executable.
 *
 * Used by a migrator.
 */
void
check_generate_scripts ()
{
  iterator_t rows;

  init_iterator (&rows, "SELECT owner,"
                        "       uuid,"
                        "       (SELECT uuid FROM users"
                        "        WHERE users.id = report_formats.owner)"
                        " FROM report_formats;");
  while (next (&rows))
    {
      resource_t owner;

      owner = iterator_int64 (&rows, 0);
      if (owner)
        {
          const gchar *report_format_uuid, *user_uuid;
          gchar *path;

          report_format_uuid = iterator_string (&rows, 1);
          if (report_format_uuid == NULL)
            continue;

          user_uuid = iterator_string (&rows, 2);
          if (user_uuid == NULL)
            continue;

          path = g_build_filename (OPENVAS_STATE_DIR,
                                   "openvasmd",
                                   "report_formats",
                                   user_uuid,
                                   report_format_uuid,
                                   "generate",
                                   NULL);

          if (chmod (path, 0755 /* rwxr-xr-x */))
            g_warning ("%s: chmod %s failed: %s\n",
                       __FUNCTION__,
                       path,
                       strerror (errno));

          g_free (path);
        }
    }
  cleanup_iterator (&rows);
}

/**
 * @brief Stop any active tasks.
 */
void
stop_active_tasks ()
{
  iterator_t tasks;
  get_data_t get;

  /* Set requested and running tasks to stopped. */

  assert (current_credentials.uuid == NULL);
  memset (&get, '\0', sizeof (get));
  init_task_iterator (&tasks, &get);
  while (next (&tasks))
    {
      switch (task_iterator_run_status (&tasks))
        {
          case TASK_STATUS_DELETE_REQUESTED:
          case TASK_STATUS_DELETE_ULTIMATE_REQUESTED:
          case TASK_STATUS_DELETE_ULTIMATE_WAITING:
          case TASK_STATUS_DELETE_WAITING:
          case TASK_STATUS_REQUESTED:
          case TASK_STATUS_RUNNING:
          case TASK_STATUS_STOP_REQUESTED_GIVEUP:
          case TASK_STATUS_STOP_REQUESTED:
          case TASK_STATUS_STOP_WAITING:
            {
              task_t index = get_iterator_resource (&tasks);
              /* Set the current user, for event checks. */
              current_credentials.uuid = task_owner_uuid (index);
              set_task_run_status (index, TASK_STATUS_STOPPED);
              free (current_credentials.uuid);
              break;
            }
          default:
            break;
        }
    }
  cleanup_iterator (&tasks);
  current_credentials.uuid = NULL;

  /* Set requested and running reports to stopped. */

  sql ("UPDATE reports SET scan_run_status = %u"
       " WHERE scan_run_status = %u"
       " OR scan_run_status = %u"
       " OR scan_run_status = %u"
       " OR scan_run_status = %u"
       " OR scan_run_status = %u"
       " OR scan_run_status = %u"
       " OR scan_run_status = %u"
       " OR scan_run_status = %u"
       " OR scan_run_status = %u;",
       TASK_STATUS_STOPPED,
       TASK_STATUS_DELETE_REQUESTED,
       TASK_STATUS_DELETE_ULTIMATE_REQUESTED,
       TASK_STATUS_DELETE_ULTIMATE_WAITING,
       TASK_STATUS_DELETE_WAITING,
       TASK_STATUS_REQUESTED,
       TASK_STATUS_RUNNING,
       TASK_STATUS_STOP_REQUESTED,
       TASK_STATUS_STOP_REQUESTED_GIVEUP,
       TASK_STATUS_STOP_WAITING);
}

/**
 * @brief Clean up database tables.
 *
 * Remove superfluous entries from tables.
 */
void
cleanup_tables ()
{
  /* Remove group and role assignments of deleted users.
   *
   * This should be a migrator, but this way is easier to backport.  */

  sql ("DELETE FROM group_users"
       " WHERE \"user\" NOT IN (SELECT id FROM users);");
  sql ("DELETE FROM group_users_trash"
       " WHERE \"user\" NOT IN (SELECT id FROM users);");
  sql ("DELETE FROM role_users"
       " WHERE \"user\" NOT IN (SELECT id FROM users);");
  sql ("DELETE FROM role_users_trash"
       " WHERE \"user\" NOT IN (SELECT id FROM users);");

  /*
   * Remove permissions of deleted users, groups and roles.
   */
  sql ("DELETE FROM permissions"
       " WHERE (subject_type = 'user'"
       "        AND subject NOT IN (SELECT id FROM users))"
       "    OR (subject_type = 'group'"
       "        AND subject_location = " G_STRINGIFY (LOCATION_TABLE)
       "        AND subject NOT IN (SELECT id FROM groups))"
       "    OR (subject_type = 'group'"
       "        AND subject_location = " G_STRINGIFY (LOCATION_TRASH)
       "        AND subject NOT IN (SELECT id FROM groups_trash))"
       "    OR (subject_type = 'role'"
       "        AND subject_location = " G_STRINGIFY (LOCATION_TABLE)
       "        AND subject NOT IN (SELECT id FROM roles))"
       "    OR (subject_type = 'role'"
       "        AND subject_location = " G_STRINGIFY (LOCATION_TRASH)
       "        AND subject NOT IN (SELECT id FROM roles_trash));");

  sql ("DELETE FROM permissions_trash"
       " WHERE (subject_type = 'user'"
       "        AND subject NOT IN (SELECT id FROM users))"
       "    OR (subject_type = 'group'"
       "        AND subject_location = " G_STRINGIFY (LOCATION_TABLE)
       "        AND subject NOT IN (SELECT id FROM groups))"
       "    OR (subject_type = 'group'"
       "        AND subject_location = " G_STRINGIFY (LOCATION_TRASH)
       "        AND subject NOT IN (SELECT id FROM groups_trash))"
       "    OR (subject_type = 'role'"
       "        AND subject_location = " G_STRINGIFY (LOCATION_TABLE)
       "        AND subject NOT IN (SELECT id FROM roles))"
       "    OR (subject_type = 'role'"
       "        AND subject_location = " G_STRINGIFY (LOCATION_TRASH)
       "        AND subject NOT IN (SELECT id FROM roles_trash));");
}

/**
 * @brief Initialize the manage library.
 *
 * Check DB version, do startup database checks, load the NVT cache.
 * Optionally also stop active tasks.
 *
 * @param[in]  log_config      Log configuration.
 * @param[in]  nvt_cache_mode  True when running in NVT caching mode.
 * @param[in]  database        Location of database.
 * @param[in]  max_ips_per_target  Max number of IPs per target.
 * @param[in]  max_email_attachment_size  Max size of email attachments.
 * @param[in]  max_email_include_size     Max size of email inclusions.
 * @param[in]  max_email_message_size     Max size of email user message text.
 * @param[in]  update_progress     Function to update progress, or NULL. *
 * @param[in]  stop_tasks          Stop any active tasks.
 * @param[in]  fork_connection     Function to fork a connection that will
 *                                 accept OMP requests.  Used to start tasks
 *                                 with OMP when an alert occurs.
 * @param[in]  skip_db_check       Skip DB check.
 * @param[in]  check_encryption_key  Check encryption key if doing DB check.
 *
 * @return 0 success, -1 error, -2 database is wrong version, -3 database needs
 *         to be initialised from server, -4 max_ips_per_target out of range.
 */
static int
init_manage_internal (GSList *log_config,
                      int nvt_cache_mode,
                      const gchar *database,
                      int max_ips_per_target,
                      int max_email_attachment_size,
                      int max_email_include_size,
                      int max_email_message_size,
                      void (*update_progress) (),
                      int stop_tasks,
                      int (*fork_connection)
                             (openvas_connection_t *, gchar *),
                      int skip_db_check,
                      int check_encryption_key)
{
  int ret;

  /* Summary of init cases:
   *
   *     daemon [--foreground]
   *         init_ompd  cache 0
   *             init_manage
   *         serve_and_schedule
   *             forks child (serve_omp)
   *                 init_ompd_process
   *                     init_manage_process
   *                 ...
   *                 event
   *                   fork_connection_for_event
   *                       fork one
   *                           init_ompd_process
   *                               init_manage_process
   *                           serve_client
   *                       fork two
   *                           omp_auth, omp_start_task_report.
   *                 ...
   *             manage_schedule
   *                 fork_connection_for_scheduler
   *                     fork one
   *                         init_ompd_process
   *                             init_manage_process
   *                         serve_client
   *                     fork two
   *                         omp_auth, omp_start_task_report, omp_resume_task_report.
   *     --rebuild --update
   *         rebuild_nvt_cache_retry
   *             forks update_or_rebuild_nvt_cache
   *                 init_ompd  cache -1 or -2
   *                     init_manage
   *                 serve_omp
   *                     init_ompd_process
   *     --create-user --delete-user --get-users
   *         manage_create, ...
   *             init_manage_helper
   *     --encrypt/decrypt-all-credentials
   *         manage_encrypt_...
   *             init_manage_helper
   *     --backup-database
   *         (no init because no db access required)
   *     --migrate
   *         manage_migrate
   *             init_manage_process (sorts out db state itself) */

  if ((max_ips_per_target <= 0)
      || (max_ips_per_target > MANAGE_ABSOLUTE_MAX_IPS_PER_TARGET))
    return -4;

  max_hosts = max_ips_per_target;
  if (max_email_attachment_size)
    max_attach_length = max_email_attachment_size;
  if (max_email_include_size)
    max_content_length = max_email_include_size;
  if (max_email_message_size)
    max_email_message_length = max_email_message_size;
  progress = update_progress;

  g_log_set_handler (G_LOG_DOMAIN,
                     ALL_LOG_LEVELS,
                     (GLogFunc) openvas_log_func,
                     log_config);

  memset (&current_credentials, '\0', sizeof (current_credentials));

  init_manage_process (0, database);

  /* Check that the versions of the databases are correct. */

  ret = check_db_versions (nvt_cache_mode);
  if (ret)
    return ret;

  /* Ensure the database is complete, removing superfluous rows.
   *
   * Assume that all other running processes are from the same Manager version,
   * because some of these checks will modify the database if it is out of
   * date.  This is relevant because the caller may be a command option process
   * like a --create-user process.  */

  if (skip_db_check == 0)
    {
      ret = check_db (check_encryption_key);
      if (ret)
        return ret;

      cleanup_tables ();

      /* Set max_hosts in db, so database server side can access it. */

      sql ("DELETE FROM meta WHERE name = 'max_hosts';");
      sql ("INSERT INTO meta (name, value) VALUES ('max_hosts', %i);", max_hosts);
    }

  if (stop_tasks && (nvt_cache_mode == 0))
    /* Stop any active tasks. */
    stop_active_tasks ();

  /* Load the NVT cache into memory. */

  if (nvti_cache == NULL)
    update_nvti_cache ();

  sql_close ();
  task_db_name = g_strdup (database);
  if (fork_connection)
    manage_fork_connection = fork_connection;
  return 0;
}

/**
 * @brief Initialize the manage library.
 *
 * Check DB version, do startup database checks, load the NVT cache.
 *
 * Ensure all tasks are in a clean initial state.
 *
 * Beware that calling this function while tasks are running may lead to
 * problems.
 *
 * @param[in]  log_config      Log configuration.
 * @param[in]  nvt_cache_mode  True when running in NVT caching mode.
 * @param[in]  database        Location of database.
 * @param[in]  max_ips_per_target  Max number of IPs per target.
 * @param[in]  max_email_attachment_size  Max size of email attachments.
 * @param[in]  max_email_include_size     Max size of email inclusions.
 * @param[in]  max_email_message_size     Max size of email user message text.
 * @param[in]  update_progress     Function to update progress, or NULL. *
 * @param[in]  fork_connection     Function to fork a connection that will
 *                                 accept OMP requests.  Used to start tasks
 *                                 with OMP when an alert occurs.
 * @param[in]  skip_db_check       Skip DB check.
 *
 * @return 0 success, -1 error, -2 database is wrong version, -3 database needs
 *         to be initialised from server, -4 max_ips_per_target out of range.
 */
int
init_manage (GSList *log_config, int nvt_cache_mode, const gchar *database,
             int max_ips_per_target, int max_email_attachment_size,
             int max_email_include_size, int max_email_message_size,
             void (*update_progress) (),
             int (*fork_connection) (openvas_connection_t*, gchar*),
             int skip_db_check)
{
  return init_manage_internal (log_config,
                               nvt_cache_mode,
                               database,
                               max_ips_per_target,
                               max_email_attachment_size,
                               max_email_include_size,
                               max_email_message_size,
                               update_progress,
                               1,  /* Stop active tasks. */
                               fork_connection,
                               skip_db_check,
                               1); /* Check encryption key if checking db. */
}

/**
 * @brief Initialize the manage library for a helper program.
 *
 * This should be called at the beginning of any program that accesses the
 * database.  Forked processes should call init_manage_process.  The daemon
 * itself calls init_manage, including in NVT cache mode (--rebuild/update).
 *
 * @param[in]  log_config      Log configuration.
 * @param[in]  database        Location of database.
 * @param[in]  max_ips_per_target  Max number of IPs per target.
 * @param[in]  update_progress     Function to update progress, or NULL. *
 *
 * @return 0 success, -1 error, -2 database is wrong version, -3 database needs
 *         to be initialised from server, -4 max_ips_per_target out of range.
 */
int
init_manage_helper (GSList *log_config, const gchar *database,
                    int max_ips_per_target, void (*update_progress) ())
{
  return init_manage_internal (log_config,
                               0,   /* Run daemon in NVT cache mode. */
                               database,
                               max_ips_per_target,
                               0,   /* Default max_email_attachment_size. */
                               0,   /* Default max_email_include_size. */
                               0,   /* Default max_email_message_size */
                               update_progress,
                               0,   /* Stop active tasks. */
                               NULL,
                               0,   /* Skip DB check. */
                               0);  /* Dummy. */
}

/**
 * @brief Cleanup the manage library.
 *
 * Optionally put any running task in the stopped state and close the database.
 *
 * @param[in]  cleanup  If TRUE perform all cleanup operations, else only
 *                      those required at the start of a forked process.
 */
void
cleanup_manage_process (gboolean cleanup)
{
  if (sql_is_open ())
    {
      if (cleanup)
        {
          if (current_scanner_task)
            set_task_run_status (current_scanner_task, TASK_STATUS_STOPPED);
          cleanup_prognosis_iterator ();
          sql_close ();
        }
      else
        sql_close_fork ();
    }
}

/**
 * @brief Cleanup as immediately as possible.
 *
 * Put any running task in the error state and close the database.
 *
 * Intended for handlers for signals like SIGSEGV and SIGABRT.
 *
 * @param[in]  signal  Dummy argument for use as signal handler.
 */
void
manage_cleanup_process_error (int signal)
{
  g_debug ("Received %s signal.\n", sys_siglist[signal]);
  if (sql_is_open ())
    {
      if (current_scanner_task)
        {
          g_warning ("%s: Error exit, setting running task to Internal Error",
                     __FUNCTION__);
          set_task_run_status (current_scanner_task, TASK_STATUS_INTERNAL_ERROR);
        }
      sql_close ();
    }
}

/**
 * @brief Cleanup as immediately as possible.
 */
void
manage_reset_currents ()
{
  current_report = 0;
  current_scanner_task = (task_t) 0;
}

/**
 * @brief Get user hash.
 *
 * This is for "file" users, now entirely stored in db.
 *
 * @param[in]  username  User name.
 *
 * @return Hash.
 */
gchar *
manage_user_hash (const gchar *username)
{
  gchar *hash, *quoted_username;
  quoted_username = sql_quote (username);
  hash = sql_string ("SELECT password FROM users WHERE name = '%s';",
                     quoted_username);
  g_free (quoted_username);
  return hash;
}

/**
 * @brief Get user uuid.
 *
 * @param[in]  username  User name.
 * @param[in]  method    Authentication method.
 *
 * @return UUID.
 */
gchar *
user_uuid_method (const gchar *username, auth_method_t method)
{
  gchar *uuid, *quoted_username, *quoted_method;
  quoted_username = sql_quote (username);
  quoted_method = sql_quote (auth_method_name (method));
  uuid = sql_string ("SELECT uuid FROM users"
                     " WHERE name = '%s' AND method = '%s';",
                     quoted_username,
                     quoted_method);
  g_free (quoted_username);
  g_free (quoted_method);
  return uuid;
}

/**
 * @brief Check whether LDAP is enabled.
 *
 * @return 0 no, else yes.
 */
static int
ldap_auth_enabled ()
{
  if (openvas_auth_ldap_enabled ())
    return sql_int ("SELECT coalesce ((SELECT CAST (value AS INTEGER) FROM meta"
                    "                  WHERE name = 'ldap_enable'),"
                    "                 0);");
  return 0;
}

/**
 * @brief Check whether RADIUS is enabled.
 *
 * @return 0 no, else yes.
 */
static int
radius_auth_enabled ()
{
  if (openvas_auth_radius_enabled ())
    return sql_int ("SELECT coalesce ((SELECT CAST (value AS INTEGER) FROM meta"
                    "                  WHERE name = 'radius_enable'),"
                    "                 0);");
  return 0;
}


/**
 * @brief Check if user exists.
 *
 * @param[in]  name    User name.
 * @param[in]  method  Auth method.
 *
 * @return 1 yes, 0 no.
 */
static int
user_exists_method (const gchar *name, auth_method_t method)
{
  gchar *quoted_name, *quoted_method;
  int ret;

  quoted_name = sql_quote (name);
  quoted_method = sql_quote (auth_method_name (method));
  ret = sql_int ("SELECT count (*) FROM users"
                 " WHERE name = '%s' AND method = '%s';",
                 quoted_name,
                 quoted_method);
  g_free (quoted_name);
  g_free (quoted_method);

  return ret;
}

/**
 * @brief Get user uuid, trying all authentication methods.
 *
 * @param[in]  name    User name.
 *
 * @return UUID.
 */
gchar *
user_uuid_any_method (const gchar *name)
{
  if (ldap_auth_enabled ()
      && user_exists_method (name, AUTHENTICATION_METHOD_LDAP_CONNECT))
    return user_uuid_method (name, AUTHENTICATION_METHOD_LDAP_CONNECT);
  if (radius_auth_enabled ()
      && user_exists_method (name, AUTHENTICATION_METHOD_RADIUS_CONNECT))
    return user_uuid_method (name, AUTHENTICATION_METHOD_RADIUS_CONNECT);
  if (user_exists_method (name, AUTHENTICATION_METHOD_FILE))
    return user_uuid_method (name, AUTHENTICATION_METHOD_FILE);
  return NULL;
}

/**
 * @brief Ensure the user exists in the database.
 *
 * @param[in]  name    User name.
 * @param[in]  method  Auth method.
 *
 * @return 0 success.
 */
int
user_ensure_in_db (const gchar *name, const gchar *method)
{
  gchar *quoted_name, *quoted_method;

  if ((method == NULL) || (strcasecmp (method, "file") == 0))
    /* A "file" user, now entirely stored in db. */
    return 0;

  /* SELECT then INSERT instead of using "INSERT OR REPLACE", so that the
   * id stays the same. */

  quoted_name = sql_quote (name);
  quoted_method = sql_quote (method);

  if (sql_int ("SELECT count(*)"
               " FROM users WHERE name = '%s' and method = '%s';",
               quoted_name,
               quoted_method))
    {
      g_free (quoted_method);
      g_free (quoted_name);
      return 0;
    }

  sql ("INSERT INTO users"
       " (uuid, owner, name, comment, password, timezone, method, hosts,"
       "  hosts_allow, ifaces, ifaces_allow, creation_time, modification_time)"
       " VALUES"
       " (make_uuid (),"
       "  (SELECT id FROM users WHERE users.uuid = '%s'),"
       "  '%s', '', NULL, NULL, '%s', '', 2, '', 2, m_now (), m_now ());",
       current_credentials.uuid,
       quoted_name,
       quoted_method);

  g_free (quoted_method);
  g_free (quoted_name);

  return 0;
}

/**
 * @brief Check if user exists.
 *
 * @param[in]  name    User name.
 *
 * @return 1 yes, 0 no.
 */
int
user_exists (const gchar *name)
{
  if (ldap_auth_enabled ()
      && user_exists_method (name, AUTHENTICATION_METHOD_LDAP_CONNECT))
    return 1;
  if (radius_auth_enabled ()
      && user_exists_method (name, AUTHENTICATION_METHOD_RADIUS_CONNECT))
    return 1;
  return user_exists_method (name, AUTHENTICATION_METHOD_FILE);
}

/**
 * @brief Set the address of scanner to connect to.
 *
 * @param[in]  uuid     Scanner UUID.
 *
 * @return 0 if success, -1 if error, -2 scanner has no cert.
 */
int
manage_scanner_set (const char *uuid)
{
  scanner_t scanner = 0;
  char *host;
  int type, ret = 0;

  if (uuid == NULL)
    return -1;

  if (!current_credentials.uuid)
    current_credentials.uuid = "";
  if (find_scanner (uuid, &scanner) || scanner == 0)
    {
      g_warning ("Failed to find scanner %s\n", uuid);
      return -1;
    }
  if (!strcmp (current_credentials.uuid, ""))
    current_credentials.uuid = NULL;

  type = scanner_type (scanner);
  if (type != SCANNER_TYPE_OPENVAS)
    {
      g_warning ("Scanner %s not an OpenVAS Scanner\n", uuid);
      return -1;
    }
  host = scanner_host (scanner);
  if (host && *host == '/')
    {
      /* XXX: Workaround for unix socket case. Should add a flag. */
      openvas_scanner_set_unix (host);
    }
  else
    {
      char *ca_pub, *key_pub, *key_priv;
      int port;

      port = scanner_port (scanner);
      if (openvas_scanner_set_address (host, port))
        {
          g_warning ("Failed to set %s:%d as scanner\n", host, port);
          g_free (host);
          return -1;
        }
      ca_pub = scanner_ca_pub (scanner);
      key_pub = scanner_key_pub (scanner);
      key_priv = scanner_key_priv (scanner);
      ret = set_certs (ca_pub, key_pub, key_priv);
      g_free (ca_pub);
      g_free (key_pub);
      g_free (key_priv);
    }
  g_free (host);
  if (ret)
    return -2;
  return 0;
}

/**
 * @brief Set the default scanner as the scanner to connect to.
 *
 * @return 0 success, -1 error, -2 scanner has no cert.
 */
int
manage_scanner_set_default ()
{
  return manage_scanner_set (SCANNER_UUID_DEFAULT);
}

/**
 * @brief Set credentials for authenticate.
 *
 * @param[in]  credentials  Credentials.
 *
 * @return 0 success, 99 permission denied.
 */
static int
credentials_setup (credentials_t *credentials)
{
  assert (credentials->uuid);

  credentials->role
    = g_strdup (user_is_super_admin (credentials->uuid)
                 ? "Super Admin"
                 : (acl_user_is_admin (credentials->uuid)
                     ? "Admin"
                     : (acl_user_is_observer (credentials->uuid)
                         ? "Observer"
                         : (acl_user_is_user (credentials->uuid)
                             ? "User"
                             : ""))));

  if (acl_user_may ("authenticate") == 0)
    {
      free (credentials->uuid);
      credentials->uuid = NULL;
      g_free (credentials->role);
      credentials->role = NULL;
      return 99;
    }

  credentials->timezone = sql_string ("SELECT timezone FROM users"
                                      " WHERE uuid = '%s';",
                                      credentials->uuid);

  credentials->severity_class
    = sql_string ("SELECT value FROM settings"
                  " WHERE name = 'Severity Class'"
                  " AND " ACL_USER_OWNS ()
                  " ORDER BY coalesce (owner, 0) DESC LIMIT 1;",
                  credentials->uuid);

  credentials->dynamic_severity
    = sql_int ("SELECT value FROM settings"
                " WHERE name = 'Dynamic Severity'"
                " AND " ACL_USER_OWNS ()
                " ORDER BY coalesce (owner, 0) DESC LIMIT 1;",
                credentials->uuid);

  credentials->default_severity
    = sql_double ("SELECT value FROM settings"
                  " WHERE name = 'Default Severity'"
                  " AND " ACL_USER_OWNS ()
                  " ORDER BY coalesce (owner, 0) DESC LIMIT 1;",
                  credentials->uuid);

  return 0;
}

/**
 * @brief Search for LDAP or RADIUS credentials in the recently-used
 * authentication cache.
 *
 * @param[in]  username     Username.
 * @param[in]  password     Password.
 * @param[in]  method       0 for LDAP, 1 for RADIUS.
 *
 * @return 0 on success, -1 on failure.
 */
static int
auth_cache_find (const char *username, const char *password, int method)
{
  char *hash, *quoted_username;
  int ret;

  quoted_username = sql_quote (username);
  hash = sql_string ("SELECT hash FROM auth_cache WHERE username = '%s'"
                     " AND method = %i AND creation_time >= m_now () - 300;",
                     quoted_username, method);
  g_free (quoted_username);
  if (!hash)
    return -1;

  ret = openvas_authenticate_classic (username, password, hash);
  g_free (hash);
  return ret;
}

/**
 * @brief Add LDAP or RADIUS credentials to the recently-used authentication
 * cache.
 *
 * @param[in]  username     Username.
 * @param[in]  password     Password.
 * @param[in]  method       0 for LDAP, 1 for RADIUS.
 */
static void
auth_cache_insert (const char *username, const char *password, int method)
{
  char *hash, *quoted_username;

  quoted_username = sql_quote (username);
  hash = get_password_hashes (GCRY_MD_MD5, password);
  sql ("INSERT INTO auth_cache (username, hash, method, creation_time)"
       " VALUES ('%s', '%s', %i, m_now ());", quoted_username, hash, method);
  /* Cleanup cache */
  sql ("DELETE FROM auth_cache WHERE creation_time < m_now () - 300");
}

/**
 * @brief Authenticate, trying any method.
 *
 * @param[in]  username     Username.
 * @param[in]  password     Password.
 * @param[out] auth_method  Auth method return.
 *
 * @return 0 authentication success, 1 authentication failure, 99 permission
 *         denied, -1 error.
 */
int
authenticate_any_method (const gchar *username, const gchar *password,
                         auth_method_t *auth_method)
{
  int ret;
  gchar *hash;

  if (openvas_auth_ldap_enabled ()
      && ldap_auth_enabled ()
      && user_exists_method (username, AUTHENTICATION_METHOD_LDAP_CONNECT))
    {
      ldap_auth_info_t info;
      int allow_plaintext;
      gchar *authdn, *host, *cacert;

      *auth_method = AUTHENTICATION_METHOD_LDAP_CONNECT;
      /* Search the LDAP authentication cache first. */
      if (auth_cache_find (username, password, 0) == 0)
        return 0;

      manage_get_ldap_info (NULL, &host, &authdn, &allow_plaintext, &cacert);
      info = ldap_auth_info_new (host, authdn, allow_plaintext);
      g_free (host);
      g_free (authdn);
      ret = ldap_connect_authenticate (username, password, info, cacert);
      ldap_auth_info_free (info);
      free (cacert);

      if (ret == 0)
        auth_cache_insert (username, password, 0);
      return ret;
    }
  if (openvas_auth_radius_enabled ()
      && radius_auth_enabled ()
      && user_exists_method (username, AUTHENTICATION_METHOD_RADIUS_CONNECT))
    {
      char *key = NULL, *host = NULL;

      *auth_method = AUTHENTICATION_METHOD_RADIUS_CONNECT;
      if (auth_cache_find (username, password, 1) == 0)
        return 0;

      manage_get_radius_info (NULL, &host, &key);
      ret = radius_authenticate (host, key, username, password);
      g_free (host);
      g_free (key);
      if (ret == 0)
        auth_cache_insert (username, password, 1);
      return ret;
    }
  *auth_method = AUTHENTICATION_METHOD_FILE;
  hash = manage_user_hash (username);
  ret = openvas_authenticate_classic (username, password, hash);
  g_free (hash);
  return ret;
}

/**
 * @brief Authenticate credentials.
 *
 * @param[in]  credentials  Credentials.
 *
 * @return 0 authentication success, 1 authentication failure, 99 permission
 *         denied, -1 error.
 */
int
authenticate (credentials_t* credentials)
{
  if (credentials->username && credentials->password)
    {
      int fail;
      auth_method_t auth_method;

      if (authenticate_allow_all)
        {
          /* This flag is set when Manager makes a connection to itself, for
           * scheduled tasks and alerts.  Take the stored uuid
           * to be able to tell apart locally authenticated vs remotely
           * authenticated users (in order to fetch the correct rules). */
          credentials->uuid = get_scheduled_user_uuid ();
          if (*credentials->uuid)
            {
              if (credentials_setup (credentials))
                return 99;

              manage_session_init (credentials->uuid);
              return 0;
            }
          return -1;
        }

      fail = authenticate_any_method (credentials->username,
                                      credentials->password,
                                      &auth_method);
      if (fail == 0)
        {
          gchar *quoted_name, *quoted_method;

          /* Authentication succeeded. */

          user_ensure_in_db (credentials->username,
                             auth_method_name (auth_method));

          quoted_name = sql_quote (credentials->username);
          quoted_method = sql_quote (auth_method_name (auth_method));
          credentials->uuid = sql_string ("SELECT uuid FROM users"
                                          " WHERE name = '%s'"
                                          " AND method = '%s';",
                                          quoted_name,
                                          quoted_method);
          g_free (quoted_name);
          g_free (quoted_method);

          if (credentials_setup (credentials))
            {
              free (credentials->uuid);
              credentials->uuid = NULL;
              credentials->role = NULL;
              return 99;
            }

          manage_session_init (credentials->uuid);

          return 0;
        }
      return fail;
    }
  return 1;
}

/**
 * @brief Return number of resources of a certain type for current user.
 *
 * @param[in]  type  Type.
 * @param[in]  get   GET params.
 *
 * @return The number of resources associated with the current user.
 */
int
resource_count (const char *type, const get_data_t *get)
{
  static const char *filter_columns[] = { "owner", NULL };
  static column_t select_columns[] = {{ "owner", NULL }, { NULL, NULL }};
  get_data_t count_get;

  const char *extra_where;

  memset (&count_get, '\0', sizeof (count_get));
  count_get.trash = get->trash;
  if (type_owned (type))
    count_get.filter = "rows=-1 first=1 permission=any owner=any";
  else
    count_get.filter = "rows=-1 first=1 permission=any";

  if (strcmp (type, "task") == 0)
    {
      extra_where = (get->trash
                      ? " AND hidden = 2"
                      : " AND hidden = 0");
    }
  else if (strcmp (type, "report") == 0)
    {
      extra_where = " AND (SELECT hidden FROM tasks"
                    "      WHERE tasks.id = task)"
                    "     = 0";
    }
  else if (strcmp (type, "result") == 0)
    {
      extra_where = " AND (severity != " G_STRINGIFY (SEVERITY_ERROR) ")"
                    " AND (SELECT hidden FROM tasks"
                    "      WHERE tasks.id = task)"
                    "     = 0";
    }
  else
    extra_where = NULL;

  return count (get->subtype ? get->subtype : type,
                &count_get,
                type_owned (type) ? select_columns : NULL,
                type_owned (type) ? select_columns : NULL,
                type_owned (type) ? filter_columns : NULL,
                0, NULL,
                extra_where,
                type_owned (type));
}

/**
 * @brief Test whether a resource of the given type and unique ID exists.
 *
 * @param[in]  type  Type.
 * @param[in]  id    Unique ID.
 *
 * @return 1 if the resource exists, 0 otherwise.
 */
int
resource_id_exists (const char *type, const char * id)
{
  return !!sql_int ("SELECT count(*)"
                    " FROM %ss"
                    " WHERE uuid='%s'"
                    " %s;",
                    type,
                    id,
                    (strcmp (type, "task") == 0) ? "AND hidden=0" : "");
}

/**
 * @brief Test Whether a resource of the given type and ID exists in the trash.
 *
 * @param[in]  type  Type.
 * @param[in]  id    Unique ID.
 *
 * @return 1 if the resource exists, 0 otherwise.
 */
int
trash_id_exists (const char *type, const char * id)
{
  if ((strcmp (type, "nvt") == 0)
      || (strcmp (type, "cpe") == 0)
      || (strcmp (type, "cve") == 0)
      || (strcmp (type, "ovaldef") == 0)
      || (strcmp (type, "cert_bund_adv") == 0)
      || (strcmp (type, "dfn_cert_adv") == 0)
      || (strcmp (type, "report") == 0)
      || (strcmp (type, "result") == 0)
      || (strcmp (type, "user") == 0))
    return 0;
  else if (strcmp (type, "task"))
    return !!sql_int ("SELECT count(*)"
                      " FROM %ss_trash"
                      " WHERE uuid='%s';",
                      type,
                      id);
  else
    return !!sql_int ("SELECT count(*)"
                      " FROM tasks"
                      " WHERE uuid='%s'"
                      " AND hidden=2;",
                      id);
}

/**
 * @brief Return the number of tasks associated with the current user.
 *
 * @param[in]  get  GET params.
 *
 * @return The number of tasks associated with the current user.
 */
unsigned int
task_count (const get_data_t *get)
{
  static const char *extra_columns[] = TASK_ITERATOR_FILTER_COLUMNS;
  static column_t columns[] = TASK_ITERATOR_COLUMNS;
  static column_t where_columns[] = TASK_ITERATOR_WHERE_COLUMNS;
  char *filter;
  gchar *value;
  int overrides, min_qod;
  gchar *extra_tables;
  int ret;

  if (get->filt_id && strcmp (get->filt_id, "0"))
    {
      filter = filter_term (get->filt_id);
      if (filter == NULL)
        return 2;
    }
  else
    filter = NULL;
  value = filter_term_value (filter ? filter : get->filter, "apply_overrides");
  overrides = value && strcmp (value, "0");
  g_free (value);

  value = filter_term_value (filter ? filter : get->filter, "min_qod");
  if (value == NULL || sscanf (value, "%d", &min_qod) != 1)
    min_qod = MIN_QOD_DEFAULT;
  g_free (value);
  free (filter);

  extra_tables = task_iterator_opts_table (overrides, min_qod, 0);

  ret = count2 ("task", get,
                columns,
                columns,
                where_columns,
                where_columns,
                extra_columns, 0,
                extra_tables,
                get->trash
                 ? " AND hidden = 2"
                 : " AND hidden = 0",
                TRUE);

  g_free (extra_tables);
  return ret;
}

/**
 * @brief Return the identifier of a task.
 *
 * @param[in]  task  Task.
 *
 * @return ID of task.
 */
unsigned int
task_id (task_t task)
{
  /** @todo The cast is a hack for compatibility with the old, alternate,
   *        FS based storage mechanism. */
  return (unsigned int) task;
}

/**
 * @brief Return the UUID of a task.
 *
 * @param[in]   task  Task.
 * @param[out]  id    Pointer to a newly allocated string.
 *
 * @return 0.
 */
int
task_uuid (task_t task, char ** id)
{
  *id = sql_string ("SELECT uuid FROM tasks WHERE id = %llu;",
                    task);
  return 0;
}

/**
 * @brief Return whether a task is in the trashcan.
 *
 * @param[in]  task  Task.
 *
 * @return 1 if in trashcan, else 0.
 */
int
task_in_trash (task_t task)
{
  return sql_int ("SELECT hidden = 2"
                  " FROM tasks WHERE id = %llu;",
                  task);
}

/**
 * @brief Return the name of the owner of a task.
 *
 * @param[in]  task  Task.
 *
 * @return Newly allocated user name.
 */
char*
task_owner_name (task_t task)
{
  return sql_string ("SELECT name FROM users WHERE id ="
                     " (SELECT owner FROM tasks WHERE id = %llu);",
                     task);
}

/**
 * @brief Return the name of the owner of a task.
 *
 * @param[in]  task  Task.
 *
 * @return Newly allocated user name.
 */
static char*
task_owner_uuid (task_t task)
{
  return sql_string ("SELECT uuid FROM users WHERE id ="
                     " (SELECT owner FROM tasks WHERE id = %llu);",
                     task);
}

/**
 * @brief Return the name of a task.
 *
 * @param[in]  task  Task.
 *
 * @return Task name.
 */
char*
task_name (task_t task)
{
  return sql_string ("SELECT name FROM tasks WHERE id = %llu;",
                     task);
}

/**
 * @brief Return the comment of a task.
 *
 * @param[in]  task  Task.
 *
 * @return Comment of task.
 */
char*
task_comment (task_t task)
{
  return sql_string ("SELECT comment FROM tasks WHERE id = %llu;",
                     task);
}

/**
 * @brief Return the hosts ordering of a task.
 *
 * @param[in]  task  Task.
 *
 * @return Hosts ordering of task.
 */
char*
task_hosts_ordering (task_t task)
{
  return sql_string ("SELECT hosts_ordering FROM tasks WHERE id = %llu;",
                     task);
}

/**
 * @brief Return the observers of a task.
 *
 * @param[in]  task  Task.
 *
 * @return Observers of task.
 */
char*
task_observers (task_t task)
{
  iterator_t users;
  GString *observers;

  observers = g_string_new ("");

  init_task_user_iterator (&users, task);
  if (next (&users))
    {
      g_string_append (observers, task_user_iterator_name (&users));
      while (next (&users))
        g_string_append_printf (observers,
                                " %s",
                                task_user_iterator_name (&users));
    }
  cleanup_iterator (&users);

  return g_string_free (observers, FALSE);
}

/**
 * @brief Return the config of a task.
 *
 * @param[in]  task  Task.
 *
 * @return Config of task.
 */
config_t
task_config (task_t task)
{
  config_t config;
  switch (sql_int64 (&config,
                     "SELECT config FROM tasks WHERE id = %llu;",
                     task))
    {
      case 0:
        return config;
      default:       /* Programming error. */
      case 1:        /* Too few rows in result of query. */
      case -1:       /* Error. */
        /* Every task should have a config. */
        assert (0);
        return 0;
        break;
    }
}

/**
 * @brief Return the UUID of the config of a task.
 *
 * @param[in]  task  Task.
 *
 * @return UUID of config of task.
 */
char*
task_config_uuid (task_t task)
{
  if (task_config_in_trash (task))
    return sql_string ("SELECT uuid FROM configs_trash WHERE id ="
                       " (SELECT config FROM tasks WHERE id = %llu);",
                       task);
  return sql_string ("SELECT uuid FROM configs WHERE id ="
                     " (SELECT config FROM tasks WHERE id = %llu);",
                     task);
}

/**
 * @brief Return the name of the config of a task.
 *
 * @param[in]  task  Task.
 *
 * @return Name of config of task.
 */
char*
task_config_name (task_t task)
{
  if (task_config_in_trash (task))
    return sql_string ("SELECT name FROM configs_trash WHERE id ="
                       " (SELECT config FROM tasks WHERE id = %llu);",
                       task);
  return sql_string ("SELECT name FROM configs WHERE id ="
                     " (SELECT config FROM tasks WHERE id = %llu);",
                     task);
}

/**
 * @brief Return whether the config of a task is in the trashcan.
 *
 * @param[in]  task  Task.
 *
 * @return 1 if in trashcan, else 0.
 */
int
task_config_in_trash (task_t task)
{
  return sql_int ("SELECT config_location = " G_STRINGIFY (LOCATION_TRASH)
                  " FROM tasks WHERE id = %llu;",
                  task);
}

/**
 * @brief Set the config of a task.
 *
 * @param[in]  task    Task.
 * @param[in]  config  Config.
 */
void
set_task_config (task_t task, config_t config)
{
  sql ("UPDATE tasks SET config = %llu, modification_time = m_now ()"
       " WHERE id = %llu;",
       config,
       task);
}

/**
 * @brief Return the target of a task.
 *
 * @param[in]  task  Task.
 *
 * @return Target of task.
 */
target_t
task_target (task_t task)
{
  target_t target = 0;
  switch (sql_int64 (&target,
                     "SELECT target FROM tasks WHERE id = %llu;",
                     task))
    {
      case 0:
        return target;
        break;
      case 1:        /* Too few rows in result of query. */
      default:       /* Programming error. */
        assert (0);
      case -1:
        return 0;
        break;
    }
}

/**
 * @brief Set the target of a task.
 *
 * @param[in]  task    Task.
 * @param[in]  target  Target.
 */
void
set_task_target (task_t task, target_t target)
{
  sql ("UPDATE tasks SET target = %llu, modification_time = m_now ()"
       " WHERE id = %llu;",
       target,
       task);
}

/**
 * @brief Set the hosts ordering of a task.
 *
 * @param[in]  task         Task.
 * @param[in]  ordering     Hosts ordering.
 */
void
set_task_hosts_ordering (task_t task, const char *ordering)
{
  char *quoted_ordering = sql_quote (ordering ?: "");
  sql ("UPDATE tasks SET hosts_ordering = '%s', modification_time = m_now ()"
       " WHERE id = %llu;", quoted_ordering, task);
  g_free (quoted_ordering);
}

/**
 * @brief Return whether the target of a task is in the trashcan.
 *
 * @param[in]  task  Task.
 *
 * @return 1 if in trash, else 0.
 */
int
task_target_in_trash (task_t task)
{
  return sql_int ("SELECT target_location = " G_STRINGIFY (LOCATION_TRASH)
                  " FROM tasks WHERE id = %llu;",
                  task);
}

/**
 * @brief Return the scanner of a task.
 *
 * @param[in]  task  Task.
 *
 * @return scanner of task.
 */
scanner_t
task_scanner (task_t task)
{
  scanner_t scanner = 0;
  switch (sql_int64 (&scanner, "SELECT scanner FROM tasks WHERE id = %llu;",
                     task))
    {
      case 0:
        return scanner;
        break;
      case 1:        /* Too few rows in result of query. */
      default:       /* Programming error. */
        assert (0);
      case -1:
        return 0;
        break;
    }
}

/**
 * @brief Set the scanner of a task.
 *
 * @param[in]  task     Task.
 * @param[in]  scanner  Scanner.
 */
void
set_task_scanner (task_t task, scanner_t scanner)
{
  sql ("UPDATE tasks SET scanner = %llu, modification_time = m_now ()"
       " WHERE id = %llu;", scanner, task);
  if (scanner_type (scanner) == SCANNER_TYPE_CVE)
    sql ("UPDATE tasks SET config = 0 WHERE id = %llu;", task);
}

/**
 * @brief Return whether the scanner of a task is in the trashcan.
 *
 * @param[in]  task  Task.
 *
 * @return 1 if in trash, else 0.
 */
int
task_scanner_in_trash (task_t task)
{
  return sql_int ("SELECT scanner_location = " G_STRINGIFY (LOCATION_TRASH)
                  " FROM tasks WHERE id = %llu;", task);
}

/**
 * @brief Return the run state of a task.
 *
 * @param[in]  task  Task.
 *
 * @return Task run status.
 */
task_status_t
task_run_status (task_t task)
{
  return (unsigned int) sql_int ("SELECT run_status FROM tasks WHERE id = %llu;",
                                 task);
}

/**
 * @brief Set a report's scheduled flag.
 *
 * Set flag if task was scheduled, else clear flag.
 *
 * @param[in]   report  Report.
 */
void
set_report_scheduled (report_t report)
{
  if (authenticate_allow_all == 1)
    /* The task was scheduled. */
    sql ("UPDATE reports SET flags = 1 WHERE id = %llu;",
         report);
  else
    sql ("UPDATE reports SET flags = 0 WHERE id = %llu;",
         report);
}

/**
 * @brief Get a report's scheduled flag.
 *
 * @param[in]   report  Report.
 */
static int
report_scheduled (report_t report)
{
  return sql_int ("SELECT flags FROM reports WHERE id = %llu;",
                  report);
}

/**
 * @brief Set the run state of a task.
 *
 * @param[in]  task    Task.
 * @param[in]  status  New run status.
 */
static void
set_task_run_status_internal (task_t task, task_status_t status)
{
  if ((task == current_scanner_task) && current_report)
    {
      sql ("UPDATE reports SET scan_run_status = %u WHERE id = %llu;",
           status,
           current_report);
      if (setting_auto_cache_rebuild_int ())
        report_cache_counts (current_report, 0, 0, NULL);
    }

  sql ("UPDATE tasks SET run_status = %u WHERE id = %llu;",
       status,
       task);
}

/**
 * @brief Set the run state of a task.
 *
 * Logs and generates event.
 *
 * @param[in]  task    Task.
 * @param[in]  status  New run status.
 */
void
set_task_run_status (task_t task, task_status_t status)
{
  char *uuid;
  char *name;

  set_task_run_status_internal (task, status);

  task_uuid (task, &uuid);
  name = task_name (task);
  g_log ("event task", G_LOG_LEVEL_MESSAGE,
         "Status of task %s (%s) has changed to %s",
         name, uuid, run_status_name (status));
  free (uuid);
  free (name);

  event (task,
         (task == current_scanner_task) ? current_report : 0,
         EVENT_TASK_RUN_STATUS_CHANGED, (void*) status);
}

/**
 * @brief Atomically set the run state of a task to requested.
 *
 * @param[in]  task    Task.
 * @param[out] status  Old run status of task.
 *
 * @return 0 success, 1 task is active already.
 */
int
set_task_requested (task_t task, task_status_t *status)
{
  task_status_t run_status;

  sql_begin_exclusive ();

  run_status = task_run_status (task);
  if (run_status == TASK_STATUS_REQUESTED
      || run_status == TASK_STATUS_RUNNING
      || run_status == TASK_STATUS_STOP_REQUESTED
      || run_status == TASK_STATUS_STOP_REQUESTED_GIVEUP
      || run_status == TASK_STATUS_STOP_WAITING
      || run_status == TASK_STATUS_DELETE_REQUESTED
      || run_status == TASK_STATUS_DELETE_ULTIMATE_REQUESTED
      || run_status == TASK_STATUS_DELETE_ULTIMATE_WAITING
      || run_status == TASK_STATUS_DELETE_WAITING)
    {
      sql_commit ();
      *status = run_status;
      return 1;
    }

  set_task_run_status (task, TASK_STATUS_REQUESTED);

  sql_commit ();

  *status = run_status;
  return 0;
}

/**
 * @brief Return number of results in a task.
 *
 * @param[in]  task     Task.
 * @param[in]  min_qod  Minimum QOD.
 *
 * @return Result count.
 */
int
task_result_count (task_t task, int min_qod)
{
  return sql_int ("SELECT count (*) FROM results"
                  " WHERE task = %llu"
                  " AND qod > %i"
                  " AND severity > " G_STRINGIFY (SEVERITY_ERROR) ";",
                  task,
                  min_qod);
}

/**
 * @brief Return the running report of a task.
 *
 * @param[in]  task  Task.
 *
 * @return Current report of task if task is active, else (report_t) 0.
 */
report_t
task_running_report (task_t task)
{
  task_status_t run_status = task_run_status (task);
  if (run_status == TASK_STATUS_REQUESTED
      || run_status == TASK_STATUS_RUNNING)
    {
      return (unsigned int) sql_int ("SELECT max(id) FROM reports"
                                     " WHERE task = %llu AND end_time IS NULL"
                                     " AND scan_run_status = %u;",
                                     task,
                                     TASK_STATUS_RUNNING);
    }
  return (report_t) 0;
}

/**
 * @brief Return the current report of a task.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return Current report of task if task is active, else (report_t) 0.
 */
report_t
task_iterator_current_report (iterator_t *iterator)
{
  task_t task = get_iterator_resource (iterator);
  task_status_t run_status = task_iterator_run_status (iterator);
  if (run_status == TASK_STATUS_REQUESTED
      || run_status == TASK_STATUS_RUNNING
      || run_status == TASK_STATUS_DELETE_REQUESTED
      || run_status == TASK_STATUS_DELETE_ULTIMATE_REQUESTED
      || run_status == TASK_STATUS_STOP_REQUESTED
      || run_status == TASK_STATUS_STOP_REQUESTED_GIVEUP
      || run_status == TASK_STATUS_STOPPED
      || run_status == TASK_STATUS_INTERNAL_ERROR)
    {
      return (unsigned int) sql_int ("SELECT max(id) FROM reports"
                                     " WHERE task = %llu"
                                     " AND (scan_run_status = %u"
                                     " OR scan_run_status = %u"
                                     " OR scan_run_status = %u"
                                     " OR scan_run_status = %u"
                                     " OR scan_run_status = %u"
                                     " OR scan_run_status = %u"
                                     " OR scan_run_status = %u"
                                     " OR scan_run_status = %u);",
                                     task,
                                     TASK_STATUS_REQUESTED,
                                     TASK_STATUS_RUNNING,
                                     TASK_STATUS_DELETE_REQUESTED,
                                     TASK_STATUS_DELETE_ULTIMATE_REQUESTED,
                                     TASK_STATUS_STOP_REQUESTED,
                                     TASK_STATUS_STOP_REQUESTED_GIVEUP,
                                     TASK_STATUS_STOPPED,
                                     TASK_STATUS_INTERNAL_ERROR);
    }
  return (report_t) 0;
}

/**
 * @brief Return the upload progress of a task.
 *
 * @param[in]  task  Task.
 *
 * @return Task upload progress, as a percentage, or -1 on error.
 */
int
task_upload_progress (task_t task)
{
  report_t report;
  report = task_running_report (task);
  if (report)
    {
      int count;
      int using_sqlite = sql_is_sqlite3 ();
      get_data_t get;
      memset (&get, 0, sizeof (get_data_t));
      get.filter = g_strdup ("min_qod=0");
      count = result_count (&get, report, NULL);
      get_data_reset (&get);

      return sql_int ("SELECT"
                      " %s (%s (((%i * 100) / upload_result_count), 100), -1)"
                      " FROM tasks"
                      " WHERE id = %llu;",
                      using_sqlite ? "max" : "greatest",
                      using_sqlite ? "min" : "least",
                      count,
                      task);
    }
  return -1;
}

/**
 * @brief Set the start time of a task.
 *
 * @param[in]  task  Task.
 * @param[in]  time  New time.  ISO format.  Freed before return.
 */
void
set_task_start_time (task_t task, char* time)
{
  sql ("UPDATE tasks SET start_time = %i, modification_time = m_now ()"
       " WHERE id = %llu;",
       parse_iso_time (time),
       task);
  free (time);
}

/**
 * @brief Set the start time of a task.
 *
 * @param[in]  task  Task.
 * @param[in]  time  New time.  Seconds since epoch.
 */
void
set_task_start_time_epoch (task_t task, int time)
{
  sql ("UPDATE tasks SET start_time = %i, modification_time = m_now ()"
       " WHERE id = %llu;",
       time,
       task);
}

/**
 * @brief Set the start time of a task.
 *
 * @param[in]  task  Task.
 * @param[in]  time  New time.  OTP format (ctime).  Freed before return.
 */
void
set_task_start_time_otp (task_t task, char* time)
{
  sql ("UPDATE tasks SET start_time = %i, modification_time = m_now ()"
       " WHERE id = %llu;",
       parse_otp_time (time),
       task);
  free (time);
}

/**
 * @brief Get the report from the most recently completed invocation of task.
 *
 * @param[in]  task    The task.
 * @param[out] report  Report return, 0 if succesfully failed to select report.
 *
 * @return 0 success, -1 error.
 */
int
task_last_report (task_t task, report_t *report)
{
  switch (sql_int64 (report,
                     "SELECT id FROM reports WHERE task = %llu"
                     " AND scan_run_status = %u"
                     " ORDER BY date DESC LIMIT 1;",
                     task,
                     TASK_STATUS_DONE))
    {
      case 0:
        break;
      case 1:        /* Too few rows in result of query. */
        *report = 0;
        return 0;
        break;
      default:       /* Programming error. */
        assert (0);
      case -1:
        return -1;
        break;
    }
  return 0;
}

/**
 * @brief Get the report from second most recently completed invocation of task.
 *
 * @param[in]  task    The task.
 * @param[out] report  Report return, 0 if succesfully failed to select report.
 *
 * @return 0 success, -1 error.
 */
static int
task_second_last_report (task_t task, report_t *report)
{
  switch (sql_int64 (report,
                     "SELECT id FROM reports WHERE task = %llu"
                     " AND scan_run_status = %u"
                     " ORDER BY date DESC LIMIT 1 OFFSET 1;",
                     task,
                     TASK_STATUS_DONE))
    {
      case 0:
        break;
      case 1:        /* Too few rows in result of query. */
        *report = 0;
        return 0;
        break;
      default:       /* Programming error. */
        assert (0);
      case -1:
        return -1;
        break;
    }
  return 0;
}

/**
 * @brief Get the report from the most recently stopped invocation of task.
 *
 * @param[in]  task    The task.
 * @param[out] report  Report return, 0 if succesfully failed to select report.
 *
 * @return 0 success, -1 error.
 */
int
task_last_stopped_report (task_t task, report_t *report)
{
  switch (sql_int64 (report,
                     "SELECT id FROM reports WHERE task = %llu"
                     " AND scan_run_status = %u"
                     " ORDER BY date DESC LIMIT 1;",
                     task,
                     TASK_STATUS_STOPPED))
    {
      case 0:
        break;
      case 1:        /* Too few rows in result of query. */
        *report = 0;
        return 0;
        break;
      default:       /* Programming error. */
        assert (0);
      case -1:
        return -1;
        break;
    }
  return 0;
}

/**
 * @brief Get report ID from second most recently completed invocation of task.
 *
 * @param[in]  task  The task.
 *
 * @return The UUID of the report as a newly allocated string.
 */
gchar*
task_second_last_report_id (task_t task)
{
  return sql_string ("SELECT uuid FROM reports WHERE task = %llu"
                     " AND scan_run_status = %u"
                     " ORDER BY date DESC LIMIT 1 OFFSET 1;",
                     task,
                     TASK_STATUS_DONE);
}

/**
 * @brief Add an alert to a task.
 *
 * @param[in]  task       Task.
 * @param[in]  alert  Alert.
 */
void
add_task_alert (task_t task, alert_t alert)
{
  sql ("INSERT INTO task_alerts (task, alert, alert_location)"
       " VALUES (%llu, %llu, " G_STRINGIFY (LOCATION_TABLE) ");",
       task,
       alert);
}

/**
 * @brief Set the alerts on a task, removing any previous alerts.
 *
 * @param[in]  task    Task.
 * @param[in]  alerts  Alerts.
 * @param[out] alert_id_return  ID of alert on "failed to find" error.
 *
 * @return 0 success, -1 error, 1 failed to find alert.
 */
int
set_task_alerts (task_t task, array_t *alerts, gchar **alert_id_return)
{
  alert_t alert = 0;
  guint index;

  sql_begin_immediate ();

  sql ("DELETE FROM task_alerts where task = %llu;", task);

  index = alerts->len;
  while (index--)
    {
      gchar *alert_id;

      alert_id = (gchar*) g_ptr_array_index (alerts, index);
      if (strcmp (alert_id, "0") == 0)
        continue;

      if (find_alert_with_permission (alert_id, &alert, "get_alerts"))
        {
          sql_rollback ();
          return -1;
        }

      if (alert == 0)
        {
          sql_rollback ();
          if (alert_id_return) *alert_id_return = alert_id;
          return 1;
        }

      sql ("INSERT INTO task_alerts (task, alert, alert_location)"
           " VALUES (%llu, %llu, " G_STRINGIFY (LOCATION_TABLE) ");",
           task,
           alert);
    }

  sql_commit ();
  return 0;
}

/**
 * @brief Set the alterable state of a task.
 *
 * @param[in]  task       Task.
 * @param[in]  alterable  Whether task is alterable.
 */
void
set_task_alterable (task_t task, int alterable)
{
  sql ("UPDATE tasks SET alterable = %i WHERE id = %llu;",
       alterable,
       task);
}

/**
 * @brief Set observer groups on a task, removing any previous groups.
 *
 * @param[in]  task    Task.
 * @param[in]  groups  Groups.
 * @param[out] group_id_return  ID of group on "failed to find" error.
 *
 * @return 0 success, -1 error, 1 failed to find group.
 */
int
set_task_groups (task_t task, array_t *groups, gchar **group_id_return)
{
  group_t group = 0;
  guint index;

  sql_begin_immediate ();

  sql ("DELETE FROM permissions"
       " WHERE resource_type = 'task'"
       " AND resource = %llu"
       " AND subject_type = 'group'"
       " AND name = 'get';",
       task);

  index = 0;
  while (index < groups->len)
    {
      gchar *group_id;

      group_id = (gchar*) g_ptr_array_index (groups, index);
      if (strcmp (group_id, "0") == 0)
        {
          index++;
          continue;
        }

      if (find_group (group_id, &group))
        {
          sql_rollback ();
          return -1;
        }

      if (group == 0)
        {
          sql_rollback ();
          if (group_id_return) *group_id_return = group_id;
          return 1;
        }

      sql ("INSERT INTO permissions"
           " (uuid, owner, name, comment, resource_type, resource,"
           "  resource_uuid, resource_location, subject_type, subject,"
           "  subject_location, creation_time, modification_time)"
           " VALUES"
           " (make_uuid (),"
           "  (SELECT id FROM users WHERE users.uuid = '%s'),"
           "  'get_tasks', '', 'task', %llu,"
           "  (SELECT uuid FROM tasks WHERE tasks.id = %llu),"
           "  " G_STRINGIFY (LOCATION_TABLE) ", 'group', %llu,"
           "  " G_STRINGIFY (LOCATION_TABLE) ", m_now (), m_now ());",
           current_credentials.uuid, task, task, group);

      index++;
    }

  sql_commit ();
  return 0;
}

/**
 * @brief Set the schedule of a task.
 *
 * @param[in]  task      Task.
 * @param[in]  schedule  Schedule.
 * @param[in]  periods   Number of schedule periods.
 *
 * @return 0 success, -1 error.
 */
int
set_task_schedule (task_t task, schedule_t schedule, int periods)
{
  sql ("UPDATE tasks"
       " SET schedule = %llu,"
       " schedule_periods = %i,"
       " schedule_next_time = (SELECT next_time (first_time,"
       "                                         period,"
       "                                         period_months,"
       "                                         timezone)"
       "                       FROM schedules"
       "                       WHERE id = %llu),"
       " modification_time = m_now ()"
       " WHERE id = %llu;",
       schedule, periods, schedule, task);

  return 0;
}

/**
 * @brief Set the schedule of a task.
 *
 * @param[in]  task_id   Task UUID.
 * @param[in]  schedule  Schedule.
 * @param[in]  periods   Number of schedule periods.
 *
 * @return 0 success, -1 error.
 */
int
set_task_schedule_uuid (const gchar *task_id, schedule_t schedule, int periods)
{
  gchar *quoted_task_id;

  quoted_task_id = sql_quote (task_id);
  sql ("UPDATE tasks"
       " SET schedule = %llu,"
       " schedule_periods = %i,"
       " schedule_next_time = (SELECT next_time (first_time,"
       "                                         period,"
       "                                         period_months,"
       "                                         timezone)"
       "                       FROM schedules"
       "                       WHERE id = %llu),"
       " modification_time = m_now ()"
       " WHERE uuid = '%s';",
       schedule, periods, schedule, quoted_task_id);
  g_free (quoted_task_id);

  return 0;
}

/**
 * @brief Set the schedule periods of a task, given a UUID.
 *
 * The task modification time stays the same.
 *
 * @param[in]  task_id   Task UUID.
 * @param[in]  periods   Schedule periods.
 *
 * @return 0 success, -1 error.
 */
int
set_task_schedule_periods (const gchar *task_id, int periods)
{
  gchar *quoted_task_id;

  quoted_task_id = sql_quote (task_id);
  sql ("UPDATE tasks"
       " SET schedule_periods = %i"
       " WHERE uuid = '%s';",
       periods, quoted_task_id);
  g_free (quoted_task_id);

  return 0;
}

/**
 * @brief Set the schedule periods of a task, given an ID.
 *
 * The task modification time stays the same.
 *
 * @param[in]  task      Task UUID.
 * @param[in]  periods   Schedule periods.
 *
 * @return 0 success, -1 error.
 */
int
set_task_schedule_periods_id (task_t task, int periods)
{
  sql ("UPDATE tasks"
       " SET schedule_periods = %i"
       " WHERE id = %llu;",
       periods, task);
  return 0;
}

/**
 * @brief Return the schedule of a task.
 *
 * @param[in]  task  Task.
 *
 * @return Schedule.
 */
schedule_t
task_schedule (task_t task)
{
  schedule_t schedule = 0;
  switch (sql_int64 (&schedule,
                     "SELECT schedule FROM tasks WHERE id = %llu;",
                     task))
    {
      case 0:
        return schedule;
        break;
      case 1:        /* Too few rows in result of query. */
      default:       /* Programming error. */
        assert (0);
      case -1:
        return 0;
        break;
    }
}

/**
 * @brief Return the schedule of a task.
 *
 * @param[in]  task_id  ID of task.
 *
 * @return Schedule.
 */
schedule_t
task_schedule_uuid (const gchar *task_id)
{
  schedule_t schedule;
  gchar *quoted_task_id;

  quoted_task_id = sql_quote (task_id);
  schedule = 0;
  switch (sql_int64 (&schedule,
                     "SELECT schedule FROM tasks WHERE uuid = '%s';",
                     quoted_task_id))
    {
      case 0:
        g_free (quoted_task_id);
        return schedule;
        break;
      case 1:        /* Too few rows in result of query. */
      default:       /* Programming error. */
        assert (0);
      case -1:
        g_free (quoted_task_id);
        return 0;
        break;
    }
}

/**
 * @brief Get whether the task schedule is in the trash.
 *
 * @param[in]  task  Task.
 *
 * @return 1 if in trash, else 0.
 */
int
task_schedule_in_trash (task_t task)
{
  return sql_int ("SELECT schedule_location = " G_STRINGIFY (LOCATION_TRASH)
                  " FROM tasks"
                  " WHERE id = %llu;",
                  task);
}

/**
 * @brief Get the number of times the period schedule should run on the task.
 *
 * @param[in]  task  Task.
 *
 * @return Number of times.
 */
int
task_schedule_periods (task_t task)
{
  return sql_int ("SELECT schedule_periods FROM tasks WHERE id = %llu;", task);
}

/**
 * @brief Set the next time a scheduled task will be due.
 *
 * @param[in]  task_id  Task UUID.
 *
 * @return Task schedule periods.
 */
int
task_schedule_periods_uuid (const gchar *task_id)
{
  gchar *quoted_task_id;
  int ret;

  quoted_task_id = sql_quote (task_id);
  ret = sql_int ("SELECT schedule_periods FROM tasks WHERE uuid = '%s';",
                 quoted_task_id);
  g_free (quoted_task_id);
  return ret;
}

/**
 * @brief Get next time a scheduled task will run, following schedule timezone.
 *
 * @param[in]  task  Task.
 *
 * @return If the task has a schedule, the next time the task will run (0 if it
 *         has already run), otherwise 0.
 */
int
task_schedule_next_time (task_t task)
{
  int next_time;

  next_time = sql_int ("SELECT schedule_next_time FROM tasks"
                       " WHERE id = %llu;",
                       task);

  return next_time;
}

/**
 * @brief Set the next time a scheduled task will be due.
 *
 * @param[in]  task_id  Task UUID.
 */
time_t
task_schedule_next_time_uuid (const gchar *task_id)
{
  gchar *quoted_task_id;
  time_t ret;

  quoted_task_id = sql_quote (task_id);
  ret = (time_t) sql_int ("SELECT schedule_next_time FROM tasks"
                          " WHERE uuid = '%s';",
                          quoted_task_id);
  g_free (quoted_task_id);
  return ret;
}

/**
 * @brief Set the next time a scheduled task will be due.
 *
 * @param[in]  task  Task.
 * @param[in]  time  New next time.
 */
void
set_task_schedule_next_time (task_t task, time_t time)
{
  sql ("UPDATE tasks SET schedule_next_time = %i WHERE id = %llu;",
       time, task);
}

/**
 * @brief Set the next time a scheduled task will be due.
 *
 * @param[in]  task_id  Task UUID.
 * @param[in]  time     New next time.
 */
void
set_task_schedule_next_time_uuid (const gchar *task_id, time_t time)
{
  gchar *quoted_task_id;

  quoted_task_id = sql_quote (task_id);
  sql ("UPDATE tasks SET schedule_next_time = %i WHERE uuid = '%s';",
       time, quoted_task_id);
  g_free (quoted_task_id);
}

/**
 * @brief Return the severity score of a task, taking overrides into account.
 *
 * @param[in]  task       Task.
 * @param[in]  overrides  Whether to apply overrides.
 * @param[in]  min_qod    Minimum QoD of results to count.
 * @param[in]  offset     Offset of report to get severity from:
 *                        0 = use last report, 1 = use next to last report
 *
 * @return Severity score of last report on task if there is one, as a freshly
 *  allocated string, else NULL.
 */
char *
task_severity (task_t task, int overrides, int min_qod, int offset)
{
  double severity;

  severity = task_severity_double (task, overrides, min_qod, offset);

  if (severity == SEVERITY_MISSING)
    return NULL;
  else
    return g_strdup_printf ("%0.1f", severity);
}

/**
 * @brief Return the severity score of a task, taking overrides into account.
 *
 * @param[in]  task       Task.
 * @param[in]  overrides  Whether to apply overrides.
 * @param[in]  min_qod    Minimum QoD of results to count.
 * @param[in]  offset     Offset of report to get severity from:
 *                        0 = use last report, 1 = use next to last report
 *
 * @return Severity score of last report on task as a double if there is one,
 *         else SEVERITY_MISSING.
 */
double
task_severity_double (task_t task, int overrides, int min_qod, int offset)
{
  report_t report;

  if (current_credentials.uuid == NULL
      || task_target (task) == 0 /* Container task. */)
    return SEVERITY_MISSING;

  sql_int64 (&report,
             "SELECT id FROM reports"
             "           WHERE reports.task = %llu"
             "           AND reports.scan_run_status = %u"
             "           ORDER BY reports.date DESC"
             "           LIMIT 1 OFFSET %d",
             task, TASK_STATUS_DONE, offset);

  return report_severity (report, overrides, min_qod);
}

/**
 * @brief Set the observers of a task.
 *
 * @param[in]  task       Task.
 * @param[in]  observers  Observers.
 *
 * @return 0 success, -1 error, 1 user name validation failed, 2 failed to find
 *         user.
 */
int
set_task_observers (task_t task, const gchar *observers)
{
  gchar **split, **point;
  GList *added;

  // TODO the tricky bit here is if you have to own the task to set observers.

  assert (current_credentials.username);

  added = NULL;
  split = g_strsplit_set (observers, " ,", 0);

  sql_begin_immediate ();

  sql ("DELETE FROM permissions"
       " WHERE resource_type = 'task' AND resource = %llu"
       " AND subject_type = 'user';",
       task);

  point = split;
  while (*point)
    {
      user_t user;
      gchar *name;

      name = *point;

      g_strstrip (name);

      if (strcmp (name, "") == 0)
        {
          point++;
          continue;
        }

      if ((strcmp (name, current_credentials.username) == 0)
          || g_list_find_custom (added, name, (GCompareFunc) strcmp))
        {
          point++;
          continue;
        }

      added = g_list_prepend (added, name);

      if (user_exists (name) == 0)
        {
          g_list_free (added);
          g_strfreev (split);
          sql_rollback ();
          return 2;
        }

      if (find_user_by_name (name, &user))
        {
          g_list_free (added);
          g_strfreev (split);
          sql_rollback ();
          return -1;
        }

      if (user == 0)
        {
          gchar *uuid;

          /** @todo Similar to validate_user in openvas-administrator. */
          if (g_regex_match_simple ("^[[:alnum:]-_]+$", name, 0, 0) == 0)
            {
              g_list_free (added);
              g_strfreev (split);
              sql_rollback ();
              return 1;
            }

          uuid = user_uuid_any_method (name);

          if (uuid == NULL)
            {
              g_list_free (added);
              g_strfreev (split);
              sql_rollback ();
              return -1;
            }

          if (sql_int ("SELECT count(*) FROM users WHERE uuid = '%s';",
                       uuid)
              == 0)
            {
              gchar *quoted_name;
              quoted_name = sql_quote (name);
              sql ("INSERT INTO users"
                   " (uuid, name, creation_time, modification_time)"
                   " VALUES"
                   " ('%s', '%s', m_now (), m_now ());",
                   uuid,
                   quoted_name);
              g_free (quoted_name);

              user = sql_last_insert_id ();
            }
          else
            {
              /* user_find should have found it. */
              assert (0);
              g_free (uuid);
              g_list_free (added);
              g_strfreev (split);
              sql_rollback ();
              return -1;
            }

          g_free (uuid);
        }

      sql ("INSERT INTO permissions"
           " (uuid, owner, name, comment, resource_type, resource,"
           "  resource_uuid, resource_location, subject_type, subject,"
           "  subject_location, creation_time, modification_time)"
           " VALUES"
           " (make_uuid (),"
           "  (SELECT id FROM users WHERE users.uuid = '%s'),"
           "  'get_tasks', '', 'task', %llu,"
           "  (SELECT uuid FROM tasks WHERE tasks.id = %llu),"
           "  " G_STRINGIFY (LOCATION_TABLE) ", 'user', %llu,"
           "  " G_STRINGIFY (LOCATION_TABLE) ", m_now (), m_now ());",
           current_credentials.uuid, task, task, user);

      point++;
    }

  g_list_free (added);
  g_strfreev (split);
  sql_commit ();
  return 0;
}

/**
 * @brief Clear once-off schedules from tasks where the duration has passed.
 *
 * @param[in]  task  Task.  0 for all.
 */
void
clear_duration_schedules (task_t task)
{
  gchar *task_element;
  const gchar *duration_expired_element;

  if (task)
    task_element = g_strdup_printf (" AND id = %llu", task);
  else
    task_element = g_strdup ("");

  duration_expired_element
   = " AND (SELECT first_time + duration FROM schedules"
     "      WHERE schedules.id = schedule)"
     "     < m_now ()";

  sql ("UPDATE tasks"
       " SET schedule = 0,"
       " schedule_next_time = 0,"
       " modification_time = m_now ()"
       " WHERE schedule > 0"
       "%s"
       " AND (SELECT period FROM schedules WHERE schedules.id = schedule) = 0"
       " AND (SELECT period_months FROM schedules"
       "       WHERE schedules.id = schedule) = 0"
       " AND (SELECT duration FROM schedules WHERE schedules.id = schedule) > 0"
       "%s"
       " AND run_status != %i"
       " AND run_status != %i;",
       task_element,
       task ? "" : duration_expired_element,
       TASK_STATUS_RUNNING,
       TASK_STATUS_REQUESTED);

  g_free (task_element);
}

/**
 * @brief Update tasks with limited run schedules which have durations.
 *
 * If a task is given, assume that the task has finished.  Otherwise only
 * update the task if more time than the duration has passed the start time.
 *
 * @param[in]  task  Task.  0 for all.
 */
void
update_duration_schedule_periods (task_t task)
{
  gchar *task_element;
  const gchar *duration_expired_element;

  if (task)
    task_element = g_strdup_printf (" AND id = %llu", task);
  else
    task_element = g_strdup ("");

  duration_expired_element
   = /* The task has started, so assume that the start time was the last
      * most recent start of the period. */
     " AND (SELECT next_time (first_time, period, period_months,"
     "                        timezone, -1)"
     "             + duration"
     "      FROM schedules"
     "      WHERE schedules.id = schedule)"
     "     < m_now ()";

  sql ("UPDATE tasks"
       " SET schedule = 0,"
       " schedule_next_time = 0,"
       " modification_time = m_now ()"
       " WHERE schedule > 0"
       "%s"
       " AND schedule_periods = 1"
       " AND ((SELECT period FROM schedules WHERE schedules.id = schedule) > 0"
       "      OR (SELECT period_months FROM schedules"
       "          WHERE schedules.id = schedule) > 0)"
       " AND (SELECT duration FROM schedules WHERE schedules.id = schedule) > 0"
       " AND schedule_next_time = 0"  /* Set as flag when starting task. */
       "%s"
       " AND run_status != %i"
       " AND run_status != %i;",
       task_element,
       task ? "" : duration_expired_element,
       TASK_STATUS_RUNNING,
       TASK_STATUS_REQUESTED);
  g_free (task_element);
}

/**
 * @brief Auto delete reports.
 */
void
auto_delete_reports ()
{
  iterator_t tasks;

  g_debug ("%s", __FUNCTION__);

  if (sql_begin_exclusive_giveup ())
    return;

  init_iterator (&tasks,
                 "SELECT id, name,"
                 "       (SELECT value FROM task_preferences"
                 "        WHERE name = 'auto_delete_data'"
                 "        AND task = tasks.id)"
                 " FROM tasks"
                 " WHERE owner is NOT NULL"
                 " AND hidden = 0"
                 " AND EXISTS (SELECT * FROM task_preferences"
                 "             WHERE task = tasks.id"
                 "             AND name = 'auto_delete'"
                 "             AND value = 'keep');");
  while (next (&tasks))
    {
      task_t task;
      iterator_t reports;
      const char *keep_string;
      int keep;

      task = iterator_int64 (&tasks, 0);

      keep_string = iterator_string (&tasks, 2);
      if (keep_string == NULL)
        continue;
      keep = atoi (keep_string);
      if (keep < AUTO_DELETE_KEEP_MIN || keep > AUTO_DELETE_KEEP_MAX)
        continue;

      g_debug ("%s: %s (%i)", __FUNCTION__,
              iterator_string (&tasks, 1),
              keep);

      init_iterator (&reports,
                     "SELECT id FROM reports"
                     " WHERE task = %llu"
                     " AND start_time IS NOT NULL"
                     " AND start_time > 0"
                     " ORDER BY start_time DESC LIMIT %s OFFSET %i;",
                     task,
                     sql_select_limit (-1),
                     keep);
      while (next (&reports))
        {
          int ret;
          report_t report;

          report = iterator_int64 (&reports, 0);
          assert (report);

          g_debug ("%s: delete %llu", __FUNCTION__, report);
          ret = delete_report_internal (report);
          if (ret == 2)
            {
              /* Report is in use. */
              g_debug ("%s: %llu is in use", __FUNCTION__, report);
              continue;
            }
          if (ret)
            {
              g_warning ("%s: failed to delete %llu (%i)\n",
                         __FUNCTION__, report, ret);
              sql_rollback ();
            }
        }
      cleanup_iterator (&reports);
    }
  cleanup_iterator (&tasks);
  sql_commit ();
}


/**
 * @brief Get definitions file from a task's config.
 *
 * @param[in]  task  Task.
 *
 * @return Definitions file.
 */
char *
task_definitions_file (task_t task)
{
  assert (task);
  return sql_string ("SELECT value FROM config_preferences"
                     " WHERE name = 'definitions_file' AND config = %llu;",
                     task_config (task));
}

/**
 * @brief Set a task's schedule so that it runs again next scheduling round.
 *
 * @param  task_id  UUID of task.
 */
void
reschedule_task (const gchar *task_id)
{
  task_t task;
  task = 0;
  switch (sql_int64 (&task,
                     "SELECT id FROM tasks WHERE uuid = '%s'"
                     " AND hidden != 2;",
                     task_id))
    {
      case 0:
        g_warning ("%s: rescheduling task '%s'\n", __FUNCTION__, task_id);
        set_task_schedule_next_time (task, time (NULL) - 1);
        break;
      case 1:        /* Too few rows in result of query. */
        break;
      default:       /* Programming error. */
        assert (0);
      case -1:
        break;
    }
}


/* Results. */

/**
 * @brief Find a result for a set of permissions, given a UUID.
 *
 * @param[in]   uuid        UUID of result.
 * @param[out]  result      Result return, 0 if succesfully failed to find
 *                          result.
 * @param[in]   permission  Permission.
 *
 * @return FALSE on success (including if failed to find result), TRUE on error.
 */
gboolean
find_result_with_permission (const char* uuid, result_t* result,
                             const char *permission)
{
  gchar *quoted_uuid = sql_quote (uuid);
  if (acl_user_has_access_uuid ("result", quoted_uuid, permission, 0) == 0)
    {
      g_free (quoted_uuid);
      *result = 0;
      return FALSE;
    }
  switch (sql_int64 (result,
                     "SELECT id FROM results WHERE uuid = '%s';",
                     quoted_uuid))
    {
      case 0:
        break;
      case 1:        /* Too few rows in result of query. */
        *result = 0;
        break;
      default:       /* Programming error. */
        assert (0);
      case -1:
        g_free (quoted_uuid);
        return TRUE;
        break;
    }

  g_free (quoted_uuid);
  return FALSE;
}

/**
 * @brief Make an OSP result.
 *
 * @param[in]  task         The task associated with the result.
 * @param[in]  host         Target host of result.
 * @param[in]  nvt          The uuid of oval definition that produced the
 *                          result, a title for the result otherwise.
 * @param[in]  type         Type of result.  "Alarm", etc.
 * @param[in]  description  Description of the result.
 * @param[in]  port         Result port.
 * @param[in]  severity     Result severity.
 * @param[in]  qod          Quality of detection.
 *
 * @return A result descriptor for the new result, 0 if error.
 */
result_t
make_osp_result (task_t task, const char *host, const char *nvt,
                 const char *type, const char *description,
                 const char *port, const char *severity, int qod)
{
  char *nvt_revision = NULL, *quoted_desc, *quoted_nvt, *result_severity;
  char *quoted_port;

  assert (task);
  assert (type);

  if (nvt && g_str_has_prefix (nvt, "oval:"))
    nvt_revision = ovaldef_version (nvt);
  quoted_desc = sql_quote (description ?: "");
  quoted_nvt = sql_quote (nvt ?: "");
  quoted_port = sql_quote (port ?: "");
  if (!severity || !strcmp (severity, ""))
    {
      if (!strcmp (type, severity_to_type (SEVERITY_ERROR)))
        result_severity = g_strdup (G_STRINGIFY (SEVERITY_ERROR));
      else
        {
          if (nvt && g_str_has_prefix (nvt, "CVE-"))
            {
              result_severity = cve_cvss_base (nvt);
              if (result_severity == NULL || strcmp (result_severity, "") == 0)
                {
                  g_free (result_severity);
                  result_severity
                    = g_strdup_printf ("%0.1f",
                                       setting_default_severity_dbl ());
                  g_debug ("%s: OSP CVE result without severity for '%s'",
                           __FUNCTION__, nvt);
                }
            }
          else
            {
              /*
              result_severity
                = g_strdup_printf ("%0.1f",
                                   setting_default_severity_dbl ());
              */
              g_warning ("%s: Non-CVE OSP result without severity for test %s",
                         __FUNCTION__, nvt ? nvt : "(unknown)");
              return 0;
            }
        }
    }
  else
    result_severity = sql_quote (severity);
  sql ("INSERT into results"
       " (owner, date, task, host, port, nvt, nvt_version, severity, type,"
       "  qod, qod_type, description, uuid)"
       " VALUES (NULL, m_now(), %llu, '%s', '%s', '%s', '%s', '%s', '%s',"
       "         %d, '', '%s', make_uuid ());",
       task, host ?: "", quoted_port, quoted_nvt, nvt_revision ?: "",
       result_severity ?: "0", type, qod, quoted_desc);
  g_free (result_severity);
  g_free (nvt_revision);
  g_free (quoted_desc);
  g_free (quoted_nvt);
  g_free (quoted_port);

  return sql_last_insert_id ();
}

/**
 * @brief Get QoD percentage for a qod_type string.
 *
 * @param[in]  qod_type   The QoD type string.
 *
 * @return A QoD percentage value, QOD_DEFAULT if string is NULL or unknown.
 */
int
qod_from_type (const char *qod_type)
{
  if (qod_type == NULL)
    return QOD_DEFAULT;
  else if (strcmp (qod_type, "exploit") == 0)
    return 100;
  else if  (strcmp (qod_type, "remote_vul") == 0)
    return 99;
  else if (strcmp (qod_type, "remote_app") == 0)
    return 98;
  else if (strcmp (qod_type, "package") == 0)
    return 97;
  else if (strcmp (qod_type, "registry") == 0)
    return 97;
  else if (strcmp (qod_type, "remote_active") == 0)
    return 95;
  else if (strcmp (qod_type, "remote_banner") == 0)
    return 80;
  else if (strcmp (qod_type, "executable_version") == 0)
    return 80;
  else if (strcmp (qod_type, "remote_analysis") == 0)
    return 70;
  else if (strcmp (qod_type, "remote_probe") == 0)
    return 50;
  else if (strcmp (qod_type, "remote_banner_unreliable") == 0)
    return 30;
  else if (strcmp (qod_type, "executable_version_unreliable") == 0)
    return 30;
  else if (strcmp (qod_type, "general_note") == 0)
    return 1;
  else
    return QOD_DEFAULT;
}

/**
 * @brief Identify a host, given an identifier.
 *
 * Find a host which has an identifier of the same name and value, and
 * which has no identifiers of the same name and a different value.
 *
 * @param[in]  host_name         Host name.
 * @param[in]  identifier_name   Host identifier name.
 * @param[in]  identifier_value  Value of host identifier.
 * @param[in]  source_type       Source of identification: result.
 * @param[in]  source            Source identifier.
 *
 * @return Host if exists, else 0.
 */
host_t
host_identify (const char *host_name, const char *identifier_name,
               const char *identifier_value, const char *source_type,
               const char *source)
{
  host_t host;
  gchar *quoted_host_name, *quoted_identifier_name, *quoted_identifier_value;

  quoted_host_name = sql_quote (host_name);
  quoted_identifier_name = sql_quote (identifier_name);
  quoted_identifier_value = sql_quote (identifier_value);

  switch (sql_int64 (&host,
                     "SELECT id FROM hosts"
                     " WHERE name = '%s'"
                     " AND owner = (SELECT id FROM users"
                     "              WHERE uuid = '%s')"
                     " AND (EXISTS (SELECT * FROM host_identifiers"
                     "              WHERE host = hosts.id"
                     "              AND owner = (SELECT id FROM users"
                     "                           WHERE uuid = '%s')"
                     "              AND name = '%s'"
                     "              AND value = '%s')"
                     "      OR NOT EXISTS (SELECT * FROM host_identifiers"
                     "                     WHERE host = hosts.id"
                     "                     AND owner = (SELECT id FROM users"
                     "                                  WHERE uuid = '%s')"
                     "                     AND name = '%s'));",
                     quoted_host_name,
                     current_credentials.uuid,
                     current_credentials.uuid,
                     quoted_identifier_name,
                     quoted_identifier_value,
                     current_credentials.uuid,
                     quoted_identifier_name))
    {
      case 0:
        break;
      case 1:        /* Too few rows in result of query. */
        host = 0;
        break;
      default:       /* Programming error. */
        assert (0);
      case -1:
        host = 0;
        break;
    }


  g_free (quoted_host_name);
  g_free (quoted_identifier_name);
  g_free (quoted_identifier_value);

  return host;
}

/**
 * @brief Notice a host.
 *
 * When a host is detected during a scan, this makes the decision about which
 * asset host is used for the host, as decribed in \ref asset_rules.  This
 * decision is revised at the end of the scan by \ref hosts_set_identifiers if
 * there are any identifiers for the host.
 *
 * @param[in]  host_name         Name of host.
 * @param[in]  identifier_type   Type of host identifier.
 * @param[in]  identifier_value  Value of host identifier.
 * @param[in]  source_type       Type of source identifier
 * @param[in]  source_id         Source identifier.
 * @param[in]  check_add_to_assets  Whether to check the 'Add to Assets'
 *                                  task preference.
 * @param[in]  check_for_existing_identifier  Whether to check for an existing
 *                                            identifier like this one.  Used
 *                                            for slaves, which call this
 *                                            repeatedly.
 *
 * @return Host if existed, else 0.
 */
host_t
host_notice (const char *host_name, const char *identifier_type,
             const char *identifier_value, const char *source_type,
             const char *source_id, int check_add_to_assets,
             int check_for_existing_identifier)
{
  host_t host;
  gchar *quoted_identifier_value, *quoted_identifier_type, *quoted_source_type;
  gchar *quoted_source_id;

  /* Only add to assets if "Add to Assets" is set on the task. */
  if (check_add_to_assets
      && g_str_has_prefix (source_type, "Report")
      && sql_int ("SELECT value = 'no' FROM task_preferences"
                  " WHERE task = (SELECT task FROM reports WHERE uuid = '%s')"
                  " AND name = 'in_assets';",
                  source_id))
    return 0;

  host = host_identify (host_name, identifier_type, identifier_value,
                        source_type, source_id);
  if (host == 0)
    {
      gchar *quoted_host_name;
      quoted_host_name = sql_quote (host_name);
      sql ("INSERT into hosts"
           " (uuid, owner, name, comment, creation_time, modification_time)"
           " VALUES"
           " (make_uuid (), (SELECT id FROM users WHERE uuid = '%s'), '%s', '',"
           "  m_now (), m_now ());",
           current_credentials.uuid,
           quoted_host_name);
      g_free (quoted_host_name);

      host = sql_last_insert_id ();
    }

  quoted_identifier_value = sql_quote (identifier_value);
  quoted_source_id = sql_quote (source_id);
  quoted_source_type = sql_quote (source_type);
  quoted_identifier_type = sql_quote (identifier_type);

  if (check_for_existing_identifier
      && sql_int ("SELECT EXISTS (SELECT * FROM host_identifiers"
                  "               WHERE host = %llu"
                  "               AND owner = (SELECT id FROM users WHERE uuid = '%s')"
                  "               AND name = '%s'"
                  "               AND value = '%s'"
                  "               AND source_type = '%s'"
                  "               AND source_id = '%s');",
                  host,
                  current_credentials.uuid,
                  quoted_identifier_type,
                  quoted_identifier_value,
                  quoted_source_type,
                  quoted_source_id))
    return 0;

  sql ("INSERT into host_identifiers"
       " (uuid, host, owner, name, comment, value, source_type, source_id,"
       "  source_data, creation_time, modification_time)"
       " VALUES"
       " (make_uuid (), %llu, (SELECT id FROM users WHERE uuid = '%s'), '%s',"
       "  '', '%s', '%s', '%s', '', m_now (), m_now ());",
       host,
       current_credentials.uuid,
       quoted_identifier_type,
       quoted_identifier_value,
       quoted_source_type,
       quoted_source_id);

  sql ("UPDATE hosts SET modification_time = (SELECT modification_time"
       "                                      FROM host_identifiers"
       "                                      WHERE id = %llu)"
       " WHERE id = %llu;",
       sql_last_insert_id (),
       host);

  g_free (quoted_identifier_type);
  g_free (quoted_identifier_value);
  g_free (quoted_source_id);
  g_free (quoted_source_type);

  return host;
}

/**
 * @brief Get a severity string from an nvt and result type.
 *
 * @param[in]  nvt_id   NVT oid.
 * @param[in]  type     Result type.
 *
 * @return A severity string, NULL if unknown type or no nvt id for Alarm type.
 */
char *
nvt_severity (const char *nvt_id, const char *type)
{
  char *severity = NULL;

  if (strcasecmp (type, "Alarm") == 0 && nvt_id)
    severity = sql_string ("SELECT coalesce(cvss_base, '0.0')"
                           " FROM nvts WHERE uuid = '%s';", nvt_id);
  else if (strcasecmp (type, "Alarm") == 0)
    g_warning ("%s result type requires an NVT", type);
  else if (strcasecmp (type, "Log Message") == 0)
    severity = g_strdup (G_STRINGIFY (SEVERITY_LOG));
  else if (strcasecmp (type, "Debug Message") == 0)
    severity = g_strdup (G_STRINGIFY (SEVERITY_DEBUG));
  else if (strcasecmp (type, "Error Message") == 0)
    severity = g_strdup (G_STRINGIFY (SEVERITY_ERROR));
  else
    g_warning ("Invalid result nvt type %s", type);
  return severity;
}

/**
 * @brief Make a result.
 *
 * @param[in]  task         The task associated with the result.
 * @param[in]  host         Host.
 * @param[in]  port         The port the result refers to.
 * @param[in]  nvt          The OID of the NVT that produced the result.
 * @param[in]  type         Type of result.  "Security Hole", etc.
 * @param[in]  description  Description of the result.
 *
 * @return A result descriptor for the new result, 0 if error.
 */
result_t
make_result (task_t task, const char* host, const char* port, const char* nvt,
             const char* type, const char* description)
{
  result_t result;
  gchar *nvt_revision, *severity;
  gchar *quoted_descr, *quoted_qod_type;
  int qod;
  nvt_t nvt_id = 0;

  if (nvt && strcmp (nvt, "") && (find_nvt (nvt, &nvt_id) || nvt_id <= 0))
    {
      g_warning ("NVT '%s' not found. Result not created.\n", nvt);
      return 0;
    }
  else if (nvt && strcmp (nvt, ""))
    {
      nvti_t *nvti;

      nvti = nvtis_lookup (nvti_cache, nvt);
      if (nvti)
        {
          gchar *qod_str, *qod_type;
          qod_str = tag_value (nvti_tag (nvti), "qod");
          qod_type = tag_value (nvti_tag (nvti), "qod_type");

          if (qod_str == NULL || sscanf (qod_str, "%d", &qod) != 1)
            qod = qod_from_type (qod_type);

          quoted_qod_type = sql_quote (qod_type);

          g_free (qod_str);
          g_free (qod_type);
        }
      else
        {
          qod = QOD_DEFAULT;
          quoted_qod_type = g_strdup ("");
        }

      nvt_revision = sql_string ("SELECT version FROM nvts WHERE uuid = '%s';",
                                 nvt);
    }
  else
    {
      qod = QOD_DEFAULT;
      quoted_qod_type = g_strdup ("");
      nvt_revision = g_strdup ("");
    }
  severity = nvt_severity (nvt, type);
  if (!severity)
    return 0;

  if (!strcmp (severity, ""))
    {
      g_free (severity);
      severity = g_strdup ("0.0");
    }
  quoted_descr = sql_quote (description ?: "");
  sql ("INSERT into results"
       " (owner, date, task, host, port, nvt, nvt_version, severity, type,"
       "  description, uuid, qod, qod_type)"
       " VALUES"
       " (NULL, m_now (), %llu, '%s', '%s', '%s', '%s', '%s', '%s',"
       "  '%s', make_uuid (), %i, '%s');",
       task, host ?: "", port ?: "", nvt ?: "", nvt_revision, severity, type,
       quoted_descr, qod, quoted_qod_type);

  g_free (quoted_descr);
  g_free (quoted_qod_type);
  g_free (nvt_revision);
  g_free (severity);
  result = sql_last_insert_id ();
  return result;
}

/**
 * @brief Make a CVE result.
 *
 * @param[in]  task         The task associated with the result.
 * @param[in]  host         Host.
 * @param[in]  nvt          The OID of the NVT that produced the result.
 * @param[in]  cvss         CVSS base.
 * @param[in]  description  Description of the result.
 *
 * @return A result descriptor for the new result, 0 if error.
 */
result_t
make_cve_result (task_t task, const char* host, const char *nvt, double cvss,
                 const char* description)
{
  gchar *quoted_descr;
  quoted_descr = sql_quote (description ?: "");
  sql ("INSERT into results"
       " (owner, date, task, host, port, nvt, nvt_version, severity, type,"
       "  description, uuid, qod, qod_type)"
       " VALUES"
       " (NULL, m_now (), %llu, '%s', '', '%s', '', '%1.1f', '%s',"
       "  '%s', make_uuid (), %i, '');",
       task, host ?: "", nvt, cvss, severity_to_type (cvss),
       quoted_descr, QOD_DEFAULT);

  g_free (quoted_descr);
  return sql_last_insert_id ();
}

/**
 * @brief Return the UUID of a result.
 *
 * @param[in]   result  Result.
 * @param[out]  id      Pointer to a newly allocated string.
 *
 * @return 0.
 */
int
result_uuid (result_t result, char ** id)
{
  *id = sql_string ("SELECT uuid FROM results WHERE id = %llu;",
                    result);
  return 0;
}

/**
 * @brief Get product detection results corresponding to a given vulnerability
 *        detection result.
 *
 * @param[in]   result      Vulnerability detection result.
 * @param[out]  ref         Detection result UUID.
 * @param[out]  product     Product name.
 * @param[out]  location    Product location.
 * @param[out]  oid         Detection script OID.
 * @param[out]  name        Detection script name.
 *
 * @return -1 on error, 0 on success.
 */
int
result_detection_reference (result_t result, char **ref, char **product,
                            char **location, char **oid, char **name)
{
  char *report, *host = NULL;
  gchar *quoted_location = NULL;

  if ((ref == NULL) || (product == NULL) || (location == NULL) || (oid == NULL)
      || (name == NULL))
    return -1;

  *ref = *product = *location = *oid = *name = NULL;
  report = sql_string ("SELECT report FROM results WHERE id = %llu;",
                       result);
  if (report == NULL)
    goto detect_cleanup;

  host = sql_string ("SELECT host FROM results where id = %llu;",
                     result);
  if (host == NULL)
    goto detect_cleanup;

  *oid = sql_string ("SELECT value"
                     " FROM report_host_details"
                     " WHERE report_host = (SELECT id"
                     "                      FROM report_hosts"
                     "                      WHERE report = %s"
                     "                      AND host = '%s')"
                     " AND name = 'detected_by'"
                     " AND source_name = (SELECT nvt FROM results"
                     "                    WHERE id = %llu);",
                     report, host, result);
  if (*oid == NULL)
    goto detect_cleanup;

  *location = sql_string ("SELECT value"
                          " FROM report_host_details"
                          " WHERE report_host = (SELECT id"
                          "                      FROM report_hosts"
                          "                      WHERE report = %s"
                          "                      AND host = '%s')"
                          " AND name = 'detected_at'"
                          " AND source_name = (SELECT nvt"
                          "                    FROM results"
                          "                    WHERE id = %llu);",
                          report, host, result);
  if (*location == NULL)
    goto detect_cleanup;
  quoted_location = sql_quote (*location);

  *product = sql_string ("SELECT name"
                         " FROM report_host_details"
                         " WHERE report_host = (SELECT id"
                         "                      FROM report_hosts"
                         "                      WHERE report = %s"
                         "                      AND host = '%s')"
                         " AND source_name = '%s'"
                         " AND name != 'detected_at'"
                         " AND value = '%s';",
                         report, host, *oid, quoted_location);
  if (*product == NULL)
    goto detect_cleanup;

  if (g_str_has_prefix (*oid, "CVE-"))
    *name = g_strdup (*oid);
  else
    *name = sql_string ("SELECT name FROM nvts WHERE oid = '%s';", *oid);
  if (*name == NULL)
    goto detect_cleanup;

  /* Get the result produced by the detection NVT when it detected the
   * product.  The result port or description must include the product
   * location in order for this to work. */
  *ref = sql_string ("SELECT uuid"
                     " FROM results"
                     " WHERE report = %s"
                     " AND host = '%s'"
                     " AND nvt = '%s'"
                     " AND (description LIKE '%%%s%%'"
                     "      OR port LIKE '%%%s%%');",
                     report, host, *oid, quoted_location, quoted_location);
  if (*ref == NULL)
    goto detect_cleanup;

  g_free (report);
  g_free (host);
  g_free (quoted_location);

  return 0;

detect_cleanup:
  g_free (report);
  g_free (host);
  g_free (quoted_location);

  return -1;
}



/* Prognostics. */

/**
 * @brief Return highest CVE for an App.
 *
 * @param[in]  cpe  CPE.
 */
double
cpe_highest_cvss (const char *cpe)
{
  int highest;
  gchar *quoted_cpe;
  quoted_cpe = sql_quote (cpe);
  highest = sql_double ("SELECT"
                        " (CASE WHEN EXISTS (SELECT id FROM cpes"
                        "                    WHERE name = '%s')"
                        "  THEN (SELECT max_cvss FROM cpes WHERE name = '%s')"
                        "  ELSE -1"
                        "  END);",
                        quoted_cpe,
                        quoted_cpe);
  g_free (quoted_cpe);
  return highest;
}

/**
 * @brief Prognosis iterator prepared statement.
 */
static sql_stmt_t *prognosis_stmt = NULL;

/**
 * @brief Cleanup the prognosis iterator prepared statement.
 */
void
cleanup_prognosis_iterator ()
{
  if (prognosis_stmt)
    {
      sql_finalize (prognosis_stmt);
      prognosis_stmt = NULL;
    }
}

/**
 * @brief Initialise a prognosis iterator.
 *
 * @param[in]  iterator  Iterator.
 * @param[in]  cpe       CPE.
 */
void
init_prognosis_iterator (iterator_t *iterator, const char *cpe)
{
  if (prognosis_stmt == NULL)
    prognosis_stmt = sql_prepare ("SELECT cves.name, cves.cvss,"
                                  "       cves.description, cpes.name"
                                  " FROM scap.cves, scap.cpes,"
                                  "      scap.affected_products"
                                  " WHERE cpes.name=$1"
                                  " AND cpes.id=affected_products.cpe"
                                  " AND cves.id=affected_products.cve"
                                  " ORDER BY CAST (cves.cvss AS NUMERIC)"
                                  " DESC;");
  else
    {
      if (sql_reset (prognosis_stmt))
        {
          g_warning ("%s: sql_reset failed\n", __FUNCTION__);
          abort ();
        }
    }

  if (prognosis_stmt == NULL)
    abort ();

  init_prepared_iterator (iterator, prognosis_stmt);

  /* Bind iterator. */
  if (sql_bind_text (prognosis_stmt, 1, cpe, -1))
    {
      g_warning ("%s: sql_bind_text failed\n", __FUNCTION__);
      abort ();
    }
}

DEF_ACCESS (prognosis_iterator_cve, 0);
DEF_ACCESS (prognosis_iterator_cvss, 1);
DEF_ACCESS (prognosis_iterator_description, 2);
DEF_ACCESS (prognosis_iterator_cpe, 3);

/**
 * @brief Get the CVSS from a result iterator as a double.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return CVSS.
 */
double
prognosis_iterator_cvss_double (iterator_t* iterator)
{
  if (iterator->done) return 0;
  return iterator_double (iterator, 1);
}

/**
 * @brief Get the location of an App for a report's host.
 *
 * @param[in]  report_host  Report host.
 * @param[in]  app          CPE.
 */
gchar *
app_location (report_host_t report_host, const gchar *app)
{
  gchar *quoted_app, *ret;

  assert (app);

  quoted_app = sql_quote (app);

  ret = sql_string ("SELECT value FROM report_host_details"
                    " WHERE report_host = %llu"
                    " AND name = '%s'"
                    " AND source_type = 'nvt'"
                    " AND source_name"
                    "     = (SELECT source_name FROM report_host_details"
                    "        WHERE report_host = %llu"
                    "        AND source_type = 'nvt'"
                    "        AND name = 'App'"
                    "        AND value = '%s');",
                    report_host,
                    quoted_app,
                    report_host,
                    quoted_app);

  g_free (quoted_app);

  return ret;
}

/**
 * @brief Return SQL WHERE for restricting a SELECT to a search phrase.
 *
 * @param[in]  search_phrase  Phrase that results must include.  All results if
 *                            NULL or "".
 *
 * @return WHERE clause for search phrase if one is required, else NULL.
 */
static GString *
prognosis_where_search_phrase (const char* search_phrase)
{
  if (search_phrase)
    {
      GString *phrase_sql;
      gchar *quoted_search_phrase;

      if (strlen (search_phrase) == 0)
        return NULL;

      quoted_search_phrase = sql_quote (search_phrase);
      phrase_sql = g_string_new ("");
      g_string_append_printf (phrase_sql,
                              " AND (cves.description %s '%%%%%s%%%%'"
                              " OR cves.name %s '%%%%%s%%%%'"
                              " OR cpes.name %s '%%%%%s%%%%')",
                              sql_ilike_op (),
                              quoted_search_phrase,
                              sql_ilike_op (),
                              quoted_search_phrase,
                              sql_ilike_op (),
                              quoted_search_phrase);
      g_free (quoted_search_phrase);

      return phrase_sql;
    }
  return NULL;
}

/**
 * @brief Return SQL WHERE for restricting a SELECT to levels.
 *
 * @param[in]  levels  String describing threat levels (message types)
 *                     to include in report (for example, "hmlgd" for
 *                     High, Medium, Low, loG and Debug).  All levels if
 *                     NULL.
 *
 * @return WHERE clause for levels if one is required, else NULL.
 */
static const char *
prognosis_where_levels (const char* levels)
{
  char *high, *medium, *low;

  if (levels == NULL || strlen (levels) == 0)
    return "";

  high = strchr (levels, 'h');
  medium = strchr (levels, 'm');
  low = strchr (levels, 'l');

  if (high && medium && low)
    return "";

  if (high && medium)
    return " AND cves.cvss > 2";

  if (high && low)
    return " AND (cves.cvss > 5 OR cves.cvss <= 2)";

  if (medium && low)
    return " AND cves.cvss <= 5";

  if (high)
    return " AND cves.cvss > 5";

  if (medium)
    return " AND cves.cvss <= 5 AND cves.cvss > 2";

  if (low)
    return " AND cves.cvss <= 2";

  return "";
}

/**
 * @brief Return SQL ORDER BY for sorting results in a prognostic report.
 *
 * @param[in]  sort_field  Name of the field to sort by.
 * @param[in]  ascending   Sort in ascending order if true, descending if false.
 *
 * @return ORDER BY clause.
 */
static GString *
prognosis_order_by (const char* sort_field, int ascending)
{
  GString* order_sql = g_string_new ("");

  if ((sort_field == NULL) || strcmp (sort_field, "ROWID") == 0)
    g_string_append_printf (order_sql,
                            " ORDER BY cves.id %s",
                            ascending ? "ASC" : "DESC");
  else if (strcmp (sort_field, "host") == 0)
    g_string_append_printf (order_sql,
                            " ORDER BY"
                            " order_inet"
                            "  ((SELECT host FROM report_hosts"
                            "    WHERE id = report_host_details.report_host))"
                            " %s,"
                            " severity DESC",
                            ascending ? "ASC" : "DESC");
  else if (strcmp (sort_field, "vulnerability") == 0)
    g_string_append_printf (order_sql,
                            " ORDER BY"
                            " vulnerability %s",
                            ascending ? "ASC" : "DESC");
  else if (strcmp (sort_field, "location") == 0)
    g_string_append_printf (order_sql,
                            " ORDER BY"
                            " location,"
                            " severity %s",
                            ascending ? "ASC" : "DESC");
  else
    g_string_append_printf (order_sql,
                            " ORDER BY"
                            " severity %s",
                            ascending ? "ASC" : "DESC");

  return order_sql;
}

/**
 * @brief Initialise a report host prognosis iterator.
 *
 * @param[in]  iterator     Iterator.
 * @param[in]  report_host  Report host whose prognosis the iterator loops over.
 *                          All report_hosts if NULL.
 * @param[in]  first_result   The result to start from.  The results are 0
 *                            indexed.
 * @param[in]  max_results    The maximum number of results returned.
 * @param[in]  levels         String describing threat levels (message types)
 *                            to include in count (for example, "hmlgd" for
 *                            High, Medium, Low, loG and Debug).  All levels if
 *                            NULL.
 * @param[in]  search_phrase  Phrase that results must include.  All results
 * @param[in]  sort_order     Whether to sort in ascending order.
 * @param[in]  sort_field     Name of the field to sort by.
 */
void
init_host_prognosis_iterator (iterator_t* iterator, report_host_t report_host,
                              int first_result, int max_results,
                              const char *levels, const char *search_phrase,
                              int sort_order, const char *sort_field)
{
  GString *phrase_sql, *order_sql;

  if (levels == NULL) levels = "hmlgdf";

  phrase_sql = prognosis_where_search_phrase (search_phrase);
  order_sql = prognosis_order_by (sort_field, sort_order);

  init_iterator (iterator,
                 "SELECT cves.name AS vulnerability,"
                 "       CAST (cves.cvss AS NUMERIC) AS severity,"
                 "       cves.description,"
                 "       cpes.name AS location,"
                 "       (SELECT host FROM report_hosts"
                 "        WHERE id = %llu) AS host"
                 " FROM scap.cves, scap.cpes, scap.affected_products,"
                 "      report_host_details"
                 " WHERE report_host_details.report_host = %llu"
                 " AND cpes.name = report_host_details.value"
                 " AND report_host_details.name = 'App'"
                 " AND cpes.id=affected_products.cpe"
                 " AND cves.id=affected_products.cve"
                 "%s%s%s"
                 " LIMIT %s OFFSET %i;",
                 report_host,
                 report_host,
                 phrase_sql ? phrase_sql->str : "",
                 prognosis_where_levels (levels),
                 order_sql ? order_sql->str : "",
                 sql_select_limit (max_results),
                 first_result);

  if (phrase_sql) g_string_free (phrase_sql, TRUE);
  if (order_sql) g_string_free (order_sql, TRUE);
}

/**
 * @brief Count all filtered results for a prognostic report.
 *
 * @param[in]   report_host    Report host for which to count.
 * @param[in]   search_phrase  Phrase that results must include.  All results
 * @param[out]  all            Number of messages to increment.
 * @param[out]  holes          Number of hole messages to increment.
 * @param[out]  infos          Number of info messages to increment.
 * @param[out]  warnings       Number of warning messages to increment.
 */
static void
prognostic_report_result_count (report_host_t report_host,
                                const char *search_phrase,
                                int *all, int *holes, int *infos, int *warnings)
{
  GString *phrase_sql;
  int host_holes, host_warnings, host_infos;

  phrase_sql = prognosis_where_search_phrase (search_phrase);

  host_holes = sql_int ("SELECT count (*)"
                        " FROM scap.cves, scap.cpes, scap.affected_products,"
                        "      report_host_details"
                        " WHERE report_host_details.report_host = %llu"
                        " AND cpes.name = report_host_details.value"
                        " AND report_host_details.name = 'App'"
                        " AND cpes.id=affected_products.cpe"
                        " AND cves.id=affected_products.cve"
                        "%s%s;",
                        report_host,
                        phrase_sql ? phrase_sql->str : "",
                        prognosis_where_levels ("h"));

  host_warnings = sql_int ("SELECT count (*)"
                           " FROM scap.cves, scap.cpes, scap.affected_products,"
                           "      report_host_details"
                           " WHERE report_host_details.report_host = %llu"
                           " AND cpes.name = report_host_details.value"
                           " AND report_host_details.name = 'App'"
                           " AND cpes.id=affected_products.cpe"
                           " AND cves.id=affected_products.cve"
                           "%s%s;",
                           report_host,
                           phrase_sql ? phrase_sql->str : "",
                           prognosis_where_levels ("m"));

  host_infos = sql_int ("SELECT count (*)"
                        " FROM scap.cves, scap.cpes, scap.affected_products,"
                        "      report_host_details"
                        " WHERE report_host_details.report_host = %llu"
                        " AND cpes.name = report_host_details.value"
                        " AND report_host_details.name = 'App'"
                        " AND cpes.id=affected_products.cpe"
                        " AND cves.id=affected_products.cve"
                        "%s%s;",
                        report_host,
                        phrase_sql ? phrase_sql->str : "",
                        prognosis_where_levels ("l"));

  *all += host_holes + host_warnings + host_infos;

  *holes += host_holes;
  *warnings += host_warnings;
  *infos += host_infos;

  if (phrase_sql) g_string_free (phrase_sql, TRUE);
}

/**
 * @brief Count total number of results for a prognostic report.
 *
 * @param[in]   report_host    Report host for which to count.
 * @param[out]  total          Total count of all results.
 */
static void
prognostic_report_result_total (report_host_t report_host, int *total)
{
  if (total)
    *total = sql_int ("SELECT count (*)"
                      " FROM scap.cves, scap.cpes, scap.affected_products,"
                      "      report_host_details"
                      " WHERE report_host_details.report_host = %llu"
                      " AND cpes.name = report_host_details.value"
                      " AND report_host_details.name = 'App'"
                      " AND cpes.id=affected_products.cpe"
                      " AND cves.id=affected_products.cve;",
                      report_host);
}


/* Reports. */

/**
 * @brief Whether to ignore the Max Rows Per Page settings.
 */
int ignore_max_rows_per_page = 0;

/**
 * @brief Clear all cached report result counts.
 *
 * @param[in]  override  Flag for override or regular case.
 */
void
reports_clear_count_cache (int override)
{
  sql ("DELETE FROM report_counts WHERE override = %d;",
       override);
}

/**
 * @brief Create a new GHashTable for containing resource rowids.
 *
 * @return The newly allocated GHashTable
 */
GHashTable *
new_resources_hashtable ()
{
  return g_hash_table_new_full (g_int64_hash, g_int64_equal, g_free, NULL);
}

/**
 * @brief Add reports affected by an override to an existing GHashtable.
 * This is used to add more reports to the hashtable from reports_for_override.
 *
 * @param[in]  reports_table The GHashtable to contain the report rowids.
 * @param[in]  override The override that selected reports must be affected by.
 */
void
reports_add_for_override (GHashTable *reports_table,
                          override_t override)
{
  result_t result;
  task_t task;
  gchar *nvt_id;
  iterator_t reports;

  if (override == 0)
    return;

  sql_int64 (&result,
             "SELECT result FROM overrides WHERE id = %llu",
             override);
  sql_int64 (&task,
             "SELECT task FROM overrides WHERE id = %llu",
             override);
  nvt_id = sql_string ("SELECT nvt FROM overrides WHERE id = %llu",
                       override);

  if (result)
    {
      report_t *report = g_malloc0 (sizeof (report_t));
      sql_int64 (report,
                 "SELECT report FROM results"
                 " WHERE id = %llu AND nvt = '%s'",
                 result, nvt_id);

      if (*report)
        g_hash_table_add (reports_table, report);
      else
        g_free (report);

      return;
    }
  else if (task)
    {
      init_iterator (&reports,
                     "SELECT DISTINCT report FROM results"
                     " WHERE task = %llu AND nvt = '%s'",
                     task, nvt_id);
    }
  else
    {
      init_iterator (&reports,
                     "SELECT DISTINCT report FROM results"
                     " WHERE nvt = '%s'",
                     nvt_id);
    }

  while (next (&reports))
    {
      report_t *report = g_malloc0 (sizeof (report_t));

      *report = iterator_int64 (&reports, 0);

      if (g_hash_table_contains (reports_table, report) == 0)
        g_hash_table_add (reports_table, report);
      else
        g_free (report);
    }
  cleanup_iterator (&reports);
}

/**
 * @brief Get reports affected by an override in a GHashTable.
 *
 * @param[in]  override The override that selected reports must be affected by.
 *
 * @return A GHashtable containing the affected report rowids.
 */
GHashTable *
reports_for_override (override_t override)
{
  GHashTable *reports_table;
  reports_table = new_resources_hashtable ();

  reports_add_for_override (reports_table, override);

  return reports_table;
}

/**
 * @brief Add all reports to an existing GHashtable.
 *
 * @param[in]  reports_table The GHashtable to contain the report rowids.
 */
void
reports_add_all (GHashTable *reports_table)
{
  iterator_t reports;

  init_iterator (&reports,
                 "SELECT id FROM reports");

  while (next (&reports))
    {
      report_t *report = g_malloc0 (sizeof (report_t));

      *report = iterator_int64 (&reports, 0);

      if (g_hash_table_contains (reports_table, report) == 0)
        g_hash_table_add (reports_table, report);
      else
        g_free (report);
    }

  cleanup_iterator (&reports);
}

/**
 * @brief Get all reports in a GHashTable.
 *
 * @return A GHashtable containing the report rowids.
 */
GHashTable *
reports_hashtable ()
{
  GHashTable *reports_table;
  reports_table = new_resources_hashtable ();

  reports_add_all (reports_table);

  return reports_table;
}

/**
 * @brief Rebuild the report count cache for all reports and users.
 *
 * @param[in]  clear        Whether to clear the cache before rebuilding.
 * @param[out] changes_out  The number of processed user/report combinations.
 */
void
reports_build_count_cache (int clear, int* changes_out)
{
  int changes;
  iterator_t reports;
  changes = 0;

  init_iterator (&reports, "SELECT id FROM reports");

  while (next (&reports))
    {
      report_t report = iterator_int64 (&reports, 0);

      report_cache_counts (report, clear, clear, NULL);
      changes ++;
    }

  cleanup_iterator (&reports);

  if (changes_out)
    *changes_out = changes;
}

/**
 * @brief Initializes an iterator for updating the report cache
 *
 * @param[in]  iterator       Iterator.
 * @param[in]  report         Report to select.
 * @param[in]  min_qod_limit  Limit for min_qod.
 * @param[in]  add_defaults   Whether to add default values.
 * @param[in]  users_where    Optional SQL clause to limit users.
 */
void
init_report_counts_build_iterator (iterator_t *iterator, report_t report,
                                   int min_qod_limit, int add_defaults,
                                   const char *users_where)
{
  gchar *report_id, *users_string;

  report_id = sql_string ("SELECT uuid FROM reports WHERE id = %llu;", report);

  users_string = acl_users_with_access_sql ("report", report_id, users_where);
  if (users_string == NULL)
    {
      init_iterator (iterator, "SELECT NULL WHERE NOT t();");
      g_free (report_id);
      return;
    }

  if (add_defaults && MIN_QOD_DEFAULT <= min_qod_limit)
    {
      init_iterator (iterator,
                     "SELECT * FROM"
                     " (WITH users_with_access (\"user\") AS %s"
                     "  SELECT DISTINCT min_qod, override, \"user\""
                     "  FROM report_counts"
                     "  WHERE report = %llu"
                     "    AND \"user\" IN (SELECT \"user\""
                     "                     FROM users_with_access)"
                     "    AND min_qod <= %d"
                     "  UNION SELECT 0 as min_qod, 0, \"user\""
                     "          FROM users_with_access"
                     "  UNION SELECT 0 as min_qod, 1, \"user\""
                     "          FROM users_with_access"
                     "  UNION SELECT %d as min_qod, 0, \"user\""
                     "          FROM users_with_access"
                     "  UNION SELECT %d as min_qod, 1, \"user\""
                     "          FROM users_with_access) AS inner_query"
                     " ORDER BY \"user\"",
                     users_string,
                     report,
                     min_qod_limit,
                     MIN_QOD_DEFAULT,
                     MIN_QOD_DEFAULT);
    }
  else if (add_defaults)
    {
      init_iterator (iterator,
                     "SELECT * FROM"
                     " (WITH users_with_access (\"user\") AS %s"
                     "  SELECT DISTINCT min_qod, override, \"user\""
                     "  FROM report_counts"
                     "  WHERE report = %llu"
                     "    AND min_qod <= %d"
                     "    AND \"user\" IN (SELECT \"user\""
                     "                     FROM users_with_access)"
                     "  UNION SELECT 0 as min_qod, 0, \"user\""
                     "          FROM users_with_access"
                     "  UNION SELECT 0 as min_qod, 1, \"user\""
                     "          FROM users_with_access) AS inner_query"
                     " ORDER BY \"user\"",
                     users_string,
                     report,
                     min_qod_limit);
    }
  else
    {
      init_iterator (iterator,
                     "WITH users_with_access (\"user\") AS %s"
                     " SELECT DISTINCT min_qod, override, \"user\""
                     " FROM report_counts"
                     " WHERE report = %llu"
                     "   AND min_qod <= %d"
                     "   AND \"user\" IN (SELECT \"user\""
                     "                    FROM users_with_access)"
                     " ORDER BY \"user\"",
                     users_string,
                     report,
                     min_qod_limit);
    }
  g_free (users_string);
  g_free (report_id);
}

/**
 * @brief Get the min_qod from a report_counts build iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The min_qod.
 */
int
report_counts_build_iterator_min_qod (iterator_t *iterator)
{
  return iterator_int (iterator, 0);
}

/**
 * @brief Get the override flag from a report_counts build iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return Whether the report counts are using overrides.
 */
int
report_counts_build_iterator_override (iterator_t *iterator)
{
  return iterator_int (iterator, 1);
}

/**
 * @brief Get the user from a report_counts build iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The min_qod.
 */
user_t
report_counts_build_iterator_user (iterator_t *iterator)
{
  return iterator_int64 (iterator, 2);
}

/**
 * @brief Cache report counts and clear existing caches if requested.
 *
 * If queue_rebuild is true, the cache will not be rebuilt immediately, but
 *  the rebuild will be added to a queue to be handled in a separate process.
 *
 * @param[in]  report             Report to cache counts of.
 * @param[in]  clear_original     Whether to clear existing cache for
 *                                 original severity.
 * @param[in]  clear_overridden   Whether to clear existing cache for
 *                                 overridden severity.
 * @param[in]  users_where        Optional SQL clause to limit users.
 */
void
report_cache_counts (report_t report, int clear_original, int clear_overridden,
                     const char* users_where)
{
  iterator_t cache_iterator;
  int debugs, holes, infos, logs, warnings, false_positives;
  double severity;
  get_data_t *get = NULL;
  gchar *old_user_id;

  old_user_id = current_credentials.uuid;
  init_report_counts_build_iterator (&cache_iterator, report, INT_MAX, 1,
                                     users_where);

  while (next (&cache_iterator))
    {
      int override = report_counts_build_iterator_override (&cache_iterator);
      int min_qod = report_counts_build_iterator_min_qod (&cache_iterator);
      user_t user = report_counts_build_iterator_user (&cache_iterator);

      current_credentials.uuid
        = sql_string ("SELECT uuid FROM users WHERE id = %llu",
                      user);
      manage_session_init (current_credentials.uuid);

      get = report_results_get_data (1, -1, override, 0, min_qod);

      if ((clear_original && override == 0) || (clear_overridden && override))
        {
          sql ("DELETE FROM report_counts"
               " WHERE report = %llu"
               "   AND \"user\" = %llu"
               "   AND override = %d"
               "   AND min_qod = %d",
               report, user, override, min_qod);
        }

      report_counts_id (report, &debugs, &holes, &infos, &logs, &warnings,
                        &false_positives, &severity, get, NULL);

      get_data_reset (get);
      g_free (get);
      g_free (current_credentials.uuid);
    }
  cleanup_iterator (&cache_iterator);
  current_credentials.uuid = old_user_id;
  manage_session_init (current_credentials.uuid);
}

/**
 * @brief Clear report counts .
 *
 * @param[in]  report  Report.
 * @param[in]  clear_original     Whether to clear existing cache for
 *                                 original severity.
 * @param[in]  clear_overridden   Whether to clear existing cache for
 *                                 overridden severity.
 * @param[in]  users_where        Optional SQL clause to limit users.
 */
void
report_clear_count_cache (report_t report,
                          int clear_original, int clear_overridden,
                          const char* users_where)
{
  gchar *extra_where = NULL;
  if (users_where)
    {
      extra_where 
        = g_strdup_printf (" AND \"user\" IN (SELECT id FROM users WHERE %s)",
                           users_where);
    }

  if (clear_original && clear_overridden)
    {
      sql ("DELETE FROM report_counts"
           " WHERE report = %llu"
           "%s",
           report,
           extra_where ? extra_where : "");
    }
  else if (clear_original || clear_overridden)
    {
      int override = clear_overridden ? 1 : 0;
      sql ("DELETE FROM report_counts"
           " WHERE report = %llu"
           "   AND override = %d"
           "%s",
           report,
           override,
           extra_where ? extra_where : "");
    }
}

/**
 * @brief Make a report.
 *
 * @param[in]  task    The task associated with the report.
 * @param[in]  uuid    The UUID of the report.
 * @param[in]  status  The run status of the scan associated with the report.
 *
 * @return A report descriptor for the new report.
 */
report_t
make_report (task_t task, const char* uuid, task_status_t status)
{
  sql ("INSERT into reports (uuid, owner, hidden, task, date, nbefile, comment,"
       " scan_run_status, slave_progress, slave_task_uuid)"
       " VALUES ('%s',"
       " (SELECT owner FROM tasks WHERE tasks.id = %llu),"
       " 0, %llu, %i, '', '', %u, 0, '');",
       uuid, task, task, time (NULL), status);
  return sql_last_insert_id ();
}

/**
 * @brief Create the current report for a task.
 *
 * @param[in]   task       The task.
 * @param[out]  report_id  Report ID.
 * @param[in]   status     Run status of scan associated with report.
 *
 * @return 0 success, -1 current_report is already set, -2 failed to generate ID.
 */
int
create_current_report (task_t task, char **report_id, task_status_t status)
{
  char *id;

  assert (current_report == (report_t) 0);

  if (current_report) return -1;

  if (report_id == NULL) report_id = &id;

  /* Generate report UUID. */

  *report_id = openvas_uuid_make ();
  if (*report_id == NULL) return -2;

  /* Create the report. */

  current_report = make_report (task, *report_id, status);

  set_report_scheduled (current_report);

  return 0;
}

/**
 * @brief Free a host detail.
 *
 * @param[in]  detail  Host detail.
 */
void
host_detail_free (host_detail_t *detail)
{
  g_free (detail->ip);
  g_free (detail->name);
  g_free (detail->source_desc);
  g_free (detail->source_name);
  g_free (detail->source_type);
  g_free (detail->value);
}

/**
 * @brief Insert a host detail into a report.
 *
 * @param[in]   report      The detail's report.
 * @param[in]   host        The detail's host.
 * @param[in]   s_type      The detail's source type.
 * @param[in]   s_name      The detail's source name.
 * @param[in]   s_desc      The detail's source description.
 * @param[in]   name        The detail's name.
 * @param[in]   value       The detail's value.
 */
void
insert_report_host_detail (report_t report, const char *host,
                           const char *s_type, const char *s_name,
                           const char *s_desc, const char *name,
                           const char *value)
{
  char *quoted_host, *quoted_source_name, *quoted_source_type;
  char *quoted_source_desc, *quoted_name, *quoted_value;

  quoted_host = sql_quote (host);
  quoted_source_type = sql_quote (s_type);
  quoted_source_name = sql_quote (s_name);
  quoted_source_desc = sql_quote (s_desc);
  quoted_name = sql_quote (name);
  quoted_value = sql_quote (value);
  sql ("INSERT INTO report_host_details"
       " (report_host, source_type, source_name, source_description,"
       "  name, value)"
       " VALUES"
       " ((SELECT id FROM report_hosts"
       "   WHERE report = %llu AND host = '%s'),"
       "  '%s', '%s', '%s', '%s', '%s');",
       report, quoted_host, quoted_source_type, quoted_source_name,
       quoted_source_desc, quoted_name, quoted_value);

  g_free (quoted_host);
  g_free (quoted_source_type);
  g_free (quoted_source_name);
  g_free (quoted_source_desc);
  g_free (quoted_name);
  g_free (quoted_value);
}

/**
 * @brief Number of results per transaction, when uploading report.
 */
#define CREATE_REPORT_CHUNK_SIZE 10

/**
 * @brief Number of microseconds to sleep between insert chunks.
 */
#define CREATE_REPORT_CHUNK_SLEEP 1000

/**
 * @brief Create a report from an array of results.
 *
 * @param[in]   results       Array of create_report_result_t pointers.
 * @param[in]   task_id       UUID of container task, or NULL to create new one.
 * @param[in]   task_name     Name for container task.
 * @param[in]   task_comment  Comment for container task.
 * @param[in]   in_assets     Whether to create assets from the report.
 * @param[in]   scan_start    Scan start time text.
 * @param[in]   scan_end      Scan end time text.
 * @param[in]   host_starts   Array of create_report_result_t pointers.  Host
 *                            name in host, time in description.
 * @param[in]   host_ends     Array of create_report_result_t pointers.  Host
 *                            name in host, time in description.
 * @param[in]   details       Array of host_detail_t pointers.
 * @param[out]  report_id     Report ID.
 *
 * @return 0 success, 99 permission denied, -1 error, -2 failed to generate ID,
 *         -3 task_name is NULL, -4 failed to find task, -5 task must be
 *         container, -6 permission to create assets denied.
 */
int
create_report (array_t *results, const char *task_id, const char *task_name,
               const char *task_comment, const char *in_assets,
               const char *scan_start, const char *scan_end,
               array_t *host_starts, array_t *host_ends, array_t *details,
               char **report_id)
{
  int index, in_assets_int, count;
  create_report_result_t *result, *end, *start;
  report_t report;
  user_t owner;
  task_t task;
  pid_t pid;
  host_detail_t *detail;

  in_assets_int
    = (in_assets && strcmp (in_assets, "") && strcmp (in_assets, "0"));

  if (in_assets_int && acl_user_may ("create_asset") == 0)
    return -6;

  g_debug ("%s", __FUNCTION__);

  if (acl_user_may ("create_report") == 0)
    return 99;

  if (task_id == NULL && task_name == NULL)
    return -3;

  /* Find or create the task. */

  sql_begin_immediate ();
  if (task_id)
    {
      int rc = 0;

      if (find_task (task_id, &task))
        rc = -1;
      else if (task == 0)
        rc = -4;
      else if (task_target (task))
        rc = -5;
      if (rc)
        {
          sql_rollback ();
          return rc;
        }
    }
  else
    task = make_task (g_strdup (task_name),
                      task_comment ? g_strdup (task_comment) : NULL,
                      1,  /* Include in assets. */
                      1); /* Log and generate event. */

  /* Generate report UUID. */

  *report_id = openvas_uuid_make ();
  if (*report_id == NULL) return -2;

  /* Create the report. */

  report = make_report (task, *report_id, TASK_STATUS_RUNNING);

  if (scan_start)
    {
      sql ("UPDATE reports SET start_time = %i WHERE id = %llu;",
           parse_iso_time (scan_start),
           report);
    }

  if (scan_end)
    {
      sql ("UPDATE reports SET end_time = %i WHERE id = %llu;",
           parse_iso_time (scan_end),
           report);
    }

  /* Show that the upload has started. */

  set_task_run_status (task, TASK_STATUS_RUNNING);
  sql ("UPDATE tasks SET upload_result_count = %llu WHERE id = %llu;",
       results->len,
       task);
  sql_commit ();

  /* Fork a child to import the results while the parent responds to the
   * client. */

  pid = fork ();
  switch (pid)
    {
      case 0:
        /* Child.  Reopen the database (required after fork) and carry on
         * to import the reports, . */
        reinit_manage_process ();
        break;
      case -1:
        /* Parent when error. */
        g_warning ("%s: fork: %s\n", __FUNCTION__, strerror (errno));
        set_task_run_status (task, TASK_STATUS_INTERNAL_ERROR);
        return -1;
        break;
      default:
        /* Parent.  Return, in order to respond to client. */
        g_debug ("%s: %i forked %i", __FUNCTION__, getpid (), pid);
        return 0;
        break;
    }

  /* Add the results. */

  if (sql_int64 (&owner,
                 "SELECT owner FROM tasks WHERE tasks.id = %llu",
                 task))
    {
      g_warning ("%s: failed to get owner of task\n", __FUNCTION__);
      return -1;
    }

  sql_begin_immediate ();
  g_debug ("%s: add hosts", __FUNCTION__);
  index = 0;
  count = 0;
  while ((start = (create_report_result_t*) g_ptr_array_index (host_starts,
                                                               index++)))
    if (start->host && start->description)
      manage_report_host_add (report, start->host,
                              parse_iso_time (start->description),
                              0);

  g_debug ("%s: add results", __FUNCTION__);
  index = 0;
  while ((result = (create_report_result_t*) g_ptr_array_index (results,
                                                                index++)))
    {
      gchar *quoted_host, *quoted_port, *quoted_nvt_oid;
      gchar *quoted_description, *quoted_scan_nvt_version, *quoted_severity;
      gchar *quoted_qod, *quoted_qod_type;

      g_debug ("%s: add results: index: %i", __FUNCTION__, index);

      quoted_host = sql_quote (result->host ? result->host : "");
      quoted_port = sql_quote (result->port ? result->port : "");
      quoted_nvt_oid = sql_quote (result->nvt_oid ? result->nvt_oid : "");
      quoted_description = sql_quote (result->description
                                       ? result->description
                                       : "");
      quoted_scan_nvt_version = sql_quote (result->scan_nvt_version
                                       ? result->scan_nvt_version
                                       : "");
      quoted_severity =  sql_quote (result->severity ? result->severity : "");
      if (result->qod && strcmp (result->qod, "") && strcmp (result->qod, "0"))
        quoted_qod = sql_quote (result->qod);
      else
        quoted_qod = g_strdup (G_STRINGIFY (QOD_DEFAULT));
      quoted_qod_type = sql_quote (result->qod_type ? result->qod_type : "");
      sql ("INSERT INTO results"
           " (uuid, owner, date, task, host, port, nvt, type, description,"
           "  nvt_version, severity, qod, qod_type)"
           " VALUES"
           " (make_uuid (), %llu, m_now (), %llu, '%s', '%s', '%s', '%s', '%s',"
           "  '%s', '%s', '%s', '%s');",
           owner,
           task,
           quoted_host,
           quoted_port,
           quoted_nvt_oid,
           result->threat
            ? threat_message_type (result->threat)
            : "Log Message",
           quoted_description,
           quoted_scan_nvt_version,
           quoted_severity,
           quoted_qod,
           quoted_qod_type);

      g_free (quoted_host);
      g_free (quoted_port);
      g_free (quoted_nvt_oid);
      g_free (quoted_description);
      g_free (quoted_scan_nvt_version);
      g_free (quoted_severity);
      g_free (quoted_qod);
      g_free (quoted_qod_type);

      report_add_result (report, sql_last_insert_id ());

      count++;
      if (count == CREATE_REPORT_CHUNK_SIZE)
        {
          sql_commit ();
          openvas_usleep (CREATE_REPORT_CHUNK_SLEEP);
          sql_begin_immediate ();
          count = 0;
        }
    }

  g_debug ("%s: add host ends", __FUNCTION__);
  index = 0;
  count = 0;
  while ((end = (create_report_result_t*) g_ptr_array_index (host_ends,
                                                             index++)))
    if (end->host)
      {
        gchar *quoted_host;

        quoted_host = sql_quote (end->host);

        if (end->description)
          sql ("UPDATE report_hosts SET end_time = %i"
               " WHERE report = %llu AND host = '%s';",
               parse_iso_time (end->description),
               report,
               quoted_host);
        else
          sql ("UPDATE report_hosts SET end_time = NULL"
               " WHERE report = %llu AND host = '%s';",
               report,
               quoted_host);

        g_free (quoted_host);

        count++;
        if (count == CREATE_REPORT_CHUNK_SIZE)
          {
            sql_commit ();
            openvas_usleep (CREATE_REPORT_CHUNK_SLEEP);
            sql_begin_immediate ();
            count = 0;
          }
      }

  g_debug ("%s: add host details", __FUNCTION__);
  index = 0;
  while ((detail = (host_detail_t*) g_ptr_array_index (details, index++)))
    if (detail->ip && detail->name)
      insert_report_host_detail
       (report, detail->ip, detail->source_type ?: "",
        detail->source_name ?: "", detail->source_desc ?: "", detail->name,
        detail->value ?: "");
  sql_commit ();

  current_scanner_task = task;
  current_report = report;
  set_task_run_status (task, TASK_STATUS_DONE);
  current_scanner_task = 0;
  current_report = 0;

  if (in_assets_int)
    {
      create_asset_report (*report_id, "");
    }

  exit (EXIT_SUCCESS);
  return 0;
}

/**
 * @brief Return the UUID of a report.
 *
 * @param[in]  report  Report.
 *
 * @return Report UUID.
 */
char*
report_uuid (report_t report)
{
  return sql_string ("SELECT uuid FROM reports WHERE id = %llu;",
                     report);
}

/**
 * @brief Return the task of a report.
 *
 * @param[in]   report  A report.
 * @param[out]  task    Task return, 0 if succesfully failed to find task.
 *
 * @return FALSE on success (including if failed to find report), TRUE on error.
 */
gboolean
report_task (report_t report, task_t *task)
{
  switch (sql_int64 (task,
                     "SELECT task FROM reports WHERE id = %llu;",
                     report))
    {
      case 0:
        break;
      case 1:        /* Too few rows in result of query. */
        *task = 0;
        break;
      default:       /* Programming error. */
        assert (0);
      case -1:
        return TRUE;
        break;
    }
  return FALSE;
}

/**
 * @brief Return the UUID of a report's slave.
 *
 * @param[in]  report  Report.
 *
 * @return Slave UUID.
 */
char*
report_slave_uuid (report_t report)
{
  return sql_string ("SELECT slave_uuid FROM reports WHERE id = %llu;",
                     report);
}

/**
 * @brief Return the name of a report's slave.
 *
 * @param[in]  report  Report.
 *
 * @return Slave name.
 */
char*
report_slave_name (report_t report)
{
  return sql_string ("SELECT slave_name FROM reports WHERE id = %llu;",
                     report);
}

/**
 * @brief Return the host of a report's slave.
 *
 * @param[in]  report  Report.
 *
 * @return Slave UUID.
 */
char*
report_slave_host (report_t report)
{
  return sql_string ("SELECT slave_host FROM reports WHERE id = %llu;",
                     report);
}

/**
 * @brief Return the port of a report's slave.
 *
 * @param[in]  report  Report.
 *
 * @return Slave port.
 */
char*
report_slave_port (report_t report)
{
  return sql_string ("SELECT slave_port FROM reports WHERE id = %llu;",
                     report);
}

/**
 * @brief Return the port of a report's slave.
 *
 * @param[in]  report  Report.
 *
 * @return Slave port.
 */
int
report_slave_port_int (report_t report)
{
  return sql_int ("SELECT slave_port FROM reports WHERE id = %llu;",
                  report);
}

/**
 * @brief Return the source interface of a report.
 *
 * @param[in]  report  Report.
 *
 * @return Source interface.
 */
char*
report_source_iface (report_t report)
{
  return sql_string ("SELECT source_iface FROM reports WHERE id = %llu;",
                     report);
}

/**
 * @brief Set the UUID of the slave on a report.
 *
 * @param[in]  report  Report.
 * @param[in]  uuid    UUID.
 */
void
report_set_slave_uuid (report_t report, const gchar *uuid)
{
  gchar *quoted_uuid;
  quoted_uuid = sql_quote (uuid);
  sql ("UPDATE reports SET slave_uuid = '%s' WHERE id = %llu;",
       quoted_uuid,
       report);
  g_free (quoted_uuid);
}

/**
 * @brief Set the name of the slave on a report.
 *
 * @param[in]  report  Report.
 * @param[in]  name    Name.
 */
void
report_set_slave_name (report_t report, const gchar *name)
{
  gchar *quoted_name;
  quoted_name = sql_quote (name);
  sql ("UPDATE reports SET slave_name = '%s' WHERE id = %llu;",
       quoted_name,
       report);
  g_free (quoted_name);
}

/**
 * @brief Set the host of the slave of a report.
 *
 * @param[in]  report  Report.
 * @param[in]  host    Host.
 */
void
report_set_slave_host (report_t report, const gchar *host)
{
  gchar *quoted_host;
  quoted_host = sql_quote (host);
  sql ("UPDATE reports SET slave_host = '%s' WHERE id = %llu;",
       quoted_host,
       report);
  g_free (quoted_host);
}

/**
 * @brief Set the port of the slave of a report.
 *
 * @param[in]  report  Report.
 * @param[in]  port    Port.
 */
void
report_set_slave_port (report_t report, int port)
{
  sql ("UPDATE reports SET slave_port = %i WHERE id = %llu;",
       port,
       report);
}

/**
 * @brief Set the source interface of a report.
 *
 * @param[in]  report  Report.
 * @param[in]  iface   Source interface.
 */
void
report_set_source_iface (report_t report, const gchar *iface)
{
  gchar *quoted_iface;
  quoted_iface = sql_quote (iface);
  sql ("UPDATE reports SET source_iface = '%s' WHERE id = %llu;",
       quoted_iface,
       report);
  g_free (quoted_iface);
}

/**
 * @brief Add a result to a report.
 *
 * @param[in]  report  The report.
 * @param[in]  result  The result.
 */
void
report_add_result (report_t report, result_t result)
{
  double severity, ov_severity;
  int qod;
  rowid_t rowid;
  iterator_t cache_iterator;
  user_t previous_user = 0;

  if (report == 0 || result == 0)
    return;

  sql ("UPDATE results SET report = %llu,"
       "                   owner = (SELECT reports.owner"
       "                            FROM reports WHERE id = %llu)"
       " WHERE id = %llu;",
       report, report, result);

  qod = sql_int ("SELECT qod FROM results WHERE id = %llu;",
                 result);

  severity = sql_double ("SELECT severity FROM results WHERE id = %llu;",
                         result);
  ov_severity = severity;

  init_report_counts_build_iterator (&cache_iterator, report, qod, 1, NULL);
  while (next (&cache_iterator))
    {
      int min_qod = report_counts_build_iterator_min_qod (&cache_iterator);
      int override = report_counts_build_iterator_override (&cache_iterator);
      user_t user = report_counts_build_iterator_user (&cache_iterator);

      if (override && user != previous_user)
        {
          char *ov_severity_str;
          gchar *owned_clause;

          owned_clause = acl_where_owned_for_get ("override", NULL);

          ov_severity_str
            = sql_string ("SELECT coalesce (overrides.new_severity, %1.1f)"
                          " FROM overrides, results"
                          " WHERE results.id = %llu"
                          " AND overrides.nvt = results.nvt"
                          " AND %s"
                          " AND ((overrides.end_time = 0)"
                          "      OR (overrides.end_time >= m_now ()))"
                          " AND (overrides.task ="
                          "      (SELECT reports.task FROM reports"
                          "       WHERE reports.id = %llu)"
                          "      OR overrides.task = 0)"
                          " AND (overrides.result = results.id"
                          "      OR overrides.result = 0)"
                          " AND (overrides.hosts is NULL"
                          "      OR overrides.hosts = ''"
                          "      OR hosts_contains (overrides.hosts,"
                          "                         results.host))"
                          " AND (overrides.port is NULL"
                          "      OR overrides.port = ''"
                          "      OR overrides.port = results.port)"
                          " AND severity_matches_ov (%1.1f,"
                          "                          overrides.severity)"
                          " ORDER BY overrides.result DESC,"
                          "   overrides.task DESC, overrides.port DESC,"
                          "   overrides.severity ASC,"
                          "   overrides.creation_time DESC"
                          " LIMIT 1",
                          severity,
                          result,
                          owned_clause,
                          report,
                          severity);

          g_free (owned_clause);

          if (ov_severity_str == NULL
              || (sscanf (ov_severity_str, "%lf", &ov_severity) != 1))
            ov_severity = severity;

          free (ov_severity_str);

          previous_user = user;
        }

      rowid = 0;
      sql_int64 (&rowid,
                 "SELECT id FROM report_counts"
                 " WHERE report = %llu"
                 " AND \"user\" = %llu"
                 " AND override = %d"
                 " AND severity = %1.1f"
                 " AND min_qod = %d",
                 report, user, override,
                 override ? ov_severity : severity,
                 min_qod);
      if (rowid)
        sql ("UPDATE report_counts"
            " SET count = count + 1"
            " WHERE id = %llu;",
            rowid);
      else
        sql ("INSERT INTO report_counts"
             " (report, \"user\", override, min_qod, severity, count, end_time)"
             " VALUES"
             " (%llu, %llu, %d, %d, %1.1f, 1, 0);",
             report, user, override,
             override ? ov_severity : severity,
             min_qod);

    }
  cleanup_iterator (&cache_iterator);

  sql ("UPDATE report_counts"
       " SET end_time = (SELECT coalesce(min(overrides.end_time), 0)"
       "                 FROM overrides, results"
       "                 WHERE overrides.nvt = results.nvt"
       "                 AND results.report = %llu"
       "                 AND overrides.end_time >= m_now ())"
       " WHERE report = %llu AND override = 1;",
       report, report);
}

/**
 * @brief Filter columns for report iterator.
 */
#define REPORT_ITERATOR_FILTER_COLUMNS                                         \
 { ANON_GET_ITERATOR_FILTER_COLUMNS, "task_id", "name", "date", "status",      \
   "task", "severity", "false_positive", "log", "low", "medium", "high",       \
   "hosts", "result_hosts", "fp_per_host", "log_per_host", "low_per_host",     \
   "medium_per_host", "high_per_host", "duration", "duration_per_host",        \
   NULL }

/**
 * @brief Report iterator columns.
 */
#define REPORT_ITERATOR_COLUMNS                                              \
 {                                                                           \
   { "id", NULL, KEYWORD_TYPE_INTEGER },                                     \
   { "uuid", NULL, KEYWORD_TYPE_STRING },                                    \
   { "iso_time (start_time)", "name", KEYWORD_TYPE_STRING },                 \
   { "''", NULL, KEYWORD_TYPE_STRING },                                      \
   { "iso_time (start_time)", NULL, KEYWORD_TYPE_STRING },                   \
   { "iso_time (end_time)", NULL, KEYWORD_TYPE_STRING },                     \
   { "start_time", "created", KEYWORD_TYPE_INTEGER },                        \
   { "end_time", "modified", KEYWORD_TYPE_INTEGER },                         \
   { "(SELECT name FROM users WHERE users.id = reports.owner)",              \
     "_owner",                                                               \
     KEYWORD_TYPE_STRING },                                                  \
   { "owner", NULL, KEYWORD_TYPE_INTEGER },                                  \
   { NULL, NULL, KEYWORD_TYPE_UNKNOWN }                                      \
 }

/**
 * @brief Report iterator columns.
 */
#define REPORT_ITERATOR_WHERE_COLUMNS                                        \
 {                                                                           \
   { "run_status_name (scan_run_status)", "status", KEYWORD_TYPE_STRING },   \
   {                                                                         \
     "(SELECT uuid FROM tasks WHERE tasks.id = task)",                       \
     "task_id",                                                              \
     KEYWORD_TYPE_STRING                                                     \
   },                                                                        \
   { "date", NULL, KEYWORD_TYPE_INTEGER },                                   \
   { "(SELECT name FROM tasks WHERE tasks.id = task)", "task" },             \
   {                                                                         \
     "report_severity (id, opts.override, opts.min_qod)",                    \
     "severity",                                                             \
     KEYWORD_TYPE_DOUBLE                                                     \
   },                                                                        \
   {                                                                         \
     "(CASE WHEN (SELECT target IS NULL FROM tasks WHERE tasks.id = task)"   \
     "  THEN 'Container'"                                                    \
     "  ELSE run_status_name (scan_run_status)"                              \
     "       || (SELECT CAST (temp / 100 AS text)"                           \
     "                  || CAST (temp / 10 AS text)"                         \
     "                  || CAST (temp % 10 as text)"                         \
     "           FROM (SELECT report_progress (id) AS temp) AS temp_sub)"    \
     "  END)",                                                               \
     "status",                                                               \
     KEYWORD_TYPE_STRING                                                     \
   },                                                                        \
   {                                                                         \
     "report_severity_count (id, opts.override, opts.min_qod,"               \
     "                       'False Positive')",                             \
     "false_positive",                                                       \
     KEYWORD_TYPE_INTEGER                                                    \
   },                                                                        \
   {                                                                         \
     "report_severity_count (id, opts.override, opts.min_qod, 'Log')",       \
     "log",                                                                  \
     KEYWORD_TYPE_INTEGER                                                    \
   },                                                                        \
   {                                                                         \
     "report_severity_count (id, opts.override, opts.min_qod, 'Low')",       \
     "low",                                                                  \
     KEYWORD_TYPE_INTEGER                                                    \
   },                                                                        \
   {                                                                         \
     "report_severity_count (id, opts.override, opts.min_qod, 'Medium')",    \
     "medium",                                                               \
     KEYWORD_TYPE_INTEGER                                                    \
   },                                                                        \
   {                                                                         \
     "report_severity_count (id, opts.override, opts.min_qod, 'High')",      \
     "high",                                                                 \
     KEYWORD_TYPE_INTEGER                                                    \
   },                                                                        \
   {                                                                         \
     "(SELECT name FROM users WHERE users.id = reports.owner)",              \
     "_owner",                                                               \
     KEYWORD_TYPE_STRING                                                     \
   },                                                                        \
   {                                                                         \
     "report_host_count (id)",                                               \
     "hosts",                                                                \
     KEYWORD_TYPE_INTEGER                                                    \
   },                                                                        \
   {                                                                         \
     "report_result_host_count (id, opts.min_qod)",                          \
     "result_hosts",                                                         \
     KEYWORD_TYPE_INTEGER                                                    \
   },                                                                        \
   {                                                                         \
     "coalesce (report_severity_count (id, opts.override, opts.min_qod,"     \
     "                                 'False Positive') * 1.0"              \
     "            / nullif (report_result_host_count (id, opts.min_qod), 0),"\
     "          0)",                                                         \
     "fp_per_host",                                                          \
     KEYWORD_TYPE_INTEGER                                                    \
   },                                                                        \
   {                                                                         \
     "coalesce (report_severity_count (id, opts.override, opts.min_qod,"     \
     "                                 'Log') * 1.0"                         \
     "            / nullif (report_result_host_count (id, opts.min_qod), 0),"\
     "          0)",                                                         \
     "log_per_host",                                                         \
     KEYWORD_TYPE_INTEGER                                                    \
   },                                                                        \
   {                                                                         \
     "coalesce (report_severity_count (id, opts.override, opts.min_qod,"     \
     "                                 'Low') * 1.0"                         \
     "            / nullif (report_result_host_count (id, opts.min_qod), 0),"\
     "          0)",                                                         \
     "low_per_host",                                                         \
     KEYWORD_TYPE_INTEGER                                                    \
   },                                                                        \
   {                                                                         \
     "coalesce (report_severity_count (id, opts.override, opts.min_qod,"     \
     "                                 'Medium') * 1.0"                      \
     "            / nullif (report_result_host_count (id, opts.min_qod), 0),"\
     "          0)",                                                         \
     "medium_per_host",                                                      \
     KEYWORD_TYPE_INTEGER                                                    \
   },                                                                        \
   {                                                                         \
     "coalesce (report_severity_count (id, opts.override, opts.min_qod,"     \
     "                                 'High') * 1.0"                        \
     "            / nullif (report_result_host_count (id, opts.min_qod), 0),"\
     "          0)",                                                         \
     "high_per_host",                                                        \
     KEYWORD_TYPE_INTEGER                                                    \
   },                                                                        \
   {                                                                         \
     "(CASE WHEN (start_time IS NULL or end_time IS NULL)"                   \
     " THEN NULL ELSE end_time - start_time END)",                           \
     "duration",                                                             \
     KEYWORD_TYPE_INTEGER                                                    \
   },                                                                        \
   {                                                                         \
     "(CASE WHEN (start_time IS NULL or end_time IS NULL"                    \
     "            or report_result_host_count (id, opts.min_qod) = 0)"       \
     " THEN NULL"                                                            \
     " ELSE (end_time - start_time)"                                         \
     "        / report_result_host_count (id, opts.min_qod) END)",           \
     "duration_per_host",                                                    \
     KEYWORD_TYPE_INTEGER                                                    \
   },                                                                        \
   { NULL, NULL, KEYWORD_TYPE_UNKNOWN }                                      \
 }

/**
 * @brief Generate the extra_tables string for a report iterator.
 *
 * @param[in]  override  Whether to apply overrides.
 * @param[in]  min_qod   Minimum QoD of results to count.
 *
 * @return Newly allocated string with the extra_tables clause.
 */
static gchar*
report_iterator_opts_table (int override, int min_qod)
{
  return g_strdup_printf (", (SELECT"
                          "   %d AS override,"
                          "   %d AS min_qod)"
                          "  AS opts",
                          override,
                          min_qod);
}

/**
 * @brief Count number of reports.
 *
 * @param[in]  get  GET params.
 *
 * @return Total number of reports in filtered set.
 */
int
report_count (const get_data_t *get)
{
  static const char *filter_columns[] = REPORT_ITERATOR_FILTER_COLUMNS;
  static column_t columns[] = REPORT_ITERATOR_COLUMNS;
  static column_t where_columns[] = REPORT_ITERATOR_WHERE_COLUMNS;
  gchar *extra_tables;
  int ret;

  extra_tables = report_iterator_opts_table (0, MIN_QOD_DEFAULT);

  ret = count2 ("report", get, columns, NULL, where_columns, NULL,
                filter_columns, 0,
                extra_tables,
                get->trash
                 ? " AND (SELECT hidden FROM tasks"
                   "      WHERE tasks.id = task)"
                   "     = 2"
                 : " AND (SELECT hidden FROM tasks"
                   "      WHERE tasks.id = task)"
                   "     = 0",
                TRUE);

  g_free (extra_tables);
  return ret;
}

/**
 * @brief Initialise a report iterator, including observed reports.
 *
 * @param[in]  iterator    Iterator.
 * @param[in]  get         GET data.
 *
 * @return 0 success, 1 failed to find report, 2 failed to find filter,
 *         -1 error.
 */
int
init_report_iterator (iterator_t* iterator, const get_data_t *get)
{
  static const char *filter_columns[] = REPORT_ITERATOR_FILTER_COLUMNS;
  static column_t columns[] = REPORT_ITERATOR_COLUMNS;
  static column_t where_columns[] = REPORT_ITERATOR_WHERE_COLUMNS;
  char *filter;
  gchar *value;
  int overrides, min_qod;
  gchar *extra_tables;
  int ret;

  if (get->filt_id && strcmp (get->filt_id, "0"))
    {
      filter = filter_term (get->filt_id);
      if (filter == NULL)
        return 2;
    }
  else
    filter = NULL;
  value = filter_term_value (filter ? filter : get->filter, "apply_overrides");
  overrides = value && strcmp (value, "0");
  g_free (value);

  value = filter_term_value (filter ? filter : get->filter, "min_qod");
  if (value == NULL || sscanf (value, "%d", &min_qod) != 1)
    min_qod = MIN_QOD_DEFAULT;
  g_free (value);
  free (filter);

  extra_tables = report_iterator_opts_table (overrides, min_qod);

  ret = init_get_iterator2 (iterator,
                            "report",
                            get,
                            /* Columns. */
                            columns,
                            NULL,
                            /* Filterable columns not in SELECT columns. */
                            where_columns,
                            NULL,
                            filter_columns,
                            0,
                            extra_tables,
                            get->trash
                             ? " AND (SELECT hidden FROM tasks"
                               "      WHERE tasks.id = task)"
                               "     = 2"
                             : " AND (SELECT hidden FROM tasks"
                               "      WHERE tasks.id = task)"
                               "     = 0",
                            TRUE,
                            FALSE,
                            NULL);
  g_free (extra_tables);
  return ret;
}

/**
 * @brief Initialise a report iterator.
 *
 * @param[in]  iterator  Iterator.
 * @param[in]  task      Task whose reports the iterator loops over.
 */
void
init_report_iterator_task (iterator_t* iterator, task_t task)
{
  assert (task);
  init_iterator (iterator,
                 "SELECT id, uuid FROM reports WHERE task = %llu;",
                 task);
}

#if 0
/**
 * @brief Get the NAME from a host iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The NAME of the host.  Caller must use only before calling
 *         cleanup_iterator.
 */
#endif


/**
 * @brief Get the UUID from a report iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return UUID, or NULL if iteration is complete.  Freed by
 *         cleanup_iterator.
 */
DEF_ACCESS (report_iterator_uuid, 1);

/**
 * @brief Read the next report from an iterator.
 *
 * @param[in]   iterator  Task iterator.
 * @param[out]  report    Report.
 *
 * @return TRUE if there was a next task, else FALSE.
 */
gboolean
next_report (iterator_t* iterator, report_t* report)
{
  if (next (iterator))
    {
      *report = iterator_int64 (iterator, 0);
      return TRUE;
    }
  return FALSE;
}

/**
 * @brief Return SQL WHERE for restricting a SELECT to levels.
 *
 * @param[in]  levels  String describing threat levels (message types)
 *                     to include in report (for example, "hmlgd" for
 *                     High, Medium, Low, loG and Debug).  All levels if
 *                     NULL.
 * @param[in]  new_severity_sql  SQL for new severity.
 *
 * @return WHERE clause for levels if one is required, else NULL.
 */
static GString *
where_levels (const char* levels, const char *new_severity_sql)
{
  int count;
  GString *levels_sql;

  /* Generate SQL for constraints on message type, according to levels. */

  if (levels == NULL || strlen (levels) == 0)
    {
      levels_sql = g_string_new ("");
      g_string_append_printf (levels_sql,
                              " AND %s != " G_STRINGIFY (SEVERITY_ERROR),
                              new_severity_sql);
      return levels_sql;
    }

  levels_sql = NULL;
  count = 0;

  /* High. */
  if (strchr (levels, 'h'))
    {
      count = 1;
      // FIX handles dynamic "severity" in caller?
      levels_sql = g_string_new ("");
      g_string_append_printf (levels_sql,
                              " AND (severity_in_level (%s, 'high')",
                              new_severity_sql);
    }

  /* Medium. */
  if (strchr (levels, 'm'))
    {
      if (count == 0)
        {
          levels_sql = g_string_new ("");
          g_string_append_printf (levels_sql,
                                  " AND (severity_in_level (%s, 'medium')",
                                  new_severity_sql);
        }
      else
        g_string_append_printf (levels_sql,
                                " OR severity_in_level (%s, 'medium')",
                                new_severity_sql);
      count++;
    }

  /* Low. */
  if (strchr (levels, 'l'))
    {
      if (count == 0)
        {
          levels_sql = g_string_new ("");
          g_string_append_printf (levels_sql,
                                  " AND (severity_in_level (%s, 'low')",
                                  new_severity_sql);
        }
      else
        g_string_append_printf (levels_sql,
                                " OR severity_in_level (%s, 'low')",
                                new_severity_sql);
      count++;
    }

  /* loG. */
  if (strchr (levels, 'g'))
    {
      if (count == 0)
        {
          levels_sql = g_string_new ("");
          g_string_append_printf (levels_sql,
                                  " AND ((%s"
                                  "       = " G_STRINGIFY (SEVERITY_LOG) ")",
                                  new_severity_sql);
        }
      else
        g_string_append_printf (levels_sql,
                                " OR (%s"
                                "     = " G_STRINGIFY (SEVERITY_LOG) ")",
                                new_severity_sql);
      count++;
    }

  /* Debug. */
  if (strchr (levels, 'd'))
    {
      if (count == 0)
        {
          levels_sql = g_string_new ("");
          g_string_append_printf (levels_sql,
                                  " AND ((%s"
                                  "       = " G_STRINGIFY
                                               (SEVERITY_DEBUG) ")",
                                  new_severity_sql);
        }
      else
        g_string_append_printf (levels_sql,
                                " OR (%s"
                                "     = " G_STRINGIFY (SEVERITY_DEBUG) ")",
                                new_severity_sql);
      count++;
    }

  /* False Positive. */
  if (strchr (levels, 'f'))
    {
      if (count == 0)
        {
          levels_sql = g_string_new ("");
          g_string_append_printf (levels_sql,
                                  " AND ((%s"
                                  "       = " G_STRINGIFY
                                               (SEVERITY_FP) ")",
                                  new_severity_sql);
        }
      else
        g_string_append_printf (levels_sql,
                                " OR (%s"
                                "     = " G_STRINGIFY (SEVERITY_FP) ")",
                                new_severity_sql);
      count++;
    }

  if (count)
    levels_sql = g_string_append (levels_sql, ")");

  if (count == 6)
    {
      /* All levels. */
      g_string_free (levels_sql, TRUE);
      levels_sql = g_string_new ("");
      g_string_append_printf (levels_sql,
                              "AND %s != " G_STRINGIFY (SEVERITY_ERROR),
                              new_severity_sql);
    }

  return levels_sql;
}

/**
 * @brief Return SQL WHERE for restricting a SELECT to levels.
 *
 * @param[in]  levels  String describing threat levels (message types)
 *                     to include in report (for example, "hmlgd" for
 *                     High, Medium, Low, loG and Debug).  All levels if
 *                     NULL.
 * @param[in]  new_severity_sql  SQL for new severity.
 * @param[in]  auto_type_sql     SQL for auto type.
 *
 * @return WHERE clause for levels if one is required, else NULL.
 */
static GString *
where_levels_auto (const char *levels, const char *new_severity_sql,
                   const char *auto_type_sql)
{
  int count;
  GString *levels_sql;

  /* Generate SQL for constraints on message type, according to levels. */

  if (levels == NULL || strlen (levels) == 0)
    {
      levels_sql = g_string_new ("");
      g_string_append_printf (levels_sql,
                              " AND %s != " G_STRINGIFY (SEVERITY_ERROR),
                              new_severity_sql);
      return levels_sql;
    }

  levels_sql = NULL;
  count = 0;

  /* High. */
  if (strchr (levels, 'h'))
    {
      count = 1;
      // FIX handles dynamic "severity" in caller?
      levels_sql = g_string_new ("");
      g_string_append_printf (levels_sql,
                              " AND (((%s IS NULL) AND (severity_in_level (%s, 'high')",
                              auto_type_sql,
                              new_severity_sql);
    }

  /* Medium. */
  if (strchr (levels, 'm'))
    {
      if (count == 0)
        {
          levels_sql = g_string_new ("");
          g_string_append_printf (levels_sql,
                                  " AND (((%s IS NULL) AND (severity_in_level (%s, 'medium')",
                                  auto_type_sql,
                                  new_severity_sql);
        }
      else
        g_string_append_printf (levels_sql,
                                " OR severity_in_level (%s, 'medium')",
                                new_severity_sql);
      count++;
    }

  /* Low. */
  if (strchr (levels, 'l'))
    {
      if (count == 0)
        {
          levels_sql = g_string_new ("");
          g_string_append_printf (levels_sql,
                                  " AND (((%s IS NULL) AND (severity_in_level (%s, 'low')",
                                  auto_type_sql,
                                  new_severity_sql);
        }
      else
        g_string_append_printf (levels_sql,
                                " OR severity_in_level (%s, 'low')",
                                new_severity_sql);
      count++;
    }

  /* loG. */
  if (strchr (levels, 'g'))
    {
      if (count == 0)
        {
          levels_sql = g_string_new ("");
          g_string_append_printf (levels_sql,
                                  " AND (((%s IS NULL) AND (severity_in_level (%s, 'log')",
                                  auto_type_sql,
                                  new_severity_sql);
        }
      else
        g_string_append_printf (levels_sql,
                                " OR severity_in_level (%s, 'log')",
                                new_severity_sql);
      count++;
    }

  /* Debug. */
  if (strchr (levels, 'd'))
    {
      if (count == 0)
        {
          levels_sql = g_string_new ("");
          g_string_append_printf (levels_sql,
                                  " AND (((%s IS NULL)"
                                  "       AND ((%s"
                                  "             = " G_STRINGIFY
                                                     (SEVERITY_DEBUG) ")",
                                  auto_type_sql,
                                  new_severity_sql);
        }
      else
        g_string_append_printf (levels_sql,
                                " OR (%s"
                                "     = " G_STRINGIFY
                                           (SEVERITY_DEBUG) ")",
                                new_severity_sql);
      count++;
    }

  /* False Positive. */
  if (strchr (levels, 'f'))
    {
      if (count == 0)
        {
          levels_sql = g_string_new ("");
          g_string_append_printf (levels_sql,
                                  " AND (((%s IS NULL)"
                                  "       AND %s"
                                  "           = " G_STRINGIFY
                                                   (SEVERITY_FP) ")"
                                  "      OR %s = 1)",
                                  auto_type_sql,
                                  new_severity_sql,
                                  auto_type_sql);
        }
      else
        g_string_append_printf (levels_sql,
                                " OR %s"
                                "    = " G_STRINGIFY
                                          (SEVERITY_FP) "))"
                                " OR %s = 1)",
                                new_severity_sql,
                                auto_type_sql);
      count++;
    }
  else if (count)
    levels_sql = g_string_append (levels_sql, ")))");

  if (count == 6)
    {
      /* All levels. */
      g_string_free (levels_sql, TRUE);
      levels_sql = g_string_new ("");
      /* It's not possible to override from or to the error severity, so no
       * need to use the overriden severity here (new_severity_sql).  This
       * helps with the default result counting performance because the
       * overridden severity is complex. */
      g_string_append_printf (levels_sql,
                              " AND severity != " G_STRINGIFY (SEVERITY_ERROR));
    }

  return levels_sql;
}

/**
 * @brief Return SQL WHERE for restricting a SELECT to a minimum QoD.
 *
 * @param[in]  min_qod  Minimum value for QoD.
 *
 * @return WHERE clause if one is required, else NULL.
 */
static gchar*
where_qod (const char* min_qod)
{
  gchar *qod_sql;

  if (min_qod && strlen (min_qod))
    {
      gchar *quoted_qod;

      if (atoi (min_qod) == 0)
        return g_strdup ("");

      quoted_qod = sql_quote (min_qod);
      qod_sql = g_strdup_printf (" AND (qod >= CAST ('%s' AS INTEGER))",
                                 quoted_qod);
      g_free (quoted_qod);
    }
  else
    qod_sql = g_strdup_printf (" AND (qod >= CAST ('%d' AS INTEGER))",
                               MIN_QOD_DEFAULT);

  return qod_sql;
}

/**
 * @brief SQL for retrieving CVSS base.
 */
#define CVSS_BASE_SQL                                            \
  "(SELECT cvss_base FROM nvts WHERE nvts.oid = results.nvt)"

/**
 * @brief Filter columns for result iterator.
 */
#define RESULT_ITERATOR_FILTER_COLUMNS                                        \
  { GET_ITERATOR_FILTER_COLUMNS, "host", "location", "nvt",                   \
    "type", "original_type", "auto_type",                                     \
    "description", "task", "report", "cvss_base", "nvt_version",              \
    "severity", "original_severity", "vulnerability", "date", "report_id",    \
    "solution_type", "qod", "qod_type", "task_id", "cve", "hostname", NULL }

// TODO Combine with RESULT_ITERATOR_COLUMNS.
/**
 * @brief Result iterator filterable columns, for severity only version .
 */
#define RESULT_ITERATOR_COLUMNS_SEVERITY_FILTERABLE                           \
  {                                                                           \
    { "id", NULL, KEYWORD_TYPE_INTEGER },                                     \
    { "uuid", NULL, KEYWORD_TYPE_STRING },                                    \
    { "(SELECT name FROM nvts WHERE nvts.oid =  nvt)",                        \
      "name",                                                                 \
      KEYWORD_TYPE_STRING },                                                  \
    { "''", "comment", KEYWORD_TYPE_STRING },                                 \
    { " iso_time ( date )", "creation_time", KEYWORD_TYPE_STRING },           \
    { " iso_time ( date )", "modification_time", KEYWORD_TYPE_STRING },       \
    { "date", "created", KEYWORD_TYPE_INTEGER },                              \
    { "date", "modified", KEYWORD_TYPE_INTEGER },                             \
    { "(SELECT name FROM users WHERE users.id = results.owner)",              \
      "_owner",                                                               \
      KEYWORD_TYPE_STRING },                                                  \
    { "owner", NULL, KEYWORD_TYPE_INTEGER },                                  \
    { "host", NULL, KEYWORD_TYPE_STRING },                                    \
    { "port", "location", KEYWORD_TYPE_STRING },                              \
    { "nvt", NULL, KEYWORD_TYPE_STRING },                                     \
    { "severity_to_type (severity)", "original_type", KEYWORD_TYPE_STRING },  \
    { "'Log Message'", /* Adjusted by init_result_get_iterator_severity. */   \
      "type",                                                                 \
      KEYWORD_TYPE_STRING },                                                  \
    { "description", NULL, KEYWORD_TYPE_STRING },                             \
    { "task", NULL, KEYWORD_TYPE_INTEGER },                                   \
    { "report", "report_rowid", KEYWORD_TYPE_INTEGER },                       \
    { "(SELECT cvss_base FROM nvts WHERE nvts.oid =  nvt)",                   \
      "cvss_base",                                                            \
      KEYWORD_TYPE_DOUBLE },                                                  \
    { "nvt_version", NULL, KEYWORD_TYPE_STRING },                             \
    { "severity", "original_severity", KEYWORD_TYPE_DOUBLE },                 \
    { "(SELECT name FROM nvts WHERE nvts.oid =  nvt)",                        \
      "vulnerability",                                                        \
      KEYWORD_TYPE_STRING },                                                  \
    { "date" , NULL, KEYWORD_TYPE_INTEGER },                                  \
    { "(SELECT uuid FROM reports WHERE id = report)",                         \
      "report_id",                                                            \
      KEYWORD_TYPE_STRING },                                                  \
    { "(SELECT solution_type FROM nvts WHERE nvts.oid = nvt)",                \
      "solution_type",                                                        \
      KEYWORD_TYPE_STRING },                                                  \
    { "qod", NULL, KEYWORD_TYPE_INTEGER },                                    \
    { "qod_type", NULL, KEYWORD_TYPE_STRING },                                \
    { "qod_type", NULL, KEYWORD_TYPE_STRING },                                \
    { "(SELECT uuid FROM tasks WHERE id = task)",                             \
      "task_id",                                                              \
      KEYWORD_TYPE_STRING },                                                  \
    { "(SELECT cve FROM nvts WHERE oid = nvt)", "cve", KEYWORD_TYPE_STRING }, \
    { "(SELECT value FROM report_host_details"                                \
      " WHERE name = 'hostname'"                                              \
      "   AND report_host = (SELECT id FROM report_hosts"                     \
      "                       WHERE report_hosts.host=results.host"           \
      "                         AND report_hosts.report = results.report)"    \
      " LIMIT 1)",                                                            \
      "hostname",                                                             \
      KEYWORD_TYPE_STRING                                                     \
    },                                                                        \
    { NULL, NULL, KEYWORD_TYPE_UNKNOWN }                                      \
  }

/**
 * @brief Result iterator columns.
 */
#define RESULT_ITERATOR_COLUMNS                                               \
  {                                                                           \
    { "id", NULL, KEYWORD_TYPE_INTEGER },                                     \
    { "uuid", NULL, KEYWORD_TYPE_STRING },                                    \
    { "(SELECT name FROM nvts WHERE nvts.oid =  nvt)",                        \
      "name",                                                                 \
      KEYWORD_TYPE_STRING },                                                  \
    { "''", "comment", KEYWORD_TYPE_STRING },                                 \
    { " iso_time ( date )", "creation_time", KEYWORD_TYPE_STRING },           \
    { " iso_time ( date )", "modification_time", KEYWORD_TYPE_STRING },       \
    { "date", "created", KEYWORD_TYPE_INTEGER },                              \
    { "date", "modified", KEYWORD_TYPE_INTEGER },                             \
    { "(SELECT name FROM users WHERE users.id = results.owner)",              \
      "_owner",                                                               \
      KEYWORD_TYPE_STRING },                                                  \
    { "owner", NULL, KEYWORD_TYPE_INTEGER },                                  \
    { "host", NULL, KEYWORD_TYPE_STRING },                                    \
    { "port", "location", KEYWORD_TYPE_STRING },                              \
    { "nvt", NULL, KEYWORD_TYPE_STRING },                                     \
    { "severity_to_type (severity)", "original_type", KEYWORD_TYPE_STRING },  \
    { "severity_to_type ((SELECT new_severity FROM result_new_severities"     \
      "                  WHERE result_new_severities.result = results.id"     \
      "                  AND result_new_severities.user"                      \
      "                      = (SELECT users.id"                              \
      "                         FROM current_credentials, users"              \
      "                         WHERE current_credentials.uuid = users.uuid)" \
      "                  AND result_new_severities.override = opts.override"  \
      "                  AND result_new_severities.dynamic = opts.dynamic"    \
      "                  LIMIT 1))",                                          \
      "type",                                                                 \
      KEYWORD_TYPE_STRING },                                                  \
    { "(SELECT autofp FROM results_autofp"                                    \
      " WHERE (result = results.id) AND (autofp_selection = opts.autofp))",   \
      "auto_type",                                                            \
      KEYWORD_TYPE_INTEGER },                                                 \
    { "description", NULL, KEYWORD_TYPE_STRING },                             \
    { "task", NULL, KEYWORD_TYPE_INTEGER },                                   \
    { "report", "report_rowid", KEYWORD_TYPE_INTEGER },                       \
    { "(SELECT cvss_base FROM nvts WHERE nvts.oid =  nvt)",                   \
      "cvss_base",                                                            \
      KEYWORD_TYPE_DOUBLE },                                                  \
    { "nvt_version", NULL, KEYWORD_TYPE_STRING },                             \
    { "severity", "original_severity", KEYWORD_TYPE_DOUBLE },                 \
    { "(SELECT new_severity FROM result_new_severities"                       \
      " WHERE result_new_severities.result = results.id"                      \
      " AND result_new_severities.user"                                       \
      "     = (SELECT users.id"                                               \
      "        FROM current_credentials, users"                               \
      "        WHERE current_credentials.uuid = users.uuid)"                  \
      " AND result_new_severities.override = opts.override"                   \
      " AND result_new_severities.dynamic = opts.dynamic"                     \
      " LIMIT 1)",                                                            \
      "severity",                                                             \
      KEYWORD_TYPE_DOUBLE },                                                  \
    { "(SELECT name FROM nvts WHERE nvts.oid =  nvt)",                        \
      "vulnerability",                                                        \
      KEYWORD_TYPE_STRING },                                                  \
    { "date" , NULL, KEYWORD_TYPE_INTEGER },                                  \
    { "(SELECT uuid FROM reports WHERE id = report)",                         \
      "report_id",                                                            \
      KEYWORD_TYPE_STRING },                                                  \
    { "(SELECT solution_type FROM nvts WHERE nvts.oid = nvt)",                \
      "solution_type",                                                        \
      KEYWORD_TYPE_STRING },                                                  \
    { "qod", NULL, KEYWORD_TYPE_INTEGER },                                    \
    { "qod_type", NULL, KEYWORD_TYPE_STRING },                                \
    { "qod_type", NULL, KEYWORD_TYPE_STRING },                                \
    { "(SELECT uuid FROM tasks WHERE id = task)",                             \
      "task_id",                                                              \
      KEYWORD_TYPE_STRING },                                                  \
    { "(SELECT cve FROM nvts WHERE oid = nvt)", "cve", KEYWORD_TYPE_STRING }, \
    { "(SELECT value FROM report_host_details"                                \
      " WHERE name = 'hostname'"                                              \
      "   AND report_host = (SELECT id FROM report_hosts"                     \
      "                       WHERE report_hosts.host=results.host"           \
      "                         AND report_hosts.report = results.report)"    \
      " LIMIT 1)",                                                            \
      "hostname",                                                             \
      KEYWORD_TYPE_STRING                                                     \
    },                                                                        \
    { NULL, NULL, KEYWORD_TYPE_UNKNOWN }                                      \
  }

/**
 * @brief Generate the extra_tables string for a result iterator.
 *
 * @param[in]  autofp    Whether to apply auto FP filter.
 * @param[in]  override  Whether to apply overrides.
 * @param[in]  dynamic   Whether to use dynamic severity scores.
 *
 * @return Newly allocated string with the extra_tables clause.
 */
static gchar*
result_iterator_opts_table (int autofp, int override, int dynamic)
{
  return g_strdup_printf (", (SELECT"
                          "   %d AS autofp,"
                          "   %d AS override,"
                          "   %d AS dynamic) AS opts",
                          autofp,
                          override,
                          dynamic);
}

/**
 * @brief Get extra_where string for a result iterator or count.
 *
 * @param[in]  get              GET data.
 * @param[in]  report           Report to restrict returned results to.
 * @param[in]  host             Host to restrict returned results to.
 * @param[in]  autofp           Whether to apply auto FP filter.
 * @param[in]  apply_overrides  Whether to apply overrides.
 * @param[in]  dynamic_severity Whether to use dynamic severity.
 * @param[in]  filter           Filter string.
 *
 * @return     Newly allocated extra_where string.
 */
static gchar*
results_extra_where (const get_data_t *get, report_t report, const gchar* host,
                     int autofp, int apply_overrides, int dynamic_severity,
                     const gchar *filter)
{
  gchar *extra_where;
  gchar *levels, *min_qod;
  gchar *report_clause, *host_clause, *min_qod_clause;
  GString *levels_clause;
  gchar *new_severity_sql, *auto_type_sql;

  // Get filter values
  min_qod = filter_term_value (filter, "min_qod");

  levels = filter_term_value (filter, "levels");
  if (levels == NULL)
    levels = g_strdup ("hmlgdf");

  // Build clause fragments

  switch (autofp)
    {
      case 1:
        auto_type_sql = g_strdup_printf
          ("(SELECT autofp FROM results_autofp"
            " WHERE result = results.id"
            " AND autofp_selection = 1)");
        break;
      case 2:
        auto_type_sql = g_strdup_printf
          ("(SELECT autofp FROM results_autofp"
            " WHERE result = results.id"
            " AND autofp_selection = 2)");
          break;

      default:
        auto_type_sql = g_strdup ("NULL");
        break;
    }

  new_severity_sql
    = g_strdup_printf("(SELECT new_severity FROM result_new_severities"
                      " WHERE result_new_severities.result = results.id"
                      " AND result_new_severities.user"
                      "     = (SELECT id FROM users WHERE uuid = '%s')"
                      " AND override = %d"
                      " AND dynamic = %d"
                      " LIMIT 1)",
                      current_credentials.uuid,
                      apply_overrides,
                      dynamic_severity);

  // Build filter clauses

  report_clause = report ? g_strdup_printf (" AND (report = %llu) ", report)
                         : NULL;

  if (host)
    {
      gchar *quoted_host = sql_quote (host);
      host_clause = g_strdup_printf (" AND (host = '%s') ", quoted_host);
      g_free (quoted_host);
    }
  else
    host_clause = NULL;

  min_qod_clause = where_qod (min_qod);
  g_free (min_qod);

  levels_clause = where_levels_auto (levels ? levels : "hmlgdf",
                                     new_severity_sql, auto_type_sql);
  g_free (levels);
  g_free (new_severity_sql);

  extra_where = g_strdup_printf("%s%s%s%s%s",
                                report_clause ? report_clause : "",
                                host_clause ? host_clause : "",
                                levels_clause->str,
                                min_qod_clause ? min_qod_clause : "",
                                report_clause
                                 ? ""
                                 : get->trash
                                    ? " AND ((SELECT (hidden = 2) FROM tasks"
                                      "       WHERE tasks.id = task))"
                                    : " AND ((SELECT (hidden = 0) FROM tasks"
                                      "       WHERE tasks.id = task))");

  g_free (auto_type_sql);
  g_free (min_qod_clause);
  g_string_free (levels_clause, TRUE);
  g_free (report_clause);
  g_free (host_clause);

  return extra_where;
}

/**
 * @brief Initialise the severity-only result iterator.
 *
 * @param[in]  iterator    Iterator.
 * @param[in]  get         GET data.
 * @param[in]  report      Report to restrict returned results to.
 * @param[in]  host        Host to limit results to.
 * @param[in]  extra_order Extra text for ORDER term in SQL.
 *
 * @return 0 success, 1 failed to find result, failed to find filter (filt_id),
 *         -1 error.
 */
int
init_result_get_iterator_severity (iterator_t* iterator, const get_data_t *get,
                                   report_t report, const char* host,
                                   const gchar *extra_order)
{
  column_t columns[3];
  static column_t static_filterable_columns[]
    = RESULT_ITERATOR_COLUMNS_SEVERITY_FILTERABLE;
  static const char *filter_columns[] = RESULT_ITERATOR_FILTER_COLUMNS;
  column_t *filterable_columns;
  int ret;
  gchar *filter, *value;
  int autofp, apply_overrides, dynamic_severity;
  gchar *extra_tables, *extra_where, *owned_clause;
  gchar *pre_sql;
  char *user_id;

  assert (report);

  columns[0].select
    = "(SELECT autofp FROM results_autofp"
      " WHERE (result = results.id) AND (autofp_selection = opts.autofp))";
  columns[0].filter = "auto_type";
  columns[0].type = KEYWORD_TYPE_INTEGER;

  dynamic_severity = setting_dynamic_severity_int ();

  if (get->filt_id && strcmp (get->filt_id, "0"))
    {
      filter = filter_term (get->filt_id);
      if (filter == NULL)
        return 2;
    }
  else
    filter = NULL;

  value = filter_term_value (filter ? filter : get->filter, "apply_overrides");
  apply_overrides = value ? strcmp (value, "0") : 0;
  g_free (value);

  filterable_columns = column_array_copy (static_filterable_columns);
  column_array_set
   (filterable_columns,
    "type",
    g_strdup_printf ("severity_to_type"
                     " ((SELECT new_severity FROM result_new_severities"
                     "   WHERE result_new_severities.result = results.id"
                     "   AND result_new_severities.user"
                     "       = (SELECT users.id"
                     "          FROM current_credentials, users"
                     "          WHERE current_credentials.uuid = users.uuid)"
                     "   AND result_new_severities.override = %i"
                     "   AND result_new_severities.dynamic = %i"
                     "   LIMIT 1))",
                     apply_overrides,
                     dynamic_severity));

  if (dynamic_severity)
    {
      if (apply_overrides)
        columns[1].select
          = "(SELECT coalesce ((SELECT new_severity FROM valid_overrides"
            "                   WHERE valid_overrides.nvt = results.nvt"
            "                   AND (valid_overrides.result = 0"
            "                        OR valid_overrides.result"
            "                           = results.id)"
            "                   AND (valid_overrides.hosts is NULL"
            "                        OR valid_overrides.hosts = ''"
            "                        OR hosts_contains"
            "                            (valid_overrides.hosts,"
            "                             results.host))"
            "                   AND (valid_overrides.port is NULL"
            "                        OR valid_overrides.port = ''"
            "                        OR valid_overrides.port"
            "                           = results.port)"
            "                   AND severity_matches_ov"
            "                        (coalesce"
            "                          ((CASE WHEN results.severity"
            "                                      > " G_STRINGIFY
                                                             (SEVERITY_LOG)
            "                            THEN (SELECT"
            "                                   CAST (cvss_base"
            "                                         AS double precision)"
            "                                  FROM nvts"
            "                                  WHERE nvts.oid"
            "                                        = results.nvt)"
            "                            ELSE results.severity"
            "                            END),"
            "                           results.severity),"
            "                         valid_overrides.severity)"
            "                   LIMIT 1),"
            "                  coalesce ((CASE WHEN results.severity"
            "                                       > " G_STRINGIFY
                                                              (SEVERITY_LOG)
            "                             THEN (SELECT"
            "                                   CAST (cvss_base"
            "                                         AS double precision)"
            "                                   FROM nvts"
            "                                   WHERE nvts.oid"
            "                                         = results.nvt)"
            "                             ELSE results.severity"
            "                             END),"
            "                            results.severity)))";
      else
        columns[1].select
          = "(SELECT coalesce ((CASE WHEN results.severity"
            "                             > " G_STRINGIFY (SEVERITY_LOG)
            "                        THEN (SELECT CAST (cvss_base"
            "                                           AS double precision)"
            "                              FROM nvts"
            "                              WHERE nvts.oid = results.nvt)"
            "                        ELSE results.severity"
            "                        END),"
            "                  results.severity))";
    }
  else
    {
      if (apply_overrides)
        columns[1].select
          = "(SELECT coalesce ((SELECT new_severity FROM valid_overrides"
            "                   WHERE valid_overrides.nvt = results.nvt"
            "                   AND (valid_overrides.result = 0"
            "                        OR valid_overrides.result"
            "                           = results.id)"
            "                   AND (valid_overrides.hosts is NULL"
            "                        OR valid_overrides.hosts = ''"
            "                        OR hosts_contains"
            "                            (valid_overrides.hosts,"
            "                             results.host))"
            "                   AND (valid_overrides.port is NULL"
            "                        OR valid_overrides.port = ''"
            "                        OR valid_overrides.port"
            "                           = results.port)"
            "                   AND severity_matches_ov"
            "                        (results.severity,"
            "                         valid_overrides.severity)"
            "                   LIMIT 1),"
            "                  results.severity))";
      else
        columns[1].select
          = "(SELECT results.severity)";
    }

  columns[1].filter = "severity";
  columns[1].type = KEYWORD_TYPE_DOUBLE;

  columns[2].select = NULL;
  columns[2].filter = NULL;
  columns[2].type = KEYWORD_TYPE_UNKNOWN;

  value = filter_term_value (filter ? filter : get->filter, "autofp");
  autofp = value ? atoi (value) : 0;
  g_free (value);

  if (autofp == 0)
    {
      columns[0].select = "0";
      extra_tables = NULL;
    }
  else
    extra_tables = result_iterator_opts_table (autofp, apply_overrides,
                                               dynamic_severity);

  extra_where = results_extra_where (get, report, host,
                                     autofp, apply_overrides, dynamic_severity,
                                     filter ? filter : get->filter);

  free (filter);

  user_id = sql_string ("SELECT id FROM users WHERE uuid = '%s';",
                        current_credentials.uuid);
  owned_clause = acl_where_owned_for_get ("override", user_id);
  free (user_id);
  pre_sql = g_strdup_printf ("WITH valid_overrides"
                             " AS (SELECT nvt, hosts, new_severity, port,"
                             "            severity, result"
                             "     FROM overrides"
                             "     WHERE %s"
                             "     AND (task = 0"
                             "          OR task = (SELECT reports.task"
                             "                     FROM reports"
                             "                     WHERE reports.id = %llu))"
                             "     AND ((end_time = 0) OR (end_time >= m_now ()))"
                             "     ORDER BY result DESC, task DESC, port DESC, severity ASC,"
                             "           creation_time DESC)"
                             " ",
                             owned_clause,
                             report);
  g_free (owned_clause);

  table_order_if_sort_not_specified = 1;
  ret = init_get_iterator2_pre (iterator,
                                "result",
                                get,
                                /* SELECT columns. */
                                columns,
                                NULL,
                                /* Filterable columns not in SELECT columns. */
                                filterable_columns,
                                NULL,
                                filter_columns,
                                0,
                                extra_tables,
                                extra_where,
                                TRUE,
                                report ? TRUE : FALSE,
                                extra_order,
                                pre_sql,
                                1);
  table_order_if_sort_not_specified = 0;
  column_array_free (filterable_columns);
  g_free (pre_sql);
  g_free (extra_tables);
  g_free (extra_where);
  return ret;
}

/**
 * @brief Initialise a result iterator.
 *
 * @param[in]  iterator    Iterator.
 * @param[in]  get         GET data.
 * @param[in]  report      Report to restrict returned results to.
 * @param[in]  host        Host to limit results to.
 * @param[in]  extra_order Extra text for ORDER term in SQL.
 *
 * @return 0 success, 1 failed to find result, failed to find filter (filt_id),
 *         -1 error.
 */
int
init_result_get_iterator (iterator_t* iterator, const get_data_t *get,
                          report_t report, const char* host,
                          const gchar *extra_order)
{
  static const char *filter_columns[] = RESULT_ITERATOR_FILTER_COLUMNS;
  static column_t columns[] = RESULT_ITERATOR_COLUMNS;
  int ret;
  gchar *filter, *value;
  int autofp, apply_overrides, dynamic_severity;

  gchar *extra_tables, *extra_where;

  if (get->filt_id && strcmp (get->filt_id, "0"))
    {
      filter = filter_term (get->filt_id);
      if (filter == NULL)
        return 2;
    }
  else
    filter = NULL;

  value = filter_term_value (filter ? filter : get->filter, "apply_overrides");
  apply_overrides = value ? strcmp (value, "0") : 0;
  g_free (value);

  value = filter_term_value (filter ? filter : get->filter, "autofp");
  autofp = value ? atoi (value) : 0;
  g_free (value);

  dynamic_severity = setting_dynamic_severity_int ();

  extra_tables
    = result_iterator_opts_table (autofp, apply_overrides, dynamic_severity);

  extra_where = results_extra_where (get, report, host,
                                     autofp, apply_overrides, dynamic_severity,
                                     filter ? filter : get->filter);

  free (filter);

  ret = init_get_iterator2 (iterator,
                            "result",
                            get,
                            /* SELECT columns. */
                            columns,
                            NULL,
                            /* Filterable columns not in SELECT columns. */
                            NULL,
                            NULL,
                            filter_columns,
                            0,
                            extra_tables,
                            extra_where,
                            TRUE,
                            report ? TRUE : FALSE,
                            extra_order);
  g_free (extra_tables);
  g_free (extra_where);
  return ret;
}

/**
 * @brief Count the number of results.
 *
 * @param[in]  get     GET params.
 * @param[in]  report  Report to limit results to.
 * @param[in]  host    Host to limit results to.
 *
 * @return Total number of results in filtered set.
 */
int
result_count (const get_data_t *get, report_t report, const char* host)
{
  static const char *filter_columns[] = RESULT_ITERATOR_FILTER_COLUMNS;
  static column_t columns[] = RESULT_ITERATOR_COLUMNS;
  int ret;
  gchar *filter, *value;
  int apply_overrides, autofp, dynamic_severity;
  gchar *extra_tables, *extra_where;

  if (get->filt_id && strcmp (get->filt_id, "0"))
    {
      filter = filter_term (get->filt_id);
      if (filter == NULL)
        return 2;
    }
  else
    filter = NULL;

  value = filter_term_value (filter, "apply_overrides");
  apply_overrides = value && strcmp (value, "0");
  g_free (value);

  value = filter_term_value (filter, "autofp");
  autofp = value ? atoi (value) : 0;
  g_free (value);

  dynamic_severity = setting_dynamic_severity_int ();

  extra_tables
    = result_iterator_opts_table (autofp, apply_overrides, dynamic_severity);

  extra_where = results_extra_where (get, report, host,
                                     autofp, apply_overrides, dynamic_severity,
                                     filter ? filter : get->filter);

  ret = count ("result", get,
                columns,
                columns,
                filter_columns, 0,
                extra_tables,
                extra_where,
                TRUE);
  g_free (extra_tables);
  g_free (extra_where);
  return ret;
}

/**
 * @brief Get the result from a result iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The result.
 */
result_t
result_iterator_result (iterator_t* iterator)
{
  if (iterator->done) return 0;
  return (result_t) iterator_int64 (iterator, 0);
}

/**
 * @brief Get the host from a result iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The host of the result.  Caller must only use before calling
 *         cleanup_iterator.
 */
DEF_ACCESS (result_iterator_host, GET_ITERATOR_COLUMN_COUNT);

/**
 * @brief Get the port from a result iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The port of the result.  Caller must only use before calling
 *         cleanup_iterator.
 */
DEF_ACCESS (result_iterator_port, GET_ITERATOR_COLUMN_COUNT + 1);

/**
 * @brief Get the NVT OID from a result iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The NVT OID of the result.  Caller must only use before calling
 *         cleanup_iterator.
 */
DEF_ACCESS (result_iterator_nvt_oid, GET_ITERATOR_COLUMN_COUNT + 2);

/**
 * @brief Get the NVT name from a result iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The name of the NVT that produced the result, or NULL on error.
 */
const char*
result_iterator_nvt_name (iterator_t *iterator)
{
  nvti_t *nvti;
  if (iterator->done) return NULL;
  nvti = nvtis_lookup (nvti_cache, result_iterator_nvt_oid (iterator));
  if (nvti)
    return nvti_name (nvti);
  return NULL;
}

/**
 * @brief Get the NVT family from a result iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The family of the NVT that produced the result, or NULL on error.
 */
const char*
result_iterator_nvt_family (iterator_t *iterator)
{
  nvti_t *nvti;
  if (iterator->done) return NULL;
  nvti = nvtis_lookup (nvti_cache, result_iterator_nvt_oid (iterator));
  if (nvti)
    return nvti_family (nvti);
  return NULL;
}

/**
 * @brief Get the NVT CVSS base value from a result iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The CVSS base of the NVT that produced the result, or NULL on error.
 */
const char*
result_iterator_nvt_cvss_base (iterator_t *iterator)
{
  nvti_t *nvti;
  if (iterator->done) return NULL;
  nvti = nvtis_lookup (nvti_cache, result_iterator_nvt_oid (iterator));
  if (nvti)
    return nvti_cvss_base (nvti);
  return NULL;
}

/**
 * @brief Get the NVT CVE from a result iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The CVE of the NVT that produced the result, or NULL on error.
 */
const char*
result_iterator_nvt_cve (iterator_t *iterator)
{
  nvti_t *nvti;
  if (iterator->done) return NULL;
  nvti = nvtis_lookup (nvti_cache, result_iterator_nvt_oid (iterator));
  if (nvti)
    return nvti_cve (nvti);
  return NULL;
}

/**
 * @brief Get the NVT BID from a result iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The BID of the NVT that produced the result, or NULL on error.
 */
const char*
result_iterator_nvt_bid (iterator_t *iterator)
{
  nvti_t *nvti;
  if (iterator->done) return NULL;
  nvti = nvtis_lookup (nvti_cache, result_iterator_nvt_oid (iterator));
  if (nvti)
    return nvti_bid (nvti);
  return NULL;
}

/**
 * @brief Get the NVT XREF from a result iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The XREF of the NVT that produced the result, or NULL on error.
 */
const char*
result_iterator_nvt_xref (iterator_t *iterator)
{
  nvti_t *nvti;
  if (iterator->done) return NULL;
  nvti = nvtis_lookup (nvti_cache, result_iterator_nvt_oid (iterator));
  if (nvti)
    return nvti_xref (nvti);
  return NULL;
}

/**
 * @brief Get the NVT tags from a result iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The tags of the NVT that produced the result, or NULL on error.
 */
const char*
result_iterator_nvt_tag (iterator_t *iterator)
{
  nvti_t *nvti;
  if (iterator->done) return NULL;
  nvti = nvtis_lookup (nvti_cache, result_iterator_nvt_oid (iterator));
  if (nvti)
    return nvti_tag (nvti);
  return NULL;
}

/**
 * @brief Get the original type from a result iterator.
 *
 * This is the column 'type'.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The original type of the result.  Caller must only use before calling
 *         cleanup_iterator.
 */
DEF_ACCESS (result_iterator_original_type, GET_ITERATOR_COLUMN_COUNT + 3);

/**
 * @brief Get the type from a result iterator.
 *
 * This is the the autofp adjusted overridden type.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The type of the result.  Caller must only use before calling
 *         cleanup_iterator.
 */
const char*
result_iterator_type (iterator_t *iterator)
{
  if (iterator->done) return NULL;
  /* auto_type */
  if (iterator_int (iterator, GET_ITERATOR_COLUMN_COUNT + 5))
    return "False Positive";
  /* new_type */
  return iterator_string (iterator, GET_ITERATOR_COLUMN_COUNT + 4);
}

/**
 * @brief Get the descr from a result iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The descr of the result.  Caller must only use before calling
 *         cleanup_iterator.
 */
DEF_ACCESS (result_iterator_descr, GET_ITERATOR_COLUMN_COUNT + 6);

/**
 * @brief Get the task from a result iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The task associated with the result, or 0 on error.
 */
task_t
result_iterator_task (iterator_t* iterator)
{
  if (iterator->done) return 0;
  return (task_t) iterator_int64 (iterator, GET_ITERATOR_COLUMN_COUNT + 7);
}

/**
 * @brief Get the report from a result iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The report associated with the result, or 0 on error.
 */
report_t
result_iterator_report (iterator_t* iterator)
{
  if (iterator->done) return 0;
  return (task_t) iterator_int64 (iterator, GET_ITERATOR_COLUMN_COUNT + 8);
}

/**
 * @brief Get the NVT version used during the scan from a result iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The version of NVT used by the scan that produced the result.
 *         Caller must only use before calling cleanup_iterator.
 */
const char*
result_iterator_scan_nvt_version (iterator_t *iterator)
{
  const char* ret;

  if (iterator->done)
    return NULL;

  /* nvt_version */
  ret = iterator_string (iterator, GET_ITERATOR_COLUMN_COUNT + 10);
  return ret ? ret : "";
}

/**
 * @brief Get the original severity from a result iterator.
 *
 * This is the original severity without overrides and autofp.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The original severity of the result.  Caller must only use before
 *         calling cleanup_iterator.
 */
const char*
result_iterator_original_severity (iterator_t *iterator)
{
  const char* ret;

  if (iterator->done)
    return NULL;

  /* severity */
  ret = iterator_string (iterator, GET_ITERATOR_COLUMN_COUNT + 11);
  return ret ? ret : "";
}

/**
 * @brief Get the severity from a result iterator.
 *
 * This is the the autofp adjusted overridden severity.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The severity of the result.  Caller must only use before calling
 *         cleanup_iterator.
 */
const char*
result_iterator_severity (iterator_t *iterator)
{
  const char* ret;

  if (iterator->done)
    return NULL;

  /* auto_type */
  if (iterator_int (iterator, GET_ITERATOR_COLUMN_COUNT + 5))
    return G_STRINGIFY (SEVERITY_FP);

  /* new_severity */
  ret = iterator_string (iterator, GET_ITERATOR_COLUMN_COUNT + 12);
  return ret ? ret : "";
}

/**
 * @brief Get the severity from a result iterator as double.
 *
 * This is the the autofp adjusted overridden severity.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The severity of the result.  Caller must only use before calling
 *         cleanup_iterator.
 */
double
result_iterator_severity_double (iterator_t *iterator)
{
  if (iterator->done)
    return 0.0;

  /* auto_type */
  if (iterator_int (iterator, GET_ITERATOR_COLUMN_COUNT + 5))
    return SEVERITY_FP;

  return iterator_double (iterator, GET_ITERATOR_COLUMN_COUNT + 12);
}

/**
 * @brief Get the original severity/threat level from a result iterator.
 *
 * This is the original level without overrides and autofp.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The original threat level of the result.  Caller must only use before
 *         calling cleanup_iterator.
 */
const char*
result_iterator_original_level (iterator_t *iterator)
{
  double severity;
  const char* ret;

  if (iterator->done)
    return NULL;

  if (iterator_null (iterator, GET_ITERATOR_COLUMN_COUNT + 11))
    return NULL;

  /* severity */
  severity = iterator_double (iterator, GET_ITERATOR_COLUMN_COUNT + 11);

  ret = severity_to_level (severity, 0);
  return ret ? ret : "";
}

/**
 * @brief Get the severity/threat level from a result iterator.
 *
 * This is the the autofp adjusted overridden level.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The threat level of the result.  Caller must only use before
 *         calling cleanup_iterator.
 */
const char*
result_iterator_level (iterator_t *iterator)
{
  double severity;
  const char* ret;

  if (iterator->done)
    return "";

  /* auto_type */
  if (iterator_int (iterator, GET_ITERATOR_COLUMN_COUNT + 5))
    return "False Positive";

  /* new_severity */
  if (iterator_null (iterator, GET_ITERATOR_COLUMN_COUNT + 12))
    return "";

  severity = iterator_double (iterator, GET_ITERATOR_COLUMN_COUNT + 12);

  ret = severity_to_level (severity, 0);
  return ret ? ret : "";
}

/**
 * @brief Get the qod from a result iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The qod of the result.  Caller must only use before calling
 *         cleanup_iterator.
 */
DEF_ACCESS (result_iterator_qod, GET_ITERATOR_COLUMN_COUNT + 17);

/**
 * @brief Get the qod_type from a result iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The qod type of the result.  Caller must only use before calling
 *         cleanup_iterator.
 */
DEF_ACCESS (result_iterator_qod_type, GET_ITERATOR_COLUMN_COUNT + 18);

/**
 * @brief Initialise a host iterator.
 *
 * @param[in]  iterator  Iterator.
 * @param[in]  report    Report whose hosts the iterator loops over.
 * @param[in]  host      Single host to iterate over.  All hosts if NULL.
 * @param[in]  report_host  Single report host to iterate over.  All if 0.
 */
void
init_report_host_iterator (iterator_t* iterator, report_t report, const char *host,
                           report_host_t report_host)
{
  if (report)
    {
      if (report_host)
        init_iterator (iterator,
                       "SELECT id, host, iso_time (start_time),"
                       " iso_time (end_time), current_port, max_port, report,"
                       " (SELECT uuid FROM reports WHERE id = report),"
                       " (SELECT uuid FROM hosts"
                       "  WHERE id = (SELECT host FROM host_identifiers"
                       "              WHERE source_type = 'Report Host'"
                       "              AND name = 'ip'"
                       "              AND source_id = (SELECT uuid"
                       "                               FROM reports"
                       "                               WHERE id = report)"
                       "              AND value = report_hosts.host"
                       "              LIMIT 1))"
                       " FROM report_hosts WHERE id = %llu"
                       " AND report = %llu"
                       "%s%s%s"
                       " ORDER BY order_inet (host);",
                       report_host,
                       report,
                       host ? " AND host = '" : "",
                       host ? host : "",
                       host ? "'" : "");
      else
        init_iterator (iterator,
                       "SELECT id, host, iso_time (start_time),"
                       " iso_time (end_time), current_port, max_port, report,"
                       " (SELECT uuid FROM reports WHERE id = report),"
                       " (SELECT uuid FROM hosts"
                       "  WHERE id = (SELECT host FROM host_identifiers"
                       "              WHERE source_type = 'Report Host'"
                       "              AND name = 'ip'"
                       "              AND source_id = (SELECT uuid"
                       "                               FROM reports"
                       "                               WHERE id = report)"
                       "              AND value = report_hosts.host"
                       "              LIMIT 1))"
                       " FROM report_hosts WHERE report = %llu"
                       "%s%s%s"
                       " ORDER BY order_inet (host);",
                       report,
                       host ? " AND host = '" : "",
                       host ? host : "",
                       host ? "'" : "");
    }
  else
    {
      if (report_host)
        init_iterator (iterator,
                       "SELECT id, host, iso_time (start_time),"
                       " iso_time (end_time), current_port, max_port, report,"
                       " (SELECT uuid FROM reports WHERE id = report),"
                       " ''"
                       " FROM report_hosts WHERE id = %llu"
                       "%s%s%s"
                       " ORDER BY order_inet (host);",
                       report_host,
                       host ? " AND host = '" : "",
                       host ? host : "",
                       host ? "'" : "");
      else
        init_iterator (iterator,
                       "SELECT id, host, iso_time (start_time),"
                       " iso_time (end_time), current_port, max_port, report,"
                       " (SELECT uuid FROM reports WHERE id = report),"
                       " ''"
                       " FROM report_hosts"
                       "%s%s%s"
                       " ORDER BY order_inet (host);",
                       host ? " WHERE host = '" : "",
                       host ? host : "",
                       host ? "'" : "");
    }
}


/**
 * @brief Get the report host from a host iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return Report host.
 */
static report_host_t
host_iterator_report_host (iterator_t* iterator)
{
  if (iterator->done) return 0;
  return (report_host_t) iterator_int64 (iterator, 0);
}

/**
 * @brief Get the host from a host iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The host of the host.  Caller must use only before calling
 *         cleanup_iterator.
 */
DEF_ACCESS (host_iterator_host, 1);

/**
 * @brief Get the start time from a host iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The start time of the host.  Caller must use only before calling
 *         cleanup_iterator.
 */
DEF_ACCESS (host_iterator_start_time, 2);

/**
 * @brief Get the end time from a host iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The end time of the host.  Caller must use only before calling
 *         cleanup_iterator.
 */
DEF_ACCESS (host_iterator_end_time, 3);

/**
 * @brief Get the current port from a host iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return Current port.
 */
int
host_iterator_current_port (iterator_t* iterator)
{
  int ret;
  if (iterator->done) return -1;
  ret = iterator_int (iterator, 4);
  return ret;
}

/**
 * @brief Get the max port from a host iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return Current port.
 */
int
host_iterator_max_port (iterator_t* iterator)
{
  int ret;
  if (iterator->done) return -1;
  ret = iterator_int (iterator, 5);
  return ret;
}

/**
 * @brief Get the report from a host iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The report of the host.
 */
static report_t
host_iterator_report (iterator_t* iterator)
{
  if (iterator->done) return 0;
  return (report_host_t) iterator_int64 (iterator, 6);
}

/**
 * @brief Get the report UUID from a host iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The UUID of the report of the host.  Caller must use only before
 *         calling cleanup_iterator.
 */
DEF_ACCESS (host_iterator_report_uuid, 7);

/**
 * @brief Get the asset UUID from a host iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The UUID of the assset associate with the host.  Caller must use
 *         only before calling cleanup_iterator.
 */
DEF_ACCESS (host_iterator_asset_uuid, 8);

/**
 * @brief Initialise a report errors iterator.
 *
 * @param[in]  iterator  Iterator.
 * @param[in]  report   The report.
 */
void
init_report_errors_iterator (iterator_t* iterator, report_t report)
{
  if (report)
    init_iterator (iterator,
                   "SELECT results.host, results.port, results.nvt,"
                   " results.description,"
                   " coalesce((SELECT name FROM nvts"
                   "           WHERE nvts.oid = results.nvt), ''),"
                   " coalesce((SELECT cvss_base FROM nvts"
                   "           WHERE nvts.oid = results.nvt), ''),"
                   " results.nvt_version, results.severity,"
                   " results.id"
                   " FROM results"
                   " WHERE results.type = 'Error Message'"
                   "  AND results.report = %llu",
                   report);
}

/**
 * @brief Get the host from a report error messages iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The host of the report error message.  Caller must use only before
 *         calling cleanup_iterator.
 */
DEF_ACCESS (report_errors_iterator_host, 0);

/**
 * @brief Get the port from a report error messages iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The port of the report error message.  Caller must use only before
 *         calling cleanup_iterator.
 */
DEF_ACCESS (report_errors_iterator_port, 1);

/**
 * @brief Get the nvt oid from a report error messages iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The nvt of the report error message.  Caller must use only before
 *         calling cleanup_iterator.
 */
DEF_ACCESS (report_errors_iterator_nvt_oid, 2);

/**
 * @brief Get the description from a report error messages iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The description of the report error message.  Caller must use only
 * before calling cleanup_iterator.
 */
DEF_ACCESS (report_errors_iterator_desc, 3);

/**
 * @brief Get the nvt name from a report error messages iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The nvt of the report error message.  Caller must use only before
 *         calling cleanup_iterator.
 */
DEF_ACCESS (report_errors_iterator_nvt_name, 4);

/**
 * @brief Get the nvt cvss base from a report error messages iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The nvt cvss base of the report error message.  Caller must use only
 *         before calling cleanup_iterator.
 */
DEF_ACCESS (report_errors_iterator_nvt_cvss, 5);

/**
 * @brief Get the nvt cvss base from a report error messages iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The nvt version at scan time of the report error message.
 *         Caller must use only before calling cleanup_iterator.
 */
DEF_ACCESS (report_errors_iterator_scan_nvt_version, 6);

/**
 * @brief Get the nvt cvss base from a report error messages iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The severity at scan time of the report error message.
 *         Caller must use only before calling cleanup_iterator.
 */
DEF_ACCESS (report_errors_iterator_severity, 7);

/**
 * @brief Get the result from a report error messages iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return Result.
 */
result_t
report_errors_iterator_result (iterator_t* iterator)
{
  if (iterator->done) return 0;
  return iterator_int64 (iterator, 8);
}

/**
 * @brief Initialise a report host details iterator.
 *
 * @param[in]  iterator     Iterator.
 * @param[in]  report_host  Report host whose details the iterator loops over.
 *                          All report_hosts if NULL.
 */
static void
init_report_host_details_iterator (iterator_t* iterator,
                                   report_host_t report_host)
{
  /* The 'detected_at' and 'detected_by' entries are filtered out of the final
   * reports as they are only used internally for product detection. */
  init_iterator (iterator,
                 "SELECT id, name, value, source_type, source_name,"
                 "       source_description, NULL"
                 " FROM report_host_details WHERE report_host = %llu"
                 " AND NOT name IN ('detected_at', 'detected_by')"
                 " UNION SELECT 0, 'Closed CVE', cve, 'openvasmd', oid,"
                 "              nvts.name, cvss_base"
                 "       FROM nvts, report_host_details"
                 "       WHERE cve != 'NOCVE'"
                 "       AND family IN (" LSC_FAMILY_LIST ")"
                 "       AND nvts.oid = report_host_details.source_name"
                 "       AND report_host = %llu"
                 "       AND report_host_details.name = 'EXIT_CODE'"
                 "       AND report_host_details.value = 'EXIT_NOTVULN';",
                 report_host,
                 report_host);
}

/**
 * @brief Get the name from a report host details iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The name of the report host detail.  Caller must use only before
 *         calling cleanup_iterator.
 */
DEF_ACCESS (report_host_details_iterator_name, 1);

/**
 * @brief Get the value from a report host details iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The value of the report host detail.  Caller must use only before
 *         calling cleanup_iterator.
 */
DEF_ACCESS (report_host_details_iterator_value, 2);

/**
 * @brief Get the source type from a report host details iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The source type of the report host detail.  Caller must use only
 *         before calling cleanup_iterator.
 */
DEF_ACCESS (report_host_details_iterator_source_type, 3);

/**
 * @brief Get the source name from a report host details iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The source name of the report host detail.  Caller must use only
 *         before calling cleanup_iterator.
 */
DEF_ACCESS (report_host_details_iterator_source_name, 4);

/**
 * @brief Get the source description from a report host details iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return The source description of the report host detail.  Caller must use
 *         only before calling cleanup_iterator.
 */
DEF_ACCESS (report_host_details_iterator_source_desc, 5);

/**
 * @brief Get the extra info from a report host details iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return Extra info of the report host detail.  Caller must use
 *         only before calling cleanup_iterator.
 */
DEF_ACCESS (report_host_details_iterator_extra, 6);

/**
 * @brief Initialise an asset iterator.
 *
 * @param[in]  iterator  Iterator.
 * @param[in]  first_result  The host to start from.  The hosts are 0
 *                           indexed.
 * @param[in]  max_results   The maximum number of hosts returned.
 * @param[in]  levels         String describing threat levels (message types)
 *                            to include in hosts (for example, "hml" for
 *                            High, Medium and Low).  All levels if NULL.
 * @param[in]  search_phrase  Phrase that host IPs must include.  All
 *                            hosts if NULL or "".
 * @param[in]  apply_overrides  Whether to apply overrides.
 */
static void
init_classic_asset_iterator (iterator_t* iterator, int first_result,
                             int max_results, const char *levels,
                             const char *search_phrase, int apply_overrides)
{
  assert (current_credentials.uuid);

  if (levels && strlen (levels))
    {
      GString *levels_sql;
      gchar *severity_sql, *new_severity_sql, *last_report_sql;

      if (setting_dynamic_severity_int ())
        severity_sql = g_strdup ("CASE WHEN results.severity"
                                 "          > " G_STRINGIFY (SEVERITY_LOG)
                                 " THEN coalesce ((SELECT CAST (cvss_base"
                                 "                              AS REAL)"
                                 "                 FROM nvts"
                                 "                 WHERE nvts.oid"
                                 "                        = results.nvt),"
                                 "                results.severity)"
                                 " ELSE results.severity END");
      else
        severity_sql = g_strdup ("results.severity");

      if (apply_overrides)
        {
          gchar *ov, *owned_clause;

          owned_clause = acl_where_owned_for_get ("override", NULL);

          ov = g_strdup_printf
                ("SELECT overrides.new_severity"
                 " FROM overrides"
                 " WHERE overrides.nvt = results.nvt"
                 " AND %s"
                 " AND ((overrides.end_time = 0)"
                 "      OR (overrides.end_time >= m_now ()))"
                 /** @todo Include tasks.hidden and task pref in_assets? */
                 " AND (overrides.task ="
                 "      (SELECT reports.task FROM reports"
                 "       WHERE reports.id = results.report)"
                 "      OR overrides.task = 0)"
                 " AND (overrides.result = results.id"
                 "      OR overrides.result = 0)"
                 " AND (overrides.hosts is NULL"
                 "      OR overrides.hosts = ''"
                 "      OR hosts_contains (overrides.hosts, results.host))"
                 " AND (overrides.port is NULL"
                 "      OR overrides.port = ''"
                 "      OR overrides.port = results.port)"
                 " AND severity_matches_ov (%s, overrides.severity)"
                 " ORDER BY overrides.result DESC, overrides.task DESC,"
                 " overrides.port DESC, overrides.severity ASC,"
                 " overrides.creation_time DESC"
                 " LIMIT 1",
                 owned_clause,
                 severity_sql);

          g_free (owned_clause);

          new_severity_sql = g_strdup_printf ("coalesce ((%s), %s)",
                                              ov, severity_sql);

          g_free (severity_sql);
          g_free (ov);
        }
      else
        new_severity_sql = g_strdup_printf ("%s",
                                            severity_sql);

      levels_sql = where_levels (levels, new_severity_sql);

      last_report_sql
       = g_strdup_printf (" (SELECT report FROM report_hosts"
                          "  WHERE report_hosts.host = distinct_host"
                          "  AND end_time IS NOT NULL"
                          "  AND report_hosts.report"
                          "      IN (SELECT reports.id FROM reports"
                          "          WHERE user_owns ('task', reports.task))"
                          "  AND (SELECT reports.scan_run_status = %u"
                          "       FROM reports"
                          "       WHERE reports.id = report)"
                          "  AND (SELECT hidden FROM tasks"
                          "       WHERE tasks.id"
                          "             = (SELECT task FROM reports"
                          "                WHERE reports.id = report))"
                          "      = 0"
                          "  AND (SELECT value FROM task_preferences"
                          "       WHERE task_preferences.task"
                          "             = (SELECT task FROM reports"
                          "                WHERE reports.id = report)"
                          "       AND task_preferences.name = 'in_assets')"
                          "      = 'yes'"
                          "  ORDER BY id DESC LIMIT 1)",
                          TASK_STATUS_DONE);

      if (search_phrase && strlen (search_phrase))
        {
          gchar *quoted_search_phrase;

          quoted_search_phrase = sql_quote (search_phrase);
          init_iterator
           (iterator,
            "SELECT"
            " distinct_host"
            " FROM (SELECT DISTINCT host AS distinct_host, order_inet (host)"
            "       FROM report_hosts"
            "       ORDER BY order_inet (host))"
            "      AS distinct_host_subquery"
            /* Search IP. */
            " WHERE (distinct_host %s '%%%s%%%'"
            /* Search hostname. */
            "        OR EXISTS"
            "        (SELECT * FROM report_host_details"
            "         WHERE report_host"
            "               = (SELECT id FROM report_hosts"
            "                  WHERE report = %s"
            "                  AND host = distinct_host)"
            "         AND (name = 'hostname'"
            "              OR name = 'best_os_txt'"
            "              OR name = 'best_os_cpe' OR name = 'App'"
            "              OR name = 'ports')"
            "         AND source_type = 'nvt'"
            "         AND value %s '%%%s%%'))"
            /* Filter levels. */
            " AND EXISTS (SELECT results.id"
            "             FROM results"
            "             WHERE results.report = %s"
            "             AND results.host = distinct_host"
            "             AND qod >= " G_STRINGIFY (MIN_QOD_DEFAULT)
            "             %s)"
            " LIMIT %s OFFSET %i;",
            sql_ilike_op (),
            quoted_search_phrase,
            last_report_sql,
            sql_ilike_op (),
            quoted_search_phrase,
            last_report_sql,
            levels_sql ? levels_sql->str : "",
            sql_select_limit (max_results),
            first_result);
          g_free (quoted_search_phrase);
        }
      else
        init_iterator
         (iterator,
          "SELECT"
          " distinct_host"
          " FROM (SELECT DISTINCT host AS distinct_host, order_inet (host)"
          "       FROM report_hosts"
          "       ORDER BY order_inet (host))"
          "      AS distinct_host_subquery"
          " WHERE EXISTS (SELECT results.id"
          "               FROM results"
          "               WHERE results.report = %s"
          "               AND results.host = distinct_host"
            "             AND qod >= " G_STRINGIFY (MIN_QOD_DEFAULT)
          "               %s)"
          " LIMIT %s OFFSET %i;",
          last_report_sql,
          levels_sql ? levels_sql->str : "",
          sql_select_limit (max_results),
          first_result);

      if (levels_sql)
        g_string_free (levels_sql, TRUE);
      g_free (new_severity_sql);
      g_free (last_report_sql);
    }
  else if (search_phrase && strlen (search_phrase))
    {
      gchar *quoted_search_phrase;

      quoted_search_phrase = sql_quote (search_phrase);
      init_iterator (iterator,
                     "SELECT host"
                     " FROM report_hosts"
                     " WHERE report_hosts.report"
                     "     IN (SELECT reports.id FROM reports"
                     "         WHERE user_owns ('task', reports.task))"
                     " AND (SELECT tasks.hidden FROM tasks, reports"
                     "      WHERE reports.task = tasks.id"
                     "      AND reports.id = report_hosts.report)"
                     "     = 0"
                     " AND (SELECT value FROM task_preferences, tasks,"
                     "                        reports"
                     "      WHERE reports.task = tasks.id"
                     "      AND reports.id = report_hosts.report"
                     "      AND task_preferences.task = tasks.id"
                     "      AND task_preferences.name = 'in_assets')"
                     "     = 'yes'"
                     " AND report_hosts.end_time IS NOT NULL"
                     " GROUP BY host"
                     " HAVING host %s '%%%s%%'"
                     " OR EXISTS"
                     " (SELECT * FROM report_host_details"
                     "  WHERE report_hosts.id = report_host"
                     "  AND (name = 'hostname' OR name = 'best_os_txt'"
                     "       OR name = 'best_os_cpe' OR name = 'App'"
                     "       OR name = 'ports')"
                     "  AND source_type = 'nvt'"
                     "  AND value %s '%%%s%%')"
                     " ORDER BY order_inet (host)"
                     " LIMIT %s OFFSET %i;",
                     sql_ilike_op (),
                     quoted_search_phrase,
                     sql_ilike_op (),
                     quoted_search_phrase,
                     sql_select_limit (max_results),
                     first_result);
      g_free (quoted_search_phrase);
    }
  else
    init_iterator (iterator,
                   "SELECT DISTINCT host, order_inet (host) FROM report_hosts"
                   " WHERE report_hosts.report"
                   "     IN (SELECT reports.id FROM reports"
                   "         WHERE user_owns ('task', reports.task))"
                   " AND (SELECT tasks.hidden FROM tasks, reports"
                   "      WHERE reports.task = tasks.id"
                   "      AND reports.id = report_hosts.report)"
                   "     = 0"
                   " AND (SELECT value FROM task_preferences, tasks,"
                   "                        reports"
                   "      WHERE reports.task = tasks.id"
                   "      AND reports.id = report_hosts.report"
                   "      AND task_preferences.task = tasks.id"
                   "      AND task_preferences.name = 'in_assets')"
                   "     = 'yes'"
                   " AND report_hosts.end_time IS NOT NULL"
                   " ORDER BY order_inet (host)"
                   " LIMIT %s OFFSET %i;",
                   sql_select_limit (max_results),
                   first_result);
}

/**
 * @brief Get the IP from a asset iterator.
 *
 * @param[in]  iterator  Iterator.
 *
 * @return Host IP.
 */
DEF_ACCESS (classic_asset_iterator_ip, 0);

/**
 * @brief Set the end time of a task.
 *
 * @param[in]  task  Task.
 * @param[in]  time  New time.  Freed before return.  If NULL, clear end time.
 */
void
set_task_end_time (task_t task, char* time)
{
  if (time)
    {
      sql ("UPDATE tasks SET end_time = %i WHERE id = %llu;",
           parse_iso_time (time),
           task);
      free (time);
    }
  else
    sql ("UPDATE tasks SET end_time = NULL WHERE id = %llu;",
         task);
}

/**
 * @brief Set the end time of a task.
 *
 * @param[in]  task  Task.
 * @param[in]  time  New time.  Freed before return.  If NULL, clear end time.
 */
void
set_task_end_time_epoch (task_t task, time_t time)
{
  if (time)
    sql ("UPDATE tasks SET end_time = %i WHERE id = %llu;", time, task);
  else
    sql ("UPDATE tasks SET end_time = NULL WHERE id = %llu;", task);
}

/**
 * @brief Get the start time of a scan.
 *
 * @param[in]  report  The report associated with the scan.
 *
 * @return Start time of scan, in a newly allocated string.
 */
char*
scan_start_time (report_t report)
{
  char *time = sql_string ("SELECT iso_time (start_time)"
                           " FROM reports WHERE id = %llu;",
                           report);
  return time ? time : g_strdup ("");
}

/**
 * @brief Get the start time of a scan, in seconds since the epoch.
 *
 * @param[in]  report  The report associated with the scan.
 *
 * @return Start time of scan, in seconds.
 */
int
scan_start_time_epoch (report_t report)
{
  return sql_int ("SELECT start_time FROM reports WHERE id = %llu;",
                  report);
}

/**
 * @brief Get the start time of a scan.
 *
 * @param[in]  uuid  The report associated with the scan.
 *
 * @return Start time of scan, in a newly allocated string.
 */
char*
scan_start_time_uuid (const char *uuid)
{
  char *time, *quoted_uuid;
  quoted_uuid = sql_quote (uuid);
  time = sql_string ("SELECT iso_time (start_time)"
                     " FROM reports WHERE uuid = '%s';",
                     quoted_uuid);
  return time ? time : g_strdup ("");
}

/**
 * @brief Set the start time of a scan.
 *
 * @param[in]  report     The report associated with the scan.
 * @param[in]  timestamp  Start time.  In ISO format.
 */
void
set_scan_start_time (report_t report, const char* timestamp)
{
  sql ("UPDATE reports SET start_time = %i WHERE id = %llu;",
       parse_iso_time (timestamp),
       report);
}

/**
 * @brief Set the start time of a scan.
 *
 * @param[in]  report     The report associated with the scan.
 * @param[in]  timestamp  Start time. Epoch format.
 */
void
set_scan_start_time_epoch (report_t report, time_t timestamp)
{
  sql ("UPDATE reports SET start_time = %i WHERE id = %llu;",
       timestamp, report);
}

/**
 * @brief Set the start time of a scan.
 *
 * @param[in]  report     The report associated with the scan.
 * @param[in]  timestamp  Start time.  In OTP format (ctime).
 */
void
set_scan_start_time_otp (report_t report, const char* timestamp)
{
  sql ("UPDATE reports SET start_time = %i WHERE id = %llu;",
       parse_otp_time (timestamp),
       report);
}

/**
 * @brief Get the end time of a scan.
 *
 * @param[in]  report  The report associated with the scan.
 *
 * @return End time of scan, in a newly allocated string.
 */
char*
scan_end_time (report_t report)
{
  char *time = sql_string ("SELECT iso_time (end_time)"
                           " FROM reports WHERE id = %llu;",
                           report);
  return time ? time : g_strdup ("");
}

/**
 * @brief Get the end time of a scan.
 *
 * @param[in]  uuid  The report associated with the scan.
 *
 * @return End time of scan, in a newly allocated string.
 */
char*
scan_end_time_uuid (const char *uuid)
{
  char *time, *quoted_uuid;
  quoted_uuid = sql_quote (uuid);
  time = sql_string ("SELECT iso_time (end_time)"
                     " FROM reports WHERE uuid = '%s';",
                     quoted_uuid);
  return time ? time : g_strdup ("");
}

/**
 * @brief Set the end time of a scan.
 *
 * @param[in]  report     The report associated with the scan.
 * @param[in]  timestamp  End time. Epoch format.
 */
void
set_scan_end_time_epoch (report_t report, time_t timestamp)
{
  if (timestamp)
    sql ("UPDATE reports SET end_time = %i WHERE id = %llu;",
         timestamp, report);
}

/**
 * @brief Set the end time of a scan.
 *
 * @param[in]  report     The report associated with the scan.
 * @param[in]  timestamp  End time.  ISO format.  If NULL, clear end time.
 */
void
set_scan_end_time (report_t report, const char* timestamp)
{
  if (timestamp)
    sql ("UPDATE reports SET end_time = %i WHERE id = %llu;",
         parse_iso_time (timestamp), report);
  else
    sql ("UPDATE reports SET end_time = NULL WHERE id = %llu;",
         report);
}

/**
 * @brief Set the end time of a scan.
 *
 * @param[in]  report     The report associated with the scan.
 * @param[in]  timestamp  End time.  OTP format (ctime).  If NULL, clear end
 *                        time.
 */
void
set_scan_end_time_otp (report_t report, const char* timestamp)
{
  if (timestamp)
    sql ("UPDATE reports SET end_time = %i WHERE id = %llu;",
         parse_otp_time (timestamp), report);
  else
    sql ("UPDATE reports SET end_time = NULL WHERE id = %llu;",
         report);
}

/**
 * @brief Get the end time of a scanned host.
 *
 * @param[in]  report     Report associated with the scan.
 * @param[in]  host       Host.
 */
int
scan_host_end_time (report_t report, const char* host)
{
  gchar *quoted_host;
  int ret;

  quoted_host = sql_quote (host);
  ret = sql_int ("SELECT end_time FROM report_hosts"
                 " WHERE report = %llu AND host = '%s';",
                 report, quoted_host);
  g_free (quoted_host);
  return ret;
}

/**
 * @brief Set the end time of a scanned host.
 *
 * @param[in]  report     Report associated with the scan.
 * @param[in]  host       Host.
 * @param[in]  timestamp  End time.  ISO format.
 */
void
set_scan_host_end_time (report_t report, const char* host,
                        const char* timestamp)
{
  gchar *quoted_host;
  quoted_host = sql_quote (host);
  if (sql_int ("SELECT COUNT(*) FROM report_hosts"
               " WHERE report = %llu AND host = '%s';",
               report, quoted_host))
    sql ("UPDATE report_hosts SET end_time = %i"
         " WHERE report = %llu AND host = '%s';",
         parse_iso_time (timestamp), report, quoted_host);
  else
    manage_report_host_add (report, host, 0, parse_iso_time (timestamp));
  g_free (quoted_host);
}

/**
 * @brief Set the end time of a scanned host.
 *
 * @param[in]  report     Report associated with the scan.
 * @param[in]  host       Host.
 * @param[in]  timestamp  End time.  OTP format (ctime).
 */
void
set_scan_host_end_time_otp (report_t report, const char* host,
                            const char* timestamp)
{
  gchar *quoted_host;
  quoted_host = sql_quote (host);
  if (sql_int ("SELECT COUNT(*) FROM report_hosts"
               " WHERE report = %llu AND host = '%s';",
               report, quoted_host))
    sql ("UPDATE report_hosts SET end_time = %i"
         " WHERE report = %llu AND host = '%s';",
         parse_otp_time (timestamp), report, quoted_host);
  else
    manage_report_host_add (report, host, 0, parse_otp_time (timestamp));
  g_free (quoted_host);
}

/**
 * @brief Set the start time of a scanned host.
 *
 * @param[in]  report     Report associated with the scan.
 * @param[in]  host       Host.
 * @param[in]  timestamp  Start time.  ISO format.
 */
void
set_scan_host_start_time (report_t report, const char* host,
                          const char* timestamp)
{
  gchar *quoted_host;
  quoted_host = sql_quote (host);
  if (sql_int ("SELECT COUNT(*) FROM report_hosts"
               " WHERE report = %llu AND host = '%s';",
               report, quoted_host))
    sql ("UPDATE report_hosts SET start_time = %i"
         " WHERE report = %llu AND host = '%s';",
         parse_iso_time (timestamp), report, quoted_host);
  else
    manage_report_host_add (report, host, parse_iso_time (timestamp), 0);
  g_free (quoted_host);
}

/**
 * @brief Set the start time of a scanned host.
 *
 * @param[in]  report     Report associated with the scan.
 * @param[in]  host       Host.
 * @param[in]  timestamp  Start time.  OTP format (ctime).
 */
void
set_scan_host_start_time_otp (report_t report, const char* host,
                              const char* timestamp)
{
  gchar *quoted_host;
  quoted_host = sql_quote (host);
  if (sql_int ("SELECT COUNT(*) FROM report_hosts"
               " WHERE report = %llu AND host = '%s';",
               report, quoted_host))
    sql ("UPDATE report_hosts SET start_time = %i"
         " WHERE report = %llu AND host = '%s';",
         parse_otp_time (timestamp), report, quoted_host);
  else
    manage_report_host_add (report, host, parse_otp_time (timestamp), 0);
  g_free (quoted_host);
}

/**
 * @brief Get the timestamp of a report.
 *
 * @todo Lacks permission check.  Caller contexts all have permission
 *       checks before calling this so it's safe.  Rework callers so
 *       they pass report_t instead of UUID string.
 *
 * @param[in]   report_id    UUID of report.
 * @param[out]  timestamp    Timestamp on success.  Caller must free.
 *
 * @return 0 on success, -1 on error.
 */
int
report_timestamp (const char* report_id, gchar** timestamp)
{
  const char* stamp;
  time_t time = sql_int ("SELECT date FROM reports where uuid = '%s';",
                         report_id);
  stamp = iso_time (&time);
  if (stamp == NULL) return -1;
  *timestamp = g_strdup (stamp);
  return 0;
}

/**
 * @brief Return the run status of the scan associated with a report.
 *
 * @param[in]   report  Report.
 * @param[out]  status  Scan run status.
 *
 * @return 0 on success, -1 on error.
 */
int
report_scan_run_status (report_t report, int* status)
{
  *status = sql_int ("SELECT scan_run_status FROM reports"
                     " WHERE reports.id = %llu;",
                     report);
  return 0;
}

/**
 * @brief Return the run status of the scan associated with a report.
 *
 * @param[in]   report  Report.
 * @param[out]  status  Scan run status.
 *
 * @return 0 on success, -1 on error.
 */
int
set_report_scan_run_status (report_t report, task_status_t status)
{
  sql ("UPDATE reports SET scan_run_status = %u WHERE id = %llu;",
       status,
       report);
  if (setting_auto_cache_rebuild_int ())
    report_cache_counts (report, 0, 0, NULL);
  return 0;
}

/**
 * @brief Prepare quick statement for report_severity_data.
 *
 * @return Statement.
 */
sql_stmt_t *
report_severity_data_prepare ()
{
  sql_stmt_t *stmt;
  gchar *owned_clause;

  owned_clause = acl_where_owned_for_get ("override", NULL);
  stmt = sql_prepare ("SELECT 1 FROM overrides"
                      " WHERE (overrides.nvt = $1)"
                      " AND %s"
                      " AND ((overrides.end_time = 0)"
                      "      OR (overrides.end_time >= m_now ()))",
                      owned_clause);
  g_free (owned_clause);
  if (stmt == NULL)
    {
      g_warning ("%s: sql_prepare stmt failed\n", __FUNCTION__);
      abort ();
    }
  return stmt;
}

/**
 * @brief Prepare quick statement for report_severity_data.
 *
 * @return Statement.
 */
sql_stmt_t *
report_severity_data_prepare_full (task_t task)
{
  sql_stmt_t *full_stmt;
  gchar *owned_clause;

  owned_clause = acl_where_owned_for_get ("override", NULL);
  full_stmt = sql_prepare
               ("SELECT severity_to_type (overrides.new_severity),"
                 "       overrides.new_severity"
                 " FROM overrides"
                 " WHERE overrides.nvt = $1" // 1
                 " AND %s"
                 " AND ((overrides.end_time = 0)"
                 "      OR (overrides.end_time >= m_now ()))"
                 " AND (overrides.task = 0"
                 "      OR overrides.task = %llu)"
                 " AND (overrides.result = 0"
                 "      OR overrides.result = $2)" // 2
                 " AND (overrides.hosts is NULL"
                 "      OR overrides.hosts = ''"
                 "      OR hosts_contains (overrides.hosts, $3))" // 3
                 " AND (overrides.port is NULL"
                 "      OR overrides.port = ''"
                 "      OR overrides.port = $4)" // 4
                 " AND severity_matches_ov ($5," // 5
                 "                          overrides.severity)"
                 " ORDER BY overrides.result DESC, overrides.task DESC,"
                 " overrides.port DESC, overrides.severity ASC,"
                 " overrides.modification_time DESC;",
                 owned_clause,
                 task);
  g_free (owned_clause);
  if (full_stmt == NULL)
    {
      g_warning ("%s: sql_prepare full_stmt failed\n", __FUNCTION__);
      abort ();
    }
  return full_stmt;
}

/**
 * @brief Get the result severity counts for a report.
 *
 * @param[in]  report     Report.
 * @param[in]  host       Host to which to limit the count.  NULL to allow all.
 * @param[in]  get        Report "get" data to retrieve filter info from.
 * @param[out] severity_data           The severity data struct to store counts in.
 * @param[out] filtered_severity_data  The severity data struct to store counts in.
 */
static void
report_severity_data (report_t report, const char *host,
                      const get_data_t* get,
                      severity_data_t* severity_data,
                      severity_data_t* filtered_severity_data)
{
  iterator_t results;

  gchar *filter, *value;
  int apply_overrides, autofp;

  if (get->filt_id && strcmp (get->filt_id, "0"))
    {
      filter = filter_term (get->filt_id);
    }
  else
    filter = NULL;

  value = filter_term_value (filter ? filter : get->filter, "apply_overrides");
  apply_overrides = value ? strcmp (value, "0") : 0;
  g_free (value);

  value = filter_term_value (filter ? filter : get->filter, "autofp");
  autofp = value ? atoi (value) : 0;
  g_free (value);

  if (severity_data)
    {
      get_data_t *get_all;

      get_all = report_results_get_data (1, -1, apply_overrides, autofp, 0);
      ignore_max_rows_per_page = 1;
      init_result_get_iterator_severity (&results, get_all, report, host, NULL);
      ignore_max_rows_per_page = 0;
      while (next (&results))
        {
          double severity;

          if (results.done)
            severity = 0.0;
          else if (iterator_int (&results, 0))
            severity = SEVERITY_FP;
          else
            severity = iterator_double (&results, 1);

          severity_data_add (severity_data, severity);
        }
      cleanup_iterator (&results);
      get_data_reset (get_all);
      free (get_all);
    }

  if (filtered_severity_data)
    {
      get_data_t get_filtered;

      memset (&get_filtered, 0, sizeof (get_data_t));
      get_filtered.filt_id = get->filt_id;
      get_filtered.filter = get->filter;
      get_filtered.type = get->type;
      get_filtered.ignore_pagination = 1;

      ignore_max_rows_per_page = 1;
      init_result_get_iterator_severity (&results, &get_filtered, report, host,
                                         NULL);
      ignore_max_rows_per_page = 0;
      while (next (&results))
        {
          double severity;

          if (results.done)
            severity = 0.0;
          else if (iterator_int (&results, 0))
            severity = SEVERITY_FP;
          else
            severity = iterator_double (&results, 1);

          severity_data_add (filtered_severity_data, severity);
        }
      cleanup_iterator (&results);
    }
}

/**
 * @brief Get the message counts for a report given the UUID.
 *
 * @todo Lacks permission check.  Caller contexts all have permission
 *       checks before calling this so it's safe.  Rework callers to
 *       use report_counts_id instead.
 *
 * @param[in]   report_id    ID of report.
 * @param[out]  debugs       Number of debug messages.
 * @param[out]  holes        Number of hole messages.
 * @param[out]  infos        Number of info messages.
 * @param[out]  logs         Number of log messages.
 * @param[out]  warnings     Number of warning messages.
 * @param[out]  false_positives  Number of false positives.
 * @param[out]  severity     Maximum severity score.
 * @param[in]   override     Whether to override the threat.
 * @param[in]   autofp       Whether to apply the auto FP filter.
 * @param[in]   min_qod      Min QOD.
 *
 * @return 0 on success, -1 on error.
 */
int
report_counts (const char* report_id, int* debugs, int* holes, int* infos,
               int* logs, int* warnings, int* false_positives, double* severity,
               int override, int autofp, int min_qod)
{
  report_t report;
  int ret;
  get_data_t *get;
  // TODO Wrap in transaction.
  if (find_report_with_permission (report_id, &report, "get_reports"))
    return -1;
  // TODO Check if report was found.

  get = report_results_get_data (1, -1, override, autofp, min_qod);
  ret = report_counts_id (report, debugs, holes, infos, logs, warnings,
                          false_positives, severity, get, NULL);
  get_data_reset (get);
  free (get);
  return ret;
}

/**
 * @brief Test if a counts cache exists for a report and the current user.
 * @param[in]           report    The report to check.
 * @param[in]           override  Whether to check for overridden results.
 * @param[in]           min_qod   Minimum QoD of results to count.
 *
 * @return 1 if cache exists, 0 otherwise.
 */
static int
report_counts_cache_exists (report_t report, int override, int min_qod)
{
  if (setting_dynamic_severity_int ())
    return 0;
  else
    return sql_int ("SELECT EXISTS (SELECT * FROM report_counts"
                    " WHERE report = %llu"
                    "   AND override = %d"
                    "   AND \"user\" = (SELECT id FROM users"
                    "                   WHERE users.uuid = '%s')"
                    "   AND min_qod = %d"
                    "   AND (end_time = 0 OR end_time >= m_now ()));",
                    report, override, current_credentials.uuid, min_qod);
}

/**
 * @brief Get cached result counts for a report and the current user.
 *
 * @param[in]           report    The report to get counts from.
 * @param[in]           override  Whether to get overridden results.
 * @param[in]           min_qod   Minimum QoD of results to count.
 * @param[out]          data      The severity_data_t to save counts in.
 */
static void
report_counts_from_cache (report_t report, int override, int min_qod,
                          severity_data_t* data)
{
  iterator_t iterator;
  init_iterator (&iterator,
                 "SELECT severity, count FROM report_counts"
                 " WHERE report = %llu"
                 "   AND override = %i"
                 "   AND \"user\" = (SELECT id FROM users"
                 "                   WHERE users.uuid = '%s')"
                 "   AND min_qod = %d"
                 "   AND (end_time = 0 OR end_time >= m_now ());",
                 report, override, current_credentials.uuid, min_qod);
  while (next (&iterator))
    {
      severity_data_add_count (data,
                               iterator_double (&iterator, 0),
                               iterator_int (&iterator, 1));
    }
  cleanup_iterator (&iterator);
}

/**
 * @brief Cache the message counts for a report.
 *
 * @param[in]   report    Report.
 * @param[in]   override  Whether overrides were applied to the results.
 * @param[in]   min_qod   The minimum QoD of the results.
 * @param[in]   data      Severity data struct containing the message counts.
 * @param[in]   make_transaction  True to wrap in an exclusive transaction.
 *
 * @return      0 if successful, 1 gave up, -1 error (see sql_giveup).
 */
static int
cache_report_counts (report_t report, int override, int min_qod,
                     severity_data_t* data, int make_transaction)
{
  /* Try cache results.  Give up if the database is locked because this could
   * happen while the caller has an SQL statement open.  If another process
   * tries to write to the database between the statement open and
   * cache_report_counts then they'll deadlock. */
  int i, ret;
  double severity;
  int end_time;

  // Do not cache results when using dynamic severity.
  if (setting_dynamic_severity_int ())
    return 0;

  if (make_transaction)
    {
      ret = sql_begin_exclusive_giveup ();
      if (ret)
        return ret;
    }

  ret = sql_giveup ("DELETE FROM report_counts"
                    " WHERE report = %llu"
                    "   AND override = %i"
                    "   AND min_qod = %i"
                    "   AND \"user\" = (SELECT id FROM users"
                    "                   WHERE users.uuid = '%s');",
                    report, override, min_qod, current_credentials.uuid);
  if (ret)
    {
      if (make_transaction)
        sql_rollback ();
      return ret;
    }

  if (data->total == 0)
    {
      /* Create dummy entry for empty reports */
      ret = sql_giveup ("INSERT INTO report_counts"
                        " (report, \"user\", override, min_qod, severity,"
                        "  count, end_time)"
                        " VALUES (%llu,"
                        "         (SELECT id FROM users"
                        "          WHERE users.uuid = '%s'),"
                        "         %d, %d, " G_STRINGIFY (SEVERITY_MISSING) ","
                        "         0, 0);",
                        report, current_credentials.uuid, override, min_qod);
      if (ret)
        {
          if (make_transaction)
            sql_rollback ();
          return ret;
        }
    }
  else
    {
      i = 0;
      if (override)
        end_time = sql_int ("SELECT coalesce(min(end_time), 0)"
                            " FROM overrides, results"
                            " WHERE overrides.nvt = results.nvt"
                            " AND results.report = %llu"
                            " AND overrides.end_time >= m_now ();",
                            report);
      else
        end_time = 0;

      severity = severity_data_value (i);
      while (severity <= (data->max + (1.0
                                       / SEVERITY_SUBDIVISIONS
                                       / SEVERITY_SUBDIVISIONS))
             && severity != SEVERITY_MISSING)
        {
          if (data->counts[i] > 0)
            {
              ret = sql_giveup ("INSERT INTO report_counts"
                                " (report, \"user\", override, min_qod,"
                                "  severity, count, end_time)"
                                " VALUES (%llu,"
                                "         (SELECT id FROM users"
                                "          WHERE users.uuid = '%s'),"
                                "         %d, %d, %1.1f, %d, %d);",
                                report, current_credentials.uuid, override,
                                min_qod, severity, data->counts[i], end_time);
              if (ret)
                {
                  if (make_transaction)
                    sql_rollback ();
                  return ret;
                }
            }
          i++;
          severity = severity_data_value (i);
        }
    }
  if (make_transaction)
    {
      ret = sql_giveup ("COMMIT;");
      if (ret)
        {
          sql_rollback ();
          return ret;
        }
    }
  return 0;
}

/**
 * @brief Get the message counts for a report.
 *
 * @param[in]   report    Report.
 * @param[out]  debugs    Number of debug messages.
 * @param[out]  holes     Number of hole messages.
 * @param[out]  infos     Number of info messages.
 * @param[out]  logs      Number of log messages.
 * @param[out]  warnings  Number of warning messages.
 * @param[out]  false_positives    Number of false positive messages.
 * @param[out]  severity  Maximum severity of the report.
 * @param[in]   get       Get data.
 * @param[in]   host      Host to which to limit the count.
 * @param[out]  filtered_debugs    Number of debug messages after filtering.
 * @param[out]  filtered_holes     Number of hole messages after filtering.
 * @param[out]  filtered_infos     Number of info messages after filtering.
 * @param[out]  filtered_logs      Number of log messages after filtering.
 * @param[out]  filtered_warnings  Number of warning messages after filtering.
 * @param[out]  filtered_false_positives  Number of false positive messages after
 *                                        filtering.
 * @param[out]  filtered_severity  Maximum severity after filtering.
 *
 * @return 0 on success, -1 on error.
 */
static int
report_counts_id_full (report_t report, int* debugs, int* holes, int* infos,
                       int* logs, int* warnings, int* false_positives,
                       double* severity,
                       const get_data_t* get, const char* host,
                       int* filtered_debugs, int* filtered_holes,
                       int* filtered_infos, int* filtered_logs,
                       int* filtered_warnings, int* filtered_false_positives,
                       double* filtered_severity)
{
  const char *filter;
  keyword_t **point;
  array_t *split;
  int filter_cacheable, unfiltered_requested, filtered_requested, cache_exists;
  const char *severity_class;
  int override, min_qod_int;
  severity_data_t severity_data, filtered_severity_data;

  unfiltered_requested = (holes || warnings || infos || logs || false_positives
                          || severity);
  filtered_requested = (filtered_holes || filtered_warnings || filtered_infos
                        || filtered_logs || filtered_false_positives
                        || filtered_severity);
  severity_class = setting_severity ();

  if (current_credentials.uuid == NULL
      || strcmp (current_credentials.uuid, "") == 0)
    g_warning ("%s: called by NULL or dummy user", __FUNCTION__);

  if (get->filt_id && strlen (get->filt_id) && strcmp (get->filt_id, "0"))
    {
      filter = filter_term (get->filt_id);
      if (filter == NULL)
        {
          return -1;
        }
    }
  else
    {
      filter = get->filter;
    }

  filter_cacheable = TRUE;
  override = 0;
  min_qod_int = MIN_QOD_DEFAULT;

  if (filter == NULL)
    filter = "";

  split = split_filter (filter);
  point = (keyword_t**) split->pdata;
  while (*point)
    {
      keyword_t *keyword;

      keyword = *point;
      if (keyword->column == NULL)
        {
          filter_cacheable = FALSE;
        }
      else if (strcasecmp (keyword->column, "first") == 0
               || strcasecmp (keyword->column, "rows") == 0
               || strcasecmp (keyword->column, "sort") == 0
               || strcasecmp (keyword->column, "sort-reverse") == 0
               || strcasecmp (keyword->column, "notes") == 0
               || strcasecmp (keyword->column, "overrides") == 0)
        {
          // ignore
        }
      else if (strcasecmp (keyword->column, "apply_overrides") == 0)
        {
          if (keyword->string
              && strcmp (keyword->string, "")
              && strcmp (keyword->string, "0"))
            override = 1;
        }
      else if (strcasecmp (keyword->column, "min_qod") == 0)
        {
          if (keyword->string == NULL
              || sscanf (keyword->string, "%d", &min_qod_int) != 1)
            min_qod_int = MIN_QOD_DEFAULT;
        }
      else if (strcasecmp (keyword->column, "autofp") == 0)
        {
          if (keyword->string
              && strcmp (keyword->string, "")
              && strcmp (keyword->string, "0"))
            {
              filter_cacheable = FALSE;
            }
        }
      else
        {
          filter_cacheable = FALSE;
        }
      point++;
    }
  filter_free (split);

  cache_exists = filter_cacheable
                 && report_counts_cache_exists (report, override, min_qod_int);
  init_severity_data (&severity_data);
  init_severity_data (&filtered_severity_data);

  /* This adds time and is out of scope of OMP threat levels, so skip it */
  if (debugs)
    *debugs = 0;

  if (filtered_debugs)
    *filtered_debugs = 0;

  if (cache_exists && filter_cacheable)
    {
      /* Get unfiltered counts from cache. */
      if (unfiltered_requested)
        report_counts_from_cache (report, override, min_qod_int,
                                  &severity_data);
      if (filtered_requested)
        report_counts_from_cache (report, override, min_qod_int,
                                  &filtered_severity_data);
    }
  else
    {
      /* Recalculate. */
      report_severity_data (report, host, get,
                            unfiltered_requested
                              ? &severity_data : NULL,
                            filtered_requested
                              ? &filtered_severity_data : NULL);
    }

  severity_data_level_counts (&severity_data, severity_class,
                              NULL, NULL, false_positives,
                              logs, infos, warnings, holes);
  severity_data_level_counts (&filtered_severity_data, severity_class,
                              NULL, NULL, filtered_false_positives,
                              filtered_logs, filtered_infos,
                              filtered_warnings, filtered_holes);

  if (severity)
    *severity = severity_data.max;
  if (filtered_severity && filtered_requested)
    *filtered_severity = filtered_severity_data.max;

  if (filter_cacheable && !cache_exists)
    {
      if (unfiltered_requested)
        cache_report_counts (report, override, 0, &severity_data, 0);
      if (filtered_requested)
        cache_report_counts (report, override, min_qod_int,
                             &filtered_severity_data, 0);
    }

  cleanup_severity_data (&severity_data);
  cleanup_severity_data (&filtered_severity_data);

  return 0;
}

/**
 * @brief Get the full, unfiltered message counts for a report.
 *
 * @param[in]   report    Report.
 * @param[out]  debugs    Number of debug messages.
 * @param[out]  holes     Number of hole messages.
 * @param[out]  infos     Number of info messages.
 * @param[out]  logs      Number of log messages.
 * @param[out]  warnings  Number of warning messages.
 * @param[out]  false_positives  Number of false positive messages.
 * @param[out]  severity  Maximum severity score.
 * @param[in]   get       Get data.
 * @param[in]   host      Host to which to limit the count.  NULL to allow all.
 *
 * @return 0 on success, -1 on error.
 */
int
report_counts_id_no_filt (report_t report, int* debugs, int* holes, int* infos,
                          int* logs, int* warnings, int* false_positives,
                          double* severity, const get_data_t *get,
                          const char *host)
{
  int ret;
  ret = report_counts_id_full (report, debugs, holes, infos, logs, warnings,
                               false_positives, severity, get, host,
                               NULL, NULL, NULL, NULL, NULL, NULL, NULL);
  return ret;
}

/**
 * @brief Get only the filtered message counts for a report.
 *
 * @param[in]   report    Report.
 * @param[out]  debugs    Number of debug messages.
 * @param[out]  holes     Number of hole messages.
 * @param[out]  infos     Number of info messages.
 * @param[out]  logs      Number of log messages.
 * @param[out]  warnings  Number of warning messages.
 * @param[out]  false_positives  Number of false positive messages.
 * @param[out]  severity  Maximum severity score.
 * @param[in]   get       Get data.
 * @param[in]   host      Host to which to limit the count.  NULL to allow all.
 *
 * @return 0 on success, -1 on error.
 */
int
report_counts_id (report_t report, int* debugs, int* holes, int* infos,
                  int* logs, int* warnings, int* false_positives,
                  double* severity, const get_data_t *get, const char *host)
{
  int ret;
  ret = report_counts_id_full (report, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
                               get, host, debugs, holes, infos, logs, warnings,
                               false_positives, severity);
  return ret;
}

/**
 * @brief Get the maximum severity of a report.
 *
 * @param[in]  report     Report.
 * @param[in]  overrides  Whether to apply overrides.
 * @param[in]  min_qod    Minimum QoD of results to count.
 *
 * @return Severity score of the report.
 */
double
report_severity (report_t report, int overrides, int min_qod)
{
  double severity;
  iterator_t iterator;

  init_iterator (&iterator,
                 "SELECT max(severity)"
                 " FROM report_counts"
                 " WHERE report = %llu"
                 " AND override = %d"
                 " AND user = (SELECT id FROM users WHERE uuid = '%s')"
                 " AND min_qod = %d"
                 " AND (end_time = 0 or end_time >= m_now ());",
                 report, overrides, current_credentials.uuid, min_qod);
  if (next (&iterator)
      && (iterator_null (&iterator, 0) == 0))
    {
      g_debug ("%s: max(severity)=%s", __FUNCTION__,
               iterator_string (&iterator, 0));
      severity = iterator_double (&iterator, 0);
    }
  else
    {
      g_debug ("%s: could not get max from cache", __FUNCTION__);
      get_data_t *get = report_results_get_data (1, -1, overrides, 0, min_qod);
      report_counts_id (report, NULL, NULL, NULL, NULL, NULL,
                        NULL, &severity, get, NULL);
      get_data_reset (get);
      free (get);
    }
  cleanup_iterator (&iterator);
  return severity;
}

/**
 * @brief Delete a report.
 *
 * It's up to the caller to provide the transaction.
 *
 * @param[in]  report  Report.
 *
 * @return 0 success, 1 report is hidden, 2 report is in use, -1 error.
 */
int
delete_report_internal (report_t report)
{
  task_t task;
  char *slave_task_uuid;

  if (sql_int ("SELECT hidden FROM reports WHERE id = %llu;", report))
    return 1;

  if (sql_int ("SELECT count(*) FROM reports WHERE id = %llu"
               " AND (scan_run_status = %u OR scan_run_status = %u"
               " OR scan_run_status = %u OR scan_run_status = %u"
               " OR scan_run_status = %u);",
               report,
               TASK_STATUS_RUNNING,
               TASK_STATUS_REQUESTED,
               TASK_STATUS_DELETE_REQUESTED,
               TASK_STATUS_DELETE_ULTIMATE_REQUESTED,
               TASK_STATUS_STOP_REQUESTED,
               TASK_STATUS_STOP_REQUESTED_GIVEUP,
               TASK_STATUS_STOP_WAITING))
    return 2;

  if (report_task (report, &task))
    return -1;

  /* Remove any associated slave task. */

  slave_task_uuid = report_slave_task_uuid (report);
  g_debug ("%s: slave_task_uuid: %s", __FUNCTION__, slave_task_uuid);
  if (slave_task_uuid)
    {
      scanner_t slave;

      /* A stopped report leaves the task on the slave.  Try delete the task. */

      slave = task_scanner (task);
      if (slave)
        {
          char *username, *password;

          username = scanner_login (slave);
          password = scanner_password (slave);
          if (username && password)
            {
              char *host;
              int port;

              /* Try with values stored on report. */
              host = report_slave_host (report);
              port = report_slave_port_int (report);
              if (host)
                delete_slave_task (host, port, username, password,
                                   slave_task_uuid);
              g_free (host);

              /* TODO If that fails, try with the current values from the
               *      slave/scanner stored on the report.  And if that fails,
               *      try with the values from the current slave of the task. */
            }
          free (username);
          free (password);
        }
    }

  /* Remove the report data. */

  sql ("DELETE FROM report_host_details WHERE report_host IN"
       " (SELECT id FROM report_hosts WHERE report = %llu);",
       report);
  sql ("DELETE FROM report_hosts WHERE report = %llu;", report);

  sql ("UPDATE tags"
       " SET resource = 0, resource_location = " G_STRINGIFY (LOCATION_TABLE)
       " WHERE resource IN"
       "   (SELECT id FROM results WHERE report = %llu);",
       report);
  sql ("UPDATE tags_trash"
       " SET resource = 0, resource_location = " G_STRINGIFY (LOCATION_TABLE)
       " WHERE resource IN"
       "   (SELECT id FROM results WHERE report = %llu);",
       report);
  sql ("DELETE FROM results WHERE report = %llu;", report);

  sql ("DELETE FROM report_counts WHERE report = %llu;", report);
  sql ("DELETE FROM reports WHERE id = %llu;", report);

  /* Adjust permissions. */

  permissions_set_orphans ("report", report, LOCATION_TABLE);
  tags_set_orphans ("report", report, LOCATION_TABLE);

  /* Update the task state. */

  switch (sql_int64 (&report,
                     "SELECT max (id) FROM reports WHERE task = %llu",
                     task))
    {
      case 0:
        if (report)
          {
            int status;
            if (report_scan_run_status (report, &status))
              return -1;
            sql ("UPDATE tasks SET run_status = %u WHERE id = %llu;",
                 status,
                 task);
          }
        else
          sql ("UPDATE tasks SET run_status = %u WHERE id = %llu;",
               TASK_STATUS_NEW,
               task);
        break;
      case 1:        /* Too few rows in result of query. */
        break;
      default:       /* Programming error. */
        assert (0);
      case -1:
        return -1;
        break;
    }

  return 0;
}

/**
 * @brief Modify a report.
 *
 * @param[in]   report_id       UUID of report.
 * @param[in]   comment         Comment on report.
 *
 * @return 0 success, 1 failed to find report, 2 report_id required, 3 comment
 * required, 99 permission denied, -1 internal error.
 */
int
modify_report (const char *report_id, const char *comment)
{
  gchar *quoted_comment;
  report_t report;

  if (report_id == NULL)
    return 2;

  if (comment == NULL)
    return 3;

  sql_begin_immediate ();

  assert (current_credentials.uuid);

  if (acl_user_may ("modify_report") == 0)
    {
      sql_rollback ();
      return 99;
    }

  report = 0;
  if (find_report_with_permission (report_id, &report, "modify_report"))
    {
      sql_rollback ();
      return -1;
    }

  if (report == 0)
    {
      sql_rollback ();
      return 1;
    }

  quoted_comment = sql_quote (comment ? comment : "");

  sql ("UPDATE reports SET"
       " comment = '%s'"
       " WHERE id = %llu;",
       quoted_comment,
       report);

  g_free (quoted_comment);

  sql_commit ();

  return 0;
}

/**
 * @brief Delete a report.
 *
 * @param[in]  report_id  UUID of report.
 * @param[in]  dummy      Dummy arg to match other delete functions.
 *
 * @return 0 success, 2 failed to find report, 99 permission denied, -1 error.
 */
int
delete_report (const char *report_id, int dummy)
{
  report_t report;
  int ret;

  sql_begin_exclusive ();

  if (acl_user_may ("delete_report") == 0)
    {
      sql_rollback ();
      return 99;
    }

  report = 0;
  if (find_report_with_permission (report_id, &report, "delete_report"))
    {
      sql_rollback ();
      return -1;
    }

  if (report == 0)
    {
      sql_rollback ();
      return 2;
    }

  ret = delete_report_internal (report);
  if (ret)
    {
      sql_rollback ();
      return ret;
    }

  sql_commit ();

  return 0;
}

/**
 * @brief Return the slave progress of a report.
 *
 * @param[in]  report  Report.
 *
 * @return Number of reports.
 */
int
report_slave_progress (report_t report)
{
  return sql_int ("SELECT slave_progress FROM reports WHERE id = %llu;",
                  report);
}

/**
 * @brief Set slave progress of a report.
 *
 * @param[in]  report    The report.
 * @param[in]  progress  The new progress value.
 *
 * @return 0 success.
 */
int
set_report_slave_progress (report_t report, int progress)
{
  sql ("UPDATE reports SET slave_progress = %i WHERE id = %llu;",
       progress,
       report);
  return 0;
}

/**
 * @brief Return the UUID of the task on the slave.
 *
 * @param[in]  report    The report.
 *
 * @return UUID of the slave task if any, else NULL.
 */
char*
report_slave_task_uuid (report_t report)
{
  char *uuid;

  uuid = sql_string ("SELECT slave_task_uuid FROM reports WHERE id = %llu;",
                     report);
  if (uuid && strlen (uuid))
    return uuid;
  free (uuid);
  return NULL;
}

/**
 * @brief Set the UUID of the slave task, on the local task.
 *
 * @param[in]  report    The report.
 * @param[in]  uuid  UUID.
 */
void
set_report_slave_task_uuid (report_t report, const char *uuid)
{
  gchar *quoted_uuid = sql_quote (uuid);
  sql ("UPDATE reports SET slave_task_uuid = '%s' WHERE id = %llu;",
       quoted_uuid,
       report);
  g_free (quoted_uuid);
}

/**
 * @brief Prepare a partial report for restarting the scan from the beginning.
 *
 * @param[in]  report  The report.
 */
void
trim_report (report_t report)
{
  /* Remove results for all hosts. */

  sql ("DELETE FROM results WHERE id IN"
       " (SELECT results.id FROM results"
       "  WHERE results.report = %llu);",
       report);

  /* Remove all hosts and host details. */

  sql ("DELETE FROM report_host_details WHERE report_host IN"
       " (SELECT id FROM report_hosts WHERE report = %llu);",
       report);
  sql ("DELETE FROM report_hosts"
       " WHERE report = %llu;",
       report);

  /* Clear and rebuild counts cache */
  if (setting_auto_cache_rebuild_int ())
    report_cache_counts (report, 1, 1, NULL);
  else
    report_clear_count_cache (report, 1, 1, NULL);
}

/**
 * @brief Prepare a partial report for resumption of the scan.
 *
 * @param[in]  report  The report.
 */
void
trim_partial_report (report_t report)
{
  /* Remove results for partial hosts. */

  sql ("DELETE FROM results WHERE id IN"
       " (SELECT results.id FROM results, report_hosts"
       "  WHERE results.report = %llu"
       "  AND report_hosts.report = %llu"
       "  AND results.host = report_hosts.host"
       "  AND report_hosts.end_time = 0);",
       report,
       report);

  /* Remove partial hosts and host details. */

  sql ("DELETE FROM report_host_details WHERE report_host IN"
       " (SELECT report_hosts.id FROM report_hosts"
       "  WHERE report_hosts.report = %llu"
       "  AND report_hosts.end_time = 0);",
       report);

  sql ("DELETE FROM report_hosts"
       " WHERE report = %llu"
       " AND end_time is NULL;",
       report);

  /* Clear and rebuild counts cache */
  if (setting_auto_cache_rebuild_int ())
    report_cache_counts (report, 1, 1, NULL);
  else
    report_clear_count_cache (report, 1, 1, NULL);
}

/**
 * @brief Compares two textual port representations, sorting descending
 * @brief by severity
 *
 * @param[in]  arg_one  First threat level.
 * @param[in]  arg_two  Second threat level.
 *
 * @return 1, 0 or -1 if first given severity is less than, equal to or greater
 *         than second.
 */
static gint
compare_severity_desc (gconstpointer arg_one, gconstpointer arg_two)
{
  double one_severity, two_severity;
  gchar *one = *((gchar**) arg_one);
  gchar *two = *((gchar**) arg_two);
  gint host;

  one += strlen (one) + 1;
  two += strlen (two) + 1;
  one_severity = g_strtod (one, NULL);
  two_severity = g_strtod (two, NULL);

  one += strlen (one) + 1;
  two += strlen (two) + 1;
  host = strcmp (one, two);
  if (host == 0)
    {
      if (one_severity > two_severity)
        return -1;
      else if (one_severity < two_severity)
        return 1;
      else
        {
          one = *((gchar**) arg_one);
          two = *((gchar**) arg_two);
          return strcmp (two, one);
        }
    }
  return host;
}

/**
 * @brief Compares two textual port representations, sorting descending
 * @brief by severity
 *
 * @param[in]  arg_one  First port.
 * @param[in]  arg_two  Second port.
 *
 * @return -1, 0 or 1 if first given severity is less than, equal to or greater
 *         than second.
 */
static gint
compare_severity_asc (gconstpointer arg_one, gconstpointer arg_two)
{
  double one_severity, two_severity;
  gchar *one = *((gchar**) arg_one);
  gchar *two = *((gchar**) arg_two);
  gint host;

  one += strlen (one) + 1;
  two += strlen (two) + 1;
  one_severity = g_strtod (one, NULL);
  two_severity = g_strtod (two, NULL);

  one += strlen (one) + 1;
  two += strlen (two) + 1;
  host = strcmp (one, two);
  if (host == 0)
    {
      if (one_severity < two_severity)
        return -1;
      else if (one_severity > two_severity)
        return 1;
      else
        {
          one = *((gchar**) arg_one);
          two = *((gchar**) arg_two);
          return strcmp (one, two);
        }
    }
  return host;
}

/**
 * @brief Some result info, for sorting.
 */
struct result_buffer
{
  gchar *host;                  ///< Host.
  gchar *port;                  ///< Port.
  gchar *severity;              ///< Severity.
  double severity_double;       ///< Severity.
};

/**
 * @brief Buffer host type.
 */
typedef struct result_buffer result_buffer_t;

/**
 * @brief Create a result buffer.
 *
 * @param[in]  host      Host.
 * @param[in]  port      Port.
 * @param[in]  severity  Severity.
 * @param[in]  severity_double  Severity.
 *
 * @return Freshly allocated result buffer.
 */
static result_buffer_t*
result_buffer_new (const gchar *host, const gchar *port, const gchar *severity,
                   double severity_double)
{
  result_buffer_t *result_buffer;
  result_buffer = g_malloc (sizeof (result_buffer_t));
  result_buffer->host = g_strdup (host);
  result_buffer->port = g_strdup (port);
  result_buffer->severity = g_strdup (severity);
  result_buffer->severity_double = severity_double;
  return result_buffer;
}

/**
 * @brief Free a result buffer.
 *
 * @param[in]  result_buffer  Result buffer.
 */
static void
result_buffer_free (result_buffer_t *result_buffer)
{
  g_free (result_buffer->host);
  g_free (result_buffer->port);
  g_free (result_buffer->severity);
  g_free (result_buffer);
}

/**
 * @brief Compares two buffered results, sorting by host, port then severity.
 *
 * @param[in]  arg_one  First result.
 * @param[in]  arg_two  Second result.
 *
 * @return -1, 0 or 1 if first given result is less than, equal to or greater
 *         than second.
 */
static gint
compare_port_severity (gconstpointer arg_one, gconstpointer arg_two)
{
  int host;
  result_buffer_t *one, *two;

  one = *((result_buffer_t**) arg_one);
  two = *((result_buffer_t**) arg_two);

  host = strcmp (one->host, two->host);
  if (host == 0)
    {
      double severity_cmp;
      int port;

      port = strcmp (one->port, two->port);
      if (port != 0)
        return port;

      severity_cmp = two->severity_double - one->severity_double;
      if (severity_cmp > 0)
        return 1;
      else if (severity_cmp < 0)
        return -1;
      else
        return 0;
    }
  return host;
}

/** @todo Defined in omp.c! */
void buffer_results_xml (GString *, iterator_t *, task_t, int, int, int, int,
                         int, int, int, const char *, iterator_t *, int);

/**
 * @brief Comparison returns.
 */
typedef enum
{
  COMPARE_RESULTS_CHANGED,
  COMPARE_RESULTS_ERROR,
  COMPARE_RESULTS_GONE,
  COMPARE_RESULTS_NEW,
  COMPARE_RESULTS_SAME
} compare_results_t;

/**
 * @brief Return the sort order of two results.
 *
 * @param[in]  results        Iterator containing first result.
 * @param[in]  delta_results  Iterator containing second result.
 * @param[in]  sort_order     Whether to sort ascending or descending.
 * @param[in]  sort_field     Field to sort on, or NULL for "type".
 *
 * @return < 0 if first comes before second, 0 if equal, > 0 if first comes
 *         after second.
 */
static compare_results_t
result_cmp (iterator_t *results, iterator_t *delta_results, int sort_order,
            const char* sort_field)
{
  const char *host, *delta_host, *port, *delta_port;
  const char *nvt, *delta_nvt, *name, *delta_name, *descr, *delta_descr;
  int ret;
  double severity, delta_severity;

  if (sort_field == NULL)
    sort_field = "type";

  g_debug ("   delta: %s: sort_order: %i", __FUNCTION__, sort_order);
  g_debug ("   delta: %s: sort_field: %s", __FUNCTION__, sort_field);

  host = result_iterator_host (results);
  delta_host = result_iterator_host (delta_results);

  port = result_iterator_port (results);
  delta_port = result_iterator_port (delta_results);

  severity = result_iterator_severity_double (results);
  delta_severity = result_iterator_severity_double (delta_results);

  nvt = result_iterator_nvt_oid (results);
  delta_nvt = result_iterator_nvt_oid (delta_results);

  name = result_iterator_nvt_name (results);
  delta_name = result_iterator_nvt_name (delta_results);

  descr = result_iterator_descr (results);
  delta_descr = result_iterator_descr (delta_results);

  /*
   * For delta reports to work correctly, the order must be the same as in
   *  init_delta_iterators, except that description should not be checked
   *  unless it is the sort_field.
   */

  /* Check sort_field first, also using sort_order (0 is descending). */
  if (strcmp (sort_field, "ROWID") == 0)
    {
      if (sort_order)
        return result_iterator_result (results)
                > result_iterator_result (delta_results);
      else
        return result_iterator_result (results)
                < result_iterator_result (delta_results);
    }
  else if (strcmp (sort_field, "host") == 0)
    {
      ret = collate_ip (NULL,
                        strlen (host), host, strlen (delta_host), delta_host);
      if (sort_order == 0)
        ret = -ret;
      g_debug ("   delta: %s: host (%s): %s VS %s (%i)",
               __FUNCTION__, sort_order ? "desc" : "asc",
               host, delta_host, ret);
      if (ret)
        return ret;
    }
  else if (strcmp (sort_field, "port") == 0
           || strcmp (sort_field, "location") == 0)
    {
      ret = strcmp (port, delta_port);
      if (sort_order == 0)
        ret = -ret;
      g_debug ("   delta: %s: port (%s): %s VS %s (%i)",
               __FUNCTION__, sort_order ? "desc" : "asc",
               port, delta_port, ret);
      if (ret)
        return ret;
    }
  else if (strcmp (sort_field, "severity") == 0)
    {
      if (severity > delta_severity)
        ret = sort_order ? 1 : -1;
      else if (severity < delta_severity)
        ret = sort_order ? -1 : 1;
      else
        ret = 0;
      g_debug ("   delta: %s: severity (%s): %f VS %f (%i)",
               __FUNCTION__, sort_order ? "desc" : "asc",
               severity, delta_severity, ret);
      if (ret)
        return ret;
    }
  else if (strcmp (sort_field, "nvt") == 0)
    {
      ret = strcmp (nvt, delta_nvt);
      if (sort_order)
        ret = -ret;
      g_debug ("   delta: %s: nvt (%s): %s VS %s (%i)",
               __FUNCTION__, sort_order ? "desc" : "asc",
               nvt, delta_nvt, ret);
      if (ret)
        return ret;
    }
  else if (strcmp (sort_field, "description") == 0)
    {
      ret = strcmp (descr, delta_descr);
      if (sort_order == 0)
        ret = -ret;
      g_debug ("   delta: %s: description (%s): %s VS %s (%i)",
               __FUNCTION__, sort_order ? "desc" : "asc",
               descr, delta_descr, ret);
      if (ret)
        return ret;
    }
  else if (strcmp (sort_field, "type") == 0)
    {
      const char *type, *delta_type;

      type = result_iterator_type (results);
      delta_type = result_iterator_type (delta_results);

      ret = strcmp (type, delta_type);
      if (sort_order == 0)
        ret = -ret;
      g_debug ("   delta: %s: type (%s): %s VS %s (%i)",
               __FUNCTION__, sort_order ? "desc" : "asc",
               descr, delta_descr, ret);
      if (ret)
        return ret;
    }
  else if (strcmp (sort_field, "original_type") == 0)
    {
      const char *type, *delta_type;

      type = result_iterator_original_type (results);
      delta_type = result_iterator_original_type (delta_results);

      ret = strcmp (type, delta_type);
      if (sort_order == 0)
        ret = -ret;
      g_debug ("   delta: %s: original_type (%s): %s VS %s (%i)",
               __FUNCTION__, sort_order ? "desc" : "asc",
               descr, delta_descr, ret);
      if (ret)
        return ret;
    }
  else
    {
      /* Default to "vulnerability" (a.k.a "name") for unknown sort fields.
       *
       * Also done in print_report_xml_start, so this is just a safety check. */
      ret = strcmp (name, delta_name);
      if (sort_order == 0)
        ret = -ret;
      if (ret)
        return ret;
    }

  /* Check remaining fields */
  if (strcmp (sort_field, "host"))
    {
      ret = collate_ip (NULL,
                        strlen (host), host, strlen (delta_host), delta_host);
      g_debug ("   delta: %s: host: %s VS %s (%i)",
               __FUNCTION__, host, delta_host, ret);
      if (ret)
        return ret;
    }
  if (strcmp (sort_field, "port")
      && strcmp (sort_field, "location"))
    {
      ret = strcmp (port, delta_port);
      g_debug ("   delta: %s: port: %s VS %s (%i)",
               __FUNCTION__, port, delta_port, ret);
      if (ret)
        return ret;
    }
  if (strcmp (sort_field, "severity"))
    {
      if (severity > delta_severity)
        ret = 1;
      else if (severity < delta_severity)
        ret = -1;
      else
        ret = 0;
      g_debug ("   delta: %s: severity: %f VS %f (%i)",
               __FUNCTION__, severity, delta_severity, ret);
      if (ret)
        return ret;
    }
  if (strcmp (sort_field, "nvt"))
    {
      ret = strcmp (nvt, delta_nvt);
      g_debug ("   delta: %s: nvt: %s VS %s (%i)",
               __FUNCTION__, nvt, delta_nvt, ret);
      if (ret)
        return ret;
    }

  return 0;
}

/**
 * @brief Compare two results.
 *
 * @param[in]  results        Iterator containing first result.
 * @param[in]  delta_results  Iterator containing second result.
 * @param[in]  sort_order     Whether to sort ascending or descending.
 * @param[in]  sort_field     Field to sort on, or NULL for "type".
 *
 * @return Result of comparison.
 */
static compare_results_t
compare_results (iterator_t *results, iterator_t *delta_results, int sort_order,
                 const char* sort_field)
{
  int ret;
  const char *descr, *delta_descr;

  g_debug ("   delta: %s", __FUNCTION__);

  ret = result_cmp (results, delta_results, sort_order, sort_field);
  if (ret > 0)
    /* The delta result sorts first, so it is new. */
    return COMPARE_RESULTS_NEW;
  if (ret < 0)
    /* The 'results' result sorts first, so it has gone. */
    return COMPARE_RESULTS_GONE;

  descr = result_iterator_descr (results);
  delta_descr = result_iterator_descr (delta_results);

  g_debug ("   delta: %s: descr: %s VS %s (%i)",
          __FUNCTION__, descr, delta_descr, strcmp (descr, delta_descr));

  ret = strcmp (descr, delta_descr);
  if (ret)
    return COMPARE_RESULTS_CHANGED;

  return COMPARE_RESULTS_SAME;
}

/**
 * @brief Compare two results, writing associated XML to a buffer.
 *
 * @param[in]  buffer         Buffer.
 * @param[in]  results        Iterator containing first result.
 * @param[in]  delta_results  Iterator containing second result.
 * @param[in]  task           Task associated with report.
 * @param[in]  notes              Whether to include notes.
 * @param[in]  notes_details      If notes, Whether to include details.
 * @param[in]  overrides          Whether to include overrides.
 * @param[in]  overrides_details  If overrides, Whether to include details.
 * @param[in]  sort_order     Whether to sort ascending or descending.
 * @param[in]  sort_field     Field to sort on, or NULL for "type".
 * @param[in]  changed        Whether to include changed results.
 * @param[in]  gone           Whether to include gone results.
 * @param[in]  new            Whether to include new results.
 * @param[in]  same           Whether to include same results.
 * @param[in]  max_results    Value to decrement if result is buffered.
 * @param[in]  first_result   Skip result and decrement if positive.
 * @param[in]  used           0 if used, 1 if skipped.
 * @param[in]  would_use      0 if would use (first_result aside), 1 if skipped.
 *
 * @return Result of comparison.
 */
static compare_results_t
compare_and_buffer_results (GString *buffer, iterator_t *results,
                            iterator_t *delta_results, task_t task, int notes,
                            int notes_details, int overrides,
                            int overrides_details, int sort_order,
                            const char* sort_field, int changed, int gone,
                            int new, int same, int *max_results,
                            int *first_result, int *used, int *would_use)
{
  compare_results_t state;
  state = compare_results (results, delta_results, sort_order, sort_field);
  *used = 0;
  *would_use = 0;
  switch (state)
    {
      case COMPARE_RESULTS_CHANGED:
        if (changed)
          {
            *would_use = 1;
            if (*first_result)
              {
                g_debug ("   delta: skip");
                (*first_result)--;
                break;
              }
            *used = 1;
            (*max_results)--;
            if (buffer)
              buffer_results_xml (buffer,
                                  results,
                                  task,
                                  notes,
                                  notes_details,
                                  overrides,
                                  overrides_details,
                                  0,
                                  0,
                                  0,
                                  "changed",
                                  delta_results,
                                  1);
          }
        break;

      case COMPARE_RESULTS_GONE:
        if (gone)
          {
            *would_use = 1;
            if (*first_result)
              {
                g_debug ("   delta: skip");
                (*first_result)--;
                break;
              }
            *used = 1;
            (*max_results)--;
            if (buffer)
              buffer_results_xml (buffer,
                                  results,
                                  task,
                                  notes,
                                  notes_details,
                                  overrides,
                                  overrides_details,
                                  0,
                                  0,
                                  0,
                                  "gone",
                                  delta_results,
                                  0);
          }
        break;

      case COMPARE_RESULTS_NEW:
        if (new)
          {
            *would_use = 1;
            if (*first_result)
              {
                g_debug ("   delta: skip");
                (*first_result)--;
                break;
              }
            *used = 1;
            (*max_results)--;
            if (buffer)
              buffer_results_xml (buffer,
                                  delta_results,
                                  task,
                                  notes,
                                  notes_details,
                                  overrides,
                                  overrides_details,
                                  0,
                                  0,
                                  0,
                                  "new",
                                  delta_results,
                                  0);
          }
        break;

      case COMPARE_RESULTS_SAME:
        if (same)
          {
            *would_use = 1;
            if (*first_result)
              {
                g_debug ("   delta: skip");
                (*first_result)--;
                break;
              }
            *used = 1;
            (*max_results)--;
            if (buffer)
              buffer_results_xml (buffer,
                                  results,
                                  task,
                                  notes,
                                  notes_details,
                                  overrides,
                                  overrides_details,
                                  0,
                                  0,
                                  0,
                                  "same",
                                  delta_results,
                                  0);
          }
        break;

      default:
        return COMPARE_RESULTS_ERROR;
    }

  return state;
}

/**
 * @brief Write to a file or close stream and exit.
 *
 * @param[in]   stream    Stream to write to.
 * @param[in]   format    Format specification.
 * @param[in]   args      Arguments.
 */
#define PRINT(stream, format, args...)                                       \
  do                                                                         \
    {                                                                        \
      gchar *msg;                                                            \
      msg = g_markup_printf_escaped (format, ## args);                       \
      if (fprintf (stream, "%s", msg) < 0)                                   \
        {                                                                    \
          g_free (msg);                                                      \
          fclose (stream);                                                   \
          return -1;                                                         \
        }                                                                    \
      g_free (msg);                                                          \
    }                                                                        \
  while (0)

/**
 * @brief Write XML to a file or close stream and return.
 *
 * @param[in]   stream  Stream to write to.
 * @param[in]   xml     XML.
 */
#define PRINT_XML(stream, xml)                                               \
  do                                                                         \
    {                                                                        \
      if (fprintf (stream, "%s", xml) < 0)                                   \
        {                                                                    \
          fclose (stream);                                                   \
          return -1;                                                         \
        }                                                                    \
    }                                                                        \
  while (0)

#if 0
void
dump (GArray *ports)
{
  int index;
  for (index = 0; index < ports->len; index++)
    {
      char *port = g_array_index (ports, char*, index);
      char *threat = port + strlen (port) + 1;
      g_debug ("  == %s %s %s", threat + strlen (threat) + 1, port, threat);
    }
}
#endif

/**
 * @brief Add a port to a port tree.
 *
 * @param[in]  ports    The tree.
 * @param[in]  results  Result iterator on result whose port to add.
 */
static void
add_port (GTree *ports, iterator_t *results)
{
  const char *port, *host;
  double *old_severity, *severity;
  GTree *host_ports;

  /* Ensure there's an inner tree for the host. */

  host = result_iterator_host (results);
  host_ports = g_tree_lookup (ports, host);
  if (host_ports == NULL)
    {
      host_ports = g_tree_new_full ((GCompareDataFunc) strcmp, NULL, g_free,
                                    g_free);
      g_tree_insert (ports, g_strdup (host), host_ports);
    }

  /* Ensure the highest threat is recorded for the port in the inner tree. */

  port = result_iterator_port (results);
  severity = g_malloc (sizeof (double));
  *severity = result_iterator_severity_double (results);

  old_severity = g_tree_lookup (host_ports, port);
  g_debug ("   delta: %s: adding %s severity %1.1f on host %s", __FUNCTION__,
          port, *severity, host);
  if (old_severity == NULL)
    g_tree_insert (host_ports, g_strdup (port), severity);
  else if (severity > old_severity)
    {
      *old_severity = *severity;
      g_free (severity);
    }
}

/**
 * @brief Print delta host ports.
 *
 * @param[in]  key     Port.
 * @param[in]  value   Threat.
 * @param[in]  data    Host and stream.
 */
static gboolean
print_host_port (gpointer key, gpointer value, gpointer data)
{
  gpointer *host_and_stream;
  host_and_stream = (gpointer*) data;
  g_debug ("   delta: %s: host %s port %s", __FUNCTION__,
          (gchar*) host_and_stream[0], (gchar*) key);
  fprintf ((FILE*) host_and_stream[1],
           "<port>"
           "<host>%s</host>"
           "%s"
           "<severity>%1.1f</severity>"
           "<threat>%s</threat>"
           "</port>",
           (gchar*) host_and_stream[0],
           (gchar*) key,
           *((double*) value),
           severity_to_level (*((double*) value), 0));
  return FALSE;
}

/**
 * @brief Print delta ports.
 *
 * @param[in]  key     Host.
 * @param[in]  value   Port tree.
 * @param[in]  stream  Stream.
 */
static gboolean
print_host_ports (gpointer key, gpointer value, gpointer stream)
{
  gpointer host_and_stream[2];
  host_and_stream[0] = key;
  host_and_stream[1] = stream;
  g_debug ("   delta: %s: host %s", __FUNCTION__, (gchar*) key);
  g_tree_foreach ((GTree*) value, print_host_port, host_and_stream);
  return FALSE;
}

/**
 * @brief Add port to ports array.
 *
 * @param[in]  key     Port.
 * @param[in]  value   Threat.
 * @param[in]  ports   Ports array.
 */
static gboolean
array_add_port (gpointer key, gpointer value, gpointer ports)
{
  gpointer *port_threat;
  port_threat = g_malloc (2 * sizeof (gpointer));
  port_threat[0] = key;
  port_threat[1] = value;
  array_add ((array_t*) ports, port_threat);
  return FALSE;
}

/**
 * @brief Print delta ports, in descending order.
 *
 * @param[in]  key     Host.
 * @param[in]  value   Port tree.
 * @param[in]  stream  Stream.
 */
static gboolean
print_host_ports_desc (gpointer key, gpointer value, gpointer stream)
{
  guint index;
  array_t *ports;

  g_debug ("   delta: %s: host %s", __FUNCTION__, (gchar*) key);

  /* Convert tree to array. */

  ports = make_array ();
  g_tree_foreach ((GTree*) value, array_add_port, ports);

  /* Print the array backwards. */

  index = ports->len;
  while (index--)
    {
      gpointer *port_threat;
      port_threat = g_ptr_array_index (ports, index);
      fprintf ((FILE*) stream,
               "<port>"
               "<host>%s</host>"
               "%s"
               "<severity>%1.1f</severity>"
               "<threat>%s</threat>"
               "</port>",
               (gchar*) key,
               (gchar*) port_threat[0],
               *((double*) port_threat[1]),
               severity_to_level (*((double*) port_threat[1]), 0));
    }

  array_free (ports);

  return FALSE;
}

/**
 * @brief Compare port severities, ascending.
 *
 * @param[in]  one  First.
 * @param[in]  two  Second.
 */
static gint
compare_ports_severity (gconstpointer one, gconstpointer two)
{
  gpointer *port_threat_one, *port_threat_two;
  port_threat_one = *((gpointer**) one);
  port_threat_two = *((gpointer**) two);
  if (*((double*) port_threat_one[1]) > *((double*) port_threat_two[1]))
    return 1;
  else if (*((double*) port_threat_one[1]) < *((double*) port_threat_two[1]))
    return -1;
  else
    return 0;
}

/**
 * @brief Compare port severities, descending.
 *
 * @param[in]  one  First.
 * @param[in]  two  Second.
 */
static gint
compare_ports_severity_desc (gconstpointer one, gconstpointer two)
{
  gpointer *port_threat_one, *port_threat_two;
  port_threat_one = *((gpointer**) one);
  port_threat_two = *((gpointer**) two);
  if (*((double*) port_threat_one[1]) < *((double*) port_threat_two[1]))
    return 1;
  else if (*((double*) port_threat_one[1]) > *((double*) port_threat_two[1]))
    return -1;
  else
    return 0;
}

/**
 * @brief Print delta ports, ordering by severity.
 *
 * @param[in]  key        Host.
 * @param[in]  value      Port tree.
 * @param[in]  stream     Stream.
 * @param[in]  ascending  Ascending or descending.
 */
static gboolean
print_host_ports_by_severity (gpointer key, gpointer value, gpointer stream,
                              int ascending)
{
  guint index, len;
  array_t *ports;

  g_debug ("   delta: %s: host %s", __FUNCTION__, (gchar*) key);

  /* Convert tree to array. */

  ports = make_array ();
  g_tree_foreach ((GTree*) value, array_add_port, ports);

  /* Sort the array. */

  if (ascending)
    g_ptr_array_sort (ports, compare_ports_severity);
  else
    g_ptr_array_sort (ports, compare_ports_severity_desc);

  /* Print the sorted array. */

  index = 0;
  len = ports->len;
  while (index < len)
    {
      gpointer *port_threat;
      port_threat = g_ptr_array_index (ports, index);
      fprintf ((FILE*) stream,
               "<port>"
               "<host>%s</host>"
               "%s"
               "<severity>%1.1f</severity>"
               "<threat>%s</threat>"
               "</port>",
               (gchar*) key,
               (gchar*) port_threat[0],
               *((double*) port_threat[1]),
               severity_to_level (*((double*) port_threat[1]), 0));
      index++;
    }

  array_free (ports);

  return FALSE;
}

/**
 * @brief Print delta ports, ordering by severity descending.
 *
 * @param[in]  key     Host.
 * @param[in]  value   Port tree.
 * @param[in]  stream  Stream.
 */
static gboolean
print_host_ports_by_severity_desc (gpointer key, gpointer value,
                                   gpointer stream)
{
  return print_host_ports_by_severity (key, value, stream, 0);
}

/**
 * @brief Print delta ports, ordering by severity ascending.
 *
 * @param[in]  key     Host.
 * @param[in]  value   Port tree.
 * @param[in]  stream  Stream.
 */
static gboolean
print_host_ports_by_severity_asc (gpointer key, gpointer value,
                                  gpointer stream)
{
  return print_host_ports_by_severity (key, value, stream, 1);
}

/**
 * @brief Free delta host ports.
 *
 * @param[in]  host_ports  Ports.
 * @param[in]  dummy       Dummy.
 */
static gboolean
free_host_ports (GTree *host_ports, gpointer dummy)
{
  g_tree_destroy (host_ports);
  return FALSE;
}

/**
 * @brief Get N'th last report_host given a host.
 *
 * The last report_host is at position 1, the second last at position 2, and
 * so on.
 *
 * @param[in]  host         Host.
 * @param[in]  report_host  Report host.
 * @param[in]  position     Position from end.
 */
gboolean
host_nthlast_report_host (const char *host, report_host_t *report_host,
                          int position)
{
  gchar *quoted_host;

  assert (current_credentials.uuid);

  if (position == 0)
    position = 1;

  quoted_host = sql_quote (host);
  switch (sql_int64 (report_host,
                     "SELECT id FROM report_hosts WHERE host = '%s'"
                     " AND user_owns ('task',"
                     "                (SELECT reports.task FROM reports"
                     "                 WHERE reports.id"
                     "                       = report_hosts.report))"
                     " AND (SELECT tasks.hidden FROM tasks, reports"
                     "      WHERE reports.task = tasks.id"
                     "      AND reports.id = report_hosts.report)"
                     "     = 0"
                     " AND (SELECT value FROM task_preferences, tasks,"
                     "                        reports"
                     "      WHERE reports.task = tasks.id"
                     "      AND reports.id = report_hosts.report"
                     "      AND task_preferences.task = tasks.id"
                     "      AND task_preferences.name = 'in_assets')"
                     "     = 'yes'"
                     " AND report_hosts.end_time IS NOT NULL"
                     " AND NOT EXISTS (SELECT * FROM report_host_details"
                     "                 WHERE report_host = report_hosts.id"
                     "                 AND name = 'CVE Scan')"
                     " ORDER BY id DESC LIMIT 1 OFFSET %i;",
                     quoted_host,
                     position - 1))
    {
      case 0:
        break;
      case 1:        /* Too few rows in result of query. */
        *report_host = 0;
        break;
      default:       /* Programming error. */
        assert (0);
      case -1:
        return TRUE;
        break;
    }

  g_free (quoted_host);
  return FALSE;
}

/**
 * @brief Count host reports.
 *
 * @param[in]  host  Host.
 *
 * @return Report count.
 */
static int
host_report_count (const char *host)
{
  int count;
  gchar *quoted_host;
  assert (current_credentials.uuid);
  quoted_host = sql_quote (host);
  count = sql_int ("SELECT count (*) FROM report_hosts WHERE host = '%s'"
                   "  AND (SELECT reports.owner FROM reports"
                   "       WHERE reports.id = report_hosts.report)"
                   "      = (SELECT id FROM users"
                   "         WHERE users.uuid = '%s')"
                   "  AND (SELECT tasks.hidden FROM tasks, reports"
                   "       WHERE reports.task = tasks.id"
                   "       AND reports.id = report_hosts.report)"
                   "      = 0"
                   "  AND (SELECT reports.scan_run_status FROM reports"
                   "       WHERE reports.id = report_hosts.report)"
                   "      = %u;",
                   quoted_host,
                   current_credentials.uuid,
                   TASK_STATUS_DONE);
  g_free (quoted_host);
  return count;
}

/**
 * @brief Count hosts.
 *
 * @return Host count.
 */
static int
host_count ()
{
  return sql_int ("SELECT count (DISTINCT host) FROM report_hosts"
                  " WHERE host != 'localhost';");
}

/**
 * @brief Count hosts with filtering.
 *
 * @param[in]  levels         Levels.
 * @param[in]  search_phrase  Phrase that host IPs must include.  All hosts
 *                            if NULL or "".
 * @param[in]  apply_overrides  Whether to apply overrides.
 *
 * @return Host count.
 */
static int
filtered_host_count (const char *levels, const char *search_phrase,
                     int apply_overrides)
{
  assert (current_credentials.uuid);

  if (levels && strlen (levels))
    {
      int ret;
      GString *levels_sql;
      gchar *severity_sql, *new_severity_sql, *last_report_sql;

      if (setting_dynamic_severity_int ())
        severity_sql = g_strdup ("CASE WHEN results.severity"
                                 "          > " G_STRINGIFY (SEVERITY_LOG)
                                 " THEN coalesce ((SELECT CAST (cvss_base"
                                 "                              AS REAL)"
                                 "                 FROM nvts"
                                 "                 WHERE nvts.oid"
                                 "                         = results.nvt),"
                                 "                results.severity)"
                                 " ELSE results.severity END");
      else
        severity_sql = g_strdup ("results.severity");

      if (apply_overrides)
        {
          gchar *ov, *owned_clause;

          owned_clause = acl_where_owned_for_get ("override", NULL);

          ov = g_strdup_printf
                ("SELECT overrides.new_severity"
                 " FROM overrides"
                 " WHERE overrides.nvt = results.nvt"
                 " AND %s"
                 " AND ((overrides.end_time = 0)"
                 "      OR (overrides.end_time >= m_now ()))"
                 " AND (overrides.task ="
                 "      (SELECT reports.task FROM reports"
                 "       WHERE reports.id = results.report)"
                 "      OR overrides.task = 0)"
                 " AND (overrides.result = results.id"
                 "      OR overrides.result = 0)"
                 " AND (overrides.hosts is NULL"
                 "      OR overrides.hosts = ''"
                 "      OR hosts_contains (overrides.hosts, results.host))"
                 " AND (overrides.port is NULL"
                 "      OR overrides.port = ''"
                 "      OR overrides.port = results.port)"
                 " AND (severity_matches_ov (%s, overrides.severity))"
                 " ORDER BY overrides.result DESC, overrides.task DESC,"
                 " overrides.port DESC, overrides.severity ASC,"
                 " overrides.creation_time DESC"
                 " LIMIT 1",
                 owned_clause,
                 severity_sql);

          g_free (owned_clause);

          new_severity_sql = g_strdup_printf ("coalesce ((%s),%s)",
                                              ov, severity_sql);

          g_free (ov);
        }
      else
        new_severity_sql = g_strdup_printf ("%s", severity_sql);

      g_free (severity_sql);

      levels_sql = where_levels (levels, new_severity_sql);

      last_report_sql
       = g_strdup_printf (" (SELECT report FROM report_hosts"
                          "  WHERE report_hosts.host = distinct_host"
                          "  AND end_time IS NOT NULL"
                          "  AND report_hosts.report"
                          "      IN (SELECT reports.id FROM reports"
                          "          WHERE user_owns ('task', reports.task))"
                          "  AND (SELECT reports.scan_run_status = %u"
                          "       FROM reports"
                          "       WHERE reports.id = report)"
                          "  AND (SELECT hidden FROM tasks"
                          "       WHERE tasks.id"
                          "             = (SELECT task FROM reports"
                          "                WHERE reports.id = report))"
                          "      = 0"
                          "  AND (SELECT value FROM task_preferences"
                          "       WHERE task_preferences.task"
                          "             = (SELECT task FROM reports"
                          "                WHERE reports.id = report)"
                          "       AND task_preferences.name = 'in_assets')"
                          "      = 'yes'"
                          "  ORDER BY id DESC LIMIT 1)",
                          TASK_STATUS_DONE);

      if (search_phrase && strlen (search_phrase))
        {
          gchar *quoted_search_phrase;

          quoted_search_phrase = sql_quote (search_phrase);
          ret = sql_int
                 ("SELECT"
                  " count (*)"
                  " FROM (SELECT DISTINCT host AS distinct_host"
                  "       FROM report_hosts)"
                  "      AS distinct_host_subquery"
                  /* Search IP. */
                  " WHERE (distinct_host %s '%%%s%%%'"
                  /* Search hostname. */
                  "        OR EXISTS"
                  "        (SELECT * FROM report_host_details"
                  "         WHERE report_host"
                  "               = (SELECT id FROM report_hosts"
                  "                  WHERE report = %s"
                  "                  AND host = distinct_host)"
                  "         AND (name = 'hostname'"
                  "              OR name = 'best_os_txt'"
                  "              OR name = 'best_os_cpe' OR name = 'App'"
                  "              OR name = 'ports')"
                  "         AND source_type = 'nvt'"
                  "         AND value %s '%%%s%%'))"
                  " AND EXISTS (SELECT results.id, %s AS new_severity"
                  "             FROM results"
                  "             WHERE results.report = %s"
                  "             AND results.host = distinct_host"
                  "             AND qod >= " G_STRINGIFY (MIN_QOD_DEFAULT)
                  "             %s);",
                  sql_ilike_op (),
                  quoted_search_phrase,
                  last_report_sql,
                  sql_ilike_op (),
                  quoted_search_phrase,
                  new_severity_sql,
                  last_report_sql,
                  levels_sql ? levels_sql->str : "");
          g_free (quoted_search_phrase);
        }
      else
        ret = sql_int
               ("SELECT"
                " count (*)"
                " FROM (SELECT DISTINCT host AS distinct_host"
                "       FROM report_hosts)"
                "      AS distinct_host_subquery"
                " WHERE EXISTS (SELECT results.id, %s AS new_severity"
                "               FROM results"
                "               WHERE results.report = %s"
                "               AND results.host = distinct_host"
                "               AND qod >= " G_STRINGIFY (MIN_QOD_DEFAULT)
                "               %s);",
                new_severity_sql,
                last_report_sql,
                levels_sql ? levels_sql->str : "");

      if (levels_sql)
        g_string_free (levels_sql, TRUE);
      g_free (new_severity_sql);
      g_free (last_report_sql);

      return ret;
    }
  else if (search_phrase && strlen (search_phrase))
    {
      int retn;
      gchar *quoted_search_phrase;

      quoted_search_phrase = sql_quote (search_phrase);
      retn = sql_int ("SELECT count(*) FROM"
                      " (SELECT host"
                      "  FROM report_hosts"
                      "  WHERE report_hosts.report"
                      "        IN (SELECT reports.id FROM reports"
                      "            WHERE user_owns ('task', reports.task))"
                      "  AND (SELECT tasks.hidden FROM tasks, reports"
                      "       WHERE reports.task = tasks.id"
                      "       AND reports.id = report_hosts.report)"
                      "      = 0"
                      "  AND report_hosts.end_time IS NOT NULL"
                      "  GROUP BY host"
                      "  HAVING host %s '%%%s%%'"
                      "  OR EXISTS"
                      "  (SELECT * FROM report_host_details"
                      "   WHERE report_hosts.id = report_host"
                      "   AND (name = 'hostname' OR name = 'best_os_txt'"
                      "        OR name = 'best_os_cpe' OR name = 'App'"
                      "        OR name = 'ports')"
                      "   AND source_type = 'nvt'"
                      "   AND value %s '%%%s%%')"
                      "  ORDER BY order_inet (host))"
                      " AS distinct_host_subquery;",
                      sql_ilike_op (),
                      quoted_search_phrase,
                      sql_ilike_op (),
                      quoted_search_phrase);
      g_free (quoted_search_phrase);
      return retn;
    }

  return sql_int ("SELECT count(*) FROM"
                  " (SELECT DISTINCT host FROM report_hosts"
                  "  WHERE report_hosts.report"
                  "        IN (SELECT reports.id FROM reports"
                  "            WHERE user_owns ('task', reports.task))"
                  "  AND (SELECT tasks.hidden FROM tasks, reports"
                  "       WHERE reports.task = tasks.id"
                  "       AND reports.id = report_hosts.report)"
                  "      = 0"
                  "  AND report_hosts.end_time IS NOT NULL)"
                  " AS distinct_host_subquery;");
}

/**
 * @brief Buffer host.
 */
struct buffer_host
{
  gchar *ip;                  ///< IP of host.
  report_host_t report_host;  ///< Report host of host.
};

/**
 * @brief Buffer host type.
 */
typedef struct buffer_host buffer_host_t;

/**
 * @brief Free print_report_xml prognostic host buffer.
 *
 * @param[in]  buffer  The buffer.
 */
static void
free_buffer (array_t *buffer)
{
  guint index;
  /* Free the buffer. */
  index = buffer->len;
  while (index--)
    {
      buffer_host_t *buffer_host;
      buffer_host = (buffer_host_t*) g_ptr_array_index (buffer, index);
      if (buffer_host)
        {
          g_free (buffer_host->ip);
          g_free (buffer_host);
        }
    }
  g_ptr_array_free (buffer, TRUE);
}

/**
 * @brief Count a report's total number of hosts.
 *
 * @return Host count.
 */
int
report_host_count (report_t report)
{
  return sql_int ("SELECT count (DISTINCT id) FROM report_hosts"
                  " WHERE report = %llu;",
                  report);
}

/**
 * @brief Count a report's total number of hosts with results.
 *
 * @param[in]   report         Report.
 * @param[in]   min_qod        Minimum QoD of results to count.
 *
 * @return The number of hosts with results
 */
int
report_result_host_count (report_t report, int min_qod)
{
  return sql_int ("SELECT count (DISTINCT id) FROM report_hosts"
                  " WHERE report_hosts.report = %llu"
                  "   AND EXISTS (SELECT * FROM results"
                  "               WHERE results.host = report_hosts.host"
                  "                 AND results.qod >= %d)",
                  report,
                  min_qod);
}

/**
 * @brief Count a report's total number of tcp/ip ports.
 * @brief Ignores ports entries in "general/..." form.
 *
 * @return Ports count.
 */
static int
report_port_count (report_t report)
{
  return sql_int ("SELECT count (DISTINCT port) FROM results"
                  " WHERE report = %llu AND port != ''"
                  "  AND port NOT %s 'general/%';",
                  report,
                  sql_ilike_op ());
}

/**
 * @brief Count a prognostic report host's total number of tcp/ip ports.
 * @brief Ignores ports entries in "general/..." form.
 *
 * @return Ports count for prognostic report host.
 */
static int
prognostic_host_port_count (report_t report, const char *host)
{
  return sql_int ("SELECT count (DISTINCT port) FROM results"
                  " WHERE report = %llu AND host = '%s'"
                  "  AND port NOT %s 'general/%';",
                  report,
                  host,
                  sql_ilike_op ());
}

/**
 * @brief Count a report's total number of closed cves.
 *
 * @return Closed CVE count.
 */
static int
report_closed_cve_count (report_t report)
{
  return sql_int (" SELECT count(id) FROM nvts"
                  " WHERE cve != 'NOCVE'"
                  " AND family IN (" LSC_FAMILY_LIST ")"
                  " AND oid IN"
                  " (SELECT source_name FROM report_host_details"
                  "  WHERE report_host IN "
                  "   (SELECT id FROM report_hosts WHERE report = %llu)"
                  "  AND name = 'EXIT_CODE'"
                  "  AND value = 'EXIT_NOTVULN');",
                  report);
}

/**
 * @brief Count a report's total number of vulnerabilities.
 *
 * @return Vulnerabilities count.
 */
static int
report_vuln_count (report_t report)
{
  return sql_int ("SELECT count (DISTINCT nvt) FROM results"
                  " WHERE report = %llu AND nvt != '0'"
                  " AND severity != " G_STRINGIFY (SEVERITY_ERROR) ";",
                  report);
}

/**
 * @brief Count a report's total number of detected Operating Systems.
 *
 * @return OS count.
 */
static int
report_os_count (report_t report)
{
  return sql_int ("SELECT count (DISTINCT value) FROM report_host_details"
                  " WHERE report_host IN"
                  "  (SELECT id from report_hosts WHERE report = %llu)"
                  "  AND name = 'best_os_cpe';",
                  report);
}

/**
 * @brief Count a report's total number of detected Apps.
 *
 * @return App count.
 */
static int
report_app_count (report_t report)
{
  return sql_int ("SELECT count (DISTINCT value) FROM report_host_details"
                  " WHERE report_host IN"
                  "  (SELECT id from report_hosts WHERE report = %llu)"
                  "  AND name = 'App';",
                  report);
}

/**
 * @brief Count a report's total number of found SSL Certificates.
 *
 * @return SSL Certificates count.
 */
static int
report_ssl_cert_count (report_t report)
{
  return sql_int ("SELECT count (DISTINCT id) FROM report_host_details"
                  " WHERE report_host IN"
                  "  (SELECT id from report_hosts WHERE report = %llu)"
                  "  AND name = 'SSLInfo';",
                  report);
}

/**
 * @brief Count a report's total number of error messages.
 *
 * @return Error Messages count.
 */
static int
report_error_count (report_t report)
{
  return sql_int ("SELECT count (id) FROM results"
                  " WHERE report = %llu and type = 'Error Message';",
                  report);
}

/**
 * @brief Write report host detail to file stream.
 *
 * On error close stream.
 *
 * @param[in]   stream    Stream to write to.
 * @param[in]   details   Report host details iterator.
 *
 * @return 0 success, -1 error.
 */
int
print_report_host_detail (FILE *stream, iterator_t *details)
{
  PRINT (stream,
        "<detail>"
        "<name>%s</name>"
        "<value>%s</value>"
        "<source>"
        "<type>%s</type>"
        "<name>%s</name>"
        "<description>%s</description>"
        "</source>"
        "<extra>%s</extra>"
        "</detail>",
        report_host_details_iterator_name (details),
        report_host_details_iterator_value (details),
        report_host_details_iterator_source_type (details),
        report_host_details_iterator_source_name (details),
        report_host_details_iterator_source_desc (details),
        report_host_details_iterator_extra (details) ?
         report_host_details_iterator_extra (details)
         : "");
  return 0;
}

/**
 * @brief Print the XML for a report's host details to a file stream.
 * @param[in]  report_host  The report host.
 * @param[in]  stream       File stream to write to.
 *
 * @return 0 on success, -1 error.
 */
static int
print_report_host_details_xml (report_host_t report_host, FILE *stream)
{
  iterator_t details;

  init_report_host_details_iterator
   (&details, report_host);
  while (next (&details))
    if (print_report_host_detail (stream, &details))
      return -1;
  cleanup_iterator (&details);

  return 0;
}

/**
 * @brief Write report error message to file stream.
 *
 * @param[in]   stream      Stream to write to.
 * @param[in]   errors      Pointer to report error messages iterator.
 * @param[in]   asset_id    Asset ID.
 */
#define PRINT_REPORT_ERROR(stream, errors, asset_id)                       \
  do                                                                       \
    {                                                                      \
      PRINT (stream,                                                       \
             "<error>"                                                     \
             "<host>"                                                      \
             "%s"                                                          \
             "<asset asset_id=\"%s\"/>"                                    \
             "</host>"                                                     \
             "<port>%s</port>"                                             \
             "<description>%s</description>"                               \
             "<nvt oid=\"%s\">"                                            \
             "<type>nvt</type>"                                            \
             "<name>%s</name>"                                             \
             "<cvss_base>%s</cvss_base>"                                   \
             "</nvt>"                                                      \
             "<scan_nvt_version>%s</scan_nvt_version>"                     \
             "<severity>%s</severity>"                                     \
             "</error>",                                                   \
             report_errors_iterator_host (errors),                         \
             asset_id ? asset_id : "",                                     \
             report_errors_iterator_port (errors),                         \
             report_errors_iterator_desc (errors),                         \
             report_errors_iterator_nvt_oid (errors),                      \
             report_errors_iterator_nvt_name (errors),                     \
             report_errors_iterator_nvt_cvss (errors),                     \
             report_errors_iterator_scan_nvt_version (errors),             \
             report_errors_iterator_severity (errors));                    \
    }                                                                      \
  while (0)

/**
 * @brief Print the XML for a report's error messages to a file stream.
 * @param[in]  report   The report.
 * @param[in]  stream   File stream to write to.
 *
 * @return 0 on success, -1 error.
 */
static int
print_report_errors_xml (report_t report, FILE *stream)
{
  iterator_t errors;

  init_report_errors_iterator
   (&errors, report);

  PRINT (stream, "<errors><count>%i</count>", report_error_count (report));
  while (next (&errors))
    {
      char *asset_id;

      asset_id = result_host_asset_id (report_errors_iterator_host (&errors),
                                       report_errors_iterator_result (&errors));
      PRINT_REPORT_ERROR (stream, &errors, asset_id);
      free (asset_id);
    }
  cleanup_iterator (&errors);
  PRINT (stream, "</errors>");

  return 0;
}

/**
 * @brief Print the XML for a report of type assets.
 * @param[in]  out              File stream to write to.
 * @param[in]  host             Host or NULL.
 * @param[in]  first_result     The result to start from. The results are 0
 *                              indexed.
 * @param[in]  max_results      The maximum number of results returned.
 * @param[in]  levels           String describing threat levels (message types)
 * @param[in]  search_phrase    Phrase that results must include.
 * @param[in]  pos              Position of report from end.
 * @param[in]  get              GET command data.
 * @param[in]  apply_overrides  Whether to apply overrides.
 * @param[in]  autofp           Whether to apply the auto FP filter.
 *
 * @return 0 on success, -1 error.
 */
static int
print_report_assets_xml (FILE *out, const char *host, int first_result, int
                         max_results, gchar *levels, gchar *search_phrase,
                         int pos, const get_data_t *get, int apply_overrides,
                         int autofp)
{
  iterator_t hosts;

  if (host)
    {
      PRINT (out,
             "<host_count>"
             "<full>1</full>"
             "<filtered>1</filtered>"
             "</host_count>"
             "<hosts start=\"1\" max=\"1\"/>");
    }
  else
    {
      // TODO Slow (about 30% of assets page).
      init_classic_asset_iterator (&hosts, first_result, max_results, levels,
                                   search_phrase, apply_overrides);
      PRINT (out,
             "<host_count>"
             "<full>%i</full>"
             "<filtered>%i</filtered>"
             "</host_count>",
             host_count (),
             // TODO Slow (about 30% of assets page).
             filtered_host_count (levels, search_phrase, apply_overrides));
      PRINT (out,
             "<hosts start=\"%i\" max=\"%i\"/>",
             /* Add 1 for 1 indexing. */
             first_result + 1,
             max_results);
    }

  while (host || next (&hosts))
    {
      report_host_t report_host;
      const char *ip;

      ip = host ? host : classic_asset_iterator_ip (&hosts);

      if (host_nthlast_report_host (ip, &report_host, pos))
        {
          if (host == NULL)
            cleanup_iterator (&hosts);

          return -1;
        }

      if (report_host)
        {
          iterator_t report_hosts;
          init_report_host_iterator (&report_hosts, 0, NULL, report_host);
          if (next (&report_hosts))
            {
              iterator_t details;
              report_t report;
              int holes, infos, logs, warnings, false_positives;
              double severity, highest_cvss;

              PRINT (out,
                     "<host>"
                     "<ip>%s</ip>"
                     "<start>%s</start>"
                     "<end>%s</end>",
                     ip,
                     host_iterator_start_time (&report_hosts),
                     host_iterator_end_time (&report_hosts));

              PRINT (out,
                     "<detail>"
                     "<name>report/@id</name>"
                     "<value>%s</value>"
                     "<source>"
                     "<type></type>"
                     "<name>openvasmd</name>"
                     "<description>UUID of current report</description>"
                     "</source>"
                     "</detail>",
                     host_iterator_report_uuid (&report_hosts));

              PRINT (out,
                     "<detail>"
                     "<name>report_count</name>"
                     "<value>%i</value>"
                     "<source>"
                     "<type></type>"
                     "<name>openvasmd</name>"
                     "<description>Number of reports</description>"
                     "</source>"
                     "</detail>",
                     host_report_count (ip));

              report = host_iterator_report (&report_hosts);

              report_counts_id (report, NULL, &holes, &infos, &logs,
                                &warnings, &false_positives, &severity,
                                get, ip);

              PRINT (out,
                     "<detail>"
                     "<name>report/result_count/high</name>"
                     "<value>%i</value>"
                     "<source>"
                     "<type></type>"
                     "<name>openvasmd</name>"
                     "<description>Number of highs</description>"
                     "</source>"
                     "</detail>",
                     holes);

              PRINT (out,
                     "<detail>"
                     "<name>report/result_count/medium</name>"
                     "<value>%i</value>"
                     "<source>"
                     "<type></type>"
                     "<name>openvasmd</name>"
                     "<description>Number of mediums</description>"
                     "</source>"
                     "</detail>",
                     warnings);

              PRINT (out,
                     "<detail>"
                     "<name>report/result_count/low</name>"
                     "<value>%i</value>"
                     "<source>"
                     "<type></type>"
                     "<name>openvasmd</name>"
                     "<description>Number of lows</description>"
                     "</source>"
                     "</detail>",
                     infos);

              PRINT (out,
                     "<detail>"
                     "<name>report/result_count/log</name>"
                     "<value>%i</value>"
                     "<source>"
                     "<type></type>"
                     "<name>openvasmd</name>"
                     "<description>Number of logs</description>"
                     "</source>"
                     "</detail>",
                     logs);
              /* Print all the host details. */

              highest_cvss = -1;
              init_report_host_details_iterator
               (&details, report_host);
              while (next (&details))
                {
                  const char *value;
                  value = report_host_details_iterator_value (&details);

                  if (print_report_host_detail (out, &details))
                    return -1;

                  if (manage_scap_loaded ()
                      && get->details
                      && (strcmp (report_host_details_iterator_name
                                   (&details),
                                  "App")
                          == 0))
                    {
                      iterator_t prognosis;
                      double cvss;
                      int first;

                      /* Print details of all CVEs on the App. */

                      first = 1;
                      cvss = -1;
                      init_prognosis_iterator (&prognosis, value);
                      while (next (&prognosis))
                        {
                          if (first)
                            {
                              cvss = prognosis_iterator_cvss_double
                                      (&prognosis);
                              first = 0;
                            }

                          PRINT (out,
                                 "<detail>"
                                 "<name>%s/CVE</name>"
                                 "<value>%s</value>"
                                 "</detail>"
                                 "<detail>"
                                 "<name>%s/%s/CVSS</name>"
                                 "<value>%s</value>"
                                 "</detail>",
                                 value,
                                 prognosis_iterator_cve (&prognosis),
                                 value,
                                 prognosis_iterator_cve (&prognosis),
                                 prognosis_iterator_cvss (&prognosis));
                        }

                      /* Print App prognosis, according to highest CVSS. */

                      if (cvss >= 0)
                        PRINT (out,
                               "<detail>"
                               "<name>%s/threat</name>"
                               "<value>%s</value>"
                               "</detail>",
                               value,
                               severity_to_level (cvss, 0));
                      cleanup_iterator (&prognosis);
                    }

                  if (manage_scap_loaded ()
                      && (strcmp (report_host_details_iterator_name
                                   (&details),
                                  "App")
                          == 0))
                    {
                      int highest;
                      /* Check if this App's CVSS is the highest CVSS for
                       * this host. */
                      highest = cpe_highest_cvss (value);
                      if (highest > highest_cvss)
                        highest_cvss = highest;
                    }
                }
              cleanup_iterator (&details);

              /* Print prognosis of host, according to highest CVSS. */

              if (highest_cvss > 0)
                PRINT (out,
                       "<detail>"
                       "<name>prognosis</name>"
                       "<value>%s</value>"
                       "</detail>",
                       severity_to_level (highest_cvss, 0));

              PRINT (out,
                     "<detail>"
                     "<name>report/pos</name>"
                     "<value>%i</value>"
                     "<source>"
                     "<type></type>"
                     "<name>openvasmd</name>"
                     "<description>Position of report from end</description>"
                     "</source>"
                     "</detail>",
                     pos);
            }
          cleanup_iterator (&report_hosts);

          PRINT (out,
                 "</host>");
        }

      if (host)
        break;
    }
  if (host == NULL)
    cleanup_iterator (&hosts);

  return 0;
}

/**
 * @brief Print the XML for a report port summary to a file.
 *
 * @param[in]  report           The report.
 * @param[in]  out              File stream.
 * @param[in]  get              Result get data.
 * @param[in]  first_result     The result to start from.  The results are 0
 *                              indexed.
 * @param[in]  max_results      The maximum number of results returned.
 * @param[in]  sort_order       Whether to sort ascending or descending.
 * @param[in]  sort_field       Field to sort on.
 * @param[out] host_ports       Hash table for counting ports per host.
 *
 * @return 0 on success, -1 error.
 */
static int
print_report_port_xml (report_t report, FILE *out, const get_data_t *get,
                       int first_result, int max_results,
                       int sort_order, const char *sort_field,
                       GHashTable *host_ports)
{
  iterator_t results;
  result_buffer_t *last_item;
  GArray *ports = g_array_new (TRUE, FALSE, sizeof (gchar*));

  init_result_get_iterator (&results, get, report, NULL, NULL);

  /* Buffer the results, removing duplicates. */

  last_item = NULL;
  while (next (&results))
    {
      const char *port = result_iterator_port (&results);
      const char *host = result_iterator_host (&results);
      double cvss_double;

      cvss_double = result_iterator_severity_double (&results);

      if (last_item
          && strcmp (port, last_item->port) == 0
          && strcmp (host, last_item->host) == 0
          && last_item->severity_double <= cvss_double)
        {
          last_item->severity_double = cvss_double;
          g_free (last_item->severity);
          last_item->severity = g_strdup (result_iterator_severity (&results));
        }
      else
        {
          const char *cvss;
          result_buffer_t *item;

          cvss = result_iterator_severity (&results);
          if (cvss == NULL)
            {
              cvss_double = 0.0;
              cvss = "0.0";
            }
          item = result_buffer_new (host, port, cvss, cvss_double);
          g_array_append_val (ports, item);
          last_item = item;
        }

    }

  /* Handle sorting by threat and ROWID. */

  if (sort_field == NULL || strcmp (sort_field, "port"))
    {
      int index, length;

      /** @todo Sort by ROWID if was requested. */

      /* Sort by port then severity. */

      g_array_sort (ports, compare_port_severity);

      /* Remove duplicates. */

      last_item = NULL;
      for (index = 0, length = ports->len; index < length; index++)
        {
          result_buffer_t *item;

          item = g_array_index (ports, result_buffer_t*, index);
          if (last_item
              && (strcmp (item->port, last_item->port) == 0)
              && (strcmp (item->host, last_item->host) == 0))
            {
              if (item->severity_double > last_item->severity_double)
                {
                  gchar *severity;
                  severity = last_item->severity;
                  last_item->severity = item->severity;
                  item->severity = severity;
                  last_item->severity_double = item->severity_double;
                }
              g_array_remove_index (ports, index);
              length = ports->len;
              index--;
            }
          else
            last_item = item;
        }

      /* Sort by severity. */

      if (sort_order)
        g_array_sort (ports, compare_severity_asc);
      else
        g_array_sort (ports, compare_severity_desc);
    }

  /* Write to file from the buffer. */

  PRINT (out,
           "<ports"
           " start=\"%i\""
           " max=\"%i\">"
           "<count>%i</count>",
           /* Add 1 for 1 indexing. */
           first_result + 1,
           max_results,
           report_port_count (report));
  {
    result_buffer_t *item;
    int index = 0;

    while ((item = g_array_index (ports, result_buffer_t*, index++)))
      {
        int host_port_count
              = GPOINTER_TO_INT (g_hash_table_lookup (host_ports, item->host));

        PRINT (out,
               "<port>"
               "<host>%s</host>"
               "%s"
               "<severity>%1.1f</severity>"
               "<threat>%s</threat>"
               "</port>",
               item->host,
               item->port,
               item->severity_double,
               severity_to_level (g_strtod (item->severity, NULL), 0));

        if (g_str_has_prefix(item->port, "general/") == FALSE)
          {
            g_hash_table_replace (host_ports,
                                  g_strdup (item->host),
                                  GINT_TO_POINTER (host_port_count + 1));
          }
        result_buffer_free (item);
      }
    g_array_free (ports, TRUE);
  }
  PRINT (out, "</ports>");
  cleanup_iterator (&results);

  return 0;
}

/**
 * @brief Print the XML for a report of type assets.
 *
 * @param[in]  out              File stream to write to.
 * @param[in]  host             Host or NULL.
 * @param[in]  first_result     The result to start from. The results are 0
 *                              indexed.
 * @param[in]  max_results      The maximum number of results returned.
 * @param[in]  levels           String describing threat levels (message types)
 * @param[in]  search_phrase    Phrase that results must include.
 * @param[in]  pos              Position of report from end.
 * @param[in]  get              GET command data.
 * @param[in]  apply_overrides  Whether to apply overrides.
 * @param[in]  autofp           Whether to apply the auto FP filter.
 * @param[in]  host_search_phrase  Phrase that results must include.  All results
 *                                 if NULL or "".  For hosts.
 * @param[in]  host_levels         String describing threat levels (message types)
 *                                 to include in count (for example, "hmlgd" for
 *                                 High, Medium, Low, loG and Debug).  All levels if
 *                                 NULL.
 * @param[in]  host_first_result   The host result to start from.  The results
 *                                 are 0 indexed.
 * @param[in]  host_max_results    The host maximum number of results returned.
 * @param[in]  result_hosts_only   Whether to show only hosts with results.
 * @param[in]  sort_order          Whether to sort in ascending order.
 * @param[in]  sort_field          Name of the field to sort by.
 *
 * @return 0 on success, -1 error.
 */
static int
print_report_prognostic_xml (FILE *out, const char *host, int first_result, int
                             max_results, gchar *levels, gchar *search_phrase,
                             int pos, const get_data_t *get,
                             int apply_overrides, int autofp,
                             const char *host_search_phrase,
                             const char *host_levels, int host_first_result,
                             int host_max_results, int result_hosts_only,
                             int sort_order, const char *sort_field)
{
  array_t *buffer, *apps;
  buffer_host_t *buffer_host;
  int index, skip, result_total, total_host_count;
  int holes, infos, logs, warnings;
  int f_holes, f_infos, f_logs, f_warnings;
  iterator_t hosts;
  time_t now;
  gchar *scan_start, *scan_end;

  time (&now);

  if (host == NULL)
    {
      host_levels = host_levels ? host_levels : "hmlgd";

      init_classic_asset_iterator (&hosts, host_first_result, host_max_results,
                                   host_levels, host_search_phrase,
                                   apply_overrides);
    }

  buffer = make_array ();
  apps = make_array ();
  holes = warnings = infos = logs = 0;
  f_holes = f_warnings = f_infos = f_logs = 0;
  skip = 0;
  total_host_count = 0;
  result_total = 0;

  /* Output the results, buffering the associated hosts. */

  PRINT (out,
         "<results start=\"%i\" max=\"%i\">",
         /* Add 1 for 1 indexing. */
         first_result + 1,
         max_results);

  while (host || next (&hosts))
    {
      iterator_t report_hosts;
      report_host_t report_host;
      const char *ip;

      ip = host ? host : classic_asset_iterator_ip (&hosts);

      if (host_nthlast_report_host (ip, &report_host, pos))
        {
          if (host == NULL)
            cleanup_iterator (&hosts);
          free_buffer (buffer);
          array_free (apps);
          return -1;
        }

      if (report_host)
        {
          int filtered, host_result_total;
          filtered = 0;
          host_result_total = 0;

          total_host_count++;

          prognostic_report_result_total (report_host, &host_result_total);
          result_total += host_result_total;

          prognostic_report_result_count (report_host, search_phrase,
                                          &filtered, &f_holes,
                                          &f_infos, &f_warnings);
          filtered = (strchr (levels, 'h') ? f_holes : 0)
                      + (strchr (levels, 'l') ? f_infos : 0)
                      + (strchr (levels, 'g') ? f_logs : 0)
                      + (strchr (levels, 'm') ? f_warnings : 0);
          if (result_hosts_only && !filtered)
            /* Skip this host. */
            report_host = 0;
        }

      if (report_host)
        {
          init_report_host_iterator (&report_hosts, 0, NULL, report_host);
          if (next (&report_hosts))
            {
              iterator_t prognosis;
              int buffered;

              buffered = 0;

              init_host_prognosis_iterator (&prognosis, report_host,
                                            0, -1,
                                            levels, search_phrase,
                                            sort_order, sort_field);
              while (next (&prognosis))
                {
                  const char *threat;

                  threat = severity_to_level (prognosis_iterator_cvss_double
                                                (&prognosis), 0);

                  array_add_new_string (apps,
                                        prognosis_iterator_cpe (&prognosis));

                  if (skip < first_result)
                    {
                      /* Skip result. */
                      skip++;
                      continue;
                    }

                   if (max_results == 0)
                     continue;

                   buffered = 1;

                   PRINT (out,
                          "<result>"
                          "<host>%s</host>"
                          "<port>0</port>"
                          "<nvt oid=\"%s\">"
                          "<type>cve</type>"
                          "<name>%s</name>"
                          "<cvss_base>%s</cvss_base>"
                          "<cpe id='%s'/>"
                          "</nvt>"
                          "<threat>%s</threat>"
                          "<description>"
                          "The host carries the product: %s\n"
                          "It is vulnerable according to: %s.\n"
                          "\n"
                          "%s"
                          "</description>"
                          "</result>",
                          ip,
                          prognosis_iterator_cve (&prognosis),
                          prognosis_iterator_cve (&prognosis),
                          prognosis_iterator_cvss (&prognosis),
                          prognosis_iterator_cpe (&prognosis),
                          threat,
                          prognosis_iterator_cpe (&prognosis),
                          prognosis_iterator_cve (&prognosis),
                          prognosis_iterator_description
                           (&prognosis));

                   max_results--;
                 }
               if (buffered || (result_hosts_only == 0))
                 {
                   /* Buffer IP and report_host. */
                   buffer_host_t *buffer_host;
                   buffer_host = (buffer_host_t*) g_malloc (sizeof (buffer_host_t));
                   buffer_host->report_host = report_host;
                   buffer_host->ip = g_strdup (ip);
                   array_add (buffer, buffer_host);
                 }
              cleanup_iterator (&prognosis);
            }
          cleanup_iterator (&report_hosts);
        }

      if (host)
        break;
    }
  if (host == NULL)
    cleanup_iterator (&hosts);

  PRINT (out,
         "</results>");

  /* Output buffered hosts. */

  if (host)
    {
      PRINT (out,
             "<host_count>"
             "<full>1</full>"
             "<filtered>1</filtered>"
             "</host_count>"
             "<hosts start=\"1\" max=\"1\"/>");
    }
  else
    {
      PRINT (out,
             "<host_count>"
             "<full>%i</full>"
             "<filtered>%i</filtered>"
             "</host_count>",
             host_count (),
             buffer->len);
      PRINT (out,
             "<hosts start=\"%i\" max=\"%i\"/>",
             /* Add 1 for 1 indexing. */
             host_first_result + 1,
             host_max_results);
    }

  scan_start = g_strdup (iso_time (&now));
  time (&now);
  scan_end = g_strdup (iso_time (&now));

  PRINT (out,
         "<scan_start>%s</scan_start>"
         "<scan_end>%s</scan_end>",
         scan_start,
         scan_end);

  array_terminate (buffer);
  index = 0;
  while ((buffer_host = g_ptr_array_index (buffer, index++)))
    {
      iterator_t report_hosts;
      init_report_host_iterator (&report_hosts, 0, NULL, buffer_host->report_host);
      if (next (&report_hosts))
        {
          report_t report;
          int h_holes, h_infos, h_logs, h_warnings, h_false_positives;
          double h_severity;

          PRINT (out,
                 "<host_start>"
                 "<host>%s</host>%s"
                 "</host_start>"
                 "<host_end>"
                 "<host>%s</host>%s"
                 "</host_end>",
                 buffer_host->ip,
                 scan_start,
                 buffer_host->ip,
                 scan_end);

          PRINT (out,
                 "<host>"
                 "<ip>%s</ip>"
                 "<start>%s</start>"
                 "<end>%s</end>",
                 buffer_host->ip,
                 host_iterator_start_time (&report_hosts),
                 host_iterator_end_time (&report_hosts));

          PRINT (out,
                 "<detail>"
                 "<name>report/@id</name>"
                 "<value>%s</value>"
                 "<source>"
                 "<type></type>"
                 "<name>openvasmd</name>"
                 "<description>UUID of current report</description>"
                 "</source>"
                 "</detail>",
                 host_iterator_report_uuid (&report_hosts));

          PRINT (out,
                 "<detail>"
                 "<name>port_count</name>"
                 "<value>%i</value>"
                 "<source>"
                 "<type></type>"
                 "<name>openvasmd</name>"
                 "<description>Ports number of current host</description>"
                 "</source>"
                 "</detail>",
                 prognostic_host_port_count
                  (host_iterator_report (&report_hosts), buffer_host->ip));

          PRINT (out,
                 "<detail>"
                 "<name>report_count</name>"
                 "<value>%i</value>"
                 "<source>"
                 "<type></type>"
                 "<name>openvasmd</name>"
                 "<description>Number of reports</description>"
                 "</source>"
                 "</detail>",
                 host_report_count (buffer_host->ip));

          report = host_iterator_report (&report_hosts);

          report_counts_id (report, NULL, &h_holes, &h_infos, &h_logs,
                            &h_warnings, &h_false_positives, &h_severity,
                            get, buffer_host->ip);

          PRINT (out,
                 "<detail>"
                 "<name>report/result_count/high</name>"
                 "<value>%i</value>"
                 "<source>"
                 "<type></type>"
                 "<name>openvasmd</name>"
                 "<description>Number of highs</description>"
                 "</source>"
                 "</detail>",
                 h_holes);

          PRINT (out,
                 "<detail>"
                 "<name>report/result_count/medium</name>"
                 "<value>%i</value>"
                 "<source>"
                 "<type></type>"
                 "<name>openvasmd</name>"
                 "<description>Number of mediums</description>"
                 "</source>"
                 "</detail>",
                 h_warnings);

          PRINT (out,
                 "<detail>"
                 "<name>report/result_count/low</name>"
                 "<value>%i</value>"
                 "<source>"
                 "<type></type>"
                 "<name>openvasmd</name>"
                 "<description>Number of lows</description>"
                 "</source>"
                 "</detail>",
                 h_infos);

          if (print_report_host_details_xml
               (buffer_host->report_host, out))
            {
              array_free (apps);
              return -1;
            }

          PRINT (out,
                 "<detail>"
                 "<name>report/pos</name>"
                 "<value>%i</value>"
                 "<source>"
                 "<type></type>"
                 "<name>openvasmd</name>"
                 "<description>Position of report from end</description>"
                 "</source>"
                 "</detail>",
                 pos);

          PRINT (out,
                 "</host>");
        }
      cleanup_iterator (&report_hosts);
    }

  g_free (scan_start);
  g_free (scan_end);
  free_buffer (buffer);

  PRINT (out,
         "<result_count>"
         "%i"
         "<full>%i</full>"
         "<filtered>%i</filtered>"
         "<debug><full>0</full><filtered>0</filtered></debug>"
         "<hole><full>%i</full><filtered>%i</filtered></hole>"
         "<info><full>%i</full><filtered>%i</filtered></info>"
         "<log><full>%i</full><filtered>%i</filtered></log>"
         "<warning><full>%i</full><filtered>%i</filtered></warning>"
         "<false_positive>"
         "<full>0</full>"
         "<filtered>0</filtered>"
         "</false_positive>"
         "</result_count>",
         result_total,
         result_total,
         (strchr (levels, 'h') ? f_holes : 0)
          + (strchr (levels, 'l') ? f_infos : 0)
          + (strchr (levels, 'g') ? f_logs : 0)
          + (strchr (levels, 'm') ? f_warnings : 0),
         holes,
         (strchr (levels, 'h') ? f_holes : 0),
         infos,
         (strchr (levels, 'l') ? f_infos : 0),
         logs,
         (strchr (levels, 'g') ? f_logs : 0),
         warnings,
         (strchr (levels, 'm') ? f_warnings : 0));

  PRINT (out,
         "<hosts><count>%i</count></hosts>",
         total_host_count);

  PRINT (out,
         "<apps><count>%i</count></apps>",
         apps->len);

  array_free (apps);

  return 0;
}

/**
 * @brief Check whether the scan of a report is active.
 *
 * @param[in]  report         Report.
 *
 * @return 1 if active, else 0.
 */
static int
report_active (report_t report)
{
  int run_status;
  report_scan_run_status (report, &run_status);
  if (run_status == TASK_STATUS_REQUESTED
      || run_status == TASK_STATUS_RUNNING
      || run_status == TASK_STATUS_DELETE_REQUESTED
      || run_status == TASK_STATUS_DELETE_ULTIMATE_REQUESTED
      || run_status == TASK_STATUS_STOP_REQUESTED
      || run_status == TASK_STATUS_STOP_REQUESTED_GIVEUP
      || run_status == TASK_STATUS_STOPPED)
    return 1;
  return 0;
}

/**
 * @brief Get progress for active report.
 *
 * @param[in]  report         Report.
 * @param[in]  maximum_hosts  Maximum number of hosts in target.
 * @param[out] hosts_xml      Return for hosts XML if required, else NULL.
 *
 * @return Progress XML.
 */
static int
report_progress_active (report_t report, long maximum_hosts, gchar **hosts_xml)
{
  long total = 0, num_hosts = 0, dead_hosts = 0;
  int total_progress;
  iterator_t hosts;
  GString *string;

  string = g_string_new ("");

  init_report_host_iterator (&hosts, report, NULL, 0);
  while (next (&hosts))
    {
      unsigned int max_port, current_port;
      long progress;

      max_port = host_iterator_max_port (&hosts);
      current_port = host_iterator_current_port (&hosts);
      if (max_port)
        {
          if (max_port == -1)
            dead_hosts++;
          progress = (current_port * 100) / max_port;
          if (progress < 0) progress = 0;
          else if (progress > 100) progress = 100;
        }
      else
        progress = current_port ? 100 : 0;

      total += progress;
      num_hosts++;

      if (hosts_xml)
        g_string_append_printf (string,
                                "<host_progress>"
                                "<host>%s</host>"
                                "%li"
                                "</host_progress>",
                                host_iterator_host (&hosts),
                                progress);
    }
  cleanup_iterator (&hosts);

  total_progress = (maximum_hosts - dead_hosts)
                   ? (total / (maximum_hosts - dead_hosts)) : 0;

#if 1
  g_debug ("   total: %li\n", total);
  g_debug ("   num_hosts: %li\n", num_hosts);
  g_debug ("   maximum_hosts: %li\n", maximum_hosts);
  g_debug ("   total_progress: %i\n", total_progress);
#endif

  if (total_progress == 0) total_progress = 1;
  else if (total_progress == 100) total_progress = 99;

  if (hosts_xml)
    *hosts_xml = g_string_free (string, FALSE);
  else
    g_string_free (string, TRUE);

  return total_progress;
}

/**
 * @brief Calculate the progress of a report.
 *
 * @param[in]  report     Report.
 * @param[in]  task       Report's task.
 * @param[out] hosts_xml  Return for hosts XML if required, else NULL.
 *
 * @return Progress.
 */
int
report_progress (report_t report, task_t task, gchar **hosts_xml)
{
  target_t target;
  char *hosts, *exclude_hosts;
  int progress;
  long maximum_hosts;

  if (report == 0)
    {
      if (hosts_xml)
        *hosts_xml = g_strdup ("");
      return -1;
    }

  /* Handles both slave and OSP cases. */
  if ((progress = report_slave_progress (report)))
    {
      if (hosts_xml)
        *hosts_xml = g_strdup ("");
      return progress;
    }

  target = task_target (task);
  if (task_target_in_trash (task))
    {
      hosts = target ? trash_target_hosts (target) : NULL;
      exclude_hosts = target ? trash_target_exclude_hosts
                                (target) : NULL;
    }
  else
    {
      hosts = target ? target_hosts (target) : NULL;
      exclude_hosts = target ? target_exclude_hosts (target) : NULL;
    }
  maximum_hosts = hosts ? manage_count_hosts_max (hosts, exclude_hosts, 0) : 0;
  g_free (hosts);
  g_free (exclude_hosts);

  if (report_active (report))
    return report_progress_active (report, maximum_hosts, hosts_xml);

  if (hosts_xml)
    *hosts_xml = g_strdup ("");

  return -1;
}

/**
 * @brief Buffer XML for a severity class.
 *
 * @param[in]  severity  Severity name.
 *
 * @return Freshly allocated XML on success, else NULL.
 */
gchar *
severity_class_xml (const gchar *severity)
{
  if (severity)
    {
      if ((strcmp (severity, "nist") == 0)
          || (strcmp (severity, "bsi") == 0))
        return g_strdup_printf ("<severity_class"
                                " id=\"d4c74cda-89e1-11e3-9c29-406186ea4fc5\">"
                                "<name>nist</name>"
                                "<full_name>%s</full_name>"
                                "<severity_range>"
                                "<name>None</name>"
                                "<min>0.0</min>"
                                "<max>0.0</max>"
                                "</severity_range>"
                                "<severity_range>"
                                "<name>Low</name>"
                                "<min>0.1</min>"
                                "<max>3.9</max>"
                                "</severity_range>"
                                "<severity_range>"
                                "<name>Medium</name>"
                                "<min>4.0</min>"
                                "<max>6.9</max>"
                                "</severity_range>"
                                "<severity_range>"
                                "<name>High</name>"
                                "<min>7.0</min>"
                                "<max>10.0</max>"
                                "</severity_range>"
                                "</severity_class>",
                                strcmp (severity, "nist") == 0
                                 ? "NVD Vulnerability Severity Ratings"
                                 : "BSI Schwachstellenampel (Germany)");
      else if (strcmp (severity, "classic") == 0)
        return g_strdup_printf ("<severity_class"
                                " id=\"dc1d556a-89e1-11e3-bc21-406186ea4fc5\">"
                                "<name>classic</name>"
                                "<full_name>OpenVAS Classic</full_name>"
                                "<severity_range>"
                                "<name>None</name>"
                                "<min>0.0</min>"
                                "<max>0.0</max>"
                                "</severity_range>"
                                "<severity_range>"
                                "<name>Low</name>"
                                "<min>0.1</min>"
                                "<max>2.0</max>"
                                "</severity_range>"
                                "<severity_range>"
                                "<name>Medium</name>"
                                "<min>2.1</min>"
                                "<max>5.0</max>"
                                "</severity_range>"
                                "<severity_range>"
                                "<name>High</name>"
                                "<min>5.1</min>"
                                "<max>10.0</max>"
                                "</severity_range>"
                                "</severity_class>");
      else if (strcmp (severity, "pci-dss") == 0)
        return g_strdup_printf ("<severity_class"
                                " id=\"e442e476-89e1-11e3-bfc6-406186ea4fc5\">"
                                "<name>pci-dss</name>"
                                "<full_name>PCI-DSS</full_name>"
                                "<severity_range>"
                                "<name>None</name>"
                                "<min>0.0</min>"
                                "<max>3.9</max>"
                                "</severity_range>"
                                "<severity_range>"
                                "<name>High</name>"
                                "<min>4.0</min>"
                                "<max>10.0</max>"
                                "</severity_range>"
                                "</severity_class>");
    }
  return NULL;
}

/**
 * @brief Restore original TZ.
 *
 * @param[in]  zone             Only revert if this is at least one character.
 *                               Freed here always.
 * @param[in]  tz               Original TZ.  Freed here if revert occurs.
 * @param[in]  old_tz_override  Original tz_override.  Freed here on revert.
 *
 * @return 0 success, -1 error.
 */
static int
tz_revert (gchar *zone, char *tz, char *old_tz_override)
{
  if (zone && strlen (zone))
    {
      gchar *quoted_old_tz_override;
      /* Revert to stored TZ. */
      if (tz)
        {
          if (setenv ("TZ", tz, 1) == -1)
            {
              g_warning ("%s: Failed to switch to original TZ", __FUNCTION__);
              g_free (tz);
              g_free (zone);
              free (old_tz_override);
              return -1;
            }
        }
      else
        unsetenv ("TZ");

      quoted_old_tz_override = sql_insert (old_tz_override);
      sql ("UPDATE current_credentials SET tz_override = %s",
           quoted_old_tz_override);
      g_free (quoted_old_tz_override);

      free (old_tz_override);
      g_free (tz);
    }
  g_free (zone);
  return 0;
}

/**
 * @brief Print the XML for a report to a file.
 *
 * @param[in]  host_summary_buffer  Summary.
 * @param[in]  host                 Host.
 * @param[in]  start_iso            Start time, in ISO format.
 * @param[in]  end_iso              End time, in ISO format.
 */
static void
host_summary_append (GString *host_summary_buffer, const char *host,
                     const char *start_iso, const char *end_iso)
{
  if (host_summary_buffer)
    {
      char start[200], end[200];

      if (start_iso)
        {
          struct tm start_tm;

          memset (&start_tm, 0, sizeof (struct tm));
          if (strptime (start_iso, "%FT%H:%M:%S", &start_tm) == NULL)
            {
              g_warning ("%s: Failed to parse start", __FUNCTION__);
              return;
            }

          if (strftime (start, 200, "%b %d, %H:%M:%S", &start_tm) == 0)
            {
              g_warning ("%s: Failed to format start", __FUNCTION__);
              return;
            }
        }
      else
        strcpy (start, "(not started)");

      if (end_iso)
        {
          struct tm end_tm;

          memset (&end_tm, 0, sizeof (struct tm));
          if (strptime (end_iso, "%FT%H:%M:%S", &end_tm) == NULL)
            {
              g_warning ("%s: Failed to parse end", __FUNCTION__);
              return;
            }

          if (strftime (end, 200, "%b %d, %H:%M:%S", &end_tm) == 0)
            {
              g_warning ("%s: Failed to format end", __FUNCTION__);
              return;
            }
        }
      else
        strcpy (end, "(not finished)");

      g_string_append_printf (host_summary_buffer,
                              "   %-15s   %-16s   %s\n",
                              host,
                              start,
                              end);
    }
}

/**
 * @brief Init delta iterators for print_report_xml.
 *
 * @param[in]  report         The report.
 * @param[in]  results        Report result iterator.
 * @param[in]  delta          Delta report.
 * @param[in]  delta_results  Delta report result iterator.
 * @param[in]  get            GET command data.
 * @param[in]  term           Filter term.
 * @param[out] sort_field     Sort field.
 *
 * @return 0 on success, -1 error.
 */
static int
init_delta_iterators (report_t report, iterator_t *results, report_t delta,
                      iterator_t *delta_results, const get_data_t *get,
                      const char *term, const char *sort_field)
{
  int res;
  gchar *order;
  get_data_t delta_get;

  /*
   * Order must be the same as in result_cmp, except for description
   *  which isn't checked there.
   */
  if ((strcmp (sort_field, "name") == 0)
      || (strcmp (sort_field, "vulnerability") == 0))
    order = g_strdup (", host, port, severity, nvt, description");
  else if (strcmp (sort_field, "host") == 0)
    order = g_strdup (", port, severity, nvt, description");
  else if ((strcmp (sort_field, "port") == 0)
           || (strcmp (sort_field, "location") == 0))
    order = g_strdup (", host, severity, nvt, description");
  else if (strcmp (sort_field, "severity") == 0)
    order = g_strdup (", host, port, nvt, description");
  else if (strcmp (sort_field, "nvt") == 0)
    order = g_strdup (", host, port, severity, description");
  else
    order = g_strdup (", host, port, severity, nvt, description");

  delta_get = *get;
  delta_get.filt_id = NULL;
  delta_get.filter = g_strdup_printf ("rows=-1 first=1 %s", term);
  ignore_max_rows_per_page = 1;

#if 0
  iterator_t results2;

  res = init_result_get_iterator (results, &delta_get, report, NULL, order);
  if (res)
    return -1;

  res = init_result_get_iterator (&results2, &delta_get, delta, NULL, order);
  if (res)
    return -1;

  g_debug ("   delta: %s: REPORT 1:", __FUNCTION__);
  while (next (results))
    g_debug ("   delta: %s: %s   %s   %s   %s   %.30s",
            __FUNCTION__,
            result_iterator_nvt_name (results),
            result_iterator_host (results),
            result_iterator_type (results),
            result_iterator_port (results),
            result_iterator_descr (results));
  cleanup_iterator (results);
  g_debug ("   delta: %s: REPORT 1 END", __FUNCTION__);

  g_debug ("   delta: %s: REPORT 2:", __FUNCTION__);
  while (next (&results2))
    g_debug ("   delta: %s: %s   %s   %s   %s   %.30s",
            __FUNCTION__,
            result_iterator_nvt_name (&results2),
            result_iterator_host (&results2),
            result_iterator_type (&results2),
            result_iterator_port (&results2),
            result_iterator_descr (&results2));
  cleanup_iterator (&results2);
  g_debug ("   delta: %s: REPORT 2 END", __FUNCTION__);
#endif

  res = init_result_get_iterator (results, &delta_get, report, NULL, order);
  if (res)
    {
      ignore_max_rows_per_page = 0;
      g_free (order);
      return -1;
    }

  res = init_result_get_iterator (delta_results, &delta_get, delta, NULL, order);
  if (res)
    {
      ignore_max_rows_per_page = 0;
      g_free (order);
      return -1;
    }

  g_free (delta_get.filter);
  ignore_max_rows_per_page = 0;
  g_free (order);

  return 0;
}

/**
 * @brief Print delta results for print_report_xml.
 *
 * @param[in]  out            File stream to write to.
 * @param[in]  results        Report result iterator.
 * @param[in]  delta_results  Delta report result iterator.
 * @param[in]  delta_states   String describing delta states to include in count
 *                            (for example, "sngc" Same, New, Gone and Changed).
 *                            All levels if NULL.
 * @param[in]  first_result   First result.
 * @param[in]  max_results    Max results.
 * @param[in]  task           The task.
 * @param[in]  notes          Whether to include notes.
 * @param[in]  notes_details  Whether to include note details.
 * @param[in]  overrides          Whether to include overrides.
 * @param[in]  overrides_details  Whether to include override details.
 * @param[in]  sort_order         Sort order.
 * @param[in]  sort_field         Sort field.
 * @param[in]  result_hosts_only  Whether to only include hosts with results.
 * @param[in]  orig_filtered_result_count  Result count.
 * @param[in]  filtered_result_count       Result count.
 * @param[in]  orig_f_debugs  Result count.
 * @param[in]  f_debugs       Result count.
 * @param[in]  orig_f_holes   Result count.
 * @param[in]  f_holes        Result count.
 * @param[in]  orig_f_infos   Result count.
 * @param[in]  f_infos        Result count.
 * @param[in]  orig_f_logs    Result count.
 * @param[in]  f_logs         Result count.
 * @param[in]  orig_f_warnings  Result count.
 * @param[in]  f_warnings       Result count.
 * @param[in]  orig_f_false_positives  Result count.
 * @param[in]  f_false_positives       Result count.
 * @param[in]  result_hosts   Result hosts.
 *
 * @return 0 on success, -1 error.
 */
static int
print_report_delta_xml (FILE *out, iterator_t *results,
                        iterator_t *delta_results, const char *delta_states,
                        int first_result, int max_results, task_t task,
                        int notes, int notes_details, int overrides,
                        int overrides_details, int sort_order,
                        const char *sort_field, int result_hosts_only,
                        int *orig_filtered_result_count,
                        int *filtered_result_count,
                        int *orig_f_debugs, int *f_debugs,
                        int *orig_f_holes, int *f_holes,
                        int *orig_f_infos, int *f_infos,
                        int *orig_f_logs, int *f_logs,
                        int *orig_f_warnings, int *f_warnings,
                        int *orig_f_false_positives, int *f_false_positives,
                        array_t *result_hosts)
{
  gboolean done, delta_done;
  int changed, gone, new, same;
  /* A tree of host, tree pairs, where the inner tree is a sorted tree
   * of port, threat pairs. */
  GTree *ports;
  gchar *msg;

  *orig_f_debugs = *f_debugs;
  *orig_f_holes = *f_holes;
  *orig_f_infos = *f_infos;
  *orig_f_logs = *f_logs;
  *orig_f_warnings = *f_warnings;
  *orig_f_false_positives = *f_false_positives;
  *orig_filtered_result_count = *filtered_result_count;

  changed = (strchr (delta_states, 'c') != NULL);
  gone = (strchr (delta_states, 'g') != NULL);
  new = (strchr (delta_states, 'n') != NULL);
  same = (strchr (delta_states, 's') != NULL);

  ports = g_tree_new_full ((GCompareDataFunc) strcmp, NULL, g_free,
                           (GDestroyNotify) free_host_ports);

  /* Compare the results in the two iterators, which are sorted. */

  g_debug ("   delta: %s: start", __FUNCTION__);
  g_debug ("   delta: %s: sort_field: %s", __FUNCTION__, sort_field);
  g_debug ("   delta: %s: sort_order: %i", __FUNCTION__, sort_order);
  g_debug ("   delta: %s: max_results: %i", __FUNCTION__, max_results);
  done = !next (results);
  delta_done = !next (delta_results);
  while (1)
    {
      GString *buffer;
      compare_results_t state;
      int used, would_use;

      if (max_results == 0)
        break;

      if (done)
        {
          if (delta_done)
            break;
          if (new)
            /* Extra results in 'delta_results'. */
            do
              {
                const char *level;

                g_debug ("   delta: %s: extra from report 2: %s",
                        __FUNCTION__,
                        result_iterator_nvt_oid (results));

                if (first_result)
                  {
                    g_debug ("   delta: skip");
                    first_result--;
                    continue;
                  }

                /* Increase the result count. */
                level = result_iterator_level (delta_results);
                (*orig_filtered_result_count)++;
                (*filtered_result_count)++;
                if (strcmp (level, "High") == 0)
                  {
                    (*orig_f_holes)++;
                    (*f_holes)++;
                  }
                else if (strcmp (level, "Medium") == 0)
                  {
                    (*orig_f_warnings)++;
                    (*f_warnings)++;
                  }
                else if (strcmp (level, "Low") == 0)
                  {
                    (*orig_f_infos)++;
                    (*f_infos)++;
                  }
                else if (strcmp (level, "Log") == 0)
                  {
                    (*orig_f_logs)++;
                    (*f_logs)++;
                  }
                else if (strcmp (level, "False Positive") == 0)
                  {
                    (*orig_f_false_positives)++;
                    (*f_false_positives)++;
                  }

                g_debug ("   delta: %s: extra from report 2: %s",
                        __FUNCTION__,
                        result_iterator_nvt_oid (delta_results));
                buffer = g_string_new ("");
                buffer_results_xml (buffer,
                                    delta_results,
                                    task,
                                    notes,
                                    notes_details,
                                    overrides,
                                    overrides_details,
                                    0,
                                    0,
                                    0,
                                    "new",
                                    NULL,
                                    0);
                if (fprintf (out, "%s", buffer->str) < 0)
                  return -1;
                g_string_free (buffer, TRUE);
                if (result_hosts_only)
                  array_add_new_string (result_hosts,
                                        result_iterator_host (delta_results));
                add_port (ports, delta_results);
                max_results--;
                if (max_results == 0)
                  break;
              }
            while (next (delta_results));
          delta_done = TRUE;
          break;
        }

      if (delta_done)
        {
          /* Extra results in 'results'. */
          if (gone)
            do
              {
                g_debug ("   delta: %s: extra from report 1: %s",
                        __FUNCTION__,
                        result_iterator_nvt_oid (results));
                if (first_result)
                  {
                    g_debug ("   delta: skip");
                    first_result--;
                    continue;
                  }
                buffer = g_string_new ("");
                buffer_results_xml (buffer,
                                    results,
                                    task,
                                    notes,
                                    notes_details,
                                    overrides,
                                    overrides_details,
                                    0,
                                    0,
                                    0,
                                    "gone",
                                    NULL,
                                    0);
                if (fprintf (out, "%s", buffer->str) < 0)
                  return -1;
                g_string_free (buffer, TRUE);
                if (result_hosts_only)
                  array_add_new_string (result_hosts,
                                        result_iterator_host (results));
                add_port (ports, results);
                max_results--;
                if (max_results == 0)
                  break;
              }
            while (next (results));
          else
            do
              {
                const char *level;

                /* Decrease the result count. */
                level = result_iterator_level (results);
                (*orig_filtered_result_count)--;
                (*filtered_result_count)--;
                if (strcmp (level, "High") == 0)
                  {
                    (*orig_f_holes)--;
                    (*f_holes)--;
                  }
                else if (strcmp (level, "Medium") == 0)
                  {
                    (*orig_f_warnings)--;
                    (*f_warnings)--;
                  }
                else if (strcmp (level, "Low") == 0)
                  {
                    (*orig_f_infos)--;
                    (*f_infos)--;
                  }
                else if (strcmp (level, "Log") == 0)
                  {
                    (*orig_f_logs)--;
                    (*f_logs)--;
                  }
                else if (strcmp (level, "False Positive") == 0)
                  {
                    (*orig_f_false_positives)--;
                    (*f_false_positives)--;
                  }
              }
            while (next (results));
          done = TRUE;
          break;
        }

      /* Compare the two results. */

      buffer = g_string_new ("");
      state = compare_and_buffer_results (buffer,
                                          results,
                                          delta_results,
                                          task,
                                          notes,
                                          notes_details,
                                          overrides,
                                          overrides_details,
                                          sort_order,
                                          sort_field,
                                          changed,
                                          gone,
                                          new,
                                          same,
                                          &max_results,
                                          &first_result,
                                          &used,
                                          &would_use);
      if (state == COMPARE_RESULTS_ERROR)
        {
          g_warning ("%s: compare_and_buffer_results failed\n",
                     __FUNCTION__);
          return -1;
        }
      if (fprintf (out, "%s", buffer->str) < 0)
        return -1;
      g_string_free (buffer, TRUE);

      if ((used == 0)
          && ((state == COMPARE_RESULTS_GONE)
              || (state == COMPARE_RESULTS_SAME)
              || (state == COMPARE_RESULTS_CHANGED)))
        {
          const char *level;

          /* Decrease the result count. */
          level = result_iterator_level (results);
          (*filtered_result_count)--;
          if (strcmp (level, "High") == 0)
            {
              (*f_holes)--;
            }
          else if (strcmp (level, "Medium") == 0)
            {
              (*f_warnings)--;
            }
          else if (strcmp (level, "Low") == 0)
            {
              (*f_infos)--;
            }
          else if (strcmp (level, "Log") == 0)
            {
              (*f_logs)--;
            }
          else if (strcmp (level, "False Positive") == 0)
            {
              (*f_false_positives)--;
            }
        }

      if ((would_use == 0)
          && ((state == COMPARE_RESULTS_GONE)
              || (state == COMPARE_RESULTS_SAME)
              || (state == COMPARE_RESULTS_CHANGED)))
        {
          const char *level;

          /* Decrease the result count. */
          level = result_iterator_level (results);
          (*orig_filtered_result_count)--;
          if (strcmp (level, "High") == 0)
            {
              (*orig_f_holes)--;
            }
          else if (strcmp (level, "Medium") == 0)
            {
              (*orig_f_warnings)--;
            }
          else if (strcmp (level, "Low") == 0)
            {
              (*orig_f_infos)--;
            }
          else if (strcmp (level, "Log") == 0)
            {
              (*orig_f_logs)--;
            }
          else if (strcmp (level, "False Positive") == 0)
            {
              (*orig_f_false_positives)--;
            }
        }

      /* Move on to the next. */

      if (state == COMPARE_RESULTS_GONE)
        {
          /* "Used" just the 'results' result. */
          if (used)
            {
              if (result_hosts_only)
                array_add_new_string (result_hosts,
                                      result_iterator_host (results));
              add_port (ports, results);
            }
          done = !next (results);
        }
      else if ((state == COMPARE_RESULTS_SAME)
               || (state == COMPARE_RESULTS_CHANGED))
        {
          /* "Used" both results. */
          if (used)
            {
              if (result_hosts_only)
                array_add_new_string (result_hosts,
                                      result_iterator_host (results));
              add_port (ports, results);
            }
          done = !next (results);
          delta_done = !next (delta_results);
        }
      else if (state == COMPARE_RESULTS_NEW)
        {
          if (would_use)
            {
              const char *level;

              /* Would have "used" just the 'delta_results' result, on
               * an earlier page. */

              /* Increase the result count. */
              level = result_iterator_level (delta_results);
              (*orig_filtered_result_count)++;
              if (strcmp (level, "High") == 0)
                {
                  (*orig_f_holes)++;
                }
              else if (strcmp (level, "Medium") == 0)
                {
                  (*orig_f_warnings)++;
                }
              else if (strcmp (level, "Low") == 0)
                {
                  (*orig_f_infos)++;
                }
              else if (strcmp (level, "Log") == 0)
                {
                  (*orig_f_logs)++;
                }
              else if (strcmp (level, "False Positive") == 0)
                {
                  (*orig_f_false_positives)++;
                }
            }

          if (used)
            {
              const char *level;

              /* "Used" just the 'delta_results' result. */

              /* Increase the result count. */
              level = result_iterator_level (delta_results);
              (*filtered_result_count)++;
              if (strcmp (level, "High") == 0)
                {
                  (*f_holes)++;
                }
              else if (strcmp (level, "Medium") == 0)
                {
                  (*f_warnings)++;
                }
              else if (strcmp (level, "Low") == 0)
                {
                  (*f_infos)++;
                }
              else if (strcmp (level, "Log") == 0)
                {
                  (*f_logs)++;
                }
              else if (strcmp (level, "False Positive") == 0)
                {
                  (*f_false_positives)++;
                }

              if (result_hosts_only)
                array_add_new_string (result_hosts,
                                      result_iterator_host
                                       (delta_results));

              add_port (ports, delta_results);
            }
          delta_done = !next (delta_results);
        }
      else
        assert (0);
    }

  /* Compare remaining results, for the filtered report counts. */

  g_debug ("   delta: %s: counting rest", __FUNCTION__);
  while (1)
    {
      compare_results_t state;
      int used, would_use;

      if (done)
        {
          if (delta_done)
            break;
          if (new)
            /* Extra results in 'delta_results'. */
            do
              {
                const char *level;

                g_debug ("   delta: %s: extra from report 2: %s",
                        __FUNCTION__,
                        result_iterator_nvt_oid (delta_results));

                /* Increase the result count. */
                level = result_iterator_level (delta_results);
                (*orig_filtered_result_count)++;
                if (strcmp (level, "High") == 0)
                  {
                    (*orig_f_holes)++;
                  }
                else if (strcmp (level, "Medium") == 0)
                  {
                    (*orig_f_warnings)++;
                  }
                else if (strcmp (level, "Low") == 0)
                  {
                    (*orig_f_infos)++;
                  }
                else if (strcmp (level, "Log") == 0)
                  {
                    (*orig_f_logs)++;
                  }
                else if (strcmp (level, "False Positive") == 0)
                  {
                    (*orig_f_false_positives)++;
                  }
              }
            while (next (delta_results));
          break;
        }

      if (delta_done)
        {
          /* Extra results in 'results'. */
          if (gone)
            do
              {
                g_debug ("   delta: %s: extra from report 1: %s",
                        __FUNCTION__,
                        result_iterator_nvt_oid (results));

                /* It's in the count already. */
              }
            while (next (results));
          else
            do
              {
                const char *level;

                /* Decrease the result count. */
                level = result_iterator_level (results);
                (*orig_filtered_result_count)--;
                if (strcmp (level, "High") == 0)
                  {
                    (*orig_f_holes)--;
                  }
                else if (strcmp (level, "Medium") == 0)
                  {
                    (*orig_f_warnings)--;
                  }
                else if (strcmp (level, "Low") == 0)
                  {
                    (*orig_f_infos)--;
                  }
                else if (strcmp (level, "Log") == 0)
                  {
                    (*orig_f_logs)--;
                  }
                else if (strcmp (level, "False Positive") == 0)
                  {
                    (*orig_f_false_positives)--;
                  }
              }
            while (next (results));
          break;
        }

      /* Compare the two results. */

      state = compare_and_buffer_results (NULL,
                                          results,
                                          delta_results,
                                          task,
                                          notes,
                                          notes_details,
                                          overrides,
                                          overrides_details,
                                          sort_order,
                                          sort_field,
                                          changed,
                                          gone,
                                          new,
                                          same,
                                          &max_results,
                                          &first_result,
                                          &used,
                                          &would_use);
      if (state == COMPARE_RESULTS_ERROR)
        {
          g_warning ("%s: compare_and_buffer_results failed\n",
                     __FUNCTION__);
          return -1;
        }

      if (state == COMPARE_RESULTS_NEW)
        {
          if (used)
            {
              const char *level;

              /* "Used" just the 'delta_results' result. */

              /* Increase the result count. */
              level = result_iterator_level (delta_results);
              (*orig_filtered_result_count)++;
              if (strcmp (level, "High") == 0)
                {
                  (*orig_f_holes)++;
                }
              else if (strcmp (level, "Medium") == 0)
                {
                  (*orig_f_warnings)++;
                }
              else if (strcmp (level, "Low") == 0)
                {
                  (*orig_f_infos)++;
                }
              else if (strcmp (level, "Log") == 0)
                {
                  (*orig_f_logs)++;
                }
              else if (strcmp (level, "False Positive") == 0)
                {
                  (*orig_f_false_positives)++;
                }
            }
        }
      else if (used)
        {
          /* It's in the count already. */
        }
      else
        {
          const char *level;

          /* Decrease the result count. */
          level = result_iterator_level (results);
          (*orig_filtered_result_count)--;
          if (strcmp (level, "High") == 0)
            {
              (*orig_f_holes)--;
            }
          else if (strcmp (level, "Medium") == 0)
            {
              (*orig_f_warnings)--;
            }
          else if (strcmp (level, "Low") == 0)
            {
              (*orig_f_infos)--;
            }
          else if (strcmp (level, "Log") == 0)
            {
              (*orig_f_logs)--;
            }
          else if (strcmp (level, "False Positive") == 0)
            {
              (*orig_f_false_positives)--;
            }
        }

      /* Move on to the next. */

      if (state == COMPARE_RESULTS_GONE)
        {
          /* "Used" just the 'results' result. */
          done = !next (results);
        }
      else if ((state == COMPARE_RESULTS_SAME)
               || (state == COMPARE_RESULTS_CHANGED))
        {
          /* "Used" both results. */
          done = !next (results);
          delta_done = !next (delta_results);
        }
      else if (state == COMPARE_RESULTS_NEW)
        {
          /* "Used" just the 'delta_results' result. */
          delta_done = !next (delta_results);
        }
      else
        assert (0);
    }
  msg = g_markup_printf_escaped ("</results>");
  if (fprintf (out, "%s", msg) < 0)
    {
      g_free (msg);
      fclose (out);
      return -1;
    }
  g_free (msg);

  /* Write ports to file. */

  msg = g_markup_printf_escaped ("<ports"
                                 " start=\"%i\""
                                 " max=\"%i\">",
                                 /* Add 1 for 1 indexing. */
                                 first_result + 1,
                                 max_results);
  if (fprintf (out, "%s", msg) < 0)
    {
      g_free (msg);
      fclose (out);
      return -1;
    }
  g_free (msg);
  if (sort_field == NULL || strcmp (sort_field, "port"))
    {
      if (sort_order)
        g_tree_foreach (ports, print_host_ports_by_severity_asc, out);
      else
        g_tree_foreach (ports, print_host_ports_by_severity_desc, out);
    }
  else if (sort_order)
    g_tree_foreach (ports, print_host_ports, out);
  else
    g_tree_foreach (ports, print_host_ports_desc, out);
  g_tree_destroy (ports);
  msg = g_markup_printf_escaped ("</ports>");
  if (fprintf (out, "%s", msg) < 0)
    {
      g_free (msg);
      fclose (out);
      return -1;
    }
  g_free (msg);

  return 0;
}

/**
 * @brief Print the main XML content for a report to a file.
 *
 * @param[in]  report      The report.
 * @param[in]  delta       Report to compare with the report.
 * @param[in]  task        Task associated with report.
 * @param[in]  xml_start   File name.
 * @param[in]  get         GET command data.
 * @param[in]  type               Type of report, NULL, "scan" or "assets".
 * @param[in]  notes_details      If notes, Whether to include details.
 * @param[in]  overrides_details  If overrides, Whether to include details.
 * @param[in]  host               Host or NULL, when type "assets".
 * @param[in]  pos                Position of report from end, when type
 *                                "assets".
 * @param[in]  host_search_phrase  Phrase that results must include.  All results
 *                                 if NULL or "".  For hosts.
 * @param[in]  host_levels         String describing threat levels (message types)
 *                                 to include in count (for example, "hmlgd" for
 *                                 High, Medium, Low, loG and Debug).  All levels if
 *                                 NULL.
 * @param[in]  host_first_result   The host result to start from.  The results
 *                                 are 0 indexed.
 * @param[in]  host_max_results    The host maximum number of results returned.
 * @param[in]  ignore_pagination   Whether to ignore pagination data.
 * @param[out] filter_term_return  Filter term used in report.
 * @param[out] zone_return         Actual zone used in report.
 * @param[out] host_summary    Summary of results per host.
 *
 * @return 0 on success, -1 error, 2 failed to find filter (before any printing).
 */
static int
print_report_xml_start (report_t report, report_t delta, task_t task,
                        gchar* xml_start, const get_data_t *get,
                        const char *type,
                        int notes_details, int overrides_details,
                        const char *host, int pos,
                        const char *host_search_phrase, const char *host_levels,
                        int host_first_result, int host_max_results,
                        int ignore_pagination, gchar **filter_term_return,
                        gchar **zone_return, gchar **host_summary)
{
  int result_hosts_only;
  int notes, overrides;

  int first_result, max_results, sort_order, autofp;

  FILE *out;
  gchar *clean, *term, *sort_field, *levels, *search_phrase;
  gchar *min_qod;
  gchar *delta_states, *timestamp;
  int min_qod_int;
  char *uuid, *tsk_uuid = NULL, *start_time, *end_time;
  int total_result_count, filtered_result_count, run_status;
  array_t *result_hosts;
  iterator_t results, delta_results;
  int debugs, holes, infos, logs, warnings, false_positives;
  int f_debugs, f_holes, f_infos, f_logs, f_warnings, f_false_positives;
  int orig_f_debugs, orig_f_holes, orig_f_infos, orig_f_logs;
  int orig_f_warnings, orig_f_false_positives, orig_filtered_result_count;
  int search_phrase_exact, apply_overrides;
  double severity, f_severity;
  gchar *tz, *zone;
  char *old_tz_override;
  GString *filters_buffer, *filters_extra_buffer, *host_summary_buffer;
  gchar *term_value;
  GHashTable *f_host_ports;
  GHashTable *f_host_holes, *f_host_warnings, *f_host_infos;
  GHashTable *f_host_logs, *f_host_false_positives;

  /* Init some vars to prevent warnings from older compilers. */
  total_result_count = filtered_result_count = 0;
  orig_filtered_result_count = 0;
  orig_f_false_positives = orig_f_warnings = orig_f_logs = orig_f_infos = 0;
  orig_f_holes = orig_f_debugs = 0;
  f_host_ports = NULL;
  f_host_holes = NULL;
  f_host_warnings = NULL;
  f_host_infos = NULL;
  f_host_logs = NULL;
  f_host_false_positives = NULL;

  /** @todo Leaks on error in PRINT and PRINT_XML.  The process normally exits
   *        then anyway. */

  /* run_status is set by report_scan_run_status when either of "delta" and
   * "report" are true.  run_status is only used by run_status_name, only when
   * either of "delta" and "report" are true, and only after a
   * report_scan_run_status call.  Still GCC 4.4.5 (Debian 4.4.5-8) gives a
   * "may be used uninitialized" warning, so init it here to quiet the
   * warning. */
  run_status = TASK_STATUS_INTERNAL_ERROR;

  if (type
      && strcmp (type, "scan")
      && strcmp (type, "assets")
      && strcmp (type, "prognostic"))
    return -1;

  if ((type == NULL) || (strcmp (type, "scan") == 0))
    {
      type = NULL;
      if (report == 0)
        {
          assert (0);
          return -1;
        }
    }

  out = fopen (xml_start, "w");

  if (out == NULL)
    {
      g_warning ("%s: fopen failed: %s\n",
                 __FUNCTION__,
                 strerror (errno));
      return -1;
    }

  assert (get);

  if ((get->filt_id && strlen (get->filt_id) && strcmp (get->filt_id, "0"))
      || (get->filter && strlen (get->filter)))
    {
      term = NULL;
      if (get->filt_id && strlen (get->filt_id) && strcmp (get->filt_id, "0"))
        {
          term = filter_term (get->filt_id);
          if (term == NULL)
            {
              fclose (out);
              return 2;
            }
        }

      /* Set the filter parameters from the filter term. */
      manage_report_filter_controls (term ? term : get->filter,
                                     &first_result, &max_results, &sort_field,
                                     &sort_order, &result_hosts_only,
                                     &min_qod, &levels, &delta_states,
                                     &search_phrase, &search_phrase_exact,
                                     &autofp, &notes, &overrides,
                                     &apply_overrides, &zone);
    }
  else
    {
      term = NULL;
      first_result = 0;
      max_results = -1;
      sort_field = NULL;
      sort_order = 1;
      result_hosts_only = 1;
      min_qod = NULL;
      levels = NULL;
      delta_states = NULL;
      search_phrase = NULL;
      search_phrase_exact = 0;
      autofp = 0;
      notes = 0;
      overrides = 0;
      apply_overrides = 0;
      zone = NULL;
    }

  max_results = manage_max_rows (max_results);

  if (delta
      && sort_field
      && strcmp (sort_field, "name")
      && strcmp (sort_field, "vulnerability")
      && strcmp (sort_field, "host")
      && strcmp (sort_field, "port")
      && strcmp (sort_field, "location")
      && strcmp (sort_field, "severity")
      && strcmp (sort_field, "type")
      && strcmp (sort_field, "original_type"))
    {
      if ((strcmp (sort_field, "task") == 0)
          || (strcmp (sort_field, "task_id") == 0)
          || (strcmp (sort_field, "report_id") == 0))
        {
          /* These don't affect delta report, so sort by vulnerability. */
          g_free (sort_field);
          sort_field = g_strdup ("vulnerability");
        }
      else
        {
          /* The remaining filterable fields for the result iterator, all of
           * which may be used as a sort field.  These could be added to
           * result_cmp.  For now sort by vulnerability. */
#if 0
          "uuid", "comment", "created", "modified", "_owner"
          "nvt",
          "auto_type",
          "report", "cvss_base", "nvt_version",
          "original_severity", "date",
          "solution_type", "qod", "qod_type", "cve", "hostname"
#endif
          g_free (sort_field);
          sort_field = g_strdup ("vulnerability");
        }
    }

  levels = levels ? levels : g_strdup ("hmlgd");

  if (task && task_uuid (task, &tsk_uuid))
    {
      fclose (out);
      g_free (term);
      g_free (sort_field);
      g_free (levels);
      g_free (search_phrase);
      g_free (min_qod);
      g_free (delta_states);
      return -1;
    }

  if (zone && strlen (zone))
    {
      gchar *quoted_zone;
      /* Store current TZ. */
      tz = getenv ("TZ") ? g_strdup (getenv ("TZ")) : NULL;

      if (setenv ("TZ", zone, 1) == -1)
        {
          g_warning ("%s: Failed to switch to timezone", __FUNCTION__);
          if (tz != NULL)
            setenv ("TZ", tz, 1);
          g_free (tz);
          g_free (zone);
          return -1;
        }

      old_tz_override = sql_string ("SELECT tz_override"
                                    " FROM current_credentials;");

      quoted_zone = sql_insert (zone);
      sql ("UPDATE current_credentials SET tz_override = %s", quoted_zone);
      g_free (quoted_zone);

      tzset ();
    }
  else
    {
      /* Keep compiler quiet. */
      tz = NULL;
      old_tz_override = NULL;
    }

  if (delta && report)
    {
      uuid = report_uuid (report);
      PRINT (out, "<report type=\"delta\" id=\"%s\">", uuid);
      free (uuid);
    }
  else if (report)
    {
      uuid = report_uuid (report);
      PRINT (out, "<report id=\"%s\">", uuid);
      free (uuid);
    }
  else if (type && (strcmp (type, "assets") == 0))
    PRINT (out, "<report scap_loaded=\"%i\" type=\"%s\">",
           manage_scap_loaded (),
           type);
  else
    PRINT (out, "<report type=\"%s\">", type);

  PRINT (out, "<omp><version>%s</version></omp>", OMP_VERSION);

  if (delta)
    {
      delta_states = delta_states ? delta_states : g_strdup ("cgns");
      report_scan_run_status (delta, &run_status);

      uuid = report_uuid (delta);
      PRINT (out,
             "<delta>"
             "<report id=\"%s\">"
             "<scan_run_status>%s</scan_run_status>",
             uuid,
             run_status_name (run_status
                               ? run_status
                               : TASK_STATUS_INTERNAL_ERROR));

      if (report_timestamp (uuid, &timestamp))
        {
          free (uuid);
          g_free (sort_field);
          g_free (levels);
          g_free (search_phrase);
          g_free (min_qod);
          g_free (delta_states);
          tz_revert (zone, tz, old_tz_override);
          return -1;
        }
      PRINT (out,
             "<timestamp>%s</timestamp>",
             timestamp);
      g_free (timestamp);

      start_time = scan_start_time (delta);
      PRINT (out,
             "<scan_start>%s</scan_start>",
             start_time);
      free (start_time);

      end_time = scan_end_time (delta);
      PRINT (out,
             "<scan_end>%s</scan_end>",
             end_time);
      free (end_time);

      PRINT (out,
             "</report>"
             "</delta>");
    }

  if (report)
    {
      if (delta == 0)
        {
          int total_debugs, total_holes, total_infos, total_logs;
          int total_warnings, total_false_positives;
          get_data_t *all_results_get;

          all_results_get = report_results_get_data (1, -1, 0, 0, 0);
          report_counts_id (report, &total_debugs, &total_holes, &total_infos,
                            &total_logs, &total_warnings,
                            &total_false_positives, NULL, all_results_get,
                            NULL);
          total_result_count = total_debugs + total_holes + total_infos
                               + total_logs + total_warnings
                               + total_false_positives;
          get_data_reset (all_results_get);
          free (all_results_get);
        }
      report_counts_id (report, &debugs, &holes, &infos, &logs, &warnings,
                        &false_positives, NULL, get, NULL);
      filtered_result_count = debugs + holes + infos + logs + warnings
                              + false_positives;
      report_scan_run_status (report, &run_status);
    }

  clean = manage_clean_filter (term
                                ? term
                                : (get->filter ? get->filter : ""));

  term_value = filter_term_value (clean, "min_qod");
  if (term_value == NULL)
    {
      gchar *new_filter;
      new_filter = g_strdup_printf ("min_qod=%i %s",
                                    MIN_QOD_DEFAULT,
                                    clean);
      g_free (clean);
      clean = new_filter;
    }
  g_free (term_value);

  term_value = filter_term_value (clean, "apply_overrides");
  if (term_value == NULL)
    {
      gchar *new_filter;
      new_filter = g_strdup_printf ("apply_overrides=%i %s",
                                    APPLY_OVERRIDES_DEFAULT,
                                    clean);
      g_free (clean);
      clean = new_filter;
    }
  g_free (term_value);

  g_free (term);
  term = clean;

  if (filter_term_return)
    *filter_term_return = g_strdup (term);

  PRINT
   (out,
    "<sort><field>%s<order>%s</order></field></sort>",
    sort_field ? sort_field : "type",
    sort_order ? "ascending" : "descending");

  filters_extra_buffer = g_string_new ("");

  if (strchr (levels, 'h'))
    g_string_append (filters_extra_buffer, "<filter>High</filter>");
  if (strchr (levels, 'm'))
    g_string_append (filters_extra_buffer, "<filter>Medium</filter>");
  if (strchr (levels, 'l'))
    g_string_append (filters_extra_buffer, "<filter>Low</filter>");
  if (strchr (levels, 'g'))
    g_string_append (filters_extra_buffer, "<filter>Log</filter>");
  if (strchr (levels, 'd'))
    g_string_append (filters_extra_buffer, "<filter>Debug</filter>");
  if (strchr (levels, 'f'))
    g_string_append (filters_extra_buffer, "<filter>False Positive</filter>");

  if (delta)
    {
      gchar *escaped_host = host ? g_markup_escape_text (host, -1) : NULL;
      gchar *escaped_delta_states = g_markup_escape_text (delta_states, -1);
      g_string_append_printf (filters_extra_buffer,
                              "<host><ip>%s</ip></host>"
                              "<delta>"
                              "%s"
                              "<changed>%i</changed>"
                              "<gone>%i</gone>"
                              "<new>%i</new>"
                              "<same>%i</same>"
                              "</delta>",
                              escaped_host ? escaped_host : "",
                              escaped_delta_states,
                              strchr (delta_states, 'c') != NULL,
                              strchr (delta_states, 'g') != NULL,
                              strchr (delta_states, 'n') != NULL,
                              strchr (delta_states, 's') != NULL);
      g_free (escaped_host);
      g_free (escaped_delta_states);
    }
  else if (type && (strcmp (type, "prognostic") == 0))
    {
      gchar *escaped_host = host ? g_markup_escape_text (host, -1) : NULL;
      g_string_append_printf (filters_extra_buffer,
                              "<host><ip>%s</ip></host>",
                              escaped_host ? escaped_host : "");
      g_free (escaped_host);
    }

  filters_buffer = g_string_new ("");
  buffer_get_filter_xml (filters_buffer, "result", get, clean,
                         filters_extra_buffer->str);
  g_string_free (filters_extra_buffer, TRUE);

  PRINT_XML (out, filters_buffer->str);
  g_string_free (filters_buffer, TRUE);

  {
    const char *severity_setting;
    gchar *class_xml;

    severity_setting = setting_severity ();
    class_xml = severity_class_xml (severity_setting);
    if (class_xml)
      {
        PRINT_XML (out, class_xml);
        g_free (class_xml);
      }
  }

  if (report)
    {
      uuid = report_uuid (report);

      PRINT (out,
            "<user_tags>"
            "<count>%i</count>",
            resource_tag_count ("report", report, 1));

      if (get->details || get->id)
        {
          iterator_t tags;

          init_resource_tag_iterator (&tags, "report", report, 1, NULL, 1);

          while (next (&tags))
            {
              PRINT (out,
                     "<tag id=\"%s\">"
                     "<name>%s</name>"
                     "<value>%s</value>"
                     "<comment>%s</comment>"
                     "</tag>",
                     resource_tag_iterator_uuid (&tags),
                     resource_tag_iterator_name (&tags),
                     resource_tag_iterator_value (&tags),
                     resource_tag_iterator_comment (&tags));
            }

          cleanup_iterator (&tags);
        }

      PRINT (out, "</user_tags>");

      free (uuid);
    }

  if (report)
    {
      PRINT
       (out,
        "<scan_run_status>%s</scan_run_status>",
        run_status_name (run_status
                          ? run_status
                          : TASK_STATUS_INTERNAL_ERROR));

      PRINT (out,
             "<hosts><count>%i</count></hosts>",
             report_host_count (report));

      PRINT (out,
             "<closed_cves><count>%i</count></closed_cves>",
             report_closed_cve_count (report));

      PRINT (out,
             "<vulns><count>%i</count></vulns>",
             report_vuln_count (report));

      PRINT (out,
             "<os><count>%i</count></os>",
             report_os_count (report));

      PRINT (out,
             "<apps><count>%i</count></apps>",
             report_app_count (report));

      PRINT (out,
             "<ssl_certs><count>%i</count></ssl_certs>",
             report_ssl_cert_count (report));

    }

  if (task && tsk_uuid)
    {
      char *tsk_name, *task_target_uuid, *comment;
      target_t target;
      gchar *progress_xml;
      iterator_t tags;

      tsk_name = task_name (task);

      comment = task_comment (task);

      target = task_target (task);
      if (task_target_in_trash (task))
        task_target_uuid = trash_target_uuid (target);
      else
        task_target_uuid = target_uuid (target);

      if ((target == 0)
          && (task_run_status (task) == TASK_STATUS_RUNNING))
        progress_xml = g_strdup_printf
                        ("%i",
                         task_upload_progress (task));
      else
        {
          int progress;
          progress = report_progress (report, task, NULL);
          progress_xml = g_strdup_printf ("%i", progress);
        }

      PRINT (out,
             "<task id=\"%s\">"
             "<name>%s</name>"
             "<comment>%s</comment>"
             "<target id=\"%s\">"
             "<trash>%i</trash>"
             "</target>"
             "<progress>%s</progress>",
             tsk_uuid,
             tsk_name ? tsk_name : "",
             comment ? comment : "",
             task_target_uuid ? task_target_uuid : "",
             task_target_in_trash (task),
             progress_xml);
      g_free (progress_xml);
      free (comment);
      free (tsk_name);
      free (tsk_uuid);

      PRINT (out,
             "<user_tags>"
             "<count>%i</count>",
             resource_tag_count ("task", task, 1));

      init_resource_tag_iterator (&tags, "task", task, 1, NULL, 1);
      while (next (&tags))
        {
          PRINT (out,
                 "<tag id=\"%s\">"
                 "<name>%s</name>"
                 "<value>%s</value>"
                 "<comment>%s</comment>"
                 "</tag>",
                 resource_tag_iterator_uuid (&tags),
                 resource_tag_iterator_name (&tags),
                 resource_tag_iterator_value (&tags),
                 resource_tag_iterator_comment (&tags));
        }
      cleanup_iterator (&tags);

      PRINT (out,
             "</user_tags>"
             "</task>");

      {
        char *slave_uuid, *slave_name, *slave_host, *slave_port, *source_iface;

        /* Info about the situation at the time of scan. */

        PRINT (out,
               "<scan>"
               "<task>");

        slave_uuid = report_slave_uuid (report);
        slave_name = report_slave_name (report);
        slave_host = report_slave_host (report);
        slave_port = report_slave_port (report);

        if (slave_uuid)
          /* @id "" means no slave.  Missing SLAVE means we don't know. */
          PRINT (out,
                 "<slave id=\"%s\">"
                 "<name>%s</name>"
                 "<host>%s</host>"
                 "<port>%s</port>"
                 "</slave>",
                 slave_uuid,
                 slave_name ? slave_name : "",
                 slave_host ? slave_host : "",
                 slave_port ? slave_port : "");

        free (slave_uuid);
        free (slave_name);
        free (slave_host);
        free (slave_port);

        source_iface = report_source_iface (report);

        if (source_iface)
          /* VALUE "" means preference was not set.  Missing PREFERENCE means
           * we don't know. */
          PRINT (out,
                 "<preferences>"
                 "<preference>"
                 "<name>Network Source Interface</name>"
                 "<scanner_name>source_iface</scanner_name>"
                 "<value>%s</value>"
                 "</preference>"
                 "</preferences>",
                 source_iface);

        free (source_iface);

        PRINT (out,
               "</task>"
               "</scan>");
      }
    }

  if (type && (strcmp (type, "assets") == 0))
    {
      int ret;

      g_free (term);

      ret = print_report_assets_xml (out, host,
                                     ignore_pagination ? 1 : first_result,
                                     ignore_pagination ? -1 : max_results,
                                     levels, search_phrase, pos,
                                     get, apply_overrides, autofp);
      g_free (sort_field);
      g_free (levels);
      g_free (search_phrase);
      g_free (min_qod);
      g_free (delta_states);

      tz_revert (zone, tz, old_tz_override);

      if (ret)
        return ret;

      if (fclose (out))
        {
          g_warning ("%s: fclose failed: %s\n",
                     __FUNCTION__,
                     strerror (errno));
          return -1;
        }

      return 0;
    }

  if (type && (strcmp (type, "prognostic") == 0))
    {
      int ret;

      g_free (term);

      ret = print_report_prognostic_xml (out, host,
                                         ignore_pagination ? 1 : first_result,
                                         ignore_pagination ? -1 : max_results,
                                         levels, search_phrase, pos, get,
                                         apply_overrides, autofp,
                                         host_search_phrase, host_levels,
                                         host_first_result, host_max_results,
                                         result_hosts_only,
                                         sort_order, sort_field);

      g_free (sort_field);
      g_free (levels);
      g_free (search_phrase);
      g_free (min_qod);
      g_free (delta_states);

      tz_revert (zone, tz, old_tz_override);

      if (ret)
        return ret;

      if (fclose (out))
        {
          g_warning ("%s: fclose failed: %s\n",
                     __FUNCTION__,
                     strerror (errno));
          return -1;
        }

      return 0;
    }

  uuid = report_uuid (report);
  if (report_timestamp (uuid, &timestamp))
    {
      free (uuid);
      g_free (term);
      tz_revert (zone, tz, old_tz_override);
      return -1;
    }
  free (uuid);
  PRINT (out,
         "<timestamp>%s</timestamp>",
         timestamp);
  g_free (timestamp);

  start_time = scan_start_time (report);
  PRINT (out,
         "<scan_start>%s</scan_start>",
         start_time);
  free (start_time);

  {
    time_t start_time_epoch;
    const char *abbrev;
    gchar *report_zone;

    start_time_epoch = scan_start_time_epoch (report);
    abbrev = NULL;
    if (zone && strlen (zone))
      report_zone = g_strdup (zone);
    else
      report_zone = setting_timezone ();
    iso_time_tz (&start_time_epoch, report_zone, &abbrev);

    if (zone_return)
      *zone_return = g_strdup (report_zone ? report_zone : "");

    PRINT (out,
           "<timezone>%s</timezone>"
           "<timezone_abbrev>%s</timezone_abbrev>",
           report_zone
            ? report_zone
            : "Coordinated Universal Time",
           abbrev ? abbrev : "UTC");
    g_free (report_zone);
  }

  /* Port summary. */

  f_host_ports = g_hash_table_new_full (g_str_hash, g_str_equal,
                                        g_free, NULL);

  if (get->details && (delta == 0))
    {
      if (print_report_port_xml (report, out, get, first_result, max_results,
                                 sort_order, sort_field, f_host_ports))
        {
          g_free (term);
          tz_revert (zone, tz, old_tz_override);
          g_hash_table_destroy (f_host_ports);
          return -1;
        }
    }

  /* Prepare result counts. */

  report_counts_id_full (report, &debugs, &holes, &infos, &logs,
                         &warnings, &false_positives, &severity,
                         get, NULL,
                         &f_debugs, &f_holes, &f_infos, &f_logs, &f_warnings,
                         &f_false_positives, &f_severity);

  /* Results. */

  if (min_qod == NULL || sscanf (min_qod, "%d", &min_qod_int) != 1)
    min_qod_int = MIN_QOD_DEFAULT;

  if (delta && get->details)
    {
      if (init_delta_iterators (report, &results, delta, &delta_results, get,
                                term, sort_field))
        {
          g_free (term);
          g_hash_table_destroy (f_host_ports);
          return -1;
        }
      g_free (term);
    }
  else if (get->details)
    {
      int res;
      g_free (term);
      res = init_result_get_iterator (&results, get, report, NULL, NULL);
      if (res)
        {
          g_hash_table_destroy (f_host_ports);
          return -1;
        }
    }
  else
    g_free (term);

  if (get->details)
    PRINT (out,
             "<results"
             " start=\"%i\""
             " max=\"%i\">",
             /* Add 1 for 1 indexing. */
             ignore_pagination ? 1 : first_result + 1,
             ignore_pagination ? -1 : max_results);
  if (get->details && result_hosts_only)
    result_hosts = make_array ();
  else
    /* Quiet erroneous compiler warning. */
    result_hosts = NULL;

  f_host_holes = g_hash_table_new_full (g_str_hash, g_str_equal,
                                        g_free, NULL);
  f_host_warnings = g_hash_table_new_full (g_str_hash, g_str_equal,
                                           g_free, NULL);
  f_host_infos = g_hash_table_new_full (g_str_hash, g_str_equal,
                                      g_free, NULL);
  f_host_logs = g_hash_table_new_full (g_str_hash, g_str_equal,
                                      g_free, NULL);
  f_host_false_positives = g_hash_table_new_full (g_str_hash, g_str_equal,
                                                  g_free, NULL);

  if (delta && get->details)
    {
      if (print_report_delta_xml (out, &results, &delta_results, delta_states,
                                  first_result, max_results, task, notes,
                                  notes_details, overrides, overrides_details,
                                  sort_order, sort_field, result_hosts_only,
                                  &orig_filtered_result_count,
                                  &filtered_result_count,
                                  &orig_f_debugs, &f_debugs,
                                  &orig_f_holes, &f_holes,
                                  &orig_f_infos, &f_infos,
                                  &orig_f_logs, &f_logs,
                                  &orig_f_warnings, &f_warnings,
                                  &orig_f_false_positives, &f_false_positives,
                                  result_hosts))
        {
          fclose (out);
          g_free (sort_field);
          g_free (levels);
          g_free (search_phrase);
          g_free (min_qod);
          g_free (delta_states);
          cleanup_iterator (&results);
          cleanup_iterator (&delta_results);
          tz_revert (zone, tz, old_tz_override);
          g_hash_table_destroy (f_host_ports);
          g_hash_table_destroy (f_host_holes);
          g_hash_table_destroy (f_host_warnings);
          g_hash_table_destroy (f_host_infos);
          g_hash_table_destroy (f_host_logs);
          g_hash_table_destroy (f_host_false_positives);

          return -1;
        }
    }
  else if (get->details)
    {
      while (next (&results))
        {
          const char* level;
          GHashTable *f_host_result_counts;
          GString *buffer = g_string_new ("");
          buffer_results_xml (buffer,
                              &results,
                              task,
                              notes,
                              notes_details,
                              overrides,
                              overrides_details,
                              1,
                              1,
                              0,
                              NULL,
                              NULL,
                              0);
          PRINT_XML (out, buffer->str);
          g_string_free (buffer, TRUE);
          if (result_hosts_only)
            array_add_new_string (result_hosts,
                                  result_iterator_host (&results));

          level = result_iterator_level (&results);
          if (strcasecmp (level, "log") == 0)
            f_host_result_counts = f_host_logs;
          else if (strcasecmp (level, "high") == 0)
            f_host_result_counts = f_host_holes;
          else if (strcasecmp (level, "medium") == 0)
            f_host_result_counts = f_host_warnings;
          else if (strcasecmp (level, "low") == 0)
            f_host_result_counts = f_host_infos;
          else if (strcasecmp (level, "false positive") == 0)
            f_host_result_counts = f_host_false_positives;
          else
            f_host_result_counts = NULL;

          if (f_host_result_counts)
            {
              const char *host = result_iterator_host (&results);
              int result_count
                    = GPOINTER_TO_INT
                        (g_hash_table_lookup (f_host_result_counts, host));

              g_hash_table_replace (f_host_result_counts,
                                    g_strdup (host),
                                    GINT_TO_POINTER (result_count + 1));
            }

        }
      PRINT (out, "</results>");
    }
  if (get->details)
    cleanup_iterator (&results);
  if (delta && get->details)
    cleanup_iterator (&delta_results);

  /* Print result counts and severity. */

  if (delta)
    /** @todo The f_debugs, etc. vars are setup to give the page count. */
    PRINT (out,
           "<result_count>"
           "<filtered>%i</filtered>"
           "<debug><filtered>%i</filtered></debug>"
           "<hole><filtered>%i</filtered></hole>"
           "<info><filtered>%i</filtered></info>"
           "<log><filtered>%i</filtered></log>"
           "<warning><filtered>%i</filtered></warning>"
           "<false_positive>"
           "<filtered>%i</filtered>"
           "</false_positive>"
           "</result_count>",
           orig_filtered_result_count,
           (strchr (levels, 'd') ? orig_f_debugs : 0),
           (strchr (levels, 'h') ? orig_f_holes : 0),
           (strchr (levels, 'l') ? orig_f_infos : 0),
           (strchr (levels, 'g') ? orig_f_logs : 0),
           (strchr (levels, 'm') ? orig_f_warnings : 0),
           (strchr (levels, 'f') ? orig_f_false_positives : 0));
  else
    {
      PRINT (out,
             "<result_count>"
             "%i"
             "<full>%i</full>"
             "<filtered>%i</filtered>"
             "<debug><full>%i</full><filtered>%i</filtered></debug>"
             "<hole><full>%i</full><filtered>%i</filtered></hole>"
             "<info><full>%i</full><filtered>%i</filtered></info>"
             "<log><full>%i</full><filtered>%i</filtered></log>"
             "<warning><full>%i</full><filtered>%i</filtered></warning>"
             "<false_positive>"
             "<full>%i</full>"
             "<filtered>%i</filtered>"
             "</false_positive>"
             "</result_count>",
             total_result_count,
             total_result_count,
             filtered_result_count,
             debugs,
             (strchr (levels, 'd') ? f_debugs : 0),
             holes,
             (strchr (levels, 'h') ? f_holes : 0),
             infos,
             (strchr (levels, 'l') ? f_infos : 0),
             logs,
             (strchr (levels, 'g') ? f_logs : 0),
             warnings,
             (strchr (levels, 'm') ? f_warnings : 0),
             false_positives,
             (strchr (levels, 'f') ? f_false_positives : 0));

      PRINT (out,
             "<severity>"
             "<full>%1.1f</full>"
             "<filtered>%1.1f</filtered>"
             "</severity>",
             severity,
             f_severity);
    }

  if (host_summary)
    {
      host_summary_buffer = g_string_new ("");
      g_string_append_printf (host_summary_buffer,
                              "   %-15s   %-16s   End\n",
                              "Host", "Start");
    }
  else
    host_summary_buffer = NULL;

  if (get->details && result_hosts_only)
    {
      gchar *host;
      int index = 0;
      array_terminate (result_hosts);
      while ((host = g_ptr_array_index (result_hosts, index++)))
        {
          gboolean present;
          iterator_t hosts;
          init_report_host_iterator (&hosts, report, host, 0);
          present = next (&hosts);
          if (delta && (present == FALSE))
            {
              cleanup_iterator (&hosts);
              init_report_host_iterator (&hosts, delta, host, 0);
              present = next (&hosts);
            }
          if (present)
            {
              const char *current_host;
              int ports_count;
              int holes_count, warnings_count, infos_count;
              int logs_count, false_positives_count;

              current_host = host_iterator_host (&hosts);

              ports_count
                = GPOINTER_TO_INT
                    (g_hash_table_lookup (f_host_ports, current_host));
              holes_count
                = GPOINTER_TO_INT
                    (g_hash_table_lookup ( f_host_holes, current_host));
              warnings_count
                = GPOINTER_TO_INT
                    (g_hash_table_lookup ( f_host_warnings, current_host));
              infos_count
                = GPOINTER_TO_INT
                    (g_hash_table_lookup ( f_host_infos, current_host));
              logs_count
                = GPOINTER_TO_INT
                    (g_hash_table_lookup ( f_host_logs, current_host));
              false_positives_count
                = GPOINTER_TO_INT
                    (g_hash_table_lookup ( f_host_false_positives, current_host));

              host_summary_append (host_summary_buffer,
                                   host,
                                   host_iterator_start_time (&hosts),
                                   host_iterator_end_time (&hosts));
              PRINT (out,
                     "<host>"
                     "<ip>%s</ip>"
                     "<asset asset_id=\"%s\"/>"
                     "<start>%s</start>"
                     "<end>%s</end>"
                     "<port_count><page>%d</page></port_count>"
                     "<result_count>"
                     "<page>%d</page>"
                     "<hole><page>%d</page></hole>"
                     "<warning><page>%d</page></warning>"
                     "<info><page>%d</page></info>"
                     "<log><page>%d</page></log>"
                     "<false_positive><page>%d</page></false_positive>"
                     "</result_count>",
                     host,
                     host_iterator_asset_uuid (&hosts)
                       ? host_iterator_asset_uuid (&hosts)
                       : "",
                     host_iterator_start_time (&hosts),
                     host_iterator_end_time (&hosts)
                       ? host_iterator_end_time (&hosts)
                       : "",
                     ports_count,
                     (holes_count + warnings_count + infos_count
                      + logs_count + false_positives_count),
                     holes_count,
                     warnings_count,
                     infos_count,
                     logs_count,
                     false_positives_count);

              if (print_report_host_details_xml
                   (host_iterator_report_host (&hosts), out))
                {
                  tz_revert (zone, tz, old_tz_override);
                  if (host_summary_buffer)
                    g_string_free (host_summary_buffer, TRUE);
                  g_hash_table_destroy (f_host_ports);
                  g_hash_table_destroy (f_host_holes);
                  g_hash_table_destroy (f_host_warnings);
                  g_hash_table_destroy (f_host_infos);
                  g_hash_table_destroy (f_host_logs);
                  g_hash_table_destroy (f_host_false_positives);
                  return -1;
                }

              PRINT (out,
                     "</host>");

              PRINT (out,
                     "<host_start>"
                     "<host>%s</host>%s"
                     "</host_start>",
                     host,
                     host_iterator_start_time (&hosts));
              PRINT (out,
                     "<host_end>"
                     "<host>%s</host>%s"
                     "</host_end>",
                     host,
                     host_iterator_end_time (&hosts)
                       ? host_iterator_end_time (&hosts)
                       : "");
            }
          cleanup_iterator (&hosts);
        }
      array_free (result_hosts);
    }
  else if (get->details)
    {
      iterator_t hosts;
      init_report_host_iterator (&hosts, report, NULL, 0);
      while (next (&hosts))
        {
          const char *current_host;
          int ports_count;
          int holes_count, warnings_count, infos_count;
          int logs_count, false_positives_count;

          current_host = host_iterator_host (&hosts);

          ports_count
            = GPOINTER_TO_INT
                (g_hash_table_lookup (f_host_ports, current_host));
          holes_count
            = GPOINTER_TO_INT
                (g_hash_table_lookup (f_host_holes, current_host));
          warnings_count
            = GPOINTER_TO_INT
                (g_hash_table_lookup (f_host_warnings, current_host));
          infos_count
            = GPOINTER_TO_INT
                (g_hash_table_lookup (f_host_infos, current_host));
          logs_count
            = GPOINTER_TO_INT
                (g_hash_table_lookup (f_host_logs, current_host));
          false_positives_count
            = GPOINTER_TO_INT
                (g_hash_table_lookup (f_host_false_positives, current_host));

          host_summary_append (host_summary_buffer,
                               host_iterator_host (&hosts),
                               host_iterator_start_time (&hosts),
                               host_iterator_end_time (&hosts));
          PRINT (out,
                 "<host>"
                 "<ip>%s</ip>"
                 "<asset asset_id=\"%s\"/>"
                 "<start>%s</start>"
                 "<end>%s</end>"
                 "<port_count><page>%d</page></port_count>"
                 "<result_count>"
                 "<page>%d</page>"
                 "<hole><page>%d</page></hole>"
                 "<warning><page>%d</page></warning>"
                 "<info><page>%d</page></info>"
                 "<log><page>%d</page></log>"
                 "<false_positive><page>%d</page></false_positive>"
                 "</result_count>",
                 host_iterator_host (&hosts),
                 host_iterator_asset_uuid (&hosts)
                   ? host_iterator_asset_uuid (&hosts)
                   : "",
                 host_iterator_start_time (&hosts),
                 host_iterator_end_time (&hosts)
                   ? host_iterator_end_time (&hosts)
                   : "",
                 ports_count,
                 (holes_count + warnings_count + infos_count
                  + logs_count + false_positives_count),
                 holes_count,
                 warnings_count,
                 infos_count,
                 logs_count,
                 false_positives_count);

          if (print_report_host_details_xml
               (host_iterator_report_host (&hosts), out))
            {
              tz_revert (zone, tz, old_tz_override);
              if (host_summary_buffer)
                g_string_free (host_summary_buffer, TRUE);
              g_hash_table_destroy (f_host_ports);
              g_hash_table_destroy (f_host_holes);
              g_hash_table_destroy (f_host_warnings);
              g_hash_table_destroy (f_host_infos);
              g_hash_table_destroy (f_host_logs);
              g_hash_table_destroy (f_host_false_positives);
              return -1;
            }

          PRINT (out,
                 "</host>");

          PRINT (out,
                 "<host_start><host>%s</host>%s</host_start>",
                 host_iterator_host (&hosts),
                 host_iterator_start_time (&hosts));
        }
      cleanup_iterator (&hosts);

      init_report_host_iterator (&hosts, report, NULL, 0);
      while (next (&hosts))
        PRINT (out,
               "<host_end><host>%s</host>%s</host_end>",
               host_iterator_host (&hosts),
               host_iterator_end_time (&hosts)
                ? host_iterator_end_time (&hosts)
                : "");
      cleanup_iterator (&hosts);
    }

  g_hash_table_destroy (f_host_ports);
  g_hash_table_destroy (f_host_holes);
  g_hash_table_destroy (f_host_warnings);
  g_hash_table_destroy (f_host_infos);
  g_hash_table_destroy (f_host_logs);
  g_hash_table_destroy (f_host_false_positives);

  end_time = scan_end_time (report);
  PRINT (out,
           "<scan_end>%s</scan_end>",
           end_time);
  free (end_time);

  if (delta == 0 && print_report_errors_xml (report, out))
    {
      tz_revert (zone, tz, old_tz_override);
      if (host_summary_buffer)
        g_string_free (host_summary_buffer, TRUE);
      return -1;
    }

  g_free (sort_field);
  g_free (levels);
  g_free (search_phrase);
  g_free (min_qod);
  g_free (delta_states);

  if (host_summary && host_summary_buffer)
    *host_summary = g_string_free (host_summary_buffer, FALSE);

  if (fclose (out))
    {
      g_warning ("%s: fclose failed: %s\n",
                 __FUNCTION__,
                 strerror (errno));
      return -1;
    }

  return 0;
}

/**
 * @brief Completes a report by adding report format info.
 *
 * @param[in]   xml_start      Path of file containing start of report.
 * @param[in]   xml_full       Path to file to print full report to.
 * @param[in]   report_format  Format of report that will be created from XML.
 */
static int
print_report_xml_end (gchar *xml_start, gchar *xml_full,
                      report_format_t report_format)
{
  FILE *out;
  iterator_t params;

  if (openvas_file_copy (xml_start, xml_full) == FALSE)
    {
      g_warning ("%s: failed to copy xml_start file", __FUNCTION__);
      return -1;
    }

  out = fopen (xml_full, "a");
  if (out == NULL)
    {
      g_warning ("%s: fopen failed: %s\n",
                 __FUNCTION__,
                 strerror (errno));
      return -1;
    }

  PRINT (out, "<report_format>");
  init_report_format_param_iterator (&params, report_format, 0, 1, NULL);
  while (next (&params))
    PRINT (out,
           "<param><name>%s</name><value>%s</value></param>",
           report_format_param_iterator_name (&params),
           report_format_param_iterator_value (&params));
  cleanup_iterator (&params);

  PRINT (out, "</report_format>");

  PRINT (out, "</report>");

  if (fclose (out))
    {
      g_warning ("%s: fclose failed: %s\n",
                 __FUNCTION__,
                 strerror (errno));
      return -1;
    }

  return 0;
}

/**
 * @brief Generate a report.
 *
 * @param[in]  report             Report.
 * @param[in]  get                GET data for report.
 * @param[in]  report_format      Report format.
 * @param[in]  notes_details      If notes, Whether to include details.
 * @param[in]  overrides_details  If overrides, Whether to include details.
 * @param[in]  type               Type of report: NULL or "scan".
 * @param[out] output_length      NULL or location for length of return.
 * @param[out] extension          NULL or location for report format extension.
 *                                Only defined on success.
 * @param[out] content_type       NULL or location for report format content
 *                                type.  Only defined on success.
 * @param[out] filter_term_return  Filter term used in report.
 * @param[out] zone_return         Actual zone used in report.
 * @param[out] host_summary    Summary of results per host.
 *
 * @return Contents of report on success, NULL on error.
 */
gchar *
manage_report (report_t report, const get_data_t *get,
               const report_format_t report_format,
               int notes_details, int overrides_details, const char *type,
               gsize *output_length, gchar **extension, gchar **content_type,
               gchar **filter_term_return, gchar **zone_return,
               gchar **host_summary)
{
  task_t task;
  gchar *report_format_id, *xml_start, *xml_file, *output_file;
  char xml_dir[] = "/tmp/openvasmd_XXXXXX";
  int ret;
  GList *used_rfps;
  GError *get_error;
  gchar *output;
  gsize output_len;

  used_rfps = NULL;

  if (type && strcmp (type, "scan"))
    return NULL;

  /* Print the report as XML to a file. */

  if (((report_format_predefined (report_format) == 0)
       && (report_format_trust (report_format) != TRUST_YES))
      || (report_task (report, &task)))
    {
      return NULL;
    }

  i