#include "osl/effect_util/pin.h"
#include "osl/record/csaString.h"
#include "osl/record/csaRecord.h"
#include "osl/oslConfig.h"

#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>
#include <iostream>
#include <fstream>

using namespace osl;
using namespace osl::effect_util;

class PinTest : public CppUnit::TestFixture 
{
  CPPUNIT_TEST_SUITE(PinTest);
  CPPUNIT_TEST(testNoPin);
  CPPUNIT_TEST(testUp);
  CPPUNIT_TEST(testDown);
  CPPUNIT_TEST(testNeighbor);
  CPPUNIT_TEST(testDiagonal);
  CPPUNIT_TEST(testEquality);
  CPPUNIT_TEST(testCount);
  CPPUNIT_TEST_SUITE_END();
public:
  void testNoPin() 
  {
    {
      NumEffectState state((SimpleState(HIRATE)));
      const PieceMask pins = Pin::make(state, BLACK);
      CPPUNIT_ASSERT(pins.none());
    }
    {
      NumEffectState state(CsaString(
			  "P1-KY-KE-GI * +OU * -GI-KE * \n"
			  "P2 * -HI-KI *  *  * -KI-KA * \n"
			  "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
			  "P4 *  *  *  *  *  *  *  *  * \n"
			  "P5 *  *  *  * -KY *  *  *  * \n"
			  "P6 *  *  *  *  *  *  *  *  * \n"
			  "P7+FU+FU+FU+FU+FU+FU+FU+FU+FU\n"
			  "P8 * +KA+KI *  *  * +KI+HI * \n"
			  "P9+KY+KE+GI * -OU * +GI+KE+KY\n"
			  "+\n").getInitialState());
      const PieceMask pins = Pin::make(state, BLACK);
      CPPUNIT_ASSERT(pins.none());
    }
  }
  void testUp()
  {
    {
      NumEffectState state(CsaString(	
			  "P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
			  "P2 *  *  *  *  *  *  * -KA * \n"
			  "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
			  "P4 *  *  *  *  *  *  *  *  * \n"
			  "P5 *  *  *  * -HI *  *  *  * \n"
			  "P6 *  *  *  *  *  *  *  *  * \n"
			  "P7+FU+FU+FU+FU+FU+FU+FU+FU+FU\n"
			  "P8 * +KA *  *  *  *  * +HI * \n"
			  "P9+KY+KE+GI+KI+OU+KI+GI+KE+KY\n"
			  "+\n").getInitialState());
      const PieceMask pins = Pin::make(state, BLACK);
      CPPUNIT_ASSERT_EQUAL(1, pins.countBit2());
      const Piece p57fu = state.pieceOnBoard(Square(5,7));
      CPPUNIT_ASSERT(pins.test(p57fu.number()));
    }
    {
      NumEffectState state(CsaString(	
			  "P1-KY-KE-GI-KI-OU-KI-GI-KE * \n"
			  "P2 * -HI *  *  *  *  * -KA * \n"
			  "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
			  "P4 *  *  *  *  *  *  *  *  * \n"
			  "P5 *  *  *  * -KY *  *  *  * \n"
			  "P6 *  *  *  *  *  *  *  *  * \n"
			  "P7+FU+FU+FU+FU+FU+FU+FU+FU+FU\n"
			  "P8 * +KA *  *  *  *  * +HI * \n"
			  "P9+KY+KE+GI+KI+OU+KI+GI+KE+KY\n"
			  "+\n").getInitialState());
      const PieceMask pins = Pin::make(state, BLACK);
      CPPUNIT_ASSERT_EQUAL(1, pins.countBit2());
      const Piece p57fu = state.pieceOnBoard(Square(5,7));
      CPPUNIT_ASSERT(pins.test(p57fu.number()));
    }
  }
  void testDown()
  {
    {
      NumEffectState state(CsaString(
			  "P1-KY-KE-GI * +OU * -GI-KE-KY\n"
			  "P2 *  * -KI *  *  * -KI-KA * \n"
			  "P3-FU-FU-FU-FU+FU-FU-FU-FU-FU\n"
			  "P4 *  *  *  *  *  *  *  *  * \n"
			  "P5 *  *  *  * -HI *  *  *  * \n"
			  "P6 *  *  *  *  *  *  *  *  * \n"
			  "P7+FU+FU+FU+FU-FU+FU+FU+FU+FU\n"
			  "P8 * +KA+KI *  *  * +KI+HI * \n"
			  "P9+KY+KE+GI * -OU * +GI+KE+KY\n"
			  "+\n").getInitialState());
      const PieceMask pins = Pin::make(state, BLACK);
      CPPUNIT_ASSERT_EQUAL(1, pins.countBit2());
      const Piece p53fu = state.pieceOnBoard(Square(5,3));
      CPPUNIT_ASSERT(pins.test(p53fu.number()));
    }
  }
  void testNeighbor()
  {
    // 駒が隣にある場合のテスト
    {
      NumEffectState state(CsaString(	
			  "P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
			  "P2 *  *  *  *  *  *  * -KA * \n"
			  "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
			  "P4 *  *  *  *  *  *  *  *  * \n"
			  "P5 *  *  *  * -HI+FU+OU *  * \n"
			  "P6 *  *  *  *  *  *  *  *  * \n"
			  "P7+FU+FU+FU+FU+FU * +FU+FU+FU\n"
			  "P8 * +KA *  *  *  *  * +HI * \n"
			  "P9+KY+KE+GI+KI * +KI+GI+KE+KY\n"
			  "+\n").getInitialState());
      const PieceMask pins = Pin::make(state, BLACK);
      CPPUNIT_ASSERT_EQUAL(1, pins.countBit2());
      const Piece p45fu = state.pieceOnBoard(Square(4,5));
      CPPUNIT_ASSERT(pins.test(p45fu.number()));
    }
    {
      NumEffectState state(CsaString(	
			  "P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
			  "P2 *  *  *  *  *  *  * -KA * \n"
			  "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
			  "P4 *  *  *  *  *  *  *  *  * \n"
			  "P5 *  *  * -HI * +FU+OU *  * \n"
			  "P6 *  *  *  *  *  *  *  *  * \n"
			  "P7+FU+FU+FU+FU+FU * +FU+FU+FU\n"
			  "P8 * +KA *  *  *  *  * +HI * \n"
			  "P9+KY+KE+GI+KI * +KI+GI+KE+KY\n"
			  "+\n").getInitialState());
      const PieceMask pins = Pin::make(state, BLACK);
      CPPUNIT_ASSERT_EQUAL(1, pins.countBit2());
      const Piece p45fu = state.pieceOnBoard(Square(4,5));
      CPPUNIT_ASSERT(pins.test(p45fu.number()));
    }
    {
      NumEffectState state(CsaString(	
			  "P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
			  "P2 *  *  *  *  *  *  * -KA * \n"
			  "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
			  "P4 *  *  *  *  *  *  *  *  * \n"
			  "P5 *  *  *  * -HI+FU * +OU * \n"
			  "P6 *  *  *  *  *  *  *  *  * \n"
			  "P7+FU+FU+FU+FU+FU * +FU+FU+FU\n"
			  "P8 * +KA *  *  *  *  * +HI * \n"
			  "P9+KY+KE+GI+KI * +KI+GI+KE+KY\n"
			  "+\n").getInitialState());
      const PieceMask pins = Pin::make(state, BLACK);
      CPPUNIT_ASSERT_EQUAL(1, pins.countBit2());
      const Piece p45fu = state.pieceOnBoard(Square(4,5));
      CPPUNIT_ASSERT(pins.test(p45fu.number()));
    }
  }
  void testDiagonal()
  {
    {
      NumEffectState state(CsaString(
			  "P1-KY-KE-GI * -OU * -GI-KE-KY\n"
			  "P2 * -HI-KI *  *  * -KI-KA * \n"
			  "P3-FU-FU-FU-FU-FU-FU * -FU-FU\n"
			  "P4 *  *  *  *  *  * -FU *  * \n"
			  "P5 *  *  *  *  *  *  *  *  * \n"
			  "P6 *  *  *  *  *  *  *  *  * \n"
			  "P7+FU+FU+FU+FU+FU+FU+FU+FU+FU\n"
			  "P8 * +OU+KI *  *  * +KI+HI * \n"
			  "P9+KY+KE+GI * +KA * +GI+KE+KY\n"
			  "+\n").getInitialState());
      const PieceMask pins = Pin::make(state, BLACK);
      CPPUNIT_ASSERT_EQUAL(1, pins.countBit2());
      const Piece pin = state.pieceOnBoard(Square(7,7));
      CPPUNIT_ASSERT(pins.test(pin.number()));
    }
  }
  void testEquality(const std::string& filename)
  {
    Record rec=CsaFile(filename).getRecord();
    NumEffectState state(rec.getInitialState());
    const vector<osl::Move> moves=rec.getMoves();

    size_t i=0;
    while (true)
    {
      const PieceMask black_pins = Pin::makeNaive(state, BLACK);
      const PieceMask black_pins2 = Pin::makeByPiece(state, BLACK);
      if (black_pins != black_pins2)
	std::cerr << state;
      CPPUNIT_ASSERT_EQUAL(black_pins, black_pins2);
      const PieceMask black_pins3 = Pin::makeStep(state, state.kingSquare<BLACK>(), BLACK);
      if (black_pins != black_pins3)
	std::cerr << state;
      CPPUNIT_ASSERT_EQUAL(black_pins, black_pins3);

      const PieceMask white_pins = Pin::makeNaive(state, WHITE);
      const PieceMask white_pins2 = Pin::makeByPiece(state, WHITE);
      if (white_pins != white_pins2)
	std::cerr << state;
      CPPUNIT_ASSERT_EQUAL(white_pins, white_pins2);

      const PieceMask white_pins3 = Pin::makeStep(state, state.kingSquare<WHITE>(), WHITE);
      if (white_pins != white_pins3)
	std::cerr << state;
      CPPUNIT_ASSERT_EQUAL(white_pins, white_pins3);

      if (i >= moves.size())
	break;
      const Move move = moves[i++];
      state.makeMove(move);
    } 
  }
  void testEquality()
  {
    std::ifstream ifs(OslConfig::testCsaFile("FILES"));
    CPPUNIT_ASSERT(ifs);
    int i=0;
    int count=100;
    if (OslConfig::inUnitTestShort())
      count=10;
    std::string file_name;
    while ((ifs >> file_name) && (++i<count)) {
      if (file_name == "") 
	break;
      if (OslConfig::verbose())
	std::cerr << file_name << " ";
      testEquality(OslConfig::testCsaFile(file_name));
    }
  }
  void testCount();
};

CPPUNIT_TEST_SUITE_REGISTRATION(PinTest);

void PinTest::testCount()
{
  {
    NumEffectState state(CsaString(	
			   "P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
			   "P2 *  *  *  *  *  *  * -KA * \n"
			   "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
			   "P4 *  *  *  *  *  *  *  *  * \n"
			   "P5 *  *  *  * -HI *  *  *  * \n"
			   "P6 *  *  *  *  *  *  *  *  * \n"
			   "P7+FU+FU+FU+FU+FU+FU+FU+FU+FU\n"
			   "P8 * +KA *  *  *  *  * +HI * \n"
			   "P9+KY+KE+GI+KI+OU+KI+GI+KE+KY\n"
			   "+\n").getInitialState());
    CPPUNIT_ASSERT_EQUAL(1, Pin::count(state, BLACK));
  }
  {
    NumEffectState state(CsaString(	
			   "P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
			   "P2 *  *  *  *  *  *  * -KA * \n"
			   "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
			   "P4 *  *  *  *  *  *  *  *  * \n"
			   "P5 *  *  *  * -HI *  *  *  * \n"
			   "P6 *  *  *  *  *  *  *  *  * \n"
			   "P7+FU+FU+FU+FU+FU+FU+FU+FU+FU\n"
			   "P8 * +KA *  *  *  *  * +GI * \n"
			   "P9+KY+KE+GI+KI+OU+KI-HI+KE+KY\n"
			   "+\n").getInitialState());
    CPPUNIT_ASSERT_EQUAL(2, Pin::count(state, BLACK));
  }
  {
    // target は玉でなくても良い
    NumEffectState state(CsaString(	
			   "P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
			   "P2 *  *  *  *  *  *  * -KA * \n"
			   "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
			   "P4 *  *  *  *  *  *  *  *  * \n"
			   "P5 *  *  *  * -HI *  *  *  * \n"
			   "P6 *  *  *  *  *  *  *  *  * \n"
			   "P7+FU+FU+FU+FU+FU+FU+FU+FU+FU\n"
			   "P8 * +KA+OU *  *  *  * +GI * \n"
			   "P9+KY+KE+GI+KI * +KI-HI+KE+KY\n"
			   "+\n").getInitialState());
    CPPUNIT_ASSERT_EQUAL(2, Pin::count(state, Square(5,9), BLACK));
  }
}

// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
