#!/usr/bin/python
# -*- coding: UTF-8 -*-

############################################################################
# Programmers: Jiva DeVoe <jiva@devoesquared.com>
# Filename: iCal.py
# http://www.devoesquared.com/Software/iCal_Module
#
# Contributors / Alterations:
# stefan natchev: april 15, 2005 - added support for dates with ranges
# stefan natchev: april 16, 2005 - added more recurrance support
# Aaron A. Seigo aseigo@kde.org: January 2006 - patches for event lookup for multi-day events. (see occursOn)
# Danil Dotsenko dd@accentsolution.com: June 15, 2006 - reworked file rutine in ICalReader to work with generic paths and added fileFilter and check for correct header.
# Danil Dotsenko dd@accentsolution.com: -----||------ - reworked other functions for efficiency.
# Paul van Erk: July 3, 2006 - fixed a bug where all-day events would be counted as 2 days (and 2-day events as 3, etc)
# Danil Dotsenko dd@accentsolution.com: Sept 04, 2006 - changed the iCalReader class init rutine to init on the basis of raw data list, not filenames. This allows reading files from remote sources.
# Warning! with this version (20060904) I am braking backward compatibility in __init__() arguments of ICalReader
#
# Wolfgang Arlt code@familiearlt.de: *** As the link above is not valid, and this file wasn't working as it should. I changed a lot of code to have it running my way. ***
#                                    *** Documentation and distribution will be found at http://www.familiearlt.de                                                            ***
#
#                                    - changed parseEvent class to adapt the ics file of Evolution, Google.calendar, ...
#                                    - ical does now calculate Z time (UTC) to local time
#                                    - changed function Interval for DAYLY, WEEKLY: startsAfterToday, startsAfter
#                                    - startsAfter does look for Date & Time. startsToday shows only open events. All others only for date 
#                                    - changed function startsToday, startsTomorrow, startsOn to give False instead of None
#                                    - add new function: isNowActive, nextStartDate, nextEndDate, endsAfter
#                                    - add new function: readUntil. If not used, the default value = Today+365. It's to take events only into account until a spezific date.
#				     - implementation of RRULE: "FREQ(DAILY, WEEKLY, MONTHLY, YEARLY); INTERVAL; UNTIL; COUNT; SETPOS; BYDAY; BYMONTHDAY; BYMONTH"   !! No other RRULE item supported !!
#				       exeptionDate not supported until now !!
#				     - All Functions startsAfter.., EndsAfter... are only working with RRULE: INTERVAL, COUNT, UNTIL. No other RRULE implemented until now !!
#				     - It should now cover all available event recurrence rules used on evolution, thunderbird lightning & google calendar
#	20110108 - fix repeating event in december, add function for leap years.
#	20110113 - fix time calculation when UTC is one day ahead/below local time
#       20110120 - fix function 'startsAfter', 'endsAfter': Looks now for date and time (before it was only looking for date)
#	20110203 - fix time calculation when UTC is one day ahead/below local time. Now it works !
#
#
# Complete rebuilded version 20110203 by Wolfgang Arlt
#
#
# (from old version 20060904)
#
# "2-Clause" BSD license
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
#
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
# OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
# IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
# INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
# NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
# DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
# THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
# THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
#
############################################################################

import os
import os.path
import re
import datetime
import time

class ICalReader:
	def __init__(self, dataLines = None):
		'''
		iCal.ICalReader([dataList])

		Initializes the iCal reader and parses the list type object for events.
		The list type object with iCal contents is optional.
		You can call readFiles(file name list) and readEvents(iCal data lines) separately.
		'''
		self.events = []
		today = datetime.datetime.today()
		today = datetime.datetime(today.year, today.month, today.day, today.hour)
		utc=datetime.datetime.utcnow()
		utc=datetime.datetime(utc.year, utc.month, utc.day, utc.hour)
		self.TZdiff = today - utc
		self.untildate = today.replace(year=today.year+10)
		if dataLines:
			self.readEvents(dataLines)

	def readUntil(self, untildate):
		self.untildate = untildate

	def readFiles(self, fileList):
		'''
		readFiles(fileList)

		This function accepts ARRAY with list of local file names.
		We put contents of the files into one big list and feed to readEvents()
		We only work with files, no folders or wildcards.
		'''
		dataLines = []
		for item in range(fileList.__len__()):
			if os.path.isfile(fileList[item]): # not sure why I am doing it, your code should...
				tempFile = open(fileList[item], 'r')
				dataLines.extend(tempFile.readlines())
				tempFile.close()
		if dataLines:
			self.readEvents(dataLines)

	def readURL(self, url):
		'''
		Put rutine to fetch and convert a resource into lines and feed it to readEvents()
		'''
		dataLines = None
		try:
			import urllib2
			tempFile = urllib2.urlopen(url)
			dataLines = tempFile.readlines()
			tempFile.close()
		except:
			tempFile = open(url,'r')
			dataLines = tempFile.readlines()
			tempFile.close()
		if dataLines:
			self.readEvents(dataLines)

	def readEvents(self, lines, emptyEvents=False):
		if emptyEvents:
			self.events = []
			# this is here for MY convenience. Don't rely on this.
			# Instead just to iCalObject.events = {} in your code.
			# The latter allows emptying events with functions like readURL and readFiles
		mask = {}
		mask['BEGIN'] = re.compile("^BEGIN:VEVENT")
		mask['END'] = re.compile("^END:VEVENT")
		inEvent= False
		for line in lines: # this will only work if there is no such thing as nested VEVENTs. It assumes there is one END:VEVENT fro every openning BEGIN
			if mask['BEGIN'].match(line):
				eventLines = []
				inEvent = True
			elif mask['END'].match(line) and inEvent:
				# note: BEGIN: and END: are not included in the list.
				self.events.append(self.parseEvent(eventLines))
				inEvent = False
			elif inEvent:
				eventLines.append(line)

	def parseEvent(self, lines):
		event = ICalEvent(self.untildate)
		startDate = None
		startDateUTC = None
		rule = None
		endDate = None
		categories = None
		#it has to have something for a summary(?) -s
		#DTStart1 & DTEnd1 implemented for Google by Wolfgang Arlt
		#changed DTstartTZ 
		event.summary = ''
		mask={}
		mask['Summary']=re.compile("^SUMMARY:(.*)")
		mask['DTStart']=re.compile("^DTSTART;.*:(.*).*")
		mask['DTEnd']=re.compile("^DTEND;.*:(.*).*")
		mask['DTStartTZ']=re.compile("^DTSTART:(.*).*Z")
		mask['DTEndTZ']=re.compile("^DTEND:(.*).*Z")
		mask['DTStart1']=re.compile("^DTSTART:(.*).*")
		mask['DTEnd1']=re.compile("^DTEND:(.*).*")
		mask['ExDate']=re.compile("^EXDATE.*:(.*)")
		mask['Categories']=re.compile("^CATEGORIES:(.*)")
		mask['RRule']=re.compile("^RRULE.*:(.*)")
		timed = '1'

		### ipmlementation of reading splited content lines as of RFC5545 by Wolfgang Arlt
		count = range(1, len(lines))
		for i in count:
			data = lines[i]
			if data.startswith(' '):
				lines[i-1]= lines[i-1].strip('\r\n') + lines[i].lstrip()
				del lines[i]
				count.insert(i,i)
				count.pop()
				
		for line in lines:
			if mask['Summary'].match(line):
				event.summary = mask['Summary'].match(line).group(1)
				event.summary = event.summary.strip('\r\n')
			#these are the dtstart/dtend with no time range
			elif mask['DTStart'].match(line):
				startDate = self.parseDate(mask['DTStart'].match(line).group(1))
				timed = '0'
			elif mask['DTEnd'].match(line):
				endDate = self.parseDate(mask['DTEnd'].match(line).group(1))
				timed = '0'
			#these are the ones that are 'ranged'
			elif mask['DTStartTZ'].match(line):
				startDateUTC = self.parseDate(mask['DTStartTZ'].match(line).group(1))
				startDate = startDateUTC + self.TZdiff
			elif mask['DTEndTZ'].match(line):
				endDateUTC = self.parseDate(mask['DTEndTZ'].match(line).group(1))
				endDate = endDateUTC + self.TZdiff
			elif mask['ExDate'].match(line):
				event.addExceptionDate(self.parseDate(mask['ExDate'].match(line).group(1)))
			elif mask['DTStart1'].match(line):
				startDate = self.parseDate(mask['DTStart1'].match(line).group(1))
			elif mask['DTEnd1'].match(line):
				endDate = self.parseDate(mask['DTEnd1'].match(line).group(1))
			elif mask['RRule'].match(line):
				rule = mask['RRule'].match(line).group(1)
				rule = rule.strip() + ';'	#Leerzeichen sowie \r\n am Anfang sowie am Ende entfernen
				rule = rule.replace(' ','')	#Leerzeichen entfernen
			elif mask['Categories'].match(line):
				categories = mask['Categories'].match(line).group(1)
				categories = categories.strip('\r\n')
		event.categories = categories
		event.startDateTime = startDate
		event.startDate = datetime.date(startDate.year, startDate.month, startDate.day)
		event.endDateTime = endDate
		event.endDate = datetime.date(endDate.year, endDate.month, endDate.day)
		event.timed = timed
		event.rule = rule
		event.addRecurrenceRule(rule)
		return event

	def parseDate(self, dateStr):
		year = int(dateStr[0:4])
		if year < 1900:
			year = 1900
		month = int(dateStr[4:4+2])
		day = int(dateStr[6:6+2])
		try:
			hour = int(dateStr[9:9+2])
			minute = int(dateStr[11:11+2])
		except:
			hour = 0
			minute = 0
		return datetime.datetime(year, month, day, hour, minute)

	def selectEvents(self, selectFunction):
		self.events.sort()
		events = filter(selectFunction, self.events)
		return events

	def todaysEvents(self, event):
		return event.startsToday()

	def tomorrowsEvents(self, event):
		return event.startsTomorrow()

	def afterTodaysEvents(self, event):
		return event.startsAfterToday()

	def eventsFor(self, date):
		self.events.sort()
		ret = []
		for event in self.events:
			#if event.startsOn(date):
			if event.occursOn(date):
				ret.append(event)
		return ret


class ICalEvent:
	def __init__(self, untildate):
		self.untildate = untildate
		self.exceptionDates = []
		self.dateSet = False

	def __str__(self):
		return self.summary

	def __eq__(self, otherEvent):
		return self.startDateTime == otherEvent.startDate

	def addExceptionDate(self, date):
		date = datetime.date(date.year, date.month, date.day)
		self.exceptionDates.append(date)

	def addRecurrenceRule(self, rule):
		self.dateSet = DateSet(self.startDateTime, self.endDateTime, rule)
		
	def startsToday(self):
		return (self.startsOn(datetime.date.today()))

	def startsTomorrow(self):
		tomorrow = datetime.date.today() + datetime.timedelta(days=1)
		return self.startsOn(tomorrow)

	def startsOn(self, date):
		typ = 'startsOn'
		if type (date) == datetime.datetime:
			date = datetime.date(date.year, date.month, date.day)
		if  self.exceptionDates.count(date) < 1:
			return (self.dateSet and self.dateSet.includes(date, typ, self.untildate))
		else:
			return False

	def occursOn(self, date):
		typ = 'occursOn'
		if type (date) == datetime.datetime:
			date = datetime.date(date.year, date.month, date.day)
		if  self.exceptionDates.count(date) < 1:
			return (self.dateSet and self.dateSet.includes(date, typ, self.untildate))
		else:
			return False

	def isNowActive(self):
		now = datetime.datetime.today()
		return (self.startDateTime <= now) and (self.endDateTime > now)

	def nextStartDate(self):
		now = datetime.datetime.today()
		nextStartDate = self.startsAfter(now)
		if  type(nextStartDate) == bool:
			nextStartDate = False
		return nextStartDate

	def startsAfterToday(self):
		aftertoday = datetime.date.today() + datetime.timedelta(days=1) # aftertoday = tomorrow at 0:00
		aftertoday = datetime.datetime(aftertoday.year, aftertoday.month, aftertoday.day)
		return self.startsAfter(aftertoday)

	def startsAfterTomorrow(self):
		aftertomorrow = datetime.date.today() + datetime.timedelta(days=2) # aftertomorrow = the day after tomorrow at 0:00
		aftertomorrow = datetime.datetime(aftertomorrow.year, aftertomorrow.month, aftertomorrow.day)
		return self.startsAfter(aftertomorrow)

	def startsAfter(self, date):
		typ = 'startsAfter'
		if  self.exceptionDates.count(date) < 1:
			return (self.dateSet and self.dateSet.includes(date, typ, self.untildate))
		else:
			return False

	def nextEndDate(self):
		now = datetime.datetime.today()
		nextEndDate = self.endsAfter(now)
		if  type(nextEndDate) == bool:
			nextEndDate = False
		return nextEndDate

	def endsAfter(self, date):
		typ = 'endsAfter'
		if  self.exceptionDates.count(date) < 1:
			return (self.dateSet and self.dateSet.includes(date, typ, self.untildate))
		else:
			return False

	def startTime(self):
		return self.startDateTime


class DateSet:
	def __init__(self, startDate, endDate, rule):
		self.startDateTime = startDate
		self.startDate = datetime.date(self.startDateTime.year, self.startDateTime.month, self.startDateTime.day)
		self.endDateTime = endDate
		self.endDate = datetime.date(self.endDateTime.year, self.endDateTime.month, self.endDateTime.day)
		self.frequency = None
		self.count = None
		self.interval = 1
		self.untilDate = None
		self.byMonth = None
		self.byMonthday = None
		self.byDate = None
		self.byDay = None
		self.bySetpos = None
		if rule != None:
			self.parseRecurrenceRule(rule)
			

	def parseRecurrenceRule(self, rule):
		if re.compile("FREQ=(.*?);").match(rule):
			self.frequency = re.compile("FREQ=(.*?);").match(rule).group(1)

		if re.compile("UNTIL=(.*?);").search(rule):
			self.untilDate = self.parse(re.compile("UNTIL=(.*?);").search(rule).group(1))

		if re.compile("COUNT=(\d*)").search(rule):
			self.count = int(re.compile("COUNT=(\d*)").search(rule).group(1))

		if re.compile("INTERVAL=(\d*)").search(rule):
			self.interval = int(re.compile("INTERVAL=(\d*)").search(rule).group(1))

		if re.compile("BYSECOND=(.*?);").search(rule):
			self.bySecond = re.compile("BYSECOND=(.*?);").search(rule).group(1)

		if re.compile("BYMINUTE=(.*?);").search(rule):
			self.byMinute = re.compile("BYMinute=(.*?);").search(rule).group(1)

		if re.compile("BYHOURE=(.*?);").search(rule):
			self.byHoure = re.compile("BYHOURE=(.*?);").search(rule).group(1)

		if re.compile("BYDAY=(.*?);").search(rule):
			if re.compile("BYDAY=(-?\d)").search(rule):
				self.bySetpos = re.compile("BYDAY=(.-?\d*)").search(rule).group(1)
				byDay = re.compile("BYDAY=-?\d*(\D*);").search(rule).group(1)
			elif re.compile("BYDAY=(\d)").search(rule):
				self.bySetpos = re.compile("BYDAY=(\d*)").search(rule).group(1)
				byDay = re.compile("BYDAY=\d*(\D*);").search(rule).group(1)
			else:
				byDay = re.compile("BYDAY=(.*?);").search(rule).group(1)
			self.byDay = re.split(',', byDay)

		if re.compile("BYSETPOS=(.*?);").search(rule):
			self.bySetpos = re.compile("BYSETPOS=(.*?);").search(rule).group(1)
		
		if re.compile("BYMONTHDAY=(.*?);").search(rule):
			byMonthday = re.compile("BYMONTHDAY=(.*?);").search(rule).group(1)
			self.byMonthday = re.split(',', byMonthday)

		if re.compile("BYYEARDAY=(.*?);").search(rule):
			self.byYearday = re.compile("BYYEARDAY=(.*?);").search(rule).group(1)

		if re.compile("BYWEEKNO=(.*?);").search(rule):
			self.byWeekno = re.compile("BYWEEKNO=(.*?);").search(rule).group(1)

		if re.compile("BYMONTH=(.*?);").search(rule):
			byMonth = re.compile("BYMONTH=(.*?);").search(rule).group(1)
			self.byMonth = re.split(',', byMonth)

		if re.compile("BYWEEKST=(.*?);").search(rule):
			self.byWeekst = re.compile("BYWEEKST=(.*?);").search(rule).group(1)


	def parse(self, dateStr):
		year = int(dateStr[0:4])
		if year < 1970:
			year = 1970
	
		month = int(dateStr[4:4+2])
		day = int(dateStr[6:6+2])
		try:
			hour = int(dateStr[9:9+2])
			minute = int(dateStr[11:11+2])
		except:
			hour = 0
			minute = 0
		return datetime.datetime(year, month, day, hour, minute)


	def includes(self, date, typ, untildate):

		if type(untildate) == datetime.datetime:
			self.untildatetime = untildate
			self.untildate = datetime.date(untildate.year, untildate.month, untildate.day)
		elif type(untildate) == datetime.date:
			self.untildatetime = datetime.datetime(untildate.year, untildate.month, untildate.day)
			self.untildate = untildate

		if typ == 'startsOn':
			if (self.startDate == date) and (self.startDate < self.untildate):
				return self.startDateTime
			
		elif typ == 'occursOn':
			if (self.startDate <= date) and (self.endDate >= date) and (self.startDate < self.untildate):
				if self.endDate == date and self.endDateTime.hour == 0 and self.endDateTime.second == 0:
					return False
				else:
					return True						
                            
		elif typ == 'startsAfter':
			if type (date) == datetime.date:
				date = datetime.datetime(date.year, date.month, date.day)
			if (self.startDateTime > date) and (self.startDate < self.untildate):
				return self.startDateTime

		elif typ == 'endsAfter':
			if type (date) == datetime.date:
				date = datetime.datetime(date.year, date.month, date.day)
			if (self.endDateTime > date) and (self.endDate < self.untildate):
				return self.endDateTime


		if self.frequency:
			if self.frequency == 'DAILY' or self.frequency == 'WEEKLY':
				if self.frequency == 'DAILY':
					if self.interval:
						increment = self.interval
					else:	
						increment = 1

				elif self.frequency == 'WEEKLY':
					if self.interval:
						increment = self.interval * 7
					else:
						increment = 7

				sd = self.startDate	
				ed = self.endDate
				sdt = self.startDateTime
				edt = self.endDateTime
				counter = 0
				counterUntil = 0

				if typ.endswith('On'):
					if self.byDay:
						MO = 0; TU = 1; WE = 2; TH = 3; FR = 4; SA = 5; SU = 6
						byday = None
						sd1 = sd
						ed1 = ed
						sdt1 = sdt
						while (byday != sd1.weekday()) or (sd1 <= date and sd1 < self.untildate):	
							for i in self.byDay:
								byday = (vars()[i.upper()])

								if self.untilDate:
									if self.untilDate < sdt1:
										return False
										break	

								if self.count and (byday == sd1.weekday()):
									counterUntil += 1
									if counterUntil > self.count:
										return False
										break

								if (sd1 == date) and (sd1 < self.untildate) and (byday == sd1.weekday()):
									if typ == 'occursOn':
										return True
										break
									return sdt1
									break


							sd1 = sd1 + datetime.timedelta(1)
							ed1 = ed1 + datetime.timedelta(1)
							sdt1 = sdt1 + datetime.timedelta(1)

							if self.frequency == 'WEEKLY' and sd1.weekday() == 0:
								sd1 = sd1 + datetime.timedelta(increment - 7)
								ed1 = ed1 + datetime.timedelta(increment - 7)
								sdt1 = sdt1 + datetime.timedelta(increment - 7)
									

					else:
						while(sd <= date):
							if self.untilDate:
								if self.untilDate < sdt:
									return False
									break	

							if self.count:
								counter += 1
								if counter > self.count:
									return False
									break

							if (sd == date) and (sd < self.untildate):
								if typ == 'occursOn':
									return True
									break
								return sdt
								break

							if typ == 'occursOn':
								if (sd <= date) and (ed >= date) and (sd < self.untildate):
									if self.endDateTime.hour == 0 and self.endDateTime.second == 0:
										return False
									else:
										return True
										break						

							sd = sd + datetime.timedelta(increment)
							ed = ed + datetime.timedelta(increment)
							sdt = sdt + datetime.timedelta(increment)
						return False
					return False

				if typ == 'startsAfter':
					sdt = self.startDateTime
					while (sdt < self.untildatetime):
						if self.untilDate:
							if self.untilDate < sdt:
								return False
								break	

						if self.count:
							counter += 1
							if counter > self.count:
								return False
								break

						if (sdt.time() > date.time() and
							sdt.day >= date.day and
							sdt.month >= date.month and
							sdt.year >= date.year and
							sdt < self.untildatetime):
							return sdt
							break

						elif (sdt.day > date.day and
							sdt.month >= date.month and
							sdt.year >= date.year and
							sdt < self.untildatetime):
							return sdt
							break

						elif (sdt.month > date.month and
							sdt.year >= date.year and
							sdt < self.untildatetime):
							return sdt
							break

						elif (sdt.year > date.year and
							sdt < self.untildatetime):
							return sdt
							break

						elif (sdt >= self.untildatetime):
							return False
							break

						sdt = sdt + datetime.timedelta(increment)

					return False

				if typ == 'endsAfter':
					edt = self.endDateTime
					while (edt < self.untildatetime):
						if self.untilDate:
							if self.untilDate < edt:
								return False
								break	

						if self.count:
							counter += 1
							if counter > self.count:
								return False
								break

						if (edt.time() > date.time() and
							edt.day >= date.day and
							edt.month >= date.month and
							edt.year >= date.year and
							edt < self.untildatetime):
							return edt
							break

						elif (edt.day > date.day and
							edt.month >= date.month and
							edt.year >= date.year and
							edt < self.untildatetime):
							return edt
							break

						elif (edt.month > date.month and
							edt.year >= date.year and
							edt < self.untildatetime):
							return edt
							break

						elif (edt.year > date.year and
							edt < self.untildatetime):
							return edt
							break

						elif (edt >= self.untildatetime):
							return False
							break

						edt = edt + datetime.timedelta(increment)

					return False
				return False

			elif self.frequency == 'MONTHLY':
				if self.interval:
					increment = self.interval
				else:
					increment = 1
				sd = self.startDate
				ed = self.endDate
				sdt = self.startDateTime
				edt = self.endDateTime
				counter = 0
				counterUntil = 0
				counterSetpos = 0
				MO = 0; TU = 1; WE = 2; TH = 3; FR = 4; SA = 5; SU = 6
				byday = None

				if typ.endswith('On'):

					if self.bySetpos and self.byDay:
						sd1 = datetime.date(sd.year, sd.month, 1)
						ed1 = ed
						sdt1 = datetime.datetime(sd.year, sd.month, 1, sdt.hour, sdt.minute)
						while (byday != sd1.weekday()) or (sd1 <= date and sd1 < self.untildate):
							for i in self.byDay:
								byday = (vars()[i.upper()])

								if self.untilDate:
									if self.untilDate < sdt1:
										return False
										break	

								if (sd1 < self.untildate) and (byday == sd1.weekday()):
									if sd1.month in [1, 3, 5, 7, 8, 10, 12]:
										days_in_month = 31
									elif sd1.month != 2:
										days_in_month = 30
									elif sd1.year % 4 == 0:
										days_in_month = 29
									else:
										days_in_month = 28
									setpos = (sd1.day - days_in_month) /7
									counterSetpos +=1
									if counterSetpos == int(self.bySetpos) or setpos == int(self.bySetpos):
										if self.count:
											counterUntil += 1
											if counterUntil > self.count:
												return False	
										if (sd1 == date):
											if typ == 'occursOn':
												return True
											return sdt1
									
							sd1 = sd1 + datetime.timedelta(1)
							ed1 = ed1 + datetime.timedelta(1)
							sdt1 = sdt1 + datetime.timedelta(1)

							if sd1.day == 1:
								counterSetpos = 0


					if self.byDay and not self.bySetpos:
						sd1 = sd
						ed1 = ed
						sdt1 = sdt
						while (byday != sd1.weekday()) or (sd1 <= date and sd1 < self.untildate):	
							for i in self.byDay:
								byday = (vars()[i.upper()])

								if self.untilDate:
									if self.untilDate < sdt1:
										return False
										break	


								if (sd1 < self.untildate) and (byday == sd1.weekday()):
									if self.count:
										counterUntil += 1
										if counterUntil > self.count:
											return False
	
									if (sd1 == date):
										if typ == 'occursOn':
											return True
											break
										return sdt1
										break

							sd1 = sd1 + datetime.timedelta(1)
							ed1 = ed1 + datetime.timedelta(1)
							sdt1 = sdt1 + datetime.timedelta(1)

	
					if self.byMonthday and not self.byDay:
						ed1 = ed
						sd1 = datetime.date(sd.year, sd.month, int(self.byMonthday[0]))
						if sd != sd1:
							counter -=1

						while(sd1 <= date):
							for i in self.byMonthday:
								sd1 = datetime.date(sd1.year, sd1.month, int(i))
								sdt1 = datetime.datetime(sd1.year, sd1.month, int(i), sdt.hour, sdt.minute)

								if self.untilDate:
									if self.untilDate < sdt1:
										return False
										break	

								if self.count:
									counterUntil += 1
									if counterUntil > self.count:
										return False	

								if (sd1 == date) and (sd1 < self.untildate):
									if typ == 'occursOn':
										return True
										break
									return sdt1
									break

								if typ == 'occursOn':
									if (sd1 <= date) and (ed >= date) and (sd1 < self.untildate):
										if self.endDateTime.hour == 0 and self.endDateTime.second == 0:
											return False
										else:
											return True
											break

							year, month = sd1.timetuple()[:2]
							day = int(self.byMonthday[0])
							new_month = (month + increment)
							if new_month > 12:
								new_year = year + 1
							else:
								new_year = year
							new_month = new_month % 12
							if new_month == 0:
								new_month = 12
							try:  # Routine to take next month if day is not valid. e.g. February does only have 28 days
								sd1 = datetime.date(new_year, new_month, day)
							except:
								new_month = (new_month + increment)
								if new_month > 12:
									new_year = year + 1
								else:
									new_year = year
								new_month = new_month % 12
								if new_month == 0:
									new_month = 12
								sd1 = datetime.date(new_year, new_month, day)


							sdt1 = datetime.datetime(sd1.year, sd1.month, sd1.day, sdt1.hour, sdt1.minute)


							year, month, day = ed1.timetuple()[:3]
							new_month = (month + increment)
							if new_month > 12:
								new_year = year + 1
							else:
								new_year = year
							new_month = new_month % 12
							if new_month == 0:
								new_month = 12
							try:  # Routine to take next month if day is not valid. e.g. February does only have 28 days
								ed1 = datetime.date(new_year, new_month, day)
							except:
								new_month = (new_month + increment)
								if new_month > 12:
									new_year = year + 1
								else:
									new_year = year
								new_month = new_month % 12
								if new_month == 0:
									new_month = 12
								ed1 = datetime.date(new_year, new_month, day)
	

					while(sd <= date) and not self.byDay and not self.byMonthday:

						if self.untilDate:
							if self.untilDate < sdt:
								return False
								break	

						if self.count:
							counter += 1
							if counter > self.count:
								return False	

						if (sd == date) and (sd < self.untildate):
							if typ == 'occursOn':
								return True
								break
							return sdt
							break

						if typ == 'occursOn':
							if (sd <= date) and (ed >= date) and (sd < self.untildate):
								if self.endDateTime.hour == 0 and self.endDateTime.second == 0:
									return False
								else:
									return True
									break


						year, month, day = sd.timetuple()[:3]
						new_month = (month + increment)
						if new_month > 12:
							new_year = year + 1
						else:
							new_year = year
						new_month = new_month % 12
						if new_month == 0:
							new_month = 12
						try:  # Routine to take next month if day is not valid. e.g. February does only have 28 days
							sd = datetime.date(new_year, new_month, day)
						except:
							new_month = (new_month + increment)
							if new_month > 12:
								new_year = year + 1
							else:
								new_year = year
							new_month = new_month % 12
							if new_month == 0:
								new_month = 12
							sd = datetime.date(new_year, new_month, day)


						sdt = datetime.datetime(sd.year, sd.month, sd.day, sdt.hour, sdt.minute)


						year, month, day = ed.timetuple()[:3]
						new_month = (month + increment)
						if new_month > 12:
							new_year = year + 1
						else:
							new_year = year
						new_month = new_month % 12
						if new_month == 0:
							new_month = 12
						try:  # Routine to take next month if day is not valid. e.g. February does only have 28 days
							ed = datetime.date(new_year, new_month, day)
						except:
							new_month = (new_month + increment)
							if new_month > 12:
								new_year = year + 1
							else:
								new_year = year
							new_month = new_month % 12
							if new_month == 0:
								new_month = 12
							ed = datetime.date(new_year, new_month, day)


					return False

				if typ == 'startsAfter':
					sdt = self.startDateTime
					while (sdt < self.untildatetime):
						if self.untilDate:
							if self.untilDate < sdt:
								return False
								break	

						if self.count:
							counter += 1
							if counter > self.count:
								return False
								break

						if (sdt.time() > date.time() and
							sdt.day >= date.day and
							sdt.month >= date.month and
							sdt.year >= date.year and
							sdt < self.untildatetime):
							return sdt
							break

						elif (sdt.day > date.day and
							sdt.month >= date.month and
							sdt.year >= date.year and
							sdt < self.untildatetime):
							return sdt
							break

						elif (sdt.month > date.month and
							sdt.year >= date.year and
							sdt < self.untildatetime):
							return sdt	
							break

						elif (sdt.year > date.year and
							sdt < self.untildatetime):
							return sdt
							break

						elif (sdt >= self.untildatetime):
							return False
							break

						year, month, day, hour, minute = sdt.timetuple()[:5]
						new_month = (month + increment)
						if new_month > 12:
							new_year = year + 1
						else:
							new_year = year
						new_month = new_month % 12
						if new_month == 0:
							new_month = 12
						try:
							sd = datetime.date(new_year, new_month, day)
						except:
							new_month = (new_month + increment)
							if new_month > 12:
								new_year = year + 1
							else:
								new_year = year
							new_month = new_month % 12
							if new_month == 0:
								new_month = 12
							sd = datetime.date(new_year, new_month, day)
						sdt = datetime.datetime(new_year, new_month, day, hour, minute)

					return False

				if typ == 'endsAfter':
					edt = self.endDateTime
					while (edt < self.untildatetime):
						if self.untilDate:
							if self.untilDate < edt:
								return False
								break	

						if self.count:
							counter += 1
							if counter > self.count:
								return False
								break

						if (edt.time() > date.time() and
							edt.day >= date.day and
							edt.month >= date.month and
							edt.year >= date.year and
							edt < self.untildatetime):
							return edt
							break

						elif (edt.day > date.day and
							edt.month >= date.month and
							edt.year >= date.year and
							edt < self.untildatetime):
							return edt
							break

						elif (edt.month > date.month and
							edt.year >= date.year and
							edt < self.untildatetime):
							return edt	
							break

						elif (edt.year > date.year and
							edt < self.untildatetime):
							return edt
							break

						elif (edt >= self.untildatetime):
							return False
							break

						year, month, day, hour, minute = edt.timetuple()[:5]
						new_month = (month + increment)
						if new_month > 12:
							new_year = year + 1
						else:
							new_year = year
#						new_year = year + (new_month / 12)
						new_month = new_month % 12
						if new_month == 0:
							new_month = 12
						try:
							ed = datetime.date(new_year, new_month, day)
						except:
							new_month = (new_month + increment)
							if new_month > 12:
								new_year = year + 1
							else:
								new_year = year
#							new_year = year + (new_month / 12)
							new_month = new_month % 12
							if new_month == 0:
								new_month = 12
							ed = datetime.date(new_year, new_month, day)
						edt = datetime.datetime(new_year, new_month, day, hour, minute)

					return False

			elif self.frequency == 'YEARLY':
				if self.interval:
					increment = self.interval
				else:
					increment = 1

				sd = self.startDate
				ed = self.endDate
				sdt = self.startDateTime
				edt = self.endDateTime
				sdt2 = None
				counter = 0
				if self.byMonth:
					for month in self.byMonth:
						sd = datetime.date(sd.year, int(month), sd.day)
						sdt = datetime.datetime(sd.year, int(month), sd.day, sdt.hour, sdt.minute)
						counterSetpos = 0

						if self.byMonthday:
							for day in self.byMonthday:
								sd = datetime.date(sd.year, sd.month, int(day))
								sdt = datetime.datetime(sd.year, sd.month, int(day), sdt.hour, sdt.minute)
						if self.byDay:
							sd = datetime.date(sd.year, sd.month, 1)
							sdt = datetime.datetime(sd.year, sd.month, 1, sdt.hour, sdt.minute)
							byday = None
							MO = 0; TU = 1; WE = 2; TH = 3; FR = 4; SA = 5; SU = 6
							for i in self.byDay:
								byday = (vars()[i.upper()])


								if typ.endswith('On'):
									while (sd <= date and sd < self.untildate):	

										if self.untilDate:
											if self.untilDate < sdt:
												return False
												break	

										if self.bySetpos and (byday == sd.weekday()):
											if sd.month in [1, 3, 5, 7, 8, 10, 12]:
												days_in_month = 31
											elif sd.month != 2:
												days_in_month = 30
											elif sd.year % 4 == 0:
												days_in_month = 29
											else:
												days_in_month = 28
											setpos = (sd.day - days_in_month) /7
											counterSetpos +=1
											if  counterSetpos == int(self.bySetpos) or setpos == int(self.bySetpos):
												if self.count:
													counterUntil += 1
													if counterUntil > self.count:
														return False	
												if (sd == date):
													if typ == 'occursOn':
														return True
													return sdt

										if self.byDay and not self.bySetpos:
											if self.count and (byday == sd.weekday()):
												counterUntil += 1
												if counterUntil > self.count:
													return False
													break

											if (sd == date) and (sd < self.untildate) and (byday == sd.weekday()):
												if typ == 'occursOn':
													return True
													break
												return sdt
												break

										if not self.bySetpos and not self.byDay:
											if self.count:
												counterUntil += 1
												if counterUntil > self.count:
													return False
													break

											if (sd == date) and (sd < self.untildate):
												if typ == 'occursOn':
													return True
													break
												return sdt
												break


											if typ == 'occursOn':
												if (sd <= date) and (ed >= date) and (sd < self.untildate):
													if self.endDateTime.hour == 0 and self.endDateTime.second == 0:
														return False
													else:
														return True
														break						

										sd = sd + datetime.timedelta(1)
										ed = ed + datetime.timedelta(1)
										sdt = sdt + datetime.timedelta(1)

										if sd.day == 1:
											sd = sd + datetime.timedelta(-1)
											ed = ed + datetime.timedelta(-1)
											sdt = sdt + datetime.timedelta(-1)
											sd = datetime.date(sd.year, sd.month, 1)
											sdt = datetime.datetime(sd.year, sd.month, 1, sdt.hour, sdt.minute)
											sd = sd.replace(year=sd.year+increment)
											ed = ed.replace(year=ed.year+increment)
											sdt = sdt.replace(year=sdt.year+increment)
											counterSetpos = 0
									return False

				if typ.endswith('On'):
					while(sd <= date):
						if self.untilDate:
							if self.untilDate < sdt:
								return False
								break	
	
						if self.count:
							counter += 1
							if counter > self.count:
								return False	
	
						if (sd == date) and (sd < self.untildate):
							if typ == 'occursOn':
								return True
								break
							return sdt
							break
	
						if typ == 'occursOn':
							if (sd <= date) and (ed >= date) and (sd < self.untildate):
								if self.endDateTime.hour == 0 and self.endDateTime.second == 0:
									return False
								else:
									return True
						if sd.month == 2 and sd.day > 28:
							new_year = sd.year+(increment)
							while not self.isleap(new_year):
								new_year=new_year+(increment)
						else:
							new_year = sd.year+(increment)
						sd = sd.replace(year=new_year)

						if ed.month == 2 and ed.day > 28:
							new_year = ed.year+(increment)
							while not self.isleap(new_year):
								new_year=new_year+(increment)
						else:
							new_year = ed.year+(increment)
						ed = ed.replace(year=new_year)

						if sdt.month == 2 and sdt.day > 28:
							new_year = sdt.year+(increment)
							while not self.isleap(new_year):
								new_year=new_year+(increment)
						else:
							new_year = sdt.year+(increment)
						sdt = sdt.replace(year=new_year)
	
					return False





				if typ == 'startsAfter':
					sdt = self.startDateTime
					while (sdt < self.untildatetime):
						if self.untilDate:
							if self.untilDate < sdt:
								return False
								break	

						if self.count:
							counter += 1
							if counter > self.count:
								return False
								break

						if (sdt.time() > date.time() and
							sdt.day >= date.day and
							sdt.month >= date.month and
							sdt.year >= date.year and
							sdt < self.untildatetime):
							return sdt
							break

						elif (sdt.day > date.day and
							sdt.month >= date.month and
							sdt.year >= date.year and
							sdt < self.untildatetime):
							return sdt
							break

						elif (sdt.month > date.month and
							sdt.year >= date.year and
							sdt < self.untildatetime):
							return sdt
							break

						elif (sdt.year > date.year and
							sdt < self.untildatetime):
							return sdt
							break

						elif (sdt >= self.untildatetime):
							return False
							break

						if sdt.month == 2 and sdt.day > 28:
							new_year = sdt.year+(increment)
							while not self.isleap(new_year):
								new_year=new_year+(increment)
						else:
							new_year = sdt.year+(increment)
						sdt = sdt.replace(year=new_year)

					return False

				if typ == 'endsAfter':
					edt = self.endDateTime
					while (edt < self.untildatetime):
						if self.untilDate:
							if self.untilDate < edt:
								return False
								break	

						if self.count:
							counter += 1
							if counter > self.count:
								return False
								break

						if (edt.time() > date.time() and
							edt.day >= date.day and
							edt.month >= date.month and
							edt.year >= date.year and
							edt < self.untildatetime):
							return edt
							break

						elif (edt.day > date.day and
							edt.month >= date.month and
							edt.year >= date.year and
							edt < self.untildatetime):
							return edt
							break

						elif (edt.month > date.month and
							edt.year >= date.year and
							edt < self.untildatetime):
							return edt
							break

						elif (edt.year > date.year and
							edt < self.untildatetime):
							return edt
							break

						elif (edt >= self.untildatetime):
							return False
							break

						if edt.month == 2 and edt.day > 28:
							new_year = edt.year+(increment)
							while not self.isleap(new_year):
								new_year=new_year+(increment)
						else:
							new_year = edt.year+(increment)
						edt = edt.replace(year=new_year)

					return False
			return False
		return False

	def isleap(self, year):
	    """Return 1 for leap years, 0 for non-leap years."""
	    return year % 4 == 0 and (year % 100 != 0 or year % 400 == 0)

if __name__ == '__main__':
	reader = ICalReader()
	reader.readURL('calendar.ics')
	for event in reader.events:
		if event.rule or event.endDate >= datetime.date.today(): # prints only events which could be valid
			print str(event.startDateTime) + '  ' + str(event.endDateTime) + '  ' + str(event)
