// © 2016 and later: Unicode, Inc. and others.
// License & terms of use: http://www.unicode.org/copyright.html#License
/*
 ******************************************************************************
 * Copyright (C) 1996-2013, International Business Machines Corporation and   *
 * others. All Rights Reserved.                                               *
 ******************************************************************************
 */

/**
 * Test for CharsetSelector
 *
 * This is a port of ucnvseltst.c from ICU4C
 *
 * Tests related to serialization are not ported in this version. In addition,
 * the TestConversionUTF8 method is not going to be ported, as UTF8 is seldom used
 * in Java.
 *
 * @author Shaopeng Jia
 */

package com.ibm.icu.dev.test.charset;

import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.JUnit4;

import com.ibm.icu.charset.CharsetICU;
import com.ibm.icu.charset.CharsetProviderICU;
import com.ibm.icu.charset.CharsetSelector;
import com.ibm.icu.dev.test.TestFmwk;
import com.ibm.icu.text.UTF16;
import com.ibm.icu.text.UnicodeSet;

@RunWith(JUnit4.class)
public class TestSelection extends TestFmwk {
    @Test
    public void TestConversionUTF16() {
        List<String> testEncodings;
        if (TestFmwk.getExhaustiveness() < 6) {
            testEncodings = Arrays.asList(
                    "Big5",
                    "EUC-JP",
                    "EUC-KR",
                    "GB2312",
                    "ISO-2022-JP",
                    "ISO-2022-KR",
                    "ISO-8859-1",
                    "ISO-8859-10",
                    "ISO-8859-13",
                    "ISO-8859-14",
                    "ISO-8859-15",
                    "ISO-8859-2",
                    "ISO-8859-3",
                    "ISO-8859-4",
                    "ISO-8859-5",
                    "ISO-8859-6",
                    "ISO-8859-7",
                    "ISO-8859-8",
                    "ISO-8859-9",
                    "KOI8-R",
                    "Shift_JIS",
                    "TIS-620",
                    "UTF-16",
                    "UTF-32",
                    "UTF-8",
                    "windows-1250",
                    "windows-1251",
                    "windows-1252",
                    "windows-1253",
                    "windows-1254",
                    "windows-1255",
                    "windows-1256",
                    "windows-1257",
                    "windows-1258"
            );
        } else {
            testEncodings = Arrays.asList(CharsetProviderICU.getAvailableNames());
        }

        UnicodeSet[] excludedSets = new UnicodeSet[] {
                UnicodeSet.EMPTY,           // empty set
                new UnicodeSet(30, 500),    // arbitrary code range for testing
        };

        for (UnicodeSet excluded : excludedSets) {
            CharsetSelector sel = new CharsetSelector(testEncodings, excluded, CharsetICU.ROUNDTRIP_SET);
            for (int i = 0; i < texts.length; i++) {
                List<String> result = sel.selectForString(texts[i]);
                verifyResultUTF16(texts[i], testEncodings, result, excluded, CharsetICU.ROUNDTRIP_SET);
            }
        }
    }

    private void verifyResultUTF16(String s, List<String> encodings, List<String> result,
            UnicodeSet excluded, int mappingType) {
        for (String enc : encodings) {
            UnicodeSet unicodePointSet = new UnicodeSet();
            Charset testCharset = CharsetICU.forNameICU(enc);
            ((CharsetICU) testCharset).getUnicodeSet(unicodePointSet, mappingType);

            boolean manualResult = true;
            int index = 0;
            while (index < s.length()) {
                int ch = UTF16.charAt(s, index);
                if (!excluded.contains(ch) && !unicodePointSet.contains(ch)) {
                    manualResult = false;
                    break;
                }
                index += UTF16.getCharCount(ch);
            }

            boolean selectResult = result.contains(enc);
            if (selectResult != manualResult) {
                errln("failure in charset selector! Charset " + enc
                        + " had conflicting results manual: " + manualResult
                        + ", selectForString(): " + selectResult + "\n");
            }
        }
    }

    /* This test is to provide better code coverage for CharsetSelector */
    @Test
    public void TestCharsetSelectorCodeCoverage() {
        List emptyList = new ArrayList();
        UnicodeSet nonEmptySet = new UnicodeSet();

        nonEmptySet.add(0x0001, 0x0FFF);

        CharsetSelector sel = null;

        sel = new CharsetSelector(emptyList, nonEmptySet, CharsetICU.ROUNDTRIP_SET);

        /* Test bogus mappingType */
        try {
            sel = new CharsetSelector(null, null, -1);
            errln("IllegalArgumentException should have been thrown by CharsetSelector when given a bogus mappingType, but got - " + sel.toString());
        } catch (IllegalArgumentException e) {
            logln("IllegalArgumentException was thrown by CharsetSelector with a bogus mappingType.");
        }
    }

    private String[] texts = {
            "Cos\'\u00E8 Unicode?\n\nUnicode assegna un numero univoco a ogni carattere,\nindipendentemente dalla piattaforma," +
            "\nindipendentemente dall\'applicazione,\nindipendentemente dalla lingua.\n\nI computer, in buona sostanza," +
            " non sanno far altro che trattare numeri. Per immagazzinare in memoria lettere o altri segni \u00E8 necessario" +
            " che a ogni carattere venga assegnato un numero. Esistono centinaia di sistemi di codifica, preesistenti a " +
            "Unicode, e ognuno di questi abbina i numeri ai caratteri in modo differente. Nessuna di queste codifiche " +
            "comprende un numero di caratteri sufficiente per tutte le circostanze. Per le sole lingue dell\'Unione " +
            "Europea, ad esempio, \u00E8 necessario utilizzare parecchi sistemi di codifica distinti. Anche considerando " +
            "una solo lingua, come l\'italiano, non esiste una codifica unica che comprenda tutte le lettere e tutti i " +
            "segni di punteggiatura e simboli tecnici di uso comune.\n\nQuesti sistemi di codifica, inoltre, sono in " +
            "contraddizione l\'uno con l\'altro. Succede che due codifiche utilizzino lo stesso numero per due caratteri " +
            "diversi o che, viceversa, adottino numeri diversi per lo stesso carattere. Qualsiasi elaboratore, e a maggior " +
            "ragione un server di rete, ha bisogno di utilizzare codifiche diverse. Il problema \u00E8 che, quando i dati " +
            "passano da una codifica a un\'altra, o da una piattaforma a un\'altra, si corre il serio rischio di perdere " +
            "informazioni.\nUnicode sta cambiando le cose!\n\nUnicode attribuisce un numero univoco a ogni carattere, " +
            "indipendentemente dalla piattaforma, dall\'applicativo, dalla lingua. Lo standard Unicode \u00E8 stato " +
            "adottato da leader di mercato del calibro di Apple, HP, IBM, JustSystem, Microsoft, Oracle, SAP, Sun, " +
            "Sybase, Unisys e molti altri. Unicode \u00E8 alla base di molti moderni standard, come XML, Java, ECMAScript " +
            "(JavaScript), LDAP, CORBA 3.0, WML eccetera, e costituisce l\'implementazione ufficiale dello standard " +
            "internazionale ISO/IEC 10646. Unicode \u00E8 supportato da molti sistemi operativi, da tutti i pi\u00F9 " +
            "moderni web browser e da molti altri prodotti. L\'emergere dello standard Unicode, unito alla recente " +
            "disponibilit\u00E0 di strumenti che lo supportano, \u00E8 fra i pi\u00F9 significativi sviluppi della " +
            "tecnologia della globalizzazione del software.\n\nL\'adozione di Unicode sui siti web e nelle applicazioni " +
            "client/server o multi-tiered, rispetto all\'utilizzo dei set di caratteri tradizionali, permette un " +
            "significativo abbattimento dei costi di gestione. Unicode consente che un\'unica versione di un software " +
            "o di un sito web siano fruibili con piattaforme, lingue e paesi diversi, evitando la necessit\u00E0 di " +
            "reingenierizzare il prodotto per ogni situazione specifica. Permette, inoltre, il trasporto del testo " +
            "fra sistemi diversi senza che abbia luogo alcuna corruzione dei dati.\n",

            "\u0427\u0442\u043E \u0442\u0430\u043A\u043E\u0435 Unicode? Unicode - \u044D\u0442\u043E" +
            " \u0443\u043D\u0438\u043A\u0430\u043B\u044C\u043D\u044B\u0439 \u043A\u043E\u0434" +
            " \u0434\u043B\u044F \u043B\u044E\u0431\u043E\u0433\u043E \u0441\u0438\u043C\u0432\u043E\u043B\u0430," +
            " \u043D\u0435\u0437\u0430\u0432\u0438\u0441\u0438\u043C\u043E \u043E\u0442 " +
            "\u043F\u043B\u0430\u0442\u0444\u043E\u0440\u043C\u044B, \u043D\u0435\u0437\u0430\u0432\u0438\u0441\u0438\u043C\u043E" +
            " \u043E\u0442 \u043F\u0440\u043E\u0433\u0440\u0430\u043C\u043C\u044B, " +
            "\u043D\u0435\u0437\u0430\u0432\u0438\u0441\u0438\u043C\u043E \u043E\u0442 \u044F\u0437\u044B\u043A\u0430. " +
            "\u041F\u043E \u0441\u0432\u043E\u0435\u0439 \u043F\u0440\u0438\u0440\u043E\u0434\u0435" +
            " \u043A\u043E\u043C\u043F\u044C\u044E\u0442\u0435\u0440\u044B \u043C\u043E\u0433\u0443\u0442 " +
            "\u0440\u0430\u0431\u043E\u0442\u0430\u0442\u044C \u043B\u0438\u0448\u044C \u0441" +
            " \u0447\u0438\u0441\u043B\u0430\u043C\u0438. \u0418 \u0434\u043B\u044F " +
            "\u0442\u043E\u0433\u043E, \u0447\u0442\u043E\u0431\u044B \u043E\u043D\u0438" +
            " \u043C\u043E\u0433\u043B\u0438 \u0445\u0440\u0430\u043D\u0438\u0442\u044C " +
            "\u0432 \u043F\u0430\u043C\u044F\u0442\u0438 \u0431\u0443\u043A\u0432\u044B " +
            "\u0438\u043B\u0438 \u0434\u0440\u0443\u0433\u0438\u0435 \u0441\u0438\u043C\u0432\u043E\u043B\u044B, " +
            "\u043A\u0430\u0436\u0434\u043E\u043C\u0443 \u0442\u0430\u043A\u043E\u043C\u0443 " +
            "\u0441\u0438\u043C\u0432\u043E\u043B\u0443 \u0434\u043E\u043B\u0436\u043D\u043E " +
            "\u0431\u044B\u0442\u044C \u043F\u043E\u0441\u0442\u0430\u0432\u043B\u0435\u043D\u043E " +
            "\u0432 \u0441\u043E\u043E\u0442\u0432\u0435\u0442\u0441\u0442\u0432\u0438\u0435 " +
            "\u0447\u0438\u0441\u043B\u043E. \u0414\u043E \u0442\u043E\u0433\u043E, " +
            "\u043A\u0430\u043A \u043F\u043E\u044F\u0432\u0438\u043B\u0441\u044F Unicode, " +
            "\u0432 \u043C\u0438\u0440\u0435 \u0438\u043C\u0435\u043B\u0438 " +
            "\u0445\u043E\u0436\u0434\u0435\u043D\u0438\u0435 \u0441\u043E\u0442\u043D\u0438 " +
            "\u0440\u0430\u0437\u043B\u0438\u0447\u043D\u044B\u0445 \u0441\u0445\u0435\u043C " +
            "\u043F\u043E\u0434\u043E\u0431\u043D\u043E\u0433\u043E " +
            "\u043A\u043E\u0434\u0438\u0440\u043E\u0432\u0430\u043D\u0438\u044F " +
            "\u0441\u0438\u043C\u0432\u043E\u043B\u043E\u0432. \u041D\u043E \u043D\u0438 " +
            "\u043E\u0434\u043D\u0430 \u0438\u0437 \u044D\u0442\u0438\u0445 " +
            "\u0441\u0445\u0435\u043C \u043D\u0435 \u0431\u044B\u043B\u0430 " +
            "\u0441\u0442\u043E\u043B\u044C \u0443\u043D\u0438\u0432\u0435\u0440\u0441\u0430\u043B\u044C\u043D\u043E\u0439," +
            " \u0447\u0442\u043E\u0431\u044B \u043E\u043F\u0438\u0441\u0430\u0442\u044C " +
            "\u0432\u0441\u0435 \u043D\u0435\u043E\u0431\u0445\u043E\u0434\u0438\u043C\u044B\u0435 " +
            "\u0441\u0438\u043C\u0432\u043E\u043B\u044B: \u043D\u0430\u043F\u0440\u0438\u043C\u0435\u0440, " +
            "\u0442\u043E\u043B\u044C\u043A\u043E \u0434\u043B\u044F " +
            "\u043A\u043E\u0434\u0438\u0440\u043E\u0432\u0430\u043D\u0438\u044F " +
            "\u0431\u0443\u043A\u0432, \u0432\u0445\u043E\u0434\u044F\u0449\u0438\u0445 \u0432 " +
            "\u0430\u043B\u0444\u0430\u0432\u0438\u0442\u044B \u044F\u0437\u044B\u043A\u043E\u0432 " +
            "\u0415\u0432\u0440\u043E\u043F\u0435\u0439\u0441\u043A\u043E\u0433\u043E " +
            "\u0421\u043E\u043E\u0431\u0449\u0435\u0441\u0442\u0432\u0430, " +
            "\u043D\u0435\u043E\u0431\u0445\u043E\u0434\u0438\u043C\u043E \u0431\u044B\u043B\u043E " +
            "\u0438\u0441\u043F\u043E\u043B\u044C\u0437\u043E\u0432\u0430\u0442\u044C " +
            "\u043D\u0435\u0441\u043A\u043E\u043B\u044C\u043A\u043E " +
            "\u0440\u0430\u0437\u043B\u0438\u0447\u043D\u044B\u0445 \u043A\u043E\u0434\u0438\u0440\u043E\u0432\u043E\u043A. " +
            "\u041F\u043E \u0431\u043E\u043B\u044C\u0448\u043E\u043C\u0443 \u0441\u0447\u0451\u0442\u0443 " +
            "\u0434\u0430\u0436\u0435 \u0438 \u0434\u043B\u044F \u043E\u0442\u0434\u0435\u043B\u044C\u043D\u043E\u0433\u043E " +
            "\u044F\u0437\u044B\u043A\u0430, \u0441\u043A\u0430\u0436\u0435\u043C, " +
            "\u0430\u043D\u0433\u043B\u0438\u0439\u0441\u043A\u043E\u0433\u043E, \u043D\u0435 " +
            "\u0441\u0443\u0449\u0435\u0441\u0442\u0432\u043E\u0432\u0430\u043B\u043E \u0435\u0434\u0438\u043D\u043E\u0439 " +
            "\u0441\u0438\u0441\u0442\u0435\u043C\u044B \u043A\u043E\u0434\u0438\u0440\u043E\u0432\u0430\u043D\u0438\u044F," +
            " \u0432\u043A\u043B\u044E\u0447\u0430\u0432\u0448\u0435\u0439 \u0432 \u0441\u0435\u0431\u044F \u0432\u0441\u0435 " +
            "\u043E\u0431\u044B\u0447\u043D\u043E \u0438\u0441\u043F\u043E\u043B\u044C\u0437\u0443\u0435\u043C\u044B\u0435 " +
            "\u0431\u0443\u043A\u0432\u044B, \u0437\u043D\u0430\u043A\u0438 \u043F\u0443\u043D\u043A\u0442\u0443\u0430\u0446\u0438\u0438" +
            " \u0438 \u0442\u0435\u0445\u043D\u0438\u0447\u0435\u0441\u043A\u0438\u0435 \u0441\u0438\u043C\u0432\u043E\u043B\u044B." +
            " \u0411\u043E\u043B\u0435\u0435 \u0442\u043E\u0433\u043E, " +
            "\u0432\u0441\u0435 \u044D\u0442\u0438 \u0441\u0445\u0435\u043C\u044B " +
            "\u043A\u043E\u0434\u0438\u0440\u043E\u0432\u0430\u043D\u0438\u044F \u0447\u0430\u0441\u0442\u043E" +
            " \u0434\u0430\u0436\u0435 \u043D\u0435 \u0431\u044B\u043B\u0438" +
            " \u0441\u043E\u0432\u043C\u0435\u0441\u0442\u0438\u043C\u044B \u0434\u0440\u0443\u0433 \u0441" +
            " \u0434\u0440\u0443\u0433\u043E\u043C. \u041A \u043F\u0440\u0438\u043C\u0435\u0440\u0443," +
            " \u0434\u0432\u0435 \u0440\u0430\u0437\u043D\u044B\u0435 \u043A\u043E\u0434\u0438\u0440\u043E\u0432\u043A\u0438 " +
            "\u043C\u043E\u0433\u043B\u0438 \u0438\u0441\u043F\u043E\u043B\u044C\u0437\u043E\u0432\u0430\u0442\u044C " +
            "\u043E\u0434\u0438\u043D \u0438 \u0442\u043E\u0442 \u0436\u0435 \u043A\u043E\u0434 \u0434\u043B\u044F " +
            "\u043F\u0440\u0435\u0434\u0441\u0442\u0430\u0432\u043B\u0435\u043D\u0438\u044F \u0434\u0432\u0443\u0445 " +
            "\u0440\u0430\u0437\u043D\u044B\u0445 \u0441\u0438\u043C\u0432\u043E\u043B\u043E\u0432 \u0438\u043B\u0438 " +
            "\u043F\u0440\u0438\u0441\u0432\u0430\u0438\u0432\u0430\u0442\u044C \u0440\u0430\u0437\u043D\u044B\u0435 " +
            "\u043A\u043E\u0434\u044B \u043E\u0434\u043D\u043E\u0439 \u0438 \u0442\u043E\u0439 \u0436\u0435 " +
            "\u0431\u0443\u043A\u0432\u0435. \u0412 \u044D\u0442\u043E\u0439 \u0441\u0438\u0442\u0443\u0430\u0446\u0438\u0438 " +
            "\u0434\u043B\u044F \u043B\u044E\u0431\u043E\u0433\u043E \u043A\u043E\u043C\u043F\u044C\u044E\u0442\u0435\u0440\u0430, " +
            "\u0430 \u043E\u0441\u043E\u0431\u0435\u043D\u043D\u043E \u0441\u0435\u0440\u0432\u0435\u0440\u0430, " +
            "\u043F\u0440\u0438\u0445\u043E\u0434\u0438\u043B\u043E\u0441\u044C " +
            "\u043F\u043E\u0434\u0434\u0435\u0440\u0436\u0438\u0432\u0430\u0442\u044C " +
            "\u043D\u0435\u0441\u043A\u043E\u043B\u044C\u043A\u043E \u0440\u0430\u0437\u043D\u044B\u0445 " +
            "\u043A\u043E\u0434\u0438\u0440\u043E\u0432\u043E\u043A, \u043A\u043E\u0442\u043E\u0440\u044B\u0435 " +
            "\u043C\u043E\u0433\u043B\u0438 \u043F\u043E\u043D\u0430\u0434\u043E\u0431\u0438\u0442\u044C\u0441\u044F, " +
            "\u043D\u043E \u0434\u0430\u0436\u0435 \u0438 \u0442\u043E\u0433\u0434\u0430 \u043F\u0440\u0438 " +
            "\u043F\u0435\u0440\u0435\u0434\u0430\u0447\u0435 \u0434\u0430\u043D\u043D\u044B\u0445 \u043D\u0430 " +
            "\u0434\u0440\u0443\u0433\u0443\u044E \u043F\u043B\u0430\u0442\u0444\u043E\u0440\u043C\u0443 " +
            "\u0438\u043B\u0438 \u043F\u0440\u0438 \u0438\u0445 " +
            "\u043F\u0440\u0435\u043E\u0431\u0440\u0430\u0437\u043E\u0432\u0430\u043D\u0438\u0438 \u0432 " +
            "\u0434\u0440\u0443\u0433\u0443\u044E \u043A\u043E\u0434\u0438\u0440\u043E\u0432\u043A\u0443 " +
            "\u0432\u0441\u0435\u0433\u0434\u0430 \u043E\u0441\u0442\u0430\u0432\u0430\u043B\u0441\u044F " +
            "\u0440\u0438\u0441\u043A, \u0447\u0442\u043E \u044D\u0442\u0438 \u0434\u0430\u043D\u043D\u044B\u0435 " +
            "\u043E\u043A\u0430\u0436\u0443\u0442\u0441\u044F " +
            "\u043F\u043E\u0432\u0440\u0435\u0436\u0434\u0451\u043D\u043D\u044B\u043C\u0438. " +
            "Unicode \u0438\u0437\u043C\u0435\u043D\u044F\u0435\u0442 \u0442\u0430\u043A\u043E\u0435 " +
            "\u043F\u043E\u043B\u043E\u0436\u0435\u043D\u0438\u0435 \u0432\u0435\u0449\u0435\u0439! " +
            "\u0421\u0438\u0441\u0442\u0435\u043C\u0430 Unicode \u043F\u0440\u0438\u0441\u0432\u0430\u0438\u0432\u0430\u0435\u0442 " +
            "\u0443\u043D\u0438\u043A\u0430\u043B\u044C\u043D\u044B\u0439 \u043A\u043E\u0434 \u043B\u044E\u0431\u043E\u043C\u0443 " +
            "\u0441\u0438\u043C\u0432\u043E\u043B\u0443, \u043D\u0435\u0437\u0430\u0432\u0438\u0441\u0438\u043C\u043E \u043E\u0442 " +
            "\u043F\u043B\u0430\u0442\u0444\u043E\u0440\u043C\u044B, \u043D\u0435\u0437\u0430\u0432\u0438\u0441\u0438\u043C\u043E " +
            "\u043E\u0442 \u043F\u0440\u043E\u0433\u0440\u0430\u043C\u043C\u044B, " +
            "\u043D\u0435\u0437\u0430\u0432\u0438\u0441\u0438\u043C\u043E \u043E\u0442 \u044F\u0437\u044B\u043A\u0430. " +
            "Unicode \u0431\u044B\u043B \u043F\u0440\u0438\u043D\u044F\u0442 \u043A\u0430\u043A " +
            "\u0441\u0442\u0430\u043D\u0434\u0430\u0440\u0442 \u0442\u0430\u043A\u0438\u043C\u0438 " +
            "\u043B\u0438\u0434\u0435\u0440\u0430\u043C\u0438 \u043A\u043E\u043C\u043F\u044C\u044E\u0442\u0435\u0440\u043D\u043E\u0439" +
            " \u0438\u043D\u0434\u0443\u0441\u0442\u0440\u0438\u0438, \u043A\u0430\u043A " +
            "Apple, HP, IBM, JustSystem, Microsoft, Oracle, SAP, Sun, Sybase, Unisys \u0434\u0430 \u0438 " +
            "\u043C\u043D\u043E\u0433\u0438\u043C\u0438 \u0434\u0440\u0443\u0433\u0438\u043C\u0438. " +
            "\u0418\u043C\u0435\u043D\u043D\u043E \u044D\u0442\u0430 \u0441\u0445\u0435\u043C\u0430 " +
            "\u043A\u043E\u0434\u0438\u0440\u043E\u0432\u0430\u043D\u0438\u044F " +
            "\u0438\u0441\u043F\u043E\u043B\u044C\u0437\u0443\u0435\u0442\u0441\u044F \u0442\u0430\u043A\u0438\u043C\u0438 " +
            "\u0441\u043E\u0432\u0440\u0435\u043C\u0435\u043D\u043D\u044B\u043C\u0438 " +
            "\u0442\u0435\u0445\u043D\u043E\u043B\u043E\u0433\u0438\u044F\u043C\u0438 \u0438 " +
            "\u0441\u0442\u0430\u043D\u0434\u0430\u0440\u0442\u0430\u043C\u0438, " +
            "\u043A\u0430\u043A \u043D\u0430\u043F\u0440\u0438\u043C\u0435\u0440 XML, Java, ECMAScript (JavaScript), LDAP, CORBA 3.0, " +
            "WML \u0438 \u0442\u0430\u043A \u0434\u0430\u043B\u0435\u0435. \u0418\u043C\u0435\u043D\u043D\u043E Unicode " +
            "\u044F\u0432\u043B\u044F\u0435\u0442\u0441\u044F \u043E\u0444\u0438\u0446\u0438\u0430\u043B\u044C\u043D\u043E\u0439 " +
            "\u0441\u0445\u0435\u043C\u043E\u0439 \u0440\u0435\u0430\u043B\u0438\u0437\u0430\u0446\u0438\u0438 ISO/IEC 10646. " +
            "\u041D\u0430\u043A\u043E\u043D\u0435\u0446, \u044D\u0442\u0430 \u043A\u043E\u0434\u0438\u0440\u043E\u0432\u043A\u0430 " +
            "\u043F\u043E\u0434\u0434\u0435\u0440\u0436\u0438\u0432\u0430\u0435\u0442\u0441\u044F \u0432\u043E " +
            "\u043C\u043D\u043E\u0436\u0435\u0441\u0442\u0432\u0435 " +
            "\u043E\u043F\u0435\u0440\u0430\u0446\u0438\u043E\u043D\u043D\u044B\u0445 \u0441\u0438\u0441\u0442\u0435\u043C, " +
            "\u0432\u043E \u0432\u0441\u0435\u0445 \u0441\u043E\u0432\u0440\u0435\u043C\u0435\u043D\u043D\u044B\u0445 " +
            "\u0431\u0440\u0430\u0443\u0437\u0435\u0440\u0430\u0445 \u0418\u043D\u0442\u0435\u0440\u043D\u0435\u0442 " +
            "\u0438 \u0432 \u0431\u043E\u043B\u044C\u0448\u043E\u043C \u043A\u043E\u043B\u0438\u0447\u0435\u0441\u0442\u0432\u0435 " +
            "\u0434\u0440\u0443\u0433\u0438\u0445 \u043F\u0440\u043E\u0433\u0440\u0430\u043C\u043C. " +
            "\u041F\u043E\u0432\u0441\u0435\u043C\u0435\u0441\u0442\u043D\u043E\u0435 " +
            "\u0440\u0430\u0441\u043F\u0440\u043E\u0441\u0442\u0440\u0430\u043D\u0435\u043D\u0438\u0435 " +
            "\u0441\u0442\u0430\u043D\u0434\u0430\u0440\u0442\u0430 Unicode \u0440\u043E\u0432\u043D\u043E " +
            "\u043A\u0430\u043A \u0438 \u0434\u043E\u0441\u0442\u0443\u043F\u043D\u043E\u0441\u0442\u044C " +
            "\u043F\u043E\u0434\u0434\u0435\u0440\u0436\u0438\u0432\u0430\u044E\u0449\u0438\u0445 \u0435\u0433\u043E " +
            "\u0441\u0440\u0435\u0434\u0441\u0442\u0432 \u0432 \u043D\u0430\u0441\u0442\u043E\u044F\u0449\u0435\u0435 " +
            "\u0432\u0440\u0435\u043C\u044F \u044F\u0432\u043B\u044F\u044E\u0442\u0441\u044F " +
            "\u043E\u0434\u043D\u0438\u043C\u0438 \u0438\u0437 \u043D\u0430\u0438\u0431\u043E\u043B\u0435\u0435 " +
            "\u0432\u0430\u0436\u043D\u044B\u0445 \u043D\u0430\u043F\u0440\u0430\u0432\u043B\u0435\u043D\u0438\u0439 " +
            "\u0440\u0430\u0437\u0432\u0438\u0442\u0438\u044F \u0438\u043D\u0434\u0443\u0441\u0442\u0440\u0438\u0438 " +
            "\u043F\u0440\u043E\u0433\u0440\u0430\u043C\u043C\u043D\u043E\u0433\u043E " +
            "\u043E\u0431\u0435\u0441\u043F\u0435\u0447\u0435\u043D\u0438\u044F. " +
            "\u0418\u0441\u043F\u043E\u043B\u044C\u0437\u043E\u0432\u0430\u043D\u0438\u0435 Unicode \u0432 " +
            "\u043C\u043D\u043E\u0433\u043E\u0443\u0440\u043E\u0432\u043D\u0435\u0432\u044B\u0445 " +
            "\u043F\u0440\u0438\u043B\u043E\u0436\u0435\u043D\u0438\u044F\u0445 \u0438\u043B\u0438 " +
            "\u043F\u0440\u043E\u0433\u0440\u0430\u043C\u043C\u043D\u044B\u0445 " +
            "\u043A\u043E\u043C\u043F\u043B\u0435\u043A\u0441\u0430\u0445, " +
            "\u043F\u043E\u0441\u0442\u0440\u043E\u0435\u043D\u043D\u044B\u0445 \u0432 \u0440\u0430\u043C\u043A\u0430\u0445 " +
            "\u0430\u0440\u0445\u0438\u0442\u0435\u043A\u0442\u0443\u0440\u044B " +
            "\u043A\u043B\u0438\u0435\u043D\u0442-\u0441\u0435\u0440\u0432\u0435\u0440, \u0430 " +
            "\u0442\u0430\u043A\u0436\u0435 \u043F\u0440\u0438 " +
            "\u043F\u0440\u0435\u0434\u0441\u0442\u0430\u0432\u043B\u0435\u043D\u0438\u0438 " +
            "\u0434\u0430\u043D\u043D\u044B\u0445 \u0432 \u0441\u0435\u0442\u0438 \u0418\u043D\u0442\u0435\u0440\u043D\u0435\u0442, " +
            "\u043F\u0440\u0438\u0432\u043E\u0434\u0438\u0442 \u043A " +
            "\u0437\u043D\u0430\u0447\u0438\u0442\u0435\u043B\u044C\u043D\u043E\u043C\u0443 " +
            "\u0441\u043D\u0438\u0436\u0435\u043D\u0438\u044E \u0440\u0430\u0441\u0445\u043E\u0434\u043E\u0432 " +
            "\u043D\u0430 \u043F\u043E\u0434\u0434\u0435\u0440\u0436\u043A\u0443 \u044D\u0442\u0438\u0445 " +
            "\u043F\u0440\u043E\u0434\u0443\u043A\u0442\u043E\u0432 \u0438\u043B\u0438 " +
            "\u0441\u0435\u0440\u0432\u0438\u0441\u043E\u0432 \u043F\u043E \u0441\u0440\u0430\u0432\u043D\u0435\u043D\u0438\u044E " +
            "\u0441\u043E \u0441\u043B\u0443\u0447\u0430\u0435\u043C " +
            "\u0438\u0441\u043F\u043E\u043B\u044C\u0437\u043E\u0432\u0430\u043D\u0438\u044F \u0441\u0442\u0430\u0440\u044B\u0445 " +
            "\u0441\u0445\u0435\u043C \u043A\u043E\u0434\u0438\u0440\u043E\u0432\u0430\u043D\u0438\u044F. " +
            "\u0414\u0435\u0439\u0441\u0442\u0432\u0438\u0442\u0435\u043B\u044C\u043D\u043E, Unicode " +
            "\u043F\u043E\u0437\u0432\u043E\u043B\u044F\u0435\u0442 \u0441\u043E\u0437\u0434\u0430\u0432\u0430\u0442\u044C " +
            "\u0435\u0434\u0438\u043D\u044B\u0439 \u043F\u0440\u043E\u0433\u0440\u0430\u043C\u043C\u043D\u044B\u0439 " +
            "\u043F\u0440\u043E\u0434\u0443\u043A\u0442 \u0438\u043B\u0438 \u0441\u0430\u0439\u0442 " +
            "\u0418\u043D\u0442\u0435\u0440\u043D\u0435\u0442 \u0434\u043B\u044F \u043C\u043D\u043E\u0436\u0435\u0441\u0442\u0432\u0430 " +
            "\u043F\u043B\u0430\u0442\u0444\u043E\u0440\u043C, \u044F\u0437\u044B\u043A\u043E\u0432 \u0438 " +
            "\u0441\u0442\u0440\u0430\u043D \u0431\u0435\u0437 \u043A\u0430\u043A\u0438\u0445-\u043B\u0438\u0431\u043E " +
            "\u043F\u0435\u0440\u0435\u0434\u0435\u043B\u043E\u043A. \u0410 \u0435\u0433\u043E " +
            "\u0438\u0441\u043F\u043E\u043B\u044C\u0437\u043E\u0432\u0430\u043D\u0438\u0435 \u043F\u0440\u0438 " +
            "\u043F\u0435\u0440\u0435\u0434\u0430\u0447\u0435 \u0434\u0430\u043D\u043D\u044B\u0445 \u043C\u0435\u0436\u0434\u0443 " +
            "\u0440\u0430\u0437\u043B\u0438\u0447\u043D\u044B\u043C\u0438 \u0441\u0438\u0441\u0442\u0435\u043C\u0430\u043C\u0438 " +
            "\u043F\u0440\u0435\u0434\u043E\u0445\u0440\u0430\u043D\u044F\u0435\u0442 \u044D\u0442\u0438 " +
            "\u0434\u0430\u043D\u043D\u044B\u0435 \u043E\u0442 \u043F\u043E\u0432\u0440\u0435\u0436\u0434\u0435",

            "\u0645\u0627 \u0647\u064A \u0627\u0644\u0634\u0641\u0631\u0629 \u0627\u0644\u0645\u0648\u062D\u062F\u0629 " +
            "\"\u064A\u0648\u0646\u0650\u0643\u0648\u062F\" \u061F\n\n\u0623\u0633\u0627\u0633\u064B\u0627\u060C " +
            "\u062A\u062A\u0639\u0627\u0645\u0644 \u0627\u0644\u062D\u0648\u0627\u0633\u064A\u0628 \u0641\u0642\u0637 \u0645\u0639 " +
            "\u0627\u0644\u0623\u0631\u0642\u0627\u0645\u060C \u0648\u062A\u0642\u0648\u0645 \u0628\u062A\u062E\u0632\u064A\u0646 " +
            "\u0627\u0644\u0623\u062D\u0631\u0641 \u0648\u0627\u0644\u0645\u062D\u0627\u0631\u0641 \u0627\u0644\u0623\u062E\u0631\u0649 " +
            "\u0628\u0639\u062F \u0623\u0646 \u062A\u064F\u0639\u0637\u064A \u0631\u0642\u0645\u0627 \u0645\u0639\u064A\u0646\u0627 " +
            "\u0644\u0643\u0644 \u0648\u0627\u062D\u062F \u0645\u0646\u0647\u0627. \u0648\u0642\u0628\u0644 " +
            "\u0627\u062E\u062A\u0631\u0627\u0639 \"\u064A\u0648\u0646\u0650\u0643\u0648\u062F\"\u060C \u0643\u0627\u0646 " +
            "\u0647\u0646\u0627\u0643 \u0645\u0626\u0627\u062A \u0627\u0644\u0623\u0646\u0638\u0645\u0629 " +
            "\u0644\u0644\u062A\u0634\u0641\u064A\u0631 \u0648\u062A\u062E\u0635\u064A\u0635 \u0647\u0630\u0647 " +
            "\u0627\u0644\u0623\u0631\u0642\u0627\u0645 \u0644\u0644\u0645\u062D\u0627\u0631\u0641\u060C \u0648\u0644\u0645 " +
            "\u064A\u0648\u062C\u062F \u0646\u0638\u0627\u0645 \u062A\u0634\u0641\u064A\u0631 \u0648\u0627\u062D\u062F " +
            "\u064A\u062D\u062A\u0648\u064A \u0639\u0644\u0649 \u062C\u0645\u064A\u0639 \u0627\u0644\u0645\u062D\u0627\u0631\u0641 " +
            "\u0627\u0644\u0636\u0631\u0648\u0631\u064A\u0629. \u0648\u0639\u0644\u0649 \u0633\u0628\u064A\u0644 " +
            "\u0627\u0644\u0645\u062B\u0627\u0644\u060C \u0641\u0625\u0646 \u0627\u0644\u0627\u062A\u062D\u0627\u062F " +
            "\u0627\u0644\u0623\u0648\u0631\u0648\u0628\u064A \u0644\u0648\u062D\u062F\u0647\u060C \u0627\u062D\u062A\u0648\u0649 " +
            "\u0627\u0644\u0639\u062F\u064A\u062F \u0645\u0646 \u0627\u0644\u0634\u0641\u0631\u0627\u062A " +
            "\u0627\u0644\u0645\u062E\u062A\u0644\u0641\u0629 \u0644\u064A\u063A\u0637\u064A \u062C\u0645\u064A\u0639 " +
            "\u0627\u0644\u0644\u063A\u0627\u062A \u0627\u0644\u0645\u0633\u062A\u062E\u062F\u0645\u0629 \u0641\u064A " +
            "\u0627\u0644\u0627\u062A\u062D\u0627\u062F. \u0648\u062D\u062A\u0649 \u0644\u0648 " +
            "\u0627\u0639\u062A\u0628\u0631\u0646\u0627 \u0644\u063A\u0629 \u0648\u0627\u062D\u062F\u0629\u060C " +
            "\u0643\u0627\u0644\u0644\u063A\u0629 \u0627\u0644\u0625\u0646\u062C\u0644\u064A\u0632\u064A\u0629\u060C " +
            "\u0641\u0625\u0646 \u062C\u062F\u0648\u0644 \u0634\u0641\u0631\u0629 \u0648\u0627\u062D\u062F \u0644\u0645 " +
            "\u064A\u0643\u0641 \u0644\u0627\u0633\u062A\u064A\u0639\u0627\u0628 \u062C\u0645\u064A\u0639 " +
            "\u0627\u0644\u0623\u062D\u0631\u0641 \u0648\u0639\u0644\u0627\u0645\u0627\u062A " +
            "\u0627\u0644\u062A\u0631\u0642\u064A\u0645 \u0648\u0627\u0644\u0631\u0645\u0648\u0632 " +
            "\u0627\u0644\u0641\u0646\u064A\u0629 \u0648\u0627\u0644\u0639\u0644\u0645\u064A\u0629 " +
            "\u0627\u0644\u0634\u0627\u0626\u0639\u0629 \u0627\u0644\u0627\u0633\u062A\u0639\u0645\u0627\u0644.\n\n" +
            " \n\n\u0648\u062A\u062C\u062F\u0631 \u0627\u0644\u0645\u0644\u0627\u062D\u0638\u0629 \u0623\u0646 " +
            "\u0623\u0646\u0638\u0645\u0629 \u0627\u0644\u062A\u0634\u0641\u064A\u0631 " +
            "\u0627\u0644\u0645\u062E\u062A\u0644\u0641\u0629 \u062A\u062A\u0639\u0627\u0631\u0636 \u0645\u0639 " +
            "\u0628\u0639\u0636\u0647\u0627 \u0627\u0644\u0628\u0639\u0636. \u0648\u0628\u0639\u0628\u0627\u0631\u0629 " +
            "\u0623\u062E\u0631\u0649\u060C \u064A\u0645\u0643\u0646 \u0623\u0646 \u064A\u0633\u062A\u062E\u062F\u0650\u0645 " +
            "\u062C\u062F\u0648\u0644\u064A \u0634\u0641\u0631\u0629 \u0646\u0641\u0633 \u0627\u0644\u0631\u0642\u0645 " +
            "\u0644\u062A\u0645\u062B\u064A\u0644 \u0645\u062D\u0631\u0641\u064A\u0646 " +
            "\u0645\u062E\u062A\u0644\u0641\u064A\u0646\u060C \u0623\u0648 \u0631\u0642\u0645\u064A\u0646 " +
            "\u0645\u062E\u062A\u0644\u0641\u064A\u0646 \u0644\u062A\u0645\u062B\u064A\u0644 \u0646\u0641\u0633 " +
            "\u0627\u0644\u0645\u062D\u0631\u0641. \u0648\u0644\u0648 \u0623\u062E\u0630\u0646\u0627 \u0623\u064A " +
            "\u062C\u0647\u0627\u0632 \u062D\u0627\u0633\u0648\u0628\u060C \u0648\u0628\u062E\u0627\u0635\u0629 " +
            "\u062C\u0647\u0627\u0632 \u0627\u0644\u0646\u0627\u062F\u0644 (server)\u060C \u0641\u064A\u062C\u0628 " +
            "\u0623\u0646 \u062A\u0643\u0648\u0646 \u0644\u062F\u064A\u0647 \u0627\u0644\u0642\u062F\u0631\u0629 " +
            "\u0639\u0644\u0649 \u0627\u0644\u062A\u0639\u0627\u0645\u0644 \u0645\u0639 \u0639\u062F\u062F " +
            "\u0643\u0628\u064A\u0631 \u0645\u0646 \u0627\u0644\u0634\u0641\u0631\u0627\u062A " +
            "\u0627\u0644\u0645\u062E\u062A\u0644\u0641\u0629\u060C \u0648\u064A\u062A\u0645 " +
            "\u062A\u0635\u0645\u064A\u0645\u0647 \u0639\u0644\u0649 \u0647\u0630\u0627 \u0627\u0644\u0623\u0633\u0627\u0633. " +
            "\u0648\u0645\u0639 \u0630\u0644\u0643\u060C \u0641\u0639\u0646\u062F\u0645\u0627 \u062A\u0645\u0631 " +
            "\u0627\u0644\u0628\u064A\u0627\u0646\u0627\u062A \u0639\u0628\u0631 \u0623\u0646\u0638\u0645\u0629 " +
            "\u0645\u062E\u062A\u0644\u0641\u0629\u060C \u062A\u0648\u062C\u062F \u0647\u0646\u0627\u0643 " +
            "\u062E\u0637\u0648\u0631\u0629 \u0644\u0636\u064A\u0627\u0639 \u0623\u0648 \u062A\u062D\u0631\u064A\u0641 " +
            "\u0628\u0639\u0636 \u0647\u0630\u0647 \u0627\u0644\u0628\u064A\u0627\u0646\u0627\u062A.\n\n " +
            "\n\n\"\u064A\u0648\u0646\u0650\u0643\u0648\u062F\" \u062A\u063A\u064A\u0631 \u0647\u0630\u0627 " +
            "\u0643\u0644\u064A\u0640\u0627 !\n\n\u062A\u062E\u0635\u0635 \u0627\u0644\u0634\u0641\u0631\u0629 " +
            "\u0627\u0644\u0645\u0648\u062D\u062F\u0629 \"\u064A\u0648\u0646\u0650\u0643\u0648\u062F\" " +
            "\u0631\u0642\u0645\u0627 \u0648\u062D\u064A\u062F\u0627 \u0644\u0643\u0644 \u0645\u062D\u0631\u0641 " +
            "\u0641\u064A \u062C\u0645\u064A\u0639 \u0627\u0644\u0644\u063A\u0627\u062A " +
            "\u0627\u0644\u0639\u0627\u0644\u0645\u064A\u0629\u060C \u0648\u0630\u0644\u0643 \u0628\u063A\u0636 " +
            "\u0627\u0644\u0646\u0638\u0631 \u0639\u0646 \u0646\u0648\u0639 \u0627\u0644\u062D\u0627\u0633\u0648\u0628 " +
            "\u0623\u0648 \u0627\u0644\u0628\u0631\u0627\u0645\u062C \u0627\u0644\u0645\u0633\u062A\u062E\u062F\u0645\u0629. " +
            "\u0648\u0642\u062F \u062A\u0640\u0645 \u062A\u0628\u0646\u064A \u0645\u0648\u0627\u0635\u0641\u0629 " +
            "\"\u064A\u0648\u0646\u0650\u0643\u0648\u062F\" \u0645\u0640\u0640\u0646 \u0642\u0628\u0640\u0644 " +
            "\u0642\u0627\u062F\u0629 \u0627\u0644\u0635\u0627\u0646\u0639\u064A\u0646 \u0644\u0623\u0646\u0638\u0645\u0629 " +
            "\u0627\u0644\u062D\u0648\u0627\u0633\u064A\u0628 \u0641\u0640\u064A \u0627\u0644\u0639\u0627\u0644\u0645\u060C " +
            "\u0645\u062B\u0644 \u0634\u0631\u0643\u0627\u062A \u0622\u064A.\u0628\u064A.\u0625\u0645. " +
            "(IBM)\u060C \u0623\u0628\u0640\u0644 (APPLE)\u060C \u0647\u0650\u064A\u0640\u0652\u0648\u0644\u0650\u0640\u062A " +
            "\u0628\u0640\u0627\u0643\u0640\u0631\u062F (Hewlett-Packard) \u060C " +
            "\u0645\u0627\u064A\u0643\u0631\u0648\u0633\u0648\u0641\u062A (Microsoft)\u060C " +
            "\u0623\u0648\u0631\u0627\u0643\u0650\u0640\u0644 (Oracle) \u060C \u0635\u0646 (Sun) " +
            "\u0648\u063A\u064A\u0631\u0647\u0627. \u0643\u0645\u0627 " +
            "\u0623\u0646 \u0627\u0644\u0645\u0648\u0627\u0635\u0641\u0627\u062A " +
            "\u0648\u0627\u0644\u0645\u0642\u0627\u064A\u064A\u0633 \u0627\u0644\u062D\u062F\u064A\u062B\u0629 " +
            "(\u0645\u062B\u0644 \u0644\u063A\u0629 \u0627\u0644\u0628\u0631\u0645\u062C\u0629 \"\u062C\u0627\u0641\u0627\" " +
            "\"JAVA\" \u0648\u0644\u063A\u0629 \"\u0625\u0643\u0633 \u0625\u0645 \u0625\u0644\" \"XML\" \u0627\u0644\u062A\u064A " +
            "\u062A\u0633\u062A\u062E\u062F\u0645 \u0644\u0628\u0631\u0645\u062C\u0629 " +
            "\u0627\u0644\u0627\u0646\u062A\u0631\u0646\u064A\u062A) \u062A\u062A\u0637\u0644\u0628 " +
            "\u0627\u0633\u062A\u062E\u062F\u0627\u0645 \"\u064A\u0648\u0646\u0650\u0643\u0648\u062F\". " +
            "\u0639\u0644\u0627\u0648\u0629 \u0639\u0644\u0649 \u0630\u0644\u0643 \u060C \u0641\u0625\u0646 " +
            "\"\u064A\u0648\u0646\u0650\u0643\u0648\u062F\" \u0647\u064A \u0627\u0644\u0637\u0640\u0631\u064A\u0640\u0642\u0640\u0629 " +
            "\u0627\u0644\u0631\u0633\u0640\u0645\u064A\u0629 \u0644\u062A\u0637\u0628\u064A\u0642 " +
            "\u0627\u0644\u0645\u0642\u064A\u0640\u0627\u0633 \u0627\u0644\u0640\u0639\u0640\u0627\u0644\u0640\u0645\u064A " +
            "\u0625\u064A\u0632\u0648 \u0661\u0660\u0666\u0664\u0666  (ISO 10646) .\n\n \n\n\u0625\u0646 \u0628\u0632\u0648\u063A " +
            "\u0645\u0648\u0627\u0635\u0641\u0629 \"\u064A\u0648\u0646\u0650\u0643\u0648\u062F\" " +
            "\u0648\u062A\u0648\u0641\u0651\u064F\u0631 \u0627\u0644\u0623\u0646\u0638\u0645\u0629 \u0627\u0644\u062A\u064A " +
            "\u062A\u0633\u062A\u062E\u062F\u0645\u0647 \u0648\u062A\u062F\u0639\u0645\u0647\u060C \u064A\u0639\u062A\u0628\u0631 " +
            "\u0645\u0646 \u0623\u0647\u0645 \u0627\u0644\u0627\u062E\u062A\u0631\u0627\u0639\u0627\u062A " +
            "\u0627\u0644\u062D\u062F\u064A\u062B\u0629 \u0641\u064A \u0639\u0648\u0644\u0645\u0629 " +
            "\u0627\u0644\u0628\u0631\u0645\u062C\u064A\u0627\u062A \u0644\u062C\u0645\u064A\u0639 " +
            "\u0627\u0644\u0644\u063A\u0627\u062A \u0641\u064A \u0627\u0644\u0639\u0627\u0644\u0645. " +
            "\u0648\u0625\u0646 \u0627\u0633\u062A\u062E\u062F\u0627\u0645 \"\u064A\u0648\u0646\u0650\u0643\u0648\u062F\" " +
            "\u0641\u064A \u0639\u0627\u0644\u0645 \u0627\u0644\u0627\u0646\u062A\u0631\u0646\u064A\u062A " +
            "\u0633\u064A\u0624\u062F\u064A \u0625\u0644\u0649 \u062A\u0648\u0641\u064A\u0631 \u0643\u0628\u064A\u0631 " +
            "\u0645\u0642\u0627\u0631\u0646\u0629 \u0645\u0639 \u0627\u0633\u062A\u062E\u062F\u0627\u0645 " +
            "\u0627\u0644\u0645\u062C\u0645\u0648\u0639\u0627\u062A \u0627\u0644\u062A\u0642\u0644\u064A\u062F\u064A\u0629 " +
            "\u0644\u0644\u0645\u062D\u0627\u0631\u0641 \u0627\u0644\u0645\u0634\u0641\u0631\u0629. \u0643\u0645\u0627 " +
            "\u0623\u0646 \u0627\u0633\u062A\u062E\u062F\u0627\u0645 \"\u064A\u0648\u0646\u0650\u0643\u0648\u062F\" " +
            "\u0633\u064A\u064F\u0645\u0643\u0651\u0650\u0646 \u0627\u0644\u0645\u0628\u0631\u0645\u062C \u0645\u0646 " +
            "\u0643\u062A\u0627\u0628\u0629 \u0627\u0644\u0628\u0631\u0646\u0627\u0645\u062C \u0645\u0631\u0629 " +
            "\u0648\u0627\u062D\u062F\u0629\u060C \u0648\u0627\u0633\u062A\u062E\u062F\u0627\u0645\u0647 \u0639\u0644\u0649 " +
            "\u0623\u064A \u0646\u0648\u0639 \u0645\u0646 \u0627\u0644\u0623\u062C\u0647\u0632\u0629",

            "\u4EC0\u9EBD\u662FUnicode(\u7D71\u4E00\u78BC/\u6A19\u6E96\u842C\u570B\u78BC)?\n\n" +
            "Unicode\u7D66\u6BCF\u500B\u5B57\u5143\u63D0\u4F9B\u4E86\u4E00\u500B\u552F\u4E00\u7684\u6578\u4F4D\uFF0C\n" +
            "\u4E0D\u8AD6\u662F\u4EC0\u9EBD\u5E73\u81FA\uFF0C\n\u4E0D\u8AD6\u662F\u4EC0\u9EBD\u7A0B\u5F0F\uFF0C\n" +
            "\u4E0D\u8AD6\u662F\u4EC0\u9EBD\u8A9E\u8A00\u3002\n\n" +
            "\u57FA\u672C\u4E0A\uFF0C\u96FB\u8166\u53EA\u662F\u8655\u7406\u6578\u4F4D\u3002\u5B83\u5011\u6307\u5B9A" +
            "\u4E00\u500B\u6578\u4F4D\uFF0C\u4F86\u5132\u5B58\u5B57\u6BCD\u6216\u5176\u4ED6\u5B57\u5143\u3002\u5728\u5275\u9020" +
            "Unicode\u4E4B\u524D\uFF0C\u6709\u6578\u767E\u7A2E\u6307\u5B9A\u9019\u4E9B\u6578\u4F4D\u7684\u7DE8\u78BC" +
            "\u7CFB\u7D71\u3002\u6C92\u6709\u4E00\u500B\u7DE8\u78BC\u53EF\u4EE5\u5305\u542B\u8DB3\u5920\u7684\u5B57" +
            "\u5143\uFF1A\u4F8B\u5982\uFF0C\u55AE\u55AE\u6B50\u5DDE\u5171\u540C\u9AD4\u5C31\u9700\u8981\u597D\u5E7E" +
            "\u7A2E\u4E0D\u540C\u7684\u7DE8\u78BC\u4F86\u5305\u62EC\u6240\u6709\u7684\u8A9E\u8A00\u3002\u5373\u4F7F" +
            "\u662F\u55AE\u4E00\u7A2E\u8A9E\u8A00\uFF0C\u4F8B\u5982\u82F1\u8A9E\uFF0C\u4E5F\u6C92\u6709\u54EA\u4E00" +
            "\u500B\u7DE8\u78BC\u53EF\u4EE5\u9069\u7528\u65BC\u6240\u6709\u7684\u5B57\u6BCD\uFF0C\u6A19\u9EDE\u7B26" +
            "\u865F\uFF0C\u548C\u5E38\u7528\u7684\u6280\u8853\u7B26\u865F\u3002\n\n\u9019\u4E9B\u7DE8\u78BC\u7CFB" +
            "\u7D71\u4E5F\u6703\u4E92\u76F8\u885D\u7A81\u3002\u4E5F\u5C31\u662F\u8AAA\uFF0C\u5169\u7A2E\u7DE8\u78BC" +
            "\u53EF\u80FD\u4F7F\u7528\u76F8\u540C\u7684\u6578\u4F4D\u4EE3\u8868\u5169\u500B\u4E0D\u540C\u7684\u5B57" +
            "\u5143\uFF0C\u6216\u4F7F\u7528\u4E0D\u540C\u7684\u6578\u4F4D\u4EE3\u8868\u76F8\u540C\u7684\u5B57\u5143" +
            "\u3002\u4EFB\u4F55\u4E00\u53F0\u7279\u5B9A\u7684\u96FB\u8166 (\u7279\u5225\u662F\u4F3A\u670D\u5668\u90FD" +
            "\u9700\u8981\u652F\u63F4\u8A31\u591A\u4E0D\u540C\u7684\u7DE8\u78BC\uFF0C\u4F46\u662F\uFF0C\u4E0D\u8AD6" +
            "\u4EC0\u9EBD\u6642\u5019\u8CC7\u6599\u901A\u904E\u4E0D\u540C\u7684\u7DE8\u78BC\u6216\u5E73\u81FA\u4E4B" +
            "\u9593\uFF0C\u90A3\u4E9B\u8CC7\u6599\u7E3D\u6703\u6709\u640D\u58DE\u7684\u5371\u96AA\u3002\nUnicode" +
            "\u6B63\u5728\u6539\u8B8A\u6240\u6709\u9019\u4E00\u5207\uFF01\n\nUnicode\u7D66\u6BCF\u500B\u5B57\u5143" +
            "\u63D0\u4F9B\u4E86\u4E00\u500B\u552F\u4E00\u7684\u6578\u4F4D\uFF0C\u4E0D\u8AD6\u662F\u4EC0\u9EBD\u5E73" +
            "\u81FA\uFF0C\u4E0D\u8AD6\u662F\u4EC0\u9EBD\u7A0B\u5F0F\uFF0C\u4E0D\u8AD6\u4EC0\u9EBD\u8A9E\u8A00\u3002" +
            "Unicode\u6A19\u6E96\u5DF2\u7D93\u88AB\u9019\u4E9B\u5DE5\u696D\u754C\u7684\u9818\u5C0E\u5011\u6240\u63A1" +
            "\u7528\uFF0C\u4F8B\u5982\uFF1AApple, HP, IBM, JustSystem, Microsoft, Oracle, SAP, Sun, Sybase, Unisys" +
            " \u548C\u5176\u4ED6\u8A31\u591A\u516C\u53F8\u3002\u6700\u65B0\u7684\u6A19\u6E96\u90FD\u9700\u8981Unicode" +
            "\uFF0C\u4F8B\u5982XML, Java, ECMAScript (JavaScript), LDAP, CORBA 3.0, WML\u7B49\u7B49\uFF0C\u4E26\u4E14" +
            "\uFF0CUnicode\u662F\u5BE6\u73FEISO/IEC 10646\u7684\u6B63\u898F\u65B9\u5F0F\u3002\u8A31\u591A\u4F5C\u696D" +
            "\u7CFB\u7D71\uFF0C\u6240\u6709\u6700\u65B0\u7684\u700F\u89BD\u5668\u548C \u8A31\u591A\u5176\u4ED6\u7523" +
            "\u54C1\u90FD\u652F\u63F4\u5B83\u3002Unicode\u6A19\u6E96\u7684\u51FA\u73FE\u548C\u652F\u63F4\u5B83\u5DE5" +
            "\u5177\u7684\u5B58\u5728\uFF0C\u662F\u8FD1\u4F86\u5168\u7403\u8EDF\u9AD4\u6280\u8853\u6700\u91CD\u8981" +
            "\u7684\u767C\u5C55\u8DA8\u52E2\u3002\n\n\u5C07Unicode\u8207\u5BA2\u6236\u4F3A\u670D\u5668\u6216\u591A" +
            "\u5C64\u61C9\u7528\u7A0B\u5F0F\u548C\u7DB2\u969B\u7DB2\u8DEF\u7D50\u5408\uFF0C\u6BD4\u4F7F\u7528\u50B3" +
            "\u7D71\u5B57\u5143\u96C6\u7BC0\u7701\u8CBB\u7528\u3002Unicode\u4F7F\u55AE\u4E00\u8EDF\u9AD4\u7523\u54C1" +
            "\u6216\u55AE\u4E00\u7DB2\u7AD9\u80FD\u5920\u8CAB\u7A7F\u591A\u500B\u5E73\u81FA\uFF0C\u8A9E\u8A00\u548C" +
            "\u570B\u5BB6\uFF0C\u800C\u4E0D\u9700\u8981\u91CD\u5EFA\u3002\u5B83\u53EF\u5C07\u8CC7\u6599\u50B3\u8F38" +
            "\u5230\u8A31\u591A\u4E0D\u540C\u7684\u7CFB\u7D71\uFF0C\u800C\u7121\u640D\u58DE\u3002\n\u95DC\u65BCUnicode" +
            "\u5B78\u8853\u5B78\u6703\n\nUnicode\u5B78\u8853\u5B78\u6703",

            "\u00C7\'\u00EBsht\u00EB UNICODE?\n\nUnicode siguron nj\u00EB num\u00EBr t\u00EB vet\u00EBm p\u00EBr \u00E7do " +
            "g\u00EBrm\u00EB,\np\u00EBr cil\u00EBndo platform\u00EB,\np\u00EBr cilindo program,\np\u00EBr cil\u00EBndo " +
            "gjuh\u00EB.\n\nN\u00EB themel, kompjuterat veprojn\u00EB me an\u00EBn e numrave. Ata ruajn\u00EB g\u00EBrmat " +
            "dhe sh\u00EBnjat (karakteret) e tjera duke u caktuar nga nj\u00EB num\u00EBr (kod). Para zbulimit t\u00EB " +
            "Unicode-s, kishte qindra sisteme kodimi t\u00EB ndrysh\u00EBm p\u00EBr t\u00EB caktuar k\u00EBta numra. Por " +
            "asnj\u00EB kodim i vet\u00EBm nuk mund t\u00EB p\u00EBrmbante mjaft sh\u00EBnja: p\u00EBr shembull, vet\u00EBm " +
            "Bashkimi Europian k\u00EBrkon disa kodime t\u00EB ndryshme p\u00EBr t\u00EB mbuluar gjith\u00EB gjuh\u00EBt e tij. " +
            "Edhe p\u00EBr nj\u00EB gjuh\u00EB t\u00EB vetme si Anglishtja asnj\u00EB kodim nuk ishte i mjaftuesh\u00EBm " +
            "p\u00EBr t\u00EB gjitha g\u00EBrmat, sh\u00EBnjat e pik\u00EBsimit, dhe simbolet teknike n\u00EB p\u00EBrdorim " +
            "t\u00EB zakonsh\u00EBm.\n\nK\u00EBta sisteme kodimi gjithashtu p\u00EBrplasen me nj\u00EBri-tjetrin. Dmth, dy " +
            "kodime mundet q\u00EB t\u00EB p\u00EBrdorin t\u00EB nj\u00EBjtin num\u00EBr p\u00EBr dy sh\u00EBnja krejt t\u00EB " +
            "ndryshme, ose dy numra t\u00EB ndrysh\u00EBm p\u00EBr t\u00EB nj\u00EBjt\u00EBn sh\u00EBnj\u00EB. Secili kompjuter " +
            "(sidomos sh\u00EBrbyesit - serverat) duhet t\u00EB jet\u00EB i aft\u00EB t\u00EB mb\u00EBshtes\u00EB shum\u00EB " +
            "kodime, dhe megjithat\u00EB kur t\u00EB dh\u00EBnat kalohen p\u00EBrmes kodimeve t\u00EB ndryshme ose platformave, " +
            "\u00EBsht\u00EB gjithmon\u00EB rreziku i korruptimit t\u00EB tyre.\nUnicode po e ndryshon gjith\u00EB k\u00EBt\u00EB " +
            "rr\u00EBmuj\u00EB!\n\nUnicode siguron nj\u00EB num\u00EBr t\u00EB vet\u00EBm p\u00EBr \u00E7do g\u00EBrm\u00EB, " +
            "p\u00EBr cil\u00EBndo platform\u00EB, p\u00EBr cilindo program, p\u00EBr cil\u00EBndo gjuh\u00EB. Standarti Unicode " +
            "\u00EBsht\u00EB f\u00EBmij\u00EBruar (adaptuar) nga udh\u00EBheq\u00EBs t\u00EB till\u00EB t\u00EB industris\u00EB " +
            "si Apple, HP, IBM, JustSystem, Microsoft, Oracle, SAP, Sun, Sybase, Unisys dhe shum\u00EB t\u00EB tjer\u00EB. " +
            "Unicode k\u00EBrkohet nga standarte moderne si XML, Java, ECMAScript (JavaScript), LDAP, CORBA 3.0, WML, etj., " +
            "dhe \u00EBsht\u00EB m\u00EBnyra zyrtare p\u00EBr t\u00EB zbatuar ISO/IEC 10646. Unicode \u00EBsht\u00EB i " +
            "mb\u00EBshtetur nga shum\u00EB sisteme vepruese (operativ\u00EB), gjith\u00EB shfletuesat (brauzerat) modern\u00EB, " +
            "dhe shum\u00EB produkte t\u00EB tjera. Dalja n\u00EB drit\u00EB e Standartit Unicode, dhe pasja e veglave q\u00EB " +
            "e mb\u00EBshtesin, jan\u00EB midis faktor\u00EBve m\u00EB dometh\u00EBn\u00EBs t\u00EB koh\u00EBve t\u00EB fundit " +
            "t\u00EB drejtimeve t\u00EB zhvillimit t\u00EB p\u00EBrgjith\u00EBshme t\u00EB teknologjive t\u00EB softuer\u00EBve." +
            "\n\nP\u00EBrtrup\u00EBzimi i Unicode n\u00EB zbatimet klient-server apo shum\u00EB-shkall\u00EBsh si dhe n\u00EB " +
            "faqet internet, krijon mund\u00EBsin\u00EB p\u00EBr kursime kostoje t\u00EB ndjeshme n\u00EB krahasim me " +
            "p\u00EBrdorimin e kodimeve t\u00EB m\u00EBparshme. Unicode b\u00EBn t\u00EB mundur q\u00EB nj\u00EB produkt " +
            "softuer ose nj\u00EB faqe interneti t\u00EB p\u00EBrdoret p\u00EBr shum\u00EB platforma, gjuh\u00EB dhe vende " +
            "pa re-inxhinierim. Ai lejon t\u00EB dh\u00EBnat t\u00EB kalohen p\u00EBrmes shum\u00EB sistemeve t\u00EB ndryshme " +
            "pa korruptim.",

            "\u092F\u0942\u0928\u093F\u0915\u094B\u0921 \u0915\u094D\u092F\u093E \u0939\u0948?\n\n\u092F\u0942\u0928\u093F\u0915" +
            "\u094B\u0921 \u092A\u094D\u0930\u0924\u094D\u092F\u0947\u0915 \u0905\u0915\u094D\u0937\u0930 \u0915\u0947 " +
            "\u0932\u093F\u090F \u090F\u0915 \u0935\u093F\u0936\u0947\u0937 \u0928\u092E\u094D\u092C\u0930 \u092A\u094D" +
            "\u0930\u0926\u093E\u0928 \u0915\u0930\u0924\u093E \u0939\u0948,\n\u091A\u093E\u0939\u0947 \u0915\u094B\u0908 " +
            "\u092D\u0940 \u092A\u094D\u0932\u0948\u091F\u092B\u0949\u0930\u094D\u092E \u0939\u094B,\n\u091A\u093E\u0939\u0947" +
            " \u0915\u094B\u0908 \u092D\u0940 \u092A\u094D\u0930\u094B\u0917\u094D\u0930\u093E\u092E \u0939\u094B," +
            "\n\u091A\u093E\u0939\u0947 \u0915\u094B\u0908 \u092D\u0940 \u092D\u093E\u0937\u093E \u0939\u094B\u0964\n" +
            "\n\u0915\u092E\u094D\u092A\u094D\u092F\u0942\u091F\u0930, \u092E\u0942\u0932 \u0930\u0942\u092A \u0938\u0947," +
            " \u0928\u0902\u092C\u0930\u094B\u0902 \u0938\u0947 \u0938\u092E\u094D\u092C\u0902\u0927 \u0930\u0916\u0924\u0947" +
            " \u0939\u0948\u0902\u0964 \u092F\u0947 \u092A\u094D\u0930\u0924\u094D\u092F\u0947\u0915 \u0905\u0915\u094D\u0937\u0930" +
            " \u0914\u0930 \u0935\u0930\u094D\u0923 \u0915\u0947 \u0932\u093F\u090F \u090F\u0915 \u0928\u0902\u092C\u0930 " +
            "\u0928\u093F\u0930\u094D\u0927\u093E\u0930\u093F\u0924 \u0915\u0930\u0915\u0947 \u0905\u0915\u094D\u0937\u0930" +
            " \u0914\u0930 \u0935\u0930\u094D\u0923 \u0938\u0902\u0917\u094D\u0930\u0939\u093F\u0924 \u0915\u0930\u0924\u0947" +
            " \u0939\u0948\u0902\u0964 \u092F\u0942\u0928\u093F\u0915\u094B\u0921 \u0915\u093E \u0906\u0935\u093F\u0937" +
            "\u094D\u0915\u093E\u0930 \u0939\u094B\u0928\u0947 \u0938\u0947 \u092A\u0939\u0932\u0947, \u0910\u0938\u0947" +
            " \u0928\u0902\u092C\u0930 \u0926\u0947\u0928\u0947 \u0915\u0947 \u0932\u093F\u090F \u0938\u0948\u0902\u0915" +
            "\u0921\u094B\u0902 \u0935\u093F\u092D\u093F\u0928\u094D\u0928 \u0938\u0902\u0915\u0947\u0924 \u0932\u093F" +
            "\u092A\u093F \u092A\u094D\u0930\u0923\u093E\u0932\u093F\u092F\u093E\u0902 \u0925\u0940\u0902\u0964 \u0915" +
            "\u093F\u0938\u0940 \u090F\u0915 \u0938\u0902\u0915\u0947\u0924 \u0932\u093F\u092A\u093F \u092E\u0947\u0902 " +
            "\u092A\u0930\u094D\u092F\u093E\u092A\u094D\u0924 \u0905\u0915\u094D\u0937\u0930 \u0928\u0939\u0940\u0902 " +
            "\u0939\u094B \u0938\u0915\u0924\u0947 \u0939\u0948\u0902 : \u0909\u0926\u093E\u0939\u0930\u0923 \u0915\u0947" +
            " \u0932\u093F\u090F, \u092F\u0942\u0930\u094B\u092A\u093F\u092F \u0938\u0902\u0918 \u0915\u094B \u0905\u0915" +
            "\u0947\u0932\u0947 \u0939\u0940, \u0905\u092A\u0928\u0940 \u0938\u092D\u0940 \u092D\u093E\u0937\u093E\u0912\u0902" +
            " \u0915\u094B \u0915\u0935\u0930 \u0915\u0930\u0928\u0947 \u0915\u0947 \u0932\u093F\u090F \u0905\u0928\u0947\u0915" +
            " \u0935\u093F\u092D\u093F\u0928\u094D\u0928 \u0938\u0902\u0915\u0947\u0924 \u0932\u093F\u092A\u093F\u092F\u094B\u0902" +
            " \u0915\u0940 \u0906\u0935\u0936\u094D\u092F\u0915\u0924\u093E \u0939\u094B\u0924\u0940 \u0939\u0948\u0964 " +
            "\u0905\u0902\u0917\u094D\u0930\u0947\u091C\u0940 \u091C\u0948\u0938\u0940 \u092D\u093E\u0937\u093E \u0915\u0947" +
            " \u0932\u093F\u090F \u092D\u0940, \u0938\u092D\u0940 \u0905\u0915\u094D\u0937\u0930\u094B\u0902, " +
            "\u0935\u093F\u0930\u093E\u092E\u091A\u093F\u0928\u094D\u0939\u094B\u0902 \u0914\u0930 \u0938\u093E\u092E" +
            "\u093E\u0928\u094D\u092F \u092A\u094D\u0930\u092F\u094B\u0917 \u0915\u0947 \u0924\u0915\u0928\u0940\u0915" +
            "\u0940 \u092A\u094D\u0930\u0924\u0940\u0915\u094B\u0902 \u0939\u0947\u0924\u0941 \u090F\u0915 \u0939\u0940" +
            " \u0938\u0902\u0915\u0947\u0924 \u0932\u093F\u092A\u093F \u092A\u0930\u094D\u092F\u093E\u092A\u094D\u0924" +
            " \u0928\u0939\u0940\u0902 \u0925\u0940\u0964\n\n\u092F\u0947 \u0938\u0902\u0915\u0947\u0924 \u0932\u093F" +
            "\u092A\u093F \u092A\u094D\u0930\u0923\u093E\u0932\u093F\u092F\u093E\u0902 \u092A\u0930\u0938\u094D\u092A\u0930" +
            " \u0935\u093F\u0930\u094B\u0927\u0940 \u092D\u0940 \u0939\u0948\u0902\u0964 \u0907\u0938\u0940\u0932\u093F\u090F," +
            " \u0926\u094B \u0938\u0902\u0915\u0947\u0924 \u0932\u093F\u092A\u093F\u092F\u093E\u0902 \u0926\u094B " +
            "\u0935\u093F\u092D\u093F\u0928\u094D\u0928 \u0905\u0915\u094D\u0937\u0930\u094B\u0902 \u0915\u0947 " +
            "\u0932\u093F\u090F, \u090F\u0915 \u0939\u0940 \u0928\u0902\u092C\u0930 \u092A\u094D\u0930\u092F\u094B\u0917" +
            " \u0915\u0930 \u0938\u0915\u0924\u0940 \u0939\u0948\u0902, \u0905\u0925\u0935\u093E \u0938\u092E\u093E\u0928" +
            " \u0905\u0915\u094D\u0937\u0930 \u0915\u0947 \u0932\u093F\u090F \u0935\u093F\u092D\u093F\u0928\u094D\u0928" +
            " \u0928\u092E\u094D\u092C\u0930\u094B\u0902 \u0915\u093E \u092A\u094D\u0930\u092F\u094B\u0917 \u0915\u0930" +
            " \u0938\u0915\u0924\u0940 \u0939\u0948\u0902\u0964 \u0915\u093F\u0938\u0940 \u092D\u0940 \u0915\u092E\u094D" +
            "\u092A\u094D\u092F\u0942\u091F\u0930 (\u0935\u093F\u0936\u0947\u0937 \u0930\u0942\u092A \u0938\u0947 " +
            "\u0938\u0930\u094D\u0935\u0930) \u0915\u094B \u0935\u093F\u092D\u093F\u0928\u094D\u0928 \u0938\u0902\u0915" +
            "\u0947\u0924 \u0932\u093F\u092A\u093F\u092F\u093E\u0902 \u0938\u0902\u092D\u093E\u0932\u0928\u0940 \u092A" +
            "\u0921\u093C\u0924\u0940 \u0939\u0948; \u092B\u093F\u0930 \u092D\u0940 \u091C\u092C \u0926\u094B \u0935" +
            "\u093F\u092D\u093F\u0928\u094D\u0928 \u0938\u0902\u0915\u0947\u0924 \u0932\u093F\u092A\u093F\u092F\u094B" +
            "\u0902 \u0905\u0925\u0935\u093E \u092A\u094D\u0932\u0948\u091F\u092B\u0949\u0930\u094D\u092E\u094B\u0902" +
            " \u0915\u0947 \u092C\u0940\u091A \u0921\u093E\u091F\u093E \u092D\u0947\u091C\u093E \u091C\u093E\u0924\u093E" +
            " \u0939\u0948 \u0924\u094B \u0909\u0938 \u0921\u093E\u091F\u093E \u0915\u0947 \u0939\u092E\u0947\u0936\u093E" +
            " \u0916\u0930\u093E\u092C \u0939\u094B\u0928\u0947 \u0915\u093E \u091C\u094B\u0916\u093F\u092E " +
            "\u0930\u0939\u0924\u093E \u0939\u0948\u0964\n\u092F\u0942\u0928\u093F\u0915\u094B\u0921 \u0938\u0947" +
            " \u092F\u0939 \u0938\u092C \u0915\u0941\u091B \u092C\u0926\u0932 \u0930\u0939\u093E \u0939\u0948!\n" +
            "\n\u092F\u0942\u0928\u093F\u0915\u094B\u0921, \u092A\u094D\u0930\u0924\u094D\u092F\u0947\u0915 " +
            "\u0905\u0915\u094D\u0937\u0930 \u0915\u0947 \u0932\u093F\u090F \u090F\u0915 \u0935\u093F\u0936\u0947\u0937" +
            " \u0928\u0902\u092C\u0930 \u092A\u094D\u0930\u0926\u093E\u0928 \u0915\u0930\u0924\u093E \u0939\u0948, " +
            "\u091A\u093E\u0939\u0947 \u0915\u094B\u0908 \u092D\u0940 \u092A\u094D\u0932\u0948\u091F\u092B\u0949\u0930\u094D\u092E" +
            " \u0939\u094B, \u091A\u093E\u0939\u0947 \u0915\u094B\u0908 \u092D\u0940 \u092A\u094D\u0930\u094B\u0917" +
            "\u094D\u0930\u093E\u092E \u0939\u094B, \u091A\u093E\u0939\u0947 \u0915\u094B\u0908 \u092D\u0940 \u092D" +
            "\u093E\u0937\u093E \u0939\u094B\u0964 \u092F\u0942\u0928\u093F\u0915\u094B\u0921 \u0938\u094D\u091F\u0948" +
            "\u0902\u0921\u0930\u094D\u0921 \u0915\u094B \u0910\u092A\u0932, \u090F\u091A.\u092A\u0940., \u0906\u0908." +
            "\u092C\u0940.\u090F\u092E., \u091C\u0938\u094D\u091F \u0938\u093F\u0938\u094D\u091F\u092E, \u092E\u093E" +
            "\u0908\u0915\u094D\u0930\u094B\u0938\u0949\u092B\u094D\u091F, \u0914\u0930\u0947\u0915\u0932, " +
            "\u0938\u0948\u092A, \u0938\u0928, \u0938\u093E\u0908\u092C\u0947\u0938, \u092F\u0942\u0928\u093F\u0938" +
            "\u093F\u0938 \u091C\u0948\u0938\u0940 \u0909\u0926\u094D\u092F\u094B\u0917 \u0915\u0940 \u092A\u094D" +
            "\u0930\u092E\u0941\u0916 \u0915\u092E\u094D\u092A\u0928\u093F\u092F\u094B\u0902 \u0914\u0930 \u0915\u0908" +
            " \u0905\u0928\u094D\u092F \u0928\u0947 \u0905\u092A\u0928\u093E\u092F\u093E \u0939\u0948\u0964 \u092F" +
            "\u0942\u0928\u093F\u0915\u094B\u0921 \u0915\u0940 \u0906\u0935\u0936\u094D\u092F\u0915\u0924\u093E " +
            "\u0906\u0927\u0941\u0928\u093F\u0915 \u092E\u093E\u0928\u0926\u0902\u0921\u094B\u0902, \u091C\u0948" +
            "\u0938\u0947 \u090F\u0915\u094D\u0938.\u090F\u092E.\u090F\u0932., \u091C\u093E\u0935\u093E, \u090F" +
            "\u0915\u092E\u093E \u0938\u094D\u0915\u094D\u0930\u093F\u092A\u094D\u091F (\u091C\u093E\u0935\u093E" +
            " \u0938\u094D\u0915\u094D\u0930\u093F\u092A\u094D\u091F), \u090F\u0932.\u0921\u0940.\u090F.\u092A\u0940.," +
            " \u0915\u094B\u0930\u094D\u092C\u093E 3.0, \u0921\u092C\u094D\u0932\u094D\u092F\u0942.\u090F\u092E." +
            "\u090F\u0932. \u0915\u0947 \u0932\u093F\u090F \u0939\u094B\u0924\u0940 \u0939\u0948 \u0914\u0930 " +
            "\u092F\u0939 \u0906\u0908.\u090F\u0938.\u0913./\u0906\u0908.\u0908.\u0938\u0940. 10646 \u0915\u094B" +
            " \u0932\u093E\u0917\u0942 \u0915\u0930\u0928\u0947 \u0915\u093E \u0905\u0927\u093F\u0915\u093E\u0930\u093F\u0915" +
            " \u0924\u0930\u0940\u0915\u093E \u0939\u0948\u0964 \u092F\u0939 \u0915\u0908 \u0938\u0902\u091A\u093E\u0932\u0928 " +
            "\u092A\u094D\u0930\u0923\u093E\u0932\u093F\u092F\u094B\u0902, \u0938\u092D\u0940 \u0906\u0927\u0941" +
            "\u0928\u093F\u0915 \u092C\u094D\u0930\u093E\u0909\u091C\u0930\u094B\u0902 \u0914\u0930 \u0915\u0908" +
            " \u0905\u0928\u094D\u092F \u0909\u0924\u094D\u092A\u093E\u0926\u094B\u0902 \u092E\u0947\u0902 \u0939" +
            "\u094B\u0924\u093E \u0939\u0948\u0964 \u092F\u0942\u0928\u093F\u0915\u094B\u0921 \u0938\u094D\u091F" +
            "\u0948\u0902\u0921\u0930\u094D\u0921 \u0915\u0940 \u0909\u0924\u094D\u092A\u0924\u093F \u0914\u0930" +
            " \u0907\u0938\u0915\u0947 \u0938\u0939\u093E\u092F\u0915 \u0909\u092A\u0915\u0930\u0923\u094B\u0902" +
            " \u0915\u0940 \u0909\u092A\u0932\u092C\u094D\u0927\u0924\u093E, \u0939\u093E\u0932 \u0939\u0940" +
            " \u0915\u0947 \u0905\u0924\u093F \u092E\u0939\u0924\u094D\u0935\u092A\u0942\u0930\u094D\u0923" +
            " \u0935\u093F\u0936\u094D\u0935\u0935\u094D\u092F\u093E\u092A\u0940 \u0938\u0949\u092B\u094D\u091F" +
            "\u0935\u0947\u092F\u0930 \u091F\u0947\u0915\u094D\u0928\u094B\u0932\u0949\u091C\u0940 \u0930\u0941" +
            "\u091D\u093E\u0928\u094B\u0902 \u092E\u0947\u0902 \u0938\u0947 \u0939\u0948\u0902\u0964\n\n\u092F" +
            "\u0942\u0928\u093F\u0915\u094B\u0921 \u0915\u094B \u0917\u094D\u0930\u093E\u0939\u0915-\u0938\u0930" +
            "\u094D\u0935\u0930 \u0905\u0925\u0935\u093E \u092C\u0939\u0941-\u0906\u092F\u093E\u092E\u0940 " +
            "\u0909\u092A\u0915\u0930\u0923\u094B\u0902 \u0914\u0930 \u0935\u0947\u092C\u0938\u093E\u0907\u091F" +
            "\u094B\u0902 \u092E\u0947\u0902 \u0936\u093E\u092E\u093F\u0932 \u0915\u0930\u0928\u0947 \u0938\u0947," +
            " \u092A\u0930\u0902\u092A\u0930\u093E\u0917\u0924 \u0909\u092A\u0915\u0930\u0923\u094B\u0902 \u0915" +
            "\u0947 \u092A\u094D\u0930\u092F\u094B\u0917 \u0915\u0940 \u0905\u092A\u0947\u0915\u094D\u0937\u093E" +
            " \u0916\u0930\u094D\u091A \u092E\u0947\u0902 \u0905\u0924\u094D\u092F\u0927\u093F\u0915 \u092C\u091A" +
            "\u0924 \u0939\u094B\u0924\u0940 \u0939\u0948\u0964 \u092F\u0942\u0928\u093F\u0915\u094B\u0921 \u0938" +
            "\u0947 \u090F\u0915 \u0910\u0938\u093E \u0905\u0915\u0947\u0932\u093E \u0938\u0949\u092B\u094D\u091F" +
            "\u0935\u0947\u092F\u0930 \u0909\u0924\u094D\u092A\u093E\u0926 \u0905\u0925\u0935\u093E \u0905\u0915" +
            "\u0947\u0932\u093E \u0935\u0947\u092C\u0938\u093E\u0907\u091F \u092E\u093F\u0932 \u091C\u093E\u0924" +
            "\u093E \u0939\u0948, \u091C\u093F\u0938\u0947 \u0930\u0940-\u0907\u0902\u091C\u0940\u0928\u093F\u092F" +
            "\u0930\u093F\u0902\u0917 \u0915\u0947 \u092C\u093F\u0928\u093E \u0935\u093F\u092D\u093F\u0928\u094D" +
            "\u0928 \u092A\u094D\u0932\u0948\u091F\u092B\u0949\u0930\u094D\u092E\u094B\u0902, \u092D\u093E\u0937" +
            "\u093E\u0913\u0902 \u0914\u0930 \u0926\u0947\u0936\u094B\u0902 \u092E\u0947\u0902 \u0909\u092A\u092F" +
            "\u094B\u0917 \u0915\u093F\u092F\u093E \u091C\u093E \u0938\u0915\u0924\u093E \u0939\u0948\u0964 \u0907" +
            "\u0938\u0938\u0947 \u0921\u093E\u091F\u093E \u0915\u094B \u092C\u093F\u0928\u093E \u0915\u093F\u0938" +
            "\u0940 \u092C\u093E\u0927\u093E \u0915\u0947 \u0935\u093F\u092D\u093F\u0928\u094D\u0928 \u092A\u094D" +
            "\u0930\u0923\u093E\u0932\u093F\u092F\u094B\u0902 \u0938\u0947 \u0939\u094B\u0915\u0930 \u0932\u0947 " +
            "\u091C\u093E\u092F\u093E \u091C\u093E \u0938\u0915\u0924\u093E \u0939\u0948\u0964\n",

            "\u0E1A\u0E40\u0E23\u0E37\u0E48\u0E2D\u0E07\u0E02\u0E2D\u0E07\u0E15\u0E31\u0E27\u0E40\u0E25\u0E02. \u0E04" +
            "\u0E2D\u0E21\u0E1E\u0E34\u0E27\u0E40\u0E15\u0E2D\u0E23\u0E4C\u0E08\u0E31\u0E14\u0E40\u0E01\u0E47\u0E1A" +
            "\u0E15\u0E31\u0E27\u0E2D\u0E31\u0E01\u0E29\u0E23\u0E41\u0E25\u0E30\u0E2D\u0E31\u0E01\u0E02\u0E23\u0E30" +
            "\u0E2D\u0E37\u0E48\u0E19\u0E46 \u0E42\u0E14\u0E22\u0E01\u0E32\u0E23\u0E01\u0E33\u0E2B\u0E19\u0E14\u0E2B" +
            "\u0E21\u0E32\u0E22\u0E40\u0E25\u0E02\u0E43\u0E2B\u0E49\u0E2A\u0E33\u0E2B\u0E23\u0E31\u0E1A\u0E41\u0E15" +
            "\u0E48\u0E25\u0E30\u0E15\u0E31\u0E27. \u0E01\u0E48\u0E2D\u0E19\u0E2B\u0E19\u0E49\u0E32\u0E17\u0E35\u0E48" +
            "\u0E4A Unicode \u0E08\u0E30\u0E16\u0E39\u0E01\u0E2A\u0E23\u0E49\u0E32\u0E07\u0E02\u0E36\u0E49\u0E19, " +
            "\u0E44\u0E14\u0E49\u0E21\u0E35\u0E23\u0E30\u0E1A\u0E1A encoding \u0E2D\u0E22\u0E39\u0E48\u0E2B\u0E25" +
            "\u0E32\u0E22\u0E23\u0E49\u0E2D\u0E22\u0E23\u0E30\u0E1A\u0E1A\u0E2A\u0E33\u0E2B\u0E23\u0E31\u0E1A\u0E01" +
            "\u0E32\u0E23\u0E01\u0E33\u0E2B\u0E19\u0E14\u0E2B\u0E21\u0E32\u0E22\u0E40\u0E25\u0E02\u0E40\u0E2B\u0E25" +
            "\u0E48\u0E32\u0E19\u0E35\u0E49. \u0E44\u0E21\u0E48\u0E21\u0E35 encoding \u0E43\u0E14\u0E17\u0E35\u0E48" +
            "\u0E21\u0E35\u0E08\u0E33\u0E19\u0E27\u0E19\u0E15\u0E31\u0E27\u0E2D\u0E31\u0E01\u0E02\u0E23\u0E30\u0E21" +
            "\u0E32\u0E01\u0E40\u0E1E\u0E35\u0E22\u0E07\u0E1E\u0E2D: \u0E22\u0E01\u0E15\u0E31\u0E27\u0E2D\u0E22\u0E48" +
            "\u0E32\u0E07\u0E40\u0E0A\u0E48\u0E19, \u0E40\u0E09\u0E1E\u0E32\u0E30\u0E43\u0E19\u0E01\u0E25\u0E38\u0E48" +
            "\u0E21\u0E2A\u0E2B\u0E20\u0E32\u0E1E\u0E22\u0E38\u0E42\u0E23\u0E1B\u0E40\u0E1E\u0E35\u0E22\u0E07\u0E41" +
            "\u0E2B\u0E48\u0E07\u0E40\u0E14\u0E35\u0E22\u0E27 \u0E01\u0E47\u0E15\u0E49\u0E2D\u0E07\u0E01\u0E32\u0E23" +
            "\u0E2B\u0E25\u0E32\u0E22 encoding \u0E43\u0E19\u0E01\u0E32\u0E23\u0E04\u0E23\u0E2D\u0E1A\u0E04\u0E25\u0E38" +
            "\u0E21\u0E17\u0E38\u0E01\u0E20\u0E32\u0E29\u0E32\u0E43\u0E19\u0E01\u0E25\u0E38\u0E48\u0E21. \u0E2B\u0E23" +
            "\u0E37\u0E2D\u0E41\u0E21\u0E49\u0E41\u0E15\u0E48\u0E43\u0E19\u0E20\u0E32\u0E29\u0E32\u0E40\u0E14\u0E35\u0E48" +
            "\u0E22\u0E27 \u0E40\u0E0A\u0E48\u0E19 \u0E20\u0E32\u0E29\u0E32\u0E2D\u0E31\u0E07\u0E01\u0E24\u0E29 \u0E01" +
            "\u0E47\u0E44\u0E21\u0E48\u0E21\u0E35 encoding \u0E43\u0E14\u0E17\u0E35\u0E48\u0E40\u0E1E\u0E35\u0E22\u0E07" +
            "\u0E1E\u0E2D\u0E2A\u0E33\u0E2B\u0E23\u0E31\u0E1A\u0E17\u0E38\u0E01\u0E15\u0E31\u0E27\u0E2D\u0E31\u0E01" +
            "\u0E29\u0E23, \u0E40\u0E04\u0E23\u0E37\u0E48\u0E2D\u0E07\u0E2B\u0E21\u0E32\u0E22\u0E27\u0E23\u0E23\u0E04" +
            "\u0E15\u0E2D\u0E19 \u0E41\u0E25\u0E30\u0E2A\u0E31\u0E0D\u0E25\u0E31\u0E01\u0E29\u0E13\u0E4C\u0E17\u0E32" +
            "\u0E07\u0E40\u0E17\u0E04\u0E19\u0E34\u0E04\u0E17\u0E35\u0E48\u0E43\u0E0A\u0E49\u0E01\u0E31\u0E19\u0E2D" +
            "\u0E22\u0E39\u0E48\u0E17\u0E31\u0E48\u0E27\u0E44\u0E1B.\n\n\u0E23\u0E30\u0E1A\u0E1A encoding \u0E40\u0E2B" +
            "\u0E25\u0E48\u0E32\u0E19\u0E35\u0E49\u0E22\u0E31\u0E07\u0E02\u0E31\u0E14\u0E41\u0E22\u0E49\u0E07\u0E0B\u0E36" +
            "\u0E48\u0E07\u0E01\u0E31\u0E19\u0E41\u0E25\u0E30\u0E01\u0E31\u0E19. \u0E19\u0E31\u0E48\u0E19\u0E01\u0E47" +
            "\u0E04\u0E37\u0E2D, \u0E43\u0E19\u0E2A\u0E2D\u0E07 encoding \u0E2A\u0E32\u0E21\u0E32\u0E23\u0E16\u0E43\u0E0A" +
            "\u0E49\u0E2B\u0E21\u0E32\u0E22\u0E40\u0E25\u0E02\u0E40\u0E14\u0E35\u0E22\u0E27\u0E01\u0E31\u0E19\u0E2A\u0E33" +
            "\u0E2B\u0E23\u0E31\u0E1A\u0E15\u0E31\u0E27\u0E2D\u0E31\u0E01\u0E02\u0E23\u0E30\u0E2A\u0E2D\u0E07\u0E15\u0E31" +
            "\u0E27\u0E17\u0E35\u0E48\u0E41\u0E15\u0E01\u0E15\u0E48\u0E32\u0E07\u0E01\u0E31\u0E19,\u0E2B\u0E23\u0E37\u0E2D" +
            "\u0E43\u0E0A\u0E49\u0E2B\u0E21\u0E32\u0E22\u0E40\u0E25\u0E02\u0E15\u0E48\u0E32\u0E07\u0E01\u0E31\u0E19\u0E2A" +
            "\u0E33\u0E2B\u0E23\u0E31\u0E1A\u0E2D\u0E31\u0E01\u0E02\u0E23\u0E30\u0E15\u0E31\u0E27\u0E40\u0E14\u0E35\u0E22" +
            "\u0E27\u0E01\u0E31\u0E19. \u0E43\u0E19\u0E23\u0E30\u0E1A\u0E1A\u0E04\u0E2D\u0E21\u0E1E\u0E34\u0E27\u0E40\u0E15" +
            "\u0E2D\u0E23\u0E4C (\u0E42\u0E14\u0E22\u0E40\u0E09\u0E1E\u0E32\u0E30\u0E40\u0E0B\u0E34\u0E23\u0E4C\u0E1F\u0E40" +
            "\u0E27\u0E2D\u0E23\u0E4C) \u0E15\u0E49\u0E2D\u0E07\u0E21\u0E35\u0E01\u0E32\u0E23\u0E2A\u0E19\u0E31\u0E1A\u0E2A" +
            "\u0E19\u0E38\u0E19\u0E2B\u0E25\u0E32\u0E22 encoding; \u0E41\u0E25\u0E30\u0E40\u0E21\u0E37\u0E48\u0E2D\u0E02\u0E49" +
            "\u0E2D\u0E21\u0E39\u0E25\u0E17\u0E35\u0E48\u0E1C\u0E48\u0E32\u0E19\u0E44\u0E1B\u0E21\u0E32\u0E23\u0E30\u0E2B" +
            "\u0E27\u0E48\u0E32\u0E07\u0E01\u0E32\u0E23\u0E40\u0E02\u0E49\u0E32\u0E23\u0E2B\u0E31\u0E2A\u0E2B\u0E23\u0E37" +
            "\u0E2D\u0E41\u0E1E\u0E25\u0E47\u0E15\u0E1F\u0E2D\u0E23\u0E4C\u0E21\u0E17\u0E35\u0E48\u0E15\u0E48\u0E32\u0E07" +
            "\u0E01\u0E31\u0E19, \u0E02\u0E49\u0E2D\u0E21\u0E39\u0E25\u0E19\u0E31\u0E49\u0E19\u0E08\u0E30\u0E40\u0E2A\u0E35" +
            "\u0E48\u0E22\u0E07\u0E15\u0E48\u0E2D\u0E01\u0E32\u0E23\u0E1C\u0E34\u0E14\u0E1E\u0E25\u0E32\u0E14\u0E40\u0E2A" +
            "\u0E35\u0E22\u0E2B\u0E32\u0E22.\nUnicode \u0E08\u0E30\u0E40\u0E1B\u0E25\u0E35\u0E48\u0E22\u0E19\u0E41\u0E1B\u0E25" +
            "\u0E07\u0E2A\u0E34\u0E48\u0E07\u0E40\u0E2B\u0E25\u0E48\u0E32\u0E19\u0E31\u0E49\u0E19\u0E17\u0E31\u0E49\u0E07\u0E2B" +
            "\u0E21\u0E14!\n\nUnicode \u0E01\u0E33\u0E2B\u0E19\u0E14\u0E2B\u0E21\u0E32\u0E22\u0E40\u0E25\u0E02\u0E40\u0E09" +
            "\u0E1E\u0E32\u0E30\u0E2A\u0E33\u0E2B\u0E23\u0E31\u0E1A\u0E41\u0E15\u0E48\u0E25\u0E30\u0E2D\u0E31\u0E01\u0E02" +
            "\u0E23\u0E30, \u0E42\u0E14\u0E22\u0E44\u0E21\u0E48\u0E2A\u0E19\u0E43\u0E08\u0E27\u0E48\u0E32\u0E40\u0E1B\u0E47" +
            "\u0E19\u0E41\u0E1E\u0E25\u0E47\u0E15\u0E1F\u0E2D\u0E23\u0E4C\u0E21\u0E43\u0E14, \u0E44\u0E21\u0E48\u0E02\u0E36" +
            "\u0E49\u0E19\u0E01\u0E31\u0E1A\u0E27\u0E48\u0E32\u0E08\u0E30\u0E40\u0E1B\u0E47\u0E19\u0E42\u0E1B\u0E23\u0E41" +
            "\u0E01\u0E23\u0E21\u0E43\u0E14\u0E41\u0E25\u0E30\u0E44\u0E21\u0E48\u0E27\u0E48\u0E32\u0E08\u0E30\u0E40\u0E1B" +
            "\u0E47\u0E19\u0E20\u0E32\u0E29\u0E32\u0E43\u0E14. \u0E21\u0E32\u0E15\u0E23\u0E10\u0E32\u0E19 Unicode \u0E44" +
            "\u0E14\u0E49\u0E16\u0E39\u0E01\u0E19\u0E33\u0E44\u0E1B\u0E43\u0E0A\u0E49\u0E42\u0E14\u0E22\u0E1C\u0E39\u0E49" +
            "\u0E19\u0E33\u0E43\u0E19\u0E2D\u0E38\u0E15\u0E2A\u0E32\u0E2B\u0E01\u0E23\u0E23\u0E21 \u0E40\u0E0A\u0E48\u0E19 " +
            "Apple, HP, IBM, JustSystem, Microsoft, Oracle, SAP, Sun, Sybase, Unisys \u0E41\u0E25\u0E30\u0E2D\u0E37\u0E48" +
            "\u0E19\u0E46 \u0E2D\u0E35\u0E01\u0E21\u0E32\u0E01. Unicode \u0E40\u0E1B\u0E47\u0E19\u0E2A\u0E34\u0E48\u0E07" +
            "\u0E17\u0E35\u0E48\u0E08\u0E33\u0E40\u0E1B\u0E47\u0E19\u0E2A\u0E33\u0E2B\u0E23\u0E31\u0E1A\u0E21\u0E32\u0E15" +
            "\u0E23\u0E10\u0E32\u0E19\u0E43\u0E2B\u0E21\u0E48\u0E46 \u0E40\u0E0A\u0E48\u0E19 XML, Java, ECMAScript (JavaScript), " +
            "LDAP, CORBA 3.0, WML \u0E2F\u0E25\u0E2F., \u0E41\u0E25\u0E30\u0E40\u0E1B\u0E47\u0E19\u0E41\u0E19\u0E27\u0E17\u0E32" +
            "\u0E07\u0E2D\u0E22\u0E48\u0E32\u0E07\u0E40\u0E1B\u0E47\u0E19\u0E17\u0E32\u0E07\u0E01\u0E32\u0E23\u0E43\u0E19\u0E01" +
            "\u0E32\u0E23\u0E17\u0E33 ISO/IEC 10646. Unicode \u0E44\u0E14\u0E49\u0E23\u0E31\u0E1A\u0E01\u0E32\u0E23\u0E2A\u0E19" +
            "\u0E31\u0E1A\u0E2A\u0E19\u0E38\u0E19\u0E43\u0E19\u0E23\u0E30\u0E1A\u0E1A\u0E1B\u0E0F\u0E34\u0E1A\u0E31\u0E15\u0E34" +
            "\u0E01\u0E32\u0E23\u0E08\u0E33\u0E19\u0E27\u0E19\u0E21\u0E32\u0E01, \u0E1A\u0E23\u0E32\u0E27\u0E40\u0E0B\u0E2D\u0E23" +
            "\u0E4C\u0E43\u0E2B\u0E21\u0E48\u0E46 \u0E17\u0E01\u0E15\u0E31\u0E27, \u0E41\u0E25\u0E30\u0E1C\u0E25\u0E34\u0E15" +
            "\u0E20\u0E31\u0E13\u0E11\u0E4C\u0E2D\u0E37\u0E48\u0E19\u0E46 \u0E2D\u0E35\u0E01\u0E21\u0E32\u0E01. \u0E01\u0E32" +
            "\u0E23\u0E40\u0E01\u0E34\u0E14\u0E02\u0E36\u0E49\u0E19\u0E02\u0E2D\u0E07 Unicode Standard \u0E41\u0E25\u0E30\u0E17" +
            "\u0E39\u0E25\u0E2A\u0E4C\u0E15\u0E48\u0E32\u0E07\u0E46 \u0E17\u0E35\u0E48\u0E21\u0E35\u0E43\u0E19\u0E01\u0E32" +
            "\u0E23\u0E2A\u0E19\u0E31\u0E1A\u0E2A\u0E19\u0E38\u0E19 Unicode, \u0E40\u0E1B\u0E47\u0E19\u0E2B\u0E19\u0E36" +
            "\u0E48\u0E07\u0E43\u0E19\u0E41\u0E19\u0E27\u0E42\u0E19\u0E49\u0E21\u0E17\u0E32\u0E07\u0E40\u0E17\u0E04\u0E42" +
            "\u0E19\u0E42\u0E25\u0E22\u0E35\u0E0B\u0E2D\u0E1F\u0E15\u0E4C\u0E41\u0E27\u0E23\u0E4C\u0E23\u0E30\u0E14\u0E31" +
            "\u0E1A\u0E42\u0E25\u0E01\u0E17\u0E35\u0E48\u0E21\u0E35\u0E04\u0E27\u0E32\u0E21\u0E2A\u0E33\u0E04\n",

            "\u4EC0\u4E48\u662FUnicode(\u7EDF\u4E00\u7801)?\n\nUnicode\u7ED9\u6BCF\u4E2A\u5B57\u7B26\u63D0\u4F9B\u4E86" +
            "\u4E00\u4E2A\u552F\u4E00\u7684\u6570\u5B57\uFF0C\n\u4E0D\u8BBA\u662F\u4EC0\u4E48\u5E73\u53F0\uFF0C\n\u4E0D" +
            "\u8BBA\u662F\u4EC0\u4E48\u7A0B\u5E8F\uFF0C\n\u4E0D\u8BBA\u662F\u4EC0\u4E48\u8BED\u8A00\u3002\n\n\u57FA\u672C" +
            "\u4E0A\uFF0C\u8BA1\u7B97\u673A\u53EA\u662F\u5904\u7406\u6570\u5B57\u3002\u5B83\u4EEC\u6307\u5B9A\u4E00\u4E2A" +
            "\u6570\u5B57\uFF0C\u6765\u50A8\u5B58\u5B57\u6BCD\u6216\u5176\u4ED6\u5B57\u7B26\u3002\u5728\u521B\u9020Unicode" +
            "\u4E4B\u524D\uFF0C\u6709\u6570\u767E\u79CD\u6307\u5B9A\u8FD9\u4E9B\u6570\u5B57\u7684\u7F16\u7801\u7CFB\u7EDF" +
            "\u3002\u6CA1\u6709\u4E00\u4E2A\u7F16\u7801\u53EF\u4EE5\u5305\u542B\u8DB3\u591F\u7684\u5B57\u7B26\uFF1A\u4F8B" +
            "\u5982\uFF0C\u5355\u5355\u6B27\u5DDE\u5171\u540C\u4F53\u5C31\u9700\u8981\u597D\u51E0\u79CD\u4E0D\u540C\u7684" +
            "\u7F16\u7801\u6765\u5305\u62EC\u6240\u6709\u7684\u8BED\u8A00\u3002\u5373\u4F7F\u662F\u5355\u4E00\u79CD\u8BED" +
            "\u8A00\uFF0C\u4F8B\u5982\u82F1\u8BED\uFF0C\u4E5F\u6CA1\u6709\u54EA\u4E00\u4E2A\u7F16\u7801\u53EF\u4EE5\u9002" +
            "\u7528\u4E8E\u6240\u6709\u7684\u5B57\u6BCD\uFF0C\u6807\u70B9\u7B26\u53F7\uFF0C\u548C\u5E38\u7528\u7684\u6280" +
            "\u672F\u7B26\u53F7\u3002\n\n\u8FD9\u4E9B\u7F16\u7801\u7CFB\u7EDF\u4E5F\u4F1A\u4E92\u76F8\u51B2\u7A81\u3002" +
            "\u4E5F\u5C31\u662F\u8BF4\uFF0C\u4E24\u79CD\u7F16\u7801\u53EF\u80FD\u4F7F\u7528\u76F8\u540C\u7684\u6570\u5B57" +
            "\u4EE3\u8868\u4E24\u4E2A\u4E0D\u540C\u7684\u5B57\u7B26\uFF0C\u6216\u4F7F\u7528\u4E0D\u540C\u7684\u6570\u5B57" +
            "\u4EE3\u8868\u76F8\u540C\u7684\u5B57\u7B26\u3002\u4EFB\u4F55\u4E00\u53F0\u7279\u5B9A\u7684\u8BA1\u7B97\u673A(" +
            "\u7279\u522B\u662F\u670D\u52A1\u5668)\u90FD\u9700\u8981\u652F\u6301\u8BB8\u591A\u4E0D\u540C\u7684\u7F16\u7801" +
            "\uFF0C\u4F46\u662F\uFF0C\u4E0D\u8BBA\u4EC0\u4E48\u65F6\u5019\u6570\u636E\u901A\u8FC7\u4E0D\u540C\u7684\u7F16" +
            "\u7801\u6216\u5E73\u53F0\u4E4B\u95F4\uFF0C\u90A3\u4E9B\u6570\u636E\u603B\u4F1A\u6709\u635F\u574F\u7684\u5371" +
            "\u9669\u3002\nUnicode\u6B63\u5728\u6539\u53D8\u6240\u6709\u8FD9\u4E00\u5207\uFF01\n\nUnicode\u7ED9\u6BCF" +
            "\u4E2A\u5B57\u7B26\u63D0\u4F9B\u4E86\u4E00\u4E2A\u552F\u4E00\u7684\u6570\u5B57\uFF0C\u4E0D\u8BBA\u662F\u4EC0" +
            "\u4E48\u5E73\u53F0\uFF0C\u4E0D\u8BBA\u662F\u4EC0\u4E48\u7A0B\u5E8F\uFF0C\u4E0D\u8BBA\u4EC0\u4E48\u8BED\u8A00" +
            "\u3002Unicode\u6807\u51C6\u5DF2\u7ECF\u88AB\u8FD9\u4E9B\u5DE5\u4E1A\u754C\u7684\u9886\u5BFC\u4EEC\u6240\u91C7" +
            "\u7528\uFF0C\u4F8B\u5982\uFF1AApple, HP, IBM, JustSystem, Microsoft, Oracle, SAP, Sun, Sybase, Unisys\u548C" +
            "\u5176\u5B83\u8BB8\u591A\u516C\u53F8\u3002\u6700\u65B0\u7684\u6807\u51C6\u90FD\u9700\u8981Unicode\uFF0C\u4F8B" +
            "\u5982XML, Java, ECMAScript (JavaScript), LDAP, CORBA 3.0, WML\u7B49\u7B49\uFF0C\u5E76\u4E14\uFF0CUnicode" +
            "\u662F\u5B9E\u73B0ISO/IEC 10646\u7684\u6B63\u89C4\u65B9\u5F0F\u3002\u8BB8\u591A\u64CD\u4F5C\u7CFB\u7EDF\uFF0C" +
            "\u6240\u6709\u6700\u65B0\u7684\u6D4F\u89C8\u5668\u548C\u8BB8\u591A\u5176\u4ED6\u4EA7\u54C1\u90FD\u652F\u6301" +
            "\u5B83\u3002Unicode\u6807\u51C6\u7684\u51FA\u73B0\u548C\u652F\u6301\u5B83\u5DE5\u5177\u7684\u5B58\u5728\uFF0C" +
            "\u662F\u8FD1\u6765\u5168\u7403\u8F6F\u4EF6\u6280\u672F\u6700\u91CD\u8981\u7684\u53D1\u5C55\u8D8B\u52BF\u3002" +
            "\n\n\u5C06Unicode\u4E0E\u5BA2\u6237\u670D\u52A1\u5668\u6216\u591A\u5C42\u5E94\u7528\u7A0B\u5E8F\u548C\u7F51" +
            "\u7AD9\u7ED3\u5408\uFF0C\u6BD4\u4F7F\u7528\u4F20\u7EDF\u5B57\u7B26\u96C6\u8282\u7701\u8D39\u7528\u3002Unicode" +
            "\u4F7F\u5355\u4E00\u8F6F\u4EF6\u4EA7\u54C1\u6216\u5355\u4E00\u7F51\u7AD9\u80FD\u591F\u8D2F\u7A7F\u591A\u4E2A" +
            "\u5E73\u53F0\uFF0C\u8BED\u8A00\u548C\u56FD\u5BB6\uFF0C\u800C\u4E0D\u9700\u8981\u91CD\u5EFA\u3002\u5B83\u53EF" +
            "\u5C06\u6570\u636E\u4F20\u8F93\u5230\u8BB8\u591A\u4E0D\u540C\u7684\u7CFB\u7EDF\uFF0C\u800C\u65E0\u635F\u574F" +
            "\u3002\n\u5173\u4E8EUnicode\u5B66\u672F\u5B66\u4F1A\n\nUnicode\u5B66\u672F\u5B66\u4F1A\u662F\u4E00\u4E2A\u975E" +
            "\u76C8\u5229\u7684\u7EC4\u7EC7\uFF0C\u662F\u4E3A\u53D1\u5C55\uFF0C\u6269\u5C55\u548C\u63A8\u5E7F\u4F7F\u7528" +
            "Unicode\u6807\u51C6\u800C\u5EFA\u7ACB\u7684\uFF0CUnicode\u5B66\u672F\u5B66\u4F1A\u8BBE\u7ACB\u4E86\u73B0\u4EE3" +
            "\u8F6F\u4EF6\u4EA7\u54C1\u548C\u6807\u51C6\u6587\u672C\u7684\u8868\u793A\u6CD5\u3002\u5B66\u672F\u5B66\u4F1A" +
            "\u7684\u4F1A\u5458\u4EE3\u8868\u4E86\u5E7F\u6CDB\u9886\u57DF\u7684\u8BA1\u7B97\u673A\u548C\u8D44\u8BAF\u5DE5" +
            "\u4E1A\u7684\u516C\u53F8\u548C\u7EC4\u7EC7\u3002\u5B66\u672F\u5B66\u4F1A\u53EA\u7531\u4F1A\u5458\u63D0\u4F9B" +
            "\u8D44\u91D1\u3002Unicode\u5B66\u672F\u5B66\u4F1A\u7684\u4F1A\u5458\u8D44\u683C\u5F00\u653E\u7ED9\u4E16\u754C" +
            "\u4E0A\u4EFB\u4F55\u652F\u6301Unicode\u6807\u51C6\u548C\u5E0C\u671B\u534F\u52A9\u5176\u6269\u5C55\u548C\u6267" +
            "\u884C\u7684\u7EC4\u7EC7\u53CA\u4E2A\u4EBA\u3002\n",

            "Mik\u00E4 Unicode on?\n\nUnicode m\u00E4\u00E4ritt\u00E4\u00E4 jokaiselle merkille yksil\u00F6llisen arvon." +
            "\nRiippumatta k\u00E4ytt\u00F6j\u00E4rjestelm\u00E4st\u00E4\nRiippumatta ohjelmistosta\nRiippumatta kielest" +
            "\u00E4\n\nTietokoneiden toiminta perustuu numeroiden k\u00E4sittelylle. My\u00F6s kirjaimet ja muut merkit " +
            "ovat tietokoneiden muistissa lukusarjoina. Ennen Unicodea oli k\u00E4yt\u00F6ss\u00E4 satoja eri tapoja esitt" +
            "\u00E4\u00E4 merkkej\u00E4 ja kirjaimia mutta yksik\u00E4\u00E4n n\u00E4ist\u00E4 koodisivuista ei kyennyt m" +
            "\u00E4\u00E4ritt\u00E4m\u00E4\u00E4n kovin kattavaa merkist\u00F6\u00E4. Jo yksinomaan Euroopan Unionissa k" +
            "\u00E4ytetyille kielille tarvitaan useita erilaisia merkist\u00F6j\u00E4 ja koodisivuja, eik\u00E4 edes englannin " +
            "kaltaisen, yksinkertaisen kielen kaikkien kirjaimien, v\u00E4limerkkien ja yleisten teknisten symbolien esitt" +
            "\u00E4minen onnistu yhdell\u00E4 n\u00E4ist\u00E4 koodisivuista.\n\nLis\u00E4ksi koodisivut ovat ristiriidassa" +
            " kesken\u00E4\u00E4n. Eri koodisivuilla voidaan k\u00E4ytt\u00E4\u00E4 samaa arvoa kahdelle t\u00E4ysin " +
            "erilaiselle merkille tai samalla merkill\u00E4 voi olla eri arvo eri koodivulla. Jokaisen tietokoneen, varsinkin" +
            " jos kyseess\u00E4 on palvelin, pit\u00E4\u00E4 tukea lukuisia eri merkist\u00F6j\u00E4 ja koodisivuja ja kun " +
            "tietoa siirret\u00E4\u00E4n j\u00E4rjestelmien ja merkist\u00F6jen v\u00E4lill\u00E4 on tiedon v\u00E4\u00E4ristymisen" +
            " ja tuhoutumisen riski suuri.\nUnicode on ratkaisu\n\nUnicodessa jokaiselle merkille on m\u00E4\u00E4ritetty " +
            "yksil\u00F6llinen, riippumaton arvo. Alan johtavat yritykset, kuten Apple, HP, IBM, JustSystem, Microsoft, " +
            "Oracle, SAP, Sun, Sybase, Unisys ja lukuisat muut toimijat k\u00E4ytt\u00E4v\u00E4t Unicodea. Unicode on sis" +
            "\u00E4llytetty vaatimuksena moniin nykyisiin standardeihin, kuten XML, Java, ECMAScript (JavaScript), LDAP, " +
            "CORBA 3.0 ja WML. Unicode on virallinen keino toteuttaa ISO/IEC 10646 -standardi. Useat k\u00E4ytt\u00F6j\u00E4" +
            "rjestelm\u00E4t, kaikki internet-selaimet ja monet muut tuotteet k\u00E4ytt\u00E4v\u00E4t Unicodea. Unicoden " +
            "syntyminen ja sit\u00E4 tukevien ty\u00F6kalujen suuri m\u00E4\u00E4r\u00E4 on yksi viime aikojen merkitt" +
            "\u00E4vimmist\u00E4 maailmanlaajuisen ohjelmistotuotannon suuntauksista.\n\nUnicoden k\u00E4ytt\u00E4minen " +
            "palvelintekniikassa, vaativissa sovelluksissa ja internet-sivustoilla tuo merkitt\u00E4vi\u00E4 kustannuss" +
            "\u00E4\u00E4st\u00F6j\u00E4 verrattuna usean eri koodisivun k\u00E4yt\u00F6lle. Unicoden ansiosta yksitt" +
            "\u00E4inen sovellus tai sivusto voidaan tarjota useille eri k\u00E4ytt\u00F6j\u00E4rjestelmille ja monilla " +
            "eri kielill\u00E4 maailmanlaajuisesti, ilman uudelleenohjelmoinnin tarvetta. Tieto kulkee j\u00E4rjestelmien " +
            "v\u00E4lill\u00E4 vailla tuhoutumisen tai v\u00E4\u00E4ristymisen vaaraa.\nUnicode-konsortio",

            "Co je Unicode?\n\nUnicode p\u0159i\u0159azuje ka\u017Ed\u00E9mu znaku jedine\u010Dn\u00E9 \u010D\u00EDslo,\nnez" +
            "\u00E1visle na platform\u011B,\nnez\u00E1visle na programu,\nnez\u00E1visle na jazyce.\n\nPo\u010D\u00EDta\u010De," +
            " ze sv\u00E9 podstaty, pracuj\u00ED pouze s \u010D\u00EDsly. P\u00EDsmena a dal\u0161\u00ED znaky ukl\u00E1daj\u00ED" +
            " tak, \u017Ee ka\u017Ed\u00E9mu z nich p\u0159i\u0159ad\u00ED \u010D\u00EDslo. P\u0159ed vznikem Unicode existovaly" +
            " stovky rozd\u00EDln\u00FDch k\u00F3dovac\u00EDch syst\u00E9m\u016F pro p\u0159i\u0159azov\u00E1n\u00ED t\u011Bchto" +
            " \u010D\u00EDsel. \u017D\u00E1dn\u00E9 z t\u011Bchto k\u00F3dov\u00E1n\u00ED nemohlo obsahovat dostatek znak\u016F:" +
            " nap\u0159\u00EDklad Evropsk\u00E1 unie sama pot\u0159ebuje n\u011Bkolik r\u016Fzn\u00FDch k\u00F3dov\u00E1n\u00ED," +
            " aby pokryla v\u0161echny sv\u00E9 jazyky. Dokonce i pro jeden jedin\u00FD jazyk, jako je angli\u010Dtina, nevyhovovalo" +
            " \u017E\u00E1dn\u00E9 k\u00F3dov\u00E1n\u00ED pro v\u0161echny p\u00EDsmena, interpunkci a b\u011B\u017En\u011B " +
            "pou\u017E\u00EDvan\u00E9 technick\u00E9 symboly.\n\nTyto k\u00F3dovac\u00ED syst\u00E9my tak\u00E9 byly v konfliktu" +
            " jeden s druh\u00FDm. To znamen\u00E1, \u017Ee dv\u011B k\u00F3dov\u00E1n\u00ED mohou pou\u017E\u00EDvat stejn\u00E9" +
            " \u010D\u00EDslo pro dva r\u016Fzn\u00E9 znaky, nebo pou\u017E\u00EDvat r\u016Fzn\u00E1 \u010D\u00EDsla pro " +
            "stejn\u00FD znak. Jak\u00FDkoli po\u010D\u00EDta\u010D (zvl\u00E1\u0161t\u011B servery) mus\u00ED podporovat " +
            "mnoho r\u016Fzn\u00FDch k\u00F3dov\u00E1n\u00ED; p\u0159esto, kdykoli jsou data p\u0159ed\u00E1v\u00E1na mezi " +
            "r\u016Fzn\u00FDmi k\u00F3dov\u00E1n\u00EDmi nebo platformami, hroz\u00ED, \u017Ee tato data budou po\u0161kozena." +
            "\n\n \nUnicode toto v\u0161echno m\u011Bn\u00ED!\n\nUnicode p\u0159i\u0159azuje ka\u017Ed\u00E9mu znaku jedine" +
            "\u010Dn\u00E9 \u010D\u00EDslo, nez\u00E1visle na platform\u011B, nez\u00E1visle na programu, nez\u00E1visle na " +
            "jazyku. Unicode Standard byl p\u0159ijat takov\u00FDmi pr\u016Fmyslov\u00FDmi v\u016Fdci, jako jsou Apple, HP, " +
            "IBM, JustSystem, Microsoft, Oracle, SAP, Sun, Sybase, Unisys a mnoha dal\u0161\u00EDmi. Unicode je vy\u017Eadov" +
            "\u00E1n modern\u00EDmi standardy, jako jsou XML, Java, ECMAScript (JavaScript), LDAP, CORBA 3.0, WML atd. a je " +
            "ofici\u00E1ln\u00ED formou implementace ISO/IEC 10646. Je podporov\u00E1n v mnoha opera\u010Dn\u00EDch syst" +
            "\u00E9mech, v\u0161ech modern\u00EDch prohl\u00ED\u017Ee\u010D\u00EDch a mnoha dal\u0161\u00EDch produktech. " +
            "To, \u017Ee se objevil Unicode Standard a dostupnost n\u00E1stroj\u016F, kter\u00E9 jej podporuj\u00ED, pat" +
            "\u0159\u00ED mezi nejv\u00FDznamn\u011Bj\u0161\u00ED ned\u00E1vn\u00E9 trendy v glob\u00E1ln\u00ED technologii" +
            " softwaru.\n\nZa\u010Dlen\u011Bn\u00ED Unicode do klient-server nebo v\u00EDcevrstv\u00FDch aplikac\u00ED a " +
            "webov\u00FDch str\u00E1nek nab\u00EDz\u00ED v\u00FDznamn\u00E9 u\u0161et\u0159en\u00ED n\u00E1klad\u016F oproti " +
            "d\u0159\u00EDv\u011Bj\u0161\u00EDm znakov\u00FDm sad\u00E1m. Unicode umo\u017E\u0148uje, aby jedin\u00FD " +
            "softwarov\u00FD produkt nebo jedin\u00E1 webov\u00E1 str\u00E1nka byla zam\u011B\u0159ena na mnoho platforem, " +
            "jazyk\u016F a zem\u00ED beze zm\u011Bn n\u00E1vrhu. To dovoluje p\u0159en\u00E1\u0161et data p\u0159es mnoho " +
            "r\u016Fzn\u00FDch syst\u00E9m\u016F bez poru\u0161en\u00ED.",

            "?\n\n\u05D9\u05D5\u05E0\u05D9\u05E7\u05D5\u05D3 \u05DE\u05E7\u05E6\u05D4 \u05DE\u05E1\u05E4\u05E8 \u05D9\u05D9" +
            "\u05D7\u05D5\u05D3\u05D9 \u05DC\u05DB\u05DC \u05EA\u05D5,\n\u05DC\u05D0 \u05DE\u05E9\u05E0\u05D4 \u05E2\u05DC " +
            "\u05D0\u05D9\u05D6\u05D5 \u05E4\u05DC\u05D8\u05E4\u05D5\u05E8\u05DE\u05D4,\n\u05DC\u05D0 \u05DE\u05E9\u05E0\u05D4" +
            " \u05D1\u05D0\u05D9\u05D6\u05D5 \u05EA\u05D5\u05DB\u05E0\u05D9\u05EA,\n\u05D5\u05DC\u05D0 \u05DE\u05E9\u05E0\u05D4" +
            " \u05D1\u05D0\u05D9\u05D6\u05D5 \u05E9\u05E4\u05D4.\n\n\u05D1\u05D0\u05D5\u05E4\u05DF \u05D1\u05E1\u05D9\u05E1\u05D9," +
            " \u05DE\u05D7\u05E9\u05D1\u05D9\u05DD \u05E2\u05D5\u05E1\u05E7\u05D9\u05DD \u05E8\u05E7 \u05D1\u05DE\u05E1\u05E4" +
            "\u05E8\u05D9\u05DD. \u05D4\u05DD \u05DE\u05D0\u05D7\u05E1\u05E0\u05D9\u05DD \u05D0\u05D5\u05EA\u05D9\u05D5\u05EA" +
            " \u05D5\u05EA\u05D5\u05D5\u05D9\u05DD \u05D0\u05D7\u05E8\u05D9\u05DD \u05E2\u05DC-\u05D9\u05D3\u05D9 \u05D4\u05E7" +
            "\u05E6\u05D0\u05EA \u05DE\u05E1\u05E4\u05E8 \u05DC\u05DB\u05DC \u05D0\u05D7\u05D3 \u05DE\u05D4\u05DD. \u05D1\u05D8" +
            "\u05E8\u05DD \u05D4\u05D5\u05DE\u05E6\u05D0 \u05D4\u05D9\u05D5\u05E0\u05D9\u05E7\u05D5\u05D3, \u05D4\u05D9\u05D5 " +
            "\u05DE\u05D0\u05D5\u05EA \u05DE\u05E2\u05E8\u05DB\u05D5\u05EA \u05E7\u05D9\u05D3\u05D5\u05D3 \u05E9\u05D5\u05E0" +
            "\u05D5\u05EA \u05DC\u05D4\u05E7\u05E6\u05D0\u05EA \u05D4\u05DE\u05E1\u05E4\u05E8\u05D9\u05DD \u05D4\u05DC\u05DC" +
            "\u05D5. \u05D0\u05E3 \u05DC\u05D0 \u05D0\u05D7\u05EA \u05DE\u05D4\u05DF \u05D9\u05DB\u05DC\u05D4 \u05DC\u05D4\u05DB" +
            "\u05D9\u05DC \u05DB\u05DE\u05D5\u05EA \u05EA\u05D5\u05D5\u05D9\u05DD \u05DE\u05E1\u05E4\u05E7\u05EA. \u05DC\u05D3" +
            "\u05D5\u05D2\u05DE\u05D0: \u05E8\u05E7 \u05DC\u05D0\u05D9\u05D7\u05D5\u05D3 \u05D4\u05D0\u05D9\u05E8\u05D5\u05E4" +
            "\u05D0\u05D9 \u05E0\u05D3\u05E8\u05E9\u05D9\u05DD \u05DB\u05DE\u05D4 \u05E1\u05D5\u05D2\u05D9 \u05E7\u05D9\u05D3" +
            "\u05D5\u05D3\u05D9\u05DD \u05E9\u05D5\u05E0\u05D9\u05DD \u05E2\u05DC \u05DE\u05E0\u05EA \u05DC\u05DB\u05E1\u05D5" +
            "\u05EA \u05D0\u05EA \u05DB\u05DC \u05D4\u05E9\u05E4\u05D5\u05EA \u05D4\u05DE\u05D3\u05D5\u05D1\u05E8\u05D5\u05EA " +
            "\u05D1\u05D5. \u05D9\u05EA\u05D9\u05E8\u05D4 \u05DE\u05D6\u05D0\u05EA \u05D0\u05E3 \u05DC\u05E9\u05E4\u05D4 " +
            "\u05D1\u05D5\u05D3\u05D3\u05EA, \u05DB\u05DE\u05D5 \u05D0\u05E0\u05D2\u05DC\u05D9\u05EA \u05DC\u05DE\u05E9\u05DC," +
            " \u05DC\u05D0 \u05D4\u05D9\u05D4 \u05D3\u05D9 \u05D1\u05DE\u05E2\u05E8\u05DB\u05EA \u05E7\u05D9\u05D3\u05D5\u05D3" +
            " \u05D0\u05D7\u05EA \u05D1\u05E2\u05D1\u05D5\u05E8 \u05DB\u05DC \u05D4\u05D0\u05D5\u05EA\u05D9\u05D5\u05EA, \u05E1" +
            "\u05D9\u05DE\u05E0\u05D9 \u05D4\u05E4\u05D9\u05E1\u05D5\u05E7 \u05D5\u05D4\u05E1\u05DE\u05DC\u05D9\u05DD \u05D4" +
            "\u05D8\u05DB\u05E0\u05D9\u05D9\u05DD \u05E9\u05D1\u05E9\u05D9\u05DE\u05D5\u05E9 \u05E9\u05D5\u05D8\u05E3.\n\n\u05DE" +
            "\u05E2\u05E8\u05DB\u05D5\u05EA \u05E7\u05D9\u05D3\u05D5\u05D3 \u05D0\u05DC\u05D5 \u05D0\u05E3 \u05E1\u05D5\u05EA" +
            "\u05E8\u05D5\u05EA \u05D6\u05D5 \u05D0\u05EA \u05D6\u05D5. \u05DB\u05DC\u05D5\u05DE\u05E8, \u05E9\u05E0\u05D9 " +
            "\u05E7\u05D9\u05D3\u05D5\u05D3\u05D9\u05DD \u05D9\u05DB\u05D5\u05DC\u05D9\u05DD \u05DC\u05D4\u05E9\u05EA\u05DE" +
            "\u05E9 \u05D1\u05D0\u05D5\u05EA\u05D5 \u05DE\u05E1\u05E4\u05E8 \u05DC\u05E9\u05E0\u05D9 \u05EA\u05D5\u05D9\u05DD" +
            " \u05E0\u05D1\u05D3\u05DC\u05D9\u05DD, \u05D0\u05D5 \u05DC\u05D4\u05E9\u05EA\u05DE\u05E9 \u05D1\u05DE\u05E1\u05E4" +
            "\u05E8\u05D9\u05DD \u05E9\u05D5\u05E0\u05D9\u05DD \u05DC\u05D0\u05D5\u05EA\u05D5 \u05EA\u05D5. \u05E2\u05DC " +
            "\u05DB\u05DC \u05DE\u05D7\u05E9\u05D1 (\u05D5\u05D1\u05DE\u05D9\u05D5\u05D7\u05D3 \u05E9\u05E8\u05EA\u05D9\u05DD)" +
            " \u05DC\u05EA\u05DE\u05D5\u05DA \u05D1\u05DE\u05E1\u05E4\u05E8 \u05E8\u05D1 \u05E9\u05DC \u05DE\u05E2\u05E8\u05DB" +
            "\u05D5\u05EA \u05E7\u05D9\u05D3\u05D5\u05D3 \u05E9\u05D5\u05E0\u05D5\u05EA; \u05D0\u05D5\u05DC\u05DD \u05DB\u05DC" +
            " \u05D0\u05D9\u05DE\u05EA \u05E9\u05E0\u05EA\u05D5\u05E0\u05D9\u05DD \u05E2\u05D5\u05D1\u05E8\u05D9\u05DD \u05D1" +
            "\u05D9\u05DF \u05DE\u05E2\u05E8\u05DB\u05D5\u05EA \u05E7\u05D9\u05D3\u05D5\u05D3 \u05D0\u05D5 \u05E4\u05DC\u05D8" +
            "\u05E4\u05D5\u05E8\u05DE\u05D5\u05EA \u05E9\u05D5\u05E0\u05D5\u05EA \u05E7\u05D9\u05D9\u05DD \u05D4\u05E1\u05D9" +
            "\u05DB\u05D5\u05DF \u05E9\u05D9\u05D9\u05E4\u05D2\u05DE\u05D5.\n\u05D9\u05D5\u05E0\u05D9\u05E7\u05D5\u05D3 " +
            "\u05DE\u05E9\u05E0\u05D4 \u05D0\u05EA \u05DB\u05DC \u05D6\u05D4!\n\n\u05D9\u05D5\u05E0\u05D9\u05E7\u05D5\u05D3 " +
            "\u05DE\u05E7\u05E6\u05D4 \u05DE\u05E1\u05E4\u05E8 \u05D9\u05D9\u05D7\u05D5\u05D3\u05D9 \u05DC\u05DB\u05DC " +
            "\u05EA\u05D5, \u05DC\u05DC\u05D0 \u05EA\u05DC\u05D5\u05EA \u05D1\u05E4\u05DC\u05D8\u05E4\u05D5\u05E8\u05DE\u05D4," +
            " \u05D1\u05EA\u05D5\u05DB\u05E0\u05D9\u05EA, \u05D0\u05D5 \u05D1\u05E9\u05E4\u05D4.\u05EA\u05E7\u05DF " +
            "\u05D4\u05D9\u05D5\u05E0\u05D9\u05E7\u05D5\u05D3 \u05D0\u05D5\u05DE\u05E5 \u05E2\u05DC-\u05D9\u05D3\u05D9 " +
            "\u05D4\u05DE\u05D5\u05D1\u05D9\u05DC\u05D9\u05DD \u05D1\u05EA\u05E2\u05E9\u05D9\u05D9\u05D4 \u05DB\u05DE\u05D5 " +
            "Apple\u200F, HP\u200F, IBM\u200F, JustSystem\u200F, Microsoft\u200F, Oracle\u200F, SAP\u200F, Sun\u200F, Sybase" +
            "\u200F, Unisys\u200F \u05D5\u05E8\u05D1\u05D9\u05DD \u05D0\u05D7\u05E8\u05D9\u05DD. \u05D9\u05D5\u05E0\u05D9\u05E7" +
            "\u05D5\u05D3 \u05E0\u05D3\u05E8\u05E9 \u05E2\u05DC-\u05D9\u05D3\u05D9 \u05EA\u05E7\u05E0\u05D9\u05DD \u05DE\u05D5" +
            "\u05D3\u05E8\u05E0\u05D9\u05D9\u05DD \u05DB\u05DE\u05D5 XML\u200F, Java\u200F, ECMAScript (JavaScript)\u200E\u200F, " +
            "LDAP\u200F, CORBA 3.0\u200E\u200F, WML\u200F \u05D5\u05DB\u05D3\u05D5\u05DE\u05D4, \u05D5\u05DE\u05D4\u05D5\u05D5\u05D4 " +
            "\u05DC\u05DE\u05E2\u05E9\u05D4 \u05D0\u05EA \u05D4\u05D9\u05D9\u05E9\u05D5\u05DD \u05D4\u05E8\u05E9\u05DE\u05D9 " +
            "\u05E9\u05DC \u05EA\u05E7\u05DF ISO/IEC 10646. \u05D4\u05D5\u05D0 \u05E0\u05EA\u05DE\u05DA \u05E2\u05DC " +
            "\u05D9\u05D3\u05D9 \u05DE\u05E2\u05E8\u05DB\u05D5\u05EA \u05D4\u05E4\u05E2\u05DC\u05D4 \u05E8\u05D1\u05D5\u05EA, " +
            "\u05DB\u05DC \u05D4\u05D3\u05E4\u05D3\u05E4\u05E0\u05D9\u05DD \u05D4\u05D7\u05D3\u05D9\u05E9\u05D9\u05DD, " +
            "\u05D5\u05DE\u05D5\u05E6\u05E8\u05D9\u05DD \u05E8\u05D1\u05D9\u05DD \u05D0\u05D7\u05E8\u05D9\u05DD. " +
            "\u05D4\u05D5\u05E4\u05E2\u05EA \u05EA\u05E7\u05DF \u05D4\u05D9\u05D5\u05E0\u05D9\u05E7\u05D5\u05D3 " +
            "\u05D5\u05D6\u05DE\u05D9\u05E0\u05D5\u05EA \u05D4\u05DB\u05DC\u05D9\u05DD \u05D4\u05EA\u05D5\u05DE\u05DB" +
            "\u05D9\u05DD \u05D1\u05D5 \u05E0\u05DE\u05E0\u05D5\u05EA \u05E2\u05DD \u05D4\u05DE\u05D2\u05DE\u05D5\u05EA" +
            " \u05D4\u05DB\u05DC\u05DC-\u05E2\u05D5\u05DC\u05DE\u05D9\u05D5\u05EA \u05D4\u05D7\u05E9\u05D5\u05D1\u05D5\u05EA" +
            " \u05D1\u05D9\u05D5\u05EA\u05E8, \u05D0\u05E9\u05E8 \u05DE\u05E1\u05EA\u05DE\u05E0\u05D5\u05EA \u05DC\u05D0" +
            "\u05D7\u05E8\u05D5\u05E0\u05D4 \u05D1\u05D8\u05DB\u05E0\u05D5\u05DC\u05D5\u05D2\u05D9\u05D9\u05EA \u05D4" +
            "\u05EA\u05D5\u05DB\u05E0\u05D4.\n\n\u05E9\u05D9\u05DC\u05D5\u05D1 \u05D9\u05D5\u05E0\u05D9\u05E7\u05D5\u05D3" +
            " \u05D1\u05D9\u05D9\u05E9\u05D5\u05DE\u05D9 \u05E9\u05E8\u05EA-\u05DC\u05E7\u05D5\u05D7 \u05D0\u05D5 \u05D1" +
            "\u05D9\u05D9\u05E9\u05D5\u05DE\u05D9\u05DD \u05E8\u05D1\u05D9-\u05E9\u05DB\u05D1\u05D5\u05EA \u05D5\u05D1\u05D0" +
            "\u05EA\u05E8\u05D9 \u05D0\u05D9\u05E0\u05D8\u05E8\u05E0\u05D8 \u05DE\u05D0\u05E4\u05E9\u05E8 \u05D7\u05D9\u05E1" +
            "\u05DB\u05D5\u05DF \u05E0\u05D9\u05DB\u05E8 \u05D1\u05E2\u05DC\u05D5\u05D9\u05D5\u05EA \u05DC\u05E2\u05D5\u05DE" +
            "\u05EA \u05D4\u05E9\u05D9\u05DE\u05D5\u05E9 \u05D1\u05E1\u05D3\u05E8\u05D5\u05EA \u05D4\u05EA\u05D5\u05D5\u05D9" +
            "\u05DD \u05D4\u05DE\u05E1\u05D5\u05E8\u05EA\u05D9\u05D5\u05EA. \u05D4\u05D5\u05D3\u05D5\u05EA \u05DC\u05D9\u05D5" +
            "\u05E0\u05D9\u05E7\u05D5\u05D3, \u05DE\u05D5\u05E6\u05E8 \u05EA\u05D5\u05DB\u05E0\u05D4 \u05D0\u05D7\u05D3 \u05D0" +
            "\u05D5 \u05D0\u05EA\u05E8 \u05D9\u05D7\u05D9\u05D3 \u05D1\u05E8\u05E9\u05EA \u05D9\u05DB\u05D5\u05DC \u05DC\u05D4" +
            "\u05E8\u05D7\u05D9\u05D1 \u05D0\u05EA \u05D9\u05E2\u05D3\u05D9\u05D5 \u05DC\u05DE\u05D2\u05D5\u05D5\u05DF \u05E4" +
            "\u05DC\u05D8\u05E4\u05D5\u05E8\u05DE\u05D5\u05EA, \u05D0\u05E8\u05E6\u05D5\u05EA \u05D5\u05E9\u05E4\u05D5\u05EA " +
            "\u05DC\u05DC\u05D0 \u05E6\u05D5\u05E8\u05DA \u05D1\u05E9\u05D9\u05E0\u05D5\u05D9\u05D9\u05DD \u05DE\u05E8\u05D7" +
            "\u05D9\u05E7\u05D9\u05DD. \u05D9\u05D5\u05E0\u05D9\u05E7\u05D5\u05D3 \u05DE\u05D0\u05E4\u05E9\u05E8 \u05DE\u05E2" +
            "\u05D1\u05E8 \u05E0\u05EA\u05D5\u05E0\u05D9\u05DD \u05D3\u05E8\u05DA \u05DE\u05E2\u05E8\u05DB\u05D5\u05EA \u05E8" +
            "\u05D1\u05D5\u05EA \u05D5\u05E9\u05D5\u05E0\u05D5\u05EA \u05DE\u05D1\u05DC\u05D9 \u05E9\u05D9\u05D9\u05E4\u05D2" +
            "\u05DE\u05D5.\n\u05E4\u05E8\u05D8\u05D9\u05DD \u05D0\u05D5\u05D3\u05D5\u05EA \u05D4\u05E7\u05D5\u05E0\u05E1\u05D5" +
            "\u05E8\u05E6\u05D9\u05D5\u05DD \u05E9\u05DC \u05D9\u05D5\u05E0\u05D9\u05E7\u05D5\u05D3 (Unicode Consortium",
            "Hva\u00F0 er Unicode?\n\nUnicode sta\u00F0allinn \u00FAthlutar hverju skrift\u00E1kni t\u00F6lu,\nsem er \u00F3h\u00E1" +
            "\u00F0 t\u00F6lvuger\u00F0,\nsem er \u00F3h\u00E1\u00F0 forriti,\nsem er \u00F3h\u00E1\u00F0 tungum\u00E1li.\n\nT" +
            "\u00F6lvur geta \u00ED e\u00F0li s\u00EDnu a\u00F0eins unni\u00F0 me\u00F0 t\u00F6lur. \u00DE\u00E6r geyma b" +
            "\u00F3kstafi og \u00F6nnur skrift\u00E1kn me\u00F0 \u00FEv\u00ED a\u00F0 \u00FAthluta \u00FEeim t\u00F6lu. " +
            "\u00C1\u00F0ur en Unicode kom til voru hundru\u00F0 mismunandi t\u00FAlkunarkerfa sem \u00FAthlutu\u00F0u " +
            "\u00FEessum t\u00F6lum. Ekkert eitt t\u00FAlkunarkerfi gat innihaldi\u00F0 n\u00E6gilegan fj\u00F6lda skrift" +
            "\u00E1kna; t.d. \u00FEarfnast Evr\u00F3pusambandi\u00F0 nokkurra mismunandi kerfa til a\u00F0 spanna \u00F6ll " +
            "tungum\u00E1l \u00FEess. Jafnvel fyrir eitt tungum\u00E1l, eins og ensku, var eitt t\u00FAlkunarkerfi ekki n" +
            "\u00F3g fyrir alla b\u00F3kstafi, greinarmerki og algengustu einingat\u00E1kn.\n\nT\u00FAlkunarkerfin hafa einnig" +
            " veri\u00F0 \u00ED andst\u00F6\u00F0u hvert vi\u00F0 anna\u00F0, \u00FE.e. tv\u00F6 kerfi geta nota\u00F0 s\u00F6mu" +
            " t\u00F6lu fyrir tv\u00F6 \u00F3l\u00EDk skrift\u00E1kn e\u00F0a nota\u00F0 tv\u00E6r mismunandi t\u00F6lur fyrir" +
            " sama t\u00E1kni\u00F0. S\u00E9rhver t\u00F6lva \u00FEarf(s\u00E9rstaklega mi\u00F0larar) a\u00F0 sty\u00F0ja margs" +
            " konar t\u00FAlkanir \u00E1 st\u00F6fum; engu a\u00F0 s\u00ED\u00F0ur er alltaf h\u00E6tta \u00E1 stafabrenglun " +
            "\u00FEegar g\u00F6gn fara \u00E1 milli t\u00F6lva og \u00E1 milli mismunandi t\u00FAlkunarkerfa.\nUnicode breytir" +
            " \u00FEessu \u00F6llu!\n\nUnicode gefur hverju skrift\u00E1kni eigin t\u00F6lu sem breytist ekki eftir t\u00F6lvuger" +
            "\u00F0, forriti e\u00F0a tungum\u00E1li. Unicode sta\u00F0allinn hefur veri\u00F0 tekinn upp af fork\u00F3lfum t" +
            "\u00F6lvui\u00F0na\u00F0arins; Apple, HP, IBM, JustSystem, Microsoft, Oracle, SAP, Sun, Sybase, Unisys og m" +
            "\u00F6rgum \u00F6\u00F0rum. Unicode er nota\u00F0 af n\u00FAt\u00EDmast\u00F6\u00F0lum eins og XML, Java, " +
            "ECMAScript (JavaScript), LDAP, CORBA 3.0, WML, o.s.frv. og er hin opinbera lei\u00F0 til a\u00F0 \u00FAtf" +
            "\u00E6ra ISO/IEC 10646. Unicode sta\u00F0allinn er studdur af m\u00F6rgum st\u00FDrikerfum, \u00F6llum n" +
            "\u00FAt\u00EDmav\u00F6frum og m\u00F6rgum \u00F6\u00F0rum b\u00FAna\u00F0i. Tilkoma Unicode sta\u00F0alsins" +
            " og b\u00FAna\u00F0ar til a\u00F0 sty\u00F0ja hann eru veigamikil skref \u00ED \u00FEr\u00F3un hnattr\u00E6nnar" +
            " t\u00E6kni.\n\nA\u00F0 nota Unicode \u00ED notendami\u00F0lurum e\u00F0a \u00ED forritum og vefs\u00ED" +
            "\u00F0um me\u00F0 m\u00F6rgum notendavi\u00F0m\u00F3tum b\u00FD\u00F0ur upp \u00E1 umtalsver\u00F0an " +
            "sparna\u00F0, \u00ED samanbur\u00F0i vi\u00F0 a\u00F0 nota eldri stafat\u00F6flur. Unicode leyfir einum " +
            "forritapakka e\u00F0a einni vefsl\u00F3\u00F0 a\u00F0 n\u00E1 til margra t\u00F6lvuger\u00F0a, tungum" +
            "\u00E1la og landa \u00E1n endurh\u00F6nnunar. Unicode gerir g\u00F6gnum kleift a\u00F0 fer\u00F0ast gegnum " +
            "m\u00F6rg mismunandi kerfi \u00E1n brenglunar.\nUm Unicode samt\u00F6kin (Consortium)\n\nUnicode Consortium " +
            "samt\u00F6kin stefna ekki a\u00F0 hagna\u00F0i. \u00DEau voru stofnu\u00F0 til a\u00F0 \u00FEr\u00F3a, \u00FA" +
            "tv\u00EDkka og koma \u00E1 framf\u00E6ri Unicode sta\u00F0linum, sem skilgreinir framsetningu \u00E1 texta " +
            "\u00ED n\u00FAt\u00EDmaforritap\u00F6kkum og st\u00F6\u00F0lum. Me\u00F0limir samtakanna eru fulltr\u00FAar " +
            "margs konar fyrirt\u00E6kja og stofnana \u00ED t\u00F6lvu- og uppl\u00FDsingai\u00F0na\u00F0inum. Samt\u00F6kin" +
            " hafa eing\u00F6ngu tekjur af a\u00F0ildargj\u00F6ldum s\u00EDnum. Fyrirt\u00E6ki, stofnanir og einstaklingar " +
            "hvar sem er \u00ED heiminum sem \u00F3"
            };
}
