//*****************************************************************************
//                                FrmMain.cpp                                 *
//                               -------------                                *
// Started     : 18/08/2003                                                   *
// Last Update : 23/09/2011                                                   *
// Copyright   : (C) 2003 by MSWaters                                         *
// Email       : M.Waters@bom.gov.au                                          *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "main/FrmMain.hpp"

// The application icons
#include "icons/gspiceui-32x32.xpm"
#include "icons/file-open.xpm"
#include "icons/file-import.xpm"
#include "icons/file-reload.xpm"
#include "icons/file-close.xpm"
#include "icons/sim-create.xpm"
#include "icons/sim-run.xpm"
#include "icons/sim-stop.xpm"
#include "icons/sim-edit.xpm"
#include "icons/sim-plot.xpm"
#include "icons/help.xpm"

//*****************************************************************************
// Implement an event table in which the events are routed to their respective
// handler functions in the class. If -1 is given as the ID, the given handler
// will be invoked for any event of the specified type.

BEGIN_EVENT_TABLE( FrmMain, wxFrame )

  EVT_MENU( ID_MNU_OPEN,    FrmMain::OnOpen       )
  EVT_MENU( ID_MNU_IMPORT,  FrmMain::OnImport     )
  EVT_MENU( ID_MNU_RELOAD,  FrmMain::OnReload     )
  EVT_MENU( ID_MNU_CLOSE,   FrmMain::OnClose      )
  EVT_MENU( ID_MNU_EXIT,    FrmMain::OnAppExit    )
  EVT_MENU( ID_MNU_CREATE,  FrmMain::OnSimCreate  )
  EVT_MENU( ID_MNU_RUN,     FrmMain::OnSimRun     )
  EVT_MENU( ID_MNU_STOP,    FrmMain::OnSimStop    )
  EVT_MENU( ID_MNU_SCHEM,   FrmMain::OnSchematic  )
  EVT_MENU( ID_MNU_VIEWER,  FrmMain::OnViewData   )
  EVT_MENU( ID_MNU_GNUCAP,  FrmMain::OnSelSimEng  )
  EVT_MENU( ID_MNU_NGSPICE, FrmMain::OnSelSimEng  )
  EVT_MENU( ID_MNU_PREFS,   FrmMain::OnPrefs      )
  EVT_MENU( ID_MNU_MAN_USR, FrmMain::OnManUser    )
  EVT_MENU( ID_MNU_MAN_NGS, FrmMain::OnManNgSpice )
  EVT_MENU( ID_MNU_MAN_GCP, FrmMain::OnManGnuCap  )
  EVT_MENU( ID_MNU_ABOUT,   FrmMain::OnAbout      )

  EVT_TOOL( ID_TBR_OPEN,    FrmMain::OnOpen       )
  EVT_TOOL( ID_TBR_IMPORT,  FrmMain::OnImport     )
  EVT_TOOL( ID_TBR_RELOAD,  FrmMain::OnReload     )
  EVT_TOOL( ID_TBR_CLOSE,   FrmMain::OnClose      )
  EVT_TOOL( ID_TBR_CREATE,  FrmMain::OnSimCreate  )
  EVT_TOOL( ID_TBR_RUN,     FrmMain::OnSimRun     )
  EVT_TOOL( ID_TBR_STOP,    FrmMain::OnSimStop    )
  EVT_TOOL( ID_TBR_SCHEM,   FrmMain::OnSchematic  )
  EVT_TOOL( ID_TBR_VIEWER,  FrmMain::OnViewData   )
  EVT_TOOL( ID_TBR_HELP,    FrmMain::OnManUser    )

  EVT_TOOL_ENTER( -1,       FrmMain::OnToolEnter  )

  EVT_CLOSE(                FrmMain::OnSysExit    )

END_EVENT_TABLE( )

//*****************************************************************************
// Constructor.
//
// Argument List :
//   poApp - A pointer to the class that created this object

FrmMain::FrmMain( const wxApp * poApp ) :
             wxFrame( (wxFrame *) NULL, -1, wxT(""), wxDefaultPosition,
                      wxDefaultSize, wxDEFAULT_FRAME_STYLE & ~wxMAXIMIZE_BOX ),
             m_oFileTsks( this ), m_oHelpTsks( this ),
             m_oSimnNgs( ), m_oSimnGcp( )
{
  wxString  os1;

  // Indicate that the main frame is open
  m_bIsOpen = TRUE;

  // Set pointer to parent application
  m_poApp = poApp;

  // Get the global configuration object
  m_poCfg = (wxConfig *) wxConfig::Get( );

  // Initialize pointers
  m_poSimn      = NULL;
  m_poPrcSimEng = NULL;
  m_poNbkSimEng = NULL;

  // Create everything
  Initialize( );

  // Set status bar text
  os1 << APP_NAME << wxT(", Version ") << APP_VERSION
      << wxT(" (") << APP_DATE << wxT(")");
  SetStatusText( wxT(" Welcome to ") + os1, ePANE_MESG );
}

//*****************************************************************************
// Destructor.

FrmMain::~FrmMain( )
{
  if( m_poNbkSimEng != NULL ) { delete m_poNbkSimEng; m_poNbkSimEng = NULL; }
  if( m_poPrcSimEng != NULL ) { delete m_poPrcSimEng; m_poPrcSimEng = NULL; }
}

//*****************************************************************************
// Initialize the main frame.

void  FrmMain::Initialize( void )
{
  long  li1;

  // Set the frame icon, font and title
  SetIcon( wxICON( gspiceui ) );
  SetFont( FONT_NORM );
  m_oFileTsks.bSetTitle( );

  // Automatically update the frame layout when it is resized
  SetAutoLayout( TRUE );

  // Call all the initialization functions
  InitMenuBar ( );
  InitToolBar ( );
  InitLstBoxs ( );
  InitNbkTCtls( );
  InitStatBar ( );
  InitToolTips( );
  InitSimEng  ( );
  InitLogFiles( );
  DoLayout( );
  InitPosnSize( );

  // Set the sweep source synchronization flag
  m_poCfg->SetPath( wxT("/Main") );
  m_poCfg->Read( wxT("SyncSwpSrcs"), &li1, 0 );
  PnlAnaBase::m_bSyncSwpSrcs = ( li1 != 0 ? TRUE : FALSE );

  // Display the desired analysis page
  m_poCfg->SetPath( wxT("/Simulator") );
  m_poNbkSimEng->bSetPage( m_poCfg->Read( wxT("Analysis"), wxT("OP") ) );

  // Initialize the FileTasks object eg. load schematic or netlist file/s
  m_oFileTsks.Initialize( );

  // If necessary load the simulation or netlist
  if( ! m_oNetLst.bIsEmpty( ) ) bSimnLoad( );

  // Display the appropriate text control page
  if(      ! m_oNbkTxtCtls.poGetPage( NbkTxtCtls::ePAGE_SIMULTN )->bIsEmpty() )
    m_oNbkTxtCtls.bSetPage( NbkTxtCtls::ePAGE_SIMULTN );
  else if( ! m_oNbkTxtCtls.poGetPage( NbkTxtCtls::ePAGE_NETLIST )->bIsEmpty() )
    m_oNbkTxtCtls.bSetPage( NbkTxtCtls::ePAGE_NETLIST );
  else
    m_oNbkTxtCtls.bSetPage( NbkTxtCtls::ePAGE_CONSOLE );
}

//*****************************************************************************
// Initialize the menu bar.

void  FrmMain::InitMenuBar( void )
{
  // Create the menu bar and menus
  wxMenuBar * poMenuBar  = new wxMenuBar;
  wxMenu    * poMenuFile = new wxMenu;
  wxMenu    * poMenuSimu = new wxMenu;
  wxMenu    * poMenuSets = new wxMenu;
  wxMenu    * poMenuHelp = new wxMenu;

  // Set the tool bar font
  poMenuBar->SetFont( FONT_NORM );

  // Load the menus with items
  poMenuFile->Append( ID_MNU_OPEN  ,           wxT(" &Open ... ")           );
  poMenuFile->Append( ID_MNU_IMPORT,           wxT(" &Import ... ")         );
  poMenuFile->Append( ID_MNU_RELOAD,           wxT(" &Reload ")             );
  poMenuFile->Append( ID_MNU_CLOSE ,           wxT(" &Close ")              );
  poMenuFile->AppendSeparator( );
  poMenuFile->Append( ID_MNU_EXIT  ,           wxT(" E&xit ")               );

  poMenuSimu->Append( ID_MNU_CREATE,           wxT(" &Create ")             );
  poMenuSimu->Append( ID_MNU_RUN   ,           wxT(" &Run ")                );
  poMenuSimu->Append( ID_MNU_STOP  ,           wxT(" &Stop ")               );
  poMenuSimu->AppendSeparator( );
  poMenuSimu->Append( ID_MNU_SCHEM ,           wxT(" Sc&hematic ... ")      );
  poMenuSimu->Append( ID_MNU_VIEWER,           wxT(" R&esults ... ")        );

  poMenuSets->AppendRadioItem( ID_MNU_GNUCAP , wxT(" &GNU-Cap ")            );
  poMenuSets->AppendRadioItem( ID_MNU_NGSPICE, wxT(" &NG-Spice ")           );
  poMenuSets->AppendSeparator( );
  poMenuSets->Append( ID_MNU_PREFS,            wxT(" &Preferences ... ")    );

  poMenuHelp->Append( ID_MNU_MAN_USR,          wxT(" &User Manual ... ")    );
// poMenuHelp->Append( ID_MNU_MAN_NGS,          wxT(" &NG-Spice Manual ...") );
// poMenuHelp->Append( ID_MNU_MAN_GCP,          wxT(" &GNU-Cap Manual ...")  );
// poMenuHelp->AppendSeparator( );
  poMenuHelp->Append( ID_MNU_ABOUT  ,          wxT(" &About ... ")          );

  // Load the menubar with menus
  poMenuBar ->Append( poMenuFile,              wxT(" &File ")               );
  poMenuBar ->Append( poMenuSimu,              wxT(" &Simulate ")           );
  poMenuBar ->Append( poMenuSets,              wxT(" S&ettings ")           );
  poMenuBar ->Append( poMenuHelp,              wxT(" &Help ")               );

  // Check the appropriate simulation engine menu item
  m_poCfg->SetPath( wxT("/Simulator") );
  if( m_poCfg->Read( wxT("Engine"), CLP_GNUCAP ) == CLP_NGSPICE )
       poMenuSets->Check( ID_MNU_NGSPICE, TRUE );
  else poMenuSets->Check( ID_MNU_GNUCAP , TRUE );

  // Attach the menu bar to the frame
  SetMenuBar( poMenuBar );

  // Disable the stop option
  poMenuBar->Enable( ID_MNU_STOP, FALSE );
}

//*****************************************************************************
// Initialize the tool bar.

void  FrmMain::InitToolBar( void )
{
  wxBitmap  * poPixMap[ 10 ];
  wxToolBar * poToolBar;

  // Create the tool bar
  poToolBar = CreateToolBar( wxHORIZONTAL | wxTB_FLAT );

  // Create the bitmaps for the tools
  poPixMap[ 0 ] = new wxBitmap( file_open_xpm   );
  poPixMap[ 1 ] = new wxBitmap( file_import_xpm );
  poPixMap[ 2 ] = new wxBitmap( file_reload_xpm );
  poPixMap[ 3 ] = new wxBitmap( file_close_xpm  );
  poPixMap[ 4 ] = new wxBitmap( sim_create_xpm  );
  poPixMap[ 5 ] = new wxBitmap( sim_run_xpm     );
  poPixMap[ 6 ] = new wxBitmap( sim_stop_xpm    );
  poPixMap[ 7 ] = new wxBitmap( sim_edit_xpm    );
  poPixMap[ 8 ] = new wxBitmap( sim_plot_xpm    );
  poPixMap[ 9 ] = new wxBitmap( help_xpm        );

  // Add the tools to the toolbar
  poToolBar->SetToolSeparation( 10 );

  poToolBar->AddTool( ID_TBR_OPEN,   wxT(""), *(poPixMap[ 0 ]),
                      wxT("Open a netlist file") );
  poToolBar->AddTool( ID_TBR_IMPORT, wxT(""), *(poPixMap[ 1 ]),
                      wxT("Import schematic file/s") );
  poToolBar->AddTool( ID_TBR_RELOAD, wxT(""), *(poPixMap[ 2 ]),
                      wxT("Reload netlist / schematic file/s") );
  poToolBar->AddTool( ID_TBR_CLOSE,  wxT(""), *(poPixMap[ 3 ]),
                      wxT("Close netlist / schematic file/s") );

  poToolBar->AddSeparator( );

  poToolBar->AddTool( ID_TBR_CREATE, wxT(""), *(poPixMap[ 4 ]),
                      wxT("Create a netlist file containing simulation instructions") );
  poToolBar->AddTool( ID_TBR_RUN,    wxT(""), *(poPixMap[ 5 ]),
                      wxT("Run the simulation engine process") );
  poToolBar->AddTool( ID_TBR_STOP,   wxT(""), *(poPixMap[ 6 ]),
                      wxT("Stop the simulation engine process") );

  poToolBar->AddSeparator( );

  poToolBar->AddTool( ID_TBR_SCHEM,  wxT(""), *(poPixMap[ 7 ]),
                      wxT("Edit / view schematic/s") );
  poToolBar->AddTool( ID_TBR_VIEWER, wxT(""), *(poPixMap[ 8 ]),
                      wxT("View results data") );

  poToolBar->AddSeparator( );

  poToolBar->AddTool( ID_TBR_HELP,   wxT(""), *(poPixMap[ 9 ]),
                      wxT("View the user manual") );

  // Realize the toolbar
  poToolBar->Realize( );

  // Delete the bitmaps
  for( int i1=0; i1<10; i1++ ) delete poPixMap[ i1 ];

  // Disable the stop button
  poToolBar->EnableTool( ID_TBR_STOP, FALSE );
}

//*****************************************************************************
// Initialize the test node and component list boxes.

void  FrmMain::InitLstBoxs( void )
{
  long  lStyle;

  // Create the node and component list labels
  lStyle = wxALIGN_CENTER;
  m_oLblNodes.Create( this, ID_UNUSED, wxT("Nodes"),      wxDefaultPosition,
                      wxDefaultSize, lStyle );
  m_oLblCpnts.Create( this, ID_UNUSED, wxT("Components"), wxDefaultPosition,
                      wxDefaultSize, lStyle );

  // Create the node and component list boxes
  lStyle = wxLB_MULTIPLE | wxLB_ALWAYS_SB;
  m_oLbxNodes.Create( this, ID_LBX_NODES, wxDefaultPosition,
                      wxSize( 100, -1 ), 0, NULL, lStyle );
  m_oLbxCpnts.Create( this, ID_LBX_CPNTS, wxDefaultPosition,
                      wxSize( 100, -1 ), 0, NULL, lStyle );
}

//*****************************************************************************
// Initialize the console notebook.

void  FrmMain::InitNbkTCtls( void )
{
  long  li1;

  // Create the text control notebook
  m_oNbkTxtCtls.bCreate( this, ID_NBK_TXTCTRLS );

  // Set the text control maximum lines
  m_poCfg->SetPath( wxT("/Main") );
  m_poCfg->Read( wxT("NbkMaxLines"), &li1, TCTL_LINESDEF );
  m_oNbkTxtCtls.bSetLinesMax( li1 );

  // Set the precision of the results data
  m_poCfg->Read( wxT("Precision"),   &li1, CNVTYPE_DFLT_FLTRES );
  ConvertType::bSetFltRes( (int) li1 );
}

//*****************************************************************************
// Initialize the status bar.
//
// Note : The first field in the status bar has benn effectively disable by
//        setting it's width to near zero. It is not used since the frame
//        insists on writing text into this field as the user moves the mouse
//        over display controls.

void  FrmMain::InitStatBar( void )
{
  wxString  os1;
  int       ia1[ ePANE_LAST+1 ];

  // Create the status bar
  CreateStatusBar( ePANE_LAST+1 );

  // Disable the status bar pane used to display menu and toolbar help
  SetStatusBarPane( -1 );

  // Determine the status bar field widths
  ia1[ ePANE_MESG ] = -1;  // Variable width
  ia1[ ePANE_SIMR ] = 144; // Fixed width
  ia1[ ePANE_VIEW ] = 116; // Fixed width
  SetStatusWidths( ePANE_LAST+1, ia1 );

  // Get the viewer application name from the configuration object
  m_poCfg->SetPath( wxT("/Viewer") );
  m_poCfg->Read( wxT("Name"), &os1, CLP_GWAVE );

  // Set status bar text
  SetStatusText( wxT(" Welcome message") , ePANE_MESG );
  SetStatusText( wxT(" Simulator : None"), ePANE_SIMR );
  SetStatusText( wxT(" Viewer : ") + os1 , ePANE_VIEW );
}

//*****************************************************************************
// Initialize the tool tips.

void  FrmMain::InitToolTips( void )
{
  long  li1;

  // Define tool tips for each control
  m_oLbxNodes.SetToolTip( wxT("Test Nodes") );
  m_oLbxCpnts.SetToolTip( wxT("Test Components") );

  // Set global tool tip attributes
  m_poCfg->SetPath( wxT("/Main") );
  m_poCfg->Read( wxT("ToolTips"), &li1, (long) 0 );
  wxToolTip::Enable( li1 != 0 ? TRUE : FALSE );
  wxToolTip::SetDelay( 700 );
}

//*****************************************************************************
// Set the electronic circuit simulator engine to use.

void  FrmMain::InitSimEng( void )
{
  // Delete any existing simulation engine notebook object
  if( m_poNbkSimEng != NULL )
  {
    GetSizer( )->Detach( m_poNbkSimEng );
    delete m_poNbkSimEng;
    m_poNbkSimEng = NULL;
  }

  // Delete any existing simulation engine process object
  if( m_poPrcSimEng != NULL )
  {
    delete m_poPrcSimEng;
    m_poPrcSimEng = NULL;
  }

  // Create the new simulation engine objects and update the status text
  if( GetMenuBar( )->IsChecked( ID_MNU_NGSPICE ) )
  {
    m_poNbkSimEng = new NbkNgSpice( this, ID_NBK_ANALYSIS );
    m_poPrcSimEng = new PrcNgSpice( );
    m_poSimn      = &m_oSimnNgs;

    SetStatusText( wxT(" Simulator : NG-Spice"), ePANE_SIMR );
  }
  else
  {
    m_poNbkSimEng = new NbkGnuCap( this, ID_NBK_ANALYSIS );
    m_poPrcSimEng = new PrcGnuCap( );
    m_poSimn      = &m_oSimnGcp;

    SetStatusText( wxT(" Simulator : GNU-Cap"), ePANE_SIMR );
  }
}

//*****************************************************************************
// Initialize the log file names in the process objects.

void  FrmMain::InitLogFiles( void )
{
  wxFileName  ofn1;
  wxString    os1;

  // Check that a netlist file has been defined
  ofn1 = m_oFileTsks.rosGetNetLstFile( );
  if( ! ofn1.IsOk( )       ) return;
  if( ! ofn1.FileExists( ) ) return;

  // Get the path to the schematic or netlist file
  os1 = ofn1.GetPath( ) + wxT("/gspiceui.log");

  // Set the log file path for the gNetList process object
  m_oFileTsks.bSetLogFile( os1 );

  // Set the log file path for the simulation process object
  if( m_poPrcSimEng != NULL )
    m_poPrcSimEng->bSetLogFile( os1 );
}

//*****************************************************************************
// Initialize the frames position amd size.

void  FrmMain::InitPosnSize( void )
{
  int  ix, iy, iw, ih;

  // Get the default position and size
  GetPosition( &ix, &iy );
  GetClientSize( &iw, &ih );

  // Get the position and size from the configuration object
  m_poCfg->SetPath( wxT("/Main") );
  ix = m_poCfg->Read( wxT("PosnX"), 1      );
  iy = m_poCfg->Read( wxT("PosnY"), iy     );
  iw = m_poCfg->Read( wxT("SizeW"), iw     );
  ih = m_poCfg->Read( wxT("SizeH"), ih+100 ); // (10/03/2011) ??? Is this line related to the main frame sizing problems when simr. engine changes?

  // Set the position and size
  Move( ix, iy );
  SetClientSize( iw, ih );
}

//*****************************************************************************
// Layout the main frame display objects.

void  FrmMain::DoLayout( void )
{
  wxGridBagSizer * poSzr;
  wxGBPosition     oGBPosn;
  wxGBSpan         oGBSpan;
  int              iFlags;
  int              iBorder;
  size_t           sz1;

  if( GetSizer( ) == NULL )
  { // Create and set the frame's sizer (at startup)
    poSzr = new wxGridBagSizer( 1, 1 );
    SetSizer( poSzr );
  }
  else
  { // Detach the display objects from the sizer
    poSzr = (wxGridBagSizer *) GetSizer( );
    poSzr->Detach( &m_oLblNodes   );
    poSzr->Detach( &m_oLbxNodes   );
    poSzr->Detach( &m_oLblCpnts   );
    poSzr->Detach( &m_oLbxCpnts   );
    poSzr->Detach(  m_poNbkSimEng );
    poSzr->Detach( &m_oNbkTxtCtls );
  }

  iFlags  = wxALL | wxALIGN_CENTER;
  iBorder = 9;

  // Add the nodes label
  oGBPosn.SetCol( 0 );        oGBPosn.SetRow( 0 );
  oGBSpan.SetColspan( 1 );    oGBSpan.SetRowspan( 1 );
  poSzr->Add( &m_oLblNodes,   oGBPosn, oGBSpan, iFlags, iBorder );

  // Add the components label
  oGBPosn.SetCol( 1 );        oGBPosn.SetRow( 0 );
  oGBSpan.SetColspan( 1 );    oGBSpan.SetRowspan( 1 );
  poSzr->Add( &m_oLblCpnts,   oGBPosn, oGBSpan, iFlags, iBorder );

  iFlags |= wxEXPAND;
  iBorder = 3;

  // Add the nodes list box
  oGBPosn.SetCol( 0 );        oGBPosn.SetRow( 1 );
  oGBSpan.SetColspan( 1 );    oGBSpan.SetRowspan( 19 );
  poSzr->Add( &m_oLbxNodes,   oGBPosn, oGBSpan, iFlags, iBorder );

  // Add the components list box
  oGBPosn.SetCol( 1 );        oGBPosn.SetRow( 1 );
  oGBSpan.SetColspan( 1 );    oGBSpan.SetRowspan( 19 );
  poSzr->Add( &m_oLbxCpnts,   oGBPosn, oGBSpan, iFlags, iBorder );

  // Add the simulator notebook
  oGBPosn.SetCol( 2 );        oGBPosn.SetRow( 0 );
  oGBSpan.SetColspan( 1 );    oGBSpan.SetRowspan( 13 );
  poSzr->Add( m_poNbkSimEng,  oGBPosn, oGBSpan, iFlags, iBorder );
#ifndef LAYOUT_MNGR // ???
  poSzr->SetItemMinSize( m_poNbkSimEng, NBKSIMR_WIDTH, NBKSIMR_HEIGHT );
#endif // LAYOUT_MNGR

  // Add the console notebook
  oGBPosn.SetCol( 2 );        oGBPosn.SetRow( 13 );
  oGBSpan.SetColspan( 1 );    oGBSpan.SetRowspan( 7 );
  poSzr->Add( &m_oNbkTxtCtls, oGBPosn, oGBSpan, iFlags, iBorder );

  // Specify how the sizer will grow when resized
  poSzr->SetNonFlexibleGrowMode( wxFLEX_GROWMODE_SPECIFIED );
  poSzr->AddGrowableCol( 0 );
  poSzr->AddGrowableCol( 1 );
  for( sz1=13; sz1<20; sz1++ )
    poSzr->AddGrowableRow( sz1 );

  // Set minimum and initial sizes as calculated by the frame's sizer
  poSzr->SetSizeHints( this );
}

//*****************************************************************************
// Load information from the Simulation object attribute m_poSimn.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  FrmMain::bSimnLoad( void )
{
  TextCtrl     * poTxtCtl;
  wxArrayString  osa1;
  size_t         sz1;

  // Load all the nodes (except ground) into the "Nodes" list box
  for( sz1=0; sz1<m_poSimn->m_osaNodeLbls.GetCount( ); sz1++ )
    m_oLbxNodes.Append( m_poSimn->m_osaNodeLbls.Item( sz1 ) );
  // Select the test nodes (if any have been specified)
  for( sz1=0; sz1<m_poSimn->rosaGetTstNodes( ).GetCount( ); sz1++ )
    m_oLbxNodes.SetStringSelection( m_poSimn->rosaGetTstNodes( ).Item( sz1 ) );

  // Load all 2-port components into the "Components" list box
  for( sz1=0; sz1<m_poSimn->m_oaCpnts.GetCount( ); sz1++ )
  {
    const Component & roCpnt = m_poSimn->m_oaCpnts.Item( sz1 );
    switch( roCpnt.m_eType )        // Only collect two terminal components
    {
      case eCPNT_SCJN  :            // Super-conductor Junction
      case eCPNT_CAP   :            // Capacitor
      case eCPNT_RES   :            // Resistor
      case eCPNT_ADM   :            // Admittance
      case eCPNT_IND   :            // Inductor
      case eCPNT_DIODE :            // Diode
      case eCPNT_ICS   :            // Independent Current Source
      case eCPNT_IVS   : break;     // Independent Voltage Source
      default          : continue;  // Do nothing
    }
    osa1.Add( roCpnt.m_osName );
  }
  osa1.Sort( &iStrCmpCpnt );
  m_oLbxCpnts.InsertItems( osa1, 0 );
  // Select the test components (if any have been specified)
  for( sz1=0; sz1<m_poSimn->rosaGetTstCpnts( ).GetCount( ); sz1++ )
    m_oLbxCpnts.SetStringSelection( m_poSimn->rosaGetTstCpnts( ).Item( sz1 ) );

  // Load the circuit description into the text control/s
  if( ! m_oNetLst.m_osaNetLst.IsEmpty( ) )
  {
    // Load the circuit description into the NetList text control
    m_oNbkTxtCtls.bSetPage( NbkTxtCtls::ePAGE_NETLIST );
    poTxtCtl = m_oNbkTxtCtls.poGetPage( );
    poTxtCtl->bClear( );
    for( sz1=0; sz1<m_oNetLst.m_osaNetLst.GetCount( ); sz1++ )
      poTxtCtl->bAppendLine( m_oNetLst.m_osaNetLst.Item( sz1 ) );
    m_oNbkTxtCtls.bSetPosn( 0 );

    // Load the circuit description into the Simulation text control ??? 06/07/2011
//    if( m_poSimn->bIsValid( ) )
//    {
//      m_oNbkTxtCtls.bSetPage( NbkTxtCtls::ePAGE_SIMULTN );
//      poTxtCtl = m_oNbkTxtCtls.poGetPage( );
//      poTxtCtl->bClear( );
//      for( sz1=0; sz1<m_oNetLst.m_osaNetLst.GetCount( ); sz1++ )
//        poTxtCtl->bAppendLine( m_oNetLst.m_osaNetLst.Item( sz1 ) );
//      m_oNbkTxtCtls.bSetPosn( 0 );
//    }
  }

  // Load the analysis notebook with simulation information
  if( ! m_poNbkSimEng->bLoad( *m_poSimn ) ) return( FALSE );

  return( TRUE );
}

//*****************************************************************************
// Reload information from the Simulation object attribute m_poSimn.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  FrmMain::bSimnReload( void )
{
  wxArrayString  osaTstNodes;
  wxArrayString  osaTstCpnts;
  size_t         sz1;
  int            i1;

  // Record the nodes that are currently selected
#if wxCHECK_VERSION( 2,8,0 )
  for( sz1=0; sz1<m_oLbxNodes.GetCount( ); sz1++ )
#else
  for( sz1=0; sz1<(size_t)m_oLbxNodes.GetCount( ); sz1++ )
#endif
    if( m_oLbxNodes.IsSelected( sz1 ) )
      osaTstNodes.Add( m_oLbxNodes.GetString( sz1 ) );

  // Record the components that are currently selected
#if wxCHECK_VERSION( 2,8,0 )
  for( sz1=0; sz1<m_oLbxCpnts.GetCount( ); sz1++ )
#else
  for( sz1=0; sz1<(size_t)m_oLbxNodes.GetCount( ); sz1++ )
#endif
    if( m_oLbxCpnts.IsSelected( sz1 ) )
      osaTstCpnts.Add( m_oLbxCpnts.GetString( sz1 ) );

  // Clear the lists of nodes and components
  m_oLbxNodes.Clear( );
  m_oLbxCpnts.Clear( );

  bSimnLoad( );  // Load the Simulation object

  // Select nodes that where previously selected
  if( ! osaTstNodes.IsEmpty( ) )
  {
    for( sz1=0; sz1<osaTstNodes.GetCount( ); sz1++ )
    {
      i1 = m_oLbxNodes.FindString( osaTstNodes.Item( sz1 ) );
      if( i1 != wxNOT_FOUND ) m_oLbxNodes.Select( i1 );
    }
  }

  // Select components that where previously selected
  if( ! osaTstCpnts.IsEmpty( ) )
  {
    for( sz1=0; sz1<osaTstCpnts.GetCount( ); sz1++ )
    {
      i1 = m_oLbxCpnts.FindString( osaTstCpnts.Item( sz1 ) );
      if( i1 != wxNOT_FOUND ) m_oLbxCpnts.Select( i1 );
    }
  }

  return( TRUE );
}

//*****************************************************************************
// Save information to the Simulation object attribute m_poSimn.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  FrmMain::bSimnSave( void )
{
  wxArrayInt  oaiSelNodes, oaiSelCpnts;
  size_t      sz1;
  int         iSelNo;
  wxString    osSel;

  // Clear all previously test points and command objects
  m_poSimn->bClrCmds( );

  // Get an array of indicies to the currently selected test points
  m_oLbxNodes.GetSelections( oaiSelNodes );
  m_oLbxCpnts.GetSelections( oaiSelCpnts );

  // Load the test nodes into the simulation object
  for( sz1=0; sz1<oaiSelNodes.GetCount( ); sz1++ )
  {
    iSelNo = oaiSelNodes.Item( sz1 );
    osSel = m_oLbxNodes.GetString( iSelNo );
    m_poSimn->bAddTstNode( osSel );
  }

  // Load the test components into the simulation object
  for( sz1=0; sz1<oaiSelCpnts.GetCount( ); sz1++ )
  {
    iSelNo = oaiSelCpnts.Item( sz1 );
    osSel = m_oLbxCpnts.GetString( iSelNo );
    m_poSimn->bAddTstCpnt( osSel );
  }

  // Transfer the simulation parameters into the simulation object
  if( ! m_poNbkSimEng->bSave( *m_poSimn ) ) return( FALSE );

  // Create the simulation file
  if( ! m_poSimn->bSaveFile( ) )            return( FALSE );

  return( TRUE );
}

//*****************************************************************************
// Check that the electronic circuit simulator engine is present and accounted
// for, if not display an error message.
//
// Argument List :
//   poPrcSimEng - The utility object to be tested
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  FrmMain::bIsOkSimEng( PrcBase * poPrcSimEng )
{
  wxString  os1;

  // Check that gnetlist exists and is accessible
  if( ! poPrcSimEng->bBinExists( ) )
  {
    os1 << wxT("Can't find ") << poPrcSimEng->roGetBinary( ).GetFullName( )
        << wxT(" which is required to run electronic simulation.\n")
        << wxT("There is no path to it or it has not been installed.\n\n");
    DlgErrMsg( wxT("Configuration Fault"), os1 );

    return( FALSE );
  }

  return( TRUE );
}

//*****************************************************************************
// Check that the waveform viewer utility is present and accounted for, if not
// display an error message.
//
// Argument List  :
//   poPrcViewer - The utility object to be tested
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  FrmMain::bIsOkViewer( PrcBase * poPrcViewer )
{
  wxString  os1;

  // Check that the waveform viewer utility exists and is accessible
  if( ! poPrcViewer->bBinExists( ) )
  {
    os1 << wxT("Can't find ") << poPrcViewer->roGetBinary( ).GetFullName( )
        << wxT(" which is required to plot simulation results.\n")
        << wxT("There is no path to it or it has not been installed.\n\n");
    DlgErrMsg( wxT("Configuration Fault"), os1 );

    return( FALSE );
  }

  return( TRUE );
}

//*****************************************************************************
// Clear the object attributes.

bool  FrmMain::bClear( void )
{
  bool  bRtn = TRUE;

  // Clear all previously selected test points
  m_oLbxNodes.Clear( );
  m_oLbxCpnts.Clear( );

  // Clear simulation object attributes
  m_oSimnGcp.bClear( );
  m_oSimnNgs.bClear( );

  // Clear the simulation object and the analysis panels
  if( m_poNbkSimEng != NULL )
    if( ! m_poNbkSimEng->bClear( ) )   bRtn = FALSE;

  // Clear the text controls
  if( ! m_oNbkTxtCtls.bClear( ) )      bRtn = FALSE;
  m_oNbkTxtCtls.bInitialize( );

  // Terminate any simulation process
  if( m_poPrcSimEng != NULL )
    if( ! m_poPrcSimEng->bKill( ) )    bRtn = FALSE;

  // Terminate any schematic editor process
  if( ! m_oPrc_gschem.bKill( ) )       bRtn = FALSE;

  // Terminate any waveform viewer process
  m_poCfg->SetPath( wxT("/Viewer") );
  if( m_poCfg->Read( wxT("Name") ) == CLP_GWAVE )
       { if( ! m_oPrc_gwave.bKill( ) ) bRtn = FALSE; }
  else { if( ! m_oPrc_gaw  .bKill( ) ) bRtn = FALSE; }

  return( bRtn );
}

//*****************************************************************************
// Display a dialog box containing an error message.
//
// Note : At start-up the first error message is held over until the the main
//        GUI is created and displayed; to display this message call this
//        function with no arguments.
//
// Argument List :
//   rosTitle - The dialogue box title
//   rosMsg   - The error message

void  FrmMain::DlgErrMsg( const wxString & rosTitle, const wxString & rosMsg )
{
  static  wxMessageDialog * poDlgMsg=NULL;
  wxStringTokenizer  ostk1;
  wxString           os1, os2, os3;

  if( poDlgMsg == NULL )
  {
    if( rosTitle.IsEmpty( ) || rosMsg.IsEmpty( ) ) return;

    // Tokenize the message into lines
    ostk1.SetString( rosMsg, wxT(" \n\r"), wxTOKEN_STRTOK );

    // Use the first complete sentence as the status bar message
    os1 << wxT(" ERROR :");
    while( ostk1.HasMoreTokens( ) )
    {
      // Get the next line
      os2 = ostk1.GetNextToken( ).Strip( wxString::both );

      // Remove the path from file names
      if( os2.Freq( wxT('/') ) > 1 )
      {
        os3 << os2.AfterLast( wxT('/') ) << wxT(',');
        os2 = os3;
      }

      // Look for the end of the sentence
      if( os2==wxT('.') || os2==wxT(':') ) break;

      // Append the line to the message
      os1 << wxT(' ') << os2;
    }

    // If debug mode is enabled send the error message to the console
    if( g_bDebug )
      std::cerr << "DEBUG : " << rosStrToLine( os1 ).mb_str( ) << "\n\n";

    // Set the status line message
    SetStatusText( os1, ePANE_MESG );

    // Create the error message dialog
    os1 = wxT('\n') + rosMsg;
    poDlgMsg = new wxMessageDialog( this, os1, rosTitle, wxOK | wxICON_ERROR );
  }

  if( ! IsShown( ) ) return;

  // Display the error message dialog
  poDlgMsg->ShowModal( );

  // Delete the error message dialog
  delete poDlgMsg;
  poDlgMsg = NULL;
}

//*****************************************************************************
//                                                                            *
//                               Event Handlers                               *
//                                                                            *
//*****************************************************************************
// Open a circuit description file.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnOpen( wxCommandEvent & roEvtCmd )
{
  // Attempt to get a netlist file name from the user
  if( ! m_oFileTsks.bDlgOpen( ) ) return;

  ::wxBeginBusyCursor( );       // Change the cursor to the wait symbol

  bClear( );                    // Clear the main frame object attributes

  // Attempt to open and read the netlist file
  m_oFileTsks.bOpen( );

  bSimnLoad( );                 // Load the simulation information
  InitLogFiles( );              // Initialize the process log files

  ::wxEndBusyCursor( );         // Change the cursor to the default
}

//*****************************************************************************
// Import a schematic file using gnetlist to convert the schematic to a circuit
// description.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnImport( wxCommandEvent & roEvtCmd )
{
  // Attempt to get the schematic file name/s from the user
  if( ! m_oFileTsks.bDlgImport( ) ) return;

  ::wxBeginBusyCursor( );       // Change the cursor to the wait symbol

  bClear( );                    // Clear the main frame object attributes

  // Attempt to import the schematic file/s and read the netlist file
  m_oFileTsks.bImport( );

  bSimnLoad( );                 // Load the simulation information
  InitLogFiles( );              // Initialize the process log files

  ::wxEndBusyCursor( );         // Change the cursor to the default
}

//*****************************************************************************
// Reload the schematic or netlist file.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnReload( wxCommandEvent & roEvtCmd )
{
  ::wxBeginBusyCursor( );  // Change the cursor to the wait symbol

  m_oFileTsks.bReload( );  // Reload schematic file/s or the netlist file
  bSimnReload( );          // Reload the simulation information

  ::wxEndBusyCursor( );    // Change the cursor to the default
}

//*****************************************************************************
// Close the circuit description file.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnClose( wxCommandEvent & roEvtCmd )
{
  m_oFileTsks.bClose( );                    // Delete temporary files
  bClear( );                                // Clear the object attributes
  m_oFileTsks.bSetTitle( );                 // Set the frame title
  m_poNbkSimEng->bSetPage( eCMD_ANA_FST );  // Display first analysis panel
}

//*****************************************************************************
// Exit the application.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnAppExit( wxCommandEvent & roEvtCmd )
{
  Close( TRUE ); // Generates a wxCloseEvent which is handled by OnSysExit( )
}

//*****************************************************************************
// Create the simulation.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnSimCreate( wxCommandEvent & roEvtCmd )
{
  TextCtrl * poTxtCtl;
  wxString   os1;

  ::wxBeginBusyCursor( );  // Change the cursor to the wait symbol

  if( ! m_oNetLst.m_osaNetLst.IsEmpty( ) )
  { // Save the simualtion info. in the simulation object
    if( bSimnSave( ) )
    {
      // Print the simulation to the simulation text control
      m_oNbkTxtCtls.bSetPage( NbkTxtCtls::ePAGE_SIMULTN );
      poTxtCtl = m_oNbkTxtCtls.poGetPage( NbkTxtCtls::ePAGE_SIMULTN );
      poTxtCtl->bLoadFile( m_poSimn->roGetSaveFile( ).GetFullPath( ) );
      poTxtCtl->SetEditable( TRUE );
    }
    else
    {
      if(      ! m_poNbkSimEng->bIsOk( ) ) os1 = m_poNbkSimEng->rosGetErrMsg();
      else if( ! m_poSimn->bIsValid( )   ) os1 = m_poSimn     ->rosGetErrMsg();
      else                                 os1 = wxT("Unknown error.");
      DlgErrMsg( wxT("Create Simulation Error"), os1 );
    }
  }
  else
  { // Display an error message dialogue
    os1 = wxT("There is no netlist loaded.");
    DlgErrMsg( wxT("Create Simulation Error"), os1 );
  }

  ::wxEndBusyCursor( );  // Change the cursor to the default

  // Set the status line text
  if( os1.IsEmpty( ) ) os1 = wxT(" Simulation created successfully");
  else                 os1 = wxT(" Error/s encountered creating simulation");
  SetStatusText( os1, ePANE_MESG );
}

//*****************************************************************************
// Run the simulation.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnSimRun( wxCommandEvent & roEvtCmd )
{
  TextCtrl             * poTxtCtl;
  NbkTxtCtls::ePageType  ePage;
  wxFileName             ofn1;
  wxString               os1;

  // Check if a schematic or netlist file is currently open
  ofn1 = m_oFileTsks.rosGetNetLstFile( );
  if( ofn1.GetFullPath( ).IsEmpty( ) )
  {
    os1 = wxT("There is no schematic or netlist file currently loaded.");
    DlgErrMsg( wxT("Run Simulation Error"), os1 );
    return;
  }

  ::wxBeginBusyCursor( );                         // Change cursor to busy sym.
  GetMenuBar( )->Enable(     ID_MNU_STOP, TRUE ); // Enable Stop option
  GetToolBar( )->EnableTool( ID_TBR_STOP, TRUE ); // Enable Stop tool

  //*****************************************************************
  // Prepare to run the simulation (and update the Simulation text page)
  poTxtCtl = m_oNbkTxtCtls.poGetPage( NbkTxtCtls::ePAGE_SIMULTN );
  if( poTxtCtl->IsModified( ) )
  { // Save the simulation in the text page to file, the user has modified it
    poTxtCtl->SaveFile( m_poSimn->roGetSaveFile( ).GetFullPath( ) );
  }
  else if( bSimnSave( ) )
  { // Load the simulation on file into the Simulation text page
    poTxtCtl->bLoadFile( m_poSimn->roGetSaveFile( ).GetFullPath( ) );
    // poTxtCtl is saved immediately after loading to clear IsModified flag
    poTxtCtl->SaveFile( m_poSimn->roGetSaveFile( ).GetFullPath( ) );

    // ??? 26/11/2008 The editable flag needs to set when there's a simulation
    //                file loaded in the text control and cleared when it's
    //                "empty". Is this the "right" place to do it tho.?
    poTxtCtl->SetEditable( TRUE );
  }
  else
  { // There's been an error, report it to the user and return
    if(      ! m_poNbkSimEng->bIsOk( ) ) os1 = m_poNbkSimEng->rosGetErrMsg( );
    else if( ! m_poSimn->bIsValid( )   ) os1 = m_poSimn     ->rosGetErrMsg( );
    else                                 os1 = wxT("Unknown error.");
    DlgErrMsg( wxT("Run Simulation Error"), os1 );
    goto OnSimRun_Exit;
  }
  m_oNbkTxtCtls.bSetPage( NbkTxtCtls::ePAGE_SIMULTN );

  //*****************************************************************
  // Run the simulation (and update the Console text page)
  m_oNbkTxtCtls.bSetPage( NbkTxtCtls::ePAGE_CONSOLE );
  poTxtCtl = m_oNbkTxtCtls.poGetPage( NbkTxtCtls::ePAGE_CONSOLE );
  poTxtCtl->bClear( );                // Clear the Console text page
  if( ! m_poPrcSimEng->bMakeArgLst( *m_poSimn ) )
  {
    DlgErrMsg( wxT( "Run Simulation Error" ), m_poPrcSimEng->rosGetErrMsg( ) );
    goto OnSimRun_Exit;
  }
  m_poPrcSimEng->PrintCmd( *poTxtCtl ); // Print simulator cmd to Console page
  SetStatusText( wxT(" Running the simulation ..."), ePANE_MESG );
  if( ! m_poPrcSimEng->bExec( ) )
  {
    DlgErrMsg( wxT( "Run Simulation Error" ), m_poPrcSimEng->rosGetErrMsg( ) );
    goto OnSimRun_Exit;
  }
  if( GetStatusBar( )->GetStatusText( 1 ).Contains( wxT("aborted") ) )
  { // Just exit ammediately if the user aborts the simulation
    goto OnSimRun_Exit;
  }
  m_poPrcSimEng->PrintRsp( *poTxtCtl ); // Print simulator output to Console page

  //*****************************************************************
  // Format the simulation results (and update the Results text page)
  SetStatusText( wxT(" Formatting the simulation results ..."), ePANE_MESG );
  wxYield( );
  if( ! m_poPrcSimEng->bFmtResults( ) )
  {
    DlgErrMsg( wxT( "Run Simulation Error" ), m_poPrcSimEng->rosGetErrMsg( ) );
    goto OnSimRun_Exit;
  }
  if( m_poNbkSimEng->eGetSimEng( ) == eSIMR_GNUCAP )
       ePage = NbkTxtCtls::ePAGE_GNUCAP;
  else ePage = NbkTxtCtls::ePAGE_NGSPICE;
  poTxtCtl = m_oNbkTxtCtls.poGetPage( ePage );
  os1 = m_poPrcSimEng->roGetResultsFile( ).GetFullPath( );
  // Load the simulation output into the results text page
  if( ! poTxtCtl->bLoadFile( os1 ) )
  {
    os1.Prepend( wxT("Couldn't load the results file : \n\n") );
    DlgErrMsg( wxT( "Run Simulation Error" ), os1 );
    goto OnSimRun_Exit;
  }
  m_oNbkTxtCtls.bSetPage( ePage );    // Change to the appropriate results page

  //*****************************************************************

  SetStatusText( wxT(" Simulation ran successfully"), ePANE_MESG );

OnSimRun_Exit:
  ::wxEndBusyCursor( );                            // Change cursor to default
  GetMenuBar( )->Enable(     ID_MNU_STOP, FALSE ); // Disable Stop option
  GetToolBar( )->EnableTool( ID_TBR_STOP, FALSE ); // Disable Stop tool
}

//*****************************************************************************
// Stop the simulation.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnSimStop( wxCommandEvent & roEvtCmd )
{
  if( m_poPrcSimEng->bIsExec( ) )
  {
    m_poPrcSimEng->bKill( );

    SetStatusText( wxT(" Simulation aborted by user"), ePANE_MESG );

    wxMessageDialog  oMsgDlg( this, wxT("\nThe simulation has been aborted."),
                              wxT("Run Simulation Event"), wxICON_INFORMATION );
    oMsgDlg.ShowModal( );

    GetMenuBar( )->Enable(     ID_MNU_STOP, FALSE );
    GetToolBar( )->EnableTool( ID_TBR_STOP, FALSE );
  }
}

//*****************************************************************************
// Edit / view the schematic file.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnSchematic( wxCommandEvent & roEvtCmd )
{
  wxString  os1;

  // Check if the schematic editor process is already running
  if( m_oPrc_gschem.bIsExec( ) )
  {
    os1 = wxT("The schematic editor process is already running.");
    DlgErrMsg( wxT("Schematic Editor Error"), os1 );
    return;
  }

  // Check if a schematic file is currently set
  m_oPrc_gschem.bSetSchems( m_oFileTsks.rosaGetSchemFiles( ) );
  if( m_oPrc_gschem.rosGetSchems( ).IsEmpty( ) )
  {
    os1 = wxT("The netlist has no associated schematic file to edit.");
    DlgErrMsg( wxT("Schematic Editor Error"), os1 );
    return;
  }

  // Execute the schematic editor process
  if( ! m_oPrc_gschem.bExec( ) )
    // Something went wrong, display an error message
    DlgErrMsg( os1, m_oPrc_gschem.rosGetErrMsg( ) );
}

//*****************************************************************************
// View the simulation results.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnViewData( wxCommandEvent & roEvtCmd )
{
  wxFileName  ofn1;
  wxString    os1 = wxT("Plot Results Error");
  wxString    os2;

  // Check if the waveform viewer process is already running
  if( m_oPrc_gaw.bIsExec( ) || m_oPrc_gwave.bIsExec( ) )
  {
    DlgErrMsg( os1, wxT("The waveform viewer process is already running.") );
    return;
  }

  // Check if a schematic or netlist file is currently open
  ofn1 = m_oFileTsks.rosGetNetLstFile( );
  if( ofn1.GetFullPath( ).IsEmpty( ) )
  {
    DlgErrMsg( os1, wxT("There is no schematic or netlist file currently loaded.") );
    return;
  }

  // Create the appropriate name for the results file
  os2 = ofn1.GetName( );
  switch( m_poNbkSimEng->eGetSimEng( ) )
  {
    case eSIMR_GNUCAP :
      if( m_oNbkTxtCtls.eGetPage( ) == NbkTxtCtls::ePAGE_NGSPICE )
           os2 << wxT(".ngspice");
      else os2 << wxT(".gnucap");
      break;

    case eSIMR_NGSPICE :
      if( m_oNbkTxtCtls.eGetPage( ) == NbkTxtCtls::ePAGE_GNUCAP )
           os2 << wxT(".gnucap");
      else os2 << wxT(".ngspice");
      break;

    default :
      return;
  }
  ofn1.SetName( os2 );
  ofn1.SetExt( m_poNbkSimEng->rosGetPage( ) );

  // Set the results file in the waveform viewer process
  if( ! ofn1.FileExists( ) )
  {
    os2.Empty( );
    os2 << wxT("The simulation results file :\n\n") << ofn1.GetFullPath( )
        << wxT("\n\nhasn't yet been generated.");
    DlgErrMsg( os1, os2 );
    return;
  }
  m_oPrc_gaw  .bSetResults( ofn1.GetFullPath( ) );
  m_oPrc_gwave.bSetResults( ofn1.GetFullPath( ) );

  // Execute the waveform viewer process
  m_poCfg->SetPath( wxT("/Viewer") );
  if( m_poCfg->Read( wxT("Name") ) == CLP_GWAVE )
  {
    if( ! m_oPrc_gwave.bExec( ) )
      DlgErrMsg( os1, m_oPrc_gwave.rosGetErrMsg( ) );
  }
  else
  {
    if( ! m_oPrc_gaw.bExec( ) )
      DlgErrMsg( os1, m_oPrc_gaw.rosGetErrMsg( ) );
  }
}

//*****************************************************************************
// Select which simulator to use.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnSelSimEng( wxCommandEvent & roEvtCmd )
{
  int  iw, ih;

  // Temporarily store the current main frame size
  GetClientSize( &iw, &ih );

  // Change cursor to the wait symbol and hide the GUI while re-constructing it
  ::wxBeginBusyCursor( );
  Show( FALSE );

  // Create and display a progress dialog
  wxProgressDialog  oDlgProgress( wxT(" GNU Spice GUI"),
                             wxT("Changing simulation engines ..."), 100, this,
                             wxPD_AUTO_HIDE | wxPD_APP_MODAL | wxPD_SMOOTH );
  oDlgProgress.CenterOnParent( );
  oDlgProgress.Update( 0 ); ::wxMilliSleep( 50 );

  // Record the simulation engine change in the configuration file
  m_poCfg->SetPath( wxT("/Simulator") );
  if( GetMenuBar( )->IsChecked( ID_MNU_GNUCAP ) )
       m_poCfg->Write( wxT("Engine"), CLP_GNUCAP  );
  else m_poCfg->Write( wxT("Engine"), CLP_NGSPICE );
  m_poCfg->Flush( ); // Write changes to the configuration file

  // Update the progress dialog
  oDlgProgress.Update( 10 );

  // Change the simulator
  InitSimEng( );

  // Update the progress dialog
  oDlgProgress.Update( 50 ); ::wxMilliSleep( 50 );

  // Set the simulator process log file name
  InitLogFiles( );

  // Update the progress dialog
  oDlgProgress.Update( 60 ); ::wxMilliSleep( 50 );

  // Update the GUI components
  DoLayout( );

  // Update the progress dialog
  oDlgProgress.Update( 70 ); ::wxMilliSleep( 50 );

  // Transfer info. from one simulation object to the other
  if( ! m_oNetLst.bIsEmpty( ) )
  {
    if( GetMenuBar( )->IsChecked( ID_MNU_GNUCAP ) ) m_oSimnGcp = m_oSimnNgs;
    else                                            m_oSimnNgs = m_oSimnGcp;
  }

  // Update the progress dialog
  oDlgProgress.Update( 80 ); ::wxMilliSleep( 50 );

  // Load the analysis notebook with the simulation values
  m_poNbkSimEng->bLoad( *m_poSimn );
//  bSimnLoad( );

  // Update the progress dialog
  oDlgProgress.Update( 90 ); ::wxMilliSleep( 50 );

  // Restore the frame's original size
  SetClientSize( iw, ih );

  // Update the progress dialog
  oDlgProgress.Update( 99 ); ::wxMilliSleep( 50 );
  oDlgProgress.Update( 100 ); // Kill the progress dialog

  // Show the GUI again and change the cursor to the default
  Show( TRUE );
  ::wxEndBusyCursor( );
}

//*****************************************************************************
// Show the application preferences dialog.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnPrefs( wxCommandEvent & roEvtCmd )
{
  DlgPrefs  oDlgPrefs( this );;
  wxString  os1;

  oDlgPrefs.CenterOnParent( );
  oDlgPrefs.ShowModal( );

  if( oDlgPrefs.GetReturnCode( ) == wxID_OK )
  {
    // Set the waveform viewer
    os1 = wxT(" Viewer : ") + oDlgPrefs.rosGetViewer( );
    SetStatusText( os1, ePANE_VIEW );

    // Set the text control maximum lines
    m_oNbkTxtCtls.bSetLinesMax( oDlgPrefs.iGetNbkMaxLines( ) );

    // Set the synchronize sweep sources flag
    PnlAnaBase::m_bSyncSwpSrcs = oDlgPrefs.bGetSyncSwpSrcs( );
  }
}

//*****************************************************************************
// Display the gSpiceUI user manual.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnManUser( wxCommandEvent & roEvtCmd )
{
  m_oHelpTsks.ManUser( );
}

//*****************************************************************************
// Display about message dialog.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnAbout( wxCommandEvent & roEvtCmd )
{
  m_oHelpTsks.About( );
}

//*****************************************************************************
// Change the cursor when a simulation is running and the mouse is over the
// stop button.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnToolEnter( wxCommandEvent & roEvtCmd )
{
  if( roEvtCmd.GetSelection( ) == ID_TBR_STOP )
  {
    if( m_poPrcSimEng->bIsExec( ) &&  ::wxIsBusy( ) )
    {
      ::wxEndBusyCursor( );    // Change the cursor to the default
      ::wxYield( );            // Allow the display to update
    }
  }
  else
  {
    if( m_poPrcSimEng->bIsExec( ) && !::wxIsBusy( ) )
    {
      ::wxBeginBusyCursor( );  // Change the cursor to the hour glass
      ::wxYield( );            // Allow the display to update
    }
  }
}

//*****************************************************************************
// Event handler for system close.
//
// Argument List :
//   roEvtClose - The event to be processed

void  FrmMain::OnSysExit( wxCloseEvent & roEvtClose )
{
  wxString  os1;
  int       ix, iy, iw, ih;

  bClear( );            // Kill any processes currently running

  m_oFileTsks.bExit( ); // Delete temporary files

  Hide( );              // Hide the main frame while exiting

  // Save the frame size and position
  m_poCfg->SetPath( wxT("/Main") );
  GetClientSize( &iw, &ih );
  GetPosition( &ix, &iy );
  m_poCfg->Write( wxT("PosnX"), (long) ix );
  m_poCfg->Write( wxT("PosnY"), (long) iy );
  m_poCfg->Write( wxT("SizeW"), (long) iw );
  m_poCfg->Write( wxT("SizeH"), (long) ih );
  m_poCfg->Flush( );

  // Record the simulation engine currently selected
  m_poCfg->SetPath( wxT("/Simulator") );
  os1 = m_poPrcSimEng->eGetSimEng( )==eSIMR_GNUCAP ? CLP_GNUCAP : CLP_NGSPICE;
  m_poCfg->Write( wxT("Engine"), os1 );

  // Record the analysis type last used
  os1 = m_poNbkSimEng->rosGetPage( );
  m_poCfg->Write( wxT("Analysis"), os1.Upper( ) );

  // Record the waveform veiwer currently selected
  m_poCfg->SetPath( wxT("/Viewer") );
  if( m_poCfg->Read( wxT("Name") ).IsEmpty( ) )
    m_poCfg->Write( wxT("Name"), CLP_GWAVE );

  m_bIsOpen = FALSE;    // Indicate that the main frame has been closed

  // Destroys the window safely (used instead of the delete operator)
  Destroy( );
}

//*****************************************************************************
