% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/anova_summary.R
\name{anova_summary}
\alias{anova_summary}
\title{Create Nice Summary Tables of ANOVA Results}
\usage{
anova_summary(object, effect.size = "ges", detailed = FALSE, observed = NULL)
}
\arguments{
\item{object}{an object of returned by either \code{\link[car]{Anova}()}, or
\code{\link[stats]{aov}()}.}

\item{effect.size}{the effect size to compute and to show in the ANOVA
results. Allowed values can be either "ges" (generalized eta squared) or
"pes" (partial eta squared) or both. Default is "ges".}

\item{detailed}{If TRUE, returns extra information (sums of squares columns,
intercept row, etc.) in the ANOVA table.}

\item{observed}{Variables that are observed (i.e, measured) as compared to
experimentally manipulated. The default effect size reported (generalized
eta-squared) requires correct specification of the observed variables.}
}
\value{
return an object of class \code{anova_test} a data frame containing
 the ANOVA table for independent measures ANOVA. However, for repeated/mixed
 measures ANOVA, it is a list containing the following components are
 returned:

 \itemize{ \item \strong{ANOVA}: a data frame containing ANOVA results \item
 \strong{Mauchly's Test for Sphericity}: If any within-Ss variables with more
 than 2 levels are present, a data frame containing the results of Mauchly's
 test for Sphericity. Only reported for effects that have more than 2 levels
 because sphericity necessarily holds for effects with only 2 levels. \item
 \strong{Sphericity Corrections}: If any within-Ss variables are present, a
 data frame containing the Greenhouse-Geisser and Huynh-Feldt epsilon values,
 and corresponding corrected p-values. }

 The \strong{returned object might have an attribute} called \code{args} if
 you compute ANOVA using the function \code{\link{anova_test}()}. The attribute \code{args} is a
 list holding the arguments used to fit the ANOVA model, including: data, dv,
 within, between, type, model, etc.


 The following abbreviations are used in the different results tables:

 \itemize{ \item DFn	Degrees of Freedom in the numerator (i.e. DF effect).
 \item DFd	Degrees of Freedom in the denominator (i.e., DF error). \item
 SSn	Sum of Squares in the numerator (i.e., SS effect). \item SSd	Sum of
 Squares in the denominator (i.e.,SS error). \item F	F-value. \item p	p-value
 (probability of the data given the null hypothesis). \item p<.05	Highlights
 p-values less than the traditional alpha level of .05. \item ges	Generalized
 Eta-Squared measure of effect size. \item GGe	Greenhouse-Geisser epsilon.
 \item p[GGe]	p-value after correction using Greenhouse-Geisser epsilon.
 \item p[GGe]<.05	Highlights p-values (after correction using
 Greenhouse-Geisser epsilon) less than the traditional alpha level of .05.
 \item HFe	Huynh-Feldt epsilon. \item p[HFe]	p-value after correction using
 Huynh-Feldt epsilon. \item p[HFe]<.05	Highlights p-values (after correction
 using Huynh-Feldt epsilon) less than the traditional alpha level of .05.
 \item W	Mauchly's W statistic }
}
\description{
Create beautiful summary tables of ANOVA test results obtained
 from either \code{\link[car]{Anova}()} or \code{\link[stats]{aov}()}.

 The results include ANOVA table, generalized effect size and some assumption
 checks.
}
\examples{
# Load data
#:::::::::::::::::::::::::::::::::::::::
data("ToothGrowth")
df <- ToothGrowth
df$dose <- as.factor(df$dose)

# Independent measures ANOVA
#:::::::::::::::::::::::::::::::::::::::::
# Compute ANOVA and display the summary
res.anova <- Anova(lm(len ~ dose*supp, data = df))
anova_summary(res.anova)

# Display both SSn and SSd using detailed = TRUE
# Show generalized eta squared using effect.size = "ges"
anova_summary(res.anova, detailed = TRUE, effect.size = "ges")

# Show partial eta squared using effect.size = "pes"
anova_summary(res.anova, detailed = TRUE, effect.size = "pes")

# Repeated measures designs using car::Anova()
#:::::::::::::::::::::::::::::::::::::::::
# Prepare the data
df$id <- as.factor(rep(1:10, 6)) # Add individuals ids
head(df)

# Easily perform repeated measures ANOVA using the car package
design <- factorial_design(df, dv = len, wid = id, within = c(supp, dose))
res.anova <- Anova(design$model, idata = design$idata, idesign = design$idesign, type = 3)
anova_summary(res.anova)

# Repeated measures designs using stats::Aov()
#:::::::::::::::::::::::::::::::::::::::::
res.anova <- aov(len ~ dose*supp + Error(id/(supp*dose)), data = df)
anova_summary(res.anova)
}
\seealso{
\code{\link{anova_test}()}, \code{\link{factorial_design}()}
}
\author{
Alboukadel Kassambara, \email{alboukadel.kassambara@gmail.com}
}
